//-----------------------------------------------------------
// Copyright Christian Arnault LAL-Orsay CNRS
// arnault@lal.in2p3.fr
// See the complete license in cmt_license.txt "http://www.cecill.info". 
//-----------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "cmt_script.h"
#include "cmt_use.h"

/*----------------------------------------------------------*/
/*                                                          */
/*  Operations on Scripts                                   */
/*                                                          */
/*----------------------------------------------------------*/

/*----------------------------------------------------------*/
void Script::action (const CmtSystem::cmt_string_vector& words,
                     ScriptType type,
                     Use* use)
{
  cmt_string name;
  Script* script;

  if (use == 0) use = &(Use::current());

  name = words[1];
  if (name == "") return;

  script = add (name, type, use);
}

/*----------------------------------------------------------*/
Script* Script::find (const cmt_string& name, Use* use)
{
  int script_index;

  if (use == 0) use = &(Use::current());

  if (use->scripts.size () == 0) return (0);

  for (script_index = 0;
       script_index < use->scripts.size ();
       script_index++)
    {
      Script& script = use->scripts[script_index];

      if (script.name == name)
        {
          return (&script);
        }
    }

  return (0);
}

/*----------------------------------------------------------*/
Script* Script::add (const cmt_string& name, ScriptType type, Use* use)
{
  cmt_string short_name = name;

  if (use == 0) use = &(Use::current());

  int pos;

  pos = short_name.find (".csh");
  if (pos != cmt_string::npos)
    {
      short_name.erase (pos);
    }

  pos = short_name.find (".sh");
  if (pos != cmt_string::npos)
    {
      short_name.erase (pos);
    }

  {
    Script* script;

    if (short_name == "") return (0);
    script = find (short_name, use);
    if (script != 0)
      {
        script->type = type;
        return (script);
      }
  }

  Script& script = use->scripts.add ();

  script.name = short_name;
  script.type = type;

  return (&script);
}

/*----------------------------------------------------------*/
void Script::all_print_clean (PrintMode mode)
{
  int number;
  int script_number;
  Use::UsePtrVector& uses = Use::get_ordered_uses ();

  if (uses.size () > 0)
    {
      for (number = 0; number < uses.size (); number++)
        {
          const Use* use = uses[number];

	  Package* p = use->get_package ();
          if (p->is_cmt ()) continue;

          for (script_number = 0;
               script_number < use->scripts.size ();
               script_number++)
            {
              Script& script = use->scripts[script_number];

              if (script.type == CleanupScript)
                {
                  if (script.print (mode, use)) cout << "; ";
                }
            }
        }
    }
}

/*----------------------------------------------------------*/
void Script::all_print (PrintMode mode)
{
  int number;
  int script_number;
  Use::UsePtrVector& uses = Use::get_ordered_uses ();

  if (uses.size () > 0)
    {
      for (number = 0; number < uses.size (); number++)
        {
          const Use* use = uses[number];

	  Package* p = use->get_package ();
          if (p->is_cmt ()) continue;

          for (script_number = 0;
               script_number < use->scripts.size ();
               script_number++)
            {
              Script& script = use->scripts[script_number];

              if (script.type == SetupScript)
                {
                  if (script.print (mode, use)) cout << "; ";
                }
            }
        }
    }

  {
    const Use& use = Use::current ();

    for (script_number = 0;
         script_number < use.scripts.size ();
         script_number++)
      {
        Script& script = use.scripts[script_number];

        if (script.type == SetupScript)
          {
            if (script.print (mode, &use)) cout << "; ";
          }
      }
  }
}

/*----------------------------------------------------------*/
void Script::clear_all ()
{
}

/*----------------------------------------------------------*/
Script::Script ()
{
}

/*----------------------------------------------------------*/
Script::~Script ()
{
}

/*----------------------------------------------------------*/
int Script::print (PrintMode mode, const Use* use)
{
  int result = 1;
  cmt_string suffix;

  if (name == "") return (0);

  switch (mode)
    {
    case Csh :
      suffix = ".csh";
      cout << "source ";
      break;
    case Sh :
      suffix = ".sh";
      cout << ". ";
      break;
    }

  if (name.find (SLASH) == cmt_string::npos)
    {
      cout << "${" << use->prefix << "ROOT}/";
      if (use->style == cmt_style) cout << "cmt/";
      else cout << "mgr/";
    }

  cout << name << suffix;

  return (result);
}
