//-----------------------------------------------------------
// Copyright Christian Arnault LAL-Orsay CNRS
// arnault@lal.in2p3.fr
// See the complete license in cmt_license.txt "http://www.cecill.info". 
//-----------------------------------------------------------

#ifndef __cmt_system_h__
#define __cmt_system_h__

#include "cmt_std.h"
#include "cmt_string.h"
#include "cmt_vector.h"
#include "cmt_regexp.h"

class Project;

class IProjectFactory
{
public:
  virtual void reset () = 0;
  virtual Project* create_project (const cmt_string& name,
				   const cmt_string& path,
				   const cmt_string& path_source,
				   Project* parent) = 0;
};

class CmtSystem
{
public:
  typedef cmt_vector<cmt_string> cmt_string_vector;

  static cmt_string pwd ();
  static bool cd (const cmt_string& dir);

  static void basename (const cmt_string& file_name, cmt_string& result);
  static void basename (const cmt_string& file_name,
                        const cmt_string& suffix,
                        cmt_string& result);

  // Resolve symbolic links in a path.
  static bool realpath (const cmt_string& path, cmt_string& result);
  // Currently the above does not resolve symbolic links in a path.
  static bool realpath_ (const cmt_string& path, cmt_string& result);

  static void dirname (const cmt_string& file_name, cmt_string& result);

  static void name (const cmt_string& file, cmt_string& result);

  static void get_suffix (const cmt_string& file, cmt_string& result);
  static void get_dot_suffix (const cmt_string& file, cmt_string& result);
  static bool has_prefix (const cmt_string& name);
  static bool absolute_path (const cmt_string& name);
  static bool has_device (const cmt_string& name);

  static cmt_string current_branch ();

    /**
     * Implementation of the "test -d" equivalent
     */
  static bool test_directory (const cmt_string& name);
    /**
     * Implementation of the "test -f" equivalent
     */               
  static bool test_file (const cmt_string& name);
    /**
     * Implementation of the "touch" function equivalent
     */               
  static bool touch_file (const cmt_string& name);    
    /**
     * Check if the file "name1" is identical to "name2"
     */
  static bool compare_files (const cmt_string& name1,
                             const cmt_string& name2);
    /**
     * Check if the file "name1" is identical to "name2"
     * if they are identical, "name1" will be simply deleted
     * otherwise "name1" will be copied to "name2" and deleted afterwards
     */
  static bool compare_and_update_files (const cmt_string& name1,
                                        const cmt_string& name2);
  static int file_size (const cmt_string& name);
  static char file_separator ();
  static void reduce_file_separators (cmt_string& text);
  static char path_separator ();
  static char command_separator ();
  static const cmt_string& ev_open ();
  static const cmt_string& ev_close ();
  static bool create_symlink (const cmt_string& oldname,
                              const cmt_string& newname);
  static bool remove_file (const cmt_string& name);
  static bool remove_directory (const cmt_string& name);
  static bool mkdir (const cmt_string& name);
  static void scan_dir (const cmt_string& dir_name,
                        cmt_string_vector& list);
  static void scan_dir (const cmt_string& dir_name,
                        const cmt_regexp& expression,
                        cmt_string_vector& list);
  static cmt_string_vector& scan_dir (const cmt_string& dir_name);
  static const cmt_string& get_cmt_root ();
  static void get_cmt_version (cmt_string& version);
  static cmt_string get_cmt_config ();
  static cmt_string get_cmt_site ();
  static void get_uname (cmt_string& uname);
  static void get_hosttype (cmt_string& hosttype);
  static cmt_string get_temporary_name ();
  
  static cmt_string get_home_package ();
  static bool is_home_package (const cmt_string& name,
                               const cmt_string& version);
  
  static cmt_string get_user_context_package ();
  static bool is_user_context_package (const cmt_string& name,
                                       const cmt_string& version);
  
  static cmt_string get_project_package ();
  static bool is_project_package (const cmt_string& name,
				  const cmt_string& version);
  
  static bool testenv (const cmt_string& name);
  static cmt_string getenv (const cmt_string& name);
  static bool putenv (const cmt_string& name, const cmt_string& value);
  static void get_cmt_paths (IProjectFactory& factory,
                             const cmt_string& init_text,
                             const cmt_string& cmt_user_context,
                             const cmt_string& cmt_home);
  static int execute (const cmt_string& command);
  static int execute (const cmt_string& command, cmt_string& output);
  static bool is_package_directory (const cmt_string& name);
  static bool is_version_directory (const cmt_string& name);
  static bool is_version_directory (const cmt_string& name, int& v, int& r, int& p);
  static void split (const cmt_string& text,
                     const cmt_string& separators,
                     cmt_string_vector& strings);
  static cmt_string quote (const cmt_string& text,
			   const cmt_string& separators);
  static int mangle (const cmt_string& text, cmt_string& out);
  static cmt_string mangle (const cmt_string& text);

  static void compress_path (const cmt_string& dir, cmt_string& new_dir);
  static void compress_path (cmt_string& dir);

  static cmt_string now ();
  static cmt_string user ();

  static void get_cvsroot (cmt_string& cvsroot);

  static bool get_home_directory (cmt_string& dir);

  static cmt_string get_makefile_suffix ();

  static void close_ostream (FILE *stream, const cmt_string& name = "" );
  static void close_stdout (void);

};

class FilePath
{
public:

  FilePath ();

  FilePath (const FilePath& other);
  FilePath (const cmt_string& other);
  FilePath (const char* other);

  FilePath& operator = (const FilePath& other);
  FilePath& operator = (const cmt_string& other);
  FilePath& operator = (const char* other);

  bool operator == (const FilePath& other) const;
  bool operator == (const cmt_string& other) const;
  bool operator == (const char* other) const;

  bool operator != (const FilePath& other) const;
  bool operator != (const cmt_string& other) const;
  bool operator != (const char* other) const;

  const cmt_string& name () const;
  operator const cmt_string& () const;

  bool in (const FilePath& other) const;
  bool in (const cmt_string& other) const;
  bool in (const char* other) const;

  bool cd () const;

private:

  void set (const FilePath& other);
  void set (const cmt_string& other);
  void set (const char* other);

  cmt_string p_name; // physical name
  cmt_string l_name; // first logical name

  CmtSystem::cmt_string_vector alternates; // other logical names
};


#endif
