# -*- coding: iso-8859-1 -*-
########################################
# author: Garonne Vincent              #
# mail: garonne@lal.in2p3.fr           #
# Description: Class and methods to    #
# build a kit                          #
# sources.                             #
# Date: june/29/2005                   #
# based on the christian shell and     #
# python script                        #
########################################

import os
import sys
import string
import popen2
import stat
import re
import time
from   os     import path
import getopt
import tempfile
import signal
import tarfile
import logging
import commands

from   cmt import *

class Kit:
##################################################################################################
#--------------------------- __init__ -----------------------------------------------------------#
##################################################################################################
    def __init__ (self, DEBUG=False):
        # Log creation
        self.debug = DEBUG
        if DEBUG:
            self.log = logging.Logger(name='kit.py', level=logging.DEBUG)
        else:
            self.log = logging.Logger(name='kit.py', level=logging.INFO)
        hdlr = logging.StreamHandler()
        fmt  = logging.Formatter("%(levelname)s:%(filename)s:%(lineno)d:%(message)s")
        hdlr.setFormatter(fmt)
        self.log.addHandler(hdlr)
        
        self.log.info(time.strftime ('%c'))
        # Default options 
        self.here               = os.getcwd()
        self.kits_dir           = self.here +'/kits' 
        self.cache_dir          = self.here +'/cache'
        self.rpm                = False 
        self.minimal            = False
        self.pacman_only        = False
        self.override           = False
        self.do_source_kit      = False
        self.patch_dir          = ''
        self.patch_requirements = ''
        self.platform           = '' 
        self.platform_suffix    = ''
        self.tempprefix         = ''
        self.tempfile1          = ''
        self.tempfile2          = ''
        self.tempcopydir        = ''
        self.tempcmtusercontext = ''
        self.patch_requirements = ''
        self.release            = ''
        self.cyclefile          = ''
        self.url                = ''
        self.pacman_base        = ''

        # CMT setup
        self.cmt        = CMT ()
        self.cmtversion = self.cmt.do ('version')
        self.log.info("cmtversion=%s", self.cmtversion)
 
##################################################################################################
#----------------------------- get_options ------------------------------------------------------#
##################################################################################################
    def get_options (self, args):
        self.log.debug("get_options")
        self.racine = os.path.dirname(os.path.realpath(args[0]))
        try:
            opts, args = getopt.getopt (args[1:] , 'r:c:P:mOp:osu:', 
                                        ['release='  , 'cycles='       , 'patches=', 
                                        'minimal'    , 'pacman_only'   , 'platform=', 
                                        'override'   , 'source'        , 'url='])
        except getopt.GetoptError:
            # print help information and exit:
            self.usage()
            sys.exit(2)

        for o,a in opts:
            if o in ('-r', '--release'):
                self.release = a
            if o in ('-c', '--cycles'):
                self.cyclefile = a
            if o in ('-P', '--patches'):
                self.patch_dir = a
            if o in ('-m', '--minimal'):
                self.minimal = True
            if o in ('-O', '--pacman_only'):
                self.pacman_only = True
            if o in ('-p', '--platform'):
                self.platform = a
            if o in ('-o', '--override'):
                self.override = True
            if o in ('-s', '--source'):
                self.do_source_kit = True
            if o in ('-u', '--url'):
                self.url = a
        for a in args:
            self.pacman_base = os.path.normpath (a)
            self.cache_dir = os.path.join (self.pacman_base, os.path.normpath ('cache'))
            self.kits_dir  = os.path.join (self.pacman_base, os.path.normpath ('kits'))
##################################################################################################
#----------------------------- make_pacman_cache ------------------------------------------------#
##################################################################################################
    def make_pacman_cache (self):
        # Prepare the target pacman cache
        self.log.debug("make_pacman_cache")
        if not os.path.isdir (self.cache_dir):
            self.log.info('Creating %s', self.cache_dir)
            os.makedirs (self.cache_dir)
        if not os.path.isdir (self.kits_dir):
            self.log.info('Creating %s', self.kits_dir)
            os.makedirs (self.kits_dir)
##################################################################################################            
#------------------------------------ make_tempory_file -----------------------------------------#
##################################################################################################
    def make_tempory_file (self):
        self.log.debug("make_tempory_file")  
        # associate handler 
        #signal.signal(signal.SIG_DFL,  self.cleanup); # 0
        #signal.signal(signal.SIG_IGN,  self.cleanup); # 1
        signal.signal(signal.SIGINT,  self.cleanup); # 2
        signal.signal(signal.SIGTERM, self.cleanup); # 15        
        # Prepare temporary file management
        self.log.debug('CMT %s', str(os.getpid ()))
        self.tempprefix = "/tmp/CMT"
        self.log.debug('temprefix=%s', self.tempprefix)
        self.tempfile1          = os.path.join (self.tempprefix, 't')
        self.tempfile2          = os.path.join (self.tempprefix, 'u')
        self.tempcopydir        = os.path.join (self.tempprefix, 'c')
        self.tempcmtusercontext = os.path.join (self.tempprefix, 'd')
        self.cleanup ()
        if self.tempprefix != '':
            os.makedirs (self.tempprefix)
##################################################################################################            
#-------------------------------- prepare_cmt_context -------------------------------------------#
##################################################################################################
    def prepare_cmt_context (self):
        # Prepare the CMT and package context
        self.log.debug("prepare_cmt_context")
        h    = self.cmt.do ('show pwd')
        root = os.path.dirname (h)
        self.cmtversion = self.cmt.do ('version')
        self.version    = self.cmt.macro_value ('version')
        if self.version == 'v*':
            self.version = 'v1'

        self.log.debug('cmtversion=%s', self.cmtversion)
        self.log.debug('version=%s',    self.version)
        
        self.package = self.cmt.macro_value ('package')
        self._normpath(self.package)
        
        self.cmtpath = self.cmt.macro_value (self.package + '_cmtpath')
        self._normpath (self.cmtpath)

        self.offset = self.cmt.macro_value (self.package + '_offset')
        self._normpath (self.offset)

        self.log.debug('package=%s', self.package)
        self.log.debug('cmtpath=%s', self.cmtpath)
        self.log.debug('offset=%s',  self.offset)
      
        self.project = self.cmt.macro_value (self.package + '_project')
        self.log.debug('package_project=%s',  self.project)
        
        self.project_id = os.path.basename (self.project)
        self.log.debug('project_id=%s',  self.project_id)
        
        self.release = self.cmt.macro_value (self.package + '_project_release')
        self.log.debug('package_project_release=%s',  self.release)
        self.log.debug('CMTPATH=%s',  self.cmt.macro_value ('CMTPATH'))
        
        if self.cmtpath == '':
            self.cmtpath = os.path.normpath (self.cmt.macro_value (self.package + '_root'))
            self.cmtpath = os.path.dirname (self.cmtpath)
            self.cmtpath = os.path.dirname (self.cmtpath)

        self.log.info('# Working on %s %s %s in %s project %s % s',\
                       self.package, self.version, self.offset, \
                       self.cmtpath, self.project, self.release)
        self.cleanup ()        
##################################################################################################
#----------------------- prepare_patch_structure ------------------------------------------------#
##################################################################################################
    def prepare_patch_structure (self):
        # Prepare the patch structure
        self.log.debug("prepare_patch_structure")
        if self.patch_dir != '':
            if os.path.isdir (os.path.join (self.patch_dir, self.project_id)):
                self.log.info("create_kit> %s/%s", self.patch_dir, self.project_id)
                os.makedirs (self.tempcmtusercontext)
                os.putenv   ('CMTUSERCONTEXT', self.tempcmtusercontext)
                self.patch_requirements = os.path.join (self.tempcmtusercontext, \
                                                        os.path.normpath ('requirements'))
                
                files = list () 
                files.append (os.path.join (self.patch_dir, self.project_id)+'/requirements')
                files.append (os.path.join (self.patch_dir, self.project_id)+self.package + \
                             '/requirements')
                # concatenate files
                outfile = open(self.patch_requirements, 'w+')
                for file in files:    
                   if os.path.isfile(file):   
                       file=open(file,'r')
                       data=file.read()
                       file.close()
                       outfile.write(data)
                outfile.close()
##################################################################################################
#--------------------------------- prepare_externals --------------------------------------------#
##################################################################################################
    def prepare_externals (self):
        # Prepare the detection of external packages
        self.log.debug("prepare_externals")
        self.native_version = self.cmt.macro_value (self.package + '_native_version')
        self.export_paths   = self.cmt.macro_value (self.package + '_export_paths')
        self.is_internal = self.export_paths == None
        if self.do_source_kit:
            self.is_internal = True
##################################################################################################
#------------------------------------------ prepare_temp ----------------------------------------#
##################################################################################################
    def prepare_temp (self):
        # Prepare the temporary copies
        self.log.debug("prepare_temp")
        os.makedirs (self.tempcopydir)
        
        copycmd = 'ln -s'
        
        if self.do_source_kit:
            self.acquire_sources()
        
##################################################################################################
#------------------------------------------------------------------------------------------------#
##################################################################################################
    def generate_pacman(self):
        # Generate the pacman file 
        self.build_pacman ()
##################################################################################################
#--------------------------------------- build_pacman -------------------------------------------#
##################################################################################################
    def build_pacman (self):
        self.log.debug("build_pacman")
        self.log.debug("%s %s %s %s %s %s", self.url, self.package, self.version, self.cmtpath, \
                       self.project_id, self.is_internal)
        self.source = '../kits'
        # External packages (those with export_paths) are split into two kits:
        # o The internal part, which follows the naming convention for internal packages
        # o The external part, which only refers to the package version (no mention of
        #   the project)
        # Naming cnvention:
        #  external packages : <package>-<version>-<platform>
        #  internal packages : <projectid>/<package>-<platform>

        if not self.is_internal:
            # first build the pacman file for the external stuff
            if self.native_version == None:
                cmd        = "echo "+self.version+" | sed -e 's#'^"+self.package+"-'##'"
                status, vv = os.getstatusoutput(cmd)
            else:
                vv = self.native_version
                self.download_filename="${package}-${vv}${platform_suffix}"
            self.download_filename = self.package + '-' + vv + self.platform_suffix

            self.pacman_filename = self.download_filename + '.pacman'
            
            self.log.info ('build_pacman> Create pacman file for external stuff %s', self.pacman_filename)
            file = os.path.join (self.cache_dir, self.pacman_filename)
            self.pacman_file_version = self.get_pacman_file_version (file)
            if os.path.isfile(file):
                os.rename(file, file + ".bak")
            # if test ! $? = 0; then
            #   print 'create_kit> failed to rename ' + cache_dir + '/' + pacman_filename + ' to ' + cache_dir + '/' + pacman_filename' + '.bak'
            #sys.exit(2)
            
        # write pacman file
        self.log.debug ('write pacman file')
        f = open (file, 'w+')
        
        content = '''
        description ('External Package %s %s %s %s')

        source = '%s'
        download = { '*':'%s.tar.gz' }        
        '''%(self.package, self.version, self.platform, file, self.source, self.download_filename)
        f.write (content)
        f.close()
        self.log.debug ('close file')
        
    # Figure out dependencies from CMT.
    # Only take directly used packages, not those used indirectly;
    # let pacman handle the cascade.
    # Filter the requirements file patch if any to get the direct use statements.
    # The cmt show uses does NOT show the use statements obtained from the CMTUSERCONTEXT thus
    # they have to be added manually here.
    #      
        cmd = '''
        `(if test ! "%s" = ""; then grep use %s | sed -e 's/use /# use /'; fi; cmt show uses) | cat | awk -f %s/get_depends.awk`'''%(self.patch_requirements, self.patch_requirements, self.racine)
        status, self.depsv = commands.getstatusoutput(cmd)
        
        if os.path.isfile(self.cyclefile):
            self.new_depsv= self.filter_deps(self.depsv)
            if self.new_depsv == self.depsv:
              print "Filtered dependencies: "
              print "depsv", self.depsv
              print "new_depsv", self.new_depsv
              self.depsv=self.new_depsv
        
        if self.project_id == "":
            self.release_id=self.release
            cmd = "`echo "+self.version+" | sed -e 's#'^"+self.package+"-'##'`"
            status, vv= commands.getstatusoutput(cmd)
        else:
            self.release_id=self.project_id
            vv=self.project_id

        # Format the depend statement according to Pacman syntax
        if self.depsv == "":
            self.depends=""
        else:
            self.project_dep=os.system("`echo "+self.project+self.platform_suffix+" | sed -e 's#/#-#'`")
    
            if self.project_dep== "contrib-CMT"+self.platform_suffix:
                self.project_dep = "CMTCONFIG-"+self.cmtversion+self.platform_suffix
   
            self.depends="package ('"+str(self.project_dep)+"')"
            first=True
            for f in self.depsv:
                p=os.system  ("`echo "+f+" | cut -d: -f1`")
                v=os.system  ("`echo "+f+" | cut -d: -f2`")
                pp=os.system ("`echo "+f+" | cut -d: -f3 | sed -e 's#/$##'`")

        # end build_pacman        
##################################################################################################
#------------------------------- usage ----------------------------------------------------------#
##################################################################################################
    def usage (self):
        print "Make a distribution kit for a CMT package"
        print "Usage: create_kit.sh [ <option> ... ] <pacman-cache>"
        print " where option can be:"
        print "  -r|--release <release-id>  : specify the release id"
        print "  -c|--cycles <cycles-file>  : specify the cycles file"
        print "  -P|--patches <patch-dir>   : specify a directory for patches "
        print "  -m|--minimal               : only store CMT files into the tar ball"
        print "  -O|--pacman_only           : do not generate the tar ball"
        print "  -p|--platform <platform>   : specify a platform suffix"
        print "  -o|--override              : override the existing tar balls"
        print "  -s|--source                : generate the source kit"
        print "Typical/example usage:"
        print "  cmt broadcast create_kit.sh -release 6.3.0 -cycles ${DEP}/cycles.txt /tmp "
        # end usage
##################################################################################################
#--------------------------------------- _normpath ----------------------------------------------#
##################################################################################################
    def _normpath (self, path):
        if path != '':
            path = os.path.normpath (path)   
##################################################################################################
#-------------------------------------- __del__ -------------------------------------------------#
##################################################################################################
    def __del__ (self):
        self.log.debug("__del__")
        self.cleanup()      
##################################################################################################    
#----------------------------- cleanup ----------------------------------------------------------#
##################################################################################################
    def cleanup (self):
        self.log.info('Cleanup')
        self.cmt.removedirs (self.tempprefix)
##################################################################################################
#------------------------------ get_project -----------------------------------------------------#
##################################################################################################
    def get_project (self):
        self.log.debug("get_project")
        if self.cmtpath != '':
            rr = os.path.basename (self.cmtpath)
            pr = os.path.dirname (self.cmtpath)
            pr = os.path.basename (pr)
            return os.path.join (pr, rr)
        else:
            self.log.error('CMTPATH empty')
            sys.exit(-1)
##################################################################################################
#------------------------------------------------------------------------------------------------#
##################################################################################################
    def get_libraries (self):
        self.log.debug("get_libraries")
        pass                
##################################################################################################
#---------------------------- get_pacman_file_version -------------------------------------------#
##################################################################################################        
    def get_pacman_file_version(self, old_file):
        self.log.debug("get_pacman_file_version")
        pacman_file_version=1
        if os.path.isfile(old_file):
            cmd = "grep description "+old_file+" | egrep '[(]v' | sed -e 's#.*(v##' -e 's#).*##'"
            status, pacman_file_version = commands.getstatusoutput (cmd)
            
        if pacman_file_version == "":
            pacman_file_version=1
        else:
            pacman_file_version=int(pacman_file_version) + 1

        return "v"+str(pacman_file_version)  
##################################################################################################
#---------------------------------------- filter_deps -------------------------------------------#
##################################################################################################
    def filter_deps (self, deps):
        #  filter_deps <cyclefile> <p> <p> ...
        #  Remove from the list of <p>s all entries found in the cyclefile
        self.log.debug("filter_deps")
        cmd = "`cat "+self.cyclefile+" | awk '{if ($NF == \"'"+self.package+"'\") {print "+deps[0]+"}}'`"
        status, cycles = commands.getstatusoutput (cmd)
 
        for f in deps: 
            found = 0
            cmd   = "`echo "+f+" | cut -d: -f1`"
            status, p = commands.getstatusoutput (cmd)
 
        for c in cycles:
            if c == p:
                found=1
        if found:
            return f
##################################################################################################
#---------------------------------------- acquire_sources ---------------------------------------#
##################################################################################################
    def acquire_sources (self):
        self.log.debug("acquire_sources")

        self.log.info("Now acquiring sources")
        f      = open ('../CVS/Repository','r')
        module = f.read() 
        f.close()
        
        self.log.debug("version=%s"    , self.version)
        self.log.debug("package=%s"    , self.package)
        self.log.debug("offset=%s"     , self.offset)
        self.log.debug("project=%s"    , self.project)
        self.log.debug("kits_dir=%s"   , self.kits_dir)
        self.log.debug("tempcopydir=%s", self.tempcopydir)
        
        self.project_id   = os.path.basename (self.project)
        self.project_name = os.path.dirname  (self.project)

        os.removedirs (self.tempcopydir)
        os.makedirs   (self.tempcopydir+"/"+self.project+"/"+self.offset)

        self.log.info("Now exporting the sources from CVS")
        os.chdir(self.tempcopydir+"/"+self.project)
        cmd = "cvs export -r "+self.version+" -d "+self.offset+self.package+" "+self.module
        self.log.debug(str(os.listdir(self.tempcopydir)))

        self.log.info("Now constructing the source tar ball")
        os.makedirs (self.kits_dir+"/"+self.project_id)
        os.chdir(self.tempcopydir)
        
        name = self.kits_dir+"/"+self.project_id+"/"+self.package+"-src.tar.gz" 
        self.log.debug("tar ball name=%s" , name)
        tarball = tarfile.open (name=name , mode='w:gz')
        tarball.add            (name='.')
        tarball.close          ()

        if self.debug:
            tarball = tarfile.open (name=name , mode='r:gz')
            self.log.debug('tar ball content:%s', str(tarball.list ()))
            tarball.close ()
##################################################################################################
#---------------------------------- acquire_CMT_files -------------------------------------------#
##################################################################################################            
    def acquire_CMT_files (self):
        #  Acquire the CMT files (requirements + make_fragments) file
        #  Result is stored (symlinked) in ${tempcopydir}/cmt
        self.log.debug ("acquire_CMT_files")
        pass 
#  echo `pwd` >${tempfile1}
#  if test -d ../fragments; then
#    up=`(cd ..; pwd)`
#    echo ${up}/fragments >>${tempfile1}
#  fi
#
#  if test -f version.cmt; then
#    versiondir=""
#  else
#    versiondir="/${version}"
#  fi
#
#  tempcmtbase=${project}/${offset}${package}${versiondir}
#
#  macro_value ${package}_CMT_files >>${tempfile1}
#
#  old_copycmd=${copycmd}
#  copycmd="cp -r"
#
#  transfer_files "CMT files" ${tempfile1} ${tempcmtbase}
#
#  copycmd=${old_copycmd}
#
#  chmod -R +w ${tempcopydir}/${tempcmtbase}/cmt
#  (cd ${tempcopydir}/${tempcmtbase}/cmt;
#   eval "/bin/rm -f setup.* cleanup.* *make* Makefile")
#
#  if test ! "${patch_requirements}" = ""; then
#    echo "create_kit> Patching requirements file with ${patch_requirements}"
#    cat requirements ${patch_requirements} >|${tempcopydir}/${tempcmtbase}/cmt/requirements
#  fi
#}
##################################################################################################
#---------------------------------- transfer_files ----------------------------------------------#
##################################################################################################
    def transfer_files (self, kind, liste, branch):
        self.log.debug ("transfer_files")
        alt       = ""
        dir_prefix= ""

        if os.path.isfile(liste): return 
#        
#  cmt -quiet filter ${list} ${tempfile2}; mv ${tempfile2} ${list}
#  sort -u ${list} >${tempfile2}; mv ${tempfile2} ${list}
#
#  n=`wc -w ${list} | awk '{print $1}'`
#  if test "${n}" = 0; then
#    /bin/rm -f ${list}
#    return
#  fi
#
#  echo "create_kit> Now acquiring ${kind}"
#
#  
#  for f in `cat ${list} | sort -u`; do
#
#    alt=`echo ${f} | egrep '^[-]s='`
#
#    if test ! "${alt}" = ""; then
#      alt=
#      dir_prefix=`echo ${f} | sed -e 's#^[-]s=##'`
#    else
#
#      test_absolute=`echo ${f} | egrep '^/'`
#
#      if test ! $? = 0; then
#        if test ! "${dir_prefix}" = ""; then
#          f=${dir_prefix}/${f}
#        fi
#      fi
#
#      ##echo "1) $f"
#
#      n=`(set -f; basename ${f})`
#      d=`(set -f; dirname ${f})`
#
#      if test -d ${d}; then
#        d=`(cd ${d}; pwd)`
#        ff=${d}/${n}
#
#        #echo "2) $ff ${tempcopydir}/${branch} d=${d}"
#
#        mkdir -p ${tempcopydir}/${branch}
#
#        is_dir=0
#        if test "`(set -f; echo ${ff}) | grep '[*]'`" = ""; then
#          if test -d `(set -f; echo ${ff})`; then
#            is_dir=1
#          fi
#        fi
#
#        ##------------------------------------
#        ## We compare ${d} and ${branch}
#        ## is ${branch} strictly included in ${d} ?
#        ## if yes, compute the offset to be added to ${branch}
#        ##
#
#        off=
#
#        aaa=`echo ${d} | grep -e ${branch}`
#
#        if test $? = 0; then
#          # echo "There may be an offset d=${d} branch=${branch}"
#          off=`echo "${d}" | sed -e 's#.*'"${branch}"'##' -e 's#^[/]##g'`
#          if test ! "${off}" = ""; then
#            off="/${off}"
#            mkdir -p ${tempcopydir}/${branch}${off}
#          fi
#          ##echo "off=${off}"
#        fi
#
#        ##echo "3) is_dir=$is_dir"
#
#        if test ${is_dir} = 1; then
#          ##echo "  > directory ${ff}"
#
#          ##echo "4) copycmd=[$copycmd]"
#
#          ${copycmd} ${ff} ${tempcopydir}/${branch}${off}/${n}
#        else
#          a=`eval "ls ${ff} 2>/dev/null"`
#
#          ##echo "4) a=[$a]"
#
#          for g in `echo ${a}`; do
#            nn=`basename ${g}`
#            ##echo "  > file ${g} branch=${branch}${off}"
#            if test ! -e ${tempcopydir}/${branch}${off}/${nn}; then
#              ${copycmd} ${g} ${tempcopydir}/${branch}${off}/${nn}
#	    fi
#          done
#        fi
#      else
#        echo "create_kit> Warning: Directory ${d} not found for file ${f}"
#      fi
#    fi
#  done
#
#  /bin/rm -f ${list}
##################################################################################################
#---------------------------------- acquire_java ------------------------------------------------#
##################################################################################################
    def acquire_java (self):
        self.log.debug ("acquire_java")
        pass
##################################################################################################
#---------------------------------- acquire_applications ----------------------------------------#
##################################################################################################    
    def acquire_applications (self):
        self.log.debug ("acquire_applications")
        pass
##################################################################################################
#---------------------------------- acquire_runtime_files ---------------------------------------#
##################################################################################################    
    def acquire_runtime_files (self):
        self.log.debug ("acquire_runtime_files")
        pass
##################################################################################################
#---------------------------------- acquire_jo_files --------------------------------------------#
##################################################################################################        
    def acquire_jo_files (self):
        self.log.debug ("acquire_jo_files")
        pass
##################################################################################################
#---------------------------------- acquire_python_modules_files --------------------------------#
##################################################################################################        
    def acquire_python_modules_files (self):
        self.log.debug ("acquire_python_modules_files")
        pass
##################################################################################################
#---------------------------------- acquire_xml_files -------------------------------------------#
##################################################################################################        
    def acquire_xml_files (self):
        self.log.debug ("acquire_xml_files")
        pass
##################################################################################################
#---------------------------------- acquire_headers ---------------------------------------------#
##################################################################################################        
    def acquire_headers (self):
        self.log.debug ("acquire_headers")
        pass
##################################################################################################
#---------------------------------- acquire_scripts_files ---------------------------------------#
##################################################################################################        
    def acquire_scripts_files (self):
        self.log.debug ("acquire_scripts_files")
        pass
##################################################################################################
#---------------------------------- acquire_lcg_modules -----------------------------------------#
##################################################################################################        
    def acquire_lcg_modules (self):
        self.log.debug ("acquire_lcg_modules")
        pass
##################################################################################################
#---------------------------------- acquire_export_installed_files ------------------------------#
##################################################################################################        
    def acquire_export_installed_files (self):
        self.log.debug ("acquire_export_installed_files")
        pass  
##################################################################################################
#---------------------------------- create_tar --------------------------------------------------#
##################################################################################################        
    def create_tar (self):
        self.log.debug ("create_ta")
        pass
##################################################################################################
#---------------------------------- acquire_export_path -----------------------------------------#
##################################################################################################        
    def acquire_export_path (self):
        self.log.debug ("acquire_export_path")
        pass
##################################################################################################
#---------------------------------- acquire_export_path -----------------------------------------#
##################################################################################################        
    def make_tar_filename (self):
        self.log.debug ("make_tar_filename")
        if self.project_id=="":
            if self.native_version=="":
                return self.package+"-"+self.version+self.platform_suffix
                #| sed -e 's#'${package}-${package}'#'${package}'#'
            else:
                print self.package+"-"+self.native_version+self.platform_suffix
        else:
            return self.project_id+"/"+self.package+self.platform_suffix
##################################################################################################
#----------------------------------      END       ----------------------------------------------#
##################################################################################################