//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4ASTARStopping.cc,v 1.8 2008/11/24 18:28:09 vnivanch Exp $
// GEANT4 tag $Name: geant4-09-03 $

//---------------------------------------------------------------------------
//
// GEANT4 Class file
//
// Description: Data on stopping power
//
// Author:      Anton Ivantchenko 21.04.2006
//
// Organisation:        QinetiQ Ltd, UK
// Customer:            ESA/ESTEC, NOORDWIJK
// Contract:            CSMAN-5288
//
// Modifications:
// 
//----------------------------------------------------------------------------
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4ASTARStopping.hh" 
#include "G4Material.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ASTARStopping::G4ASTARStopping()
{
  currentE = 0.0;
  currentMaterial = 0;
  index = 0;
  matIndex = -1;
  Initialise();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ASTARStopping::~G4ASTARStopping()
{}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4int G4ASTARStopping:: GetIndex (const G4Material* mat)
{  
  if (mat == currentMaterial) return matIndex;
  G4String matName = mat->GetName();
  for (G4int i=0; i<74; i++){
    if (matName == name[i]){
      matIndex = i;
      currentMaterial = mat;
      return i;
    }
  }
  return -1;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4ASTARStopping::GetElectronicDEDX(G4int i, G4double energy)
{
  if (matIndex == i && energy == currentE) return res;       

  if (i<0 || i>73){
    G4cout << "### G4ASTARStopping WARNING: index " 
	   << i << " is out of range!" << G4endl; 
    res = 0.0;
    return res;
  }
  matIndex = i;
  currentE = energy;

  if (energy < kinE[index] || energy > kinE[index+1]){

    if (energy <= kinE[0]) {
      index = 0;
      res = e[matIndex][0]*std::sqrt(energy/kinE[0]);
      return res;
    } else if (energy >= kinE[77]) {
      index = 76;
      res = e[matIndex][77];
      return res;
    }
    for (index = 0; index<77; index++){if (energy <= kinE[index+1]) break;}
  }

  G4double t1 = kinE[index];
  G4double t2 = kinE[index+1];
  G4double e1 = e[matIndex][index];
  G4double e2 = e[matIndex][index+1];
  res = e1*std::exp(std::log(energy/t1)*std::log(e2/e1)/std::log(t2/t1));
  return res;
 }
 
void G4ASTARStopping::Initialise()
{

G4int i; 
G4double Z[74] = {
  6.0,  6.0,  6.0, 47.0,  7.0, 13.0, 13.0, 18.0, 79.0, 6.0,
  4.0,  6.0,  6.0,  6.0,  6.0,  6.0,  6.0, 20.0, 40.0, 6.0,
  6.0, 54.0, 29.0, 26.0, 26.0, 64.0, 32.0, 14.0,  1.0, 2.0,
  6.0, 36.0,  5.0,  6.0,  9.0,  6.0,  6.0, 42.0,  6.0, 6.0,
  6.0,  6.0,  7.0, 55.0, 10.0,  6.0,  8.0,  6.0, 82.0, 6.0,
  6.0,  6.0,  6.0,  6.0,  6.0,  6.0,  6.0,  6.0,  6.0, 6.0,
 78.0, 14.0, 14.0,  6.0, 22.0, 50.0,  6.0,  6.0,  6.0,92.0,
 74.0,  8.0,  8.0, 54.0 
};

 for(i=0; i<74; i++) {effZ[i]=Z[i];}
 
name [0] = "G4_A-150_TISSUE";  
G4double T0[78] = { 0.001, 0.0015, 0.002, 0.0025, 0.003, 0.004, 0.005, 0.006, 0.007, 0.008, 0.009, 0.01, 0.0125, 0.015, 0.0175, 0.02, 0.0225, 0.025, 0.0275, 0.03, 0.035, 0.04, 0.045, 0.05, 0.055, 0.06, 0.065, 0.07, 0.075, 0.08, 0.085, 0.09, 0.095, 0.1, 0.125, 0.15, 0.175, 0.2, 0.225, 0.25, 0.275, 0.3, 0.35, 0.4, 0.45, 0.5, 0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 1, 1.25, 1.5, 1.75, 2, 2.25, 2.5, 2.75, 3, 3.5, 4, 4.5, 5, 5.5, 6, 6.5, 7, 7.5, 8, 8.5, 9, 9.5, 10 }; 
 
G4double e0[78] = { 168.8, 204.2, 233.8, 259.7, 283, 324.3, 360.4, 393, 422.8, 450.5, 476.5, 501, 557.1, 607.7, 654.1, 697.1, 737.4, 775.4, 811.5, 845.9, 910.3, 970, 1026, 1078, 1128, 1175, 1220, 1263, 1305, 1344, 1383, 1419, 1455, 1489, 1646, 1782, 1901, 2005, 2097, 2178, 2249, 2312, 2413, 2488, 2539, 2572, 2589, 2593, 2586, 2571, 2548, 2521, 2489, 2453, 2416, 2377, 2176, 1989, 1825, 1683, 1561, 1457, 1369, 1292, 1165, 1062, 978.5, 908.4, 848.8, 797.4, 752.4, 712.9, 677.7, 646.1, 617.7, 591.9, 568.4, 547 }; 
 
 for(i=0; i<78; i++) {e[0][i]=e0[i]*MeV*cm2/g;}
 for(i=0; i<78; i++) {kinE[i]=T0[i];}
 
name [1] = "G4_ACETYLENE"; 
G4double e1[78] = { 192.2, 229.4, 260.2, 287, 311, 353.1, 389.7, 422.5, 452.4, 480.1, 506, 530.3, 585.9, 635.7, 681.2, 723.2, 762.5, 799.4, 834.3, 867.5, 929.6, 986.9, 1040, 1090, 1137, 1182, 1224, 1265, 1304, 1341, 1376, 1411, 1444, 1475, 1619, 1741, 1847, 1938, 2017, 2085, 2144, 2195, 2275, 2331, 2367, 2388, 2397, 2395, 2386, 2370, 2350, 2326, 2299, 2271, 2241, 2210, 2056, 1913, 1783, 1666, 1562, 1468, 1384, 1308, 1177, 1072, 986.8, 915.5, 855, 802.8, 757.2, 717.1, 681.4, 649.5, 620.7, 594.7, 570.9, 549.2 }; 
 
 for(i=0; i<78; i++) {e[1][i]=e1[i]*MeV*cm2/g;}
 
name [2] = "G4_ADIPOSE_TISSUE_ICRP"; 
G4double e2[78] = { 169.4, 204.2, 233.3, 258.8, 281.8, 322.4, 358.1, 390.2, 419.7, 447.2, 472.9, 497.2, 552.9, 603.2, 649.4, 692.4, 732.7, 770.7, 806.8, 841.3, 906, 966.1, 1022, 1075, 1125, 1173, 1219, 1262, 1304, 1344, 1383, 1420, 1456, 1491, 1650, 1787, 1907, 2012, 2104, 2185, 2256, 2317, 2417, 2489, 2539, 2571, 2588, 2592, 2587, 2573, 2554, 2529, 2500, 2468, 2434, 2398, 2213, 2035, 1874, 1731, 1605, 1499, 1408, 1328, 1197, 1091, 1004, 932.2, 870.7, 817.8, 771.5, 730.8, 694.5, 662.1, 632.9, 606.4, 582.3, 560.2 }; 
 
 for(i=0; i<78; i++) {e[2][i]=e2[i]*MeV*cm2/g;}
 
name [3] = "G4_Ag"; 
G4double e3[78] = { 31.26, 38.13, 43.91, 48.98, 53.56, 61.66, 68.79, 75.21, 81.11, 86.6, 91.74, 96.6, 107.7, 117.8, 127, 135.6, 143.6, 151.2, 158.4, 165.3, 178.2, 190.2, 201.4, 212, 222, 231.6, 240.7, 249.5, 257.9, 266, 273.9, 281.4, 288.8, 295.9, 328.8, 357.8, 383.8, 407.4, 428.8, 448.4, 466.3, 482.8, 511.8, 536.2, 556.6, 573.5, 587.3, 598.4, 607.2, 613.8, 618.5, 621.6, 623.4, 623.8, 623.3, 621.8, 604.4, 578.7, 550.8, 523.3, 497.6, 474.4, 453.6, 435, 402.7, 375.9, 353.2, 333.6, 316.9, 302, 288.8, 276.9, 266.2, 256.5, 247.7, 239.5, 232, 225.1 }; 
 
 for(i=0; i<78; i++) {e[3][i]=e3[i]*MeV*cm2/g;}
 
name [4] = "G4_AIR"; 
G4double e4[78] = { 87.5, 108.6, 126.7, 142.7, 157.3, 183.5, 206.7, 227.9, 247.5, 265.9, 283.2, 299.6, 337.7, 372.3, 404.3, 434.3, 462.6, 489.4, 515, 539.5, 585.8, 629, 669.6, 708.1, 744.8, 779.8, 813.4, 845.7, 876.8, 906.8, 935.9, 964, 991.3, 1018, 1140, 1247, 1343, 1429, 1506, 1575, 1637, 1693, 1787, 1861, 1918, 1961, 1990, 2008, 2017, 2019, 2013, 2002, 1987, 1968, 1946, 1922, 1774, 1625, 1494, 1382, 1287, 1205, 1133, 1072, 968.6, 885.9, 818, 760.7, 711.7, 669.6, 632.7, 600.2, 571.2, 545.3, 521.9, 500.6, 481.2, 463.4 }; 
 
 for(i=0; i<78; i++) {e[4][i]=e4[i]*MeV*cm2/g;}
 
name [5] = "G4_Al"; 
G4double e5[78] = { 55.8, 71.89, 86.05, 98.92, 110.9, 132.7, 152.5, 170.9, 188.2, 204.5, 220.1, 235.1, 270.1, 302.5, 332.8, 361.5, 388.6, 414.6, 439.4, 463.3, 508.4, 550.4, 589.6, 626.5, 661.1, 693.7, 724.3, 753.2, 780.3, 805.9, 830.1, 852.8, 874.3, 894.5, 980, 1045, 1096, 1136, 1168, 1195, 1216, 1234, 1262, 1280, 1291, 1297, 1298, 1297, 1293, 1286, 1279, 1269, 1259, 1248, 1237, 1225, 1161, 1098, 1039, 984.9, 937.6, 895.4, 856.8, 821, 756.1, 698.6, 647.8, 604.8, 567.8, 535.7, 507.4, 482.4, 460, 439.8, 421.6, 405.1, 389.9, 376 }; 
 
 for(i=0; i<78; i++) {e[5][i]=e5[i]*MeV*cm2/g;}
 
name [6] = "G4_ALUMINUM_OXIDE"; 
G4double e6[78] = { 71.39, 88.59, 103.3, 116.4, 128.4, 149.9, 169, 186.5, 202.8, 218, 232.3, 246, 277.7, 306.6, 333.4, 358.5, 382.1, 404.6, 426, 446.5, 485, 520.9, 554.3, 585.8, 615.5, 643.5, 670, 695.2, 719.1, 741.9, 763.6, 784.3, 804, 822.9, 905.9, 974, 1031, 1080, 1122, 1159, 1191, 1220, 1267, 1304, 1332, 1353, 1368, 1378, 1383, 1385, 1384, 1381, 1376, 1370, 1362, 1352, 1308, 1253, 1187, 1115, 1044, 982.5, 928.4, 880, 801.2, 737, 683.4, 638.1, 599.1, 565.2, 535.4, 508.9, 485.3, 464, 444.8, 427.3, 411.4, 396.7 }; 
 
 for(i=0; i<78; i++) {e[6][i]=e6[i]*MeV*cm2/g;}
 
name [7] = "G4_Ar"; 
G4double e7[78] = { 34.54, 44.56, 53.39, 61.42, 68.88, 82.53, 94.95, 106.5, 117.3, 127.6, 137.4, 146.8, 168.8, 189.3, 208.5, 226.7, 244.1, 260.8, 276.8, 292.3, 321.9, 349.8, 376.5, 401.9, 426.4, 450, 472.8, 494.9, 516.3, 537.1, 557.3, 577, 596.1, 614.8, 702.1, 780.3, 850.8, 914.4, 971.9, 1024, 1070, 1111, 1181, 1233, 1272, 1298, 1314, 1321, 1321, 1315, 1305, 1291, 1276, 1258, 1240, 1222, 1130, 1047, 974, 911.2, 856.7, 809, 767, 729.8, 666.1, 613.2, 569.7, 532.7, 501, 473.4, 449.1, 427.4, 408.3, 391, 375.4, 361.1, 348.1, 336 }; 
 
 for(i=0; i<78; i++) {e[7][i]=e7[i]*MeV*cm2/g;}
 
name [8] = "G4_Au"; 
G4double e8[78] = { 11.5, 14.55, 17.2, 19.57, 21.76, 25.72, 29.27, 32.54, 35.59, 38.46, 41.18, 43.78, 49.83, 55.39, 60.56, 65.44, 70.06, 74.46, 78.68, 82.74, 90.45, 97.69, 104.5, 111.1, 117.3, 123.3, 129.1, 134.6, 140, 145.2, 150.3, 155.2, 160, 164.7, 186.4, 205.8, 223.5, 239.5, 254.2, 267.7, 280.1, 291.5, 311.6, 328.5, 342.6, 354.3, 363.8, 371.5, 377.5, 382.1, 385.5, 387.8, 389.1, 389.6, 389.4, 388.7, 384.4, 375.7, 362.6, 346.8, 330.5, 316.4, 304, 293, 274.1, 258.4, 245, 233.4, 223.1, 214, 205.8, 198.4, 191.6, 185.5, 179.7, 174.5, 169.6, 165 }; 
 
 for(i=0; i<78; i++) {e[8][i]=e8[i]*MeV*cm2/g;}
 
name [9] = "G4_B-100_BONE"; 
G4double e9[78] = { 138.6, 167.7, 192, 213.3, 232.4, 266.3, 296, 322.7, 347.2, 369.9, 391.2, 411.3, 457.3, 498.8, 536.8, 572, 605, 636.1, 665.7, 693.8, 746.5, 795.4, 841, 883.9, 924.5, 963.1, 999.9, 1035, 1069, 1101, 1133, 1163, 1192, 1220, 1348, 1460, 1557, 1643, 1719, 1786, 1845, 1897, 1983, 2047, 2093, 2124, 2142, 2149, 2148, 2139, 2125, 2106, 2084, 2059, 2031, 2003, 1861, 1726, 1600, 1484, 1379, 1290, 1214, 1147, 1035, 945.3, 871.8, 810, 757.7, 712.5, 672.9, 638.1, 607, 579.2, 554.1, 531.3, 510.5, 491.5 }; 
 
 for(i=0; i<78; i++) {e[9][i]=e9[i]*MeV*cm2/g;}
 
name [10] = "G4_Be"; 
G4double e10[78] = { 146.3, 177.1, 202.9, 225.4, 245.6, 281.3, 312.5, 340.6, 366.2, 390, 412.2, 433.2, 481.1, 524.1, 563.3, 599.7, 633.6, 665.4, 695.6, 724.2, 777.7, 827, 872.7, 915.5, 955.7, 993.7, 1030, 1064, 1096, 1128, 1157, 1186, 1213, 1239, 1356, 1453, 1534, 1602, 1659, 1706, 1745, 1776, 1821, 1846, 1856, 1855, 1845, 1829, 1809, 1785, 1759, 1732, 1703, 1674, 1645, 1617, 1499, 1406, 1326, 1254, 1189, 1129, 1075, 1024, 935.3, 858.7, 792.9, 737.6, 690, 648.7, 612.5, 580.4, 552, 526.4, 503.3, 482.4, 463.3, 445.8 }; 
 
 for(i=0; i<78; i++) {e[10][i]=e10[i]*MeV*cm2/g;}
 
name [11] = "G4_BONE_COMPACT_ICRU"; 
G4double e11[78] = { 125.9, 152.6, 175, 194.8, 212.5, 244.1, 271.8, 296.8, 319.8, 341.1, 361.2, 380.2, 423.8, 463.1, 499.3, 533, 564.6, 594.5, 622.8, 649.9, 700.8, 748.1, 792.4, 834.2, 873.7, 911.4, 947.4, 981.9, 1015, 1047, 1078, 1107, 1136, 1164, 1291, 1401, 1498, 1584, 1660, 1727, 1787, 1839, 1926, 1991, 2039, 2072, 2092, 2103, 2105, 2100, 2089, 2075, 2056, 2035, 2011, 1986, 1846, 1705, 1576, 1460, 1358, 1271, 1196, 1130, 1020, 931.6, 859.4, 798.8, 747.4, 702.9, 664, 629.7, 599.1, 571.7, 547.1, 524.7, 504.3, 485.6 }; 
 
 for(i=0; i<78; i++) {e[11][i]=e11[i]*MeV*cm2/g;}
 
name [12] = "G4_C"; 
G4double e12[78] = { 190.6, 225.6, 254.3, 279, 300.9, 339.2, 372.1, 401.4, 427.9, 452.3, 475, 496.2, 544.3, 586.9, 625.5, 661, 693.8, 724.6, 753.5, 780.8, 831.5, 877.9, 920.8, 960.6, 998, 1033, 1066, 1098, 1128, 1156, 1184, 1210, 1235, 1259, 1366, 1455, 1531, 1595, 1650, 1698, 1738, 1772, 1825, 1862, 1886, 1900, 1906, 1906, 1901, 1892, 1880, 1866, 1850, 1833, 1814, 1795, 1684, 1571, 1466, 1372, 1288, 1213, 1145, 1085, 981.9, 898.2, 829.6, 772, 722.9, 680.4, 643.1, 610.2, 580.8, 554.4, 530.6, 509, 489.2, 471.2 }; 
 
 for(i=0; i<78; i++) {e[12][i]=e12[i]*MeV*cm2/g;}
 
name [13] = "G4_C_Graphite"; 
G4double e13[78] = { 192.3, 228.9, 259, 285.1, 308.3, 348.9, 384, 415.3, 443.7, 469.9, 494.2, 517, 568.9, 615.1, 657, 695.5, 731.3, 764.8, 796.3, 826.2, 881.8, 932.7, 979.8, 1024, 1065, 1104, 1141, 1175, 1209, 1240, 1271, 1300, 1327, 1354, 1473, 1574, 1658, 1731, 1793, 1845, 1890, 1929, 1987, 2027, 2051, 2063, 2065, 2060, 2049, 2034, 2015, 1993, 1969, 1944, 1917, 1891, 1751, 1620, 1502, 1400, 1310, 1231, 1161, 1099, 994.3, 909, 839.2, 780.7, 730.8, 687.6, 649.8, 616.4, 586.6, 559.8, 535.7, 513.8, 493.8, 475.5 }; 
 
 for(i=0; i<78; i++) {e[13][i]=e13[i]*MeV*cm2/g;}
 
name [14] = "G4_ETHYLENE"; 
G4double e14[78] = { 159.8, 195.8, 226, 252.7, 276.8, 319.7, 357.4, 391.5, 422.9, 452.1, 479.5, 505.4, 565.1, 619, 668.6, 714.8, 758.1, 799.1, 838.1, 875.3, 945.4, 1011, 1072, 1130, 1185, 1237, 1288, 1336, 1383, 1428, 1471, 1514, 1555, 1594, 1779, 1945, 2094, 2230, 2354, 2468, 2571, 2666, 2830, 2961, 3060, 3130, 3172, 3189, 3185, 3164, 3129, 3086, 3038, 2986, 2931, 2876, 2578, 2306, 2084, 1907, 1763, 1642, 1539, 1450, 1303, 1185, 1089, 1010, 941.9, 883.6, 832.8, 788.1, 748.4, 712.9, 680.9, 652, 625.7, 601.7 }; 
 
 for(i=0; i<78; i++) {e[14][i]=e14[i]*MeV*cm2/g;}
 
name [15] = "G4_C-552"; 
G4double e15[78] = { 124.1, 148.8, 169.3, 187.2, 203.3, 231.5, 256.1, 278.2, 298.4, 317.1, 334.5, 351, 388.6, 422.4, 453.2, 481.8, 508.4, 533.5, 557.3, 579.9, 622.2, 661.3, 697.8, 732, 764.3, 794.9, 824, 851.9, 878.5, 904.1, 928.7, 952.4, 975.2, 997.3, 1097, 1184, 1259, 1325, 1384, 1435, 1481, 1521, 1588, 1639, 1678, 1706, 1726, 1738, 1744, 1745, 1743, 1736, 1727, 1716, 1703, 1688, 1603, 1511, 1421, 1338, 1261, 1190, 1126, 1067, 965.5, 883.8, 816.4, 759.6, 711.1, 669.1, 632.3, 599.9, 571, 545, 521.6, 500.3, 480.9, 463.1 }; 
 
 for(i=0; i<78; i++) {e[15][i]=e15[i]*MeV*cm2/g;}
 
name [16] = "G4_CARBON_DIOXIDE"; 
G4double e16[78] = { 93.28, 114.2, 131.9, 147.5, 161.6, 186.6, 208.6, 228.5, 246.8, 263.8, 279.8, 295, 329.8, 361.2, 390.2, 417.1, 442.4, 466.4, 489.1, 510.8, 551.8, 589.8, 625.6, 659.4, 691.5, 722.2, 751.6, 779.8, 807.1, 833.4, 858.9, 883.7, 907.7, 931.1, 1040, 1137, 1225, 1306, 1380, 1449, 1513, 1571, 1676, 1763, 1834, 1891, 1933, 1961, 1977, 1982, 1978, 1967, 1952, 1932, 1911, 1888, 1752, 1614, 1490, 1383, 1290, 1209, 1139, 1077, 973.3, 890, 821.5, 764, 714.9, 672.6, 635.5, 602.6, 573.5, 547.3, 523.7, 502.4, 482.9, 465 }; 
 
 for(i=0; i<78; i++) {e[16][i]=e16[i]*MeV*cm2/g;}
 
name [17] = "G4_CALCIUM_FLUORIDE"; 
G4double e17[78] = { 72.78, 90.74, 106.1, 119.8, 132.3, 154.7, 174.6, 192.8, 209.7, 225.5, 240.4, 254.5, 287.2, 317, 344.6, 370.4, 394.6, 417.6, 439.5, 460.5, 499.9, 536.5, 570.8, 603.1, 633.6, 662.6, 690.1, 716.4, 741.5, 765.6, 788.6, 810.8, 832.1, 852.6, 944.4, 1022, 1088, 1145, 1194, 1237, 1275, 1308, 1361, 1401, 1431, 1453, 1467, 1476, 1481, 1481, 1479, 1474, 1466, 1458, 1447, 1436, 1352, 1256, 1166, 1086, 1016, 954.4, 900.2, 852.3, 772.1, 708.1, 655.5, 610.7, 573.3, 540.6, 511.9, 486.6, 464, 443.8, 425.4, 408.7, 393.4, 379.4 }; 
 
 for(i=0; i<78; i++) {e[17][i]=e17[i]*MeV*cm2/g;}
 
name [18] = "G4_CERIC_SULFATE"; 
G4double e18[78] = { 99.73, 123.7, 144.1, 162.3, 178.8, 208.3, 234.6, 258.5, 280.5, 301.2, 320.6, 339.1, 381.8, 420.6, 456.5, 490, 521.6, 551.5, 580.1, 607.4, 658.9, 707, 752.2, 795, 835.7, 874.6, 911.8, 947.6, 982, 1015, 1047, 1078, 1109, 1138, 1273, 1391, 1497, 1591, 1676, 1753, 1822, 1884, 1990, 2074, 2140, 2190, 2227, 2251, 2266, 2273, 2272, 2266, 2254, 2238, 2220, 2198, 2049, 1886, 1737, 1607, 1495, 1399, 1315, 1242, 1120, 1023, 942.8, 875.6, 818.4, 769.2, 726.1, 688, 654.2, 624, 596.7, 572, 549.5, 528.9 }; 
 
 for(i=0; i<78; i++) {e[18][i]=e18[i]*MeV*cm2/g;}
 
name [19] = "G4_CELLULOSE_NITRATE"; 
G4double e19[78] = { 157.6, 186.4, 209.9, 230.3, 248.3, 279.7, 306.8, 330.8, 352.6, 372.6, 391.2, 408.7, 448.1, 483.2, 514.9, 544.1, 571.1, 596.4, 620.3, 642.8, 684.8, 723.2, 758.8, 792, 823.2, 852.7, 880.6, 907.2, 932.6, 956.9, 980.2, 1003, 1024, 1045, 1139, 1220, 1290, 1353, 1408, 1457, 1502, 1541, 1608, 1662, 1704, 1736, 1761, 1778, 1789, 1796, 1798, 1796, 1790, 1782, 1772, 1760, 1674, 1574, 1476, 1385, 1302, 1227, 1160, 1099, 993.8, 909.3, 839.7, 781.1, 731.2, 688.1, 650.2, 616.7, 587, 560.3, 536.2, 514.4, 494.5, 476.2 }; 
 
 for(i=0; i<78; i++) {e[19][i]=e19[i]*MeV*cm2/g;}
 
name [20] = "G4_BONE_CORTICAL_ICRP"; 
G4double e20[78] = { 109.1, 132.8, 152.8, 170.4, 186.2, 214.4, 239.3, 261.7, 282.4, 301.6, 319.7, 336.8, 376.1, 411.7, 444.5, 475, 503.6, 530.7, 556.5, 581.1, 627.4, 670.4, 710.8, 748.9, 785, 819.4, 852.4, 883.9, 914.3, 943.5, 971.7, 999, 1025, 1051, 1168, 1270, 1360, 1440, 1511, 1574, 1630, 1680, 1763, 1826, 1872, 1905, 1927, 1939, 1943, 1941, 1933, 1921, 1905, 1887, 1866, 1844, 1720, 1593, 1475, 1369, 1274, 1193, 1123, 1062, 959.2, 877.1, 809.7, 753, 704.9, 663.4, 627, 594.9, 566.3, 540.6, 517.5, 496.5, 477.4, 459.8 }; 
 
 for(i=0; i<78; i++) {e[20][i]=e20[i]*MeV*cm2/g;}
 
name [21] = "G4_CESIUM_IODIDE"; 
G4double e21[78] = { 40.6, 48.54, 55.16, 60.95, 66.15, 75.33, 83.38, 90.64, 97.29, 103.5, 109.3, 114.8, 127.3, 138.7, 149.2, 158.9, 168, 176.7, 184.9, 192.8, 207.6, 221.3, 234.2, 246.4, 258, 269.1, 279.7, 289.9, 299.7, 309.2, 318.3, 327.1, 335.7, 344, 382.3, 415.8, 445.6, 472.1, 495.7, 516.7, 535.4, 551.8, 578.9, 599.2, 613.8, 623.8, 629.9, 633, 633.6, 632.3, 629.5, 625.5, 620.6, 615.1, 609.1, 602.8, 567.7, 533, 501.8, 474.4, 450.3, 429, 410.1, 393.3, 364.6, 340.9, 321, 304, 289.2, 275.9, 264.1, 253.6, 243.9, 235, 226.7, 218.9, 211.9, 205.4 }; 
 
 for(i=0; i<78; i++) {e[21][i]=e21[i]*MeV*cm2/g;}
 
name [22] = "G4_Cu"; 
G4double e22[78] = { 29.51, 36.49, 42.42, 47.68, 52.45, 60.98, 68.54, 75.4, 81.74, 87.65, 93.23, 98.51, 110.7, 121.8, 132, 141.5, 150.5, 159, 167.2, 174.9, 189.5, 203.2, 216, 228.1, 239.7, 250.7, 261.3, 271.4, 281.2, 290.7, 299.8, 308.7, 317.3, 325.6, 364.2, 398.4, 429.1, 456.8, 482.1, 505.1, 526.2, 545.4, 579, 607, 630.1, 649.1, 664.4, 676.6, 686.2, 693.4, 698.8, 702.4, 704.6, 705.7, 705.7, 704.9, 692.5, 672.6, 650.3, 627.6, 606, 585.6, 566.3, 548, 514, 483.4, 455.8, 431, 408.4, 388, 369.6, 353.7, 339.1, 325.7, 313.4, 302, 291.7, 282.3 }; 
 
 for(i=0; i<78; i++) {e[22][i]=e22[i]*MeV*cm2/g;}
 
name [23] = "G4_Fe"; 
G4double e23[78] = { 54.06, 65.42, 74.91, 83.2, 90.66, 103.8, 115.3, 125.6, 135.1, 143.8, 152, 159.8, 177.4, 193.3, 207.8, 221.3, 233.9, 245.7, 256.9, 267.6, 287.7, 306.2, 323.5, 339.8, 355.2, 369.8, 383.8, 397.1, 410, 422.3, 434.2, 445.7, 456.9, 467.6, 517.1, 560.6, 599.4, 634.3, 665.9, 694.6, 720.8, 744.8, 786.6, 821.5, 850.4, 874.1, 893.1, 908.1, 919.6, 928, 933.6, 936.8, 938, 937.3, 935.1, 931.5, 899.9, 856.7, 810.9, 766.9, 727.3, 691.9, 660.2, 631.5, 581.8, 540.1, 504.6, 473.9, 447.2, 423.6, 402.6, 383.5, 366.9, 352, 338.3, 325.7, 314.1, 303.6 }; 
 
 for(i=0; i<78; i++) {e[23][i]=e23[i]*MeV*cm2/g;}
 
name [24] = "G4_FERROUS_SULFATE"; 
G4double e24[78] = { 99.58, 123.5, 144, 162.1, 178.6, 208.1, 234.3, 258.2, 280.2, 300.8, 320.3, 338.7, 381.4, 420.2, 456, 489.5, 521, 551, 579.5, 606.8, 658.3, 706.3, 751.5, 794.2, 834.9, 873.7, 910.9, 946.6, 981, 1014, 1046, 1077, 1108, 1137, 1271, 1390, 1495, 1590, 1675, 1751, 1820, 1882, 1988, 2072, 2138, 2188, 2225, 2250, 2265, 2271, 2271, 2264, 2253, 2237, 2218, 2197, 2048, 1886, 1738, 1609, 1497, 1401, 1317, 1244, 1122, 1025, 944.6, 877.4, 820, 770.7, 727.5, 689.3, 655.4, 625.2, 597.8, 573.1, 550.5, 529.9 }; 
 
 for(i=0; i<78; i++) {e[24][i]=e24[i]*MeV*cm2/g;}
 
name [25] = "G4_Gd"; 
G4double e25[78] = { 38.11, 45.4, 51.41, 56.61, 61.25, 69.34, 76.36, 82.61, 88.29, 93.53, 98.41, 103, 113.4, 122.7, 131.1, 138.9, 146.1, 152.9, 159.3, 165.4, 176.7, 187.1, 196.8, 205.9, 214.4, 222.5, 230.3, 237.6, 244.7, 251.5, 258, 264.3, 270.3, 276.2, 303.1, 326.6, 347.4, 366.2, 383.1, 398.5, 412.5, 425.4, 447.8, 466.5, 482.1, 494.8, 505.1, 513.2, 519.4, 523.9, 526.9, 528.6, 529.1, 528.6, 527.3, 525.2, 505.6, 481, 456.8, 434.6, 414.6, 396.6, 380.5, 366, 340.9, 319.7, 301.6, 285.8, 272, 259.8, 248.8, 239, 230.1, 222, 214.6, 207.7, 201.4, 195.6 }; 
 
 for(i=0; i<78; i++) {e[25][i]=e25[i]*MeV*cm2/g;}
 
name [26] = "G4_Ge"; 
G4double e26[78] = { 47.65, 57.57, 65.83, 73.05, 79.53, 90.94, 100.9, 109.9, 118, 125.6, 132.7, 139.4, 154.6, 168.3, 180.8, 192.3, 203.1, 213.3, 222.9, 232, 249.1, 264.8, 279.5, 293.2, 306.1, 318.4, 330, 341.1, 351.7, 361.8, 371.5, 380.8, 389.8, 398.4, 437.2, 469.9, 497.8, 521.7, 542.3, 560.1, 575.5, 588.9, 610.7, 627.4, 640.1, 649.9, 657.3, 662.8, 666.8, 669.6, 671.4, 672.3, 672.4, 671.9, 670.9, 669.4, 656.9, 639.5, 619.8, 599.4, 578.4, 557.6, 537.5, 518.5, 483.6, 452.9, 425.8, 401.8, 380.5, 361.5, 344.3, 328.6, 315.1, 303.1, 292.1, 282, 272.6, 263.8 }; 
 
 for(i=0; i<78; i++) {e[26][i]=e26[i]*MeV*cm2/g;}
 
name [27] = "G4_Pyrex_Glass"; 
G4double e27[78] = { 83.85, 102.1, 117.5, 131, 143.3, 165, 184.1, 201.3, 217.2, 232, 245.9, 259, 289.2, 316.5, 341.5, 364.8, 386.7, 407.4, 427.1, 445.8, 481.1, 513.8, 544.4, 573.2, 600.6, 626.6, 651.5, 675.3, 698.1, 720.1, 741.3, 761.7, 781.5, 800.6, 888, 963.9, 1031, 1090, 1142, 1189, 1231, 1268, 1330, 1378, 1414, 1441, 1460, 1473, 1481, 1485, 1484, 1481, 1475, 1466, 1456, 1445, 1389, 1323, 1249, 1171, 1095, 1029, 971.4, 920.7, 835.9, 768, 711.5, 663.8, 622.8, 587.4, 556.1, 528.4, 503.7, 481.5, 461.4, 443.2, 426.5, 411.2 }; 
 
 for(i=0; i<78; i++) {e[27][i]=e27[i]*MeV*cm2/g;}
 
name [28] = "G4_H"; 
G4double e28[78] = { 212, 275.4, 331.7, 383, 430.9, 518.8, 599.2, 674, 744.5, 811.5, 875.6, 937.1, 1082, 1217, 1344, 1465, 1580, 1691, 1798, 1901, 2098, 2285, 2463, 2634, 2798, 2956, 3108, 3256, 3399, 3538, 3674, 3805, 3933, 4058, 4637, 5152, 5610, 6019, 6383, 6705, 6988, 7235, 7632, 7915, 8100, 8203, 8240, 8222, 8161, 8067, 7947, 7808, 7655, 7494, 7327, 7157, 6345, 5647, 5068, 4588, 4188, 3860, 3585, 3351, 2971, 2675, 2438, 2242, 2078, 1939, 1818, 1713, 1620, 1538, 1464, 1398, 1337, 1283 }; 
 
 for(i=0; i<78; i++) {e[28][i]=e28[i]*MeV*cm2/g;}
 
name [29] = "G4_HELIUM"; 
G4double e29[78] = { 87.26, 110.8, 131.3, 149.8, 166.8, 197.7, 225.5, 251.1, 275, 297.5, 318.9, 339.4, 387, 430.9, 471.8, 510.4, 547, 581.9, 615.3, 647.5, 708.7, 766.1, 820.5, 872.3, 921.8, 969.3, 1015, 1059, 1102, 1143, 1183, 1221, 1259, 1296, 1465, 1614, 1748, 1867, 1975, 2071, 2156, 2233, 2361, 2460, 2535, 2588, 2624, 2644, 2652, 2650, 2639, 2621, 2598, 2571, 2540, 2507, 2323, 2139, 1976, 1832, 1706, 1593, 1493, 1402, 1249, 1130, 1033, 953.8, 886.8, 829.5, 779.8, 736.4, 697.9, 663.7, 632.9, 605.2, 580, 557 }; 
 
 for(i=0; i<78; i++) {e[29][i]=e29[i]*MeV*cm2/g;}
 
name [30] = "G4_KAPTON"; 
G4double e30[78] = { 165.5, 197.6, 224.2, 247.2, 267.8, 304, 335.4, 363.5, 389.1, 412.8, 434.8, 455.6, 502.9, 545.2, 583.8, 619.4, 652.6, 683.8, 713.3, 741.2, 793.5, 841.6, 886.3, 928.2, 967.7, 1005, 1041, 1074, 1107, 1138, 1167, 1196, 1223, 1250, 1369, 1471, 1560, 1636, 1703, 1762, 1812, 1857, 1928, 1981, 2018, 2042, 2056, 2062, 2061, 2054, 2043, 2028, 2010, 1990, 1969, 1946, 1819, 1688, 1565, 1451, 1350, 1264, 1190, 1125, 1017, 929.8, 858.3, 798.2, 747, 702.8, 664, 629.8, 599.3, 571.9, 547.2, 524.8, 504.4, 485.7 }; 
 
 for(i=0; i<78; i++) {e[30][i]=e30[i]*MeV*cm2/g;}
 
name [31] = "G4_Kr"; 
G4double e31[78] = { 10.15, 13.69, 16.93, 19.96, 22.84, 28.23, 33.29, 38.08, 42.66, 47.08, 51.35, 55.5, 65.43, 74.84, 83.85, 92.52, 100.9, 109.1, 117, 124.7, 139.7, 154.1, 168, 181.5, 194.6, 207.4, 219.9, 232.1, 244, 255.7, 267.2, 278.4, 289.5, 300.3, 351.9, 399.6, 443.7, 484.4, 522.1, 556.7, 588.4, 617.2, 666.6, 705.7, 735.5, 757, 771.4, 779.9, 783.4, 783, 779.4, 773.6, 765.9, 757, 747.2, 736.9, 683.7, 635.6, 595.2, 561.5, 533, 508.6, 487.3, 468.5, 439.9, 417.9, 398.8, 381.4, 364.9, 349.2, 334.2, 319.8, 306.2, 293.7, 282.8, 272.8, 263.6, 255.2 }; 
 
 for(i=0; i<78; i++) {e[31][i]=e31[i]*MeV*cm2/g;}
 
name [32] = "G4_LITHIUM_TETRABORATE"; 
G4double e32[78] = { 132.7, 159.4, 181.6, 200.9, 218.2, 248.6, 275, 298.7, 320.4, 340.4, 359.1, 376.6, 416.8, 452.7, 485.5, 515.8, 544, 570.6, 595.8, 619.7, 664.3, 705.6, 743.9, 779.9, 813.9, 846.1, 876.8, 906.1, 934.2, 961.1, 987, 1012, 1036, 1059, 1166, 1258, 1338, 1410, 1474, 1531, 1583, 1628, 1706, 1767, 1814, 1849, 1874, 1891, 1899, 1902, 1899, 1891, 1875, 1852, 1825, 1795, 1638, 1494, 1375, 1276, 1193, 1121, 1058, 1002, 908.8, 833.2, 770.5, 717.6, 672.4, 633.2, 598.8, 568.3, 541.2, 516.9, 494.8, 474.9, 456.6, 439.9 }; 
 
 for(i=0; i<78; i++) {e[32][i]=e32[i]*MeV*cm2/g;}
 
name [33] = "G4_LITHIUM_FLUORIDE"; 
G4double e33[78] = { 118.7, 142.6, 162.5, 179.7, 195.2, 222.4, 246, 267.2, 286.5, 304.3, 321, 336.6, 372.4, 404.4, 433.5, 460.4, 485.5, 509.1, 531.4, 552.6, 592.1, 628.5, 662.4, 694.1, 724.1, 752.4, 779.4, 805.1, 829.6, 853.2, 875.9, 897.6, 918.7, 938.9, 1031, 1110, 1179, 1240, 1294, 1341, 1383, 1421, 1483, 1530, 1566, 1592, 1610, 1621, 1627, 1627, 1624, 1618, 1609, 1598, 1585, 1571, 1483, 1387, 1296, 1214, 1140, 1074, 1015, 962.2, 872, 799.7, 739.5, 688.5, 644.8, 606.9, 573.7, 544.4, 518.3, 494.9, 473.6, 454.3, 436.7, 420.6 }; 
 
 for(i=0; i<78; i++) {e[33][i]=e33[i]*MeV*cm2/g;}
 
name [34] = "G4_M3_WAX"; 
G4double e34[78] = { 173.6, 209, 238.5, 264.3, 287.6, 328.7, 364.7, 397.1, 426.9, 454.5, 480.4, 504.9, 561, 611.6, 658, 701.1, 741.6, 779.8, 816, 850.6, 915.4, 975.6, 1032, 1085, 1135, 1182, 1228, 1271, 1313, 1353, 1392, 1429, 1465, 1499, 1657, 1792, 1910, 2012, 2101, 2179, 2247, 2305, 2398, 2463, 2507, 2533, 2544, 2544, 2534, 2517, 2494, 2468, 2437, 2405, 2371, 2335, 2154, 1981, 1825, 1686, 1564, 1460, 1372, 1295, 1167, 1064, 980.4, 910.2, 850.4, 798.9, 753.8, 714.2, 678.9, 647.3, 618.8, 593, 569.4, 547.9 }; 
 
 for(i=0; i<78; i++) {e[34][i]=e34[i]*MeV*cm2/g;}
 
name [35] = "G4_MS20_TISSUE"; 
G4double e35[78] = { 163.5, 196.6, 224.1, 248.2, 269.8, 308, 341.4, 371.4, 398.9, 424.4, 448.3, 470.9, 522.5, 569, 611.6, 651.1, 688, 722.9, 756, 787.5, 846.6, 901.2, 952.3, 1000, 1046, 1089, 1130, 1169, 1207, 1243, 1278, 1311, 1344, 1375, 1516, 1638, 1744, 1836, 1916, 1986, 2048, 2101, 2185, 2247, 2288, 2314, 2327, 2330, 2325, 2312, 2295, 2273, 2249, 2222, 2193, 2163, 2002, 1845, 1703, 1575, 1463, 1368, 1286, 1215, 1097, 1001, 923.2, 857.7, 802, 753.9, 711.7, 674.6, 641.6, 611.9, 585.2, 561, 539, 518.8 }; 
 
 for(i=0; i<78; i++) {e[35][i]=e35[i]*MeV*cm2/g;}
 
name [36] = "G4_METHANE"; 
G4double e36[78] = { 197.5, 241.8, 279.3, 312.2, 342, 394.9, 441.5, 483.7, 522.4, 558.5, 592.4, 624.4, 698.1, 764.8, 826, 883.1, 936.6, 987.3, 1035, 1081, 1168, 1249, 1324, 1396, 1464, 1529, 1591, 1650, 1708, 1764, 1818, 1870, 1920, 1970, 2198, 2402, 2586, 2754, 2907, 3047, 3175, 3291, 3492, 3652, 3773, 3856, 3905, 3921, 3899, 3849, 3783, 3708, 3627, 3544, 3460, 3378, 2998, 2685, 2432, 2226, 2055, 1911, 1789, 1683, 1508, 1370, 1257, 1163, 1084, 1015, 955.9, 903.7, 857.5, 816.2, 779.1, 745.5, 714.9, 687.1 }; 
 
 for(i=0; i<78; i++) {e[36][i]=e36[i]*MeV*cm2/g;}
 
name [37] = "G4_Mo"; 
G4double e37[78] = { 58.22, 68.98, 77.79, 85.4, 92.16, 103.9, 114.1, 123.1, 131.3, 138.8, 145.8, 152.4, 167.3, 180.5, 192.5, 203.5, 213.8, 223.4, 232.4, 241, 256.9, 271.6, 285.1, 297.8, 309.8, 321.1, 331.8, 342.1, 351.9, 361.3, 370.3, 379, 387.4, 395.5, 432.4, 464.5, 492.9, 518.2, 541, 561.6, 580.4, 597.4, 626.9, 651.2, 671.2, 687.3, 700.1, 710, 717.3, 722.3, 725.4, 726.7, 726.5, 725, 722.3, 718.8, 690.9, 655, 617.8, 582.6, 552.1, 525.7, 502.3, 481.5, 445.5, 415.5, 389.9, 367.7, 348.1, 330.6, 314.8, 300.9, 288.6, 277.3, 266.9, 257.6, 249, 241.2 }; 
 
 for(i=0; i<78; i++) {e[37][i]=e37[i]*MeV*cm2/g;}
 
name [38] = "G4_MUSCLE_WITH_SUCROSE"; 
G4double e38[78] = { 111.7, 137.5, 159.4, 178.8, 196.5, 227.9, 255.7, 281, 304.4, 326.2, 346.7, 366.2, 411.2, 452, 489.7, 525, 558.1, 589.5, 619.5, 648.1, 702.2, 752.5, 799.9, 844.6, 887.2, 927.9, 966.8, 1004, 1040, 1075, 1108, 1141, 1172, 1203, 1343, 1467, 1576, 1674, 1762, 1841, 1912, 1975, 2081, 2165, 2230, 2277, 2310, 2331, 2342, 2344, 2339, 2329, 2313, 2293, 2270, 2244, 2074, 1896, 1739, 1607, 1494, 1398, 1314, 1241, 1120, 1022, 942.4, 875.3, 818.1, 768.9, 725.8, 687.7, 653.9, 623.7, 596.4, 571.7, 549.2, 528.6 }; 
 
 for(i=0; i<78; i++) {e[38][i]=e38[i]*MeV*cm2/g;}
 
name [39] = "G4_MUSCLE_WITHOUT_SUCROSE"; 
G4double e39[78] = { 108.2, 133.5, 155, 174.1, 191.4, 222.4, 249.8, 274.8, 297.8, 319.3, 339.6, 358.9, 403.3, 443.7, 481, 515.9, 548.7, 579.8, 609.5, 637.9, 691.4, 741.4, 788.3, 832.7, 875, 915.3, 953.9, 991.1, 1027, 1061, 1095, 1127, 1158, 1189, 1328, 1451, 1560, 1658, 1745, 1824, 1895, 1958, 2065, 2150, 2215, 2263, 2298, 2320, 2332, 2335, 2331, 2321, 2306, 2287, 2265, 2240, 2075, 1901, 1746, 1613, 1501, 1404, 1320, 1247, 1125, 1027, 946.2, 878.8, 821.4, 772, 728.6, 690.4, 656.4, 626.1, 598.7, 573.9, 551.3, 530.6 }; 
 
 for(i=0; i<78; i++) {e[39][i]=e39[i]*MeV*cm2/g;}
 
name [40] = "G4_MUSCLE_SKELETAL_ICRP"; 
G4double e40[78] = { 104.5, 129.2, 150.2, 168.8, 185.8, 216.1, 242.9, 267.4, 290, 311.1, 331, 349.8, 393.5, 433.1, 469.7, 503.9, 536.2, 566.7, 595.9, 623.7, 676.3, 725.4, 771.5, 815.1, 856.6, 896.2, 934.1, 970.6, 1006, 1040, 1072, 1104, 1135, 1165, 1302, 1422, 1530, 1626, 1712, 1790, 1860, 1922, 2029, 2113, 2178, 2228, 2263, 2286, 2299, 2304, 2302, 2293, 2280, 2262, 2242, 2218, 2059, 1888, 1735, 1604, 1492, 1396, 1312, 1240, 1118, 1021, 941.1, 874.1, 817, 767.9, 724.9, 686.8, 653.1, 622.9, 595.7, 571, 548.6, 528 }; 
 
 for(i=0; i<78; i++) {e[40][i]=e40[i]*MeV*cm2/g;}
 
name [41] = "G4_MUSCLE_STRIATED_ICRU"; 
G4double e41[78] = { 104.7, 129.4, 150.5, 169.2, 186.2, 216.5, 243.4, 267.9, 290.5, 311.7, 331.6, 350.5, 394.2, 433.9, 470.6, 504.9, 537.2, 567.8, 596.9, 624.9, 677.5, 726.7, 772.8, 816.5, 858.1, 897.8, 935.8, 972.3, 1007, 1041, 1074, 1106, 1137, 1167, 1304, 1425, 1532, 1629, 1715, 1793, 1863, 1925, 2032, 2116, 2181, 2230, 2265, 2289, 2302, 2306, 2304, 2295, 2282, 2264, 2244, 2220, 2063, 1894, 1742, 1611, 1498, 1401, 1318, 1244, 1123, 1025, 944.5, 877.2, 819.9, 770.6, 727.3, 689.2, 655.3, 625, 597.7, 572.9, 550.4, 529.7 }; 
 
 for(i=0; i<78; i++) {e[41][i]=e41[i]*MeV*cm2/g;}
 
name [42] = "G4_N"; 
G4double e42[78] = { 82.8, 103.7, 121.6, 137.7, 152.3, 178.7, 202.3, 223.8, 243.8, 262.5, 280.2, 297.1, 336.2, 372, 405.1, 436.2, 465.5, 493.4, 520.1, 545.7, 594.1, 639.3, 682, 722.5, 761, 797.9, 833.4, 867.4, 900.3, 932, 962.8, 992.5, 1021, 1049, 1179, 1293, 1395, 1486, 1567, 1640, 1706, 1765, 1864, 1940, 1999, 2041, 2069, 2086, 2093, 2091, 2083, 2069, 2051, 2030, 2005, 1978, 1819, 1660, 1522, 1406, 1308, 1224, 1151, 1088, 982.9, 898.5, 829.5, 771.1, 721.3, 678.5, 641, 608, 578.6, 552.3, 528.5, 506.9, 487.2, 469.1 }; 
 
 for(i=0; i<78; i++) {e[42][i]=e42[i]*MeV*cm2/g;}
 
name [43] = "G4_SODIUM_IODIDE"; 
G4double e43[78] = { 35.86, 44.19, 51.24, 57.48, 63.13, 73.21, 82.12, 90.2, 97.64, 104.6, 111.1, 117.3, 131.6, 144.5, 156.4, 167.6, 178, 187.9, 197.3, 206.3, 223.3, 239.1, 253.9, 268, 281.3, 294, 306.2, 317.9, 329.2, 340.1, 350.6, 360.8, 370.6, 380.2, 424.5, 463.5, 498.4, 529.9, 558.2, 583.8, 607, 627.8, 663.4, 691.6, 713.4, 729.6, 740.9, 748.1, 751.7, 752.4, 750.7, 747, 741.7, 735.1, 727.6, 719.5, 664.4, 611.2, 569.4, 538.3, 514, 491.6, 472.1, 454, 422.9, 393.7, 370.2, 349.6, 331.4, 315.3, 301.2, 288.5, 276.8, 266.1, 256.3, 247.1, 238.9, 231.5 }; 
 
 for(i=0; i<78; i++) {e[43][i]=e43[i]*MeV*cm2/g;}
 
name [44] = "G4_Ne"; 
G4double e44[78] = { 68.73, 83.7, 96.26, 107.3, 117.2, 134.8, 150.3, 164.2, 177, 188.8, 199.9, 210.4, 234.5, 256.2, 276.1, 294.6, 311.9, 328.3, 343.8, 358.6, 386.3, 412, 436.1, 458.8, 480.3, 500.8, 520.3, 539.1, 557.1, 574.5, 591.2, 607.5, 623.2, 638.4, 708.4, 770.3, 825.6, 875.7, 921.2, 962.8, 1001, 1036, 1098, 1150, 1194, 1230, 1260, 1285, 1304, 1319, 1330, 1338, 1343, 1344, 1344, 1341, 1296, 1229, 1160, 1095, 1035, 981.4, 932.6, 888.5, 812.1, 748.3, 695.6, 650.4, 611.3, 577, 546.8, 520, 496, 474.4, 454.8, 437, 420.7, 405.7 }; 
 
 for(i=0; i<78; i++) {e[44][i]=e44[i]*MeV*cm2/g;}
 
name [45] = "G4_NYLON-6/6"; 
G4double e45[78] = { 167.4, 201.6, 230.3, 255.4, 278, 317.9, 352.9, 384.4, 413.4, 440.3, 465.5, 489.3, 544, 593.3, 638.6, 680.6, 720.1, 757.3, 792.7, 826.4, 889.7, 948.5, 1003, 1055, 1104, 1151, 1195, 1238, 1279, 1318, 1356, 1392, 1427, 1461, 1617, 1750, 1867, 1969, 2059, 2138, 2207, 2267, 2363, 2433, 2482, 2512, 2527, 2531, 2525, 2511, 2491, 2467, 2439, 2408, 2375, 2340, 2160, 1987, 1830, 1690, 1567, 1463, 1374, 1297, 1169, 1066, 981.3, 910.8, 850.9, 799.3, 754.1, 714.4, 679, 647.4, 618.9, 593, 569.5, 547.9 }; 
 
 for(i=0; i<78; i++) {e[45][i]=e45[i]*MeV*cm2/g;}
 
name [46] = "G4_O"; 
G4double e46[78] = { 105.8, 117.3, 257.6, 1.139e-05, 5.976e-06, 0.005, 248.1, 71.83, 352, 3.186e-05, 2.012e-05, 0.015, 412.7, 40.96, 503.1, 6.782e-05, 5.098e-05, 0.035, 610.6, 24.07, 700.7, 0.0001156, 9.613e-05, 0.07, 820, 15.89, 884.2, 0.0001583, 0.0001354, 0.125, 1124, 8.719, 1280, 0.0002775, 0.0002641, 0.3, 1573, 4.478, 1697, 0.0004528, 0.000448, 0.65, 1818, 2.655, 1822, 0.0006468, 0.0006403, 1, 1667, 1.474, 1431, 0.00138, 0.001539, 2.75, 1038, 0.7076, 860.9, 0.003906, 0.004519, 6, 616.1, 0.3845, 556.8, 0.009419, 0.01034, 9.5, 452, 0.2295, 334.7, 0.03447, 0.04325, 27.5, 196.3, 0.09039, 156.5, 0.1784, 0.2149, 60 }; 
 
 for(i=0; i<78; i++) {e[46][i]=e46[i]*MeV*cm2/g;}
 
name [47] = "G4_PARAFFIN"; 
G4double e47[78] = { 193.8, 233, 265.8, 294.4, 320.3, 365.9, 405.9, 441.9, 475, 505.7, 534.5, 561.7, 624.2, 680.6, 732.3, 780.4, 825.5, 868.2, 908.6, 947.2, 1020, 1087, 1150, 1209, 1265, 1319, 1370, 1419, 1465, 1510, 1554, 1595, 1636, 1675, 1852, 2004, 2137, 2253, 2354, 2442, 2518, 2584, 2688, 2762, 2810, 2837, 2848, 2845, 2831, 2810, 2782, 2749, 2713, 2674, 2633, 2592, 2381, 2184, 2007, 1851, 1714, 1598, 1500, 1414, 1272, 1158, 1066, 988.4, 922.7, 866.1, 816.7, 773.2, 734.5, 699.9, 668.8, 640.6, 614.9, 591.4 }; 
 
 for(i=0; i<78; i++) {e[47][i]=e47[i]*MeV*cm2/g;}
 
name [48] = "G4_Pb"; 
G4double e48[78] = { 17.96, 22.18, 25.76, 28.92, 31.8, 36.93, 41.47, 45.6, 49.4, 52.95, 56.3, 59.46, 66.77, 73.41, 79.52, 85.23, 90.6, 95.69, 100.5, 105.2, 113.9, 122, 129.7, 136.9, 143.8, 150.3, 156.6, 162.7, 168.5, 174.1, 179.5, 184.8, 189.9, 194.9, 217.7, 237.9, 256, 272.3, 287, 300.4, 312.6, 323.6, 342.6, 358.2, 370.7, 380.6, 388.3, 394.2, 398.4, 401.3, 403.1, 404, 404.1, 403.5, 402.4, 401, 392.5, 380.4, 365.3, 348.6, 332, 317.6, 304.9, 293.7, 274.4, 258.3, 244.7, 232.9, 222.5, 213.2, 205, 197.5, 190.7, 184.4, 178.7, 173.4, 168.4, 163.9 }; 
 
 for(i=0; i<78; i++) {e[48][i]=e48[i]*MeV*cm2/g;}
 
name [49] = "G4_PHOTOGRAPHIC_EMULSION"; 
G4double e49[78] = { 46.69, 57.18, 66.06, 73.91, 81.04, 93.75, 105, 115.2, 124.7, 133.5, 141.8, 149.6, 167.8, 184.3, 199.6, 213.8, 227.3, 240, 252.1, 263.8, 285.7, 306.2, 325.4, 343.6, 361, 377.5, 393.4, 408.7, 423.3, 437.5, 451.2, 464.5, 477.3, 489.8, 547, 597.3, 641.7, 681.2, 716.3, 747.6, 775.5, 800.2, 841.5, 873.5, 897.8, 915.8, 928.6, 937.1, 940, 938.4, 933.8, 927, 918.6, 909.2, 899.1, 888.5, 833.7, 781.8, 735.1, 693.7, 657.1, 624.6, 595.5, 569.4, 524.5, 487.2, 455.6, 428.5, 404.9, 383.9, 365.2, 348.6, 333.8, 320.5, 308.3, 297.2, 287, 277.8 }; 
 
 for(i=0; i<78; i++) {e[49][i]=e49[i]*MeV*cm2/g;}
 
name [50] = "G4_PLASTIC_SC_VINYLTOLUENE"; 
G4double e50[78] = { 192.4, 229.8, 260.8, 287.8, 312, 354.4, 391.4, 424.6, 454.8, 482.8, 509, 533.7, 590, 640.5, 686.6, 729.3, 769.2, 806.7, 842.2, 876, 939.2, 997.5, 1052, 1103, 1151, 1197, 1240, 1281, 1321, 1359, 1395, 1430, 1464, 1497, 1644, 1769, 1878, 1971, 2053, 2123, 2184, 2236, 2319, 2377, 2414, 2436, 2445, 2443, 2433, 2417, 2396, 2371, 2343, 2314, 2283, 2251, 2091, 1936, 1791, 1657, 1537, 1435, 1349, 1273, 1148, 1047, 964.8, 895.9, 837.3, 786.7, 742.4, 703.5, 668.8, 637.7, 609.7, 584.3, 561.2, 540 }; 
 
 for(i=0; i<78; i++) {e[50][i]=e50[i]*MeV*cm2/g;}
 
name [51] = "G4_POLYCARBONATE"; 
G4double e51[78] = { 146.3, 178.1, 204.7, 228.1, 249.2, 286.4, 319.1, 348.5, 375.5, 400.6, 424.1, 446.2, 497, 542.8, 584.7, 623.5, 659.9, 694.2, 726.6, 757.6, 815.5, 869.1, 919, 965.9, 1010, 1052, 1092, 1130, 1167, 1202, 1235, 1268, 1299, 1329, 1465, 1581, 1681, 1768, 1843, 1908, 1965, 2013, 2090, 2144, 2180, 2201, 2210, 2209, 2202, 2188, 2170, 2149, 2125, 2099, 2072, 2044, 1912, 1785, 1661, 1543, 1433, 1340, 1261, 1191, 1076, 982.4, 906, 842, 787.5, 740.5, 699.2, 662.9, 630.5, 601.5, 575.3, 551.6, 529.9, 510.1 }; 
 
 for(i=0; i<78; i++) {e[51][i]=e51[i]*MeV*cm2/g;}
 
name [52] = "G4_POLYETHYLENE"; 
G4double e52[78] = { 168.2, 205.8, 237.5, 265.4, 290.7, 335.4, 374.8, 410.5, 443.2, 473.7, 502.3, 529.3, 591.4, 647.6, 699.1, 747.1, 792.1, 834.6, 874.9, 913.5, 985.9, 1053, 1116, 1175, 1232, 1285, 1336, 1385, 1432, 1478, 1521, 1564, 1604, 1644, 1825, 1983, 2122, 2246, 2355, 2453, 2539, 2615, 2740, 2833, 2899, 2940, 2962, 2967, 2958, 2938, 2908, 2871, 2829, 2782, 2732, 2681, 2418, 2183, 1987, 1826, 1691, 1577, 1480, 1396, 1257, 1145, 1053, 977.1, 912.4, 856.5, 807.8, 764.9, 726.7, 692.6, 661.8, 633.9, 608.6, 585.4 }; 
 
 for(i=0; i<78; i++) {e[52][i]=e52[i]*MeV*cm2/g;}
 
name [53] = "G4_MYLAR"; 
G4double e53[78] = { 163.2, 195.3, 221.8, 244.9, 265.6, 301.9, 333.5, 361.8, 387.6, 411.5, 433.8, 454.8, 502.7, 545.7, 584.8, 621, 654.8, 686.6, 716.7, 745.3, 798.7, 847.9, 893.8, 936.8, 977.4, 1016, 1052, 1087, 1121, 1153, 1183, 1213, 1241, 1269, 1393, 1500, 1593, 1673, 1744, 1806, 1860, 1908, 1985, 2042, 2083, 2110, 2126, 2133, 2133, 2126, 2115, 2100, 2082, 2061, 2038, 2014, 1874, 1731, 1599, 1482, 1379, 1290, 1215, 1148, 1038, 948.5, 875.3, 813.8, 761.5, 716.3, 676.6, 641.6, 610.5, 582.5, 557.3, 534.4, 513.6, 494.5 }; 
 
 for(i=0; i<78; i++) {e[53][i]=e53[i]*MeV*cm2/g;}
 
name [54] = "G4_PLEXIGLASS"; 
G4double e54[78] = { 118.7, 146.6, 170.3, 191.4, 210.4, 244.5, 274.7, 302.1, 327.3, 350.9, 373.1, 394.2, 442.7, 486.8, 527.4, 565.2, 600.9, 634.6, 666.7, 697.4, 755.1, 808.9, 859.3, 906.9, 952, 995, 1036, 1075, 1113, 1150, 1185, 1219, 1252, 1283, 1428, 1554, 1665, 1762, 1848, 1923, 1990, 2049, 2144, 2216, 2266, 2300, 2320, 2328, 2327, 2318, 2302, 2282, 2259, 2232, 2203, 2173, 2008, 1847, 1702, 1574, 1464, 1369, 1288, 1217, 1099, 1004, 925.6, 860.1, 804.4, 756.3, 714.1, 676.9, 643.8, 614.2, 587.4, 563.2, 541.1, 520.8 }; 
 
 for(i=0; i<78; i++) {e[54][i]=e54[i]*MeV*cm2/g;}
 
name [55] = "G4_POLYPROPYLENE"; 
G4double e55[78] = { 193.7, 232.8, 265.4, 293.9, 319.6, 365, 404.8, 440.6, 473.4, 503.9, 532.5, 559.6, 621.6, 677.5, 728.8, 776.5, 821.2, 863.4, 903.5, 941.8, 1014, 1080, 1142, 1201, 1257, 1309, 1360, 1408, 1454, 1499, 1541, 1583, 1622, 1661, 1836, 1986, 2117, 2231, 2331, 2417, 2492, 2557, 2660, 2732, 2779, 2806, 2817, 2814, 2801, 2780, 2752, 2720, 2684, 2646, 2606, 2565, 2359, 2165, 1991, 1836, 1700, 1585, 1488, 1403, 1263, 1150, 1058, 981.3, 916.1, 860, 810.9, 767.8, 729.5, 695.1, 664.3, 636.2, 610.8, 587.5 }; 
 
 for(i=0; i<78; i++) {e[55][i]=e55[i]*MeV*cm2/g;}
 
name [56] = "G4_POLYSTYRENE"; 
G4double e56[78] = { 153, 187.5, 216.6, 242.2, 265.3, 306.5, 342.7, 375.5, 405.6, 433.6, 460, 484.9, 542.1, 593.8, 641.3, 685.4, 726.9, 766, 803.1, 838.5, 905, 966.6, 1024, 1078, 1130, 1178, 1225, 1269, 1312, 1352, 1392, 1429, 1466, 1501, 1661, 1797, 1915, 2017, 2106, 2183, 2249, 2305, 2394, 2454, 2492, 2511, 2516, 2509, 2494, 2471, 2443, 2410, 2376, 2339, 2301, 2262, 2070, 1895, 1742, 1609, 1493, 1395, 1312, 1239, 1119, 1021, 941.4, 874.6, 817.8, 768.7, 725.7, 687.9, 654.1, 623.9, 596.6, 571.9, 549.4, 528.7 }; 
 
 for(i=0; i<78; i++) {e[56][i]=e56[i]*MeV*cm2/g;}
 
name [57] = "G4_TEFLON"; 
G4double e57[78] = { 82.7, 99.97, 114.4, 127.1, 138.4, 158.5, 176.2, 192, 206.5, 220, 232.6, 244.5, 271.9, 296.4, 319, 339.8, 359.4, 377.8, 395.3, 412, 443.3, 472.2, 499.3, 524.7, 548.8, 571.7, 593.6, 614.5, 634.6, 653.9, 672.5, 690.4, 707.8, 724.6, 801.6, 868.9, 928.6, 981.9, 1030, 1073, 1112, 1148, 1209, 1260, 1301, 1334, 1361, 1382, 1397, 1408, 1415, 1419, 1420, 1418, 1414, 1409, 1399, 1373, 1319, 1246, 1165, 1095, 1034, 979.4, 887.4, 813.7, 752.5, 700.8, 656.5, 618.1, 584.5, 554.8, 528.3, 504.6, 483, 463.5, 445.6, 429.3 }; 
 
 for(i=0; i<78; i++) {e[57][i]=e57[i]*MeV*cm2/g;}
 
name [58] = "G4_POLYVINYL_CHLORIDE"; 
G4double e58[78] = { 94.27, 114.4, 131.3, 146.3, 159.9, 184, 205.5, 224.9, 242.9, 259.7, 275.5, 290.5, 325.3, 357, 386.3, 413.7, 439.5, 464.1, 487.6, 510.1, 552.6, 592.4, 629.9, 665.5, 699.4, 731.8, 762.9, 792.9, 821.8, 849.7, 876.7, 902.9, 928.2, 952.9, 1067, 1167, 1255, 1335, 1405, 1468, 1525, 1574, 1657, 1719, 1764, 1795, 1814, 1823, 1824, 1818, 1807, 1792, 1774, 1753, 1731, 1708, 1591, 1482, 1385, 1299, 1222, 1153, 1092, 1037, 941.7, 862.4, 795.8, 739.8, 692.3, 651.2, 615.4, 583.7, 555.6, 530.3, 507.5, 486.9, 468, 450.8 }; 
 
 for(i=0; i<78; i++) {e[58][i]=e58[i]*MeV*cm2/g;}
 
name [59] = "G4_PROPANE"; 
G4double e59[78] = { 194.5, 234.7, 268.4, 298, 324.7, 372, 413.6, 451.2, 485.8, 518, 548.2, 576.8, 642.6, 702.2, 756.9, 808, 855.9, 901.2, 944.4, 985.6, 1063, 1135, 1203, 1267, 1327, 1385, 1440, 1492, 1543, 1592, 1639, 1684, 1728, 1770, 1964, 2131, 2277, 2404, 2516, 2613, 2698, 2771, 2887, 2969, 3022, 3053, 3065, 3061, 3046, 3021, 2990, 2953, 2912, 2868, 2822, 2775, 2564, 2372, 2191, 2022, 1868, 1739, 1629, 1534, 1377, 1252, 1150, 1065, 993, 931.1, 877.2, 829.9, 787.8, 750.2, 716.4, 685.8, 658, 632.5 }; 
 
 for(i=0; i<78; i++) {e[59][i]=e59[i]*MeV*cm2/g;}
 
name [60] = "G4_Pt"; 
G4double e60[78] = { 14.45, 17.87, 20.78, 23.37, 25.71, 29.9, 33.61, 36.99, 40.1, 43.01, 45.75, 48.35, 54.36, 59.81, 64.84, 69.54, 73.97, 78.17, 82.17, 86, 93.22, 99.96, 106.3, 112.3, 118, 123.5, 128.7, 133.8, 138.7, 143.4, 147.9, 152.4, 156.7, 160.8, 180.3, 197.6, 213.3, 227.7, 240.9, 253, 264.3, 274.7, 293.2, 309, 322.4, 333.7, 343.1, 350.7, 356.8, 361.6, 365.1, 367.6, 369.1, 369.8, 369.9, 369.3, 364.7, 357, 347.3, 336.6, 325.5, 314.4, 303.6, 293.2, 274.3, 258.5, 245.1, 233.4, 223.1, 214, 205.8, 198.4, 191.6, 185.4, 179.7, 174.4, 169.5, 164.9 }; 
 
 for(i=0; i<78; i++) {e[60][i]=e60[i]*MeV*cm2/g;}
 
name [61] = "G4_Si"; 
G4double e61[78] = { 47.32, 61.63, 74.34, 85.98, 96.82, 116.8, 135.1, 152.1, 168.2, 183.4, 198.1, 212.1, 245.3, 276.1, 305.2, 332.7, 359.1, 384.4, 408.7, 432.2, 477, 519.3, 559.4, 597.5, 633.9, 668.6, 701.9, 733.7, 764.3, 793.5, 821.6, 848.5, 874.4, 899.1, 1008, 1097, 1168, 1225, 1270, 1307, 1336, 1359, 1390, 1408, 1416, 1417, 1413, 1405, 1395, 1383, 1370, 1356, 1341, 1326, 1310, 1295, 1218, 1146, 1081, 1023, 970.6, 923.9, 881.4, 842.4, 773.1, 713.1, 661.2, 617.4, 579.7, 547, 518.2, 492.7, 470, 449.5, 430.9, 414, 398.6, 384.4 }; 
 
 for(i=0; i<78; i++) {e[61][i]=e61[i]*MeV*cm2/g;}
 
name [62] = "G4_SILICON_DIOXIDE"; 
G4double e62[78] = { 68.9, 85.1, 98.85, 111, 122.1, 141.8, 159.3, 175.1, 189.8, 203.4, 216.3, 228.5, 256.6, 282.1, 305.7, 327.6, 348.3, 367.9, 386.5, 404.4, 437.9, 469.2, 498.5, 526.3, 552.6, 577.7, 601.7, 624.7, 646.9, 668.2, 688.8, 708.7, 728, 746.7, 832.2, 906.8, 972.6, 1031, 1083, 1129, 1170, 1207, 1268, 1315, 1351, 1377, 1396, 1409, 1417, 1421, 1421, 1419, 1414, 1406, 1397, 1387, 1349, 1301, 1237, 1163, 1087, 1021, 964, 913.7, 829.4, 762.2, 706.3, 659, 618.4, 583.3, 552.3, 524.9, 500.4, 478.4, 458.5, 440.4, 423.9, 408.7 }; 
 
 for(i=0; i<78; i++) {e[62][i]=e62[i]*MeV*cm2/g;}
 
name [63] = "G4_STILBENE"; 
G4double e63[78] = { 192, 228.9, 259.4, 286, 309.7, 351.2, 387.4, 419.7, 449.2, 476.4, 501.9, 525.8, 580.4, 629.2, 673.8, 714.9, 753.3, 789.4, 823.5, 856, 916.5, 972.3, 1024, 1073, 1119, 1162, 1203, 1243, 1280, 1316, 1351, 1384, 1416, 1446, 1585, 1703, 1805, 1892, 1968, 2034, 2090, 2139, 2215, 2268, 2303, 2323, 2331, 2330, 2321, 2307, 2287, 2265, 2240, 2212, 2184, 2155, 1995, 1841, 1703, 1582, 1477, 1385, 1305, 1233, 1113, 1015, 936, 869.6, 813, 764.1, 721.3, 683.7, 650.1, 620, 592.9, 568.3, 545.9, 525.4 }; 
 
 for(i=0; i<78; i++) {e[63][i]=e63[i]*MeV*cm2/g;}
 
name [64] = "G4_Ti"; 
G4double e64[78] = { 59.24, 72.81, 84.28, 94.41, 103.6, 119.9, 134.3, 147.4, 159.4, 170.6, 181.1, 191.1, 214, 234.7, 253.8, 271.6, 288.3, 304.1, 319.1, 333.5, 360.5, 385.5, 409, 431.2, 452.2, 472.2, 491.4, 509.7, 527.3, 544.3, 560.6, 576.4, 591.7, 606.5, 674.2, 733.1, 785, 831, 871.7, 907.9, 940, 968.3, 1015, 1051, 1077, 1095, 1106, 1112, 1114, 1112, 1107, 1100, 1092, 1082, 1071, 1059, 995.6, 933.7, 877.6, 827.9, 785.2, 747.8, 714.5, 684.4, 631.9, 587.1, 548.3, 514.3, 484.2, 457.3, 433.4, 412.5, 393.7, 376.7, 361.5, 347.6, 334.9, 323.2 }; 
 
 for(i=0; i<78; i++) {e[64][i]=e64[i]*MeV*cm2/g;}
 
name [65] = "G4_Sn"; 
G4double e65[78] = { 27.39, 33.96, 39.55, 44.52, 49.04, 57.11, 64.28, 70.8, 76.82, 82.45, 87.76, 92.79, 104.4, 115, 124.8, 133.9, 142.5, 150.6, 158.4, 165.8, 179.8, 192.9, 205.2, 216.8, 227.8, 238.4, 248.4, 258.1, 267.4, 276.4, 285.1, 293.5, 301.6, 309.5, 345.7, 377.3, 405.3, 430.1, 452.1, 471.8, 489.3, 504.8, 530.7, 550.9, 566.3, 577.9, 586.3, 592.2, 596.1, 598.4, 599.3, 599.1, 598.1, 596.4, 594.2, 591.5, 573.8, 552.8, 531.2, 510.1, 488.3, 467, 447, 428.6, 396.3, 369.4, 346.8, 327.7, 311.4, 297.1, 283.9, 271.9, 260.8, 250.6, 241.7, 233.6, 226, 218.9 }; 
 
 for(i=0; i<78; i++) {e[65][i]=e65[i]*MeV*cm2/g;}
 
name [66] = "G4_TISSUE_METHANE"; 
G4double e66[78] = { 135.1, 165.6, 191.3, 213.9, 234.4, 270.7, 302.8, 331.8, 358.4, 383.2, 406.5, 428.6, 479.3, 525.2, 567.4, 606.7, 643.6, 678.5, 711.7, 743.4, 803.1, 858.6, 910.7, 960.1, 1007, 1052, 1095, 1136, 1176, 1214, 1251, 1287, 1322, 1356, 1514, 1655, 1783, 1899, 2006, 2104, 2193, 2275, 2418, 2535, 2626, 2693, 2737, 2760, 2764, 2752, 2728, 2695, 2655, 2612, 2566, 2519, 2278, 2060, 1878, 1727, 1602, 1494, 1403, 1323, 1191, 1085, 998.2, 926, 864.6, 811.7, 765.6, 724.9, 688.8, 656.5, 627.4, 601.1, 577.1, 555.2 }; 
 
 for(i=0; i<78; i++) {e[66][i]=e66[i]*MeV*cm2/g;}
 
name [67] = "G4_TISSUE_PROPANE"; 
G4double e67[78] = { 149.7, 181.5, 208.2, 231.7, 252.8, 290.4, 323.5, 353.4, 380.9, 406.5, 430.5, 453.3, 505.7, 553, 596.6, 637.2, 675.4, 711.5, 745.8, 778.6, 840.4, 897.8, 951.6, 1003, 1051, 1097, 1141, 1183, 1224, 1263, 1301, 1337, 1372, 1407, 1563, 1701, 1822, 1929, 2024, 2109, 2185, 2252, 2362, 2446, 2507, 2549, 2573, 2583, 2581, 2569, 2550, 2524, 2494, 2461, 2426, 2389, 2211, 2043, 1887, 1744, 1616, 1507, 1415, 1334, 1201, 1094, 1006, 933.3, 871.4, 818.1, 771.5, 730.5, 694.1, 661.5, 632.1, 605.6, 581.4, 559.3 }; 
 
 for(i=0; i<78; i++) {e[67][i]=e67[i]*MeV*cm2/g;}
 
name [68] = "G4_TOLUENE"; 
G4double e68[78] = { 192.5, 230, 261, 288.1, 312.3, 354.9, 392, 425.3, 455.6, 483.7, 510, 534.8, 591.3, 642.1, 688.4, 731.3, 771.4, 809.1, 844.8, 878.8, 942.4, 1001, 1056, 1107, 1155, 1201, 1245, 1287, 1327, 1365, 1401, 1437, 1471, 1504, 1652, 1779, 1888, 1983, 2065, 2136, 2197, 2250, 2333, 2392, 2430, 2452, 2461, 2459, 2449, 2433, 2411, 2386, 2358, 2328, 2296, 2264, 2109, 1960, 1817, 1683, 1560, 1456, 1368, 1291, 1163, 1060, 976.6, 906.6, 847.1, 795.7, 750.8, 711.3, 676.1, 644.6, 616.2, 590.5, 567, 545.6 }; 
 
 for(i=0; i<78; i++) {e[68][i]=e68[i]*MeV*cm2/g;}
 
name [69] = "G4_U"; 
G4double e69[78] = { 13.2, 16.72, 19.77, 22.52, 25.04, 29.61, 33.73, 37.51, 41.03, 44.35, 47.5, 50.51, 57.52, 63.96, 69.97, 75.62, 80.99, 86.11, 91.01, 95.73, 104.7, 113.1, 121.1, 128.7, 136, 143, 149.8, 156.3, 162.6, 168.8, 174.7, 180.5, 186.2, 191.7, 217.4, 240.5, 261.5, 280.8, 298.5, 314.8, 329.8, 343.7, 368, 388.4, 405.3, 419, 429.8, 438.1, 444.2, 448.4, 450.8, 451.8, 451.6, 450.3, 448.2, 445.4, 420, 390.1, 362.9, 339.6, 319.8, 303.1, 288.9, 276.7, 256.9, 241.7, 229.4, 218.6, 209.1, 200.6, 193, 186.1, 179.8, 174, 168.6, 163.7, 159.1, 154.8 }; 
 
 for(i=0; i<78; i++) {e[69][i]=e69[i]*MeV*cm2/g;}
 
name [70] = "G4_W"; 
G4double e70[78] = { 20.75, 25.23, 28.99, 32.29, 35.26, 40.51, 45.11, 49.26, 53.06, 56.59, 59.9, 63.02, 70.18, 76.63, 82.54, 88.02, 93.16, 98.01, 102.6, 107, 115.2, 122.9, 130, 136.7, 143.1, 149.2, 155, 160.6, 165.9, 171.1, 176.1, 180.9, 185.6, 190.1, 211.1, 229.6, 246.2, 261.3, 275.1, 287.7, 299.3, 310, 328.8, 344.7, 358.1, 369.2, 378.3, 385.6, 391.3, 395.6, 398.7, 400.6, 401.7, 401.9, 401.4, 400.4, 389.5, 374.8, 359.5, 344.8, 331, 318.3, 306.5, 295.7, 276.7, 260.8, 247.3, 235.5, 225.2, 216.1, 207.8, 200.3, 193.5, 187.3, 181.5, 176.2, 171.3, 166.7 }; 
 
 for(i=0; i<78; i++) {e[70][i]=e70[i]*MeV*cm2/g;}
 
name [71] = "G4_WATER"; 
G4double e71[78] = { 98.91, 122.7, 143.1, 161.1, 177.5, 206.9, 233, 256.8, 278.8, 299.3, 318.6, 337, 379.5, 418.1, 453.8, 487.1, 518.6, 548.3, 576.7, 603.9, 655.2, 703, 748, 790.5, 831, 869.6, 906.6, 942.2, 976.4, 1009, 1041, 1072, 1102, 1131, 1265, 1383, 1488, 1582, 1666, 1743, 1811, 1873, 1978, 2062, 2128, 2178, 2215, 2240, 2255, 2262, 2262, 2256, 2245, 2230, 2211, 2190, 2050, 1896, 1752, 1624, 1511, 1414, 1329, 1255, 1132, 1034, 952.7, 884.8, 826.9, 777.1, 733.4, 694.9, 660.7, 630.1, 602.5, 577.6, 554.8, 534 }; 
 
 for(i=0; i<78; i++) {e[71][i]=e71[i]*MeV*cm2/g;}
 
name [72] = "G4_WATER_VAPOR"; 
G4double e72[78] = { 126.3, 154.7, 178.7, 199.8, 218.8, 252.7, 282.5, 309.5, 334.3, 357.3, 379, 399.5, 446.7, 489.3, 528.5, 565, 599.2, 631.6, 662.4, 691.9, 747.3, 798.8, 847.2, 892.9, 936.3, 977.8, 1018, 1056, 1093, 1128, 1163, 1196, 1228, 1260, 1405, 1535, 1652, 1758, 1855, 1942, 2022, 2094, 2216, 2309, 2377, 2422, 2446, 2455, 2453, 2446, 2436, 2423, 2409, 2392, 2371, 2346, 2169, 1971, 1797, 1653, 1537, 1437, 1351, 1275, 1149, 1049, 965.8, 896.6, 837.6, 786.9, 742.5, 703.3, 668.6, 637.5, 609.5, 584.2, 561.1, 539.9 }; 
 
 for(i=0; i<78; i++) {e[72][i]=e72[i]*MeV*cm2/g;}
 
name [73] = "G4_Xe"; 
G4double e73[78] = { 18.11, 23.3, 27.86, 31.99, 35.83, 42.84, 49.2, 55.1, 60.63, 65.87, 70.87, 75.65, 86.89, 97.3, 107.1, 116.3, 125.1, 133.5, 141.7, 149.5, 164.4, 178.6, 192, 204.9, 217.3, 229.2, 240.7, 251.8, 262.6, 273.1, 283.4, 293.3, 303, 312.5, 356.7, 396.5, 432.5, 465.2, 494.9, 521.8, 546.1, 567.9, 604.7, 633.2, 654.5, 669.3, 678.8, 683.6, 684.6, 682.5, 678, 671.7, 663.9, 655.1, 645.6, 635.6, 584.6, 538.7, 500.4, 468.8, 446.7, 430.5, 417.1, 405.3, 383.9, 364, 344.9, 326.8, 309.7, 294.6, 281.3, 269.5, 258.7, 248.8, 239.6, 231.1, 223.2, 216.2 }; 
 
 for(i=0; i<78; i++) {e[73][i]=e73[i]*MeV*cm2/g;}
} 
