//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// This is the *basic* version of Hadrontherapy, a Geant4-based application
// See more at: http://g4advancedexamples.lngs.infn.it/Examples/hadrontherapy
//
// To obtain the full version visit the pages: http://sites.google.com/site/hadrontherapy/

// ----------------------------------------------------------------------------
//                 GEANT 4 - Hadrontherapy example
// ----------------------------------------------------------------------------
// Main Authors:
//
// G.A.P. Cirrone(a)°, G.Cuttone(a), F.Di Rosa(a), E.Mazzaglia(a), F.Romano(a)
// 
// Contributor authors:
// P.Kaitaniemi(d), A.Heikkinen(d), G.Danielsen (d)
//
// Past authors:
// M.G.Pia(b), S.Guatelli(c), G.Russo(a), M.Russo(a), A.Lechner(e) 
//
// (a) Laboratori Nazionali del Sud 
//     of the INFN, Catania, Italy
//
// (b) INFN Section of Genova, Italy
// 
// (c) University of Wallongong, Australia
//
// (d) Helsinki Institute of Physics, Helsinki, Finland
//
// (e) CERN, (CH)
//
//  *Corresponding author, email to cirrone@lns.infn.it
// ----------------------------------------------------------------------------

#include "G4RunManager.hh"
#include "G4UImanager.hh"
#include "G4UIterminal.hh"
#include "G4UItcsh.hh"
#include "HadrontherapyEventAction.hh"
#include "HadrontherapyPhysicsList.hh"
#include "HadrontherapyDetectorSD.hh"
#include "HadrontherapyPrimaryGeneratorAction.hh"
#include "HadrontherapyRunAction.hh"
#include "HadrontherapyMatrix.hh"
#include "Randomize.hh"  
#include "G4RunManager.hh"
#include "G4UImanager.hh"
#include "G4UImessenger.hh"
#include "globals.hh"
#include "HadrontherapySteppingAction.hh"
#include "HadrontherapyAnalysisManager.hh"
#include "HadrontherapyGeometryController.hh"
#include "HadrontherapyGeometryMessenger.hh"
#include "HadrontherapyInteractionParameters.hh"
#include "G4ScoringManager.hh"
#include "IAEAScoreWriter.hh"

#if defined(G4UI_USE_TCSH)
#include "G4UIterminal.hh"
#include "G4UItcsh.hh"
#endif

#ifdef G4UI_USE_XM
#include "G4UIXm.hh"
#endif

#ifdef G4VIS_USE
#include "G4VisExecutive.hh"
#endif

#ifdef G4UI_USE_QT
#include "G4UIQt.hh"
#include "G4Qt.hh"
#endif

#ifdef G4UI_USE
#include "G4UIExecutive.hh"
#endif
//////////////////////////////////////////////////////////////////////////////////////////////

int main(int argc ,char ** argv)
{
		// Set the Random engine
	CLHEP::HepRandom::setTheEngine(new CLHEP::RanecuEngine());
        
	// Only if an initial random seed is needed
	// G4int seed = time(0);
	// CLHEP::HepRandom::setTheSeed(seed);
	
	G4RunManager* runManager = new G4RunManager;
	// Geometry controller is responsible for instantiating the
	// geometries. All geometry specific setup tasks are now in class
	// HadrontherapyGeometryController.
	HadrontherapyGeometryController *geometryController = new HadrontherapyGeometryController();
	
	// Connect the geometry controller to the G4 user interface
	HadrontherapyGeometryMessenger *geometryMessenger = new HadrontherapyGeometryMessenger(geometryController);
	
	G4ScoringManager *scoringManager = G4ScoringManager::GetScoringManager();
	scoringManager->SetVerboseLevel(1);
	scoringManager->SetScoreWriter(new IAEAScoreWriter());
	
		// Initialize the default Hadrontherapy geometry
	geometryController->SetGeometry("default");
	
		// Initialize command based scoring
	G4ScoringManager::GetScoringManager();
	
		// Initialize the physics 
	runManager -> SetUserInitialization(new HadrontherapyPhysicsList());
	
		// Initialize the primary particles
	HadrontherapyPrimaryGeneratorAction *pPrimaryGenerator = new HadrontherapyPrimaryGeneratorAction();
	runManager -> SetUserAction(pPrimaryGenerator);
	
		// Optional UserActions: run, event, stepping
	HadrontherapyRunAction* pRunAction = new HadrontherapyRunAction();
	runManager -> SetUserAction(pRunAction);
	
	HadrontherapyEventAction* pEventAction = new HadrontherapyEventAction();
	runManager -> SetUserAction(pEventAction);
	
	HadrontherapySteppingAction* steppingAction = new HadrontherapySteppingAction(pRunAction); 
	runManager -> SetUserAction(steppingAction);    
	
		// Interaction data: stopping powers
	HadrontherapyInteractionParameters* pInteraction = new HadrontherapyInteractionParameters(true);
	
#ifdef G4VIS_USE
		// Visualization manager
	G4VisManager* visManager = new G4VisExecutive;
	visManager -> Initialize();
#endif 
	
	G4UImanager* UI = G4UImanager::GetUIpointer();
	
	// Batch mode: in this configuration, i.e. when the User explicitly call a macro file,
	// the program run and, at the end, authomatically call the exit state.
	// With a such configuration, it is possible to send in background a simulation
	// and correctly get the output file.
	if (argc!=1)   
	{
		G4UIsession* session = 0;
		G4String command = "/control/execute ";
		G4String fileName = argv[1];
		UI -> ApplyCommand(command+fileName);
#if defined(G4UI_USE_TCSH)
		session = new G4UIterminal(new G4UItcsh);
#else
		session = new G4UIterminal();
#endif
		// Only if the User Interface is needed!
		//session -> SessionStart();
		//delete session;
	}
	else  // interactive mode : define visualization UI terminal
	{
		G4UIsession* session = 0;
		
			// If the enviroment variable for the TCSH terminal is active, it is used and the
			// defaultMacro.mac file is executed
#if defined(G4UI_USE_TCSH)
		session = new G4UIterminal(new G4UItcsh);
		UI->ApplyCommand("/control/execute defaultMacro.mac");
		session -> SessionStart();
		
		// Alternatively (if G4UI_USE_TCSH is not defined)  the program search for the
		// G4UI_USE_QT variable. It starts a graphical user interface based on the QT libraries
		// In the following case the GUI.mac file is also executed
#elif defined(G4UI_USE_QT)
		session = new G4UIQt(argc,argv);
		UI->ApplyCommand("/control/execute macro/GUI.mac");      
		session -> SessionStart();
		// As final option, the basic user interface terminal is opened
#else
		session = new G4UIterminal();
		
		UI -> ApplyCommand("/control/execute defaultMacro.mac");
		session -> SessionStart();
#endif
		delete session;
	}
	
		// Job termination
#ifdef G4ANALYSIS_USE_ROOT
    HadrontherapyAnalysisManager::GetInstance() -> flush();     // Finalize the root file 
#endif

#ifdef G4UI_USE
      G4UIExecutive * ui = new G4UIExecutive(argc,argv);      
      ui->SessionStart();
      delete ui;
#endif
  
  delete geometryMessenger;
  delete geometryController;
  delete pInteraction; 
  delete runManager;
  return 0;
  
}
