//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// $Id: G4OpenGLQtViewer.cc,v 1.10 2008/01/30 10:54:13 lgarnier Exp $
// GEANT4 tag $Name:  $
//
// 
// G4OpenGLQtViewer : Class to provide Qt specific
//                     functionality for OpenGL in GEANT4
//
// 27/06/2003 : G.Barrand : implementation (at last !).

#ifdef G4VIS_BUILD_OPENGLQT_DRIVER

#define GEANT4_QT_DEBUG

#include "G4OpenGLQtViewer.hh"

#include "G4ios.hh"
#include "G4VisExtent.hh"
#include "G4LogicalVolume.hh"
#include "G4VSolid.hh"
#include "G4Point3D.hh"
#include "G4Normal3D.hh"
#include "G4Scene.hh"
#include "G4OpenGLQtExportDialog.hh"
#include "G4UnitsTable.hh"
#include "G4Qt.hh"
#include "G4UImanager.hh"
#include "G4UIcommandTree.hh"
#include <qapplication.h>
#include <qlayout.h>
#include <qdialog.h>

#if QT_VERSION >= 0x040000
#include <qmenu.h>
#include <qimagewriter.h>
#else
#include <qaction.h>
#include <qwidgetlist.h>
#include <qpopupmenu.h>
#include <qimage.h>
#endif

#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qgl.h> // include <qglwidget.h>
#include <qdialog.h>
#include <qevent.h> //include <qcontextmenuevent.h>


//////////////////////////////////////////////////////////////////////////////
/**
   Implementation of virtual method of G4VViewer
*/
void G4OpenGLQtViewer::SetView (
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::SetView ++++++++++++++++++++\n");
#endif
  //   if(!fHDC) return;
  //   if(!fHGLRC) return;
  //   ::wglMakeCurrent(fHDC,fHGLRC);
  //  fWindow->makeCurrent();
  G4OpenGLViewer::SetView ();
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::SetView --------------------\n");
#endif
}

/**
 * Set the viewport of the scene
 */
void G4OpenGLQtViewer::setupViewport(int aWidth, int aHeight)
{
  int side = aWidth;
  if (aHeight < aWidth) side = aHeight;
  glViewport((aWidth - side) / 2, (aHeight - side) / 2, side, side);
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-0.5, +0.5, +0.5, -0.5, 4.0, 15.0);
  glMatrixMode(GL_MODELVIEW);
}


//////////////////////////////////////////////////////////////////////////////
/**
   Implementation of virtual method of G4VViewer
*/
void G4OpenGLQtViewer::ShowView (
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::ShowView  +++++++++++++++++++++\n");
#endif
  if (!GLWindow) {
    G4cerr << "Visualization window not defined, please choose one before\n" << G4endl;
  } else {
#if QT_VERSION < 0x040000
    GLWindow->setActiveWindow();
#else
    GLWindow->activateWindow();
#endif
#ifdef GEANT4_QT_DEBUG
    printf("G4OpenGLQtViewer::ShowView -----------------------\n");
#endif
  }
  glFlush ();
  //   // Empty the Windows message queue :
  //   MSG event;
  //   while ( ::PeekMessage(&event, NULL, 0, 0, PM_REMOVE) ) {
  //     ::TranslateMessage(&event);
  //     ::DispatchMessage (&event);
  //   }
}



//////////////////////////////////////////////////////////////////////////////
void G4OpenGLQtViewer::CreateGLQtContext (
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::CreateGLQtContext \n");
#endif
}


//////////////////////////////////////////////////////////////////////////////
void G4OpenGLQtViewer::CreateMainWindow (
 QGLWidget* glWidget
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{

  if(fWindow) return; //Done.
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::CreateMainWindow glWidget\n");
#endif

  // launch Qt if not
  G4Qt* interactorManager = G4Qt::getInstance ();
  //  G4UImanager* UI = G4UImanager::GetUIpointer();

  fWindow = glWidget ;
  //  fWindow->makeCurrent();

  // create window
  if (((QApplication*)interactorManager->GetMainInteractor())) {
    // look for the main window
    bool found = false;
#if QT_VERSION < 0x040000
    // theses lines does nothing exept this one "GLWindow = new QDialog(0..."
    // but if I comment them, it doesn't work...
    QWidgetList  *list = QApplication::allWidgets();
    QWidgetListIt it( *list );         // iterate over the widgets
    QWidget * widget;
    while ( (widget=it.current()) != 0 ) {  // for each widget...
      ++it;
      if ((found== false) && (widget->inherits("QMainWindow"))) {
        GLWindow = new QDialog(0,0,FALSE,Qt::WStyle_Title | Qt::WStyle_SysMenu | Qt::WStyle_MinMax );
        found = true;
      }
    }
    delete list;                      // delete the list, not the widgets
#else
    foreach (QWidget *widget, QApplication::allWidgets()) {
      if ((found== false) && (widget->inherits("QMainWindow"))) {
        GLWindow = new QDialog(0,Qt::WindowTitleHint | Qt::WindowSystemMenuHint | Qt::WindowMinMaxButtonsHint);
        found = true;
      }
    }
#endif

#if QT_VERSION < 0x040000
    glWidget->reparent(GLWindow,0,QPoint(0,0));  
#else
    glWidget->setParent(GLWindow);  
#endif

    if (found==false) {
#ifdef GEANT4_QT_DEBUG
      printf("G4OpenGLQtViewer::CreateMainWindow case Qapp exist, but not found\n");
#endif
      GLWindow = new QDialog();
    }
  } else {
#ifdef GEANT4_QT_DEBUG
    printf("G4OpenGLQtViewer::CreateMainWindow case Qapp exist\n");
#endif
    GLWindow = new QDialog();
  }

  QHBoxLayout *mainLayout = new QHBoxLayout(GLWindow);

  mainLayout->addWidget(fWindow);

#if QT_VERSION < 0x040000
  GLWindow->setCaption( tr( "QGl Viewer" ));
#else
  GLWindow->setLayout(mainLayout);
  GLWindow->setWindowTitle(tr("QGl Viewer"));
#endif
  GLWindow->resize(300, 300);
  GLWindow->move(900,300);
  GLWindow->show();
  
  // delete the pointer if close this
  //  GLWindow->setAttribute(Qt::WA_DeleteOnClose);

#if QT_VERSION >= 0x040000
//   QObject ::connect(GLWindow, 
//                     SIGNAL(rejected()),
//                     this, 
//                     SLOT(dialogClosed()));
#endif

  WinSize_x = 400;
  WinSize_y = 400;
  if (WinSize_x < fVP.GetWindowSizeHintX ())
    WinSize_x = fVP.GetWindowSizeHintX ();
  if (WinSize_y < fVP.GetWindowSizeHintY ())
    WinSize_y = fVP.GetWindowSizeHintY ();

  if(!fWindow) return;
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::CreateMainWindow glWidget END\n");
#endif

  if (!fContextMenu) 
    createPopupMenu();

}

#if QT_VERSION >= 0x040000
/**  Close the dialog and set the pointer to NULL
 */
// void G4OpenGLQtViewer::dialogClosed() {
// #ifdef GEANT4_QT_DEBUG
//   printf("G4OpenGLQtViewer::dialogClosed END\n");
// #endif
//   //  GLWindow = NULL;
// }
#endif

//////////////////////////////////////////////////////////////////////////////
G4OpenGLQtViewer::G4OpenGLQtViewer (
                                    G4OpenGLSceneHandler& scene
                                    )
  :G4VViewer (scene, -1)
  ,G4OpenGLViewer (scene)
  ,fWindow(0)
  ,fContextMenu(0)
  ,fMouseAction(STYLE1)
  ,fDeltaRotation(1)
  ,fDeltaSceneTranslation(0.01)
  ,fDeltaDepth(0.01)
  ,fDeltaZoom(0.1)
  ,holdKeyEvent(false)
{
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::G4OpenGLQtViewer \n");
#endif
}

//////////////////////////////////////////////////////////////////////////////
G4OpenGLQtViewer::~G4OpenGLQtViewer (
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::~G4OpenGLQtViewer \n");
#endif
}


/**
   Create a popup menu for the widget. This menu is activated by right-mouse click
*/
void G4OpenGLQtViewer::createPopupMenu()    {

#if QT_VERSION < 0x040000
  fContextMenu = new QPopupMenu( GLWindow,"All" );
#else
  fContextMenu = new QMenu("All");
#endif

#if QT_VERSION < 0x040000
  QPopupMenu *mMouseAction = new QPopupMenu( fContextMenu );
  fContextMenu->insertItem("&Mouse actions",mMouseAction);
#if QT_VERSION < 0x030200
  fRotateAction =  new QAction("&Rotate","&Rotate",CTRL+Key_R,mMouseAction,0,true);
  fMoveAction =  new QAction("&Move","&Move",CTRL+Key_M,mMouseAction,0,true);
  fPickAction =  new QAction("&Pick","&Pick",CTRL+Key_P,mMouseAction,0,true);
  fShortcutsAction =  new QAction("&Show shortcuts","&Show shortcuts",CTRL+Key_S,mMouseAction,0,true);
#else
  fRotateAction =  new QAction("&Rotate",CTRL+Key_R,mMouseAction,0,true);
  fMoveAction =  new QAction("&Move",CTRL+Key_M,mMouseAction,0,true);
  fPickAction =  new QAction("&Pick",CTRL+Key_P,mMouseAction,0,true);
  fShortcutsAction =  new QAction("&Show shortcuts",CTRL+Key_S,mMouseAction,0,true);
#endif
  fRotateAction->addTo(mMouseAction);
  fMoveAction->addTo(mMouseAction);
  fPickAction->addTo(mMouseAction);
  fShortcutsAction->addTo(mMouseAction);

  fRotateAction->setToggleAction(true);
  fMoveAction->setToggleAction(true);
  fPickAction->setToggleAction(true);
  fShortcutsAction->setToggleAction(true);

  fRotateAction->setOn(true);
  fMoveAction->setOn(false);
  fPickAction->setOn(false);
  fShortcutsAction->setOn(false);


  QObject ::connect(fRotateAction, 
                    SIGNAL(activated()),
                    this,
                    SLOT(actionMouseRotate()));

  QObject ::connect(fMoveAction, 
                    SIGNAL(activated()),
                    this,
                    SLOT(actionMouseMove()));

  QObject ::connect(fPickAction, 
                    SIGNAL(activated()),
                    this,
                    SLOT(actionMousePick()));

  QObject ::connect(fShortcutsAction, 
                    SIGNAL(activated()),
                    this,
                    SLOT(actionMouseShortcuts()));

#else
  QMenu *mMouseAction = fContextMenu->addMenu("&Mouse actions");

  fRotateAction = mMouseAction->addAction("Rotate");
  fMoveAction = mMouseAction->addAction("Move");
  fPickAction = mMouseAction->addAction("Pick");
  fShortcutsAction = mMouseAction->addAction("Show shortcuts");

  fRotateAction->setCheckable(true);
  fMoveAction->setCheckable(false);
  fPickAction->setCheckable(false);
  fShortcutsAction->setCheckable(false);

  fRotateAction->setChecked(true);
  fMoveAction->setChecked(false);
  fPickAction->setChecked(false);
  fShortcutsAction->setChecked(false);

  QObject ::connect(fRotateAction, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionMouseRotate()));

  QObject ::connect(fMoveAction, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionMouseMove()));

  QObject ::connect(fPickAction, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionMousePick()));

  QObject ::connect(fShortcutsAction, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionMouseShortcuts()));
#endif

#if QT_VERSION < 0x040000
  // === Style Menu ===
  QPopupMenu *mStyle = new QPopupMenu(fContextMenu);

  QPopupMenu *mRepresentation = new QPopupMenu(fContextMenu);

  QPopupMenu *mProjection = new QPopupMenu(fContextMenu);

#if QT_VERSION < 0x030200
  QAction *polyhedron = new QAction("&Polyhedron","&Polyhedron",CTRL+Key_P,mRepresentation,0,true);
  QAction *nurbs = new QAction("&NURBS","&NURBS",CTRL+Key_N,mRepresentation,0,true);

  QAction *ortho = new QAction("&Orthographic","&Orthographic",CTRL+Key_O,mProjection,0,true);
  QAction *perspective = new QAction("&Perspective","&Perspective",CTRL+Key_P,mProjection,0,true);
#else
  QAction *polyhedron = new QAction("&Polyhedron",CTRL+Key_P,mRepresentation);
  QAction *nurbs = new QAction("&NURBS",CTRL+Key_N,mRepresentation);

  QAction *ortho = new QAction("&Orthographic",CTRL+Key_O,mProjection);
  QAction *perspective = new QAction("&Perspective",CTRL+Key_P,mProjection);
  polyhedron->setToggleAction(true);
  nurbs->setToggleAction(true);
  ortho->setToggleAction(true);
  perspective->setToggleAction(true);
#endif
  polyhedron->addTo(mRepresentation);
  nurbs->addTo(mRepresentation);

  ortho->addTo(mProjection);
  perspective->addTo(mProjection);

  mStyle->insertItem("&Representation",mRepresentation);
  mStyle->insertItem("&Projection",mProjection);
  fContextMenu->insertItem("&Style",mStyle);


#else
  // === Style Menu ===
  QMenu *mStyle = fContextMenu->addMenu("&Style");

  QMenu *mRepresentation = mStyle->addMenu("&Representation");
  QMenu *mProjection = mStyle->addMenu("&Projection");
  QAction *polyhedron = mRepresentation->addAction("Polyhedron");
  QAction *nurbs = mRepresentation->addAction("NURBS");

  QAction *ortho = mProjection->addAction("Orthographic");
  QAction *perspective = mProjection->addAction("Persepective");
#endif

  // INIT mRepresentation
  G4ViewParameters::RepStyle style;
  style = fVP.GetRepStyle();
  if (style == G4ViewParameters::polyhedron) {
    createRadioAction(polyhedron,nurbs,SLOT(toggleRepresentation(bool)),1);
  } else if (style == G4ViewParameters::nurbs) {
    createRadioAction(polyhedron,nurbs,SLOT(toggleRepresentation(bool)),2);
  } else {
    mRepresentation->clear();
  }

  // INIT mProjection
  if (fVP.GetFieldHalfAngle() == 0) {
    createRadioAction(ortho, perspective,SLOT(toggleProjection(bool)),1);
  } else {
    createRadioAction(ortho, perspective,SLOT(toggleProjection(bool)),2);
  }

#if QT_VERSION < 0x040000
  // === Drawing Menu ===
  QPopupMenu *mDrawing = new QPopupMenu(fContextMenu);
  fContextMenu->insertItem("&Drawing",mDrawing);

  fDrawingWireframe = new QPopupMenu(mDrawing);
  mDrawing->insertItem("&Wireframe",fDrawingWireframe);

  mDrawing->setCheckable(true);
  fDrawingWireframe->setCheckable(true);

  fDrawingLineRemoval = new QPopupMenu(mDrawing);
  mDrawing->insertItem("&Hidden line removal",fDrawingLineRemoval);
  fDrawingLineRemoval->setCheckable(true);

  fDrawingSurfaceRemoval = new QPopupMenu(mDrawing);
  mDrawing->insertItem("&Hidden surface removal",fDrawingSurfaceRemoval);
  fDrawingSurfaceRemoval->setCheckable(true);

  fDrawingLineSurfaceRemoval = new QPopupMenu(mDrawing);
  mDrawing->insertItem("&Hidden line and surface removal",fDrawingLineSurfaceRemoval);
  fDrawingLineSurfaceRemoval->setCheckable(true);

#else
  // === Drawing Menu ===
  QMenu *mDrawing = mStyle->addMenu("&Drawing");

  fDrawingWireframe = mDrawing->addAction("Wireframe");
  fDrawingWireframe->setCheckable(true);

  fDrawingLineRemoval = mDrawing->addAction("Hidden line removal");
  fDrawingLineRemoval->setCheckable(true);

  fDrawingSurfaceRemoval = mDrawing->addAction("Hidden Surface removal");
  fDrawingSurfaceRemoval->setCheckable(true);

  fDrawingLineSurfaceRemoval = mDrawing->addAction("Hidden line and surface removal");
  fDrawingLineSurfaceRemoval->setCheckable(true);
#endif
  // INIT Drawing
  G4ViewParameters::DrawingStyle d_style;
  d_style = fVP.GetDrawingStyle();
  
#if QT_VERSION < 0x040000
  if (d_style == G4ViewParameters::wireframe) {
    fDrawingWireframe->setItemChecked(0,true);
  } else if (d_style == G4ViewParameters::hlr) {
    fDrawingLineRemoval->setItemChecked(0,true);
  } else if (d_style == G4ViewParameters::hsr) {
    fDrawingSurfaceRemoval->setItemChecked(0,true);
  } else if (d_style == G4ViewParameters::hlhsr) {
    fDrawingLineSurfaceRemoval->setItemChecked(0,true);
  } else {
    mDrawing->clear();
  }
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer:: fDrawingWireframe 1\n");
#endif
  QObject ::connect(fDrawingWireframe, 
                    SIGNAL(activated()),
                    this, 
                    SLOT(actionDrawingWireframe()));
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer:: fDrawingWireframe 2\n");
#endif
  QObject ::connect(fDrawingLineRemoval, 
                    SIGNAL(activated()),
                    this, 
                    SLOT(actionDrawingLineRemoval()));
  QObject ::connect(fDrawingSurfaceRemoval, 
                    SIGNAL(activated()),
                    this, 
                    SLOT(actionDrawingSurfaceRemoval()));
  QObject ::connect(fDrawingLineSurfaceRemoval, 
                    SIGNAL(activated()),
                    this, 
                    SLOT(actionDrawingLineSurfaceRemoval()));
#else
  if (d_style == G4ViewParameters::wireframe) {
    fDrawingWireframe->setChecked(true);
  } else if (d_style == G4ViewParameters::hlr) {
    fDrawingLineRemoval->setChecked(true);
  } else if (d_style == G4ViewParameters::hsr) {
    fDrawingSurfaceRemoval->setChecked(true);
  } else if (d_style == G4ViewParameters::hlhsr) {
    fDrawingLineSurfaceRemoval->setChecked(true);
  } else {
    mDrawing->clear();
  }
  QObject ::connect(fDrawingWireframe, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionDrawingWireframe()));
  QObject ::connect(fDrawingLineRemoval, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionDrawingLineRemoval()));
  QObject ::connect(fDrawingSurfaceRemoval, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionDrawingSurfaceRemoval()));
  QObject ::connect(fDrawingLineSurfaceRemoval, 
                    SIGNAL(triggered(bool)),
                    this, 
                    SLOT(actionDrawingLineSurfaceRemoval()));
#endif



#if QT_VERSION < 0x040000
  QPopupMenu *mBackground = new QPopupMenu(mStyle);
  mStyle->insertItem("&Background color",mBackground);

#if QT_VERSION < 0x030200
  QAction *white = new QAction("&White","&White",CTRL+Key_W,mBackground,0,true);
  QAction *black =  new QAction("&Black","&Black",CTRL+Key_B,mBackground,0,true);
#else
  QAction *white = new QAction("&White",CTRL+Key_W,mBackground);
  QAction *black =  new QAction("&Black",CTRL+Key_B,mBackground);
  white->setToggleAction(true);
  black->setToggleAction(true);
#endif
  white->addTo(mBackground);
  black->addTo(mBackground);

#else
  QMenu *mBackground = mStyle->addMenu("&Background color");
  QAction *white = mBackground->addAction("White");
  QAction *black = mBackground->addAction("Black");

#endif
  if (background.GetRed() == 1. &&
      background.GetGreen() == 1. &&
      background.GetBlue() == 1.) {
    createRadioAction(white,black,SLOT(toggleBackground(bool)),1);
  } else {
    createRadioAction(white,black,SLOT(toggleBackground(bool)),2);
  }


#if QT_VERSION < 0x040000
  // === Action Menu ===
  QPopupMenu *mActions = new QPopupMenu(fContextMenu);
  fContextMenu->insertItem("&Actions",mActions);

#if QT_VERSION < 0x030200
  QAction *createEPS =  new QAction("&Save as ...","&Save as ...",CTRL+Key_S,mActions,0,true);
#else
  QAction *createEPS =  new QAction("&Save as ...",CTRL+Key_S,mActions);
#endif
  createEPS->addTo(mActions);
  QObject ::connect(createEPS, 
                    SIGNAL(activated()),
                    this,
                    SLOT(actionCreateEPS()));

#else
  // === Action Menu ===
  QMenu *mActions = fContextMenu->addMenu("&Actions");
  QAction *createEPS = mActions->addAction("Save as ...");
  QObject ::connect(createEPS, 
                    SIGNAL(triggered()),
                    this,
                    SLOT(actionCreateEPS()));
#endif



#if QT_VERSION < 0x040000
  // === Special Menu ===
  QPopupMenu *mSpecial = new QPopupMenu(fContextMenu);
  fContextMenu->insertItem("S&pecial",mSpecial);

  QPopupMenu *mTransparency = new QPopupMenu(mSpecial);
  mSpecial->insertItem("Transparency",mTransparency);

#if QT_VERSION < 0x030200
  QAction *transparencyOn = new QAction("&On","&On",CTRL+Key_O,mTransparency,0,true);
  QAction *transparencyOff = new QAction("&Off","&Off",CTRL+Key_F,mTransparency,0,true);
#else
  QAction *transparencyOn = new QAction("&On",CTRL+Key_O,mTransparency);
  QAction *transparencyOff = new QAction("&Off",CTRL+Key_F,mTransparency);
  transparencyOn->setToggleAction(true);
  transparencyOff->setToggleAction(true);
#endif
  transparencyOn->addTo(mTransparency);
  transparencyOff->addTo(mTransparency);

#else
  // === Special Menu ===
  QMenu *mSpecial = fContextMenu->addMenu("S&pecial");
  QMenu *mTransparency = mSpecial->addMenu("Transparency");
  QAction *transparencyOn = mTransparency->addAction("On");
  QAction *transparencyOff = mTransparency->addAction("Off");
#endif

  if (transparency_enabled == false) {
    createRadioAction(transparencyOn,transparencyOff,SLOT(toggleTransparency(bool)),2);
  } else if (transparency_enabled == true) {
    createRadioAction(transparencyOn,transparencyOff,SLOT(toggleTransparency(bool)),1);
  } else {
    mSpecial->clear();
  }


#if QT_VERSION < 0x040000
  QPopupMenu *mAntialiasing = new QPopupMenu(mSpecial);
  mSpecial->insertItem("Antialiasing",mAntialiasing);

#if QT_VERSION < 0x030200
  QAction *antialiasingOn = new QAction("&On","&On",CTRL+Key_O,mAntialiasing,0,true);
  QAction *antialiasingOff = new QAction("&Off","&Off",CTRL+Key_F,mAntialiasing,0,true);
#else
  QAction *antialiasingOn = new QAction("&On",CTRL+Key_O,mAntialiasing);
  QAction *antialiasingOff = new QAction("&Off",CTRL+Key_F,mAntialiasing);
  antialiasingOn->setToggleAction(true);
  antialiasingOff->setToggleAction(true);
#endif
  antialiasingOn->addTo(mAntialiasing);
  antialiasingOff->addTo(mAntialiasing);

#else
  QMenu *mAntialiasing = mSpecial->addMenu("Antialiasing");
  QAction *antialiasingOn = mAntialiasing->addAction("On");
  QAction *antialiasingOff = mAntialiasing->addAction("Off");
#endif

  if (antialiasing_enabled == false) {
    createRadioAction(antialiasingOn,antialiasingOff,SLOT(toggleAntialiasing(bool)),2);
  } else if (antialiasing_enabled == true) {
    createRadioAction(antialiasingOn,antialiasingOff,SLOT(toggleAntialiasing(bool)),1);
  } else {
    mAntialiasing->clear();
  }

#if QT_VERSION < 0x040000
  QPopupMenu *mHaloing = new QPopupMenu(mSpecial);
  mSpecial->insertItem("Haloing",mHaloing);

#if QT_VERSION < 0x030200
  QAction *haloingOn = new QAction("&On","&On",CTRL+Key_O,mHaloing,0,true);
  QAction *haloingOff = new QAction("&Off","&Off",CTRL+Key_F,mHaloing,0,true);
#else
  QAction *haloingOn = new QAction("&On",CTRL+Key_O,mHaloing);
  QAction *haloingOff = new QAction("&Off",CTRL+Key_F,mHaloing);
  haloingOn->setToggleAction(true);
  haloingOff->setToggleAction(true);
#endif
  haloingOn->addTo(mHaloing);
  haloingOff->addTo(mHaloing);
#else
  QMenu *mHaloing = mSpecial->addMenu("Haloing");
  QAction *haloingOn = mHaloing->addAction("On");
  QAction *haloingOff = mHaloing->addAction("Off");
#endif
  if (haloing_enabled == false) {
    createRadioAction(haloingOn,haloingOff,SLOT(toggleHaloing(bool)),2);
  } else if (haloing_enabled == true) {
    createRadioAction(haloingOn,haloingOff,SLOT(toggleHaloing(bool)),1);
  } else {
    mHaloing->clear();
  }

#if QT_VERSION < 0x040000
  QPopupMenu *mAux = new QPopupMenu(mSpecial);
  mSpecial->insertItem("Auxiliairy edges",mAux);

#if QT_VERSION < 0x030200
  QAction *auxOn = new QAction("&On","&On",CTRL+Key_O,mAux,0,true);
  QAction *auxOff = new QAction("&Off","&Off",CTRL+Key_F,mAux,0,true);
#else
  QAction *auxOn = new QAction("&On",CTRL+Key_O,mAux);
  QAction *auxOff = new QAction("&Off",CTRL+Key_F,mAux);
  auxOn->setToggleAction(true);
  auxOff->setToggleAction(true);
#endif
  auxOn->addTo(mAux);
  auxOff->addTo(mAux);

#else
  QMenu *mAux = mSpecial->addMenu("Auxiliary edges");
  QAction *auxOn = mAux->addAction("On");
  QAction *auxOff = mAux->addAction("Off");
#endif
  if (!fVP.IsAuxEdgeVisible()) {
    createRadioAction(auxOn,auxOff,SLOT(toggleAux(bool)),1);
  } else {
    createRadioAction(auxOn,auxOff,SLOT(toggleAux(bool)),2);
  }



#if QT_VERSION < 0x040000
  QPopupMenu *mFullScreen = new QPopupMenu(mSpecial);
  mSpecial->insertItem("Full screen",mFullScreen);

#if QT_VERSION < 0x030200
  fFullScreenOn = new QAction("&On","&On",CTRL+Key_O,mFullScreen,0,true);
  fFullScreenOff = new QAction("&Off","&Off",CTRL+Key_F,mFullScreen,0,true);
#else
  fFullScreenOn = new QAction("&On",CTRL+Key_O,mFullScreen);
  fFullScreenOff = new QAction("&Off",CTRL+Key_F,mFullScreen);
  fFullScreenOn->setToggleAction(true);
  fFullScreenOff->setToggleAction(true);
#endif
  fFullScreenOn->addTo(mFullScreen);
  fFullScreenOff->addTo(mFullScreen);
#else
  QMenu *mFullScreen = mSpecial->addMenu("&Full screen");
  fFullScreenOn = mFullScreen->addAction("On");
  fFullScreenOff = mFullScreen->addAction("Off");
#endif
  createRadioAction(fFullScreenOn,fFullScreenOff,SLOT(toggleFullScreen(bool)),2);

}

void G4OpenGLQtViewer::manageContextMenuEvent(QContextMenuEvent *e)
{
  if (!GLWindow) {
    G4cerr << "Visualization window not defined, please choose one before\n" << G4endl;
  } else {
  
    if (!fContextMenu) 
      createPopupMenu();
    
    // launch menu
    if ( fContextMenu ) {
      fContextMenu->exec( e->globalPos() );
      //    delete fContextMenu;
    }
  }
  e->accept();
}


/**
   Create a radio button menu. The two menu will be connected. When click on one,
   eatch state will be invert and callback method will be called.
   @param action1 first action to connect
   @param action2 second action to connect
   @param method callback method
   @param nCheck: 1 : first action will be set true. 2 : second action will be set true
*/
#if QT_VERSION < 0x040000
void G4OpenGLQtViewer::createRadioAction(QAction *action1,QAction *action2, const std::string& method,unsigned int nCheck) {

  if (action1->parent()->inherits("QPopupMenu")){
    ((QPopupMenu*)action1->parent())->setCheckable(true);
    ((QPopupMenu*)action2->parent())->setCheckable(true);
  }
  action1->setOn(false);
   action2->setOn(false);

  if (nCheck ==1)
    action1->setOn(true);
  else
    action2->setOn(true);
   
  //FIXME : Should not work on Qt3
  QObject ::connect(action1, SIGNAL(activated()),action2, SLOT(toggle()));
  QObject ::connect(action2, SIGNAL(activated()),action1, SLOT(toggle()));

  QObject ::connect(action1, SIGNAL(toggled(bool)),this, method.c_str());
}

#else
void G4OpenGLQtViewer::createRadioAction(QAction *action1,QAction *action2, const std::string& method,unsigned int nCheck) {

  action1->setCheckable(true);
  action2->setCheckable(true);

  if (nCheck ==1)
    action1->setChecked (true);
  else
    action2->setChecked (true);
   
  QObject ::connect(action1, SIGNAL(triggered(bool)),action2, SLOT(toggle()));
  QObject ::connect(action2, SIGNAL(triggered(bool)),action1, SLOT(toggle()));

  QObject ::connect(action1, SIGNAL(toggled(bool)),this, method.c_str());

}
#endif

/**
   Slot activate when mouseAction->rotate menu is set 
 */
void G4OpenGLQtViewer::actionMouseRotate() {
  emit toggleMouseAction(STYLE1);
}


/**
   Slot activate when mouseAction->rotate menu is set 
 */
void G4OpenGLQtViewer::actionMouseMove() {
  emit toggleMouseAction(STYLE2);
}


/**
   Slot activate when mouseAction->zoom menu is set 
 */
void G4OpenGLQtViewer::actionMousePick() {
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::actionMousePick \n");
#endif
  emit toggleMouseAction(STYLE3);
}

/**
   Slot activate when mouseAction->zoom menu is set 
 */
void G4OpenGLQtViewer::actionMouseShortcuts() {
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::actionMouseShortcuts \n");
#endif
  emit toggleMouseAction(STYLE4);
}

/**
   Slot activate when drawing->wireframe menu is set 
 */
void G4OpenGLQtViewer::actionDrawingWireframe() {
  emit toggleDrawingAction(1);
}

/**
   Slot activate when drawing->line removal menu is set 
 */
void G4OpenGLQtViewer::actionDrawingLineRemoval() {
  emit toggleDrawingAction(2);
}

/**
   Slot activate when drawing->surface removal menu is set 
 */
void G4OpenGLQtViewer::actionDrawingSurfaceRemoval() {
  emit toggleDrawingAction(3);
}

/**
   Slot activate when drawing->wireframe menu is set 
 */
void G4OpenGLQtViewer::actionDrawingLineSurfaceRemoval() {
  emit toggleDrawingAction(4);
}


/**
   Slot activated when mouse action is toggle
   @param aAction : STYLE1, STYLE2, STYLE3
 */
void G4OpenGLQtViewer::toggleMouseAction(mouseActions aAction) {
  
  if ((aAction == STYLE1) || //initialize all
      (aAction == STYLE2) ||
      (aAction == STYLE3) ||
      (aAction == STYLE4))  {
#if QT_VERSION < 0x040000
    fRotateAction->setOn (false);
    fMoveAction->setOn (false);
    fPickAction->setOn (false);
    fShortcutsAction->setOn (false);
#else
    fRotateAction->setChecked (false);
    fMoveAction->setChecked (false);
    fPickAction->setChecked (false);
    fShortcutsAction->setChecked (false);
#endif
    fVP.SetPicking(false);
    fMouseAction = aAction;
  }
  // rotate
  if (aAction == STYLE1) {  // rotate
    G4cout << "Click and move mouse to rotate volume \n" << G4endl;
    G4cout << "Press left/right arrows to move volume left/right\n" << G4endl;
    G4cout << "Press up/down arrows to move volume up/down\n" << G4endl;
    G4cout << "Press ALT+up/down arrows to move volume toward/forward\n" << G4endl;
    G4cout << "Press SHIFT+left/right arrows to rotate volume left/right\n" << G4endl;
    G4cout << "Press SHIFT+up/down arrows to rotate volume up/down\n" << G4endl;
    G4cout << "Press +/- to zoom into volume\n" << G4endl;
#if QT_VERSION < 0x040000
    fRotateAction->setOn (true);
#else
    fRotateAction->setChecked (true);
#endif
  } else  if (aAction == STYLE2) { //move
    G4cout << "Move camera point of view with mouse\n" << G4endl;
#if QT_VERSION < 0x040000
    fMoveAction->setOn (true);
#else
    fMoveAction->setChecked (true);
#endif
  } else  if (aAction == STYLE3) { //pick
    G4cout << "Click and pick \n" << G4endl;
#if QT_VERSION < 0x040000
    fPickAction->setOn (true);
#else
    fPickAction->setChecked (true);
#endif
    fVP.SetPicking(true);
  } else  if (aAction == STYLE4) {  // display shortcuts
#if QT_VERSION < 0x040000
    fShortcutsAction->setOn (true);
#else
    fShortcutsAction->setChecked (true);
#endif

    G4cout << "Click and move mouse to rotate volume \n" << G4endl;
    G4cout << "Press left/right arrows to move volume left/right\n" << G4endl;
    G4cout << "Press up/down arrows to move volume up/down\n" << G4endl;
    G4cout << "Press CONTROL+up/down arrows to move volume toward/forward\n" << G4endl;
    G4cout << "Press SHIFT+left/right arrows to rotate volume left/right\n" << G4endl;
    G4cout << "Press SHIFT+up/down arrows to rotate volume up/down\n" << G4endl;
    G4cout << "Press +/- to zoom into volume\n" << G4endl;
  }

}
/**
   Slot activated when drawing menu is toggle
   Warning : When G4OpenGLStoredQtViewer::DrawView() method call,
   KernelVisitDecision () will be call and will set the fNeedKernelVisit
   to 1. See G4XXXStoredViewer::CompareForKernelVisit for explanations.
   It will cause a redraw of the view
   @param aAction : 1 wireframe, 2 line removal, 3 surface removal, 4 line & surface removal
   @see G4OpenGLStoredQtViewer::DrawView
   @see G4XXXStoredViewer::CompareForKernelVisit
 */
void G4OpenGLQtViewer::toggleDrawingAction(int aAction) {

  G4ViewParameters::DrawingStyle d_style;
  

  if (aAction ==1) {
#if QT_VERSION < 0x040000
    fDrawingWireframe->setItemChecked (0,true);
    fDrawingLineRemoval->setItemChecked (0,false);
    fDrawingSurfaceRemoval->setItemChecked (0,false);
    fDrawingLineSurfaceRemoval->setItemChecked (0,false);
#else
    fDrawingWireframe->setChecked (true);
    fDrawingLineRemoval->setChecked (false);
    fDrawingSurfaceRemoval->setChecked (false);
    fDrawingLineSurfaceRemoval->setChecked (false);
#endif

    d_style = G4ViewParameters::wireframe;

  } else  if (aAction ==2) {
#if QT_VERSION < 0x040000
    fDrawingWireframe->setItemChecked (0,false);
    fDrawingLineRemoval->setItemChecked (0,true);
    fDrawingSurfaceRemoval->setItemChecked (0,false);
    fDrawingLineSurfaceRemoval->setItemChecked (0,false);
#else
    fDrawingWireframe->setChecked (false);
    fDrawingLineRemoval->setChecked (true);
    fDrawingSurfaceRemoval->setChecked (false);
    fDrawingLineSurfaceRemoval->setChecked (false);
#endif

    d_style = G4ViewParameters::hlr;

  } else  if (aAction ==3) {
#if QT_VERSION < 0x040000
    fDrawingWireframe->setItemChecked (0,false);
    fDrawingLineRemoval->setItemChecked (0,false);
    fDrawingSurfaceRemoval->setItemChecked (0,true);
    fDrawingLineSurfaceRemoval->setItemChecked (0,false);
#else
    fDrawingWireframe->setChecked (false);
    fDrawingLineRemoval->setChecked (false);
    fDrawingSurfaceRemoval->setChecked (true);
    fDrawingLineSurfaceRemoval->setChecked (false);
#endif

    d_style = G4ViewParameters::hsr;

  } else  if (aAction ==4) {
#if QT_VERSION < 0x040000
    fDrawingWireframe->setItemChecked (0,false);
    fDrawingLineRemoval->setItemChecked (0,false);
    fDrawingSurfaceRemoval->setItemChecked (0,false);
    fDrawingLineSurfaceRemoval->setItemChecked (0,true);
#else
    fDrawingWireframe->setChecked (false);
    fDrawingLineRemoval->setChecked (false);
    fDrawingSurfaceRemoval->setChecked (false);
    fDrawingLineSurfaceRemoval->setChecked (true);
#endif
    d_style = G4ViewParameters::hlhsr;
  }
  fVP.SetDrawingStyle(d_style);

  updateQWidget();
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::toggleDrawingAction\n");
#endif
}


/**
   SLOT Activate by a click on the representation menu
   Warning : When G4OpenGLStoredQtViewer::DrawView() method call,
   KernelVisitDecision () will be call and will set the fNeedKernelVisit
   to 1. See G4XXXStoredViewer::CompareForKernelVisit for explanations.
   It will cause a redraw of the view
   @param check : 1 polyhedron, 0 nurbs
   @see G4OpenGLStoredQtViewer::DrawView
   @see G4XXXStoredViewer::CompareForKernelVisit
*/
void G4OpenGLQtViewer::toggleRepresentation(bool check) {

  G4ViewParameters::RepStyle style;
  if (check == 1) {
    style = G4ViewParameters::polyhedron;
  } else {
    style = G4ViewParameters::nurbs;
  }
  fVP.SetRepStyle (style);

  updateQWidget();
}

/**
   SLOT Activate by a click on the projection menu
   Warning : When G4OpenGLStoredQtViewer::DrawView() method call,
   KernelVisitDecision () will be call and will set the fNeedKernelVisit
   to 1. See G4XXXStoredViewer::CompareForKernelVisit for explanations.
   It will cause a redraw of the view
   @param check : 1 orthographic, 2 perspective
   @see G4OpenGLStoredQtViewer::DrawView
   @see G4XXXStoredViewer::CompareForKernelVisit
*/
void G4OpenGLQtViewer::toggleProjection(bool check) {

  if (check == 1) {
    fVP.SetFieldHalfAngle (0);
  } else {

    // look for the default parameter hidden in G4UIcommand parameters
    G4UImanager* UI = G4UImanager::GetUIpointer();
    if(UI==NULL)
      return;
    G4UIcommandTree * treeTop = UI->GetTree();

    // find command
    G4UIcommand* command = treeTop->FindPath("/vis/viewer/set/projection");
    if (!command)
      return;

    // find param
    G4UIparameter * angleParam = NULL;
    for(G4int i=0;  i<command->GetParameterEntries(); i++)
    {
      if( command->GetParameter(i)->GetParameterName() == "field-half-angle" ) {
        angleParam = command->GetParameter(i);
      }
    }
    if (!angleParam)
      return;

    // find unit
    G4UIparameter * unitParam = NULL;
    for(G4int i=0;  i<command->GetParameterEntries(); i++)
    {
      if( command->GetParameter(i)->GetParameterName() == "unit" ) {
        unitParam = command->GetParameter(i);
      }
    }
    if (!unitParam)
      return;

    G4double defaultValue = command->ConvertToDouble(angleParam->GetDefaultValue())
                            * G4UnitDefinition::GetValueOf(unitParam->GetDefaultValue()); 
    if (defaultValue > 89.5 || defaultValue <= 0.0) {
      G4cerr << "Field half angle should be 0 < angle <= 89.5 degrees. Check your default Field half angle parameter\n";
    } else {
      G4cout << "Perspective view has been set to default value. Field half angle="<<angleParam->GetDefaultValue() <<" \n" << G4endl;
      fVP.SetFieldHalfAngle (defaultValue);
      SetView ();
    }
  }  
  updateQWidget();
}


/**
   SLOT Activate by a click on the background menu
@param check : 1 white, 0 black
*/
void G4OpenGLQtViewer::toggleBackground(bool check) {

  //   //I need to revisit the kernel if the background colour changes and
  //   //hidden line removal is enabled, because hlr drawing utilises the
  //   //background colour in its drawing...
  //   // (Note added by JA 13/9/2005) Background now handled in view
  //   // parameters.  A kernel visit is triggered on change of background.
  if (check == 1) {
    ((G4ViewParameters&)this->GetViewParameters()).
      SetBackgroundColour(G4Colour(1.,1.,1.));  // White
  } else {
    ((G4ViewParameters&)this->GetViewParameters()).
      SetBackgroundColour(G4Colour(0.,0.,0.));  // Black
  }
  updateQWidget();
}

/**
   SLOT Activate by a click on the transparency menu
@param check : 1 , 0
*/
void G4OpenGLQtViewer::toggleTransparency(bool check) {
  
  if (check) {
    transparency_enabled = false;
  } else {
    transparency_enabled = true;
  }
  SetNeedKernelVisit (true);
  updateQWidget();
}

/**
   SLOT Activate by a click on the antialiasing menu
@param check : 1 , 0
*/
void G4OpenGLQtViewer::toggleAntialiasing(bool check) {

  if (!check) {
    antialiasing_enabled = false;
    glDisable (GL_LINE_SMOOTH);
    glDisable (GL_POLYGON_SMOOTH);
  } else {
    antialiasing_enabled = true;
    glEnable (GL_LINE_SMOOTH);
    glHint (GL_LINE_SMOOTH_HINT, GL_NICEST);
    glEnable (GL_POLYGON_SMOOTH);
    glHint (GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  }

  updateQWidget();
}

/**
   SLOT Activate by a click on the haloing menu
@param check : 1 , 0
*/
//FIXME : I SEE NOTHING...
void G4OpenGLQtViewer::toggleHaloing(bool check) {
  if (check) {
    haloing_enabled = false;
  } else {
    haloing_enabled = true;
  }

  updateQWidget();

}

/**
   SLOT Activate by a click on the auxiliaire edges menu
@param check : 1 , 0
*/
void G4OpenGLQtViewer::toggleAux(bool check) {
  if (check) {
    fVP.SetAuxEdgeVisible(false);
  } else {
    fVP.SetAuxEdgeVisible(true);
  }
  SetNeedKernelVisit (true);
  updateQWidget();

}

/**
   SLOT Activate by a click on the full screen menu
*/
void G4OpenGLQtViewer::toggleFullScreen(bool check) {
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::toggleFullScreen checked:%d() \n",check);
#endif
  if (check != GLWindow->isFullScreen()) { //toggle
#if QT_VERSION >= 0x030200
#if QT_VERSION < 0x040000
    GLWindow->setWindowState(GLWindow->windowState() ^ Qt::WindowFullScreen);
//     fFullScreenOn->setOn(check);
//     fFullScreenOff->setOn(!check);
#else
    GLWindow->setWindowState(GLWindow->windowState() ^ Qt::WindowFullScreen);
//     fFullScreenOn->setChecked(check);
//     fFullScreenOff->setChecked(!check);
#endif
#else
    G4cerr << "This version of Qt could not do fullScreen. Resizing the widget is the only solution available.\n" << G4endl;
#endif
  }
}

void G4OpenGLQtViewer::actionCreateEPS() {
  QString filters;
#if QT_VERSION < 0x040000
  QStrList listFormat=QImageIO::outputFormats();
  char *tmp=listFormat.first();
  while (tmp!=0) {
    filters += QString(tmp) + ";;";
    tmp=listFormat.next();
  }
#else
  QList<QByteArray> formats =  QImageWriter::supportedImageFormats ();
  for (int i = 0; i < formats.size(); ++i) {
    filters +=formats.at(i) + ";;";
  }
#endif
  filters += "eps;;";
  filters += "ps;;";
  filters += "pdf";
  QString* selectedFormat = new QString();
#if QT_VERSION < 0x040000
  QString nomFich =  QFileDialog::getSaveFileName ( ".",
                                                    filters,
                                                    GLWindow,
                                                    "Save file dialog",
                                                    tr("Save as ..."),
                                                    selectedFormat ); 
#else
  QString nomFich =  QFileDialog::getSaveFileName ( GLWindow,
                                                    tr("Save as ..."),
                                                    ".",
                                                    filters,
                                                    selectedFormat ); 
#endif
  // bmp jpg jpeg png ppm xbm xpm
  if (nomFich == "") {
    return;
  }
#if QT_VERSION < 0x040000
  nomFich += "."+QString(selectedFormat->ascii());
  QString format = selectedFormat->lower();
#else
  nomFich += "."+QString(selectedFormat->toStdString().c_str());
  QString format = selectedFormat->toLower();
#endif
  G4OpenGLQtExportDialog* exportDialog= new G4OpenGLQtExportDialog(GLWindow,format,fWindow->height(),fWindow->width());
  if(  exportDialog->exec()) {

    QImage image;
    bool res = false;
    if ((exportDialog->getWidth() !=fWindow->width()) ||
        (exportDialog->getHeight() !=fWindow->height())) {
      if (format != QString("eps")) {
      G4cerr << "Export->Change Size : This function is not implemented, to export in another size, please resize your frame to what you need\n" << G4endl;
      
      //    rescaleImage(exportDialog->getWidth(),exportDialog->getHeight());// re-scale image
      //      QGLWidget* glResized = fWindow;

      // FIXME :
      // L.Garnier : I've try to implement change size function, but the problem is 
      // the renderPixmap function call the QGLWidget to resize and it doesn't draw
      // the content of this widget... It only draw the background.

      //      fWindow->renderPixmap (exportDialog->getWidth()*2,exportDialog->getHeight()*2,true );

      //      QPixmap pixmap = fWindow->renderPixmap ();
      
      //      image = pixmap->toImage();
      //      glResized->resize(exportDialog->getWidth()*2,exportDialog->getHeight()*2);
      //      image = glResized->grabFrameBuffer();
      }      
    } else {
      image = fWindow->grabFrameBuffer();
    }    
    if (format == QString("eps")) {
      if (exportDialog->getVectorEPS()) {
        res = generateVectorEPS(nomFich,exportDialog->getWidth(),exportDialog->getHeight(),image);
      } else {
        res = generateEPS(nomFich,exportDialog->getNbColor(),image);
      }
    } else if ((format == "ps") || (format == "pdf")) {
      res = generatePS_PDF(nomFich,exportDialog->getNbColor(),image);
    } else if ((format == "tif") ||
               (format == "tiff") ||
               (format == "jpg") ||
               (format == "jpeg") ||
               (format == "png") ||
               (format == "pbm") ||
               (format == "pgm") ||
               (format == "ppm") ||
               (format == "bmp") ||
               (format == "xbm") ||
               (format == "xpm")) {
#if QT_VERSION < 0x040000
      res = image.save(nomFich,selectedFormat->ascii(),exportDialog->getSliderValue());
#else
      res = image.save(nomFich,0,exportDialog->getSliderValue());
#endif
    } else {
      G4cerr << "This version of G4UI Could not generate the selected format\n" << G4endl;
    }
    if (res == false) {
#if QT_VERSION < 0x040000
      G4cerr << "Error while saving file... "<<nomFich.ascii()<<"\n" << G4endl;
#else
      G4cerr << "Error while saving file... "<<nomFich.toStdString().c_str()<<"\n" << G4endl;
#endif
    } else {
#if QT_VERSION < 0x040000
      G4cout << "File "<<nomFich.ascii()<<" has been saved \n" << G4endl;
#else
      G4cout << "File "<<nomFich.toStdString().c_str()<<" has been saved \n" << G4endl;
#endif
    }
    
  } else { // cancel selected
    return;
  }
  
#ifdef GEANT4_QT_DEBUG
  printf("G4OpenGLQtViewer::actionCreateEPS() \n");
#endif
}

/*
// http://www.google.com/codesearch?hl=en&q=+jpg+Qt+quality+QDialog+show:FZkUoth8oiw:TONpW2mR-_c:tyTfrKMO-xI&sa=N&cd=2&ct=rc&cs_p=http://soft.proindependent.com/src/qtiplot-0.8.9.zip&cs_f=qtiplot-0.8.9/qtiplot/src/application.cpp#a0

void Graph::exportToSVG(const QString& fname)
{
  // enable workaround for Qt3 misalignments
  QwtPainter::setSVGMode(true);
  QPicture picture;
  QPainter p(&picture);
  d_plot->print(&p, d_plot->rect());
  p.end();

  picture.save(fname, "svg");
}
*/




/**
   Save the current mouse press point
   @param p mouse click point
*/
void G4OpenGLQtViewer::G4MousePressEvent(QPoint p)
{
  fLastPos = p;
  if (fMouseAction == STYLE2){  // pick
    Pick(p.x(),p.y());
  }
}


/**
   @param pos_x mouse x position
   @param pos_y mouse y position
   @param mButtons mouse button active
*/

#if QT_VERSION < 0x040000
void G4OpenGLQtViewer::G4MouseEvent(int pos_x, int pos_y,Qt::ButtonState mButtons)
#else
void G4OpenGLQtViewer::G4MouseEvent(int pos_x, int pos_y,Qt::MouseButtons mButtons)
#endif
{

  int dx = fLastPos.x() - pos_x;
  int dy = fLastPos.y() - pos_y;


  if (fMouseAction == STYLE1) {  // rotate
    if (mButtons & Qt::LeftButton) {
      G4MouseRotateEvent(dx,dy);
    }
  } else if (fMouseAction == STYLE2) {  // move
    if (mButtons & Qt::LeftButton) {
      G4double coef = ((G4double)getSceneNearWidth())/((G4double)WinSize_x);
      if (WinSize_y <WinSize_x) {
        coef = ((G4double)getSceneNearWidth())/((G4double)WinSize_y);
      }
      fVP.IncrementPan(dx*coef,-dy*coef);
      updateQWidget();
    }
  }
  fLastPos = QPoint(pos_x, pos_y);
}

/**
   Move the scene of dx, dy, dz values.
   @param dx delta mouse x position
   @param dy delta mouse y position
*/

void G4OpenGLQtViewer::G4MouseMoveEvent(G4double dx,G4double dy, G4double dz)
{
  GLdouble coefTrans = getSceneNearWidth()*fDeltaSceneTranslation;
  GLdouble coefDepth = getSceneDepth()*fDeltaDepth;

  fVP.IncrementPan(-dx*coefTrans,dy*coefTrans,dz*coefDepth);
 
  updateQWidget();
}


/**
   @param dx delta mouse x position
   @param dy delta mouse y position
*/

void G4OpenGLQtViewer::G4MouseRotateEvent(G4double dx, G4double dy)
{
  dx = dx * fDeltaRotation;
  dy = dy * fDeltaRotation;

  //phi spin stuff here
  
  G4Vector3D vp = fVP.GetViewpointDirection ().unit ();
  G4Vector3D up = fVP.GetUpVector ().unit ();
  
  G4Vector3D yprime = (up.cross(vp)).unit();
  G4Vector3D zprime = (vp.cross(yprime)).unit();
  
  G4double delta_alpha;
  G4double delta_theta;
  
  if (fVP.GetLightsMoveWithCamera()) {
    delta_alpha = dy;
    delta_theta = -dx;
  } else {
    delta_alpha = -dy;
    delta_theta = dx;
  }    
  
  delta_alpha *= deg;
  delta_theta *= deg;
  
  G4Vector3D new_vp = std::cos(delta_alpha) * vp + std::sin(delta_alpha) * zprime;
  
  G4Vector3D new_up;
  if (fVP.GetLightsMoveWithCamera()) {
    new_up = (new_vp.cross(yprime)).unit();
    fVP.SetUpVector(new_up);
  } else {
    new_up = up;
  }
  ////////////////
  // Rotates by fixed azimuthal angle delta_theta.
  
  G4double cosalpha = new_up.dot (new_vp.unit());
  G4double sinalpha = std::sqrt (1. - std::pow (cosalpha, 2));
  yprime = (new_up.cross (new_vp.unit())).unit ();
  G4Vector3D xprime = yprime.cross (new_up);
  // Projection of vp on plane perpendicular to up...
  G4Vector3D a1 = sinalpha * xprime;
  // Required new projection...
  G4Vector3D a2 =
    sinalpha * (std::cos (delta_theta) * xprime + std::sin (delta_theta) * yprime);
  // Required Increment vector...
  G4Vector3D delta = a2 - a1;
  // So new viewpoint is...
  G4Vector3D viewPoint = new_vp.unit() + delta;
  
  fVP.SetViewAndLights (viewPoint);
  updateQWidget();
  
}

/** This is the benning of a rescale function. It does nothing for the moment
    @param aWidth : new width
    @param aHeight : new height
*/
void G4OpenGLQtViewer::rescaleImage(
 int aWidth
,int aHeight
){
#ifdef GEANT4_QT_DEBUG
  printf("should rescale \n");
#endif
  GLfloat* feedback_buffer;
  GLint returned;
  FILE* file;
  
//   feedback_buffer = new GLfloat[size];
//   glFeedbackBuffer (size, GL_3D_COLOR, feedback_buffer);
//   glRenderMode (GL_FEEDBACK);
  
//   glViewport (0, 0, aWidth, aHeight);
//   DrawView();
//   returned = glRenderMode (GL_RENDER);

}

/**
   Generate Vectorial Encapsulated Postscript form image
   @param aFilename : name of file
   @param aInColor : numbers of colors : 1->BW 2->RGB 3->RGB+Alpha
   @param aImage : Image to print
*/
bool G4OpenGLQtViewer::generateVectorEPS (
 QString aFilename
,int aWidth
,int aHeight
,QImage aImage
)
{
  // Print vectored PostScript
  
  G4int size = 5000000;

  GLfloat* feedback_buffer;
  GLint returned;
  FILE* file;
  
  feedback_buffer = new GLfloat[size];
  glFeedbackBuffer (size, GL_3D_COLOR, feedback_buffer);
  glRenderMode (GL_FEEDBACK);
  
  int side = aWidth;
  if (aHeight < aWidth) side = aHeight;
  glViewport((aWidth - side) / 2, (aHeight - side) / 2, side, side);
  DrawView();

  returned = glRenderMode (GL_RENDER);
  
  
#if QT_VERSION < 0x040000
  file = fopen (aFilename.ascii(), "w");
#else
  file = fopen (aFilename.toStdString().c_str(), "w");
#endif
  if (file) {
    spewWireframeEPS (file, returned, feedback_buffer, "rendereps");
  } else {
#if QT_VERSION < 0x040000
    G4cerr << "Could not open "<< aFilename.ascii()<<"\n" << G4endl;
#else
    G4cerr << "Could not open "<< aFilename.toStdString().c_str()<<"\n" << G4endl;
#endif
  }
  
  delete[] feedback_buffer;

  return true;
}

/**
   Generate Encapsulated Postscript form image
   @param aFilename : name of file
   @param aInColor : numbers of colors : 1->BW 2->RGB 3->RGB+Alpha
   @param aImage : Image to print
*/
bool G4OpenGLQtViewer::generateEPS (
 QString aFilename
,int aInColor
,QImage aImage
)
{
  // FIXME
#ifdef GEANT4_QT_DEBUG
  printf("saving EPS\n");
#endif

  FILE* fp;

  if (aImage.bits () == NULL)
    return false;
  
#if QT_VERSION < 0x040000
  fp = fopen (aFilename.ascii(), "w");
#else
  fp = fopen (aFilename.toStdString().c_str(), "w");
#endif
  if (fp == NULL) {
    return false;
  }
  
  fprintf (fp, "%%!PS-Adobe-2.0 EPSF-1.2\n");
#if QT_VERSION < 0x040000
  fprintf (fp, "%%%%Title: %s\n", aFilename.ascii());
#else
  fprintf (fp, "%%%%Title: %s\n", aFilename.toStdString().c_str());
#endif
  fprintf (fp, "%%%%Creator: OpenGL pixmap render output\n");
  fprintf (fp, "%%%%BoundingBox: 0 0 %d %d\n", aImage.width(), aImage.height());
  fprintf (fp, "%%%%EndComments\n");
  fprintf (fp, "gsave\n");
  fprintf (fp, "/bwproc {\n");
  fprintf (fp, "    rgbproc\n");
  fprintf (fp, "    dup length 3 idiv string 0 3 0 \n");
  fprintf (fp, "    5 -1 roll {\n");
  fprintf (fp, "    add 2 1 roll 1 sub dup 0 eq\n");
  fprintf (fp, "    { pop 3 idiv 3 -1 roll dup 4 -1 roll dup\n");
  fprintf (fp, "       3 1 roll 5 -1 roll } put 1 add 3 0 \n");
  fprintf (fp, "    { 2 1 roll } ifelse\n");
  fprintf (fp, "    }forall\n");
  fprintf (fp, "    pop pop pop\n");
  fprintf (fp, "} def\n");
  fprintf (fp, "systemdict /colorimage known not {\n");
  fprintf (fp, "   /colorimage {\n");
  fprintf (fp, "       pop\n");
  fprintf (fp, "       pop\n");
  fprintf (fp, "       /rgbproc exch def\n");
  fprintf (fp, "       { bwproc } image\n");
  fprintf (fp, "   }  def\n");
  fprintf (fp, "} if\n");
  fprintf (fp, "/picstr %d string def\n", aImage.width() * aInColor);
  fprintf (fp, "%d %d scale\n", aImage.width(), aImage.height());
  fprintf (fp, "%d %d %d\n", aImage.width(), aImage.height(), 8);
  fprintf (fp, "[%d 0 0 %d 0 0]\n", aImage.width(), aImage.height());
  fprintf (fp, "{currentfile picstr readhexstring pop}\n");
  fprintf (fp, "false %d\n", aInColor);
  fprintf (fp, "colorimage\n");
  

  int width = aImage.width();
  int height = aImage.height();
  int depth = aImage.depth();
  int size = width*height;
  
  if (depth == 1)
    size = (width+7)/8*height;
  else if (aInColor == 1)
    size = size*3;
  
  int i = 0;
  //  if ( aInColor ==1 ) {
  // FIXME : L. Garnier. For the moment 10 dec 2007, I could not find a way
  // to save correctly grayscale Image. I mean that color or grayscale image
  // have the same file save size !
  
  /* } else*/ if (depth == 8) {
    for(int y=height-1; y >=0 ; y--) {
      const uchar * s = aImage.scanLine(y);
      for(int x=0; x <width; x++) {
        QRgb rgb = aImage.color(s[x]);
        if (aInColor == 1) {
          fprintf (fp, " %02hx ",(unsigned char)qGray(rgb));
          i++;
        } else {
          fprintf (fp, " %02hx %02hx %02hx",
                   (unsigned char) qRed(rgb),
                   (unsigned char) qGreen(rgb),
                   (unsigned char) qBlue(rgb));
          i += 3;
        }
      }
      fprintf (fp, "\n");
    }
  } else {
#if QT_VERSION < 0x040000
    bool alpha = aImage.hasAlphaBuffer();
#else
    bool alpha = aImage.hasAlphaChannel();
    for(int y=height-1; y >=0 ; y--) {
      QRgb * s = (QRgb*)(aImage.scanLine(y));
      for(int x=0; x <width; x++) {
        QRgb rgb = (*s++);
        if (alpha && qAlpha(rgb) < 0x40) // 25% alpha, convert to white -
          rgb = qRgb(0xff, 0xff, 0xff);
        if (aInColor == 1) {
          fprintf (fp, " %02hx ",(unsigned char)qGray(rgb));
          i++;
        } else {
          fprintf (fp, " %02hx %02hx %02hx",
                   (unsigned char) qRed(rgb),
                   (unsigned char) qGreen(rgb),
                   (unsigned char) qBlue(rgb));
          i += 3;
        }
      }
      fprintf (fp, "\n");
    } 
#endif

  }

  fprintf (fp, "grestore\n");
  fprintf (fp, "showpage\n");
  fclose (fp);

  return true;
}
/**
   Generate Postscript or PDF form image
   @param aFilename : name of file
   @param aInColor : numbers of colors : 1->BW 2->RGB
   @param aImage : Image to print
*/
bool G4OpenGLQtViewer::generatePS_PDF (
 QString aFilename
,int aInColor
,QImage aImage
)
{

#if QT_VERSION < 0x040000
#ifdef Q_WS_MAC || Q_WS_X11
  QPrinter printer;
  //  printer.setPageSize(pageSize);
  if (aInColor == 1) {
    printer.setColorMode(QPrinter::GrayScale);
  } else {
    printer.setColorMode(QPrinter::Color);
  }

  /* FIXME : I don't know which format it will save...
     if (aFilename.endsWith(".ps")) {
     printer.setOutputFormat(QPrinter::PostScriptFormat);
     } else {
     printer.setOutputFormat(QPrinter::PdfFormat);
     }
  */
  printer.setOutputFileName(aFilename);
  //  printer.setFullPage ( true);
  QPainter paint(&printer);
  paint.drawImage (0,0,aImage );
  paint.end();
#else
  G4cerr << "This fonction is only supported on Mac OsX or X11 with Qt3. Full platform supported with Qt4\n" << G4endl;
#endif
#else
  QPrinter printer;
  //  printer.setPageSize(pageSize);

  // FIXME : L. Garnier 4/12/07
  // This is not working, it does nothing. Image is staying in color mode
  // So I have desactivate the B/W button in GUI
  if ((!aImage.isGrayscale ()) &&(aInColor ==1 )) {
#if QT_VERSION < 0x040000
    aImage = aImage.convertDepth(1,Qt::MonoOnly);
#else
    aImage = aImage.convertToFormat ( aImage.format(), Qt::MonoOnly);
#endif
  }


  if (aFilename.endsWith(".ps")) {
#if QT_VERSION > 0x040200
    printer.setOutputFormat(QPrinter::PostScriptFormat);
#endif
  } else {
#if QT_VERSION > 0x040100
    printer.setOutputFormat(QPrinter::PdfFormat);
#endif
  }
#if QT_VERSION > 0x040100
  printer.setOutputFileName(aFilename);
#endif
  //  printer.setFullPage ( true);
  QPainter paint(&printer);
  paint.drawImage (0,0,aImage);
  paint.end();
#endif
  return true;
}


void G4OpenGLQtViewer::G4keyPressEvent (QKeyEvent * event) 
{
  if (holdKeyEvent)
    return;

  holdKeyEvent = true;

#if QT_VERSION < 0x040000
  if ((event->key() == Qt::Key_Down) && (event->state() & Qt::AltButton )) { // go backward
#else
  if ((event->key() == Qt::Key_Down) && (event->modifiers() & Qt::AltModifier )) { // go backward
#endif
    
    G4MouseMoveEvent(0,0,1);
  }
#if QT_VERSION < 0x040000
  else if ((event->key() == Qt::Key_Up) && (event->state() & Qt::AltButton)) { // go forward
#else
  else if ((event->key() == Qt::Key_Up) && (event->modifiers() & Qt::AltModifier)) { // go forward
#endif
    G4MouseMoveEvent(0,0,-1);
  }
#if QT_VERSION < 0x040000
  if ((event->key() == Qt::Key_Down) && (event->state() & Qt::ShiftButton)) { // rotate phi
#else
  if ((event->key() == Qt::Key_Down) && (event->modifiers() & Qt::ShiftModifier)) { // rotate phi
#endif
    G4MouseRotateEvent(0,-1);
  }
#if QT_VERSION < 0x040000
  else if ((event->key() == Qt::Key_Up) && (event->state() & Qt::ShiftButton)) { // rotate phi
#else
  else if ((event->key() == Qt::Key_Up) && (event->modifiers() & Qt::ShiftModifier)) { // rotate phi
#endif
    G4MouseRotateEvent(0,1);
  }
#if QT_VERSION < 0x040000
  if ((event->key() == Qt::Key_Left) && (event->state() & Qt::ShiftButton)) { // rotate theta
#else
  if ((event->key() == Qt::Key_Left) && (event->modifiers() & Qt::ShiftModifier)) { // rotate theta
#endif
    G4MouseRotateEvent(1,0);
  }
#if QT_VERSION < 0x040000
  else if ((event->key() == Qt::Key_Right) && (event->state() & Qt::ShiftButton)) { // rotate theta
#else
  else if ((event->key() == Qt::Key_Right) && (event->modifiers() & Qt::ShiftModifier)) { // rotate theta
#endif
    G4MouseRotateEvent(-1,0);
  }
  if (event->key() == Qt::Key_Escape) { // escaped from full screen
#if QT_VERSION >= 0x030200
    if (GLWindow->isFullScreen()) {
#if QT_VERSION < 0x040000
      fFullScreenOn->activated();
#else
      fFullScreenOn->trigger();
#endif
    }
#endif
  }
  if (event->key() == Qt::Key_Plus) { // zoom in
    fVP.SetZoomFactor(fVP.GetZoomFactor()*(1+fDeltaZoom)); 
    updateQWidget();
  }
  else if (event->key() == Qt::Key_Minus) { // zoom out
    fVP.SetZoomFactor(fVP.GetZoomFactor()*(1-fDeltaZoom)); 
    updateQWidget();
  }
  // with no modifiers
#if QT_VERSION < 0x040000
  if (event->state() == Qt::NoButton) {
#else
  if (event->modifiers() == Qt::NoModifier) {
#endif
    if (event->key() == Qt::Key_Down) { // go down
      G4MouseMoveEvent(0,1,0);
    }
    else if (event->key() == Qt::Key_Up) {  // go up
      G4MouseMoveEvent(0,-1,0);
    }
    if (event->key() == Qt::Key_Left) { // go left
      G4MouseMoveEvent(-1,0,0);
    }
    else if (event->key() == Qt::Key_Right) { // go right
      G4MouseMoveEvent(1,0,0);
    }
  }
  holdKeyEvent = false;
}

/*
  
void MultiLayer::exportToSVG(const QString& fname)
{
  QPicture picture;
  QPainter p(&picture);
  for (int i=0;i<(int)graphsList->count();i++)
    {
      Graph *gr=(Graph *)graphsList->at(i);
      Plot *myPlot= (Plot *)gr->plotWidget();
      
      QPoint pos=gr->pos();
      
      int width=int(myPlot->frameGeometry().width());
      int height=int(myPlot->frameGeometry().height());
      
      myPlot->print(&p, QRect(pos,QSize(width,height)));
    }
  
  p.end();
  picture.save(fname, "svg");
}
*/
#endif
