#include "ELYSE/DetectorConstruction.hh"

// Geant 4
#include "G4MaterialTable.hh"
#include "G4ElementTable.hh"
#include "G4Box.hh"
#include "G4SubtractionSolid.hh"
#include "G4VPhysicalVolume.hh"
#include "G4OpticalSurface.hh"
#include "G4LogicalVolume.hh"
#include "G4LogicalSkinSurface.hh"
#include "G4LogicalBorderSurface.hh"
#include "G4PVPlacement.hh"
#include "G4SDManager.hh"
#include "G4Material.hh"
#include "G4Element.hh"

#include "G4GeometryManager.hh"
#include "G4PhysicalVolumeStore.hh"
#include "G4LogicalVolumeStore.hh"
#include "G4SolidStore.hh"
#include "G4RunManager.hh"
#include "G4Colour.hh"
#include "G4VisAttributes.hh"

//ELYSE
#include "ELYSE/TrappingVolume.hh"
#include "ELYSE/DetectorMessenger.hh"
#include "ELYSE/Cast.hh"
#include "ELYSE/Analysis.hh"

ELYSE::DetectorConstruction::DetectorConstruction() {

  //-----------------
  // Create Materials
  //-----------------
    
  ConstructMaterials();

  //-----------------
  // Initilize SD pointers
  //-----------------

  aTrappingVolume  = 0; 

  //-----------------
  // Volume/Surface lengths/size (default)
  //-----------------

  //The Hall
  expHall_x = expHall_y = expHall_z = 1.0*m;
  //The target volume
  water_x    = water_y    = water_z    = 10.0*cm;
  //tyvek sheet thickness
  tyvek_thickness = 1.0*mm;


  updated = true;

  //Initialize the optical surface 
  OpWaterTyvekSurface = 0;
  OpWaterAirSurface   = 0;


  //-----------------
  // Make the detector messenger to allow changing geometry
  //-----------------

  messenger = new DetectorMessenger(this);


}//Ctor

//----------------------------------------------------------------------------------------------
ELYSE::DetectorConstruction::~DetectorConstruction(){
  delete messenger;
}

//-----------------------------------------------------------------------------------------------

G4VPhysicalVolume* ELYSE::DetectorConstruction::Construct() {

  //-----------------------
  // The Solids
  //-----------------------

  //-----------------------
  // A box of water delimited by a tyvek sheet and surrounding by a air volume sensitive
  //-----------------------


  G4Box* solidExpHall = new G4Box("expHall",
				  expHall_x/2.,
				  expHall_y/2.,
				  expHall_z/2.);
  

  //a volume sensitive to trap every particles...
  G4Box*  solidSensiVolFull = new G4Box("SensiVolFull",
				    water_x/2.+ 2.*tyvek_thickness,
				    water_y/2.+ 2.*tyvek_thickness,
				    water_z/2.+ 2.*tyvek_thickness
				    );
  

  G4Box*  solidTyvekFull = new G4Box("TyvekFull",
				     water_x/2.+tyvek_thickness,
				     water_y/2.+tyvek_thickness,
				     water_z/2.+tyvek_thickness
				     );

  G4Box* solidWaterTank = new G4Box("waterTank",water_x/2.,water_y/2.,water_z/2.);



  G4SubtractionSolid* solidSensiVol = new G4SubtractionSolid("SensiVol",solidSensiVolFull,solidTyvekFull);

  //Make the tyvek volume as an subtraction to define a proper tyvek volume for the oprtical surface
  //overwise we would have done a tyvek volume and place inside a water volume...
  G4SubtractionSolid* solidTyvek = new G4SubtractionSolid("TyvekSheet",solidTyvekFull,solidWaterTank);


  //-----------------------
  // The Logical and Physical volumes
  // The vizualization attributes are defined in scripts/Styles/ELYSE.styles
  //-----------------------



  G4LogicalVolume* logicExpHall = 
    new G4LogicalVolume(solidExpHall,
			G4Material::GetMaterial("Air"),
			"expHall",
			0,0,0);

  G4VPhysicalVolume* physicExpHall = 
    new G4PVPlacement(0,G4ThreeVector(),
		      logicExpHall,
		      "expHall",
		      0,false,0);


  G4LogicalVolume* logicSensiVol
    = new G4LogicalVolume(solidSensiVol,G4Material::GetMaterial("Glass"),"SensiVol",0,0,0);


  G4VPhysicalVolume* physicSensiVol
    = new G4PVPlacement(0,G4ThreeVector(),logicSensiVol,"SensiVol",
			logicExpHall,false,0);


  G4LogicalVolume* logicTyvek 
    = new G4LogicalVolume(solidTyvek,G4Material::GetMaterial("Tyvek"),"TyvekSheet",0,0,0);

  G4VPhysicalVolume* physicTyvek
    = new G4PVPlacement(0,G4ThreeVector(),logicTyvek,"TyvekSheet",
			logicExpHall,false,0);

  G4LogicalVolume* logicWaterTank
    = new G4LogicalVolume(solidWaterTank,G4Material::GetMaterial("Water"),"waterTank",0,0,0);
  

  G4VPhysicalVolume* physicWaterTank
    = new G4PVPlacement(0,G4ThreeVector(),logicWaterTank,"waterTank",
                        logicExpHall,false,0);
  

  //introduce the interfaces Water-Tyvek and Tyvek-Glass for optical photons
  new G4LogicalBorderSurface("waterTyvekInterface",
			       physicWaterTank,physicTyvek,OpWaterTyvekSurface);
  
  new G4LogicalBorderSurface("tyvekGlassInterface",
			       physicTyvek,physicSensiVol,OpTyvekGlassSurface);

  //--------------
  // Make sensitive detectors (this is quite artificial for the moment 13/2/07)
  //--------------

  G4SDManager* SDman = G4SDManager::GetSDMpointer();
  
  if (!aTrappingVolume) {
    //JEC 25/1/06 add the "this" pointer, so WCSD can access to the PMT location wo static functions
    aTrappingVolume = new TrappingVolume( "/ELYSE/TrappingVolume", this ); 
    SDman->AddNewDetector( aTrappingVolume );
  }
  logicSensiVol->SetSensitiveDetector( aTrappingVolume );



  // Return the pointer to the physical experimental hall
  return physicExpHall;
} //Construct

//---------------------------------------------------------------------------------------------
void ELYSE::DetectorConstruction::ConstructMaterials() {
    
  //---Vaccuum
  G4double density     = universe_mean_density;              //from PhysicalConstants.h
  G4double pressure    = 1.e-19*pascal;
  G4double temperature = 0.1*kelvin;
  G4double a = 1.01*g/mole;
  new G4Material("Vaccuum", 1., a, density,
                   kStateGas,temperature,pressure);

  //---Water
  
  a = 1.01*g/mole;
  G4Element* elH 
    = new G4Element("Hydrgen","H", 1,a);
  
  a = 16.00*g/mole;
  G4Element* elO 
    = new G4Element("Oxygen","O", 8,a);
  
  density = 1.00*g/cm3;
  G4Material* Water 
    = new G4Material("Water",density,2);
  Water->AddElement(elH, 2);
  Water->AddElement(elO, 1);

  //---Air
  
  a = 14.01*g/mole;
  G4Element* elN 
    = new G4Element("Nitrogen","N", 7,a);
  
  density = 1.290*mg/cm3;
  G4Material* Air 
    = new G4Material("Air",density,2);
  Air->AddElement(elN, 70.*perCent);
  Air->AddElement(elO, 30.*perCent);
  

  //---Glass
  a = 28.09*g/mole;
  G4Element* elSi = new G4Element("Silicon", "Si", 14., a); 
 
  density = 2.20*g/cm3;
  G4Material* SiO2 = new G4Material("SiO2",density,2);
  SiO2->AddElement(elSi, 1);
  SiO2->AddElement(elO , 2);

  a = 10.81*g/mole;
  G4Element* elB = new G4Element("Boron", "B", 5, a);  

  density = 2.46*g/cm3;
  G4Material* B2O3 = new G4Material("B2O3",density,2);
  B2O3->AddElement(elB, 2);
  B2O3->AddElement(elO, 3);

  a = 22.99*g/mole;
  G4Element* elNa = new G4Element("Sodium", "Na", 11, a);  

  density = 2.27*g/cm3;
  G4Material* Na2O = new G4Material("Na2O",density,2);
  Na2O->AddElement(elNa, 2);
  Na2O->AddElement(elO, 1);

  a = 26.98*g/mole;
  G4Element* elAl = new G4Element("Aluminum", "Al", 13, a);  

  density = 4.00*g/cm3;
  G4Material* Al2O3 = new G4Material("Al2O3",density,2);
  Al2O3->AddElement(elAl, 2);
  Al2O3->AddElement(elO, 3);

  density = 2.23*g/cm3;
  G4Material* Glass = new G4Material("Glass",density,4);
  
  Glass->AddMaterial(SiO2, 80.6*perCent);
  Glass->AddMaterial(B2O3, 13.0*perCent);
  Glass->AddMaterial(Na2O, 4.0*perCent);
  Glass->AddMaterial(Al2O3, 2.4*perCent);


  //---Tyvek
  //JEC : 9/2/07 as MEMPHYS/SK::Blacksheet in a version 0
  //JEC + SDC: 9/2/07 use Auger Liner definition
  a= 12.01*g/mole;
  G4Element* elC 
    = new G4Element("Carbon","C", 6,a);

  density = 0.94*g/cm3;
  G4Material* Tyvek = new G4Material("Tyvek",density,2);
  Tyvek->AddElement(elC, 2);
  Tyvek->AddElement(elH, 4);


  // -------------------------------------------------------------
  // Generate & Add Material Properties Table
  // -------------------------------------------------------------
  
  const G4int NUMENTRIES_general=60; //number of energy bins to simulate material properties
  
  //--------------
  //-------- Water
  //--------------
  
  G4double ENERGY_water[NUMENTRIES_general] =
    { 1.56962*eV, 1.58974*eV, 1.61039*eV, 1.63157*eV, 
      1.65333*eV, 1.67567*eV, 1.69863*eV, 1.72222*eV, 
      1.74647*eV, 1.77142*eV, 1.79710*eV, 1.82352*eV, 
      1.85074*eV, 1.87878*eV, 1.90769*eV, 1.93749*eV, 
      1.96825*eV, 1.99999*eV, 2.03278*eV, 2.06666*eV,
      2.10169*eV, 2.13793*eV, 2.17543*eV, 2.21428*eV, 
      2.25454*eV, 2.29629*eV, 2.33962*eV, 2.38461*eV, 
      2.43137*eV, 2.47999*eV, 2.53061*eV, 2.58333*eV, 
      2.63829*eV, 2.69565*eV, 2.75555*eV, 2.81817*eV, 
      2.88371*eV, 2.95237*eV, 3.02438*eV, 3.09999*eV,
      3.17948*eV, 3.26315*eV, 3.35134*eV, 3.44444*eV, 
      3.54285*eV, 3.64705*eV, 3.75757*eV, 3.87499*eV, 
      3.99999*eV, 4.13332*eV, 4.27585*eV, 4.42856*eV, 
      4.59258*eV, 4.76922*eV, 4.95999*eV, 5.16665*eV, 
      5.39129*eV, 5.63635*eV, 5.90475*eV, 6.19998*eV };

  G4double RINDEX1[NUMENTRIES_general] =  
    { 1.32885, 1.32906, 1.32927, 1.32948, 1.3297, 1.32992, 1.33014, 
      1.33037, 1.3306, 1.33084, 1.33109, 1.33134, 1.3316, 1.33186, 1.33213,
      1.33241, 1.3327, 1.33299, 1.33329, 1.33361, 1.33393, 1.33427, 1.33462,
      1.33498, 1.33536, 1.33576, 1.33617, 1.3366, 1.33705, 1.33753, 1.33803,
      1.33855, 1.33911, 1.3397, 1.34033, 1.341, 1.34172, 1.34248, 1.34331,
      1.34419, 1.34515, 1.3462, 1.34733, 1.34858, 1.34994, 1.35145, 1.35312,
      1.35498, 1.35707, 1.35943, 1.36211, 1.36518, 1.36872, 1.37287, 1.37776,
      1.38362, 1.39074, 1.39956, 1.41075, 1.42535 };
  
  G4double ABSORPTION_water[NUMENTRIES_general] =
    {25.3504*cm, 31.7938*cm, 39.9915*cm, 50.454*cm, 63.85*cm, 
     81.0584*cm, 103.24*cm, 131.93*cm, 169.172*cm, 217.694*cm, 
     224.921*cm, 249.688*cm, 262.674*cm, 273*cm, 321.13*cm, 339.789*cm,
     351.617*cm, 363.108*cm, 385.802*cm, 461.042*cm, 707.714*cm, 
     1038.42*cm, 1383.7*cm, 1558.36*cm, 1722.65*cm, 1939.11*cm, 
     2092.49*cm, 2240.14*cm, 2962.96*cm, 4967.03*cm, 6368.58*cm, 
     8207.56*cm, 10634.2*cm, 13855.3*cm, 18157.3*cm, 23940.2*cm, 
     31766*cm, 42431.6*cm, 57074.9*cm, 77335.9*cm, 105598*cm, 
     145361*cm, 192434*cm, 183898*cm, 176087*cm, 168913*cm, 162301*cm, 
     156187*cm, 150516*cm, 145243*cm, 140327*cm, 135733*cm, 131430*cm, 
     127392*cm, 123594*cm, 120016*cm, 116640*cm, 113448*cm, 110426*cm, 
     107562*cm};
  
  // M Fechner: Rayleigh scattering -- as of version 4.6.2 of GEANT,
  // one may use one's own Rayleigh scattering lengths (the buffer is no
  // longer overwritten for "water", see 4.6.2 release notes)
  
  // RAYFF = 1/ARAS, for those of you who know SKdetsim...
  // actually that's not quite right because the scattering models
  // are different; in G4 there is no scattering depolarization
  // std value at SK = 0.6. But Mie scattering is implemented
  // in SKdetsim and not in G4
  
  // april 2005 : reduced reflections, let's increase scattering...
  //   G4double RAYFF = 1.0/1.65;
  G4double RAYFF = 1.0/1.5;
  
  G4double RAYLEIGH_water[NUMENTRIES_general] = {
    167024.4*cm*RAYFF, 158726.7*cm*RAYFF, 150742*cm*RAYFF,
    143062.5*cm*RAYFF, 135680.2*cm*RAYFF, 128587.4*cm*RAYFF,
    121776.3*cm*RAYFF, 115239.5*cm*RAYFF, 108969.5*cm*RAYFF,
    102958.8*cm*RAYFF, 97200.35*cm*RAYFF, 91686.86*cm*RAYFF,
    86411.33*cm*RAYFF, 81366.79*cm*RAYFF, 76546.42*cm*RAYFF,
    71943.46*cm*RAYFF, 67551.29*cm*RAYFF, 63363.36*cm*RAYFF,
    59373.25*cm*RAYFF, 55574.61*cm*RAYFF, 51961.24*cm*RAYFF,
    48527.00*cm*RAYFF, 45265.87*cm*RAYFF, 42171.94*cm*RAYFF,
    39239.39*cm*RAYFF, 36462.50*cm*RAYFF, 33835.68*cm*RAYFF,
    31353.41*cm*RAYFF, 29010.30*cm*RAYFF, 26801.03*cm*RAYFF,
    24720.42*cm*RAYFF, 22763.36*cm*RAYFF, 20924.88*cm*RAYFF,
    19200.07*cm*RAYFF, 17584.16*cm*RAYFF, 16072.45*cm*RAYFF,
    14660.38*cm*RAYFF, 13343.46*cm*RAYFF, 12117.33*cm*RAYFF,
    10977.70*cm*RAYFF, 9920.416*cm*RAYFF, 8941.407*cm*RAYFF,
    8036.711*cm*RAYFF, 7202.470*cm*RAYFF, 6434.927*cm*RAYFF,
    5730.429*cm*RAYFF, 5085.425*cm*RAYFF, 4496.467*cm*RAYFF,
    3960.210*cm*RAYFF, 3473.413*cm*RAYFF, 3032.937*cm*RAYFF,
    2635.746*cm*RAYFF, 2278.907*cm*RAYFF, 1959.588*cm*RAYFF,
    1675.064*cm*RAYFF, 1422.710*cm*RAYFF, 1200.004*cm*RAYFF,
    1004.528*cm*RAYFF, 833.9666*cm*RAYFF, 686.1063*cm*RAYFF
  };

  // SDC: new parameters for scintillation

  G4double SCINTILFAST[NUMENTRIES_general] = 
    { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0};


   G4double SCINTILSLOW[NUMENTRIES_general] = 
     { 0 , 0 , 0 , 0 , 0 , 0 , 0,  
       0 , 0 , 0 , 0 , 0 , 0 , 
       0 , 0 , 0 , 0 , 0 , 0 , 
       0.505 , 1.5 , 2.5 , 3.5 , 4.5 , 5.5 , 
       6.5 , 7.5 , 8.5 , 8.5 , 7.5 , 6.5 , 
       5 , 3.5 , 2.5 , 1.5 , 0.505 , 0.505 , 
       1.5 , 2.5 , 3.5 , 4.5 , 5.5 , 6.5 , 
       7.5 , 8.5 , 8.5 , 7.5 , 6.5 , 5.5 , 
       4.5 , 0 , 0 , 0 , 0 , 0 , 
       0 , 0 , 0 , 0 , 0  };

   G4MaterialPropertiesTable *myMPT1 = new G4MaterialPropertiesTable();
   myMPT1->AddProperty("RINDEX", ENERGY_water, RINDEX1, NUMENTRIES_general);
   myMPT1->AddProperty("ABSLENGTH",ENERGY_water, ABSORPTION_water, NUMENTRIES_general);

   myMPT1->AddProperty("FASTCOMPONENT",ENERGY_water,SCINTILFAST,NUMENTRIES_general);
   myMPT1->AddProperty("SLOWCOMPONENT",ENERGY_water,SCINTILSLOW,NUMENTRIES_general);

   // M Fechner: new, don't let G4 compute it.
   myMPT1->AddProperty("RAYLEIGH",ENERGY_water,RAYLEIGH_water,NUMENTRIES_general);

   //SDC : scintillation part if needed (G4 exemple ???)
   myMPT1->AddConstProperty("SCINTILLATIONYIELD",50./MeV);
   myMPT1->AddConstProperty("RESOLUTIONSCALE",1.0);
   myMPT1->AddConstProperty("FASTTIMECONSTANT", 1.*ns);
   myMPT1->AddConstProperty("SLOWTIMECONSTANT",10.*ns);
   myMPT1->AddConstProperty("YIELDRATIO",0.8);


   Water->SetMaterialPropertiesTable(myMPT1);
   
   G4cout << "ConstructMaterial: Dump Water properties" << G4endl;
   myMPT1->DumpTable();
   

   //--------------
   //------------ Air
   //--------------
  G4double RINDEX_air[NUMENTRIES_general] = 
    { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
      1.0, 1.0, 1.0, 1.0, 1.0, 1.0};

   //utter fiction at this stage, does not matter
   G4double RAYLEIGH_air[NUMENTRIES_general] =
     { 0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,
       0.001*m,0.001*m,0.001*m,0.001*m,0.001*m,0.001*m};

   G4double ABSORPTION_air[NUMENTRIES_general] =
     { 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm, 1.0e-9*cm,
       1.0e-9*cm, 1.0e-9*cm,1.0e-9*cm, 1.0e-9*cm};


   G4MaterialPropertiesTable *myMPT2 = new G4MaterialPropertiesTable();
   myMPT2->AddProperty("RINDEX", ENERGY_water, RINDEX_air, NUMENTRIES_general);
   // M Fechner : what is that ?????
   //   myMPT2->AddProperty("ABSLENGTH", ENERGY_water, BLACKABS_blacksheet, NUMENTRIES_general);
   // JEC: replace BLACKABS_blacksheet by  ABSORPTION_air
   myMPT2->AddProperty("ABSLENGTH", ENERGY_water, ABSORPTION_air, NUMENTRIES_general);
   myMPT2->AddProperty("RAYLEIGH",ENERGY_water, RAYLEIGH_air, NUMENTRIES_general);
   Air->SetMaterialPropertiesTable(myMPT2);


   //--------------
   //----------------  Tyvek
   //--------------
   //JEC 9/2/07 : as Blacksheet in a version 0
   //JEC +SDC 9/2/07: introduce the Auger simulation of the liner
   
   const G4int NUMENTRIES_tyvek = 30;

   G4double ENERGY_tyvek[NUMENTRIES_tyvek] = {
     2.08*eV , 2.16*eV , 2.19*eV , 2.23*eV , 2.27*eV , 2.32*eV , 2.36*eV , 2.41*eV , 2.46*eV , 2.50*eV , 
     2.56*eV , 2.61*eV , 2.67*eV , 2.72*eV , 2.79*eV , 2.85*eV , 2.92*eV , 2.99*eV , 3.06*eV , 3.14*eV , 
     3.22*eV , 3.31*eV , 3.4*eV  , 3.49*eV , 3.59*eV , 3.7*eV  , 3.81*eV , 3.94*eV , 4.07*eV , 4.20*eV };

   G4double ABSORPTION_Tyvek[NUMENTRIES_tyvek] =
     { 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 
       10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 
       10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m , 10.0*m };

   G4MaterialPropertiesTable *myMPT4 = new G4MaterialPropertiesTable();
   myMPT4->AddProperty("ABSLENGTH", ENERGY_tyvek, ABSORPTION_Tyvek, NUMENTRIES_tyvek);
   Tyvek->SetMaterialPropertiesTable(myMPT4);
   

   //--------------
   //--------- Glass
   //--------------

  G4double RINDEX_glass[NUMENTRIES_general] =
    { 1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600, 1.600, 1.600, 1.600, 1.600, 1.600,
      1.600, 1.600 }; 
  
  //JEC: To do adapt to the # of entries  and do not include quantum eff. of a PMTs... as in MEMPHYS case 9/2/07
//   G4double ABSORPTION_glass[NUMENTRIES] = 
//     { 100.0*cm, 110.0*cm, 120.0*cm, 130.0*cm, 140.0*cm, 150.0*cm, 160.0*cm,
//       165.0*cm, 170.0*cm, 175.0*cm, 180.0*cm, 185.0*cm, 190.0*cm, 195.0*cm,
//       200.0*cm, 200.0*cm, 200.0*cm, 200.0*cm, 200.0*cm, 195.0*cm, 190.0*cm,
//       185.0*cm, 180.0*cm, 175.0*cm, 170.0*cm, 160.0*cm, 150.0*cm, 140.0*cm,
//       130.0*cm, 120.0*cm, 110.0*cm, 100.0*cm };

  //JEC dummy values to be able to compile !!! 14/2/07
   G4double ABSORPTION_glass[NUMENTRIES_general] =
     { 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm, 150.0*cm,
       150.0*cm, 150.0*cm,150.0*cm, 150.0*cm};


  
   G4MaterialPropertiesTable *myMPT5 = new G4MaterialPropertiesTable();
   myMPT5->AddProperty("RINDEX", ENERGY_water, RINDEX_glass, NUMENTRIES_general);
   myMPT5->AddProperty("ABSLENGTH",ENERGY_water, ABSORPTION_glass, NUMENTRIES_general);
   Glass->SetMaterialPropertiesTable(myMPT5);
    
   //	------------- Surfaces --------------

   //--------------------
   // Water - Tyvek
   //--------------------
   //JEC 9/2/07: From the BackSheet Simulation in MEMPHYS/SK
   //JEC + SDC: input properties from Auger

   OpWaterTyvekSurface = new G4OpticalSurface("WaterTyvekSurface");
   OpWaterTyvekSurface->SetType(dielectric_dielectric);
   OpWaterTyvekSurface->SetModel(unified);
   OpWaterTyvekSurface->SetFinish(groundbackpainted);
   OpWaterTyvekSurface->SetSigmaAlpha( 0.17 );
   
   //JEC + SDC 9/2/07 : adapted from Auger
   const G4int NUMENTRIES_water_tyvek = 3;
   G4double PP[NUMENTRIES_water_tyvek]                    = { 2.08*eV, 3.0*eV, 4.2*eV };
   G4double RINDEX_tyvek[NUMENTRIES_water_tyvek]          = { 0.0    , 0.0   ,  0.0   }; //JEC ????
   G4double SPECULARLOBECONSTANT[NUMENTRIES_water_tyvek]  = { 0.2    , 0.2   ,  0.2   };
   G4double SPECULARSPIKECONSTANT[NUMENTRIES_water_tyvek] = { 0.0    , 0.0   ,  0.0   };
   G4double BACKSCATTERCONSTANT[NUMENTRIES_water_tyvek]   = { 0.0    , 0.0   ,  0.0   };

   //JEC 9/2/07 from MEMPHYS/SK Blacksheet: for sure should be modified!
   //JEC + SDC 9/2/07 adapted from Auger
   G4double REFLECTIVITY_tyvek[NUMENTRIES_tyvek] =
     { 0.968816 , 0.968427 , 0.967940 , 0.967357 , 0.967065 , 0.966870 , 0.966870 , 0.967162 , 0.967649 , 0.968524 ,
       0.969497 , 0.970568 , 0.971541 , 0.972319 , 0.973000 , 0.973000 , 0.972319 , 0.970762 , 0.967940 , 0.963562 , 
       0.957432 , 0.948967 , 0.938167 , 0.924350 , 0.908198 , 0.886208 , 0.861008 , 0.831039 , 0.795719 , 0.754561  };

   
   G4MaterialPropertiesTable *myST1 = new G4MaterialPropertiesTable();
   myST1->AddProperty("RINDEX", ENERGY_tyvek, RINDEX_tyvek, NUMENTRIES_tyvek);
   myST1->AddProperty("SPECULARLOBECONSTANT", PP, SPECULARLOBECONSTANT, NUMENTRIES_water_tyvek);
   myST1->AddProperty("SPECULARSPIKECONSTANT", PP, SPECULARSPIKECONSTANT, NUMENTRIES_water_tyvek);
   myST1->AddProperty("BACKSCATTERCONSTANT", PP, BACKSCATTERCONSTANT, NUMENTRIES_water_tyvek);
   myST1->AddProperty("REFLECTIVITY", ENERGY_tyvek, REFLECTIVITY_tyvek, NUMENTRIES_tyvek);
   OpWaterTyvekSurface->SetMaterialPropertiesTable(myST1);


   //------------------
   //Tyvek - Glass
   //------------------
   //JEC 13/2/07 this is a stopping surface as the photocathode in MEMPHYS
   //            reflectivity = 0 && efficiency = 100%   => absorption
   
   OpTyvekGlassSurface = new G4OpticalSurface("TyvekGlassSurface");
   OpTyvekGlassSurface->SetType(dielectric_metal); //as for LXe G4 example
   OpTyvekGlassSurface->SetModel(glisur);          //as for LXe G4 example
   OpTyvekGlassSurface->SetFinish(polished);
   const G4int NUM = 2;
   G4double REFLECTIVITY_tyvekglass[NUM] = { 0.0, 0.0 };
   G4double EFFICIENCY_tyvekglass[NUM]   = { 1.0, 1.0 };

   G4MaterialPropertiesTable *myST2 = new G4MaterialPropertiesTable();
   myST2->AddProperty("REFLECTIVITY", PP, REFLECTIVITY_tyvekglass, NUM);
   myST2->AddProperty("EFFICIENCY", PP, EFFICIENCY_tyvekglass, NUM);
   OpTyvekGlassSurface->SetMaterialPropertiesTable(myST2);

   //------------------
   //Water - Air
   //------------------

   //JEC: 9/2/07 certainly from a G4 novice exercice ==> should be clarified
   OpWaterAirSurface = new G4OpticalSurface("WaterAirSurface");
   OpWaterAirSurface->SetType(dielectric_dielectric);
   OpWaterAirSurface->SetFinish(ground);
   OpWaterAirSurface->SetModel(unified);
   
   const G4int num_WaterAir = 2;
   G4double Ephoton_WaterAir[num_WaterAir]         = {2.038*eV, 4.144*eV};
   G4double RefractiveIndex_WaterAir[num_WaterAir] = {1.35, 1.40};
   G4double SpecularLobe_WaterAir[num_WaterAir]    = {0.3, 0.3};
   G4double SpecularSpike_WaterAir[num_WaterAir]   = {0.2, 0.2};
   G4double Backscatter_WaterAir[num_WaterAir]     = {0.2, 0.2};
   
   G4MaterialPropertiesTable* myST11 = new G4MaterialPropertiesTable();
   
   myST11->AddProperty("RINDEX",                Ephoton_WaterAir, RefractiveIndex_WaterAir, num_WaterAir);
   myST11->AddProperty("SPECULARLOBECONSTANT",  Ephoton_WaterAir, SpecularLobe_WaterAir,    num_WaterAir);
   myST11->AddProperty("SPECULARSPIKECONSTANT", Ephoton_WaterAir, SpecularSpike_WaterAir,   num_WaterAir);
   myST11->AddProperty("BACKSCATTERCONSTANT",   Ephoton_WaterAir, Backscatter_WaterAir,     num_WaterAir);
   
   OpWaterAirSurface->SetMaterialPropertiesTable(myST11);
}//ConstructMaterials

//-----------------------------------------------------------------------------------------------

void ELYSE::DetectorConstruction::SetWaterDimensions(G4ThreeVector dim){
  water_x = dim[0];
  water_y = dim[1];
  water_z = dim[2];
  updated = true;
}//SetWaterDimensions

//-----------------------------------------------------------------------------------------------
void ELYSE::DetectorConstruction::SetTyvekThickness(G4double thick) {
  tyvek_thickness = thick;
  updated = true;
}//SetTyvekThickness
//-----------------------------------------------------------------------------------------------

void ELYSE::DetectorConstruction::UpdateGeometry() {
  // clean-up previous geometry
  G4GeometryManager::GetInstance()->OpenGeometry();

  G4PhysicalVolumeStore::GetInstance()->Clean();
  G4LogicalVolumeStore::GetInstance()->Clean();
  G4SolidStore::GetInstance()->Clean();
  G4LogicalSkinSurface::CleanSurfaceTable();
  G4LogicalBorderSurface::CleanSurfaceTable();

  //define new one
  G4RunManager::GetRunManager()->DefineWorldVolume(Construct());
  G4RunManager::GetRunManager()->GeometryHasBeenModified();

  updated=false;
}
