/******************************/
/* BAOCONTROL                 */
/*                            */
/* Franck RICHARD             */
/* franckrichard033@gmail.com */
/* Juin 2011                  */
/******************************/


#include "baocontrol.h"



/**************************************************************************************
**Constructeur
***************************************************************************************/

BAOcontrol::BAOcontrol()
{
    float a1, a2, a3;

    // Le choix de la couleur des caractères est adapté pour une fenêtre de terminal
    // ayant un fond blanc. pour un fond noir, mettre couleurs=2 dans le fichier params

    ChoixCouleurs=1;


    Affiche("\n\n\n\n\n \
 ******************************************\n \
 *             BAORadio Control           *\n \
 * Laboratoire de l'Accélérateur Linéaire *\n \
 *                                        *\n \
 *             v0.2  20/06/2011           *\n \
 *        franckrichard033@gmail.com      *\n \
 ******************************************\n\n\n", true);

    //Initialisation des variables globales et des pointeurs


    delaitransit  = 0;
    delaitracking = 0;
    lognum        = 0;
    numAntenne    = 0;
    numobjets     = 0;
    runnum        = 1;

    Pression      = 1013.0;
    Temperature   = 10.0;

    NoExit        = true;
    Transit       = false;
    run           = false;
    exitrun       = false;

    fd            = NULL;
    client_socket = NULL;

    // chaines de caractères et tableaux

    LatitudeChar  = "";
    LongitudeChar = "";
    Serveur       = "";
    Port          = "";

    Antennes 	  = new DefAntenne[MAXANTENNES];
    objets        = new DefObjets[MAXOBJETS];

    //on initalise aussi les tableaux

    for (int i=0; i<MAXANTENNES; i++)
    {
        Antennes[i].ok=0;
        Antennes[i].ip=0;
    }

    for (int i=0; i<MAXOBJETS; i++)
    {
        objets[i].JJ=0;
        objets[i].exec=false;
    }

    // Chargement du fichier contenant les paramètres

    if (!ChargementParametres("params"))
    {
        Erreur("Le fichier de configuration 'params' contient des erreurs ou n'existe pas.\n \
        Merci de vérifier.\n\n");
        exit(1);
    }

    // Conversion de la latitude et de la longitude en radians

    Decomposition(LatitudeChar, 0, &a1, &a2, &a3);

    double Latitude = (a1 + a2 / 60.0 + a3 / 3600.0) * Pidiv180;

    Decomposition(LongitudeChar, 1, &a1, &a2, &a3);

    double Longitude = Pi2 - ( a1 + a2 / 60.0 + a3 / 3600.0 ) * Pidiv180;

    // Transfert de la latitude, longitude, pression, température à la classe Astro

    DefinirLongitudeLatitude(Longitude, Latitude);

    DefinirPressionTemp(Pression, Temperature);

    // Ouverture du socket avec indi_BAO

    client_socket = new ClientSocket("localhost", atoi(Port.c_str()) );

    // On se connecte au pilote indi_BAO

    Connect(true);
}



/**************************************************************************************
** Destructeur
***************************************************************************************/

BAOcontrol::~BAOcontrol()
{
    // destruction des tableaux

    delete [] Antennes;
    delete [] objets;

    // destruction de la fenêtre graphique, des caractères

    if (d!=NULL)
    {
        XFreeGC(d, noir);
        XFreeGC(d, vert);
        XFreeGC(d, rouge);
        XFreeGC(d, gris);
        XFreeFont(d, fd);
        XDestroyWindow(d, w);
        XCloseDisplay(d);
    }

    // destruction du socket

    delete client_socket;
}


/**************************************************************************************
** Affiche le message à l'écran (si AfficheEcran==true) et le sauvegarde dans le fichier
** BAOControl.log
***************************************************************************************/

void BAOcontrol::Affiche(string Message, bool AfficheEcran)
{
    FILE *pFile = NULL;

    if ( (pFile = fopen ("BAOcontrol.log", "a")) != NULL)
    {
        if (AfficheEcran) cout << Message;

        fprintf (pFile, Message.c_str());

        fclose (pFile);
    }
}

void BAOcontrol::Affiche(stringstream *Message, bool AfficheEcran)
{
    Affiche(Message->str(), AfficheEcran);

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Affiche les messages d'erreur en rouge
***************************************************************************************/

void BAOcontrol::Erreur(string chaine)
{
    ChoixCouleurs==1 ? cout << red1  : cout << red2;
    Affiche(chaine, true);
    ChoixCouleurs==1 ? cout << grey1 : cout << grey2;
}

void BAOcontrol::Erreur(stringstream *Message)
{
    Erreur(Message->str());

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Initialise la fenêtre graphique
***************************************************************************************/

void BAOcontrol::initialiserFenetre()
{
    XGCValues gcv;

    // contacte le serveur graphique

    if ((d = XOpenDisplay(getenv("DISPLAY"))) == NULL)
    {
        Erreur("Impossible de contacter le serveur\n\n");
        exit(1);
    }

    // crée une fenêtre

    w = XCreateSimpleWindow(d, RootWindow(d, DefaultScreen(d)),
                            0, 0, larg_fenetre, haut_fenetre,
                            2, BlackPixel(d, DefaultScreen(d)),
                            WhitePixel(d, DefaultScreen(d)));

    // Création du double buffer

    db = XCreatePixmap(d, w, larg_fenetre, haut_fenetre, DefaultDepth(d, DefaultScreen(d)));

    // Chargement de la police de caractères

    if ((fd = XLoadQueryFont(d, "fixed")) == NULL)
    {
        Erreur("Impossible de charger la police fixed\n\n");
        exit(1);
    }

    // Création des couleurs

    gcv.font = fd->fid;
    gcv.foreground = BlackPixel(d, DefaultScreen(d));
    noir = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*255;
    vert = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 255*256*256;
    rouge = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*256*256+200*256+200;
    gris = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    // Affichage de fenêtre

    XMapWindow(d, w);
}


/**************************************************************************************
** Dessine les éléments dans la fenêtre graphique
***************************************************************************************/

void BAOcontrol::Dessiner()
{
    char *chaine;
    GC color;

    chaine = new char [100];

    // On efface la fenêtre

    XSetForeground(d, noir, WhitePixel(d, DefaultScreen(d)));
    XFillRectangle(d, db, noir, 0, 0, larg_fenetre, haut_fenetre);
    XSetForeground(d, noir, BlackPixel(d, DefaultScreen(d)));

    // On récupère la date et l'heure contenue dans la classe astro (et le jour julien)

    double h  = GetHeure();
    double mi = GetMin();
    double s  = GetSec();
    double a  = GetAnnee();
    double m  = GetMois();
    double j  = GetJour();
    double JJ = GetJJ();

    // on évite les affichages de l'heure du genre 12:56:60 ou 24:00:00

    if (s==60) {
        s = 0;
        mi++;
        if (mi == 60) {
            mi = 0;
            h++;
            if (h == 24) {
                h = 0;
                j++;
            }
        }
    }

    //affichage de la date et de l'heure

    sprintf(chaine, "%02.0f:%02.0f:%02.1f TU    %02.0f/%02.0f/%04.0f    JJ=%10.5f",
            h, mi, s, j, m, a, JJ);

    XDrawString(d, db, noir, 10, 10, chaine, strlen(chaine));

    //En cas d'exécution d'un fichier de mouvements, on affiche la prochaine étape

    if (run)
    {
        sprintf(chaine, "Prochaine etape dans %4.1fs", (objets[runnum].JJ - JJ) * 3600.0 * 24.0);

        if ((objets[runnum].JJ - JJ) * 3600.0 * 24.0 >= 0.0)
            XDrawString(d, db, noir, 350, 10, chaine, strlen(chaine));
    }

    // affichage des cercles d'état
    // vert = l'antenne fonctionne
    // rouge = problème détecté sur l'antenne
    // gris = antenne non connectée

    for (int i=0; i <20; i++)
    {
        for (int j=0; j<2; j++)
        {
            switch (Antennes[20*j+i].ok)
            {
            case 1 :
                color=vert;
                break;
            case 2 :
                color=rouge;
                break;
            default:
                color=gris;
                break;
            }

            XFillArc(d, db ,color, 10+i*30, 20+j*30, 15, 15, 0, 360*64);
            XDrawArc(d, db ,noir,  10+i*30, 20+j*30, 15, 15, 0, 360*64);
        }
    }


    // Affichage des messages d'indi_BAO
    // gère aussi le défilement

    if (lognum>10)
    {
        for (int i=lognum-1; i>lognum-11; i--)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+(i-lognum+10)*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }
    else
    {
        for (int i=0; i<lognum; i++)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+i*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }

    // On affiche la copie de l'écran une fois que tout est dessiné
    // on évite ainsi les scientillements de la fenêtre

    XCopyArea(d, db, w, noir, 0, 0, larg_fenetre, haut_fenetre, 0, 0);

    // On force le rafraichissement de l'écran

    XFlush(d);

    delete [] chaine;
}


/**************************************************************************************
** Gestion des événements affectant la fenêtre graphique
***************************************************************************************/

void BAOcontrol::rouler()
{
    XEvent e;

    if (XCheckWindowEvent(d, w, ExposureMask, &e))
    {
        switch (e.type)
        {
        default:
            //cout << "J'ai reçu un événement " <<  e.type << endl
            break;
        case Expose :
            Dessiner();
            break;
        }
    }
}


/**************************************************************************************
** Récupère les messages envoyés par indi_BAO
***************************************************************************************/

void BAOcontrol::LireReponse()
{
    string reponse, memreponse, decomp;
    bool test=false;

    *client_socket >> reponse;
    
   // printf("%s\n", reponse.c_str());

    do
    {
        if (reponse.find("message")!=string::npos)
        {
            // On ne garde que la partie intéressante
            // La partie suivant "message="...
            reponse = reponse.substr(reponse.find("message") + 9);

            memreponse = reponse;

            // ...jusqu'au "
            reponse=reponse.substr(0, reponse.find("\""));

            // On récupère l'adresse ip de l'antenne qui nous parle
            // et on met à jour le tabmeau des états des antennes

            if (reponse.find(". (Antennes connectees")!=string::npos)
            {
                decomp=reponse.substr(0, reponse.find(". (Antennes connectees"));
                decomp=decomp.substr(decomp.rfind(".")+1);

                // si cette antenne est déjà connectée, on change son état

                for (int i=0; i<numAntenne; i++)
                {
                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok=1;
                        test=true;
                        break;
                    }
                }

                // ou si c'est une nouvelle antenne, on l'ajoute au tableau

                if (!test)
                {
                    Antennes[numAntenne].ip = atoi(decomp.c_str());
                    Antennes[numAntenne++].ok = 1;
                }
            }

            // erreur sur une antenne -> on change son état dans la fenêtre

            if  ((reponse.find("ALERTE antenne ")!=string::npos) ||
                    (reponse.find("Erreur sur l antenne ")!=string::npos))
            {
                decomp = reponse.substr(0, reponse.find(" :"));
                decomp = decomp.substr(decomp.rfind(".") + 1);

                for (int i=0; i<numAntenne; i++)
                {
                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok = 2;  // Erreur sur l'antenne i
                    }
                }
            }

            // On sauvegarde le message dans le tableau logs

            stringstream os;

            os << lognum << " : " << reponse;

            Affiche(os.str() + "\n", false);

            if (lognum<MAXLOG-1) logs[lognum++] = os.str();

            reponse = memreponse;
        } 
        else reponse="";
	  

    } while ( reponse !="" );

     // on actualise la fenêtre

        Dessiner();

    // on envoie un message pour actualiser la fenêtre graphique

    rouler();
}


/**************************************************************************************
** Est-ce que la réponse du pilote indi_BAO est conforme à ce que l'on attendait ?
***************************************************************************************/

bool BAOcontrol::VerifReponse(string reponseattendue)
{
    string reponse = "";
    string memreponse = "";
    int    duree = 0;
    bool   test = false;

    // TODO :La réponse du driver ne vient pas tout de suite et on peut recevoir
    // des messages intermédiaires un peu étranges impliquant la fonction CONNECT, pourquoi ?
    // Là on lit les messages trois fois de suite pour essayer de voir passer le bon
    // Essayer de voir si on ne peut pas faire mieux...

    for (int i=0; i<3 ;i++)
    {
        do
        {
            usleep(1000); // attendre 1 ms pour laisser le temps au pilote indi_BAO de répondre

            *client_socket >> reponse;

            duree++;
        }
        while (reponse.length() == 0 && duree<10 ); // on attend un message pendant 10 ms max

        // on ajoute tous les messages reçus dans memmreponse
        // ->intéressant pour le debug

        memreponse += reponse;

        if (reponse.find(reponseattendue) != string::npos)
        {
            test = true;
            break;
        }
    }

    if (!test)
    {
        // réponse inconnue -> une erreur ?

        int pos = memreponse.find("message=");

        if ( pos != string::npos )
        {
            memreponse = memreponse.substr(pos + 9);

            memreponse = memreponse.substr(0, memreponse.find("\""));

            Erreur("Réponse du pilote indi_BAO :" + memreponse + "\n\n");
        }

        return false;
    }

    return true;
}


/**************************************************************************************
** Calcul du jour julien et du temps sidéral local
***************************************************************************************/

void BAOcontrol::Update()
{
    struct tm date;
    time_t t;
    struct timeval tv;
    struct timezone tz;

    // On récupère la date et l'heure depuis l'horloge du système

    time(&t);
    date=*gmtime(&t);
    gettimeofday(&tv, &tz);

    double Annee = (double)date.tm_year + 1900.0;
    double Mois  = (double)date.tm_mon + 1.0;
    double Jour  = (double)date.tm_mday;
    double Heu   = (double)date.tm_hour;
    double Min   = (double)date.tm_min;
    double Sec   = (double)date.tm_sec + tv.tv_usec / 1.0E6;
    double UTCP  = 0.0;//(double)date.tm_isdst;

    /*Annee=2011;
    Mois=6;
    Jour=17;
    Heu=15;
    Min=53.0;
    Sec=00.0;*/

    // On actualise la date et l'heure dans la classe Astro

    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);


    // On lance le calcul

    CalculTSL();
}


/**************************************************************************************
** gestion du thread permettant
** -l'exécution d'une liste de mouvements programmés
** -l'actualisation de la fenêtre graphique
** -la récupération des messages du pilote indi_BAO
***************************************************************************************/

void *BAOcontrol::my_thread_process ()
{
    stringstream os;

    // Tant ue l'utilisateur n'a pas taper la commande 'exit'
    while (NoExit)
    {
        //On actualise le jour julien et le temps sidéral local

        Update();

        // On récupère le jour julien

        double JJ = GetJJ();


        // Faut-il executer un fichier de mouvements ?
        // il faut qu'il y ait plus d'un mouvement
        // et exxecuter les commande dans ce test jusqu'au dernier mouvement

        if (run && numobjets>1 && runnum<numobjets)
        {
            // On lance le mouvement numéro runnum si la date correspond au début
            // de son exécution et que ce mouvement n'a pas été encore réalisé

            if (JJ >= objets[runnum].JJ && !objets[runnum].exec)
            {
                // petits messages pour l'utilisateur

                if (runnum==1) Affiche("Début de l'exécution du fichier de mouvements\n\n", true);

                os << "Suivi de l'objet n°" << runnum << endl;
                Affiche(&os, true);

                // on indique que le mouvement est en cours d'exécution
                //ou qu'il a été déjà exécuté

                objets[runnum].exec=true;

                //on envoie l'ordre goto correspondant

                Goto(objets[runnum].ad, objets[runnum].de, Transit, J2000);
            }


            // On arrive à la fin de l'exécution du mouvements runnum

            if (JJ >= objets[runnum].JJ + objets[runnum].Duree / 3600.0 / 24.0 )
            {
                // On interrompt le mouvement

                Abort();

                // S'il s'agit du dernier mouvement -> Fin de la commande run

                if ( runnum + 1 >= numobjets )
                {
                    Affiche("Fin de l'execution du fichier de mouvements\n\n", true);

                    // on réaffiche le prompt

                    cout << endl;
                    ChoixCouleurs==1 ? cout << blue1 : cout << blue2;
                    cout << ">";

                    // on réinitiale toutes les variables impliquées dans la commande run

                    run=false;
                    runnum=1;
                    numobjets=1;

                    // On sort du programme dans le cas BAOControl -r fileName

                    NoExit=!exitrun;
                }
                else
                {
                    // on passe au mouvement suivant

                    runnum++;
                }
            }
        }

        // on lit et affiche la réponse des microcontrolleurs

        LireReponse();

        Dessiner();

        // pause de 100 ms

        usleep(100000);
    }

    // l'utilisateur a tapé exit... On sort du thread...


    pthread_exit (0);
}


void* LancementThread(BAOcontrol * appli)
{
    appli->my_thread_process();

    return 0;
}




/**************************************************************************************
** Décrypte les commandes saisies par l'utilisateur
** Exemples :
** DecompositionCommande("goto 10:10:10 25:10:10", "goto", &chaine1, &chaine2)
** retourne chaine1  = "10:10:10"
**          chaine2  = "25:10:10"
**
** DecompositionCommande("goto messier 1","goto", &chaine1, NULL) (avec chaine2=NULLL)
** retourne chaine1  = "messier 1"
**          chaine2  = NULL
***************************************************************************************/

bool BAOcontrol::DecompositionCommande(string chaine, string commande, string *chaine1, string *chaine2)
{
    size_t pos = chaine.find(commande);

    // si la commande est présente dans chaine, on continue

    if (pos != string::npos)
    {
        // on s'intéresse aux caractères qui suivent la commande

        pos += commande.length();

        chaine = chaine.substr(pos);

        // on ignore les espaces superflus

        while (chaine[0] == ' ') chaine = chaine.substr(1);

        // si on veut prendre les deux arguments d'un commande

        if (chaine2 != NULL)
        {
            // on prend le premier bloc jusqu'au prochain espace -> c'est la chaine1

            *chaine1 = chaine.substr(0, chaine.find(" "));

            // puis on prend le deuxième bloc -> c'est la chaine2
            pos = chaine.find(" ");

            if (pos != string::npos)
            {
                chaine = chaine.substr(pos + 1);

                while (chaine[0] == ' ') chaine = chaine.substr(1);

                *chaine2 = chaine.substr(0, chaine.find(" "));
            }
        }
        else
        {
            // on prend tout d'un bloc après la commande -> chaine 1
            // utile pour une commande ty type goto m 42 qui doit retourner "m 42" dans une
            // même chaine

            *chaine1 = chaine;
        }
    }

    return true;
}



/**************************************************************************************
** Envoie la longitude et la latitude du lieu d'observation au pilote indi_BAO
***************************************************************************************/

bool BAOcontrol::EnvoyerCoordGeographiques()
{
    // la communication avec le pilote indi_BAO
    // se fait par l'envoie de petit fichier xml
    // d'où le côté complexe des commandes

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"GEOGRAPHIC_COORD\">";
        *client_socket << "<oneNumber name=\"LAT\">";
        *client_socket << LatitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"LONG\">";
        *client_socket << LongitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"GEOGRAPHIC_COORD\" state=\"Ok\""))
        {
            Erreur("ERREUR fct EnvoyerCoordGeographiques : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Les coordonnées géographiques ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}



/**************************************************************************************
** Place les antennes en position de repos
***************************************************************************************/

bool BAOcontrol::Park()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"\" >";
        *client_socket << "<oneSwitch name=\"PARK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("PARK OK"))
        {
            Erreur("La commande PARK a échoué.\n\n");;

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Le pilote indi_BAO a bien reçu la commande PARK.\n\n", true);

    return true;
}



/**************************************************************************************
** Annule le mouvement en cours
***************************************************************************************/

bool BAOcontrol::Abort()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ABORT_MOTION\" >";
        *client_socket << "<oneSwitch name=\"ABORT\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("ABORT OK"))
        {
            Erreur("L'annulation a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Le pilote indi_BAO a bien reçu la commande ABORT.\n\n", true);

    return true;
}



/**************************************************************************************
** Diriger l'antenne vers les coordonnées ar et dec et suivre l'objet
** en activant le mode transit ou tracking.
** si J2000 == true, cela signifie que les coordonnées ar et dec sont données dans le
** le système de coordonnées J2000 (écliptique et équinoxe du 1 janvier 2000 à 0 h TU)
** Des calculs supplémentaires (précession, nutation, abérration) sont alors nécessaires
***************************************************************************************/

bool BAOcontrol::Goto(string ar, string dec, bool Transit, bool J2000)
{
    float  ar1, ar2, ar3;
    float  dec1, dec2, dec3;
    double arf, decf;
    double azi, hau;


     /*
     //Pour les tests
     ar="05:39:14.8";
     dec="23:19:24.1";
     printf("JJ=%10.10f\n", GetJJ());
     J2000=true;
     */

    // Conversion de l'AD et de la déclinaison en radians

    Decomposition(ar, 2, &ar1, &ar2, &ar3);
    Decomposition(dec, 0, &dec1, &dec2, &dec3);
    
    arf  = ( ar1  + ar2 / 60.0  + ar3 / 3600.0 ) * 15.0 * Pidiv180;
    decf = ( fabs(dec1) + dec2 / 60.0 + dec3 / 3600.0 ) * Pidiv180;

    if (dec[0]=='-') decf = -decf;

    // calculs pupplémentaires pour se ramener
    // à l'époque de l'observation

    if (J2000)
    {
        Precession(&arf, &decf);
        NutationEtoile(&arf, &decf);
        AberrationAnnuelle(&arf, &decf);
    }

    // on réécrit l'ar et la dec dans le format d'indi_BAO

    ar  = DHMS(arf * N180divPi, true);

    dec = DHMS(decf * N180divPi, false);

    // on en informe l'utilisateur

    Affiche("Coordonnées apparentes de l'objet :\n", true);

    Affiche("AD=" + ar  + "   Dec=" + dec +"\n", true);

   
    /*
    //Tests
    Azimut(arf, decf, &azi, &hau);

    hau=RefractionAtmospherique(hau);

    Affiche("Coordonnées rectangulaires liées au lieu d'observation :\n", true);

    printf("Azimut = %s   Hauteur = %s\n", DHMS(azi*N180divPi,false).c_str(), DHMS(hau*N180divPi, false).c_str());
    */

    // On transmet les coordonnées au pilote BAO

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ON_COORD_SET\">";
        *client_socket << "<oneSwitch name=\"TRANSIT\">";
        if (Transit) *client_socket << "On";
        else *client_socket << "Off";
        *client_socket << "</oneSwitch>";

        *client_socket << "<oneSwitch name=\"TRACKING\">";
        if (Transit)  *client_socket << "Off";
        else *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ON_COORD_SET\""))
        {
            Erreur("Le changement de mode TRANSIT/TRACKING a échoué.\n\n");

            return false;
        }

        *client_socket << "<newNumberVector device=\"BAO\" name=\"EQUATORIAL_EOD_COORD_REQUEST\">";
        *client_socket << "<oneNumber name=\"RA\">";
        *client_socket << ar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"DEC\">";
        *client_socket << dec;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"EQUATORIAL_EOD_COORD_REQUEST\""))
        {
            Erreur("Le transfert des coordonnées de l'objet a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Les nouvelles coordonnées AD=" + ar + " Dec=" + dec, true);
    Affiche(" ont été envoyées au pilote indi_BAO.\n\n", true);
    return true;
}



/**************************************************************************************
** Se connecte ou se déconnecte au pilote indi_BAO
***************************************************************************************/

bool BAOcontrol::Connect(bool connect)
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"CONNECTION\">";

        if (connect) *client_socket << "<oneSwitch name=\"CONNECT\">";
        else *client_socket << "<oneSwitch name=\"DISCONNECT\">";

        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (connect)
        {
            if (!VerifReponse("BAORadio is online"))
            {
                Erreur("La connexion a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (!VerifReponse("BAORadio is offline"))
            {
                Erreur("La déconnexion a échoué.\n\n");

                return false;
            }
        }
    }

    catch ( SocketException& e )
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    if (connect)
    {
        Affiche("La connexion a été établie avec le pilote indi_BAO.\n\n", true);

        EnvoyerCoordGeographiques();
    }
    else
    {
        Affiche("BAOcontrol s'est bien déconnecté du pilote indi_BAO.\n\n", true);
    }

    return true;
}



/**************************************************************************************
** Lecture des mouvements planifiés contenus dans le fichier fileName
***************************************************************************************/

bool BAOcontrol::LectureFichierMouvements(string fileName)
{
    string section;
    string key;
    char * value   = NULL;

    stringstream os;
    float a1, a2, a3, h;

    numobjets=1;

    Affiche("Lecture du fichier " + fileName +"\n\n", true);

    //Lecture de la rubrique Paramètres

    section = "parametres";
    key     = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = ( strstr(value, "transit") != NULL);
        delete [] value;

        if (Transit) Affiche("Mode transit activé.\n", true);
        else Affiche("Mode tracking activé.\n", true);
    }
    else
    {
        Erreur("Le parametre mode est incorrect.\n");
        return false;
    }

    key = "J2000";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        J2000 = ( strstr(value, "on") != NULL);
        delete [] value;

        if (J2000) Affiche("Mode J2000 activé.\n\n", true);
        else Affiche("Mode J2000 désactivé.\n\n", true);
    }
    else
    {
        Erreur("Le parametre J2000 est incorrect.\n");
        return false;
    }

    // Lecture de la liste des observations

    section = "objet 1";
    key     = "date";


    while ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (numobjets<MAXOBJETS))
    {
        os << "object n°" << numobjets << endl;
        Affiche(&os, true);
        delete [] value;
        value = NULL;


        // Lecture de la date et de l'heure de l'observation

        key = "date";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 3, &a1, &a2, &a3)))
        {
            Affiche("Date du début de l'observation : le " + (string)value, true);
            delete [] value;
            value = NULL;

            objets[numobjets].JJ = CalculJJ(a3, a2, a1, 0.0);
        }
        else
        {
            os << "La date de la rubrique [objet " << numobjets << "] est incorrecte !" << endl;
            Erreur(&os);
            return false;
        }

        key = "heure";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 2, &a1, &a2, &a3)))
        {
            Affiche(" à " + (string)value, true) ;
            delete [] value;
            value = NULL;


            objets[numobjets].JJ += ( a1 + a2 / 60.0 + a3 / 3600.0 ) / 24.0;

            os << " TU  -> JJ=" << fixed << objets[numobjets].JJ << scientific << endl;

            Affiche(&os, true);

            for (int i=1; i<numobjets; i++)
            {
                if (objets[i].JJ==objets[numobjets].JJ)
                {
                    os << "Attention, les observations des objets " << i << " et ";
                    os << numobjets << " sont définies à la même date !" << endl;
                    Erreur(&os);
                }

                if (objets[i].JJ+objets[i].Duree/3600/24.0>objets[numobjets].JJ)
                {
                    os << "Attention, les observations des objets " << i << " et " ;
                    os << numobjets << " se chevauchent !" << endl;
                    Erreur(&os);
                }
            }
        }
        else
        {
            os << "L'heure de la rubrique [objet " << numobjets << "] est incorrecte !" << endl;
            Erreur(&os);
            return false;
        }


        // Lecture de la durée de l'observation

        key = "duree";

        objets[numobjets].Duree=0.0;

        if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
        {
            Affiche("Durée de l'observation : " + (string)value + " secondes\n", true);
            objets[numobjets].Duree=atof(value);
            delete [] value;
            value = NULL;
        }
        else
        {
            os << "La duree d'observation de la rubrique [objet " << numobjets << "] est incorrecte !\n";
            Erreur(&os);
            return false;
        }

        // Lecture de l'ascension droite de l'objet

        key = "ad";

        objets[numobjets].ad = "";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 2, NULL, NULL, NULL)))
        {
            Affiche("Ascension droite de l'objet : " + (string)value + "\n", true);
            objets[numobjets].ad = value;
            delete [] value;
            value = NULL;
        }
        else
        {
            os << "L'ascension droite de la rubrique [objet " << numobjets << "] est incorrecte !";
            os << endl;
            Erreur(&os);
            return false;
        }

        // Lecture de la déclinaison de l'objet

        key = "de";

        objets[numobjets].de = "";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 0, NULL, NULL, NULL)))
        {
            Affiche("Déclinaison de l'objet : " + (string)value + "\n\n", true);
            objets[numobjets].de = value;
            delete [] value;
            value = NULL;
        }
        else
        {
            os << "La déclinaison de la rubrique [objet " << numobjets << "] est incorrecte !";
            os << endl;
            Erreur(&os);
            return false;
        }

        objets[numobjets].exec = false;

        // On passe à l'objet suivant

        numobjets++;

        os << "objet "<< numobjets;

        section = os.str();
        os.str("");
    }

    if (value) delete [] value;

    return true;
}



/**************************************************************************************
** Execute le fichier de mouvements
***************************************************************************************/

bool BAOcontrol::Run(string fichier)
{
    stringstream os;

    bool exec = LectureFichierMouvements(fichier);

    Update();

    if ( exec && numobjets>1 )
    {
        // Si le chargement du fichier s'est bien déroulé et qu'il y a plus d'un mouvement dedans

        if (objets[numobjets-1].JJ + objets[numobjets-1].Duree / 3600.0 / 24.0 < GetJJ())
        {
            Erreur("Le fichier contient des dates qui sont toutes périmées. Exécution annulée.\n\n");
            NoExit=!exitrun;
            return false;
        }
        else
        {
            Affiche("Execution du fichier " + fichier + " en attente...\n\n", true);
            run=true;
        }
    }
    else
    {
        Erreur("\nLe fichier " + fichier + " contient des erreurs ou n'existe pas...\n\n\n");
        NoExit=!exitrun;
        return false;
    }

    return true;
}



/**************************************************************************************
** Décomposition et vérification des dates, heures, AD et déclinaisons
**
** type = 0 -> Déclinaison/latitude
** type = 1 -> longitude
** type = 2 -> AD ou heure
** type = 3 -> date
**
** Exemple de Decomposition("15:23:12", 2, a, b, c) retourne a=15, b=23, c=12
** si la chaine a un format incorrect, la fct retourne false
***************************************************************************************/

bool BAOcontrol::Decomposition(string chaine, char type, float *a1, float *a2, float *a3)
{
    string car, s;
    float a, b, c;

    // pour les heures et les coordonnées, on attend ":" comme caractère séparateur, sinon
    // on attend d'avoir des "/" pour une date

    (type==3) ? car="/" : car=":";

    // Y a-t-il 2 caractères ':' ou '/' dans la chaine ?
    // C'est indispensable dans tous les cas

    int test=0;
    for (int i=0; i<chaine.length(); i++) if (chaine[i] == car[0]) test++;
    if (test<2) return false;

    // Extraction des trois nombres

    s = chaine.substr(0, chaine.find(car));

    a = atoi(s.c_str());

    s = chaine.substr(chaine.find(car)+1, chaine.rfind(car) - chaine.find(car) - 1);

    b = atoi(s.c_str());

    s = chaine.substr(chaine.rfind(car)+1);

    c = Arrondi(atof(s.c_str())*100.0)/100.0;

    //vérification de la cohérence des infos contenues dans la chaine

    if (type < 3 )
    {
        // pour une déclinaison
        if ((type==0) && (a>90.0  || a<-90.0)) return false;
        // pour une AD
        if ((type==1) && (a>360.0 || a<0.0  )) return false;
        // pour une heure
        if ((type==2) && (a>23.0  || a<0.0  )) return false;
        // pour les minutes
        if (b<0.0 || b>59.0 ) return false;
        //pour les secondes
        if (c<0.0 || c>=60.0) return false;
    }
    else
    {
        //pour les jours
        if (a<0.0 || a>31.0) return false;
        //pour les mois
        if (b<0.0 || b>12.0) return false;
    }

    if (a1!=NULL) *a1 = a;
    if (a2!=NULL) *a2 = b;
    if (a3!=NULL) *a3 = c;

    return true;
}




/**************************************************************************************
** identification des commandes entrées par l'utilisateur
** puis exécution des commandes
***************************************************************************************/

void BAOcontrol::DecodageEntreesUtilisateur(string chaine)
{
    string ar, de, objet, fichier;
    stringstream os;

    // Permet de savoir si la dernière saisie de l'utilisateur
    // est une commande valide
    bool CommandeOK = false;


    // Commande goto

    if ( chaine.find("goto") != string::npos )
    {
        CommandeOK = true;

        // L'utilisateur a fait suivre la commande goto de l'indication J2000

        if (J2000 = (chaine.find("j2000") != string::npos))
        {
            Affiche("Prise en compte de la précession, de la nutation et de l'aberration...\n", true);

            // On je garde que la première partie de la chaine avant J2000

            chaine = chaine.substr(0, chaine.find("J2000"));
        }

        if (DecompositionCommande(chaine, "goto", &ar, &de) && Decomposition(ar, 2, NULL, NULL, NULL) && Decomposition(de, 0, NULL, NULL, NULL))
        {
            // L'utilisateur a indiqué deux coordonnées valides -> on exécute le goto

            Goto(ar, de, Transit, J2000);
        }
        else
        {
            // L'utilisateur a indiqué un nom d'objet après le goto -> on lance la recherche sur internet
            // puis on envoie les coordonnées de l'objet au pilote indi_BAO

            if ((DecompositionCommande(chaine, "goto", &objet, NULL)) && (chaine.find(":") == string::npos))
            {
                Coordonnees reponse;
		
		if (objet != "sun")
		{
		    reponse = ServeurNED(objet);
		    
		    if (reponse.ar!="" && reponse.dec!="")  Goto(reponse.ar, reponse.dec, Transit, true);
		}
		else
		{   
		    printf(" lancement\n");
		   
		    CalculARDecSoleil(&reponse);
		    
		    printf("%s   %s \n", reponse.ar.c_str(), reponse.dec.c_str());
		    
		    Goto(reponse.ar, reponse.dec, Transit, false);
		}                
            }
            else
            {
                Erreur("Erreur goto : le format de la commande est goto AD DEC\n \
                    L AD doit être de la forme xx:yy:zz   (ex: 12:00:03 pour 12h 0m 3s)\n \
                    La déclinaison doit être de la forme +/-xx:yy:zz (ex: -12:50:01 pour -12° 50' 01'')\n\n \
                    Il est egalement possible de faire suivre la commande d'un nom d'un objet\n \
                    Exemple goto M 31 ou encore goto ngc145\n \
                    Ne pas préciser dans ce cas J2000.\n\n");
            }
        }
    }

    // Annulation du suivi en cours

    if (chaine.find("abort") != string::npos)
    {
        CommandeOK = true;

        Abort();

        // Annulation de l'exécution d'un fichier de mouvements

        if (chaine.find("run") != string::npos)
        {
            run=false;
            runnum=1;
            numobjets=1;

            Affiche("Annulation de l'exécution du fichier de mouvements.\n\n", true);

            return;
        }
    }

    // Lance l'exécution d'un fichier de mouvements

    if (chaine.find("run") != string::npos)
    {
        CommandeOK = true;

        if (DecompositionCommande(chaine, "run", &fichier, NULL))
        {
            Run(fichier);
        }
        else
        {
            Erreur("Erreur run : la commande run doit être suivie par un nom de fichier.\n\n");
        }
    }

    // se connecte/déconnecte au serveur

    if (chaine.find("connect") != string::npos)
    {
        CommandeOK = true;

        Connect(chaine.find("disconnect") == string::npos);
    }

    // Active le mode de suivi "transit"

    if (chaine.find("transit") != string::npos)
    {
        CommandeOK = true;

        Affiche("Mode transit activé.\n\n", true);

        Transit=true;
    }

    // Active le mode de suivi "tracking"

    if (chaine.find("tracking") != string::npos)
    {
        CommandeOK = true;

        Affiche("Mode tracking activé.\n\n", true);

        Transit=false;
    }

    // Place les antennes en position de repos

    if (chaine.find("park") != string::npos)
    {
        CommandeOK = true;

        Park();
    }

    // interroge un serveur de temps pour mettre l'horloge du PC à jour

    if (chaine.find("updatetime") != string::npos)
    {
        CommandeOK = true;

        int rc = system("sntp -s fr.pool.ntp.org");

        if (WEXITSTATUS(rc) ==-1 || WEXITSTATUS(rc) ==127)
        {
            Erreur("La commande sntp n'est pas accessible. Merci de vérifier.\n\n");
        }
    }

    // Affiche tous les paramètres du programme

    if (chaine.find("status") != string::npos)
    {
        CommandeOK = true;

        os << "Latitude = " << (string)LatitudeChar << "\n";
        os << "Longitude = " << (string)LongitudeChar << "\n\n";
        os << "Serveur Indi = " << (string)Serveur << "\n";
        os << "Port du serveur Indi = " << Port << "\n\n";
        os << "Pression atmosphérique  = " << Pression << " mBar\n";
        os << "Température = " << Temperature << " °C\n\n";
        Affiche(&os, true);
        if (Transit) Affiche("Mode Transit activé.\n\n", true);
        else Affiche ("Mode Tracking activé.\n\n", true);
    }

    // interroge le serveur NED sur internet -> retourne les coordonnées de l'objet

    if (chaine.find("search")!=string::npos)
    {
        string objet;

        CommandeOK = true;

        if (DecompositionCommande(chaine, "search", &objet, NULL))
        {
            ServeurNED(objet);
        }
    }

    // Aide du programme

    if (chaine.find("help") != string::npos)
    {
        CommandeOK = true;

        cout << endl;

        cout << "connect :           se connecte au pilote indi_BAO." << endl;
        cout << "disconnect :        se déconnecte du pilote indi_BAO." << endl;
        cout << "goto AD Dec :       pointe l'objet situé aux coordonnées apparentes AD Dec." << endl;
        cout << "                    AD doit être au format xx:yy:zz et dec au format +/-xx:yy:zz" << endl;
        cout << "                    exemple goto 12:10:05 -05:10:11 permettra de pointer l'objet" << endl;
        cout << "                    situé aux coordonnées AD=12h10m5s et dec=-5°10'11''" << endl;
        cout << "goto AD Dec J2000 : pointe l'objet situé aux coordonnées AD Dec dans le repère J2000." << endl;
        cout << "                    Avant de pointer l'objet, le programme calcule la précession, la nutation et l'aberration" << endl;
        cout << "                    pour ramener les coordonnées à l'écliptique et à l'équinoxe de l'observation." << endl;
        cout << "goto nom_objet :    interroge la base NED pour trouver les coordonnées de l'objet puis le pointe." << endl;
        cout << "                    exemples: goto m 31, goto messier 1, goto ngc 175, goto ic 434 etc..." << endl;
        cout << "                    Dans ce mode, les coordonnées sont automatiquement rapportées" << endl;
        cout << "                    à l'écliptique et à l'équinoxe de l'observation." << endl;
        cout << "                    Aussi, merci de ne pas faire suivre les coordonnées de l'indication J2000."<< endl;
        cout << "                    Goto sun permet de suivre le soleil." << endl;
        cout << "search nom_objet :  interroge le serveur NED sur internet et retourne les coordonnées de l'objet." << endl;
        cout << "transit :           active le mode de suivi transit." << endl;
        cout << "tracking :          active le mode de suivi tracking." << endl;
        cout << "status :            liste les paramètres du programme." << endl;
        cout << "run filename :      exécute le fichier de mouvements filename." << endl;
        cout << "abort run :         annule l'exécution du fichier de mouvements." << endl;
        cout << "abort :             annule le mouvement en cours." << endl;
        cout << "park :              place les antennes en position de repos." << endl;
        cout << "updatetime :        synchronise l'horloge du PC avec un serveur de temps." << endl;
        cout << "exit :              sortir de BAOControl." << endl;

        cout << endl;
    }

    // Bye !

    if (chaine.find("exit")!=string::npos)
    {
        CommandeOK = true;

        Abort();

        Connect(false);

        NoExit = false;
    }

    // La commande n'a pas été identifiée !

    if (!CommandeOK) Erreur("Commande inconnue... Tapez help pour obtenir la liste des commandes disponibles.\n\n");
}


/**************************************************************************************
** Interroge le serveur NED sur internet et retourne les coordonnées dans le repère J2000
** Si pas de réponse, alors message d'erreur et les coordonnées sont égales à AR="" dec=""
***************************************************************************************/

Coordonnees BAOcontrol::ServeurNED(string objet)
{
    Coordonnees reponse;
    stringstream send;

    FILE *pFile = NULL;
    int rc;

    //initialsation des variables de retour

    reponse.ar = "";
    reponse.dec = "";

    // le nom de l'objet ne doit pas contenir des espace pour la requête NED
    // on les remplace par des '+'

    for (int i=0; i<objet.length(); i++) if (objet[i] == ' ') objet[i]='+';

    // Effacer le fichier result s'il existe dans le répertoire du programme

    if ((pFile = fopen ("result", "r")) != NULL )
    {
        fclose (pFile);
        rc = system( "rm result");
    }

    // On utilise l'utilitaire curl pour passer la requête au serveur NED
    // On sauvegarde le résultat dans le fichier result

    send << "curl >> result \"http://ned.ipac.caltech.edu/cgi-bin/nph-objsearch?objname=";
    send << objet;
    send << "&extend=no&hconst=73&omegam=0.27&omegav=0.73&corr_z=1&out_csys=Equatorial&out_equinox=J2000.0";
    send << "&obj_sort=RA+or+Longitude&of=ascii_bar&zv_breaker=30000.0&list_limit=5&img_stamp=NO\"";

    Affiche("Envoi de la requête au serveur NED...\n", true);

    // on lance la requête

    rc = system(send.str().c_str() );

    // traitement du résultat

    if (WEXITSTATUS(rc) ==-1 || WEXITSTATUS(rc) ==127)
    {
        Erreur("La commande curl n'est pas accessible. Merci de vérifier.\n\n");
    }
    else
    {
        // On a obtenu quelque chose

        int nbRead = 0;

        char *pBuf;

        pBuf = new char [100000];

        // On laisse un peu de temps au serveur pour répondre.
        // C'est pas vraiment indispensable mais bon...

        sleep(1);

        // On ouvre le fichier result

        pFile = fopen ("result", "r");

        if (pFile)
        {
            nbRead = readline (pFile, pBuf);

            // Si le fichier n'est pas vide
            if (nbRead!=-1)
            {
                if (strstr(pBuf, "Error") != NULL)
                {
                    // Le fichier contient une indication d'erreur -> pas de réponse exploitable

                    Affiche("\nLe serveur NED n'a pas trouvé de réponse à votre demande...\n", true);
                }
                else
                {
                    // on analyse la réponse

                    Affiche("\nRéponse du serveur :\n", true);

                    do
                    {
                        // on cherche la première suggestion de NED
                        // Il faudra peut-être un peu complexifier par la suite

                        if (pBuf[0] == '1')
                        {
                            string chaine = (string)pBuf;
                            string chaine2;

                            chaine=chaine.substr(chaine.find('|')+1);

                            // affiche le nom de l'objet dans le serveur NED

                            Affiche(chaine.substr(0, chaine.find('|')) + "\n", true);

                            chaine = chaine.substr(chaine.find('|') + 1);

                            chaine2 = chaine.substr(0, chaine.find('|'));

                            double ar = atof(chaine2.c_str());

                            // affiche l'ascension droite

                            Affiche("AD (J2000) = " + DHMS(ar, true) + "\n", true);

                            reponse.ar = DHMS(ar, true);

                            chaine=chaine.substr(chaine.find('|')+1);

                            chaine2 = chaine.substr(0, chaine.find('|'));

                            double dec = atof(chaine2.c_str());

                            // on affiche la déclinaison

                            Affiche("Dec (J2000) = " + DHMS(dec, false)+"\n", true);

                            reponse.dec = DHMS(dec, false);
                        }

                        nbRead = readline (pFile, pBuf);

                    } while (nbRead!=-1);
                }
            }
            else
            {
                Erreur("\nLa connexion semble impossible avec le serveur NED !\n\n");
            }

            fclose(pFile);
        }
        else
        {
            Erreur("\nLa connexion semble impossible avec le serveur NED !\n\n");
        }

        delete [] pBuf;

        Affiche("\n", true);
    }

    return reponse;
}



/**************************************************************************************
** Lecteur d'un fichier ligne après ligne
** retourne -1 à la fin
***************************************************************************************/

int BAOcontrol::readline (FILE * pfile, char *tab)
{
    int nbchar = 0;
    char c;

    while ((c = getc (pfile)) != '\n')
    {
        if (c == EOF)
        {
            break;
        }
        tab[nbchar++] = c;
    }
    tab[nbchar] = '\0';

    while (nbchar>0 && tab[nbchar-1]==' ')
    {
        tab[--nbchar] = '\0';
    }

    while (tab[0]==' ') for (int i=1; i<=nbchar; i++) tab[i-1] = tab[i];

    if (c == EOF) nbchar=-1;

    return (nbchar);
}



/**************************************************************************************
** chargement des paramètres contenus dans le fichier params
***************************************************************************************/

bool BAOcontrol::ChargementParametres(string fileName)
{
    string section;
    string key;
    char * value;
    stringstream os;



    Affiche("Lecture du fichier de configuration 'params' :\n\n", true);


    /////////////////////////////////////////
    // Rubrique Coodonnées géographiques

    section = "coordonnees geographiques";

    key = "latitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 0, NULL, NULL, NULL)))
    {
        LatitudeChar = (string)value;
        delete [] value;
        Affiche("latitude = " + LatitudeChar +"\n", true);
    }
    else
    {
        Erreur("La latitude est incorrecte !\n");
        return false;
    }


    key = "longitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 1, NULL, NULL, NULL)))
    {
        LongitudeChar = (string)value;
        delete [] value;
        Affiche("longitude = " + LongitudeChar +"\n\n", true);
    }
    else
    {
        Erreur("La longitude est incorrecte !\n");
        return false;
    }


    /////////////////////////////////////////
    // rubrique connexion

    section = "connexion indi";

    key = "serveur";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Serveur = (string)value;
        delete [] value;
        Affiche("serveur = " + Serveur +"\n", true);
    }
    else
    {
        Erreur("Nom du serveur invalide !\n");
        return false;
    }


    key = "port";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Port = (string)value;
        delete [] value;
        Affiche("port = " + Port +"\n\n", true);
    }
    else
    {
        Erreur("Numéro de port incorrect !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique paramètres de l'atmosphère

    section = "atmosphere";

    key = "pression";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Pression = atof(value);
        delete [] value;
        os << "pression = " << Pression << endl;
        Affiche(&os, true);
    }
    else
    {
        os << "La pression atmosph&rique est incorrecte !" << endl;
        Erreur(&os);
        return false;
    }

    key = "temperature";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Temperature=atof(value);
        delete [] value;
        os << "température = " << Temperature << endl << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("La température est incorrecte !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique suivi

    section = "suivi";

    key = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = (strstr(value, "transit") != NULL);
        Affiche("mode suivi = " + (string)value + "\n", true);
        delete [] value;
    }
    else
    {
        Erreur("Le paramètre mode est incorrect !\n");
        return false;
    }



    key = "delai_transit";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitransit=atoi(value);
        delete [] value;
        os << "delai transit = " << delaitransit << " sec" << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("Le paramètre delai_transit est incorrect !\n");
        return false;
    }


    key = "delai_tracking";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitracking=atoi(value);
        delete [] value;
        os << "delai tracking = " << delaitracking << " sec" << endl << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("Le paramètre delai_tracking est incorrect !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique divers

    section = "divers";

    key = "couleurs";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        ChoixCouleurs=atoi(value);
        delete [] value;
    }
    else
    {
        /*os << "Le paramètre couleurs est incorrect !" << endl;
        Erreur(os.str());
        return false;*/
    }

    return true;
}



/**************************************************************************************
** Routine principale
***************************************************************************************/

int BAOcontrol::init(int argc, char **argv)
{
    string chaine;

    pthread_t th1;

    // gestion au démarrage des paramètres en ligne de commande

    if ( argc >1 )
    {
        if (strstr(argv[1], "-h")!=NULL)
        {
            cout << "usage :\n \
       BAOControl -h ou --help : affiche l'aide\n \
       BAOControl -r FileName : exécute les mouvements définis dans le fichier FileName\n\n";

            return 0;
        }
    }

    if (argc == 1)  cout << "Tapez help pour obtenir la liste des commandes disponibles.\n" << endl << endl;

    // On lance la fenêtre graphique

    initialiserFenetre();

    XSelectInput(d,w, ExposureMask );

    // On lance le thread pour gérer la fenêtre graphique
    // la communication avec indi_BAO et le suivi des objets

    if (pthread_create(&th1, NULL, (void*(*)(void*))LancementThread, this) < 0) {
        Erreur("Impossible de créer le thread\n");
        return -1;
    }



    // si on a utilisé l'option -r en ligne de commande
    // alors on charge et on lance l'exécution d'un fichier de mouvements

    if (argc == 3)
    {
        if (strstr(argv[1], "-r")!=NULL)
        {
            exitrun=true;

            Run(argv[2]);

            // on attend ici la fin de l'exécution du fichier de mouvements

            do
            {

            } while (NoExit);
        }
        else
        {
            Erreur("Usage incorrect\n\n");
        }
    }
    else
    {
        // Saisie et décodage des commandes entrées par l'utilisateur

        do
        {
            // Prompt
            ChoixCouleurs==1 ? cout << blue1 : cout << blue2;
            cout << '>';
            // saisie
            getline(cin, chaine);
            ChoixCouleurs==1 ? cout << grey1 : cout << grey2;

            // on met en minuscules
            for (int i=0; i<chaine.size();++i) chaine[i] = tolower(chaine[i]);

            // on décode et exécute les commandes
            DecodageEntreesUtilisateur(chaine);
        }
        while (NoExit);

        pthread_join(th1 ,NULL);
        //pthread_detach(th1);
    }

    return 0;
}
