/*
#############################
##
## BAORadio Indi driver
## Franck RICHARD
## franckrichard033@gmail.com
## Décembre 2011
##
#############################
*/

#include "BAO.h"


auto_ptr<BAO> telescope(0);

const int  POLLMS = 1;					// Period of update, 1 ms.

const char *mydev = "BAO";				// Name of our device.

const char *BASIC_GROUP    = "Main Control";		// Main Group
const char *OPTIONS_GROUP  = "Options";			// Options Group

/* Handy Macros */
//#define currentRA	EquatorialCoordsRN[0].value
//#define currentDEC	EquatorialCoordsRN[1].value
#define targetRA	EquatorialCoordsWN[0].value
#define targetDEC	EquatorialCoordsWN[1].value


static void ISPoll(void *);

void* LancementThread(BAO * appli);


/**************************************************************************************
** Initialisation de la classe BAO
***************************************************************************************/

BAO::BAO()
{
    init_properties();

    // télescope en état IDLE
    ConnectSP.s    = IPS_IDLE;

    // dernière actualisation
    lastRA         = 0.0;
    lastDEC        = 0.0;
    JJAnc          = 0.0;

    currentSet     =  0;
    lastSet        = -1;
    SocketsNumber  =  1;
    TrackingMode   =  1;

    // délais en sec entre deux actualisations de la
    // position dans les modes transit et tracking

    ActualisationTM1 = 15.0 * 60.0;
    ActualisationTM2 = 5.0;

    UpdateGoto      = true;
    InitThreadOK    = false;
    ActualisationPosition = false;
    Abort           = false;
    Park            = false;
    Suivi           = false;
    Exit            = false;


    // initialisation des sockets (Antennes)
    
    for (int i=0; i<MAXHOSTNAME; i++)
    {
        Sockets[i].Connected = false;
        Sockets[i].IP = "";
		
	Sockets[i].Delta.x = 0.0;
	Sockets[i].Delta.y = 0.0;
    }

    // initialisations supplémentaires
    
    InitAntennes();

    // Numéro de version
    
    IDLog("Initilizing from BAO device...\n");
    IDLog("Driver Version: 2011-12-02\n");

    //connect_telescope();
}


/**************************************************************************************
** Destructeur
** Lorsque l'on lance indi_BAO depuis indiserver dans un terminal
** Le destructeur ne semble jamais être atteint lorsque l'on sort... A vérifier
***************************************************************************************/
BAO::~BAO()
{
    Exit = true;
    sleep(1);
    pthread_join (th1, NULL);
}


/************************************************************************************
* Initialisation des paramètres des antennes
*
************************************************************************************/
void BAO::InitAntennes()
{
    for (int i=0; i<MAXHOSTNAME; i++)
    {
        Sockets[i].status             = 0;
        Sockets[i].sendalertes        = 0;
        Sockets[i].AttenteExecution   = 0;
        Sockets[i].AnomaliesExecution = 0;
        Sockets[i].etape              = 0;

        Sockets[i].ack_status = false;
        Sockets[i].ack_pos    = false;
        Sockets[i].ack_park   = false;
        Sockets[i].ack_abort  = false;
        Sockets[i].ack_goto   = false;

        Sockets[i].PosValides = false;
        Sockets[i].GotoOk     = false;
    }
}


/**************************************************************************************
** Initialisation des boutons et des zones d'affichage dans la boîte de dialogue INDI
***************************************************************************************/

void BAO::init_properties()
{
    // Connection
    IUFillSwitch(&ConnectS[0], "CONNECT", "Connect", ISS_OFF);
    IUFillSwitch(&ConnectS[1], "DISCONNECT", "Disconnect", ISS_ON);
    IUFillSwitchVector(&ConnectSP, ConnectS, NARRAY(ConnectS), mydev, "CONNECTION", "Connection", BASIC_GROUP, IP_RW, ISR_1OFMANY, 60, IPS_IDLE);

    // Coord Set
    IUFillSwitch(&OnCoordSetS[0], "TRANSIT", "Transit", ISS_ON);
    IUFillSwitch(&OnCoordSetS[1], "TRACKING", "Tracking", ISS_OFF);
    IUFillSwitchVector(&OnCoordSetSP, OnCoordSetS, NARRAY(OnCoordSetS), mydev, "ON_COORD_SET", "On Set", BASIC_GROUP, IP_RW, ISR_1OFMANY, 0, IPS_IDLE);

    // Abort
    IUFillSwitch(&AbortSlewS[0], "ABORT", "Abort", ISS_OFF);
    IUFillSwitchVector(&AbortSlewSP, AbortSlewS, NARRAY(AbortSlewS), mydev, "ABORT_MOTION", "Abort", BASIC_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Park
    IUFillSwitch(&ParkS[0], "PARK", "Park", ISS_OFF);
    IUFillSwitchVector(&ParkSP, ParkS, NARRAY(ParkS), mydev, "", "Park", BASIC_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Object Name
    IUFillText(&ObjectT[0], "OBJECT_NAME", "Name", "--");
    IUFillTextVector(&ObjectTP, ObjectT, NARRAY(ObjectT), mydev, "OBJECT_INFO", "Object", BASIC_GROUP, IP_RW, 0, IPS_IDLE);

    // Equatorial Coords - SET
    IUFillNumber(&EquatorialCoordsWN[0], "RA", "RA  H:M:S", "%10.6m",  0., 24., 0., 0.);
    IUFillNumber(&EquatorialCoordsWN[1], "DEC", "Dec D:M:S", "%10.6m", -90., 90., 0., 0.);
    IUFillNumberVector(&EquatorialCoordsWNP, EquatorialCoordsWN, NARRAY(EquatorialCoordsWN), mydev, "EQUATORIAL_EOD_COORD_REQUEST" , "Equatorial JNow", BASIC_GROUP, IP_WO, 0, IPS_IDLE);

    // Geographic coord - SET
    IUFillNumber(&GeographicCoordsWN[0], "LAT", "Lat  D", "%10.6m",  -90., 90., 0., 0.);
    IUFillNumber(&GeographicCoordsWN[1], "LONG", "Long D", "%10.6m", 0., 360., 0., 0.);
    IUFillNumberVector(&GeographicCoordsWNP, GeographicCoordsWN, NARRAY(GeographicCoordsWN), mydev, "GEOGRAPHIC_COORD" , "Geographic coords", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);

    // Actualisation - SET
    IUFillNumber(&ActualisationN1[0], "DELAY", "Transit mode delay (Sec)", "%10.6m",  0., 3600., 0., 0.);
    IUFillNumberVector(&ActualisationNP1, ActualisationN1, NARRAY(ActualisationN1), mydev, "DELAY1" , "", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);

    IUFillNumber(&ActualisationN2[0], "DELAY", "Tracking mode delay (Sec)", "%10.6m",  0., 3600., 0., 0.);
    IUFillNumberVector(&ActualisationNP2, ActualisationN2, NARRAY(ActualisationN2), mydev, "DELAY2" , "", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);
}


/**************************************************************************************
** Initialisation de la boîte de dialogue INDI (suite)
***************************************************************************************/

void BAO::ISGetProperties(const char *dev)
{

    if (dev && strcmp (mydev, dev))
        return;

    // Main Control
    IDDefSwitch(&ConnectSP, NULL);
    IDDefText(&ObjectTP, NULL);
    IDDefNumber(&EquatorialCoordsWNP, NULL);
    IDDefNumber(&GeographicCoordsWNP, NULL);
    IDDefSwitch(&OnCoordSetSP, NULL);
    IDDefSwitch(&AbortSlewSP, NULL);
    IDDefSwitch(&ParkSP, NULL);

    // Options
    IDDefNumber(&ActualisationNP1, NULL);
    IDDefNumber(&ActualisationNP2, NULL);
}


/**************************************************************************************
** Initialisation des vecteurs INDI
***************************************************************************************/

void BAO::reset_all_properties()
{
    ConnectSP.s			= IPS_IDLE;
    OnCoordSetSP.s		= IPS_IDLE;
    AbortSlewSP.s		= IPS_IDLE;
    ParkSP.s			= IPS_IDLE;
    ObjectTP.s			= IPS_IDLE;
    EquatorialCoordsWNP.s	= IPS_IDLE;
    GeographicCoordsWNP.s	= IPS_IDLE;
    ActualisationNP1.s		= IPS_IDLE;
    ActualisationNP2.s		= IPS_IDLE;

    IUResetSwitch(&OnCoordSetSP);
    IUResetSwitch(&AbortSlewSP);
    IUResetSwitch(&ParkSP);

    OnCoordSetS[0].s = ISS_ON;
    ConnectS[0].s = ISS_OFF;
    ConnectS[1].s = ISS_ON;

    IDSetSwitch(&ConnectSP, NULL);
    IDSetSwitch(&OnCoordSetSP, NULL);
    IDSetSwitch(&AbortSlewSP, NULL);
    IDSetSwitch(&ParkSP, NULL);
    IDSetText(&ObjectTP, NULL);
    IDSetNumber(&EquatorialCoordsWNP, NULL);
    IDSetNumber(&GeographicCoordsWNP, NULL);
    IDSetNumber(&ActualisationNP1, NULL);
    IDSetNumber(&ActualisationNP2, NULL);
}


/**************************************************************************************
** En cas de changement de texte dans la boîte de dialogue (par exemple : changement du
** nom de l'objet) alors suivre l'objet... 
** Cette fonction n'est pas encore utilisée
***************************************************************************************/

void BAO::ISNewText (const char *dev, const char *name, char *texts[], char *names[], int n)
{
    // Ignore if not ours
    if (strcmp (dev, mydev))
        return;

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! Please connect before issuing any commands.");
        reset_all_properties();
        return;
    }

    // ===================================
    // Object Name
    // ===================================
    if (!strcmp (name, ObjectTP.name))
    {
        if (IUUpdateText(&ObjectTP, texts, names, n) < 0)
            return;

        ObjectTP.s = IPS_OK;
        IDSetText(&ObjectTP, NULL);
        return;
    }
}


/**************************************************************************************
** En cas de changement d'une valeur numérique dans la boîte de dialogue Indi
** Exemple : longitude, latitude, ar, dec etc...) -> prendre en compte les modifications
**
***************************************************************************************/

void BAO::ISNewNumber (const char *dev, const char *name, double values[], char *names[], int n)
{

    // Ignore if not ours
    if (strcmp (dev, mydev))
        return;

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! BAO is offline. Please connect before issuing any commands.");
        reset_all_properties();
        return;
    }


    // ===================================
    // Geographic  Coords
    // ===================================
    if (!strcmp (name, GeographicCoordsWNP.name))
    {
        int i=0, nset=0, error_code=0;

        Latitude=0.0;
        Longitude=0.0;

        for (nset = i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&GeographicCoordsWNP, names[i]);
            if (eqp == &GeographicCoordsWN[0])
            {
                Latitude = values[i];
                nset += Latitude >= -90.0 && Latitude <= 90.0;

                Latitude *= Pidiv180;
            }
            else if (eqp == &GeographicCoordsWN[1])
            {
                Longitude = values[i];
                nset += Longitude >= 0.0 && Longitude <= 360.0;

                Longitude *= -Pidiv180;
            }
        }

        // Si la longitude et la latitude sont correctes
        // on envoie les coordonnées à la classe Astro
        if (nset == 2)
        {
            //Vérification
            //IDLog("Geographic : RA %5.2f - DEC %5.2f\n", Latitude, Longitude);

            GeographicCoordsWNP.s = IPS_OK;
            IDSetNumber(&GeographicCoordsWNP, NULL);
        }
        else
        {
            GeographicCoordsWNP.s = IPS_ALERT;
            IDSetNumber(&GeographicCoordsWNP, "Latitude or Longitude missing or invalid");

            Latitude=0.0;
            Longitude=0.0;
        }

        DefinirLongitudeLatitude(Longitude, Latitude);

        return;
    }


    // ===================================
    // Equatorial Coords
    // ===================================
    if (!strcmp (name, EquatorialCoordsWNP.name))
    {
        int i=0, nset=0, error_code=0;
        double newRA =0, newDEC =0;

        for (nset = i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&EquatorialCoordsWNP, names[i]);
            if (eqp == &EquatorialCoordsWN[0])
            {
                newRA = values[i];
                nset += newRA >= 0 && newRA <= 24.0;
            }
            else if (eqp == &EquatorialCoordsWN[1])
            {
                newDEC = values[i];
                nset += newDEC >= -90.0 && newDEC <= 90.0;
            }
        }


        // si les coordonnées de l'objet sont correctes
        if (nset == 2)
        {
            char RAStr[32], DecStr[32];
            double targetAZ, targetAlt;

            targetRA  = newRA;
            targetDEC = newDEC;

            fs_sexa(RAStr, newRA, 2, 3600);
            fs_sexa(DecStr, newDEC, 2, 3600);

            IDLog("We received JNow RA %s - DEC %s\n", RAStr, DecStr);

            // on convertit les coordonnées équatoriales de la zone du ciel observée
            // en unités de codeurs des moteurs

            ADDEC2Motor(newRA, newDEC);

            if (process_coords() == false)
            {
                EquatorialCoordsWNP.s = IPS_ALERT;
                IDSetNumber(&EquatorialCoordsWNP, NULL);
            }
        }
        else
        {
            EquatorialCoordsWNP.s = IPS_ALERT;
            IDSetNumber(&EquatorialCoordsWNP, "Error ! RA or Dec missing or invalid");
        }

        return;
    }

    // ===================================
    // Actualisation
    // ===================================
    if (!strcmp (name, ActualisationNP1.name))
    {
        int i=0, nset=0, error_code=0;
        double newAct1 =0;

        for (nset = i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&ActualisationNP1, names[i]);
            if (eqp == &ActualisationN1[0])
            {
                newAct1 = values[i];

                if (newAct1 >= 0.0 && newAct1 <= 3600.0)
                {
                    ActualisationTM1 = newAct1;

                    ActualisationNP1.s = IPS_OK;
                    IDSetNumber(&ActualisationNP1, NULL);
                }
                else
                {
                    ActualisationNP1.s = IPS_ALERT;
                    IDSetNumber(&ActualisationNP1, "Error ! Delay invalid");
                }
            }
        }
    }

    if (!strcmp (name, ActualisationNP2.name))
    {
        int i=0, nset=0, error_code=0;
        double newAct2 =0;

        for (nset = i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&ActualisationNP2, names[i]);
            if (eqp == &ActualisationN2[0])
            {
                newAct2 = values[i];

                if (newAct2 >= 0.0 && newAct2 <= 3600.0)
                {
                    ActualisationTM2 = newAct2;

                    ActualisationNP2.s = IPS_OK;
                    IDSetNumber(&ActualisationNP2, NULL);
                }
                else
                {
                    ActualisationNP2.s = IPS_ALERT;
                    IDSetNumber(&ActualisationNP2, "Error ! Delay invalid");
                }
            }
        }
    }
}


/**************************************************************************************
** L'utilisateur clique sur l'un des boutons de la boîte Indi
***************************************************************************************/

void BAO::ISNewSwitch (const char *dev, const char *name, ISState *states, char *names[], int n)
{
    // ignore if not ours //
    if (strcmp (mydev, dev))
        return;

    // ===================================
    // Connect Switch
    // ===================================
    if (!strcmp (name, ConnectSP.name))
    {
        if (IUUpdateSwitch(&ConnectSP, states, names, n) < 0)
            return;

        connect_telescope();

        return;
    }

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! BAORadio is offline. Please connect before issuing any commands.");
        reset_all_properties();
        return;
    }

    // ===================================
    // Coordinate Set
    // ===================================
    if (!strcmp(name, OnCoordSetSP.name))
    {
        if (IUUpdateSwitch(&OnCoordSetSP, states, names, n) < 0)
            return;

        currentSet = get_switch_index(&OnCoordSetSP);
        OnCoordSetSP.s = IPS_OK;
        IDSetSwitch(&OnCoordSetSP, NULL);
    }

    // ===================================
    // Abort slew
    // ===================================
    if (!strcmp (name, AbortSlewSP.name))
    {
        Abort=true;

        IUResetSwitch(&AbortSlewSP);

        if (EquatorialCoordsWNP.s == IPS_OK)
        {
            AbortSlewSP.s = IPS_OK;
            EquatorialCoordsWNP.s = IPS_IDLE;
            ObjectTP.s = IPS_IDLE;

            IDSetSwitch(&ConnectSP, "Envoi de la commande Abort\n");
            IDSetNumber(&EquatorialCoordsWNP, NULL);
            IDSetText(&ObjectTP, NULL);
        }

        return;
    }


    // ===================================
    // Park
    // ===================================
    if (!strcmp (name, ParkSP.name))
    {
        Park=true;

        IUResetSwitch(&ParkSP);

        if (EquatorialCoordsWNP.s == IPS_OK)
        {
            AbortSlewSP.s = IPS_OK;
            EquatorialCoordsWNP.s = IPS_IDLE;
            ObjectTP.s = IPS_IDLE;

            IDSetSwitch(&ConnectSP, "Envoi de la commande Park\n");
            IDSetNumber(&EquatorialCoordsWNP, NULL);
            IDSetText(&ObjectTP, NULL);
        }

        return;
    }
}



/**************************************************************************************
** Gestion du thread
** permet de suivre la connexion/déconnexion des antennes toutes les secondes
**
** l'utilisation d'un thread permet de contourner le problème de la fonction accept
** qui est bloquante.
***************************************************************************************/

void *BAO::pThreadSocket ()
{
    do
    {
        try
        {
            server.accept( Sockets[SocketsNumber].new_sock );

            Sockets[SocketsNumber].IP = server.recupip(Sockets[SocketsNumber].new_sock);

            Sockets[SocketsNumber++].Connected = true;

            InitThreadOK = true;
        }
        catch ( SocketException& e )
        {
            /*IDLog("Indi_BAO, pThreadSocket exception : ");
            IDLog(e.description().c_str());
            IDLog("\n");*/
        }
        
        sleep(1); // faire une pause pour éviter de consommer trop de temps CPU -> à vérifier 
    }
    while (!Exit);

    pthread_exit (0);
}



/**************************************************************************************
** Astuce pour lancer le thread depuis la classe BAO
**
***************************************************************************************/

void* LancementThread(BAO * appli)
{
    appli->pThreadSocket();

    return 0;
}


/**************************************************************************************
** Extraction de la position de l'antenne après l'envoi de la commande P
** Le retour de la commande P est POSITION/valeur_az/valeur_alt/
** ExtractPosition retourne donc Valeur_az et Valeur_alt
***************************************************************************************/

bool BAO::ExtractPosition(string str, Position *result)
{
    string str2;

    int pos = str.find("/");

    if (pos != string::npos)
    {
        str2 = str.substr(pos + 1);

        pos = str2.find("/");

        if (pos != string::npos)
        {
            result->x = atol(str2.substr(0, pos).c_str());

            result->y = atol(str2.substr(pos + 1).c_str());
	    
	    return true;
        }        
    } 

    IDLog((str +" failed !\n").c_str());

    return false;
}



/************************************************************************************
* cette procédure convertit les coordonnées équatoriales de l'objet visé
* en unités de codeurs des paraboles (nb de tours des deux axes moteurs depuis la position PARK)
************************************************************************************/

void BAO::ADDEC2Motor(double newRA, double newDEC)
{
    double targetAz;
    double targetAlt;
    char AzStr[32];
    char AltStr[32];


    // Calcule la hauteur et l'azimut de la zone du ciel pointée (en fonction de la date et du lieu d'observation)

    Azimut( newRA * 15.0 * Pidiv180, newDEC * Pidiv180, &targetAz, &targetAlt);
    
    // Correction de la réfraction atmosphérique
    
    targetAlt = RefractionAtmospherique(targetAlt);

    // Petit calcul pour faire en sorte que le sud soit à 0° d'azimut

    targetAz = VerifAngle( targetAz + Pi );

    // On convertit les angles précédents en degrés

    targetAlt *= N180divPi;
    targetAz  *= N180divPi;

    // Affichage dans les logs

    fs_sexa(AzStr, targetAz, 2, 3600);
    fs_sexa(AltStr, targetAlt, 2, 3600);

    IDLog("Horizontal coords : Az = %s     Alt = %s\n", AzStr, AltStr);


    //Conversion des deux angles en pas codeurs

    if ( targetAlt < 30.0 )
    {
        // L'objet est trop bas (<30°). On annule le suivi...

        IDSetSwitch(&OnCoordSetSP, "Erreur ! L objet suivi est situe a moins de 30° au-dessus de l horizon. Goto annule.");

        Suivi = false;

        ActualisationPosition = false;

        InitAntennes();
    }
    else
    {
        // Si la hauteur est supérieure à 90°, on doit ajouter 180° à l'azimut et corriger
        // la hauteur en appliquant hauteur=180°-hauteur

        if (targetAlt > 90.0)
        {
            targetAlt = 180.0 - targetAlt;
            targetAz += 180.0;
        }

        // On applique la formule de Marc pour convertir la hauteur en nombre de pas codeur alt

        double I0 = 81.0;
        double I2 = 128279.4 - 129723.4 * sin( (targetAlt - 20.2345) * Pidiv180 );

        double Codalt = (sqrt(I2) - I0) / 0.078947;


        TargetPosition.y = (int) Arrondi(Codalt);

        //4000 pas pour 360° sur l'axe az

        TargetPosition.x = (int) Arrondi( targetAz * 4000.0 / 360.0);

        IDLog("Nbre de pas codeurs Az = %i        Alt = %i\n", TargetPosition.x, TargetPosition.y);
    }
}


/**************************************************************************************
** Le fichier file existe-t-il et n'est-il pas vide ?
**************************************************************************************/

bool BAO::is_readable( const std::string & file ) 
{ 
    std::ifstream fichier( file.c_str() ); 
    if (fichier.fail()) return false;
    
    // sauvegarder la position courante
    long pos = fichier.tellg();
    // se placer en fin de fichier
    fichier.seekg( 0 , std::ios_base::end );
    // récupérer la nouvelle position = la taille du fichier
    long size = fichier.tellg() ;
    // restaurer la position initiale du fichier
    fichier.seekg( pos,  std::ios_base::beg ) ;
    return size ;
}


/**************************************************************************************
** Chargement des paramètres d'alignement des antennes
**************************************************************************************/

bool BAO::ChargementParametresAlignement(string fileName)
{
    string value;
    
    stringstream os;
    
    char *delta_az = NULL;
    
    char *delta_ha = NULL;
        
    if (!is_readable(fileName)) return false;
     
    //IDLog("Chargement du fichier d alignement des antennes %s\n", fileName.c_str());
     
    //Chargement des corrections des antennes
    
    for (int i=0; i<SocketsNumber; i++)
    {
        if (Sockets[i].Connected)
        {
            os << "Alignement antenne ip x.x.x." << Sockets[i].IP.substr(Sockets[i].IP.rfind(".")+1);

            value="delta_az";

            if (readINI((char*)os.str().c_str(), (char*)value.c_str(), &delta_az,  (char*)fileName.c_str()))

            value="delta_ha";

            if (readINI((char*)os.str().c_str(), (char*)value.c_str(), &delta_ha,  (char*)fileName.c_str()))

            if (delta_az && delta_ha)
            {
	       // Si les corrections de l'antenne i ont été modifiées depuis le démarrage du programme
	       // -> on les applique...
               if (Sockets[i].Delta.x != atol(delta_az) || Sockets[i].Delta.y != atol(delta_ha))
               {
                  Sockets[i].Delta.x = atol(delta_az);

                  Sockets[i].Delta.y = atol(delta_ha);

                  IDLog("Correction antenne ip %s  deltaAz=%i   deltaHa=%i\n", Sockets[i].IP.c_str(), Sockets[i].Delta.x, Sockets[i].Delta.y);
               }
            }

            if (delta_az) {
                delete [] delta_az;
                delta_az=NULL;
            }

            if (delta_ha) {
                delete [] delta_ha;
                delta_ha=NULL;
            }

            os.str("");
        }
    }
    
    return true;
}


/************************************************************************************
* Retourne le nombre d'antennes connectées
* 
************************************************************************************/

int BAO::AntennesConnectees()
{
    int num = 0;

    for (int i=1; i<SocketsNumber; i++) if (Sockets[i].Connected) num++;

    return num;
}


/**************************************************************************************
** En cas de problème
** Déconnecter l'antenne utilisant le socket num
***************************************************************************************/

void BAO::DeconnecterSocket(int num)
{
    IDLog("Deconnexion de l antenne : %s\n", Sockets[num].IP.c_str());
    Sockets[num].new_sock.shutdown();
    Sockets[num].Connected = false;
    Sockets[num].IP = "";
}


/**************************************************************************************
** Procédure principale
** Elle est appelée toutes les ms
***************************************************************************************/

void BAO::ISPoll()
{
    static int memSocketsNumber = -1;      // Combien y avait-il d'antennes connectées lors de l'appel précédent d'ISPoll ?
    static unsigned int compt   =  0;      // Compteur de la fonction ISPoll

    struct tm date;
    time_t t;
    struct timeval tv;
    struct timezone tz;

    //si pas de connexion avec le seveur d'indi -> on sort

    if (!is_connected()) return;


    // toutes les 100 millisec, on actualise le jour julien
    // le temps sidéral local etc...

    if ( compt%100 == 0 )
    {
        //Récupération de la date et de l'heure

        time(&t);
        date=*gmtime(&t);
        gettimeofday(&tv, &tz);

        double Annee=(double)(date.tm_year+1900);
        double Mois=(double)(date.tm_mon+1);
        double Jour=(double)date.tm_mday;
        double Heu=(double)date.tm_hour;
        double Min=(double)date.tm_min;
        double Sec=(double)date.tm_sec+tv.tv_usec/1.0E6;
        double UTCP=0.0;//(double)date.tm_isdst;

        // On transmet la date et l'heure à la classe Astro

        DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);

        //Puis on calule le temps sidéral local, le JJ etc.

        CalculTSL();
    }
    
    // On chargement les paramètres de corrections des antennes toutes les secondes
    
    if ( compt%1000 == 0 ) ChargementParametresAlignement("/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg");


    // Il faut que le thread soit actif

    if (InitThreadOK)
    {

        // Nouvelle connexion sur le socket ?

        if (SocketsNumber > memSocketsNumber)
        {
            memSocketsNumber = SocketsNumber;

            IDSetSwitch(&ConnectSP, "Connexion de l antenne %s. (Antennes connectees : %i)",
                        Sockets[SocketsNumber-1].IP.c_str(), AntennesConnectees());
        }


        /////////////////////////////////////////////////
        // Début des échanges avec les microcontrôleurs

        // Analyse des réponses des microcontrôleurs

        for (int i=1; i<SocketsNumber; i++)
        {
            if (Sockets[i].Connected)
            {
                try
                {
                    string reponse, buffereponse;

                    // on récupère la réponse du microcontrôleur

                    Sockets[i].new_sock >> reponse;

                    // Dans le cas où plusieurs trames seraient arrivées entre deux appels de POLLMS
                    // les traiter successivement

                    // d'où l'intérêt de mettre un '\n' à la fin des trames
                    // pour différencier une trame de la précédente

                    int pos = reponse.find("\n");

                    // S'il y a une réponse

                    while ((pos != string::npos) && (reponse.length() > 1))
                    {
                        // on garde en stock la deuxième partie de la trame
                        // pour un traitement ultérieur
                        buffereponse = reponse.substr(pos + 1);

                        // Partie traitée
                        reponse = reponse.substr(0, pos);
			
			IDLog("Reponse recue de %s : %s\n", Sockets[i].IP.c_str(), reponse.c_str());

                        // On vérifie ici les acknowledges
                        if ((reponse.find("ACK") != string::npos) && (reponse.find("NACK") == string::npos))
                        {
                            if (reponse.find("POSITION")  != string::npos)
                            {
                                Sockets[i].ack_pos=true;
                            }
                            else if (reponse.find("GOTO") != string::npos)
                            {
                                Sockets[i].ack_goto=true;
                            }
                            else if (reponse.find("PARK") != string::npos)
                            {
                                Sockets[i].ack_park=true;
                            }
                            else if (reponse.find("ABORT")!= string::npos)
                            {
                                Sockets[i].ack_abort=true;
                            }
                        }
                        else
                        {
                            //réponse à la requête POSITION

                            if (reponse.find("POSITION") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    //problème concernant la commande P
                                    OnCoordSetSP.s = IPS_ALERT;
                                    IDSetSwitch(&OnCoordSetSP, "ALERTE antenne %s : position de l antenne inconnue !\n",
                                                Sockets[i].IP.c_str());
                                    Sockets[i].PosValides = false;
                                    // Si la position de l'antenne est inconnue, on déconnecte l'antenne
                                    Sockets[i].Connected  = false;
                                }
                                else if (Sockets[i].ack_pos)
                                {                              
				    if ( Sockets[i].PosValides = (ExtractPosition(reponse, &Sockets[i].Pos) == true) )
                              	    {
				      OnCoordSetSP.s = IPS_OK;
                                      IDSetSwitch(&OnCoordSetSP, "Antenne %s : POSITION OK  (x=%i, y=%i)\n",
                                                Sockets[i].IP.c_str(), Sockets[i].Pos.x, Sockets[i].Pos.y);
				    }
				    else
				    {
				      OnCoordSetSP.s = IPS_ALERT;
                                      IDSetSwitch(&OnCoordSetSP, "Antenne %s : La position n est pas valide !\n",
                                                Sockets[i].IP.c_str());
				    }
				      
                                }
                            }

                            //réponse à la requête PARK

                            if (reponse.find("PARK") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    ParkSP.s = IPS_ALERT;
                                    IDSetSwitch(&ParkSP, "ALERTE antenne %s : erreur PARK !\n", Sockets[i].IP.c_str());
                                }
                                else if (Sockets[i].ack_park && reponse.find("OK")!=string::npos)
                                {
                                    ParkSP.s = IPS_OK;
                                    IDSetSwitch(&ParkSP, "Antenne %s : PARK OK\n",  Sockets[i].IP.c_str());
                                }
                            }

                            //réponse à la requête ABORT

                            if (reponse.find("ABORT") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    AbortSlewSP.s = IPS_ALERT;
                                    IDSetSwitch(&AbortSlewSP, "ALERTE antenne %s : erreur ABORT !\n",  Sockets[i].IP.c_str());
                                }
                                else if (Sockets[i].ack_abort && reponse.find("OK")!=string::npos)
                                {
                                    AbortSlewSP.s = IPS_OK;
                                    IDSetSwitch(&AbortSlewSP, "Antenne %s : ABORT OK\n",  Sockets[i].IP.c_str());
                                }

                            }

                            //réponse à la requête GOTO

                            if (reponse.find("GOTO") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    OnCoordSetSP.s = IPS_ALERT;
                                    IDSetSwitch(&OnCoordSetSP, "ALERTE antenne %s : Erreur GOTO !\n",  Sockets[i].IP.c_str());
                                    DeconnecterSocket(i);
                                }
                                else if (Sockets[i].ack_goto)
                                {
                                    if (reponse.find("OK") != string::npos)
                                    {
                                        // On a ici la confirmation que l'antenne 'i' a bien réalisé
                                        // le goto

                                        // On prend note

                                        Sockets[i].GotoOk = true;

                                        // Message pour l'utilisateur
					
					OnCoordSetSP.s = IPS_OK;
                                        IDSetSwitch(&OnCoordSetSP, "Antenne %s : GOTO OK.\n",  Sockets[i].IP.c_str());

                                        // Fin du Goto pour toutes les antennes ?

                                        int num=0;

                                        for (int j=1; j<SocketsNumber; j++)
					{
					  if (Sockets[j].Connected)
                                            {
                                                if (Sockets[j].GotoOk) num++;
                                            }
					}

                                        if ((num == AntennesConnectees()) && (num>0))
                                        {
					    // C'est bon ! Tout marche bien...
                                            // On actualise l'AR et la dec dans la boîte de dialogue

                                            lastRA  = targetRA;
                                            lastDEC = targetDEC;

                                            // On a fini le mouvement. On attend le prochain goto...

                                            ActualisationPosition=false;

                                            // Réinitialisation des paramètres des antennes en vue d'un prochain goto

                                            InitAntennes();

                                            // On dessine les voyants de la boîte de dialogue en vert
                                            OnCoordSetSP.s = IPS_OK;
                                            EquatorialCoordsWNP.s = IPS_OK;
                                            IDSetNumber (&EquatorialCoordsWNP, NULL);

                                            // Confirmation dans la boîte de dialogue que toutes
                                            // les antennes sont OK	    
					    OnCoordSetSP.s = IPS_OK;
                                            IDSetSwitch(&OnCoordSetSP, "GOTO OK !");
					    
					    UpdateGoto=true;
                                        }                                        
                                    }
                                }
                            }
                        }

                        // On passe à la trame suivante si memreponse n'est pas vide

                        reponse=buffereponse;
                        pos=reponse.find("\n");
                    }
                }
                catch (SocketException& e) //Aïe
                {
                    DeconnecterSocket(i);

                    IDLog("Indi_BAO, SocketException IsPoll : ");
                    IDLog(e.description().c_str());
                    IDLog("\n");
                }
            }
        }


        ///////////////////////////////////////
        // L'utilisateur a demandé l'annulation du mouvement en cours

        if (Abort)
        {
            // On arrête le suivi d'un objet

            Suivi = false;

            // On arrête l'enchaînement des actions
            // pour réaliser un goto

            ActualisationPosition = false;

            // On envoie l'ordre ABORT à toutes les antennes

            for (int i=1; i<SocketsNumber; i++)
            {
                if (Sockets[i].Connected)
                {
                    if (!ABORT(i)) Sockets[i].sendalertes++;
                }
            }

            IDSetSwitch(&OnCoordSetSP, "ABORT OK !");

            // Réinititialisation des paramètres des antennes

            InitAntennes();

            //Pour permettre de refaire un abort

            Abort=false;
        }

        ///////////////////////////////////////
        // L'utilisateur a demandé de mettre les antennes au repos

        if (Park)
        {
            // On arrête le suivi d'un objet

            Suivi = false;

            // On arrête l'enchaînement des actions
            // pour réaliser un goto

            ActualisationPosition = false;

            // On envoie l'ordre PARK à toutes les antennes

            for (int i=1; i<SocketsNumber; i++)
            {
                if (Sockets[i].Connected)
                {
                    if (!PARK(i)) Sockets[i].sendalertes++;
                }
            }

            IDSetSwitch(&OnCoordSetSP, "PARK OK !");

            // Réinititialisation des paramètres des antennes

            InitAntennes();

            //Pour permettre de refaire un park

            Park=false;
        }


        ///////////////////////////////////////
        // Gestion du suivi

        if ((Suivi) && (UpdateGoto))
        {
            // Délais entre deux actualisations

            double delai=ActualisationTM1 / 3600.0 / 24.0;   // Actualisation toutes les 15 minutes en mode transit

            if (TrackingMode==2) delai=ActualisationTM2 / 3600.0 / 24.0;   //et 5 secs en mode tracking


            // On actualise la position si le delai est dépassé

            if (GetJJ() - JJAnc > delai)
            {
                UpdateGoto = false;

                // Conversion des coordonnées en pas moteur

                ADDEC2Motor(targetRA, targetDEC);

                // On réinitialise les antennes en vue du goto

                InitAntennes();

                ActualisationPosition = true;

                // On sauvegarde la date

                JJAnc = GetJJ();
            }

            //Plus d'antenne ! On arrête le suivi

            if (AntennesConnectees() == 0)
            {
                if ( compt % 1000 == 0)
                {
                    IDSetSwitch(&OnCoordSetSP, "Erreur ! Plus d antennes connectees !");

                    if (Suivi) IDLog("Arrêt du suivi !");

                    ActualisationPosition=false;

                    Suivi=false;

                    InitAntennes();
                }
            }
        }



        // Exécution de la procédure complète de lecture de la position de l'antenne
        // puis envoi d'une commande Goto

        if (ActualisationPosition)
        {
            for (int i=1; i<SocketsNumber; i++)
            {
                if (Sockets[i].Connected)
                {
                    switch (Sockets[i].etape)
                    {

                        //Envoi de la commande POS

                    case 0 :
                    {
                        Sockets[i].ack_pos    = false;
                        Sockets[i].PosValides = false;

                        if (!POSITION(i)) Sockets[i].sendalertes++;

                        Sockets[i].etape++;
                    }
                    break;


                    // A-ton bien reçu l'ack POS ?

                    case 1 :
                    {
                        if (Sockets[i].ack_pos)
                        {
                            // tout marche bien
                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;
                            Sockets[i].etape++;
                            i--; //permet de revenir sur ce même socket malgré la boucle
                            // -> plus rapide
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe

                            // On garde une trace de l'anomalie
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // on recommence depuis le début
                                Sockets[i].etape = 0;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

                            // Toujours rien -> plus sérieux : il faut déconnecter l'antenne
                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
			        OnCoordSetSP.s = IPS_ALERT;
                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : pas d acknowledge recu apres l ordre POSITION. \
    Deconnexion de l antenne.", Sockets[i].IP.c_str());
                                DeconnecterSocket(i);
                            }
                        }
                    }
                    break;


                    //Les valeurs retournées pas la commande POSITION sont-elles valides ?

                    case 2 :
                    {
                        if (Sockets[i].PosValides)
                        {
                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;
                            Sockets[i].etape++; // on passe à l'étape suivante
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // on attend encore la réponse posvalides                                
                                Sockets[i].etape = 2;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
			        OnCoordSetSP.s = IPS_ALERT;
                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : la position retournee n est pas valide. \
    Deconnexion de l antenne.", Sockets[i].IP.c_str());
                                DeconnecterSocket(i);
                            }
                        }
                    }
                    break;


                    // A-ton reçu l'acknowledge de la commande goto ?

                    case 4 :
                    {
                        if (Sockets[i].ack_goto)
                        {
                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;
                            Sockets[i].etape++; // on passe à l'étape suivante
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // On prolonge l'attente                                
                                Sockets[i].etape = 4;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
			        OnCoordSetSP.s = IPS_ALERT;
                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : pas d acknowledge recu apres l ordre GOTO. \
    Deconnexion de l antenne.", Sockets[i].IP.c_str());
                                DeconnecterSocket(i);
                            }
                        }
                    }
                    break;


                    //  Confirmation goto ok ?

                    case 5 :
                    {
                        if (Sockets[i].GotoOk)
                        {
                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;
                            Sockets[i].etape++;
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // On prolonge l'attente
                                
                                Sockets[i].etape = 5;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

			    // On déconnecte l'antenne s'il n'y a pas de coonfirmation du goto au bout de 2 minutes    
                            if (Sockets[i].AnomaliesExecution > MAXANOMALIESGOTO)
                            {
			        OnCoordSetSP.s = IPS_ALERT;
                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : l antenne n a pas renvoye GOTO/OK. \
    Deconnexion de l antenne.", Sockets[i].IP.c_str());
                                DeconnecterSocket(i);
                            }
                        }
                    }
                    break;
                    }
                }
            }
        }


        ///////////////////////////////////////
        //On attend que toutes les antennes soient prêtes pour lancer l'ordre Goto -> meilleure synchronisation

        int num=0;

        for (int i=1; i<SocketsNumber; i++)
        {
            if (Sockets[i].Connected)
            {
                if (Sockets[i].etape == 3) num++; //fin de la procédure ActualisationPosition.
                // num antennes sont prêtes à recevoir l'ordre GOTO
            }
        }

        if ((num == AntennesConnectees()) && (num>0))
        {
            for (int i=1; i<SocketsNumber; i++ )
            {
                if (Sockets[i].Connected)
                {
                    Sockets[i].ack_goto = false;
                    Sockets[i].AttenteExecution = 0;
                    Sockets[i].AnomaliesExecution = 0;

                    if (!GOTO(i, TargetPosition.x - Sockets[i].Pos.x - Sockets[i].Delta.x, TargetPosition.y - Sockets[i].Pos.y - Sockets[i].Delta.y)) Sockets[i].sendalertes++;

                    Sockets[i].etape++;
                }
            }
        }

        ///////////////////////////////////////
        // Détection d'anomalies sur la socket. Déconnexion du micro-cont ?

        for (int i=1; i<SocketsNumber; i++)
        {
            if (Sockets[i].Connected)
            {
                if (Sockets[i].sendalertes > 0)
                {
		    OnCoordSetSP.s = IPS_ALERT;
                    IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : deconnexion de l antenne.", Sockets[i].IP.c_str());

                    DeconnecterSocket(i);
                }
            }
        }
    }
    
    //incrémentation du compteur
    compt++;
}



/**************************************************************************************
** Mode transit ou tracking
***************************************************************************************/

bool BAO::process_coords()
{
    switch (currentSet)
    {
        // Transit
    case BAO_TRANSIT:

        //Eteindre les voyants dans la boîte de dialogue Indi
        EquatorialCoordsWNP.s = IPS_BUSY;
        AbortSlewSP.s = IPS_IDLE;
        ParkSP.s = IPS_IDLE;

        IDSetNumber (&EquatorialCoordsWNP, NULL);
        IDSetSwitch (&AbortSlewSP, NULL);
        IDSetSwitch (&ParkSP, NULL);

        // On prépare les antennes pour le prochain goto

        InitAntennes();

        JJAnc = GetJJ();
	
	ADDEC2Motor(targetRA, targetDEC);

        TrackingMode = 1;

        Suivi = true;

        UpdateGoto = false;

        ActualisationPosition = true;

        break;

        // Tracking
    case BAO_TRACKING:

        //Eteindre les voyants dans la boîte de dialogue Indi
        EquatorialCoordsWNP.s = IPS_BUSY;
        AbortSlewSP.s = IPS_IDLE;
        ParkSP.s = IPS_IDLE;

        IDSetNumber (&EquatorialCoordsWNP, NULL);
        IDSetSwitch (&AbortSlewSP, NULL);
        IDSetSwitch (&ParkSP, NULL);

        InitAntennes();

        JJAnc=GetJJ();
	
	ADDEC2Motor(targetRA, targetDEC);

        TrackingMode = 2;

        Suivi = true;

        UpdateGoto = false;

        ActualisationPosition = true;       

        break;
    }

    return true;
}



/**************************************************************************************
** Connexion / Déconnexion avec le télescope
***************************************************************************************/

void BAO::connect_telescope()
{
    switch (ConnectSP.sp[0].s)
    {
    case ISS_ON:

        // Etats des voyants

        ConnectS[0].s = ISS_ON;
        ConnectS[1].s = ISS_OFF;
        ConnectSP.s = IPS_OK;
        IDSetSwitch (&ConnectSP, "BAORadio is online. Retrieving basic data...");

        // Petit message

        IDLog("\nHello BAORadio !\n");

        // On lance le thread !

        Exit=false;

        if (pthread_create (&th1, NULL, (void*(*)(void*))LancementThread, this) < 0)
        {
            IDLog("pthread_create error for threadSocket\n");
        }

        break;

    case ISS_OFF:

        // On sort du thread

        Exit = true;
        sleep(1);
        pthread_join (th1, NULL);
        InitThreadOK = false;

        // Etat des voyants

        ConnectS[0].s = ISS_OFF;
        ConnectS[1].s = ISS_ON;
        ConnectSP.s = IPS_IDLE;
        IDSetSwitch (&ConnectSP, "BAORadio is offline.");
        IDLog("Telescope is offline.");

        // On déconnecte tous les sockets

        for (int i=0; i<MAXHOSTNAME; i++)
        {
            DeconnecterSocket(i);
        }

        // init

        InitAntennes();

        SocketsNumber = 1;

        break;
    }
}




/**************************************************************************************
**  Envoie une commande sur le socket numsocket
***************************************************************************************/

bool BAO::COMMANDE(int numsocket, char* Commande, char* Params)
{
    char chaine[MAXCARACTERES];

    try
    {
        sprintf(chaine, "%s%s\n", Commande, Params);
	
	Sockets[numsocket].new_sock << chaine;
	
	IDLog("Commande envoyee a %s: %s", Sockets[numsocket].IP.c_str(), chaine);
    }
    catch (SocketException& e)
    {
        DeconnecterSocket(numsocket);

        IDLog("Indi_BAO, COMMANDE exception : ");
        IDLog(e.description().c_str());
        IDLog("\n");

        return false;
    }

    return true;
}


/**************************************************************************************
** Commande POSITION
***************************************************************************************/

bool BAO::POSITION(int numsocket)
{
    return COMMANDE(numsocket, (char*)"P", (char*)"");
}

/**************************************************************************************
** Commande PARK
***************************************************************************************/

bool BAO::PARK(int numsocket)
{
    return COMMANDE(numsocket, (char*)"Z", (char*)"");
}

/**************************************************************************************
** Commande ABORT
***************************************************************************************/

bool BAO::ABORT(int numsocket)
{
    return COMMANDE(numsocket, (char*)"A", (char*)"");
}


/**************************************************************************************
** Commande GOTO
***************************************************************************************/

bool BAO::GOTO(int numsocket, int deltaAz, int deltaAlt)
{
    char Params[MAXCARACTERES];
    char sensAz;
    char sensAlt;

    sensAlt = 1;
    sensAz  = 1;

    if ( deltaAz < 0 )
    {
        deltaAz = -deltaAz;
        sensAz  = 0;
    }

    if ( deltaAlt < 0 )
    {
        deltaAlt = -deltaAlt;
        sensAlt  = 0;
    }

    // Vérification du nombre de pas à faire au niveau de l'axe azimut
    // Rappel : un tour complet autour de l'axe az fait 4000 pas codeur (#define NBREPASCODEURSAZ dans BAO.h)

    // problème 1 : si deltaAz > à un demi-tours - soit 2000 pas alors
    // on fait le trajet dans le sens inverse pour aller plus vite

    // problème 2 : Passage au méridien
    // on se situe à quelques secondes de degrés avant le sud par exemple (nb de pas 3995 par exemple)
    // et on franchit le méridien (nb de pas codeur = 5 par exemple)
    // On risque alors de faire un tour complet en sens inverse pour rattraper l'objet -> à éviter

    // ne pas faire des tours complets en Az pour rien...
    while (deltaAz > NBREPASCODEURSAZ) deltaAz -= NBREPASCODEURSAZ;

    // Doit résoudre tous les problèmes concernant l'azimut...
    if (deltaAz > NBREPASCODEURSAZ / 2 )
    {
        deltaAz = NBREPASCODEURSAZ - deltaAz;
        sensAz = 1 - sensAz;
    }

    //on envoie les coordonnées au driver

    if (sensAz == 1 )  sensAz='f'; else sensAz='b';
    
    if (sensAlt == 1 ) sensAlt='f'; else sensAlt='b';

    sprintf(Params, "%c%04i%c%04i", sensAz, deltaAz, sensAlt, deltaAlt);

    return COMMANDE(numsocket, (char*)"G", Params);
}



/**************************************************************************************
** Les fonctions qui suivent sont nécessaires pour construire le pilote Indi_BAO
** Elles sont appelées par le noyau d'Indi
***************************************************************************************/

/**************************************************************************************
** Initialisation du pilote BAO
***************************************************************************************/

void ISInit()
{
    //Il ne faut exécuter la fonction qu'une seule fois
    static int isInit = 0;

    if (isInit) return;

    if (telescope.get() == 0) telescope.reset(new BAO());

    isInit = 1;

    //initialisation du timer

    IEAddTimer (POLLMS, ISPoll, NULL);
}


void ISGetProperties (const char *dev)
{
    ISInit();

    telescope->ISGetProperties(dev);
}

void ISNewSwitch (const char *dev, const char *name, ISState *states, char *names[], int n)
{
    ISInit();
    telescope->ISNewSwitch(dev, name, states, names, n);
}

void ISNewText (const char *dev, const char *name, char *texts[], char *names[], int n)
{
    ISInit();
    telescope->ISNewText(dev, name, texts, names, n);
}

void ISNewNumber (const char *dev, const char *name, double values[], char *names[], int n)
{
    ISInit();
    telescope->ISNewNumber(dev, name, values, names, n);
}

void ISNewBLOB (const char *dev, const char *name, int sizes[], int blobsizes[], char *blobs[], char *formats[], char *names[], int n)
{
    INDI_UNUSED(dev);
    INDI_UNUSED(name);
    INDI_UNUSED(sizes);
    INDI_UNUSED(blobsizes);
    INDI_UNUSED(blobs);
    INDI_UNUSED(formats);
    INDI_UNUSED(names);
    INDI_UNUSED(n);
}

void ISSnoopDevice (XMLEle *root)
{
    INDI_UNUSED(root);
}

void ISPoll (void *p)
{
    INDI_UNUSED(p);

    telescope->ISPoll();

    IEAddTimer (POLLMS, ISPoll, NULL);
}


/**************************************************************************************
**
***************************************************************************************/

int BAO::get_switch_index(ISwitchVectorProperty *sp)
{
    for (int i=0; i < sp->nsp ; i++)
        if (sp->sp[i].s == ISS_ON)
            return i;

    return -1;
}


/**************************************************************************************
**
***************************************************************************************/
void BAO::get_initial_data()
{
    //  IDSetNumber (&EquatorialCoordsRNP, NULL);
}

/**************************************************************************************
**
***************************************************************************************/
void BAO::slew_error(int slewCode)
{
    OnCoordSetSP.s = IPS_IDLE;

    if (slewCode == 1)
        IDSetSwitch (&OnCoordSetSP, "Object below horizon.");
    else if (slewCode == 2)
        IDSetSwitch (&OnCoordSetSP, "Object below the minimum elevation limit.");
    else
        IDSetSwitch (&OnCoordSetSP, "Slew failed.");
}

/**************************************************************************************
**
***************************************************************************************/

bool BAO::is_connected()
{
    // return (ConnectSP.sp[0].s == ISS_ON);
    return (ConnectSP.s == IPS_OK);
}

/**************************************************************************************
**
***************************************************************************************/
void BAO::connection_lost()
{
    ConnectSP.s = IPS_IDLE;
    IDSetSwitch(&ConnectSP, "The connection to the telescope is lost.");
    IDLog("arret");
    return;
}

/**************************************************************************************
**
***************************************************************************************/
void BAO::connection_resumed()
{
    ConnectS[0].s = ISS_ON;
    ConnectS[1].s = ISS_OFF;
    ConnectSP.s = IPS_OK;

    IDSetSwitch(&ConnectSP, "The connection to the telescope has been resumed.");
}


/**************************************************************************************
** Gère les erreurs de communication avec la boîte Indi
***************************************************************************************/
/*
void BAO::handle_error(INumberVectorProperty *nvp, int err, const char *msg)
{
    nvp->s = IPS_ALERT;

    // If the error is a time out, then the device doesn't support this property
    if (err == -2)
    {
        nvp->s = IPS_ALERT;
        IDSetNumber(nvp, "Device timed out. Current device may be busy or does not support %s. Will retry again.", msg);
    }
    else
        // Changing property failed, user should retry.
        IDSetNumber( nvp , "%s failed.", msg);

    fault = true;
}*/


/**************************************************************************************
**
***************************************************************************************/
/*
void BAO::correct_fault()
{
    fault = false;
    IDMessage(mydev, "Telescope is online.");
}*/



