/*
The stellarium telescope library helps building
telescope server programs, that can communicate with stellarium
by means of the stellarium TCP telescope protocol.
It also contains smaple server classes (dummy, Meade LX200).

Author and Copyright of this file and of the stellarium telescope library:
Johannes Gajdosik, 2006

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "ServerBAO.hpp"
#include "Socket.hpp" // GetNow

#include <math.h>

ServerDummy::ServerDummy(int port)
            :Server(port)
{
	last_pos[0] = current_pos[0] = desired_pos[0] = 1.0;
	last_pos[1] = current_pos[1] = desired_pos[1] = 0.0;
	last_pos[2] = current_pos[2] = desired_pos[2] = 0.0;
	next_pos_time = -0x8000000000000000LL;


        float a1, a2, a3;

        // Le choix de la couleur des caractères est adapté pour une fenêtre de terminal
        // ayant un fond blanc. pour un fond noir, mettre couleurs=2 dans le fichier params

        ChoixCouleurs=1;


        Affiche("\n\n\n\n\n \
     ******************************************\n \
     *           Serveur Stellarium           *\n \
     * Laboratoire de l'Accélérateur Linéaire *\n \
     *                                        *\n \
     *             v0.1  20/08/2011           *\n \
     *        franckrichard033@gmail.com      *\n \
     ******************************************\n\n\n", true);

        //Initialisation des variables globales et des pointeurs


        delaitransit  = 0;
        delaitracking = 0;
        lognum        = 0;
        numAntenne    = 0;
        numobjets     = 0;
        runnum        = 1;

        Pression      = 1013.0;
        Temperature   = 10.0;

        NoExit        = true;
        Transit       = false;
        run           = false;
        exitrun       = false;

        fd            = NULL;
        client_socket = NULL;

        // chaines de caractères et tableaux

        LatitudeChar  = "";
        LongitudeChar = "";
        Serveur       = "";
        Port          = "";

        Antennes      = new DefAntenne[MAXANTENNES];
        objets        = new DefObjets[MAXOBJETS];

        //on initalise aussi les tableaux

        for (int i=0; i<MAXANTENNES; i++)
        {
            Antennes[i].ok=0;
            Antennes[i].ip=0;
        }

        for (int i=0; i<MAXOBJETS; i++)
        {
            objets[i].JJ=0;
            objets[i].exec=false;
        }

        // Chargement du fichier contenant les paramètres

        if (!ChargementParametres("params"))
        {
            Erreur("Le fichier de configuration 'params' contient des erreurs ou n'existe pas.\n \
            Merci de vérifier.\n\n");
            exit(1);
        }

        // Conversion de la latitude et de la longitude en radians

        Decomposition(LatitudeChar, 0, &a1, &a2, &a3);

        double Latitude = (a1 + a2 / 60.0 + a3 / 3600.0) * Pidiv180;

        Decomposition(LongitudeChar, 1, &a1, &a2, &a3);

        double Longitude = Pi2 - ( a1 + a2 / 60.0 + a3 / 3600.0 ) * Pidiv180;

        // Transfert de la latitude, longitude, pression, température à la classe Astro

        DefinirLongitudeLatitude(Longitude, Latitude);

        DefinirPressionTemp(Pression, Temperature);

        // Ouverture du socket avec indi_BAO

        client_socket = new ClientSocket("localhost", atoi(Port.c_str()) );

        // On se connecte au pilote indi_BAO

        Connect(true);

        // On lance la fenêtre graphique

        initialiserFenetre();

        XSelectInput(d,w, ExposureMask );
}

void ServerDummy::step(long long int timeout_micros)
{

  if (last_pos[0]!=current_pos[0] || last_pos[1]!=current_pos[1] || last_pos[2]!=current_pos[2])
  {
		current_pos[0] = desired_pos[0];
			current_pos[1] = desired_pos[1];
			current_pos[2] = desired_pos[2];
			
		const double ra = atan2(current_pos[1],current_pos[0]);
		const double dec = atan2(current_pos[2],
		                         sqrt(current_pos[0]*current_pos[0]+current_pos[1]*current_pos[1]));
		
			//On actualise le jour julien et le temps sidéral local
		
		Goto(DHMS(VerifAngle(ra)*180.0/M_PI, true), DHMS(dec*180.0/M_PI, false), Transit, true);
		
		last_pos[0] = current_pos[0];
			last_pos[1] = current_pos[1];
			last_pos[2] = current_pos[2];
  }

    Update();

    

    // on lit et affiche la réponse des microcontrolleurs

    LireReponse();

    Dessiner();
    
	
	

	
	Server::step(timeout_micros);
}

void ServerDummy::gotoReceived(unsigned int ra_int, int dec_int)
{
	const double ra = ra_int*(M_PI/(unsigned int)0x80000000);
	const double dec = dec_int*(M_PI/(unsigned int)0x80000000);
	const double cdec = cos(dec);
	desired_pos[0] = cos(ra)*cdec;
	desired_pos[1] = sin(ra)*cdec;
	desired_pos[2] = sin(dec);
}


/**************************************************************************************
** Affiche le message à l'écran (si AfficheEcran==true) et le sauvegarde dans le fichier
** BAOControl.log
***************************************************************************************/

void ServerDummy::Affiche(string Message, bool AfficheEcran)
{
    FILE *pFile = NULL;

    if ( (pFile = fopen ("BAOcontrol.log", "a")) != NULL)
    {
        if (AfficheEcran) cout << Message;

        fprintf (pFile, Message.c_str());

        fclose (pFile);
    }
}

void ServerDummy::Affiche(stringstream *Message, bool AfficheEcran)
{
    Affiche(Message->str(), AfficheEcran);

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Affiche les messages d'erreur en rouge
***************************************************************************************/

void ServerDummy::Erreur(string chaine)
{
    ChoixCouleurs==1 ? cout << red1  : cout << red2;
    Affiche(chaine, true);
    ChoixCouleurs==1 ? cout << grey1 : cout << grey2;
}

void ServerDummy::Erreur(stringstream *Message)
{
    Erreur(Message->str());

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Initialise la fenêtre graphique
***************************************************************************************/

void ServerDummy::initialiserFenetre()
{
    XGCValues gcv;

    // contacte le serveur graphique

    if ((d = XOpenDisplay(getenv("DISPLAY"))) == NULL)
    {
        Erreur("Impossible de contacter le serveur\n\n");
        exit(1);
    }

    // crée une fenêtre

    w = XCreateSimpleWindow(d, RootWindow(d, DefaultScreen(d)),
                            0, 0, larg_fenetre, haut_fenetre,
                            2, BlackPixel(d, DefaultScreen(d)),
                            WhitePixel(d, DefaultScreen(d)));

    // Création du double buffer

    db = XCreatePixmap(d, w, larg_fenetre, haut_fenetre, DefaultDepth(d, DefaultScreen(d)));

    // Chargement de la police de caractères

    if ((fd = XLoadQueryFont(d, "fixed")) == NULL)
    {
        Erreur("Impossible de charger la police fixed\n\n");
        exit(1);
    }

    // Création des couleurs

    gcv.font = fd->fid;
    gcv.foreground = BlackPixel(d, DefaultScreen(d));
    noir = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*255;
    vert = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 255*256*256;
    rouge = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*256*256+200*256+200;
    gris = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    // Affichage de fenêtre

    XMapWindow(d, w);
}


/**************************************************************************************
** Dessine les éléments dans la fenêtre graphique
***************************************************************************************/

void ServerDummy::Dessiner()
{
    char *chaine;
    GC color;

    chaine = new char [100];

    // On efface la fenêtre

    XSetForeground(d, noir, WhitePixel(d, DefaultScreen(d)));
    XFillRectangle(d, db, noir, 0, 0, larg_fenetre, haut_fenetre);
    XSetForeground(d, noir, BlackPixel(d, DefaultScreen(d)));

    // On récupère la date et l'heure contenue dans la classe astro (et le jour julien)

    double h  = GetHeure();
    double mi = GetMin();
    double s  = GetSec();
    double a  = GetAnnee();
    double m  = GetMois();
    double j  = GetJour();
    double JJ = GetJJ();

    // on évite les affichages de l'heure du genre 12:56:60 ou 24:00:00

    if (s==60) {
        s = 0;
        mi++;
        if (mi == 60) {
            mi = 0;
            h++;
            if (h == 24) {
                h = 0;
                j++;
            }
        }
    }

    //affichage de la date et de l'heure

    sprintf(chaine, "%02.0f:%02.0f:%02.1f TU    %02.0f/%02.0f/%04.0f    JJ=%10.5f",
            h, mi, s, j, m, a, JJ);

    XDrawString(d, db, noir, 10, 10, chaine, strlen(chaine));

    //En cas d'exécution d'un fichier de mouvements, on affiche la prochaine étape

    if (run)
    {
        sprintf(chaine, "Prochaine etape dans %4.1fs", (objets[runnum].JJ - JJ) * 3600.0 * 24.0);

        if ((objets[runnum].JJ - JJ) * 3600.0 * 24.0 >= 0.0)
            XDrawString(d, db, noir, 350, 10, chaine, strlen(chaine));
    }

    // affichage des cercles d'état
    // vert = l'antenne fonctionne
    // rouge = problème détecté sur l'antenne
    // gris = antenne non connectée

    for (int i=0; i <20; i++)
    {
        for (int j=0; j<2; j++)
        {
            switch (Antennes[20*j+i].ok)
            {
            case 1 :
                color=vert;
                break;
            case 2 :
                color=rouge;
                break;
            default:
                color=gris;
                break;
            }

            XFillArc(d, db ,color, 10+i*30, 20+j*30, 15, 15, 0, 360*64);
            XDrawArc(d, db ,noir,  10+i*30, 20+j*30, 15, 15, 0, 360*64);
        }
    }


    // Affichage des messages d'indi_BAO
    // gère aussi le défilement

    if (lognum>10)
    {
        for (int i=lognum-1; i>lognum-11; i--)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+(i-lognum+10)*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }
    else
    {
        for (int i=0; i<lognum; i++)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+i*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }

    // On affiche la copie de l'écran une fois que tout est dessiné
    // on évite ainsi les scientillements de la fenêtre

    XCopyArea(d, db, w, noir, 0, 0, larg_fenetre, haut_fenetre, 0, 0);

    // On force le rafraichissement de l'écran

    XFlush(d);

    delete [] chaine;
}


/**************************************************************************************
** Gestion des événements affectant la fenêtre graphique
***************************************************************************************/

void ServerDummy::rouler()
{
    XEvent e;

    if (XCheckWindowEvent(d, w, ExposureMask, &e))
    {
        switch (e.type)
        {
        default:
            //cout << "J'ai reçu un événement " <<  e.type << endl
            break;
        case Expose :
            Dessiner();
            break;
        }
    }
}


/**************************************************************************************
** Récupère les messages envoyés par indi_BAO
***************************************************************************************/

void ServerDummy::LireReponse()
{
    string reponse, memreponse, decomp;
    bool test=false;

    *client_socket >> reponse;

   // printf("%s\n", reponse.c_str());

    do
    {
        if (reponse.find("message")!=string::npos)
        {
            // On ne garde que la partie intéressante
            // La partie suivant "message="...
            reponse = reponse.substr(reponse.find("message") + 9);

            memreponse = reponse;

            // ...jusqu'au "
            reponse=reponse.substr(0, reponse.find("\""));

            // On récupère l'adresse ip de l'antenne qui nous parle
            // et on met à jour le tabmeau des états des antennes

            if (reponse.find(". (Antennes connectees")!=string::npos)
            {
                decomp=reponse.substr(0, reponse.find(". (Antennes connectees"));
                decomp=decomp.substr(decomp.rfind(".")+1);

                // si cette antenne est déjà connectée, on change son état

                for (int i=0; i<numAntenne; i++)
                {
                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok=1;
                        test=true;
                        break;
                    }
                }

                // ou si c'est une nouvelle antenne, on l'ajoute au tableau

                if (!test)
                {
                    Antennes[numAntenne].ip = atoi(decomp.c_str());
                    Antennes[numAntenne++].ok = 1;
                }
            }

            // erreur sur une antenne -> on change son état dans la fenêtre

            if  ((reponse.find("ALERTE antenne ")!=string::npos) ||
                    (reponse.find("Erreur sur l antenne ")!=string::npos))
            {
                decomp = reponse.substr(0, reponse.find(" :"));
                decomp = decomp.substr(decomp.rfind(".") + 1);

                for (int i=0; i<numAntenne; i++)
                {
                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok = 2;  // Erreur sur l'antenne i
                    }
                }
            }

            // On sauvegarde le message dans le tableau logs

            stringstream os;

            os << lognum << " : " << reponse;

            Affiche(os.str() + "\n", false);

            if (lognum<MAXLOG-1) logs[lognum++] = os.str();

            reponse = memreponse;
        }
        else reponse="";


    } while ( reponse !="" );

     // on actualise la fenêtre

        Dessiner();

    // on envoie un message pour actualiser la fenêtre graphique

    rouler();
}


/**************************************************************************************
** Est-ce que la réponse du pilote indi_BAO est conforme à ce que l'on attendait ?
***************************************************************************************/

bool ServerDummy::VerifReponse(string reponseattendue)
{
    string reponse = "";
    string memreponse = "";
    int    duree = 0;
    bool   test = false;

    // TODO :La réponse du driver ne vient pas tout de suite et on peut recevoir
    // des messages intermédiaires un peu étranges impliquant la fonction CONNECT, pourquoi ?
    // Là on lit les messages trois fois de suite pour essayer de voir passer le bon
    // Essayer de voir si on ne peut pas faire mieux...

    for (int i=0; i<3 ;i++)
    {
        do
        {
            usleep(1000); // attendre 1 ms pour laisser le temps au pilote indi_BAO de répondre

            *client_socket >> reponse;

            duree++;
        }
        while (reponse.length() == 0 && duree<10 ); // on attend un message pendant 10 ms max

        // on ajoute tous les messages reçus dans memmreponse
        // ->intéressant pour le debug

        memreponse += reponse;

        if (reponse.find(reponseattendue) != string::npos)
        {
            test = true;
            break;
        }
    }

    if (!test)
    {
        // réponse inconnue -> une erreur ?

        int pos = memreponse.find("message=");

        if ( pos != (int)string::npos )
        {
            memreponse = memreponse.substr(pos + 9);

            memreponse = memreponse.substr(0, memreponse.find("\""));

            Erreur("Réponse du pilote indi_BAO :" + memreponse + "\n\n");
        }

        return false;
    }

    return true;
}


/**************************************************************************************
** Calcul du jour julien et du temps sidéral local
***************************************************************************************/

void ServerDummy::Update()
{
    struct tm date;
    time_t t;
    struct timeval tv;
    struct timezone tz;

    // On récupère la date et l'heure depuis l'horloge du système

    time(&t);
    date=*gmtime(&t);
    gettimeofday(&tv, &tz);

    double Annee = (double)date.tm_year + 1900.0;
    double Mois  = (double)date.tm_mon + 1.0;
    double Jour  = (double)date.tm_mday;
    double Heu   = (double)date.tm_hour;
    double Min   = (double)date.tm_min;
    double Sec   = (double)date.tm_sec + tv.tv_usec / 1.0E6;
    //double UTCP  = 0.0;//(double)date.tm_isdst;

    /*Annee=2011;
    Mois=6;
    Jour=17;
    Heu=15;
    Min=53.0;
    Sec=00.0;*/

    // On actualise la date et l'heure dans la classe Astro

    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);


    // On lance le calcul

    CalculTSL();
}





/**************************************************************************************
** Envoie la longitude et la latitude du lieu d'observation au pilote indi_BAO
***************************************************************************************/

bool ServerDummy::EnvoyerCoordGeographiques()
{
    // la communication avec le pilote indi_BAO
    // se fait par l'envoie de petit fichier xml
    // d'où le côté complexe des commandes

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"GEOGRAPHIC_COORD\">";
        *client_socket << "<oneNumber name=\"LAT\">";
        *client_socket << LatitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"LONG\">";
        *client_socket << LongitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"GEOGRAPHIC_COORD\" state=\"Ok\""))
        {
            Erreur("ERREUR fct EnvoyerCoordGeographiques : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Les coordonnées géographiques ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}



/**************************************************************************************
** Place les antennes en position de repos
***************************************************************************************/

bool ServerDummy::Park()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"\" >";
        *client_socket << "<oneSwitch name=\"PARK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("PARK OK"))
        {
            Erreur("La commande PARK a échoué.\n\n");;

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Le pilote indi_BAO a bien reçu la commande PARK.\n\n", true);

    return true;
}



/**************************************************************************************
** Annule le mouvement en cours
***************************************************************************************/

bool ServerDummy::Abort()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ABORT_MOTION\" >";
        *client_socket << "<oneSwitch name=\"ABORT\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("ABORT OK"))
        {
            Erreur("L'annulation a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Le pilote indi_BAO a bien reçu la commande ABORT.\n\n", true);

    return true;
}



/**************************************************************************************
** Diriger l'antenne vers les coordonnées ar et dec et suivre l'objet
** en activant le mode transit ou tracking.
** si J2000 == true, cela signifie que les coordonnées ar et dec sont données dans le
** le système de coordonnées J2000 (écliptique et équinoxe du 1 janvier 2000 à 0 h TU)
** Des calculs supplémentaires (précession, nutation, abérration) sont alors nécessaires
***************************************************************************************/

bool ServerDummy::Goto(string ar, string dec, bool Transit, bool J2000)
{
    float  ar1, ar2, ar3;
    float  dec1, dec2, dec3;
    double arf, decf;



     /*
     //Pour les tests
     ar="05:39:14.8";
     dec="23:19:24.1";
     printf("JJ=%10.10f\n", GetJJ());
     J2000=true;
     */

    // Conversion de l'AD et de la déclinaison en radians

    Decomposition(ar, 2, &ar1, &ar2, &ar3);
    Decomposition(dec, 0, &dec1, &dec2, &dec3);

    arf  = ( ar1  + ar2 / 60.0  + ar3 / 3600.0 ) * 15.0 * Pidiv180;
    decf = ( fabs(dec1) + dec2 / 60.0 + dec3 / 3600.0 ) * Pidiv180;

    if (dec[0]=='-') decf = -decf;

    // calculs pupplémentaires pour se ramener
    // à l'époque de l'observation

    if (J2000)
    {
        Precession(&arf, &decf);
        NutationEtoile(&arf, &decf);
        AberrationAnnuelle(&arf, &decf);
    }

    // on réécrit l'ar et la dec dans le format d'indi_BAO

    ar  = DHMS(arf * N180divPi, true);

    dec = DHMS(decf * N180divPi, false);

    // on en informe l'utilisateur

    Affiche("Coordonnées apparentes de l'objet :\n", true);

    Affiche("AD=" + ar  + "   Dec=" + dec +"\n", true);


    /*
    //Tests
    Azimut(arf, decf, &azi, &hau);

    hau=RefractionAtmospherique(hau);

    Affiche("Coordonnées rectangulaires liées au lieu d'observation :\n", true);

    printf("Azimut = %s   Hauteur = %s\n", DHMS(azi*N180divPi,false).c_str(), DHMS(hau*N180divPi, false).c_str());
    */

    // On transmet les coordonnées au pilote BAO

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ON_COORD_SET\">";
        *client_socket << "<oneSwitch name=\"TRANSIT\">";
        if (Transit) *client_socket << "On";
        else *client_socket << "Off";
        *client_socket << "</oneSwitch>";

        *client_socket << "<oneSwitch name=\"TRACKING\">";
        if (Transit)  *client_socket << "Off";
        else *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ON_COORD_SET\""))
        {
            Erreur("Le changement de mode TRANSIT/TRACKING a échoué.\n\n");

            return false;
        }

        *client_socket << "<newNumberVector device=\"BAO\" name=\"EQUATORIAL_EOD_COORD_REQUEST\">";
        *client_socket << "<oneNumber name=\"RA\">";
        *client_socket << ar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"DEC\">";
        *client_socket << dec;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"EQUATORIAL_EOD_COORD_REQUEST\""))
        {
            Erreur("Le transfert des coordonnées de l'objet a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    Affiche("Les nouvelles coordonnées AD=" + ar + " Dec=" + dec, true);
    Affiche(" ont été envoyées au pilote indi_BAO.\n\n", true);
    return true;
}



/**************************************************************************************
** Se connecte ou se déconnecte au pilote indi_BAO
***************************************************************************************/

bool ServerDummy::Connect(bool connect)
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"CONNECTION\">";

        if (connect) *client_socket << "<oneSwitch name=\"CONNECT\">";
        else *client_socket << "<oneSwitch name=\"DISCONNECT\">";

        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (connect)
        {
            if (!VerifReponse("BAORadio is online"))
            {
                Erreur("La connexion a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (!VerifReponse("BAORadio is offline"))
            {
                Erreur("La déconnexion a échoué.\n\n");

                return false;
            }
        }
    }

    catch ( SocketException& e )
    {
        Erreur("Exception was caught:" + e.description() + "\n");
        return false;
    }

    if (connect)
    {
        Affiche("La connexion a été établie avec le pilote indi_BAO.\n\n", true);

        EnvoyerCoordGeographiques();
    }
    else
    {
        Affiche("BAOcontrol s'est bien déconnecté du pilote indi_BAO.\n\n", true);
    }

    return true;
}






/**************************************************************************************
** Décomposition et vérification des dates, heures, AD et déclinaisons
**
** type = 0 -> Déclinaison/latitude
** type = 1 -> longitude
** type = 2 -> AD ou heure
** type = 3 -> date
**
** Exemple de Decomposition("15:23:12", 2, a, b, c) retourne a=15, b=23, c=12
** si la chaine a un format incorrect, la fct retourne false
***************************************************************************************/

bool ServerDummy::Decomposition(string chaine, char type, float *a1, float *a2, float *a3)
{
    string car, s;
    float a, b, c;

    // pour les heures et les coordonnées, on attend ":" comme caractère séparateur, sinon
    // on attend d'avoir des "/" pour une date

    (type==3) ? car="/" : car=":";

    // Y a-t-il 2 caractères ':' ou '/' dans la chaine ?
    // C'est indispensable dans tous les cas

    int test=0;
    for (int i=0; i<(int)chaine.length(); i++) if (chaine[i] == car[0]) test++;
    if (test<2) return false;

    // Extraction des trois nombres

    s = chaine.substr(0, chaine.find(car));

    a = atoi(s.c_str());

    s = chaine.substr(chaine.find(car)+1, chaine.rfind(car) - chaine.find(car) - 1);

    b = atoi(s.c_str());

    s = chaine.substr(chaine.rfind(car)+1);

    c = Arrondi(atof(s.c_str())*100.0)/100.0;

    //vérification de la cohérence des infos contenues dans la chaine

    if (type < 3 )
    {
        // pour une déclinaison
        if ((type==0) && (a>90.0  || a<-90.0)) return false;
        // pour une AD
        if ((type==1) && (a>360.0 || a<0.0  )) return false;
        // pour une heure
        if ((type==2) && (a>23.0  || a<0.0  )) return false;
        // pour les minutes
        if (b<0.0 || b>59.0 ) return false;
        //pour les secondes
        if (c<0.0 || c>=60.0) return false;
    }
    else
    {
        //pour les jours
        if (a<0.0 || a>31.0) return false;
        //pour les mois
        if (b<0.0 || b>12.0) return false;
    }

    if (a1!=NULL) *a1 = a;
    if (a2!=NULL) *a2 = b;
    if (a3!=NULL) *a3 = c;

    return true;
}






/**************************************************************************************
** Lecteur d'un fichier ligne après ligne
** retourne -1 à la fin
***************************************************************************************/

int ServerDummy::readline (FILE * pfile, char *tab)
{
    int nbchar = 0;
    char c;

    while ((c = getc (pfile)) != '\n')
    {
        if (c == EOF)
        {
            break;
        }
        tab[nbchar++] = c;
    }
    tab[nbchar] = '\0';

    while (nbchar>0 && tab[nbchar-1]==' ')
    {
        tab[--nbchar] = '\0';
    }

    while (tab[0]==' ') for (int i=1; i<=nbchar; i++) tab[i-1] = tab[i];

    if (c == EOF) nbchar=-1;

    return (nbchar);
}



/**************************************************************************************
** chargement des paramètres contenus dans le fichier params
***************************************************************************************/

bool ServerDummy::ChargementParametres(string fileName)
{
    string section;
    string key;
    char * value;
    stringstream os;



    Affiche("Lecture du fichier de configuration 'params' :\n\n", true);


    /////////////////////////////////////////
    // Rubrique Coodonnées géographiques

    section = "coordonnees geographiques";

    key = "latitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 0, NULL, NULL, NULL)))
    {
        LatitudeChar = (string)value;
        delete [] value;
        Affiche("latitude = " + LatitudeChar +"\n", true);
    }
    else
    {
        Erreur("La latitude est incorrecte !\n");
        return false;
    }


    key = "longitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 1, NULL, NULL, NULL)))
    {
        LongitudeChar = (string)value;
        delete [] value;
        Affiche("longitude = " + LongitudeChar +"\n\n", true);
    }
    else
    {
        Erreur("La longitude est incorrecte !\n");
        return false;
    }


    /////////////////////////////////////////
    // rubrique connexion

    section = "connexion indi";

    key = "serveur";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Serveur = (string)value;
        delete [] value;
        Affiche("serveur = " + Serveur +"\n", true);
    }
    else
    {
        Erreur("Nom du serveur invalide !\n");
        return false;
    }


    key = "port";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Port = (string)value;
        delete [] value;
        Affiche("port = " + Port +"\n\n", true);
    }
    else
    {
        Erreur("Numéro de port incorrect !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique paramètres de l'atmosphère

    section = "atmosphere";

    key = "pression";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Pression = atof(value);
        delete [] value;
        os << "pression = " << Pression << endl;
        Affiche(&os, true);
    }
    else
    {
        os << "La pression atmosph&rique est incorrecte !" << endl;
        Erreur(&os);
        return false;
    }

    key = "temperature";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Temperature=atof(value);
        delete [] value;
        os << "température = " << Temperature << endl << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("La température est incorrecte !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique suivi

    section = "suivi";

    key = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = (strstr(value, "transit") != NULL);
        Affiche("mode suivi = " + (string)value + "\n", true);
        delete [] value;
    }
    else
    {
        Erreur("Le paramètre mode est incorrect !\n");
        return false;
    }



    key = "delai_transit";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitransit=atoi(value);
        delete [] value;
        os << "delai transit = " << delaitransit << " sec" << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("Le paramètre delai_transit est incorrect !\n");
        return false;
    }


    key = "delai_tracking";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitracking=atoi(value);
        delete [] value;
        os << "delai tracking = " << delaitracking << " sec" << endl << endl;
        Affiche(&os, true);
    }
    else
    {
        Erreur("Le paramètre delai_tracking est incorrect !\n");
        return false;
    }


    /////////////////////////////////////////
    // Rubrique divers

    section = "divers";

    key = "couleurs";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        ChoixCouleurs=atoi(value);
        delete [] value;
    }
    else
    {
        /*os << "Le paramètre couleurs est incorrect !" << endl;
        Erreur(os.str());
        return false;*/
    }

    return true;
}


