////////////////////////////////
// Classe Alignement          //
// Franck RICHARD             //
// franckrichard033@gmail.com //
// BAORadio                   //
// février 2011               //
////////////////////////////////


#include "alignement.h"


/**************************************************************************************
** Constructeur de la classe Alignement
**
***************************************************************************************/

Alignement::Alignement()
{
    //Initialisation des paramètres des antennes

    InitAlignement();
}


/**************************************************************************************
** Destructeur de la classe Alignement
**
***************************************************************************************/

Alignement::~Alignement()
{
}


/**************************************************************************************
** Initialisations des vecteurs et des paramètres d'alignement
**
***************************************************************************************/

void Alignement::InitAlignement()
{
    // On initialise les points de correction

    for (int i=0; i< MAXALIGNEMENTANTENNE; i++)
    {
        ad[i]         = 0.0;
        de[i]         = 0.0;
        delta_ad[i]   = 0;
        delta_de[i]   = 0;
        tsl[i]        = 0.0;
    }

    // initialisation de la matrice de rotation (à la matrice identité)

    Identity();


    // initialisation de la direction en azimut de l'étoile polaire

    delta_az_polar    = 0;

    // Nbre de points (de mesures) actuellement disponibles pour
    // bâtir les matrices de correction

    nbrcorrections    = 0;

    // Est-ce que l'antenne est en cours d'alignement ?

    AlignementEnCours = 0;

    // La matrice de correction a-t-elle été calculée ?

    Matrice_ok        = false;

    // Méthode d'alignement par défaut

    MethodeAlignement = SIMPLE;
}



/**************************************************************************************
**  Initialisation des paramètres de la classe astro
**
***************************************************************************************/

void Alignement::TransmettreParametresClasseAstro(double Annee, double Mois, double Jour, double Heu, double Min, double Sec, double Longitude, double Latitude, double Pression, double Temp)
{
    DefinirLongitudeLatitude(Longitude, Latitude);
    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);
    DefinirPressionTemp(Pression, Temp);
}


/**************************************************************************************
**  1 0 0
**  0 1 0
**  0 0 1
***************************************************************************************/

void Alignement::Identity()
{
    MATRICE[1][1] = 1.0;
    MATRICE[2][1] = 0.0;
    MATRICE[3][1] = 0.0;

    MATRICE[1][2] = 0.0;
    MATRICE[2][2] = 1.0;
    MATRICE[3][2] = 0.0;

    MATRICE[1][3] = 0.0;
    MATRICE[2][3] = 0.0;
    MATRICE[3][3] = 1.0;
}



/**************************************************************************************
**  Calcule la matrice de rotation d'un angle alpha autour d'un axe défini
**  par les paramètres t et r dans un système de coordonnées spĥériques
**  Le signe - devant le cos permet seulement de rendre le repère compatible avec WinStars
** que j'utilise pour faire des vérifications...
** Fonction utile pour de débogage de la fct d'alignement
**
***************************************************************************************/

void Alignement::RotationAutourDunAxe(double t, double r, double alpha)
{
    double x = -cos(t) * cos(r);
    double y = sin(t) * cos(r);
    double z = sin(r);

    double c = cos(alpha);
    double s = sin(alpha);


    MATRICE[1][1] = x * x + (1.0 - x * x) * c;
    MATRICE[1][2] = x * y * (1.0 - c) - z * s;
    MATRICE[1][3] = x * z * (1.0 - c) + y * s;

    MATRICE[2][1] = x * y * (1.0 - c) + z * s;
    MATRICE[2][2] = y * y + (1.0 - y * y) * c;
    MATRICE[2][3] = y * z * (1.0 - c) - x * s;

    MATRICE[3][1] = x * z * (1.0 - c) - y * s;
    MATRICE[3][2] = y * z * (1.0 - c) + x * s;
    MATRICE[3][3] = z * z + (1.0 - z * z) * c;
}


/**************************************************************************************
** Matrice intermédiaire nécessaire au calcul de la matrice de correction
** dans le mode d'alignement TAKI
** Consultez la documentation pour comprendre le principe de la correction AFFINE/TAKI
**
***************************************************************************************/

void Alignement::Calculer_Matrice_LMN(Coord p1, Coord p2, Coord p3)
{
    double temp[4][4];
    double UnitVect[4][4];

    temp[1][1] = p2.x - p1.x;
    temp[2][1] = p3.x - p1.x;

    temp[1][2] = p2.y - p1.y;
    temp[2][2] = p3.y - p1.y;

    temp[1][3] = p2.z - p1.z;
    temp[2][3] = p3.z - p1.z;

    UnitVect[1][1] = (temp[1][2] * temp[2][3]) - (temp[1][3] * temp[2][2]);
    UnitVect[1][2] = (temp[1][3] * temp[2][1]) - (temp[1][1] * temp[2][3]);
    UnitVect[1][3] = (temp[1][1] * temp[2][2]) - (temp[1][2] * temp[2][1]);
    UnitVect[2][1] = pow(UnitVect[1][1], 2.0) + pow(UnitVect[1][2], 2.0) + pow (UnitVect[1][3], 2.0);
    UnitVect[2][2] = sqrt(UnitVect[2][1]);
    if (UnitVect[2][2] != 0.0) UnitVect[2][3] = 1.0 / UnitVect[2][2];

    for (char i=1; i<=2; i++) for (char j=1; j<=3; j++) GETLMN[i][j] = temp[i][j];

    GETLMN[3][1] = UnitVect[2][3] * UnitVect[1][1];
    GETLMN[3][2] = UnitVect[2][3] * UnitVect[1][2];
    GETLMN[3][3] = UnitVect[2][3] * UnitVect[1][3];
}



/**************************************************************************************
** Calcul de la matrice de correction dans le mode d'alignement TAKI
**
***************************************************************************************/

int Alignement::Calculer_Matrice_Taki(double x, double y, Coord a1, Coord a2, Coord a3, Coord m1, Coord m2, Coord m3)
{
    double Det;
    double EQLMN1[4][4];
    double EQLMN2[4][4];
    double EQMI_T[4][4];

    //construit les matrices EQLMN1 & 2

    Calculer_Matrice_LMN(a1, a2, a3);

    for (char i=1; i<=3; i++) for (char j=1; j<=3; j++) EQLMN1[i][j]=GETLMN[i][j];

    Calculer_Matrice_LMN(m1, m2, m3);

    for (char i=1; i<=3; i++) for (char j=1; j<=3; j++) EQLMN2[i][j]=GETLMN[i][j];

    // Calcule le déterminant de EQLMN1

    Det = EQLMN1[1][1] * ((EQLMN1[2][2] * EQLMN1[3][3]) - (EQLMN1[3][2] * EQLMN1[2][ 3]));
    Det = Det - (EQLMN1[1][2] * ((EQLMN1[2][1] * EQLMN1[3][3]) - (EQLMN1[3][1] * EQLMN1[2][3])));
    Det = Det + (EQLMN1[1][3] * ((EQLMN1[2][1] * EQLMN1[3][2]) - (EQLMN1[3][1] * EQLMN1[2][2])));


    // Calcule la matrice inverse EQMI_T de EQLMN1

    if (Det == 0.0)
    {
        ErreurLog("Erreur Matrice_Taki. Impossible de calculer l'inverse de la matrice avec un determinant = 0 \n");
        return 0;
    }
    else
    {
        EQMI_T[1][1] = ((EQLMN1[2][2] * EQLMN1[3][3]) - (EQLMN1[3][2] * EQLMN1[2][3])) / Det;
        EQMI_T[1][2] = ((EQLMN1[1][3] * EQLMN1[3][2]) - (EQLMN1[1][2] * EQLMN1[3][3])) / Det;
        EQMI_T[1][3] = ((EQLMN1[1][2] * EQLMN1[2][3]) - (EQLMN1[2][2] * EQLMN1[1][3])) / Det;
        EQMI_T[2][1] = ((EQLMN1[2][3] * EQLMN1[3][1]) - (EQLMN1[3][3] * EQLMN1[2][1])) / Det;
        EQMI_T[2][2] = ((EQLMN1[1][1] * EQLMN1[3][3]) - (EQLMN1[3][1] * EQLMN1[1][3])) / Det;
        EQMI_T[2][3] = ((EQLMN1[1][3] * EQLMN1[2][1]) - (EQLMN1[2][3] * EQLMN1[1][1])) / Det;
        EQMI_T[3][1] = ((EQLMN1[2][1] * EQLMN1[3][2]) - (EQLMN1[3][1] * EQLMN1[2][2])) / Det;
        EQMI_T[3][2] = ((EQLMN1[1][2] * EQLMN1[3][1]) - (EQLMN1[3][2] * EQLMN1[1][1])) / Det;
        EQMI_T[3][3] = ((EQLMN1[1][1] * EQLMN1[2][2]) - (EQLMN1[2][1] * EQLMN1[1][2])) / Det;
    }


    // Effectue le produit des matrices EQMI_T et EQLMN2

    MATRICE[1][1] = (EQMI_T[1][1] * EQLMN2[1][1]) + (EQMI_T[1][2] * EQLMN2[2][1]) + (EQMI_T[1][3] * EQLMN2[3][1]);
    MATRICE[1][2] = (EQMI_T[1][1] * EQLMN2[1][2]) + (EQMI_T[1][2] * EQLMN2[2][2]) + (EQMI_T[1][3] * EQLMN2[3][2]);
    MATRICE[1][3] = (EQMI_T[1][1] * EQLMN2[1][3]) + (EQMI_T[1][2] * EQLMN2[2][3]) + (EQMI_T[1][3] * EQLMN2[3][3]);

    MATRICE[2][1] = (EQMI_T[2][1] * EQLMN2[1][1]) + (EQMI_T[2][2] * EQLMN2[2][1]) + (EQMI_T[2][3] * EQLMN2[3][1]);
    MATRICE[2][2] = (EQMI_T[2][1] * EQLMN2[1][2]) + (EQMI_T[2][2] * EQLMN2[2][2]) + (EQMI_T[2][3] * EQLMN2[3][2]);
    MATRICE[2][3] = (EQMI_T[2][1] * EQLMN2[1][3]) + (EQMI_T[2][2] * EQLMN2[2][3]) + (EQMI_T[2][3] * EQLMN2[3][3]);

    MATRICE[3][1] = (EQMI_T[3][1] * EQLMN2[1][1]) + (EQMI_T[3][2] * EQLMN2[2][1]) + (EQMI_T[3][3] * EQLMN2[3][1]);
    MATRICE[3][2] = (EQMI_T[3][1] * EQLMN2[1][2]) + (EQMI_T[3][2] * EQLMN2[2][2]) + (EQMI_T[3][3] * EQLMN2[3][2]);
    MATRICE[3][3] = (EQMI_T[3][1] * EQLMN2[1][3]) + (EQMI_T[3][2] * EQLMN2[2][3]) + (EQMI_T[3][3] * EQLMN2[3][3]);


    // Calcule l'offset sur le ciel

    VECT_OFFSET.x = m1.x - ((a1.x * MATRICE[1][1]) + (a1.y * MATRICE[2][1]) + (a1.z * MATRICE[3][1]));
    VECT_OFFSET.y = m1.y - ((a1.x * MATRICE[1][2]) + (a1.y * MATRICE[2][2]) + (a1.z * MATRICE[3][2]));
    VECT_OFFSET.z = m1.z - ((a1.x * MATRICE[1][3]) + (a1.y * MATRICE[2][3]) + (a1.z * MATRICE[3][3]));



    if (x + y == 0 )
    {
        return 0;
    }
    else
    {
        return  PointSitueDansSurfaceTriangle(x, y, a1.x, a1.y, a2.x, a2.y, a3.x, a3.y);
    }
}


void Alignement::AppliquerMatriceCorrectionTaki(Coord * result, Coord vect)
{
    // Applique la correction sur le vecteur orientation vect avec
    // vect.x = ascension droite
    // vect.y = déclinaison
    // vect.z = 1

    result->x = VECT_OFFSET.x + ((vect.x * MATRICE[1][1]) + (vect.y * MATRICE[2][1]) + (vect.z * MATRICE[3][1]));
    result->y = VECT_OFFSET.y + ((vect.x * MATRICE[1][2]) + (vect.y * MATRICE[2][2]) + (vect.z * MATRICE[3][2]));
    result->z = VECT_OFFSET.z + ((vect.x * MATRICE[1][3]) + (vect.y * MATRICE[2][3]) + (vect.z * MATRICE[3][3]));
}




/**************************************************************************************
** Matrice intermédiaire nécessaire au calcul de la matrice de correction
** dans le mode d'alignement AFFINE
**
***************************************************************************************/

void Alignement::Calculer_Matrice_GETPQ(Coord p1, Coord p2, Coord p3)
{
    GETPQ[1][1] = p2.x - p1.x;
    GETPQ[2][1] = p3.x - p1.x;
    GETPQ[1][2] = p2.y - p1.y;
    GETPQ[2][2] = p3.y - p1.y;
}



/**************************************************************************************
** Calcul de la matrice de correction dans le mode d'alignement AFFINE
**
***************************************************************************************/

int Alignement::Calculer_Matrice_Affine( double x, double y, Coord a1, Coord a2, Coord a3, Coord m1, Coord m2, Coord m3)
{
    double Det;
    double EQMI[4][4];
    double EQMP[4][4];
    double EQMQ[4][4];

    //Construit les matrices EQMP et EQMQ

    Calculer_Matrice_GETPQ(a1, a2, a3);

    for (char i=1; i<=3; i++) for (char j=1; j<=3; j++) EQMP[i][j]=GETPQ[i][j];

    Calculer_Matrice_GETPQ(m1, m2, m3);

    for (char i=1; i<=3; i++) for (char j=1; j<=3; j++) EQMQ[i][j]=GETPQ[i][j];

    // Calcule l'inverse de EQMP

    Det = (EQMP[1][1] * EQMP[2][2]) - (EQMP[1][2] * EQMP[2][1]);

    if (Det == 0)
    {
        ErreurLog("Erreur Matrice_Affine. Impossible de calculer l'inverse de la matrice avec un determinant = 0 \n");
        return 0;
    }
    else
    {
        // Si le déterminant n'est pas nul
        EQMI[1][1] =  EQMP[2][2] / Det;
        EQMI[1][2] = -EQMP[1][2] / Det;
        EQMI[2][1] = -EQMP[2][1] / Det;
        EQMI[2][2] =  EQMP[1][1] / Det;
    }


    // MATRICE = EQMI * EQMQ

    MATRICE[1][1] = (EQMI[1][1] * EQMQ[1][1]) + (EQMI[1][2] * EQMQ[2][1]);
    MATRICE[1][2] = (EQMI[1][1] * EQMQ[1][2]) + (EQMI[1][2] * EQMQ[2][2]);
    MATRICE[2][1] = (EQMI[2][1] * EQMQ[1][1]) + (EQMI[2][2] * EQMQ[2][1]);
    MATRICE[2][2] = (EQMI[2][1] * EQMQ[1][2]) + (EQMI[2][2] * EQMQ[2][2]);

    // Calcul du vecteur offset sur le ciel

    VECT_OFFSET.x = m1.x - ((a1.x * MATRICE[1][1]) + (a1.y * MATRICE[2][1]));
    VECT_OFFSET.y = m1.y - ((a1.x * MATRICE[1][2]) + (a1.y * MATRICE[2][2]));

    return 1;
}



/**************************************************************************************
** Calcule la quantité CoordObjets * MATRICECorrection + VECT_offset
**
***************************************************************************************/

void Alignement::AppliquerMatriceCorrectionAffine( Coord * result, Coord ob)
{
    result->x = VECT_OFFSET.x + ((ob.x * MATRICE[1][1]) + (ob.y * MATRICE[2][1]));
    result->y = VECT_OFFSET.y + ((ob.x * MATRICE[1][2]) + (ob.y * MATRICE[2][2]));
}



/**************************************************************************************
** Calcule la matrice de correction issue de la procédure d'alignement des antennes
**
** Cette méthode qui utilise trois points de correction seulement est particulièrement
** simple mais repose sur la qualité des pointages effectués lors de l'alignement sur
** les trois étoiles et ne prend en compte de la torsion de la monture etc...
**
** Les variables a1, a2, a3 contiennent les coordonnées horaires calculées (AD & Dec) des trois
** étoiles visées...
** Les variables m1, m2, m3 contiennent les coordonnées horaires effectivement mesurées
** de ces trois mêmes objets
**
***************************************************************************************/

int Alignement::Calculer_Matrice_Simple( Coord a1, Coord a2, Coord a3, Coord m1, Coord m2, Coord m3)
{
    double x, y, z;

    Coord aa1, aa2, aa3;
    Coord mm1, mm2, mm3;

    stringstream os;

    // Pour chaque variable coordonnées, on calcule d'abord l'azimut et la hauteur de l'objet correspondant

    Azimut(a1.x, a1.y, &aa1.x, &aa1.y);

    // puis on transforme les coordonnées sphériques (azi, haut) en coordonnées cartésiennes x, y, z
    x=cos(aa1.x) * cos(aa1.y);
    y=sin(aa1.x) * cos(aa1.y);
    z=sin(aa1.y);
    aa1.x=x;
    aa1.y=y;
    aa1.z=z;

    Azimut(a2.x, a2.y, &aa2.x, &aa2.y);
    os << "a2.y " << a2.y << "\n";
    x=cos(aa2.x) * cos(aa2.y);
    y=sin(aa2.x) * cos(aa2.y);
    z=sin(aa2.y);
    aa2.x=x;
    aa2.y=y;
    aa2.z=z;

    Azimut(a3.x, a3.y, &aa3.x, &aa3.y);
    os << "a3.y " << a3.y << "\n";
    x=cos(aa3.x) * cos(aa3.y);
    y=sin(aa3.x) * cos(aa3.y);
    z=sin(aa3.y);
    aa3.x=x;
    aa3.y=y;
    aa3.z=z;

    Azimut(m1.x, m1.y, &mm1.x, &mm1.y);
    os << "m1.y " << m1.y << "\n";
    x=cos(mm1.x) * cos(mm1.y);
    y=sin(mm1.x) * cos(mm1.y);
    z=sin(mm1.y);
    mm1.x=x;
    mm1.y=y;
    mm1.z=z;

    Azimut(m2.x, m2.y, &mm2.x, &mm2.y);
    os << "m2.y " << m2.y << "\n";
    x=cos(mm2.x) * cos(mm2.y);
    y=sin(mm2.x) * cos(mm2.y);
    z=sin(mm2.y);
    mm2.x=x;
    mm2.y=y;
    mm2.z=z;

    Azimut(m3.x, m3.y, &mm3.x, &mm3.y);
    os << "m3.y " << m3.y << "\n";
    x=cos(mm3.x) * cos(mm3.y);
    y=sin(mm3.x) * cos(mm3.y);
    z=sin(mm3.y);
    mm3.x=x;
    mm3.y=y;
    mm3.z=z;

    // On se retouvre à devoir résoudre les equations:

    // mm1 = aa1 * MATRICE_Correction
    // mm2 = aa2 * MATRICE_Correction
    // mm3 = aa3 * MATRICE_Correction

    // et donc à trouver les éléments de la matrice MATRICE_Correction en fonction des paramètres x, y, z des vecteurs
    // aa1, aa2, aa3, mm1, mm2, mm3

    // Les éléments de la matrice se trouvent aisément

    double div= aa1.z*aa2.y*aa3.x - aa1.y*aa2.z*aa3.x - aa1.z*aa2.x*aa3.y +
                aa1.x*aa2.z*aa3.y + aa1.y*aa2.x*aa3.z - aa1.x*aa2.y*aa3.z;

    if ( div !=0.0 )
    {

        double mm11 = -(((-aa2.z)*aa3.y*mm1.x + aa2.y*aa3.z*mm1.x + aa1.z*aa3.y*mm2.x - aa1.y*aa3.z*mm2.x -
                         aa1.z*aa2.y*mm3.x + aa1.y*aa2.z*mm3.x)/div);

        double mm12 = -((aa2.z*aa3.x*mm1.x - aa2.x*aa3.z*mm1.x - aa1.z*aa3.x*mm2.x + aa1.x*aa3.z*mm2.x + aa1.z*aa2.x*mm3.x - aa1.x*aa2.z*mm3.x)/div);

        double mm13 = -(((-aa2.y)*aa3.x*mm1.x + aa2.x*aa3.y*mm1.x + aa1.y*aa3.x*mm2.x - aa1.x*aa3.y*mm2.x - aa1.y*aa2.x*mm3.x + aa1.x*aa2.y*mm3.x)/div);

        double mm21 = -(((-aa2.z)*aa3.y*mm1.y + aa2.y*aa3.z*mm1.y + aa1.z*aa3.y*mm2.y - aa1.y*aa3.z*mm2.y -
                         aa1.z*aa2.y*mm3.y + aa1.y*aa2.z*mm3.y)/div);

        double mm22 = -((aa2.z*aa3.x*mm1.y - aa2.x*aa3.z*mm1.y - aa1.z*aa3.x*mm2.y + aa1.x*aa3.z*mm2.y + aa1.z*aa2.x*mm3.y -
                         aa1.x*aa2.z*mm3.y)/div);

        double mm23 = -(((-aa2.y)*aa3.x*mm1.y + aa2.x*aa3.y*mm1.y + aa1.y*aa3.x*mm2.y - aa1.x*aa3.y*mm2.y -
                         aa1.y*aa2.x*mm3.y + aa1.x*aa2.y*mm3.y)/div);

        double mm31 = -(((-aa2.z)*aa3.y*mm1.z + aa2.y*aa3.z*mm1.z + aa1.z*aa3.y*mm2.z - aa1.y*aa3.z*mm2.z -
                         aa1.z*aa2.y*mm3.z + aa1.y*aa2.z*mm3.z)/div);

        double mm32 = -((aa2.z*aa3.x*mm1.z - aa2.x*aa3.z*mm1.z - aa1.z*aa3.x*mm2.z + aa1.x*aa3.z*mm2.z + aa1.z*aa2.x*mm3.z -
                         aa1.x*aa2.z*mm3.z)/div);

        double mm33 = -(((-aa2.y)*aa3.x*mm1.z + aa2.x*aa3.y*mm1.z + aa1.y*aa3.x*mm2.z - aa1.x*aa3.y*mm2.z -
                         aa1.y*aa2.x*mm3.z + aa1.x*aa2.y*mm3.z)/div);



        MATRICE[1][1] = mm11;
        MATRICE[2][1] = mm12;
        MATRICE[3][1] = mm13;

        MATRICE[1][2] = mm21;
        MATRICE[2][2] = mm22;
        MATRICE[3][2] = mm23;

        MATRICE[1][3] = mm31;
        MATRICE[2][3] = mm32;
        MATRICE[3][3] = mm33;

        //Pour vérifs
        //RotationAutourDunAxe(0.3,Pi/2.0-0.1, 0.2);

        return 1;
    }
    else
    {
        return 0;
    }
}


void Alignement::AppliquerMatriceCorrectionSimple(Coord * result, Coord vect)
{
    // Applique la correction sur le vecteur orientation vect avec

    result->x = vect.x * MATRICE[1][1] + vect.y * MATRICE[2][1] + vect.z * MATRICE[3][1];
    result->y = vect.x * MATRICE[1][2] + vect.y * MATRICE[2][2] + vect.z * MATRICE[3][2];
    result->z = vect.x * MATRICE[1][3] + vect.y * MATRICE[2][3] + vect.z * MATRICE[3][3];
}




/**************************************************************************************
** Calcule la surface d'un triangle de coordonnées (px1,py1) (px2, py2) (px3, py3)
**
***************************************************************************************/

double Alignement::Surface_Triangle(double px1, double py1, double px2, double py2, double px3, double py3)
{
    double ta;

    ta = (((px2 * py1) - (px1 * py2)) ) + (((px3 * py2) - (px2 * py3))  ) + (((px1 * py3) - (px3 * py1))  );

    return  fabs(ta) / 2.0;
}


/**************************************************************************************
** Est-ce que le point de coordonnées (px, py) se trouve dans la surface du triangle
** de coordonnées (px1,py1) (px2, py2) (px3, py3) ?
**
***************************************************************************************/

bool Alignement::PointSitueDansSurfaceTriangle(double px, double py, double px1, double py1, double px2, double py2, double px3, double py3)
{
    double ta;
    double t1;
    double t2;
    double t3;

    ta = Surface_Triangle(px1, py1, px2, py2, px3, py3);
    t1 = Surface_Triangle(px,  py,  px2, py2, px3, py3);
    t2 = Surface_Triangle(px1, py1, px,  py,  px3, py3);
    t3 = Surface_Triangle(px1, py1, px2, py2, px,  py);

    stringstream os;

    os << fabs(ta - t1 - t2 - t3)<< "\n";
    AfficherLog(os.str().c_str(), true);

    if ( DistanceAngulaireEntre2Points(px, py, px1, py1) > 1.2 ) return false;
    if ( DistanceAngulaireEntre2Points(px, py, px2, py2) > 1.2 ) return false;
    if ( DistanceAngulaireEntre2Points(px, py, px3, py3) > 1.2 ) return false;

    return ( fabs(ta - t1 - t2 - t3) < 1e-10 );
}


/**************************************************************************************
** Calcule la matrice de correction issue de la procédure d'alignement des antennes
**
** Routine principale. On construit ici trois vecteurs c1, c2 et c3 qui contiennent
** les coordonnées (issues du catalogue) de trois objets visés pendant la procédure
** d'alignement. Les vecteurs m1, m2 et m3 contiennent les coordonnées mesurées
** de ces mêmes objets dans le viseur de l'instrument.
**
***************************************************************************************/

void Alignement::CalculerMatriceCorrection(double ar, double dec)
{
    Coord c1, c2, c3, c4, m1, m2, m3, m4;

    double a1, b1, a2, b2;

    int map[MAXALIGNEMENTANTENNE];

    stringstream os;

    int best1 = 0;
    int best2 = 0;
    int best3 = 0;


    // On calcule le temps sidéral local, etc...

    CalculTSL();

    switch ( MethodeAlignement )
    {
    case SIMPLE :
    {
        best1 = 1;
        best2 = 2;
        best3 = 3;

        for (int i=0; i<MAXALIGNEMENTANTENNE; i++) map[i] = i;
    }
    break;


    case AFFINE:
    case TAKI  :
    {

        double *distances;

        distances = new double [nbrcorrections+1];

        os << "Coordonnées visees : ad=" << DHMS(ar*N180divPi,true) << "  de=" << DHMS(dec*N180divPi,false) << "\n";

        Azimut(VerifAngle(ar-GetTSL()), dec, &a2, &b2);

        for (int i=1; i<=nbrcorrections; i++)
        {
            Azimut(VerifAngle(ad[i]-tsl[i]),de[i], &a1, &b1);

            distances[i] = DistanceAngulaireEntre2Points(a1, b1, a2, b2);

            map[i]=i;
        }

        for (int i=1; i<=nbrcorrections; i++)
        {
            for (int j=i+1; j<=nbrcorrections; j++)
            {
                if (distances[j]<distances[i])
                {
                    double bout=distances[i];
                    distances[i]=distances[j];
                    distances[j]=bout;

                    int bout2=map[i];
                    map[i]=map[j];
                    map[j]=bout2;
                }
            }
        }

        for (int i=1; i<=nbrcorrections; i++)
        {
            os << "Point " << map[i] << "  d=" << distances[i] << "\n";
        }

        for (int i=1; i<=nbrcorrections; i++)
        {
            for (int j=i+1; j<=nbrcorrections; j++)
            {
                for (int k=j+1; k<=nbrcorrections; k++)
                {
                    double a,b;

                    os << "Triangle : " << map[i] << " " << map[j] << " " << map[k]  << "\n";

                    os << "Objet ";

                    os << ": az=" << DHMS(a2*N180divPi, false) << "  ha=" <<  DHMS(b2*N180divPi, false) << "\n";

                    os << "Etoile " << i ;

                    Azimut(VerifAngle(ad[map[i]]-tsl[map[i]]), de[map[i]], &a, &b);

                    os << ": az=" << DHMS(a*N180divPi, false) << "  ha=" <<  DHMS(b*N180divPi, false) << "\n";

                    os << "Etoile " << j ;

                    Azimut(VerifAngle(ad[map[j]]-tsl[map[j]]), de[map[j]], &a, &b);

                    os << ": az=" << DHMS(a*N180divPi, false) << "  ha=" <<  DHMS(b*N180divPi, false) << "\n";

                    os << "Etoile " << k ;

                    Azimut(VerifAngle(ad[map[k]]-tsl[map[k]]), de[map[k]], &a, &b);

                    os << ": az=" << DHMS(a*N180divPi, false) << "  ha=" <<  DHMS(b*N180divPi, false) << "\n";

                    AfficherLog(&os, true);

                    if (PointSitueDansSurfaceTriangle(VerifAngle(ar-GetTSL()), dec,
                                                      VerifAngle(ad[map[i]]-tsl[map[i]]), de[map[i]],
                                                      VerifAngle(ad[map[j]]-tsl[map[j]]), de[map[j]],
                                                      VerifAngle(ad[map[k]]-tsl[map[k]]), de[map[k]]))
                    {
                        best1=i;
                        best2=j;
                        best3=k;
                        k=1000;
                        j=1000;
                        i=1000;
                    }
                }
            }
        }

        delete [] distances;
    }
    break;
    }



    if ( best1 == 0 && best2 == 0 && best3 == 0 )
    {
        if (MethodeAlignement == AFFINE || MethodeAlignement == TAKI)
        {
            ErreurLog("Erreur ! Pas de triangle disponible pour la correction !\nLe positionnement de l antenne sera approximatif !\n");
        }

        Identity();

        Matrice_ok = false;

        return;
    }
    else
    {
        os << "\nUtilisation du triangle : " << map[best1] << " " << map[best2] << " " << map[best3]  << "\n" ;

        if (MethodeAlignement == SIMPLE)
        {
            c1.x = VerifAngle(ad[map[best1]] - tsl[map[best1]] + GetTSL());
            c1.y = de[map[best1]];
            c1.z = 1.0;

            c2.x = VerifAngle(ad[map[best2]] - tsl[map[best2]] + GetTSL());
            c2.y = de[map[best2]];
            c2.z = 1.0;

            c3.x = VerifAngle(ad[map[best3]] - tsl[map[best3]] + GetTSL());
            c3.y = de[map[best3]];
            c3.z = 1.0;
        }
        else
        {
            c1.x = VerifAngle(ad[map[best1]] - tsl[map[best1]]);
            c1.y = de[map[best1]];
            c1.z = 1.0;

            c2.x = VerifAngle(ad[map[best2]] - tsl[map[best2]]);
            c2.y = de[map[best2]];
            c2.z = 1.0;

            c3.x = VerifAngle(ad[map[best3]] - tsl[map[best3]]);
            c3.y = de[map[best3]];
            c3.z = 1.0;
        }


        os << "Point 1 :  ho=" << DHMS(c1.x * N180divPi, false);
        os <<  "   de=" << DHMS(c1.y *N180divPi, false) <<"\n";
        os << "Point 2 :  ho=" << DHMS(c2.x * N180divPi, false);
        os <<  "   de=" << DHMS(c2.y * N180divPi, false) <<"\n";
        os << "Point 3 :  ho=" << DHMS(c3.x * N180divPi, false);
        os <<  "   de=" << DHMS(c3.y * N180divPi, false) <<"\n\n";


        m1.x = VerifAngle(c1.x + (double)delta_ad[map[best1]] * PasDeltaAD);
        m1.y = c1.y + (double)delta_de[map[best1]] * PasDeltaDe;
        m1.z = 1.0;

        m2.x = VerifAngle(c2.x + (double)delta_ad[map[best2]] * PasDeltaAD);
        m2.y = c2.y + (double)delta_de[map[best2]] * PasDeltaDe;
        m2.z = 1.0;

        m3.x = VerifAngle(c3.x + (double)delta_ad[map[best3]] * PasDeltaAD);
        m3.y = c3.y + (double)delta_de[map[best3]] * PasDeltaDe;
        m3.z = 1.0;
    }

    // Calcul de la matrice de correction en fonction de la méthode d'alignement

    switch (MethodeAlignement)
    {
    case SIMPLE:
        os << "Methode d alignement SIMPLE\n";
        Matrice_ok = Calculer_Matrice_Simple(c1, c2, c3 ,m1, m2, m3);
        break;
    case AFFINE:
        os << "Methode d alignement AFFINE\n";
        Matrice_ok = Calculer_Matrice_Affine( VerifAngle( ar - GetTSL() ), dec, c1, c2, c3 ,m1, m2, m3);
        break;
    case TAKI:
        os << "Methode d alignement TAKI\n";
        Matrice_ok = Calculer_Matrice_Taki  ( VerifAngle( ar - GetTSL() ), dec, c1, c2, c3 ,m1, m2, m3);
        break;
    }


    // Affiche de la matrice pour vérif

    os << "\nCalcul de la matrice de correction:\n";
    os << MATRICE[1][1] << "   \t" << MATRICE[2][1] << "    \t" << MATRICE[3][1] << "\n";
    os << MATRICE[1][2] << "   \t" << MATRICE[2][2] << "    \t" << MATRICE[3][2] << "\n";
    os << MATRICE[1][3] << "   \t" << MATRICE[2][3] << "    \t" << MATRICE[3][3] << "\n";






    //Vérifications supp
    /*
    Coord result, mm1;
    double targetAz, targetAlt;

    os << "\n\n\n" << GetLatitude()*N180divPi << "      " << GetLongitude()*N180divPi << "\n";
    os << GetPression() << "      " << GetTemperature() << "\n";
    os << GetHeure() << ":" << GetMin() << ":" << GetSec() << "\n";
    os << DHMS(GetTSL()*N180divPi, true) << "\n\n";

      Azimut(c3.x, c3.y, &mm1.x, &mm1.y);

    os << mm1.x*N180divPi << "   "  <<  mm1.y*N180divPi << "\n";


    Azimut(m3.x, m3.y, &mm1.x, &mm1.y);

    os << mm1.x*N180divPi << "   "  <<  mm1.y*N180divPi << "\n";


    Azimut(c3.x, c3.y, &mm1.x, &mm1.y);

    double x=cos(mm1.x) * cos(mm1.y);
    double y=sin(mm1.x) * cos(mm1.y);
    double z=sin(mm1.y);
    mm1.x=x;
    mm1.y=y;
    mm1.z=z;

    AppliquerMatriceCorrectionSimple(&result, mm1);

    if (result.x != 0.0)
    {
        targetAz = atan( result.y / result.x );

        if (result.x < 0) targetAz+=Pi;
    }
    else targetAz = Pidiv2;

    targetAz  = VerifAngle(targetAz);

    targetAlt = asin(result.z);

    os << targetAz*N180divPi << "   "  <<  targetAlt*N180divPi << "\n";
    */



    //Afficher les messages de cette fonction

    AfficherLog(os.str(), true);
}



/**************************************************************************************
** Le fichier file existe-t-il et n'est-il pas vide ?
**************************************************************************************/

bool Alignement::is_readable( const std::string & file )
{
    std::ifstream fichier( file.c_str() );
    if (fichier.fail()) return false;

    // sauvegarder la position courante
    long pos = fichier.tellg();
    // se placer en fin de fichier
    fichier.seekg( 0 , std::ios_base::end );
    // récupérer la nouvelle position = la taille du fichier
    long size = fichier.tellg();
    return (size > 0);
}


/**************************************************************************************
** Chargement des paramètres d'alignement des antennes
**************************************************************************************/

bool Alignement::ChargementParametresAlignement(string IP, string fileName, double a, double b)
{
    static double mema = a;

    static double memb = b;

    string value;

    stringstream os, os2;

    char *delta_az_polar_data = NULL;

    char *ad_data = NULL;

    char *de_data = NULL;

    char *delta_ad_data = NULL;

    char *delta_de_data = NULL;

    char *align_data = NULL;

    char *alignment_method_data = NULL;

    char *tsl_data = NULL;


    if (!is_readable(fileName)) return false;

    // Chargement des corrections des antennes

    bool modificationAlignement = false;

    // On sélectionne la bonne antenne (à partir de son adresse ip).

    if (IP.rfind(".") != string::npos) IP = IP.substr(IP.rfind(".")+1);


    os << "Alignement antenne ip x.x.x." << IP;


    //nbrcorrections = 0; TODO semble poser un problème


    for (int j=1; j < MAXALIGNEMENTANTENNE; j++)
    {
        os2 << "ad " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &ad_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "de " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &de_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "delta_ad " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_ad_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "delta_de " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_de_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "tsl " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &tsl_data,  (char*)fileName.c_str());

        os2.str("");


        if ( ad_data && de_data && delta_ad_data && delta_de_data
                && tsl_data && atof(tsl_data)!=0.0 && atof(ad_data)!=0.0 && atof(de_data)!=0.0)
        {
            // Si les corrections de l'antenne i ont été modifiées depuis le démarrage du programme
            // -> on les applique...

            nbrcorrections = j;

            if (delta_ad[j] != atol(delta_ad_data) || delta_de[j] != atol(delta_de_data))
            {
                ad[j] = atof(ad_data);

                de[j] = atof(de_data);

                delta_ad[j] = atol(delta_ad_data);

                delta_de[j] = atol(delta_de_data);

                tsl[j] = atof(tsl_data);

                os2 << "Correction antenne ip=" << IP <<  " ad=" << ad[j] << " de=" << de[j] << " deltaAD=" << delta_ad[j] << " deltaDe=" << delta_de[j] << " tsl=" << tsl[j] << "\n";

                AfficherLog(&os2, true);

                modificationAlignement = true;
            }
        }

        SAFEDELETE_TAB(ad_data);

        SAFEDELETE_TAB(de_data);

        SAFEDELETE_TAB(delta_ad_data);

        SAFEDELETE_TAB(delta_de_data);

        SAFEDELETE_TAB(tsl_data);
    }

    os2 << "delta_az_polar";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_az_polar_data,  (char*)fileName.c_str());

    os2.str("");

    if (delta_az_polar_data)
    {
        if ( delta_az_polar != atol(delta_az_polar_data) )
        {
            delta_az_polar = atol(delta_az_polar_data);

            modificationAlignement = true;
        }
    }

    os2 << "alignment_method";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &alignment_method_data,  (char*)fileName.c_str());

    os2.str("");

    /*  char chaine[100];
    sprintf(chaine, "************** %s\n",alignment_method_data);
      cout << chaine;*/

    if (alignment_method_data)
    {
        if ( MethodeAlignement != atoi(alignment_method_data) )
        {
            MethodeAlignement = atoi(alignment_method_data);

            modificationAlignement = true;
        }
    }

    os2 << "align";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &align_data,  (char*)fileName.c_str());

    os2.str("");


    if (align_data)
    {
        // Si la procédure d'alignement se termine, il faut calculer la matrice de correction

        if (AlignementEnCours != atoi(align_data))
        {
            AlignementEnCours = atoi(align_data);

            modificationAlignement = true;
        }
    }

    if (mema!=a || memb!=b)
    {
        mema = a;

        memb = b;

        modificationAlignement = true;
    }

    // si nbrcorrections=0 (suite à l'utilisation de la commande reset dans BAOcontrol par ex) il faut
    // réinitialiser la matrice

    if (nbrcorrections == 0 ) modificationAlignement = true;


    // Si un des paramètres d'une antenne change pendant la procédure d'alignement ou que l'on a
    // plus de deux points d'alignement sur une antenne, alors on calcule la matrice de correction

    if (modificationAlignement)
    {
        if  (nbrcorrections >=3 && AlignementEnCours == 0)
        {
            // IDLog("Calcul matrice\n", true);
            CalculerMatriceCorrection(a, b);
        }
        else
        {
            Identity();

            Matrice_ok = false;
        }
    }

    SAFEDELETE_TAB(delta_az_polar_data);

    SAFEDELETE_TAB(alignment_method_data);

    SAFEDELETE_TAB(align_data);

    return true;
}



/**************************************************************************************
** Sauvegarde des paramètres d'alignement des antennes
***************************************************************************************/

bool Alignement::EnregistrementParametresAlignement(string IP, string fileName)
{
    string section;
    string key;
    string value;

    stringstream os;


    //AfficherLog("Enregistrement de l'alignement des antennes dans le fichier " + fileName + "\n", true);

    //Enregistrement des corrections des l'antennes

    os << "Alignement antenne ip x.x.x." << IP;
    section = os.str();
    os.str("");

    os << "delta_az_polar";
    key     = os.str();
    os.str("");
    os << delta_az_polar;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "align";
    key     = os.str();
    os.str("");
    (AlignementEnCours == true) ? os << "1" : os << "0";
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "alignment_method";
    key     = os.str();
    os.str("");
    os << MethodeAlignement;
    value = os.str();
    os.str("");


    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    /*    char chaine[100];
    sprintf(chaine, "s************** %i\n",MethodeAlignement);
       cout << chaine;*/


    for (int j = 1; j <= MAXALIGNEMENTANTENNE; j++)
    {
        os << "ad " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "de " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_ad " << j;
        key     = os.str();
        os.str("");
        os << 0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_de " << j;
        key     = os.str();
        os.str("");
        os << 0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "tsl " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());
    }


    for (int j = 1; j <= nbrcorrections; j++)
    {
        os << "ad " << j;
        key     = os.str();
        os.str("");
        os << ad[j];
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "de " << j;
        key     = os.str();
        os.str("");
        os << de[j];
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_ad " << j;
        key     = os.str();
        os.str("");
        os << delta_ad[j];
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_de " << j;
        key     = os.str();
        os.str("");
        os << delta_de[j];
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "tsl " << j;
        key     = os.str();
        os.str("");
        os << tsl[j];
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());
    }


    return true;
}


bool Alignement::EnregistrementParametresAlignement(int IP, string fileName)
{
    stringstream os;

    os << IP;

    return EnregistrementParametresAlignement(os.str(), fileName);
}
