/******************************/
/* BAOCONTROL                 */
/*                            */
/* Franck RICHARD             */
/* franckrichard033@gmail.com */
/* Février 2012               */
/******************************/


#include "baocontrol.h"



/**************************************************************************************
** Constructeur de la classe BAOcontrol
**
***************************************************************************************/

BAOcontrol::BAOcontrol()
{
    float a1, a2, a3;
    
    char Version[200];

    // Le choix de la couleur des caractères est adapté pour une fenêtre de terminal
    // ayant un fond blanc. pour un fond noir, mettre couleurs = 2 dans le fichier params

    ChoixCouleurs=1;

    // Affichage de la version du programme

    sprintf(Version,"\n\n\n\n\n \
 ******************************************\n \
 *             BAORadio Control           *\n \
 * Laboratoire de l'Accélérateur Linéaire *\n \
 *                                        *\n \
 *             v%s   %s           *\n \
 *        franckrichard033@gmail.com      *\n \
 ******************************************\n\n\n", VERSION, VERSION_DATE);
    
    AfficherLog(Version, true);


    // Initialisation des variables globales et des pointeurs
    // pour la signification des variables, voir baocontrol.h

    //BAOcontrol

    delaitransit           = 0;
    delaitracking          = 0;
    lognum                 = 0;
    numAntennes            = 0;
    numobjets              = 0;
    numEtoiles             = 0;
    runnum                 = 1;

    MethodeAlignement      = SIMPLE;

    Pression               = 1013.0;
    Temperature            = 10.0;

    NoExit                 = true;
    Transit                = false;
    run                    = false;
    exitrun                = false;
    ModificationAlignement = false;

    fd                     = NULL;
    client_socket          = NULL;

    // chaînes de caractères et tableaux

    LatitudeChar           = "";
    LongitudeChar          = "";
    Serveur                = "";
    Port                   = "";
    chaineDateHeure[0]     = 0;

    Antennes 	           = new DefAntenne[ MAXANTENNES ];
    objets                 = new DefObjets [ MAXOBJETS ];
    Etoiles                = new DefEtoiles[ MAXETOILES];


    // on initialise aussi les structures

    for (int i=0; i<MAXANTENNES; i++)
    {
        Antennes[i].ok = 0;
        Antennes[i].ip = 0;

        Antennes[i].AlignementAntenne = new Alignement();
    }

    for (int i=0; i<MAXOBJETS; i++)
    {
        objets[i].JJ   = 0.0;
        objets[i].exec = false;
    }

    for (int i=0; i<MAXETOILES; i++)
    {
        Etoiles[i].nom          = "";
        Etoiles[i].ad           = 0.0;
        Etoiles[i].de           = 0.0;
        Etoiles[i].az           = 0.0;
        Etoiles[i].ha           = 0.0;
        Etoiles[i].mag          = 0.0;
        Etoiles[i].selectionnee = false;
    }


    // Chargement du fichier contenant les paramètres de l'application BAOcontrol
    // La variable permet de récupérer le nom de l'utilisateur ayant ouvert la session sous linux
    // c'est dans le répertoire /home/USER/ que BAOcontrol cherche le fichier de paramètre baocontrol_params

    if (!ChargementParametres("/home/" + (string)getenv("USER") + "/baocontrol_params"))
    {
        ErreurLog("Le fichier de configuration 'params' contient des erreurs ou n'existe pas.\n");
        ErreurLog("Il devrait se trouver dans le répertoire /home/" + (string)getenv("USER") +"\n");
        ErreurLog("Merci de vérifier...\n\n");
        exit(1);
    }


    // Conversion de la latitude et de la longitude en radians

    Decomposition(LatitudeChar, 0, &a1, &a2, &a3);

    double Latitude = (a1 + a2 / 60.0 + a3 / 3600.0) * Pidiv180;

    Decomposition(LongitudeChar, 1, &a1, &a2, &a3);

    double Longitude = Pi2 - ( a1 + a2 / 60.0 + a3 / 3600.0 ) * Pidiv180;


    /*
        Latitude=(48.0+41.0/60.0+58.0/3600.0)*Pidiv180;
        Longitude = -(2.0+10.0/60.0+18.0/3600.0)*Pidiv180;
        Pression=0.0;
    */



    // Transfert de la latitude, longitude, pression, température à la classe Astro

    DefinirLongitudeLatitude(Longitude, Latitude);

    DefinirPressionTemp(Pression, Temperature);




      //DefinirDateHeure(2012.0, 6.0, 29.0, 21.0, 56.0, 48.0);




    // On lance les calculs (Temps sidéral local etc...)

    CalculTSL();


      /*double c,d;
      double e,f;
      stringstream os;

      os << "tsl=" << DHMS(GetTSL()*N180divPi, true)<< "  " << GetTSL() << "\n";

      AfficherLog(os.str(), true);
      os.str("");

      // double a=15.0*(5.0+55.0/60.0+10.3/3600.0)*Pidiv180;
      // double b=(7.0+24.0/60.0+25.0/3600.0)*Pidiv180;

      //  double a=15.0*(6.0+45.0/60.0+8.7/3600.0)*Pidiv180;
      //double b=-(16.0+43.0/60.0+0.3/3600.0)*Pidiv180;

      //double a=15.0*(5.0+55.0/60.0+50.0/3600.0)*Pidiv180;
      //double b=(7.0+24.0/60.0+30.0/3600.0)*Pidiv180;

      //  double a=15.0*(23.0+7.0/60.0+8.0/3600.0)*Pidiv180;
      // double b=(59.0+29.0/60.0+12.0/3600.0)*Pidiv180;

      double a=15.0*(14.0+15.0/60.0+39.5/3600.0)*Pidiv180;
      double b=(19.0+10.0/60.0+46.7/3600.0)*Pidiv180;

      //CoordonneesHorairesDouble Soleil;

      //CalculARDecSoleil(&Soleil);

      //a=Soleil.ar;
      //b=Soleil.dec;
      // Precession(&a, &b);
      // NutationEtoile(&a, &b);

      //  AberrationAnnuelle(&a, &b);
      
      os << "ad=" << a << "   de=" << b  << "\n";

      Azimut(a,b,&c,&d);
      
      c += 2.91488 * Pidiv180;
      d -= 8.08241 * Pidiv180;
      
      AzHa2ADDe(c, d, &a, &b);
      
      os << "ad=" << DHMS(a*N180divPi, true) << "   de=" << DHMS(b*N180divPi, false) << "\n";
      os << "ad=" << a << "   de=" << b  << "\n";

      AfficherLog(os.str(), true);

      Azimut(a,b,&c,&d);
      
      os.str("");

      os << "az=" << DHMS(c*N180divPi, false)<< "   ha=" << DHMS(d*N180divPi, false) << "\n";

      AfficherLog(os.str(), true);*/

   



    //Chargement du catalogue d'étoiles que l'on utilise pour l'alignement des antennes

    if ( !ChargementCatalogueEtoiles("/home/" + (string)getenv("USER") + "/bao_catalogue.dat") )
    {
        ErreurLog("Le catalogue d'étoiles 'bao_catalogue' est introuvable.\n");
        ErreurLog("Il devrait se trouver dans le répertoire /home/" + (string)getenv("USER") +"\n");
        ErreurLog("Merci de vérifier...\n\n");
        exit(1);
    }


    // Ouverture du socket avec indi_BAO

    client_socket = new ClientSocket("localhost", atoi(Port.c_str()) );

    // On se connecte au pilote indi_BAO

    Connect(true);
}



/**************************************************************************************
** Destructeur de la classe BAOcontrol
**
***************************************************************************************/

BAOcontrol::~BAOcontrol()
{

    AfficherLog("\nNettoyage de la mémoire...\n\n", true);

    // destruction des pointeurs et des tableaux

    for (int i=0; i<MAXANTENNES; i++)
    {
        delete Antennes[i].AlignementAntenne;
    }

    delete [] Antennes;
    delete [] objets;
    delete [] Etoiles;


    // destruction du socket

    delete client_socket;
}



/**************************************************************************************
** Dessine les éléments dans la fenêtre graphique
**
***************************************************************************************/

void BAOcontrol::Dessiner()
{
    // On récupère la date et l'heure contenue dans la classe astro (ainsi que le jour julien)

    double h  = GetHeure();
    double mi = GetMin();
    double s  = GetSec();
    double a  = GetAnnee();
    double m  = GetMois();
    double j  = GetJour();
    double JJ = GetJJ();


    // on évite les affichages de l'heure du genre 12:56:60 ou 24:00:00

    if (s == 60) {
        s = 0;
        mi++;
        if (mi == 60) {
            mi = 0;
            h++;
            if (h == 24) {
                h = 0;
                j++;
            }
        }
    }

    //affichage de la date et de l'heure

    sprintf(chaineDateHeure, "%02.0f:%02.0f:%02.0f TU    %02.0f/%02.0f/%04.0f    JJ=%10.5f    Tsl=%s",
            h, mi, s, j, m, a, JJ, DHMS(GetTSL()*N180divPi, true).c_str());



    // En cas d'exécution d'un fichier de mouvements, on affiche la durée avant la prochaine étape

    if (run)
    {
//       sprintf(chaine, "Prochaine etape dans %4.1fs", (objets[runnum].JJ - JJ) * 3600.0 * 24.0);

    }
}



/**************************************************************************************
** Récupère les messages envoyés par le serveur indi_BAO
**
***************************************************************************************/

void BAOcontrol::LireReponse()
{
    string reponse, memreponse, decomp;

    bool test = false;

    // on récupère les messages provenant d'indi_BAO

    *client_socket >> reponse;

    do
    {
        // s'il contient le mot message (et vient donc d'indi_BAO)

        if (reponse.find("message") != string::npos)
        {
            // On ne garde que la partie intéressante
            // La partie qui suit "message="...

            reponse = reponse.substr(reponse.find("message") + 9);

            // (on en garde une trace)

            memreponse = reponse;

            // ...on extrait le message jusqu'au caractère "

            reponse = reponse.substr(0, reponse.find("\""));

            // On récupère l'adresse ip de l'antenne qui nous parle
            // et on met à jour le tableau des états de l'antenne correspondante
            // Dans les messages d'indi_BAO, l'adresse ip précède les mots "Antennes connectées"

            if (reponse.find(" (Antennes connectees")!=string::npos)
            {
                decomp = reponse.substr(0, reponse.find(" (Antennes connectees"));
                decomp = decomp.substr(decomp.rfind(".")+1);

                // si cette antenne est déjà connectée, on change son état si nécessaire

                if ( atoi(decomp.c_str()) != 0 ) for (int i=0; i<numAntennes; i++)
                    {
                        // on a trouvé l'antenne correspondante

                        if (atoi(decomp.c_str()) == Antennes[i].ip)
                        {
                            Antennes[i].ok = 1;
                            test = true;
                            break;
                        }
                    }

                // ou si c'est une nouvelle antenne, on l'ajoute au tableau Antennes

                if (!test)
                {
                    Antennes[numAntennes].ip = atoi(decomp.c_str());

                    Antennes[numAntennes++].ok = 1;
                }
            }

            // erreur sur une antenne -> on change son état dans la fenêtre
            // on trouve les erreurs dans les messages d'indi_BAO formulée
            // de la façon suivante: "ALERTE antenne" ou bien "Erreur sur l antenne"

            if  ((reponse.find("ALERTE antenne ") != string::npos) ||
                    (reponse.find("Erreur sur l antenne ") != string::npos))
            {
                decomp = reponse.substr(0, reponse.find(" :"));
                decomp = decomp.substr(decomp.rfind(".") + 1);

                for (int i=0; i<numAntennes; i++)
                {
                    // On identifie l'antenne dans le tableau Antennes et on change son état

                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok = 2;  // Erreur sur l'antenne i
                    }
                }
            }

            // On sauvegarde le message dans le fichier logs mais on ne l'affiche pas

            stringstream os;

            os << lognum << " : " << reponse;

            AfficherLog(os.str() + "\n", false);

            if ( lognum < MAXLOG - 1 ) logs[ lognum++ ] = os.str();

            reponse = memreponse;
        }
        else reponse = "";


    } while ( reponse != "" );

    // on actualise la fenêtre

    Dessiner();

}



/**************************************************************************************
** Est-ce que la réponse du pilote indi_BAO est conforme à ce que l'on attendait ?
**
**
***************************************************************************************/

bool BAOcontrol::VerifReponse(string reponseattendue, string *retourreponse)
{
    string reponse = "";
    string memreponse = "";
    int    duree = 0;
    bool   test = false;


    // TODO :La réponse du driver ne vient pas tout de suite et on peut recevoir
    // des messages intermédiaires un peu étranges impliquant la fonction CONNECT, pourquoi ?
    // Là on lit les messages cinq fois de suite pour essayer de voir passer le bon
    // Essayer de voir si on ne peut pas faire mieux...

    if (retourreponse) *retourreponse = "";

    for (int i=0; i<5 ; i++)
    {
        do
        {
            usleep(1000); // attendre 1 ms pour laisser le temps au pilote indi_BAO de répondre

            *client_socket >> reponse;

            duree++;
        }
        while (reponse.length() == 0 && duree < MAX_DELAI_REPONSE ); // on attend un message pendant 10 ms max

        // on ajoute tous les messages reçus dans memmreponse
        // ->intéressant pour le debug

        memreponse += reponse;

        if (reponse.find(reponseattendue) != string::npos)
        {
            test = true;
            break;
        }
    }


    if (!test)
    {
        // réponse inconnue -> une erreur ?

        size_t pos = memreponse.find("message=");

        if ( pos != string::npos )
        {
            memreponse = memreponse.substr(pos + 9);

            memreponse = memreponse.substr(0, memreponse.find("\""));
        }

        if (memreponse != "")  ErreurLog("Réponse du pilote indi_BAO :" + memreponse + "\n\n");

        return false;
    }

    if (retourreponse)
    {
        size_t pos = memreponse.find(reponseattendue);

        if ( pos != string::npos )
        {
            memreponse = memreponse.substr(pos);

            memreponse = memreponse.substr(0, memreponse.find("\""));
        }

        *retourreponse = memreponse;
    }

    return true;
}



/**************************************************************************************
** Calcul du jour julien et du temps sidéral local
**
***************************************************************************************/

void BAOcontrol::UpdateTime()
{
    time_t t;
    struct tm date;
    struct timeval tv;
 
    // On récupère la date et l'heure depuis l'horloge du système

    time(&t);
    date=*gmtime(&t);
    gettimeofday(&tv, NULL);

    double Annee = (double)date.tm_year + 1900.0;
    double Mois  = (double)date.tm_mon + 1.0;
    double Jour  = (double)date.tm_mday;
    double Heu   = (double)date.tm_hour;
    double Min   = (double)date.tm_min;
    double Sec   = (double)date.tm_sec + tv.tv_usec / 1.0E6;
    //double UTCP  = 0.0;//(double)date.tm_isdst;

    // On actualise la date et l'heure dans la classe Astro

    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);

    // On lance les calculs (Temps sidéral local etc...)

    CalculTSL();  
}



/**************************************************************************************
** gestion du thread permettant
** - l'exécution d'une liste de mouvements programmés
** - l'actualisation de la fenêtre graphique
** - la récupération des messages du pilote indi_BAO
**
***************************************************************************************/

void BAOcontrol::thread_process ()
{
    stringstream os;
    struct timeval tv;

    // Tant que l'utilisateur n'a pas tapé la commande 'exit'


    {
        // On actualise le jour julien et le temps sidéral local

        UpdateTime();

        // On récupère le jour julien

        double JJ = GetJJ();


        // Faut-il exécuter un fichier de mouvements ?

        if (run && numobjets > 1 && runnum < numobjets)
        {
            // On lance le mouvement numéro runnum si la date et l'heure correspondent au début
            // de son exécution et que ce mouvement n'a pas été encore réalisé

            if ( JJ >= objets[runnum].JJ && !objets[runnum].exec )
            {
                // petit message pour l'utilisateur au début de l'exécution du tout premier mouvement

                if ( runnum == 1 ) AfficherLog("Début de l'exécution du fichier de mouvements\n\n", true);

                os << "Suivi de l'objet n°" << runnum << endl;

                AfficherLog(&os, true);

                // on note que le mouvement est en cours d'exécution
                // ou qu'il a été déjà exécuté

                objets[runnum].exec = true;

                // on pointe l'objet correspondant

                Goto( objets[runnum].ad, objets[runnum].de, Transit, J2000 );
            }


            // On arrive à la fin de l'exécution du mouvements runnum

            if (JJ >= objets[runnum].JJ + objets[runnum].Duree / 3600.0 / 24.0 )
            {
                // On interrompt le mouvement

                Abort();

                // S'il s'agit du dernier mouvement -> Fin de la commande run

                if ( runnum + 1 >= numobjets )
                {
                    AfficherLog("Fin de l'execution du fichier de mouvements\n\n", true);

                    // on réaffiche le prompt

                    cout << endl;
                    ChoixCouleurs==1 ? cout << blue1 : cout << blue2;
                    cout << ">";

                    // on réinitialise toutes les variables impliquées dans la commande run

                    run		= false;
                    runnum	= 1;
                    numobjets	= 1;

                    // On sort du programme dans le cas de BAOControl -r fileName

                    NoExit = !exitrun;
                }
                else
                {
                    // on passe au mouvement suivant

                    runnum++;
                }
            }
        }

        // on lit et affiche la réponse des micro-contrôleurs

        LireReponse();

        // pause de 100 ms

        // usleep(100000);
    }


}






/**************************************************************************************
** Décrypte les commandes saisies par l'utilisateur
** Exemples :
** DecompositionCommande("goto 10:10:10 25:10:10", "goto", &chaine1, &chaine2)
** retourne chaine1  = "10:10:10"
**          chaine2  = "25:10:10"
**
** DecompositionCommande("goto messier 1","goto", &chaine1, NULL) (avec chaine2=NULL)
** retourne chaine1  = "messier 1"
**          chaine2  = NULL
**
***************************************************************************************/

bool BAOcontrol::DecompositionCommande(string chaine, string commande, string *chaine1, string *chaine2)
{
    size_t pos = chaine.find(commande);

    // si la commande est présente dans chaîne, on continue

    if (pos != string::npos)
    {
        // on s'intéresse aux caractères qui suivent la commande

        pos += commande.length();

        chaine = chaine.substr(pos);

        // on ignore les espaces superflus

        while (chaine[0] == ' ') chaine = chaine.substr(1);

        // si on veut prendre les deux arguments d'une commande

        if (chaine2 != NULL)
        {
            // on prend le premier bloc jusqu'au prochain espace -> c'est la chaine1

            *chaine1 = chaine.substr(0, chaine.find(" "));

            // puis on prend le deuxième bloc -> c'est la chaine2

            pos = chaine.find(" ");

            if (pos != string::npos)
            {
                chaine = chaine.substr(pos + 1);

                while (chaine[0] == ' ') chaine = chaine.substr(1);

                *chaine2 = chaine.substr(0, chaine.find(" "));
            }
        }
        else
        {
            // on prend tout d'un bloc après la commande -> chaine 1
            // utile pour une commande du type "goto m 42" qui doit retourner "m 42" dans une
            // même chaîne

            *chaine1 = chaine;
        }
    } else return false;

    return true;
}



/**************************************************************************************
** Envoie la longitude et la latitude du lieu d'observation au pilote indi_BAO
** sous la forme de vecteurs xml
** (voir les particularités des drivers développés sous Indi)
**
***************************************************************************************/

bool BAOcontrol::EnvoyerCoordGeographiques()
{
    // la communication avec le pilote indi_BAO
    // se fait par l'envoi de trames au format xml

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"GEOGRAPHIC_COORD\">";
        *client_socket << "<oneNumber name=\"LAT\">";
        *client_socket << LatitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"LONG\">";
        *client_socket << LongitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"GEOGRAPHIC_COORD\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerCoordGeographiques : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les coordonnées géographiques ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}


/**************************************************************************************
** Transmet la pression et la température au pilote indi_BAO
**
***************************************************************************************/

bool BAOcontrol::EnvoyerPressionTemperature()
{
    stringstream os;

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"PRESSION_DATA\">";
        *client_socket << "<oneNumber name=\"Pression\">";
        os << Pression;
        *client_socket << os.str();
        os.str("");
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"Temperature\">";
        os << Temperature;
        *client_socket << os.str();
        os.str("");
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"PRESSION_DATA\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerPressionTemperature : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("La pression et la température ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}

/**************************************************************************************
** Envoie les paramètres delaitransit et delaitracking au pilote indi_BAO
** Ces paramètres définissent la durée (en sec) entre deux actualisations
** dans les modes tracking et transit
**
***************************************************************************************/

bool BAOcontrol::EnvoyerDelaisModesTransitEtTracking()
{
    stringstream os;

    try
    {
        os << delaitransit;
        *client_socket << "<newNumberVector device=\"BAO\" name=\"DELAY1\">";
        *client_socket << "<oneNumber name=\"DELAY\">";
        *client_socket << os.str();
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";
        os.str("");

        if (!VerifReponse("name=\"DELAY1\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerDelaisModesTransitEtTracking : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }

        os << delaitracking;
        *client_socket << "<newNumberVector device=\"BAO\" name=\"DELAY2\">";
        *client_socket << "<oneNumber name=\"DELAY\">";
        *client_socket << os.str();
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";
        os.str("");

        if (!VerifReponse("name=\"DELAY2\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerDelaisModesTransitEtTracking : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    return true;
}



/**************************************************************************************
** Envoie la méthode d'alignement au pilote indi_bao
** Ce paramètre peut être SIMPLE, AFFINE ou TAKI
**
***************************************************************************************/

bool BAOcontrol::EnvoyerMethodeAlignement()
{
    stringstream os;

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_SET\">";
        *client_socket << "<oneSwitch name=\"SIMPLE\">";
        (MethodeAlignement == SIMPLE) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "<oneSwitch name=\"AFFINE\">";
        (MethodeAlignement == AFFINE) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "<oneSwitch name=\"TAKI\">";
        (MethodeAlignement == TAKI)   ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ALIGNMENT_SET\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerMethodeAlignement : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    return true;
}


/**************************************************************************************
** Envoie une commande aux antennes
** exemples : P, Z, A, Gf1000f0100
**
***************************************************************************************/

bool BAOcontrol::EnvoyerCommande(string commande, string *Message)
{
    stringstream os;

    if (Message) *Message = "";

    try
    {
        *client_socket << "<newTextVector device=\"BAO\" name=\"COMMAND_SET\">";
        *client_socket << "<oneText name=\"COMMAND\">";
        *client_socket << commande;
        *client_socket << "</oneText>";
        *client_socket << "</newTextVector>";

        if (commande == "X")
        {
            if (!VerifReponse("Position:", Message))
            {
                ErreurLog("Pas de réponse de l'antenne à la requête X.\n\n");

                return false;
            }

            return true;
        }

        if (commande == "D")
        {
            if (!VerifReponse("Deltas:", Message))
            {
                ErreurLog("Pas de réponse de l'antenne à la requête D.\n\n");

                return false;
            }

            return true;
        }

        if (commande == "C")
        {
            if (!VerifReponse("Corrections:", Message))
            {
                ErreurLog("Pas de réponse de l'antenne à la requête C.\n\n");

                return false;
            }

            return true;
        }

        if (commande == "M")
        {
            if (!VerifReponse("Calcul des matrices de correction", Message))
            {
                ErreurLog("Pas de réponse de l'antenne à la requête M.\n\n");

                return false;
            }

            return true;
        }

        if (commande == "O")
        {
            if (!VerifReponse("Optimisation de la geometrie de l antenne", Message))
            {
                ErreurLog("Pas de réponse de l'antenne à la requête O.\n\n");

                return false;
            }

            return true;
        }

        if (!VerifReponse("name=\"COMMAND_SET\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerCommande : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    if (Message)
    {
        if (*Message != "") AfficherLog(*Message +"\n\n", true);
    }

    return true;
}



/**************************************************************************************
** Place les antennes en position de repos
**
***************************************************************************************/

bool BAOcontrol::Park()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"\" >";
        *client_socket << "<oneSwitch name=\"PARK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("PARK OK", NULL))
        {
            ErreurLog("La commande PARK a échoué.\n\n");;

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien reçu la commande PARK.\n\n", true);

    return true;
}



/**************************************************************************************
** Annule le mouvement en cours
**
***************************************************************************************/

bool BAOcontrol::Abort()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ABORT_MOTION\" >";
        *client_socket << "<oneSwitch name=\"ABORT\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("ABORT OK", NULL))
        {
            ErreurLog("L'annulation a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien reçu la commande ABORT.\n\n", true);

    return true;
}


/**************************************************************************************
** Alignement de l'antenne ip
**
***************************************************************************************/

bool BAOcontrol::SelectionIP(string ip)
{
    try
    {
        *client_socket << "<newTextVector device=\"BAO\" name=\"ALIGNEMENT_IP\">";
        *client_socket << "<oneText name=\"IP\">";
        *client_socket << ip;
        *client_socket << "</oneText>";
        *client_socket << "</newTextVector>";


        if (!VerifReponse("est maintenant prete pour l alignement", NULL))
        {
            ErreurLog("La sélection de l'antenne située à l'adresse ip a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien sélectionné l'antenne situé à l'adresse ip pour l'alignement.\n\n", true);

    return true;
}


/**************************************************************************************
** Augmenter ou diminuer le delta en ascension droite ou en déclinaison
**
***************************************************************************************/

bool BAOcontrol::AlignementDelta(string delta, bool azimut)
{
    string Message = "";

    try
    {
        if (azimut)
        {
            if (delta == "-1")
            {   *client_socket << "<newSwitchVector device=\"BAO\" name=\"AZ\" >";
                *client_socket << "<oneSwitch name=\"AZM\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";
            }
            else
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"AZ\" >";
                *client_socket << "<oneSwitch name=\"AZP\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";
            }

            if (!VerifReponse("Delta", &Message))
            {
                ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (delta == "-1")
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALT_N\" >";
                *client_socket << "<oneSwitch name=\"ALTN\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";

                if (!VerifReponse("Delta", &Message))
                {
                    ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                    return false;
                }
            }
            else
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALT_P\" >";
                *client_socket << "<oneSwitch name=\"ALTP\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";

                if (!VerifReponse("Delta", &Message))
                {
                    ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                    return false;
                }
            }
        }

        if (Message != "") AfficherLog(Message + "\n\n", true);
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

//   AfficherLog("Le pilote indi_BAO a bien sélectionné l'antenne situé à l'adresse ip pour l'alignement.\n\n", true);

    return true;
}


/**************************************************************************************
** Validation de l'alignement sur l'étoile visée actuellement
**
***************************************************************************************/

bool BAOcontrol::ValidationAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_OK\" >";
        *client_socket << "<oneSwitch name=\"OK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("alignement ont ete valides", NULL))
        {
            ErreurLog("L'alignement sur l'étoile n'a pu être validé.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("L'alignement sur l'étoile a bien été validé.\n\n", true);

    return true;
}


/**************************************************************************************
** Sauvegarde de l'alignement
**
***************************************************************************************/

bool BAOcontrol::SauvegardeAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_OK\" >";
        *client_socket << "<oneSwitch name=\"SAUV\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("Les parametres de l alignement ont ete sauvegardes", NULL))
        {
            ErreurLog("La sauvegarde des paramètres de l'alignement a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les paramètres de l'alignemeent ont bien été sauvegardés.\n\n", true);

    return true;
}



/**************************************************************************************
** Sauvegarde de l'alignement
**
***************************************************************************************/

bool BAOcontrol::ResetAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_RESET\" >";
        *client_socket << "<oneSwitch name=\"RESET\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("reinitialises", NULL))
        {
            ErreurLog("Impossible de réinitialiser les paramètres d'alignement de l'antenne.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les paramètres d'alignement de l'antenne ont été réinitialisés.\n\n", true);

    return true;
}


/**************************************************************************************
** Sélection vitesse alignement
**
***************************************************************************************/

bool BAOcontrol::VitesseAlignement(int vit)
{
    string Message = "";

    try
    {
        if (vit == 1)
        {
            *client_socket << "<newSwitchVector device=\"BAO\" name=\"RAQ\" >";
            *client_socket << "<oneSwitch name=\"RAQ1\">";
            *client_socket << "On";
            *client_socket << "</oneSwitch>";
            *client_socket << "</newSwitchVector>";

            if (!VerifReponse("La vitesse de la raquette est fixee a 1x", &Message))
            {
                ErreurLog("Impossible de modifier la vitesse de la raquette.\n\n");

                return false;
            }
        }
        else
        {
            *client_socket << "<newSwitchVector device=\"BAO\" name=\"RAQ\" >";
            *client_socket << "<oneSwitch name=\"RAQ10\">";
            *client_socket << "On";
            *client_socket << "</oneSwitch>";
            *client_socket << "</newSwitchVector>";

            if (!VerifReponse("La vitesse de la raquette est fixee a 10x", &Message))
            {
                ErreurLog("Impossible de modifier la vitesse de la raquette.\n\n");

                return false;
            }
        }



    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    if (Message != "") AfficherLog(Message +"\n\n", true);

    return true;
}

/**************************************************************************************
** Dirige l'antenne vers les coordonnées ar et dec et suit l'objet  en activant le mode
** transit ou tracking.
**
** si J2000 == true, cela signifie que les coordonnées ar et dec sont données dans le
** le système de coordonnées J2000 (écliptique et équinoxe du 1 janvier 2000 à 0 h TU)
** Des calculs supplémentaires (précession, nutation, aberration) sont alors réalisés
** pour ramener les coordonnées horaires à l'écliptique et l'équinoxe de la date de
** l'observation.
**
***************************************************************************************/

bool BAOcontrol::Goto(string ar, string dec, bool Transit, bool J2000)
{
    double arf, decf;
    float  ar1, ar2, ar3;
    float  dec1, dec2, dec3;


    // Conversion de l'AD et de la déclinaison en radians

    Decomposition(ar, 2, &ar1, &ar2, &ar3);

    Decomposition(dec, 0, &dec1, &dec2, &dec3);

    arf  = ( ar1  + ar2 / 60.0  + ar3 / 3600.0 ) * 15.0 * Pidiv180;

    decf = ( fabs(dec1) + dec2 / 60.0 + dec3 / 3600.0 ) * Pidiv180;

    if (dec[0] == '-') decf = -decf;

    // Calculs supplémentaires pour ramener les coordonnées horaires
    // à l'époque de l'observation

    if ( J2000 )
    {
        Precession(&arf, &decf);
        NutationEtoile(&arf, &decf);
        AberrationAnnuelle(&arf, &decf);
    }

    // on réécrit l'ar et la dec dans le format d'indi_BAO

    ar  = DHMS(arf * N180divPi, true);

    dec = DHMS(decf * N180divPi, false);

    // on en informe l'utilisateur

    AfficherLog("Coordonnées apparentes de l'objet :\n", true);

    AfficherLog("AD=" + ar  + "   Dec=" + dec +"\n", true);


    // On transmet les coordonnées au pilote BAO

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ON_COORD_SET\">";
        *client_socket << "<oneSwitch name=\"TRANSIT\">";
        (Transit) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";

        *client_socket << "<oneSwitch name=\"TRACKING\">";
        (Transit) ? *client_socket << "Off" : *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ON_COORD_SET\"", NULL))
        {
            ErreurLog("Le changement de mode TRANSIT/TRACKING a échoué.\n\n");

            return false;
        }

        *client_socket << "<newNumberVector device=\"BAO\" name=\"EQUATORIAL_EOD_COORD_REQUEST\">";
        *client_socket << "<oneNumber name=\"RA\">";
        *client_socket << ar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"DEC\">";
        *client_socket << dec;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"EQUATORIAL_EOD_COORD_REQUEST\"", NULL))
        {
            ErreurLog("Le transfert des coordonnées de l'objet a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les nouvelles coordonnées AD=" + ar + " Dec=" + dec, true);
    AfficherLog(" ont été envoyées au pilote indi_BAO.\n\n", true);

    return true;
}



/**************************************************************************************
** Se connecte ou se déconnecte au pilote indi_BAO
**
***************************************************************************************/

bool BAOcontrol::Connect(bool connect)
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"CONNECTION\">";

        (connect) ? *client_socket << "<oneSwitch name=\"CONNECT\">" : *client_socket << "<oneSwitch name=\"DISCONNECT\">";

        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (connect)
        {
            if (!VerifReponse("BAORadio is online", NULL))
            {
                ErreurLog("La connexion a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (!VerifReponse("BAORadio is offline", NULL))
            {
                ErreurLog("La déconnexion a échoué.\n\n");

                return false;
            }
        }
    }

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    if (connect)
    {
        AfficherLog("La connexion a été établie avec le pilote indi_BAO.\n\n", true);

        // Au moment de la connexion avec le driver indi_BAO
        // On transmet les paramètres du lieu de l'observation, le mode de suivi
        // ainsi que la méthode d'alignement utilisée

        usleep(500000);

        EnvoyerCoordGeographiques();

        EnvoyerPressionTemperature();

        EnvoyerDelaisModesTransitEtTracking();

        EnvoyerMethodeAlignement();
    }
    else
    {
        AfficherLog("BAOcontrol s'est bien déconnecté du pilote indi_BAO.\n\n", true);
    }

    return true;
}


/**************************************************************************************
** Gestion du clavier en mode raw pendant la procédure d'alignement
** Cela permet de ne pas avoir à taper sur la touche 'entrée' après avoir appuyé sur
** une flèche de direction
**
***************************************************************************************/

void mode_raw(int activer)
{
    static struct termios cooked;
    static int raw_actif = 0;

    if (raw_actif == activer)
        return;

    if (activer)
    {
        struct termios raw;

        tcgetattr(STDIN_FILENO, &cooked);

        raw = cooked;
        cfmakeraw(&raw);
        tcsetattr(STDIN_FILENO, TCSANOW, &raw);
    }
    else
        tcsetattr(STDIN_FILENO, TCSANOW, &cooked);

    raw_actif = activer;
}


/**************************************************************************************
** Alignement de l'antenne située à l'adresse ip
**
** Le principe consiste à utiliser des étoiles comme point de repère pour la calibration
** des antennes : on compare la position calculée de chaque objet avec la position
** effectivement mesurée lors de la procédure d'alignement. On en tire  une matrice
** de rotation qui doit corriger les défauts d'alignement de l'antenne (le 0 pas codeur
** pas  parfaitement dirigé vers le sud et un axe de rotation az pas forcément
** perpendiculaire au sol...)
**
***************************************************************************************/

bool BAOcontrol::AlignementAntenneIP( string ip )
{

    stringstream os;

    int i;

    ModificationAlignement=false;


    // on a identifié l'antenne

    if ( SelectionIP(ip) )
    {
        // L'alignement de l'antenne 'num' est en cours. Il ne faut pas calculer la
        // matrice de correction tant que ce n'est pas fini...

        AfficherLog("Début de l'alignement de l'antenne.\n", true);

        AfficherLog("\n\n\nListe des étoiles brillantes situées à plus de 30° au-dessus de l'horizon :\n\n", true);

        // Mise à jour  du temps sidéral local

        UpdateTime();

        // recherche les étoiles situées à une hauteur supérieure à 30 °
        // (et donc accessibles à l'antenne)

        for (i=0; i<numEtoiles; i++)
        {
            Azimut(Etoiles[i].ad, Etoiles[i].de, &Etoiles[i].az, &Etoiles[i].ha);

            // calcule la réfraction atmosphérique

            Etoiles[i].ha = RefractionAtmospherique(Etoiles[i].ha);

            // Sélectionne et affiche les étoiles > 30°
            // et donc la magnitude < MAGNITUDEMAXETOILESCALIBRATION

            if ( Etoiles[i].selectionnee = ((Etoiles[i].ha > (HAUTMIN + 5.0) * Pidiv180) && (Etoiles[i].mag < MAGNITUDEMAXETOILESCALIBRATION)) )
            {
                os.width(20);
                os << Etoiles[i].nom;
                os << "\t(" << Etoiles[i].cons << ")\taz=" << Etoiles[i].az * N180divPi << "°\tha=";
                os << Etoiles[i].ha * N180divPi << "°\tmagnitude=" << Etoiles[i].mag << "\n";

                AfficherLog(&os, true);
            }

            //TODO à réactiver
            /* if (Etoiles[i].selectionnee)
             {
                 for (int j=0; j < Antennes[num].AlignementAntenne->nbrcorrections; j++)
                 {
                     // Là, un point important: on ne prend pas en compte l'AD des objets dans le reste des calculs
                     // Mais l'angle horaire de ces objets avec le méridien. Voir les détails dans la documentation....

                     double distances = DistanceAngulaireEntre2Points(
                                            VerifAngle(Antennes[num].AlignementAntenne->ad[j] - Antennes[num].AlignementAntenne->tsl[j]),
                                            Antennes[num].AlignementAntenne->de[j],
                                            VerifAngle(Etoiles[i].ad - GetTSL()),
                                            Etoiles[i].de);

                     // Ici, on ne sélectionne que des étoiles qui ne sont pas trop proches de positions ayant déjà servies à la calibration
                     // de l'antenne et qui multiplierait les mesures sans amélioration de la précision

                     if ( distances < MIN_DISTANCE_ALIGNEMENT ) Etoiles[i].selectionnee = false;
                 }
             }*/
        }
    }
    else
    {
        ErreurLog("L'antenne située à l'adresse ip indiquée n'existe pas\n\n");

        return false;
    }

    return true;
}





/**************************************************************************************
** Lance l'exécution d'un fichier de mouvements
**
***************************************************************************************/

bool BAOcontrol::Run(string fichier)
{
    stringstream os;

    bool exec = LectureFichierMouvements(fichier);

    UpdateTime();

    if ( exec && numobjets > 1 )
    {
        // Si le chargement du fichier s'est bien déroulé et qu'il y a plus d'un mouvement dedans

        if (objets[numobjets-1].JJ + objets[numobjets-1].Duree / 3600.0 / 24.0 < GetJJ())
        {
            ErreurLog("Le fichier contient des dates qui sont toutes périmées. Exécution annulée.\n\n");

            NoExit = !exitrun;

            return false;
        }
        else
        {
            AfficherLog("Exécution du fichier " + fichier + " en attente...\n\n", true);
            run = true;
        }
    }
    else
    {
        ErreurLog("\nLe fichier " + fichier + " contient des erreurs ou n'existe pas...\n\n\n");

        NoExit=!exitrun;

        return false;
    }

    return true;
}




/**************************************************************************************
** identification des commandes entrées par l'utilisateur
** puis exécution des commandes
** Chaine peut contenir par ex "goto betelgeuse", "goto 12:23:01 +22:53:00", "status" etc...
**
***************************************************************************************/

string BAOcontrol::DecodageEntreesUtilisateur(string chaine)
{
    string ar, de, objet, fichier;
    stringstream os;

    // Permet de savoir si la dernière commande saisie par l'utilisateur
    // est une commande valide

    bool CommandeOK = false;


    // Commande goto

    if ( chaine.find("goto") != string::npos )
    {

        // L'utilisateur a fait suivre la commande goto de l'indication J2000

        if (J2000 = (chaine.find("j2000") != string::npos))
        {
            AfficherLog("Prise en compte de la précession, de la nutation et de l'aberration...\n", true);

            // On ne garde que la première partie de la chaine (avant le mot "J2000")

            chaine = chaine.substr(0, chaine.find("J2000"));
        }

        if (DecompositionCommande(chaine, "goto", &ar, &de) &&
                Decomposition(ar, 2, NULL, NULL, NULL) &&
                Decomposition(de, 0, NULL, NULL, NULL))
        {
            // L'utilisateur a indiqué deux coordonnées valides -> on exécute le goto

            CommandeOK = true;

            Goto(ar, de, Transit, J2000);
        }
        else
        {
            // L'utilisateur a indiqué un nom d'objet après le goto -> on lance la recherche sur internet
            // puis on envoie les coordonnées de l'objet au pilote indi_BAO

            CommandeOK = true;

            if ((DecompositionCommande(chaine, "goto", &objet, NULL)) && (chaine.find(":") == string::npos))
            {
                CoordonneesHoraires reponse;

                if (objet != "sun")
                {
                    bool CestUneEtoile = false;

                    for (int i=0; i<numEtoiles; i++)
                    {
                        if (objet == Etoiles[i].nom)
                        {
                            AfficherLog(Etoiles[i].nom + "\n", true);

                            Goto( DHMS(Etoiles[i].ad*N180divPi, true), DHMS(Etoiles[i].de*N180divPi, false), Transit, true);

                            CestUneEtoile = true;

                            break;
                        }
                    }

                    if (!CestUneEtoile)
                    {
                        reponse = ServeurNED(objet);

                        if (reponse.ar!="" && reponse.dec!="")  Goto(reponse.ar, reponse.dec, Transit, true);
                    }
                }
                else
                {
                    CoordonneesHorairesDouble pos_sun;

                    CalculARDecSoleil(&pos_sun);

                    printf("Goto Sun\n");

                    Goto(DHMS(pos_sun.ar * N180divPi, true), DHMS(pos_sun.dec * N180divPi, false), Transit, false);
                }
            }
            else
            {
                ErreurLog("Erreur goto : le format de la commande est goto AD DEC\n \
                 L'AD doit être de la forme xx:yy:zz  J2000 (ex: 12:00:03 pour 12h 0m 3s)\n \
                 La déclinaison doit être de la forme +/-xx:yy:zz (ex: -12:50:01.1 pour -12° 50' 01.1'')\n \
                 J2000 est un paramètre optionnel qui sert à indiquer que les coordonnées horaires\n \
                 sont exprimées dans le repère du même nom.\n\n \
                 Il est egalement possible de faire suivre la commande d'un nom d'un objet\n \
                 Exemple goto M 31 ou encore goto ngc145.\n \
                 Ne pas préciser dans ce cas J2000.\n\n");
            }
        }
    }



    // Commande goto

    if ( chaine.find("raq") != string::npos )
    {
        int Vitesse = 10;
        // L'utilisateur a indiqué un nom d'objet après le goto -> on lance la recherche sur internet
        // puis on envoie les coordonnées de l'objet au pilote indi_BAO

        CommandeOK = true;

        char c;

        do
        {

            cout << "Appuyez sur les touches Z,Q,D,X pour déplacer l'objet dans le viseur.\n";
            cout << "Appuyez sur S pour changer de vitesse et sur A pour envoyer un ABORT à l'antenne.\n";
            cout << "(A la fin de l'opération, appuyez sur ESC pour sortir de la raquette).\n\n";

            mode_raw(1);

            c = getchar();

            mode_raw(0);

            // pause de 10ms

            usleep(10000);

            // Gestion des touches du clavier. On modifie l'orientation de l'instrument
            // sur les axes AD et Dec...
            // On sauvegarde également le temps sidéral local. Ceci nous permettra de calculer
            // l'angle horaire de l'objet pointé


            switch (c)
            {
            case 'z' :
                AlignementDelta("1", false);
                break;

            case 'd' :
                AlignementDelta("1", true);
                break;

            case 'q' :
                AlignementDelta("-1", true);
                break;

            case 'x' :
                AlignementDelta("-1", false);
                break;

                // abort
            case 'a' :
                Abort();
                sleep(2);

                break;

                // vitesse 1x ou 10x
            case 's' :
                (Vitesse == 1 ) ? Vitesse = 10 : Vitesse = 1;
                VitesseAlignement(Vitesse);
                os << "Vitesse=" << Vitesse << "x\n\n";
                AfficherLog(&os,  true);

                break;
            }
            //touche ESC -> on sort
        } while (c!=27);
    }






// Annulation du suivi en cours

    if (chaine.find("abort") != string::npos)
    {
        CommandeOK = true;

        Abort();

        // Annulation de l'exécution d'un fichier de mouvements

        if (chaine.find("run") != string::npos)
        {
            run       = false;
            runnum    = 1;
            numobjets = 1;

            AfficherLog("Annulation de l'exécution du fichier de mouvements.\n\n", true);

            return "";
        }
    }

// Lance l'exécution d'un fichier de mouvements

    if (chaine.find("run") != string::npos)
    {
        CommandeOK = true;

        if (DecompositionCommande(chaine, "run", &fichier, NULL))
        {
            Run(fichier);
        }
        else
        {
            ErreurLog("Erreur run : la commande run doit être suivie par un nom de fichier.\n\n");
        }
    }

// se connecte/déconnecte au serveur

    if (chaine.find("connect") != string::npos)
    {
        CommandeOK = true;

        Connect(chaine.find("disconnect") == string::npos);
    }



// Place les antennes en position de repos

    if (chaine.find("park") != string::npos)
    {
        CommandeOK = true;

        Park();
    }

// interroge un serveur de temps pour mettre l'horloge du PC à jour

    if (chaine.find("updatetime") != string::npos)
    {
        CommandeOK = true;

        int rc = system("sntp -s fr.pool.ntp.org");

        if (WEXITSTATUS(rc) ==-1 || WEXITSTATUS(rc) ==127)
        {
            ErreurLog("La commande sntp n'est pas accessible. Merci de vérifier.\n\n");
        }
    }

// Affiche tous les paramètres du programme

    if (chaine.find("status") != string::npos)
    {
        CommandeOK = true;

        os << "Latitude = " << LatitudeChar << "\n";
        os << "Longitude = " << LongitudeChar << "\n\n";
        os << "Serveur Indi = " << Serveur << "\n";
        os << "Port du serveur Indi = " << Port << "\n\n";
        os << "Pression atmosphérique  = " << Pression << " mBar\n";
        os << "Température = " << Temperature << " °C\n\n";

        os << "Antennes connectées : " << numAntennes << "\n\n";
        os << "Mode ";
        (Transit) ? os << "transit" : os << "tracking";
        os << " activé.\n\n";
        os << "Durée entre deux actualisations : ";
        (Transit) ? os << delaitransit : os << delaitracking;
        os << " sec(s)\n\n";
        os << "Méthode d'alignement : ";
        switch (MethodeAlignement)
        {
        case SIMPLE :
            os << "simple\n";
            break;
        case AFFINE :
            os << "affine\n";
            break;
        case TAKI   :
            os << "taki\n";
            break;
        }
        os << "\n\n";

        AfficherLog(&os, true);
    }

// interroge le serveur NED sur internet -> retourne les coordonnées de l'objet

    if (chaine.find("search")!=string::npos)
    {
        string objet;

        bool CestUneEtoile = false;

        CommandeOK = true;

        if (DecompositionCommande(chaine, "search", &objet, NULL))
        {
            for (int i=0; i<numEtoiles; i++)
            {
                if (objet == Etoiles[i].nom)
                {
                    os << Etoiles[i].nom << "\n";
                    os << "AD (J2000) = " << DHMS(Etoiles[i].ad*N180divPi, true) << "\n";
                    os << "Dec (J2000) = " << DHMS(Etoiles[i].de*N180divPi, false) << "\n";
                    AfficherLog(os.str(), true);

                    CestUneEtoile = true;

                    break;
                }
            }

            if (!CestUneEtoile) ServeurNED(objet);
        }
    }


// Envoyer une commande aux antennes

    if (chaine.find("send")!=string::npos)
    {
        string commande;

        CommandeOK = true;

        if (DecompositionCommande(chaine, "send", &commande, NULL))
        {
            switch (commande[0])
            {
            case 'g' :
                commande[0]='G';
                if ((commande[1]!='f' && commande[1]!='b') || (commande[6]!='f' && commande[6]!='b'))
                {
                    ErreurLog("Erreur ! La commande GOTO n'a pas un format correct !\n\n");
                    return "";
                }
                break;
            case 'a' :
                commande[0]='A';
                break;
            case 'z' :
                commande[0]='Z';
                break;
            case 'p' :
                commande[0]='P';
                break;
            case 'x' :
                commande[0]='X';
                break;
            case 'd' :
                commande[0]='D';
                break;
            case 'm' :
                commande[0]='M';
                break;
            case 'o' :
                commande[0]='O';
                break;
            }



            string Message;

            if (!EnvoyerCommande((string)commande, &Message))
            {
                ErreurLog("Erreur ! La commande n'a pas un format correct !\n\n");
            }
            else
            {
                AfficherLog(Message +"\n\n", true);

                return Message;
            }
        }
    }



// correction de l'alignement d'une antenne par rapport à l'objet visé

    if (chaine.find("align")!=string::npos)
    {
        string ip;

        CommandeOK = true;

        if (DecompositionCommande(chaine, "align", &ip, NULL))
        {
            AlignementAntenneIP(ip);
        }
        else
        {
            ErreurLog("Erreur align : la commande align doit être suivie par l'ip de l'antenne qui faut aligner.\n\n");
        }
    }


// Réinitialisation de l'alignement d'une antenne

    if (chaine.find("reset")!=string::npos)
    {
        string ip;

        CommandeOK = true;

        if (DecompositionCommande(chaine, "reset", &ip, NULL))
        {
            if (SelectionIP(ip))
            {
                ResetAlignement();
                return "";
            }

            ErreurLog("Erreur reset : la commande reset doit être suivie par l'ip de l'antenne qu'il faut aligner.\n\n");
        }
    }


// Commande set

    if (chaine.find("set") != string::npos)
    {
        string commande, commande2, commande3;

        // est-ce que la commande commande par set ! -> si oui, la variable commande contient alors le reste de
        // la phrase saisie par l'utilisateur

        if (DecompositionCommande(chaine, "set", &commande, NULL))
        {
            // A-t-on set transit xxxx ?

            if (DecompositionCommande(commande, "transit", &commande2, &commande3))
            {
                //at-ton set transit delay xxx ?

                if (commande2 == "delay")
                {
                    if (atoi(commande3.c_str()) > 0)
                    {
                        CommandeOK = true;

                        delaitransit = atoi(commande3.c_str());

                        EnvoyerDelaisModesTransitEtTracking();

                        AfficherLog("Delai entre deux actualisations en mode transit : " + commande3 + "\n\n" , true);
                    }
                    else
                    {
                        AfficherLog("Le nombre qui suit le mot clé delay doit être entier et > 0  \n\n" , true);
                    }
                }

                // Active le mode de suivi "transit"

                if (commande2 == "on")
                {
                    CommandeOK = true;

                    AfficherLog("Mode transit activé.\n\n", true);

                    Transit = true;
                }
            }

            // a-t-on set tracking xxxx ?

            if (DecompositionCommande(commande, "tracking", &commande2,  &commande3))
            {
                // a-ton set tracking delay xxx ?

                if (commande2 == "delay")
                {
                    if (atoi(commande3.c_str())>0)
                    {
                        CommandeOK = true;

                        delaitracking = atoi(commande3.c_str());

                        EnvoyerDelaisModesTransitEtTracking();

                        AfficherLog("Delai entre deux actualisations en mode tracking : " + commande3 + "\n\n" , true);
                    }
                    else
                    {
                        AfficherLog("Le nombre qui suit le mot clé delay doit être entier et > 0  \n\n" , true);
                    }
                }

                // Active le mode de suivi "transit"

                if (commande2 == "on")
                {
                    CommandeOK = true;

                    AfficherLog("Mode tracking activé.\n\n", true);

                    Transit = false;
                }
            }

            // A-t-on set alignment ?

            if (DecompositionCommande(commande, "alignment", &commande2,  &commande3))
            {

                // A-ton set alignment method xxx ?

                if (commande2 == "method")
                {
                    MethodeAlignement = NONE;

                    if (commande3 == "simple") MethodeAlignement = SIMPLE;

                    if (commande3 == "affine") MethodeAlignement = AFFINE;

                    if (commande3 == "taki")   MethodeAlignement = TAKI;

                    if (MethodeAlignement == NONE )
                    {
                        MethodeAlignement = SIMPLE;

                        AfficherLog("Erreur: la syntaxe doit être 'set alignment method x' avec x = simple/affine/taki)\n\n" , true);
                    }
                    else
                    {
                        CommandeOK = true;

                        EnvoyerMethodeAlignement();

                        switch (MethodeAlignement)
                        {
                        case  SIMPLE :
                            AfficherLog("Activation de la méthode d'alignement simple\n\n", true);
                            break;
                        case  AFFINE :
                            AfficherLog("Activation de la méthode d'alignement affine\n\n", true);
                            break;
                        case  TAKI :
                            AfficherLog("Activation de la méthode d'alignement taki\n\n", true);
                            break;
                        }
                    }
                }
                else
                {
                    AfficherLog("Erreur: la syntaxe doit être 'set alignment method x' avec x = simple/affine/taki)\n\n" , true);
                }
            }
        }
    }






// Aide du programme

    if (chaine.find("help") != string::npos)
    {
        CommandeOK = true;

        cout << endl;

        cout << "connect :                Se connecte au pilote indi_BAO." << endl;
        cout << "disconnect :             Se déconnecte du pilote indi_BAO." << endl;
        cout << "goto AD Dec :            Pointe l'objet situé aux coordonnées apparentes AD Dec." << endl;
        cout << "                         AD doit être au format xx:yy:zz et dec au format +/-xx:yy:zz" << endl;
        cout << "                         exemple goto 12:10:05 -05:10:11 permettra de pointer l'objet" << endl;
        cout << "                         situé aux coordonnées AD=12h10m5s et dec=-5°10'11''" << endl;
        cout << "goto AD Dec J2000 :      Pointe l'objet situé aux coordonnées AD Dec dans le repère J2000." << endl;
        cout << "                         Avant de pointer l'objet, le programme calcule la précession, la nutation et l'aberration" << endl;
        cout << "                         pour ramener les coordonnées à l'écliptique et à l'équinoxe de l'observation." << endl;
        cout << "goto nom_objet :         Interroge la base NED pour trouver les coordonnées de l'objet puis le pointe." << endl;
        cout << "                         exemples: goto m 31, goto messier 1, goto ngc 175, goto ic 434 etc..." << endl;
        cout << "                         Dans ce mode, les coordonnées sont automatiquement rapportées" << endl;
        cout << "                         à l'écliptique et à l'équinoxe de l'observation." << endl;
        cout << "                         Aussi, merci de ne pas faire suivre les coordonnées de l'indication J2000."<< endl;
        cout << "                         Goto sun permet de suivre le soleil." << endl;
        cout << "search nom_objet :       Interroge le serveur NED sur internet et retourne les coordonnées de l'objet." << endl;
        cout << "align ip :               Lance la procédure d'alignement de l'antenne d'adresse ip." << endl;
        cout << "reset ip :               Réinitialise les paramètres d'alignement de l'antenne d'adresse ip." << endl;
        cout << "send comm :              Envoie la commande 'comm' aux antennes." << endl;
        cout << "                         exemples : send P, send A, send Gf1000f1000" << endl;
        cout << "set transit on :         Active le mode de suivi transit." << endl;
        cout << "set transit delay x :    Fixe la durée entre deux actualisations à x seconde(s) dans le mode transit" << endl;
        cout << "set tracking on :        Active le mode de suivi tracking." << endl;
        cout << "set tracking delay x :   Fixe la durée entre deux actualisations à x seconde(s) dans le mode tracking" << endl;
        cout << "set alignment method x : Aligne les antennes en utilisant la méthode x (simple/affine/taki)" << endl;
        cout << "status :                 Liste les paramètres du programme." << endl;
        cout << "run filename :           Exécute le fichier de mouvements filename." << endl;
        cout << "abort run :              Annule l'exécution du fichier de mouvements." << endl;
        cout << "abort :                  Annule le mouvement en cours." << endl;
        cout << "park :                   Place les antennes en position de repos." << endl;
        cout << "updatetime :             Synchronise l'horloge du PC avec un serveur de temps." << endl;
        cout << "exit :                   Sortir de BAOControl." << endl;

        cout << endl;
    }

// Bye !

    if (chaine.find("exit") != string::npos)
    {
        CommandeOK = true;

        if ( numAntennes > 0 ) Abort();

        usleep(500000);

        Connect(false);

        NoExit = false;

        //  pthread_join(th1 ,NULL);
    }

// La commande n'a pas été identifiée !

    if (!CommandeOK) ErreurLog("Commande inconnue... Tapez help pour obtenir la liste des commandes disponibles.\n\n");

    return "";
}



/**************************************************************************************
** Interroge le serveur NED sur internet et retourne les coordonnées dans le repère J2000
** Si pas de réponse, alors message d'erreur et les coordonnées retournées sont
** égales à AD="" dec=""
**
***************************************************************************************/

CoordonneesHoraires BAOcontrol::ServeurNED(string objet)
{
    CoordonneesHoraires reponse;
    stringstream send;

    FILE *pFile = NULL;
    int rc;

    //initialisation des variables de retour

    reponse.ar = "";
    reponse.dec = "";

    // le nom de l'objet ne doit pas contenir des espace pour la requête NED
    // on les remplace par des '+'

    for (uint i=0; i<objet.length(); i++) if (objet[i] == ' ') objet[i]='+';

    // Effacer le fichier result s'il existe dans le répertoire du programme

    if ((pFile = fopen ("result_recherche_bao", "r")) != NULL )
    {
        fclose (pFile);
        rc = system( "rm -rf result_recherche_bao");
    }

    // On utilise l'utilitaire curl pour passer la requête au serveur NED
    // On sauvegarde le résultat dans le fichier result

    send << "curl >> result_recherche_bao \"http://ned.ipac.caltech.edu/cgi-bin/nph-objsearch?objname=";
    send << objet;
    send << "&extend=no&hconst=73&omegam=0.27&omegav=0.73&corr_z=1&out_csys=Equatorial&out_equinox=J2000.0";
    send << "&obj_sort=RA+or+Longitude&of=ascii_bar&zv_breaker=30000.0&list_limit=5&img_stamp=NO\"";

    AfficherLog("Envoi de la requête au serveur NED...\n", true);

    // on lance la requête

    rc = system(send.str().c_str() );

    // traitement du résultat

    if (WEXITSTATUS(rc) ==-1 || WEXITSTATUS(rc) ==127)
    {
        ErreurLog("La commande curl n'est pas accessible. Merci de vérifier.\n\n");
    }
    else
    {
        // On a reçu quelque chose

        int nbRead = 0;

        char *pBuf;

        pBuf = new char [100000];

        // On laisse un peu de temps au serveur pour répondre.
        // C'est pas vraiment indispensable mais bon...

        sleep(1);

        // On ouvre le fichier result

        pFile = fopen ("result_recherche_bao", "r");

        if (pFile)
        {
            nbRead = readline (pFile, pBuf);

            // Si le fichier n'est pas vide
            if (nbRead!=-1)
            {
                if (strstr(pBuf, "Error") != NULL)
                {
                    // Le fichier contient une indication d'erreur -> pas de réponse exploitable

                    AfficherLog("\nLe serveur NED n'a pas trouvé de réponse à votre demande...\n", true);
                }
                else
                {
                    // on analyse la réponse

                    AfficherLog("\nRéponse du serveur :\n", true);

                    do
                    {
                        // on cherche la première suggestion de NED
                        // Il faudra peut-être un peu complexifier par la suite

                        if (pBuf[0] == '1')
                        {
                            string chaine = (string)pBuf;

                            chaine=chaine.substr(chaine.find('|')+1);

                            // affiche le nom de l'objet dans le serveur NED

                            AfficherLog(chaine.substr(0, chaine.find('|')) + "\n", true);

                            chaine = chaine.substr(chaine.find('|') + 1);

                            string chaine2 = chaine.substr(0, chaine.find('|'));

                            double ar = atof(chaine2.c_str());

                            // affiche l'ascension droite

                            AfficherLog("AD (J2000) = " + DHMS(ar, true) + "\n", true);

                            reponse.ar = DHMS(ar, true);

                            chaine=chaine.substr(chaine.find('|')+1);

                            chaine2 = chaine.substr(0, chaine.find('|'));

                            double dec = atof(chaine2.c_str());

                            // on affiche la déclinaison

                            AfficherLog("Dec (J2000) = " + DHMS(dec, false)+"\n", true);

                            reponse.dec = DHMS(dec, false);
                        }

                        nbRead = readline (pFile, pBuf);

                    } while (nbRead!=-1);
                }
            }
            else
            {
                ErreurLog("\nLa connexion semble impossible avec le serveur NED !\n\n");
            }

            fclose(pFile);
        }
        else
        {
            ErreurLog("\nLa connexion semble impossible avec le serveur NED !\n\n");
        }

        delete [] pBuf;

        AfficherLog("\n", true);
    }

    return reponse;
}



/**************************************************************************************
** Lecture d'un fichier ligne après ligne
** retourne -1 à la fin
**
***************************************************************************************/

int BAOcontrol::readline (FILE * pfile, char *tab)
{
    int nbchar = 0;

    char c;

    while ((c = getc (pfile)) != '\n')
    {
        if (c == EOF)
        {
            break;
        }
        tab[nbchar++] = c;
    }
    tab[nbchar] = '\0';

    while (nbchar>0 && tab[nbchar-1]==' ')
    {
        tab[--nbchar] = '\0';
    }

    while (tab[0]==' ') for (int i=1; i<=nbchar; i++) tab[i-1] = tab[i];

    if (c == EOF) nbchar=-1;

    return (nbchar);
}



/**************************************************************************************
** chargement des paramètres contenus dans le fichier params
**
***************************************************************************************/

bool BAOcontrol::ChargementParametres(string fileName)
{
    string section;
    string key;
    char * value;
    stringstream os;


    // Si le fichier n'existe pas -> on sort

    if (!is_readable(fileName)) return false;


    AfficherLog("Lecture du fichier de configuration 'params' :\n\n", true);


    /////////////////////////////////////////
    // Rubrique Coordonnées géographiques

    section = "coordonnees geographiques";

    key = "latitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 0, NULL, NULL, NULL)))
    {
        LatitudeChar = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("latitude = " + LatitudeChar +"\n", true);
    }
    else
    {
        ErreurLog("La latitude est incorrecte !\n");

        return false;
    }


    key = "longitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 1, NULL, NULL, NULL)))
    {
        LongitudeChar = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("longitude = " + LongitudeChar +"\n\n", true);
    }
    else
    {
        ErreurLog("La longitude est incorrecte !\n");

        return false;
    }


    /////////////////////////////////////////
    // rubrique connexion

    section = "connexion indi";

    key = "serveur";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Serveur = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("serveur = " + Serveur +"\n", true);
    }
    else
    {
        ErreurLog("Nom du serveur invalide !\n");

        return false;
    }

    /////////////////////////////////////////
    // Port de connexion du serveur Indi

    key = "port";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Port = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("port = " + Port +"\n\n", true);
    }
    else
    {
        ErreurLog("Numéro de port incorrect !\n");

        return false;
    }


    /////////////////////////////////////////
    // Rubrique paramètres de l'atmosphère

    section = "atmosphere";

    key = "pression";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Pression = atof(value);
        SAFEDELETE_TAB(value);
        os << "pression = " << Pression << endl;
        AfficherLog(&os, true);
    }
    else
    {
        os << "La pression atmosphérique est incorrecte !" << endl;
        ErreurLog(&os);

        return false;
    }

    key = "temperature";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Temperature = atof(value);
        SAFEDELETE_TAB(value);
        os << "température = " << Temperature << endl << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("La température est incorrecte !\n");

        return false;
    }

    /////////////////////////////////////////
    // Rubrique alignement

    section = "alignement";

    key = "methode_alignement";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        AfficherLog("methode d'alignement = " + (string)value + "\n", true);
        switch (value[0])
        {
        case 's' :
        case 'S' :
            MethodeAlignement = SIMPLE;
            break;

        case 'a' :
        case 'A' :
            MethodeAlignement = AFFINE;
            break;

        case 't' :
        case 'T' :
            MethodeAlignement = TAKI;
            break;
        }

        SAFEDELETE_TAB(value);
    }
    else
    {
        ErreurLog("Le paramètre methode_alignement est incorrect !\n");

        return false;
    }



    /////////////////////////////////////////
    // Rubrique suivi

    section = "suivi";

    key = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = (strstr(value, "transit") != NULL);
        AfficherLog("mode suivi = " + (string)value + "\n", true);
        SAFEDELETE_TAB(value);
    }
    else
    {
        ErreurLog("Le paramètre mode suivi est incorrect !\n");

        return false;
    }



    key = "delai_transit";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitransit = atoi(value);
        SAFEDELETE_TAB(value);
        os << "delai transit = " << delaitransit << " sec" << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("Le paramètre delai_transit est incorrect !\n");

        return false;
    }


    key = "delai_tracking";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitracking = atoi(value);
        SAFEDELETE_TAB(value);
        os << "delai tracking = " << delaitracking << " sec" << endl << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("Le paramètre delai_tracking est incorrect !\n");

        return false;
    }


    /////////////////////////////////////////
    // Rubrique divers

    section = "divers";

    key = "couleurs";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        ChoixCouleurs=atoi(value);
        SAFEDELETE_TAB(value);
    }
    else
    {
        /*os << "Le paramètre couleurs est incorrect !" << endl;
        ErreurLog(os.str());
        return false;*/
    }

    return true;
}






/**************************************************************************************
** Lecture des mouvements planifiés contenus dans le fichier fileName
** Ces fichiers de mouvement planifiés permettent des observations automatisées
** d'une liste d'objets (ou zone du ciel) pour lesquels on passe de l'un à l'autre
** en fonction de l'heure
**
***************************************************************************************/

bool BAOcontrol::LectureFichierMouvements(string fileName)
{
    string section;
    string key;
    char * value   = NULL;

    stringstream os;
    float a1, a2, a3;


    // Si le fichier n'existe pas -> on sort

    if (!is_readable( fileName )) return false;

    AfficherLog("Lecture du fichier " + fileName +"\n\n", true);

    // initialisation du compteur nombre objets

    numobjets = 1;

    //Lecture de la rubrique Paramètres

    section = "parametres";

    // Mode transit ou tracking ?

    key     = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = ( strstr(value, "transit") != NULL);
        // il faut détruire les tableaux de caratères qui sont crées
        // dans la fonction ReadINI
        SAFEDELETE_TAB(value);

        if (Transit) AfficherLog("Mode transit activé.\n", true);
        else AfficherLog("Mode tracking activé.\n", true);
    }
    else
    {
        ErreurLog("Le parametre mode est incorrect.\n");

        return false;
    }

    // Les coordonnées sont-elles dans le système de coordonnées J2000 ?

    key = "J2000";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        J2000 = ( strstr(value, "on") != NULL);

        SAFEDELETE_TAB(value);

        if (J2000) AfficherLog("Mode J2000 activé.\n\n", true);
        else AfficherLog("Mode J2000 désactivé.\n\n", true);
    }
    else
    {
        ErreurLog("Le parametre J2000 est incorrect.\n");

        return false;
    }

    // Lecture de la liste des observations

    section = "objet 1";
    key     = "date";


    while ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (numobjets<MAXOBJETS))
    {
        os << "object n°" << numobjets << endl;
        AfficherLog(&os, true);

        SAFEDELETE_TAB(value);


        // Lecture de la date et de l'heure de l'observation

        key = "date";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 3, &a1, &a2, &a3)))
        {
            AfficherLog("Date du début de l'observation : le " + (string)value, true);
            SAFEDELETE_TAB(value);

            objets[numobjets].JJ = CalculJJ(a3, a2, a1, 0.0);
        }
        else
        {
            os << "La date de la rubrique [objet " << numobjets << "] est incorrecte !" << endl;
            ErreurLog(&os);

            return false;
        }

        key = "heure";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 2, &a1, &a2, &a3)))
        {
            AfficherLog(" à " + (string)value, true) ;
            SAFEDELETE_TAB(value);


            objets[numobjets].JJ += ( a1 + a2 / 60.0 + a3 / 3600.0 ) / 24.0;

            os << " TU  -> JJ=" << fixed << objets[numobjets].JJ << scientific << endl;

            AfficherLog(&os, true);

            for (int i=1; i<numobjets; i++)
            {
                if ( objets[i].JJ == objets[numobjets].JJ )
                {
                    os << "Attention, les observations des objets " << i << " et ";
                    os << numobjets << " sont définies à la même date !" << endl;
                    ErreurLog(&os);
                }

                if ( objets[i].JJ + objets[i].Duree / 3600 / 24.0 > objets[numobjets].JJ )
                {
                    os << "Attention, les observations des objets " << i << " et " ;
                    os << numobjets << " se chevauchent !" << endl;
                    ErreurLog(&os);
                }
            }
        }
        else
        {
            os << "L'heure de la rubrique [objet " << numobjets << "] est incorrecte !" << endl;
            ErreurLog(&os);

            return false;
        }


        // Lecture de la durée de l'observation

        key = "duree";

        objets[numobjets].Duree=0.0;

        if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
        {
            AfficherLog("Durée de l'observation : " + (string)value + " secondes\n", true);
            objets[numobjets].Duree=atof(value);
            SAFEDELETE_TAB(value);
        }
        else
        {
            os << "La duree d'observation de la rubrique [objet " << numobjets << "] est incorrecte !\n";
            ErreurLog(&os);

            return false;
        }

        // Lecture de l'ascension droite de l'objet

        key = "ad";

        objets[numobjets].ad = "";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 2, NULL, NULL, NULL)))
        {
            AfficherLog("Ascension droite de l'objet : " + (string)value + "\n", true);
            objets[numobjets].ad = value;
            SAFEDELETE_TAB(value);
        }
        else
        {
            os << "L'ascension droite de la rubrique [objet " << numobjets << "] est incorrecte !";
            os << endl;
            ErreurLog(&os);

            return false;
        }

        // Lecture de la déclinaison de l'objet

        key = "de";

        objets[numobjets].de = "";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 0, NULL, NULL, NULL)))
        {
            AfficherLog("Déclinaison de l'objet : " + (string)value + "\n\n", true);
            objets[numobjets].de = value;
            SAFEDELETE_TAB(value);
        }
        else
        {
            os << "La déclinaison de la rubrique [objet " << numobjets << "] est incorrecte !";
            os << endl;
            ErreurLog(&os);

            return false;
        }

        objets[numobjets].exec = false;

        // On passe à l'objet suivant

        numobjets++;

        os << "objet "<< numobjets;

        section = os.str();

        os.str("");
    }

    SAFEDELETE_TAB(value);

    return true;
}


/**************************************************************************************
** chargement du catalogue d'étoiles nécessaire à la procédure d'alignement
**
***************************************************************************************/

bool BAOcontrol::ChargementCatalogueEtoiles(string fileName)
{
    string section;
    string key;
    char * value   = NULL;

    stringstream os;
    float a1, a2, a3;

    // Si le fichier n'existe pas -> on sort !

    if (!is_readable(fileName)) return false;

    AfficherLog("Chargement du catalogue d'étoiles : ", true);

    UpdateTime();

    numEtoiles = 0;

    // Lecture des étoiles

    section = "star 0";
    key     = "name";

    while (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        // Nom de l'étoile

        Etoiles[numEtoiles].nom = value;
        SAFEDELETE_TAB(value);

        // On convertit le nom des étoiles en minuscules

        for (uint i=0; i<strlen(Etoiles[numEtoiles].nom.c_str()); i++) Etoiles[numEtoiles].nom[i] = tolower(Etoiles[numEtoiles].nom[i]);


        //Constellation

        key = "const";

        readINI(section.c_str(), key.c_str(), &value, fileName.c_str());

        Etoiles[numEtoiles].cons = value;
        SAFEDELETE_TAB(value);


        // Lecture des coordonnées de l'étoile

        key = "ar";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 2, &a1, &a2, &a3)))
        {
            SAFEDELETE_TAB(value);

            Etoiles[numEtoiles].ad = (a1 + a2 / 60.0 + a3 / 3600.0) * 15.0 * Pidiv180;
        }
        else
        {
            os << "Les coordonnées AR de l'étoile " << numEtoiles << " sont incorrectes !" << endl;
            ErreurLog(&os);

            return false;
        }

        key = "de";

        if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
                && (Decomposition(value, 0, &a1, &a2, &a3)))
        {
            Etoiles[numEtoiles].de = (fabs(a1) + a2 / 60.0 + a3 / 3600.0 ) * Pidiv180;

            if (value[0] == '-')  Etoiles[numEtoiles].de = - Etoiles[numEtoiles].de;

            SAFEDELETE_TAB(value);
        }
        else
        {
            os << "Les coordonnées DE de l'étoile " << numEtoiles << " sont incorrectes !" << endl;
            ErreurLog(&os);

            return false;
        }


        // Lecture de la durée de l'observation

        key = "mag";

        if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
        {
            Etoiles[numEtoiles].mag = atof(value);
            SAFEDELETE_TAB(value);
        }
        else
        {
            os << "La magnitude de l'étoile " << numEtoiles << " est incorrecte !\n";
            ErreurLog(&os);

            return false;
        }


        // Passage du repère J2000 à l'écliptique et à l'équinoxe liée à la date de l'observation

        Precession        (&Etoiles[numEtoiles].ad, &Etoiles[numEtoiles].de);
        NutationEtoile    (&Etoiles[numEtoiles].ad, &Etoiles[numEtoiles].de);
        AberrationAnnuelle(&Etoiles[numEtoiles].ad, &Etoiles[numEtoiles].de);


        // On passe à l'objet suivant

        numEtoiles++;

        key     = "name";

        os << "star "<< numEtoiles;

        section = os.str();

        os.str("");
    }

    if (value) SAFEDELETE_TAB(value);

    os << numEtoiles-1 << " étoiles chargées.\n\n";

    AfficherLog(os.str(), true);

    return true;
}




/**************************************************************************************
** Routine principale
***************************************************************************************/

int BAOcontrol::init(int argc, char **argv)
{
    string chaine;


    // gestion au démarrage des paramètres en ligne de commande

    if ( argc >1 )
    {
        if (strstr(argv[1], "-h")!=NULL)
        {
            cout << "usage :\n \
       BAOControl -h ou --help : affiche l'aide\n \
       BAOControl -r FileName : exécute les mouvements définis dans le fichier FileName\n\n";

            return 0;
        }
    }

    if (argc == 1)  cout << "Tapez help pour obtenir la liste des commandes disponibles.\n" << endl << endl;


    // On lance le thread pour gérer la fenêtre graphique
    // la communication avec indi_BAO et le suivi des objets



    // si on a utilisé l'option -r en ligne de commande
    // alors on charge et on exécute le fichier de mouvements

    if (argc == 3)
    {
        if (strstr(argv[1], "-r")!=NULL)
        {
            exitrun=true;

            Run(argv[2]);

            // on attend ici la fin de l'exécution du fichier de mouvements

            do
            {

            } while (NoExit);
        }
        else
        {
            ErreurLog("Usage incorrect\n\n");
        }
    }
    else
    {
        // Saisie et décodage des commandes entrées par l'utilisateur
        /*
                do
                {
                    // Prompt
                    ChoixCouleurs==1 ? IDLog(blue2) : IDLog(blue1);

                    cout << '>';

                    // saisie
                    getline(cin, chaine);

                    ChoixCouleurs==1 ? IDLog(grey1) : IDLog(grey2);

                    AfficherLog(">" + chaine + "\n\n", false);

                    // on met en minuscules
                    for (int i=0; i<chaine.size(); ++i) chaine[i] = tolower(chaine[i]);

                    // on décode et exécute les commandes
                    DecodageEntreesUtilisateur(chaine);

                }
                while (NoExit);

                pthread_join(th1 ,NULL);*/
    }

    return 0;
}




/**************************************************************************************
** Le fichier file existe-t-il et n'est-il pas vide ?
**
**************************************************************************************/

bool BAOcontrol::is_readable( const string & file )
{
    ifstream fichier( file.c_str() );
    if (fichier.fail()) return false;

    // sauvegarder la position courante

    // long pos = fichier.tellg();

    // se placer en fin de fichier

    fichier.seekg( 0 , ios_base::end );

    // récupérer la nouvelle position = la taille du fichier

    long size = fichier.tellg();

    return (size > 0);
}
