////////////////////////////////
// Classe Alignement          //
// Franck RICHARD             //
// franckrichard033@gmail.com //
// BAORadio                   //
// février 2011               //
////////////////////////////////


#include "alignement.h"
//#include <QtCore/qcoreevent.h>

/**************************************************************************************
** Constructeur de la classe Alignement
**
***************************************************************************************/

Alignement::Alignement()
{
    //Initialisation des paramètres des antennes

    InitAlignement();


    // initialisation des paramètres géométriques de l'antenne

    InitParametresInstrument();
}


/**************************************************************************************
** Destructeur de la classe Alignement
**
***************************************************************************************/

Alignement::~Alignement()
{
}


/**************************************************************************************
** Initialisations des vecteurs et des paramètres d'alignement
**
***************************************************************************************/

void Alignement::InitAlignement()
{
    // On initialise les points de correction

    for (int i=0; i< MAXALIGNEMENTANTENNE; i++)
    {
        ad[i]                           = 0.0;
        de[i]                           = 0.0;
        delta_ad[i]                     = 0.0;
        delta_de[i]                     = 0.0;
        tsl[i]                          = 0.0;
        SelectionnePourCalculMatrice[i] = false;
    }

    // initialisation de la matrice de rotation (à la matrice identité)

    Identity();


    // initialisation de la direction en azimut de l'étoile polaire

    delta_az_polar    = 0;

    // Initialisation des deltas sur les axes az et ha
    // pour gérer la raquette hors procédure d'alignement

    deltaAZ           = 0;
    deltaHA           = 0;

    // Nbre de points (de mesures) actuellement disponibles pour
    // bâtir les matrices de correction

    nbrcorrections    = 0;

    // Est-ce que l'antenne est en cours d'alignement ?

    AlignementEnCours = 0;

    // La matrice de correction a-t-elle été calculée ?

    Matrice_ok        = false;

    // Méthode d'alignement par défaut

    MethodeAlignement = SIMPLE;
}



/**************************************************************************************
**  Initialisation des paramètres de la classe astro
**
***************************************************************************************/

void Alignement::TransmettreParametresClasseAstro(double Annee, double Mois, double Jour, double Heu, double Min, double Sec, double Longitude, double Latitude, double Pression, double Temp)
{
    DefinirLongitudeLatitude(Longitude, Latitude);
    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);
    DefinirPressionTemp(Pression, Temp);
}


/**************************************************************************************
**  1 0 0
**  0 1 0
**  0 0 1
***************************************************************************************/

void Alignement::Identity()
{
    MATRICE[1][1] = 1.0;
    MATRICE[2][1] = 0.0;
    MATRICE[3][1] = 0.0;

    MATRICE[1][2] = 0.0;
    MATRICE[2][2] = 1.0;
    MATRICE[3][2] = 0.0;

    MATRICE[1][3] = 0.0;
    MATRICE[2][3] = 0.0;
    MATRICE[3][3] = 1.0;
}

void Alignement::AfficherMatrice()
{
    stringstream os;

    os.str("");

    // Affiche de la matrice pour vérif

    os << "\nCalcul de la matrice de correction:\n";
    os.width(15);
    os << MATRICE[1][1];
    os.width(15);
    os << MATRICE[2][1];
    os.width(15);
    os << MATRICE[3][1];
    os << "\n";
    os.width(15);
    os << MATRICE[1][2];
    os.width(15);
    os << MATRICE[2][2];
    os.width(15);
    os << MATRICE[3][2];
    os << "\n";
    os.width(15);
    os << MATRICE[1][3];
    os.width(15);
    os << MATRICE[2][3];
    os.width(15);
    os << MATRICE[3][3];
    os << "\n\nDeterminant :" << Determinant() << "\n";

    AfficherLog(os.str(), true);
}



/**************************************************************************************
**  Calcule la matrice de rotation d'un angle alpha autour d'un axe défini
**  par les paramètres t et r dans un système de coordonnées spĥériques
**  Le signe - devant le cos permet seulement de rendre le repère compatible avec WinStars
** que j'utilise pour faire des vérifications...
** Fonction utile pour de débogage de la fct d'alignement
**
***************************************************************************************/

void Alignement::RotationAutourDunAxe(double t, double r, double alpha)
{
    double x = -cos(t) * cos(r);
    double y = sin(t) * cos(r);
    double z = sin(r);

    double c = cos(alpha);
    double s = sin(alpha);


    MATRICE[1][1] = x * x + (1.0 - x * x) * c;
    MATRICE[1][2] = x * y * (1.0 - c) - z * s;
    MATRICE[1][3] = x * z * (1.0 - c) + y * s;

    MATRICE[2][1] = x * y * (1.0 - c) + z * s;
    MATRICE[2][2] = y * y + (1.0 - y * y) * c;
    MATRICE[2][3] = y * z * (1.0 - c) - x * s;

    MATRICE[3][1] = x * z * (1.0 - c) - y * s;
    MATRICE[3][2] = y * z * (1.0 - c) + x * s;
    MATRICE[3][3] = z * z + (1.0 - z * z) * c;
}


/**************************************************************************************
** Matrice intermédiaire nécessaire au calcul de la matrice de correction
** dans le mode d'alignement TAKI
** Consultez la documentation pour comprendre le principe de la correction AFFINE/TAKI
**
***************************************************************************************/

void Alignement::Calculer_Matrice_LMN(Coord p1, Coord p2, Coord p3)
{
    double temp[4][4];
    double UnitVect[4][4];

    temp[1][1] = p2.x - p1.x;
    temp[2][1] = p3.x - p1.x;

    temp[1][2] = p2.y - p1.y;
    temp[2][2] = p3.y - p1.y;

    temp[1][3] = p2.z - p1.z;
    temp[2][3] = p3.z - p1.z;

    UnitVect[1][1] = (temp[1][2] * temp[2][3]) - (temp[1][3] * temp[2][2]);
    UnitVect[1][2] = (temp[1][3] * temp[2][1]) - (temp[1][1] * temp[2][3]);
    UnitVect[1][3] = (temp[1][1] * temp[2][2]) - (temp[1][2] * temp[2][1]);
    UnitVect[2][1] = pow(UnitVect[1][1], 2.0) + pow(UnitVect[1][2], 2.0) + pow (UnitVect[1][3], 2.0);
    UnitVect[2][2] = sqrt(UnitVect[2][1]);
    UnitVect[2][3] = 0.0;
    if (UnitVect[2][2] != 0.0) UnitVect[2][3] = 1.0 / UnitVect[2][2];

    for (int i=1; i<=2; i++) for (int j=1; j<=3; j++) GETLMN[i][j] = temp[i][j];

    GETLMN[3][1] = UnitVect[2][3] * UnitVect[1][1];
    GETLMN[3][2] = UnitVect[2][3] * UnitVect[1][2];
    GETLMN[3][3] = UnitVect[2][3] * UnitVect[1][3];
}



/**************************************************************************************
** Calcul de la matrice de correction dans le mode d'alignement TAKI
**
***************************************************************************************/

int Alignement::Calculer_Matrice_Taki(double x, double y, Coord aa1, Coord aa2, Coord aa3, Coord mm1, Coord mm2, Coord mm3)
{
    double Det;
    double EQLMN1[4][4];
    double EQLMN2[4][4];
    double EQMI_T[4][4];

    Coord a1, a2, a3, m1, m2, m3;

    AzHa2XY(aa1.x, aa1.y, &a1.x, &a1.y);
    AzHa2XY(aa2.x, aa2.y, &a2.x, &a2.y);
    AzHa2XY(aa3.x, aa3.y, &a3.x, &a3.y);

    AzHa2XY(mm1.x, mm1.y, &m1.x, &m1.y);
    AzHa2XY(mm2.x, mm2.y, &m2.x, &m2.y);
    AzHa2XY(mm3.x, mm3.y, &m3.x, &m3.y);

    //construit les matrices EQLMN1 & 2

    Calculer_Matrice_LMN(m1, m2, m3);

    for (int i=1; i<=3; i++) for (int j=1; j<=3; j++) EQLMN1[i][j]=GETLMN[i][j];

    Calculer_Matrice_LMN(a1, a2, a3);

    for (int i=1; i<=3; i++) for (int j=1; j<=3; j++) EQLMN2[i][j]=GETLMN[i][j];

    // Calcule le déterminant de EQLMN1

    Det = EQLMN1[1][1] * ((EQLMN1[2][2] * EQLMN1[3][3]) - (EQLMN1[3][2] * EQLMN1[2][ 3]));
    Det = Det - (EQLMN1[1][2] * ((EQLMN1[2][1] * EQLMN1[3][3]) - (EQLMN1[3][1] * EQLMN1[2][3])));
    Det = Det + (EQLMN1[1][3] * ((EQLMN1[2][1] * EQLMN1[3][2]) - (EQLMN1[3][1] * EQLMN1[2][2])));


    // Calcule la matrice inverse EQMI_T de EQLMN1

    if (Det == 0.0)
    {
        ErreurLog("Erreur Matrice_Taki. Impossible de calculer l'inverse de la matrice avec un determinant = 0 \n");
        return 0;
    }
    else
    {
        EQMI_T[1][1] = ((EQLMN1[2][2] * EQLMN1[3][3]) - (EQLMN1[3][2] * EQLMN1[2][3])) / Det;
        EQMI_T[1][2] = ((EQLMN1[1][3] * EQLMN1[3][2]) - (EQLMN1[1][2] * EQLMN1[3][3])) / Det;
        EQMI_T[1][3] = ((EQLMN1[1][2] * EQLMN1[2][3]) - (EQLMN1[2][2] * EQLMN1[1][3])) / Det;
        EQMI_T[2][1] = ((EQLMN1[2][3] * EQLMN1[3][1]) - (EQLMN1[3][3] * EQLMN1[2][1])) / Det;
        EQMI_T[2][2] = ((EQLMN1[1][1] * EQLMN1[3][3]) - (EQLMN1[3][1] * EQLMN1[1][3])) / Det;
        EQMI_T[2][3] = ((EQLMN1[1][3] * EQLMN1[2][1]) - (EQLMN1[2][3] * EQLMN1[1][1])) / Det;
        EQMI_T[3][1] = ((EQLMN1[2][1] * EQLMN1[3][2]) - (EQLMN1[3][1] * EQLMN1[2][2])) / Det;
        EQMI_T[3][2] = ((EQLMN1[1][2] * EQLMN1[3][1]) - (EQLMN1[3][2] * EQLMN1[1][1])) / Det;
        EQMI_T[3][3] = ((EQLMN1[1][1] * EQLMN1[2][2]) - (EQLMN1[2][1] * EQLMN1[1][2])) / Det;
    }


    // Effectue le produit des matrices EQMI_T et EQLMN2

    MATRICE[1][1] = (EQMI_T[1][1] * EQLMN2[1][1]) + (EQMI_T[1][2] * EQLMN2[2][1]) + (EQMI_T[1][3] * EQLMN2[3][1]);
    MATRICE[1][2] = (EQMI_T[1][1] * EQLMN2[1][2]) + (EQMI_T[1][2] * EQLMN2[2][2]) + (EQMI_T[1][3] * EQLMN2[3][2]);
    MATRICE[1][3] = (EQMI_T[1][1] * EQLMN2[1][3]) + (EQMI_T[1][2] * EQLMN2[2][3]) + (EQMI_T[1][3] * EQLMN2[3][3]);

    MATRICE[2][1] = (EQMI_T[2][1] * EQLMN2[1][1]) + (EQMI_T[2][2] * EQLMN2[2][1]) + (EQMI_T[2][3] * EQLMN2[3][1]);
    MATRICE[2][2] = (EQMI_T[2][1] * EQLMN2[1][2]) + (EQMI_T[2][2] * EQLMN2[2][2]) + (EQMI_T[2][3] * EQLMN2[3][2]);
    MATRICE[2][3] = (EQMI_T[2][1] * EQLMN2[1][3]) + (EQMI_T[2][2] * EQLMN2[2][3]) + (EQMI_T[2][3] * EQLMN2[3][3]);

    MATRICE[3][1] = (EQMI_T[3][1] * EQLMN2[1][1]) + (EQMI_T[3][2] * EQLMN2[2][1]) + (EQMI_T[3][3] * EQLMN2[3][1]);
    MATRICE[3][2] = (EQMI_T[3][1] * EQLMN2[1][2]) + (EQMI_T[3][2] * EQLMN2[2][2]) + (EQMI_T[3][3] * EQLMN2[3][2]);
    MATRICE[3][3] = (EQMI_T[3][1] * EQLMN2[1][3]) + (EQMI_T[3][2] * EQLMN2[2][3]) + (EQMI_T[3][3] * EQLMN2[3][3]);


    // Calcule l'offset sur le ciel

    VECT_OFFSET.x = m1.x - ((a1.x * MATRICE[1][1]) + (a1.y * MATRICE[2][1]) + (a1.z * MATRICE[3][1]));
    VECT_OFFSET.y = m1.y - ((a1.x * MATRICE[1][2]) + (a1.y * MATRICE[2][2]) + (a1.z * MATRICE[3][2]));
    VECT_OFFSET.z = m1.z - ((a1.x * MATRICE[1][3]) + (a1.y * MATRICE[2][3]) + (a1.z * MATRICE[3][3]));



    if (x + y == 0 )
    {
        return 0;
    }
    else
    {
        return  PointSitueDansSurfaceTriangle(x, y, aa1.x, aa1.y, aa2.x, aa2.y, aa3.x, aa3.y);
    }
}


void Alignement::AppliquerMatriceCorrectionTaki(Coord * result, Coord vect)
{
    // Applique la correction sur le vecteur orientation vect avec
    // vect.x = ascension droite
    // vect.y = déclinaison
    // vect.z = 1

    result->x = -VECT_OFFSET.x + ((vect.x * MATRICE[1][1]) + (vect.y * MATRICE[2][1]) + (vect.z * MATRICE[3][1]));
    result->y = -VECT_OFFSET.y + ((vect.x * MATRICE[1][2]) + (vect.y * MATRICE[2][2]) + (vect.z * MATRICE[3][2]));
    result->z = -VECT_OFFSET.z + ((vect.x * MATRICE[1][3]) + (vect.y * MATRICE[2][3]) + (vect.z * MATRICE[3][3]));
}




/**************************************************************************************
** Matrice intermédiaire nécessaire au calcul de la matrice de correction
** dans le mode d'alignement AFFINE
**
***************************************************************************************/

void Alignement::Calculer_Matrice_GETPQ(Coord p1, Coord p2, Coord p3)
{
    GETPQ[1][1] = p2.x - p1.x;
    GETPQ[2][1] = p3.x - p1.x;
    GETPQ[1][2] = p2.y - p1.y;
    GETPQ[2][2] = p3.y - p1.y;
}



/**************************************************************************************
** Calcul de la matrice de correction dans le mode d'alignement AFFINE
**
***************************************************************************************/

int Alignement::Calculer_Matrice_Affine( double x, double y, Coord aa1, Coord aa2, Coord aa3, Coord mm1, Coord mm2, Coord mm3)
{
    double Det;
    double EQMI[4][4];
    double EQMP[4][4];
    double EQMQ[4][4];

    Coord a1, a2, a3, m1, m2, m3, c1, c2;

    AzHa2XY(aa1.x, aa1.y, &a1.x, &a1.y);
    AzHa2XY(aa2.x, aa2.y, &a2.x, &a2.y);
    AzHa2XY(aa3.x, aa3.y, &a3.x, &a3.y);

    AzHa2XY(mm1.x, mm1.y, &m1.x, &m1.y);
    AzHa2XY(mm2.x, mm2.y, &m2.x, &m2.y);
    AzHa2XY(mm3.x, mm3.y, &m3.x, &m3.y);


    //Construit les matrices EQMP et EQMQ

    Calculer_Matrice_GETPQ(m1, m2, m3);

    for (int i=1; i<=3; i++) for (int j=1; j<=3; j++) EQMP[i][j]=GETPQ[i][j];

    Calculer_Matrice_GETPQ(a1, a2, a3);

    for (int i=1; i<=3; i++) for (int j=1; j<=3; j++) EQMQ[i][j]=GETPQ[i][j];

    // Calcule l'inverse de EQMP

    Det = (EQMP[1][1] * EQMP[2][2]) - (EQMP[1][2] * EQMP[2][1]);

    if (Det == 0)
    {
        ErreurLog("Erreur Matrice_Affine. Impossible de calculer l'inverse de la matrice avec un determinant = 0 \n");
        return 0;
    }
    else
    {
        // Si le déterminant n'est pas nul
        EQMI[1][1] =  EQMP[2][2] / Det;
        EQMI[1][2] = -EQMP[1][2] / Det;
        EQMI[2][1] = -EQMP[2][1] / Det;
        EQMI[2][2] =  EQMP[1][1] / Det;
    }


    // MATRICE = EQMI * EQMQ

    MATRICE[1][1] = (EQMI[1][1] * EQMQ[1][1]) + (EQMI[1][2] * EQMQ[2][1]);
    MATRICE[1][2] = (EQMI[1][1] * EQMQ[1][2]) + (EQMI[1][2] * EQMQ[2][2]);
    MATRICE[2][1] = (EQMI[2][1] * EQMQ[1][1]) + (EQMI[2][2] * EQMQ[2][1]);
    MATRICE[2][2] = (EQMI[2][1] * EQMQ[1][2]) + (EQMI[2][2] * EQMQ[2][2]);



    // Calcul du vecteur offset sur le ciel

    VECT_OFFSET.x =m1.x - ((a1.x * MATRICE[1][1]) + (a1.y * MATRICE[2][1]));
    VECT_OFFSET.y =m1.y - ((a1.x * MATRICE[1][2]) + (a1.y * MATRICE[2][2]));

    if (x + y == 0 )
    {
        return 0;
    }
    else
    {
        return  PointSitueDansSurfaceTriangle(x, y, aa1.x, aa1.y, aa2.x, aa2.y, aa3.x, aa3.y);
    }
}



/**************************************************************************************
** Calcule la quantité CoordObjets * MATRICECorrection + VECT_offset
**
***************************************************************************************/

void Alignement::AppliquerMatriceCorrectionAffine( Coord * result, Coord ob)
{
    result->x = VECT_OFFSET.x + ((ob.x * MATRICE[1][1]) + (ob.y * MATRICE[2][1]));
    result->y = VECT_OFFSET.y + ((ob.x * MATRICE[1][2]) + (ob.y * MATRICE[2][2]));
}


inline int Alignement::ElementsMatriceSimple(Coord aa1, Coord aa2, Coord aa3, Coord mm1, Coord mm2, Coord mm3)
{
    // mm1 = aa1 * MATRICE_Correction
    // mm2 = aa2 * MATRICE_Correction
    // mm3 = aa3 * MATRICE_Correction

    // On doit donc trouver les éléments de la matrice MATRICE_Correction en fonction des paramètres x, y, z des vecteurs
    // aa1, aa2, aa3, mm1, mm2, mm3

    // Les éléments de la matrice se trouvent aisément :

    double div= aa1.z*aa2.y*aa3.x - aa1.y*aa2.z*aa3.x - aa1.z*aa2.x*aa3.y +
                aa1.x*aa2.z*aa3.y + aa1.y*aa2.x*aa3.z - aa1.x*aa2.y*aa3.z;

    if ( div !=0.0 )
    {
        double mm11 = -(((-aa2.z)*aa3.y*mm1.x + aa2.y*aa3.z*mm1.x + aa1.z*aa3.y*mm2.x - aa1.y*aa3.z*mm2.x -
                         aa1.z*aa2.y*mm3.x + aa1.y*aa2.z*mm3.x)/div);

        double mm12 = -((aa2.z*aa3.x*mm1.x - aa2.x*aa3.z*mm1.x - aa1.z*aa3.x*mm2.x + aa1.x*aa3.z*mm2.x + aa1.z*aa2.x*mm3.x - aa1.x*aa2.z*mm3.x)/div);

        double mm13 = -(((-aa2.y)*aa3.x*mm1.x + aa2.x*aa3.y*mm1.x + aa1.y*aa3.x*mm2.x - aa1.x*aa3.y*mm2.x - aa1.y*aa2.x*mm3.x + aa1.x*aa2.y*mm3.x)/div);

        double mm21 = -(((-aa2.z)*aa3.y*mm1.y + aa2.y*aa3.z*mm1.y + aa1.z*aa3.y*mm2.y - aa1.y*aa3.z*mm2.y -
                         aa1.z*aa2.y*mm3.y + aa1.y*aa2.z*mm3.y)/div);

        double mm22 = -((aa2.z*aa3.x*mm1.y - aa2.x*aa3.z*mm1.y - aa1.z*aa3.x*mm2.y + aa1.x*aa3.z*mm2.y + aa1.z*aa2.x*mm3.y -
                         aa1.x*aa2.z*mm3.y)/div);

        double mm23 = -(((-aa2.y)*aa3.x*mm1.y + aa2.x*aa3.y*mm1.y + aa1.y*aa3.x*mm2.y - aa1.x*aa3.y*mm2.y -
                         aa1.y*aa2.x*mm3.y + aa1.x*aa2.y*mm3.y)/div);

        double mm31 = -(((-aa2.z)*aa3.y*mm1.z + aa2.y*aa3.z*mm1.z + aa1.z*aa3.y*mm2.z - aa1.y*aa3.z*mm2.z -
                         aa1.z*aa2.y*mm3.z + aa1.y*aa2.z*mm3.z)/div);

        double mm32 = -((aa2.z*aa3.x*mm1.z - aa2.x*aa3.z*mm1.z - aa1.z*aa3.x*mm2.z + aa1.x*aa3.z*mm2.z + aa1.z*aa2.x*mm3.z -
                         aa1.x*aa2.z*mm3.z)/div);

        double mm33 = -(((-aa2.y)*aa3.x*mm1.z + aa2.x*aa3.y*mm1.z + aa1.y*aa3.x*mm2.z - aa1.x*aa3.y*mm2.z -
                         aa1.y*aa2.x*mm3.z + aa1.x*aa2.y*mm3.z)/div);



        MATRICE[1][1] = mm11;
        MATRICE[2][1] = mm12;
        MATRICE[3][1] = mm13;

        MATRICE[1][2] = mm21;
        MATRICE[2][2] = mm22;
        MATRICE[3][2] = mm23;

        MATRICE[1][3] = mm31;
        MATRICE[2][3] = mm32;
        MATRICE[3][3] = mm33;

        //Pour vérifs
        //RotationAutourDunAxe(0.3,Pi/2.0-0.1, 0.2);

        return 1;
    }
    else
    {
        return 0;
    }
}

/**************************************************************************************
** Calcule la matrice de correction issue de la procédure d'alignement des antennes
**
** Cette méthode qui utilise trois points de correction seulement est particulièrement
** simple mais repose sur la qualité des pointages effectués lors de l'alignement sur
** les trois étoiles et ne prend en compte de la torsion de la monture etc...
**
** Les variables a1, a2, a3 contiennent les coordonnées horaires calculées (AD & Dec) des trois
** étoiles visées...
** Les variables m1, m2, m3 contiennent les coordonnées horaires effectivement mesurées
** de ces trois mêmes objets
**
***************************************************************************************/

inline int Alignement::Calculer_Matrice_Simple( Coord a1, Coord a2, Coord a3, Coord m1, Coord m2, Coord m3)
{
    double x, y, z;

    Coord aa1, aa2, aa3;
    Coord mm1, mm2, mm3;

    // Pour chaque variable coordonnées, on calcule d'abord l'azimut et la hauteur de l'objet correspondant

    Azimut(a1.x, a1.y, &aa1.x, &aa1.y);

    // puis on transforme les coordonnées sphériques (azi, haut) en coordonnées cartésiennes x, y, z
    x=cos(aa1.x) * cos(aa1.y);
    y=sin(aa1.x) * cos(aa1.y);
    z=sin(aa1.y);
    aa1.x=x;
    aa1.y=y;
    aa1.z=z;

    Azimut(a2.x, a2.y, &aa2.x, &aa2.y);
    x=cos(aa2.x) * cos(aa2.y);
    y=sin(aa2.x) * cos(aa2.y);
    z=sin(aa2.y);
    aa2.x=x;
    aa2.y=y;
    aa2.z=z;

    Azimut(a3.x, a3.y, &aa3.x, &aa3.y);
    x=cos(aa3.x) * cos(aa3.y);
    y=sin(aa3.x) * cos(aa3.y);
    z=sin(aa3.y);
    aa3.x=x;
    aa3.y=y;
    aa3.z=z;

    Azimut(m1.x, m1.y, &mm1.x, &mm1.y);
    x=cos(mm1.x) * cos(mm1.y);
    y=sin(mm1.x) * cos(mm1.y);
    z=sin(mm1.y);
    mm1.x=x;
    mm1.y=y;
    mm1.z=z;

    Azimut(m2.x, m2.y, &mm2.x, &mm2.y);
    x=cos(mm2.x) * cos(mm2.y);
    y=sin(mm2.x) * cos(mm2.y);
    z=sin(mm2.y);
    mm2.x=x;
    mm2.y=y;
    mm2.z=z;

    Azimut(m3.x, m3.y, &mm3.x, &mm3.y);
    x=cos(mm3.x) * cos(mm3.y);
    y=sin(mm3.x) * cos(mm3.y);
    z=sin(mm3.y);
    mm3.x=x;
    mm3.y=y;
    mm3.z=z;

    return ElementsMatriceSimple(aa1, aa2, aa3, mm1, mm2, mm3);
}


void Alignement::AppliquerMatriceCorrectionSimple(Coord * result, Coord vect)
{
    // Applique la correction sur le vecteur orientation vect avec

    result->x = vect.x * MATRICE[1][1] + vect.y * MATRICE[2][1] + vect.z * MATRICE[3][1];
    result->y = vect.x * MATRICE[1][2] + vect.y * MATRICE[2][2] + vect.z * MATRICE[3][2];
    result->z = vect.x * MATRICE[1][3] + vect.y * MATRICE[2][3] + vect.z * MATRICE[3][3];
}


double Alignement::Determinant()
{
    double Det;

    Det = MATRICE[1][1] * ((MATRICE[2][2] * MATRICE[3][3]) - (MATRICE[3][2] * MATRICE[2][ 3]));
    Det = Det - (MATRICE[1][2] * ((MATRICE[2][1] * MATRICE[3][3]) - (MATRICE[3][1] * MATRICE[2][3])));
    Det = Det + (MATRICE[1][3] * ((MATRICE[2][1] * MATRICE[3][2]) - (MATRICE[3][1] * MATRICE[2][2])));

    return Det;
}


/**************************************************************************************
** Projection simple d'un point situé aux coordonnées (az, ha) dans un cercle de
** rayon Pi / 2
**
***************************************************************************************/

void Alignement::AzHa2XY(double az, double ha, double *x, double *y)
{
    double r = Pidiv2 - ha;

    *x = r * cos(az);

    *y = r * sin(az);
}


/**************************************************************************************
** Calcule la surface d'un triangle de coordonnées (px1,py1) (px2, py2) (px3, py3)
**
***************************************************************************************/

double Alignement::Surface_Triangle(double px1, double py1, double px2, double py2, double px3, double py3)
{
    double ta;

    double pxx1, pyy1, pxx2, pyy2, pxx3, pyy3;

    AzHa2XY(px1, py1, &pxx1, &pyy1);

    AzHa2XY(px2, py2, &pxx2, &pyy2);

    AzHa2XY(px3, py3, &pxx3, &pyy3);

    ta = (((pxx2 * pyy1) - (pxx1 * pyy2)) ) + (((pxx3 * pyy2) - (pxx2 * pyy3))  ) + (((pxx1 * pyy3) - (pxx3 * pyy1))  );

    return  fabs(ta) / 2.0;
}


/**************************************************************************************
** Est-ce que le point de coordonnées (px, py) se trouve dans la surface du triangle
** de coordonnées (px1,py1) (px2, py2) (px3, py3) ?
**
***************************************************************************************/

bool Alignement::PointSitueDansSurfaceTriangle(double px, double py, double px1, double py1, double px2, double py2, double px3, double py3)
{
    double ta;
    double t1;
    double t2;
    double t3;

    ta = Surface_Triangle(px1, py1, px2, py2, px3, py3);
    t1 = Surface_Triangle(px,  py,  px2, py2, px3, py3);
    t2 = Surface_Triangle(px1, py1, px,  py,  px3, py3);
    t3 = Surface_Triangle(px1, py1, px2, py2, px,  py);

    stringstream os;

    os << fabs(ta - t1 - t2 - t3)<< "\n";
    AfficherLog(os.str().c_str(), true);

    /* if ( DistanceAngulaireEntre2Points(px, py, px1, py1) > 1.2 ) return false;
      if ( DistanceAngulaireEntre2Points(px, py, px2, py2) > 1.2 ) return false;
      if ( DistanceAngulaireEntre2Points(px, py, px3, py3) > 1.2 ) return false;*/

    return ( fabs(ta - t1 - t2 - t3) < 1e-10 );
}


/**************************************************************************************
** Initialise les cotes de l'instrument
**
***************************************************************************************/

void Alignement::InitParametresInstrument()
{
    //paramètres de l'instrument

    ED          = ED0;
    CD          = CD0;
    BC          = BC0;
    AE          = AE0;
    I0          = I00;
    Theta       = Theta0;
    CoeffMoteur = CoeffMoteur0;
}



/**************************************************************************************
** Convertit un nombre de pas en hauteur en un angle exprimé en degrés
**
***************************************************************************************/

double Alignement::Motor2Alt(double pas)
{
    return asin(( - pow( ( pas * CoeffMoteur0 ) + I00, 2.0 ) + 128279.4) / 129723.4 ) * N180divPi + Theta0;
}


double Alignement::Alt2Motor(double targetAlt)
{
    double I2 = 128279.4 - 129723.4 * sin( (targetAlt - Theta0) * Pidiv180 );

    double Codalt = (sqrt(I2) - I00) / CoeffMoteur0;

    return Codalt;
}



/**************************************************************************************
** Convertit une hauteur (exprimée en degrés) en nombre de pas codeur
**
***************************************************************************************/

double Alignement::Alt2Motor2(double targetAlt)
{
    double CD2pBC2 = CD * CD + BC * BC;

    double I2 = -2.0 * ED * sqrt( CD2pBC2 ) * sin(  ( targetAlt - Theta ) * Pidiv180 );

    I2 += ED * ED + CD2pBC2 - AE * AE;

    double Codalt = ( sqrt(I2) - I0 ) / CoeffMoteur;

    return Codalt;
}

double Alignement::Motor2Alt2(double pas)
{
    double CD2pBC2 = CD * CD + BC * BC;

    double I2 = pow(( CoeffMoteur * pas + I0 ) ,2.0);

    I2 -= ED * ED + CD2pBC2 - AE * AE;

    double angle = asin( I2 / ( -2.0 * ED * sqrt(CD2pBC2) )) * N180divPi + Theta;

    return angle;
}

void Alignement::OptimisationGeometrieAntenne(bool Chercher)
{
    Coord c1, c2, c3, c4, m1, m2, m3, mm1, mm2, mm3, mm,mmm, result;

    double i, j, k, l, m;

    double memBC = BC0;
    double memCD = CD0;
    double memED = ED0;
    double memAE = AE0;
    double memI0 = I00;
    double memCoeffMoteur = CoeffMoteur0;
    double memTheta = Theta0;

    stringstream os;

    double det;

    double maxdet = 0.1;

    double distmax2 = 1e50;

    double pas = 1.0; //0.1

    double interv = 20.0; //0.5



    /* for (int i=0; i<nbrcorrections; i++)
     {
         Precession(&ad[i], &de[i]);
        NutationEtoile(&ad[i], &de[i]);

         AberrationAnnuelle(&ad[i], &de[i]);

     Precession(&delta_ad[i], &delta_de[i]);
        NutationEtoile(&delta_ad[i], &delta_de[i]);

         AberrationAnnuelle(&delta_ad[i], &delta_de[i]);
     }*/

    c1.x = VerifAngle(ad[0] - tsl[0] + GetTSL());
    c1.y = de[0];

    Azimut(c1.x, c1.y, &mm.x, &mm.y);
    c1.x = cos(mm.x) * cos(mm.y);
    c1.y = sin(mm.x) * cos(mm.y);
    c1.z = sin(mm.y);

    c2.x = VerifAngle(ad[1] - tsl[1] + GetTSL());
    c2.y = de[1];

    Azimut(c2.x, c2.y, &mm.x, &mm.y);
    c2.x = cos(mm.x) * cos(mm.y);
    c2.y = sin(mm.x) * cos(mm.y);
    c2.z = sin(mm.y);

    c3.x = VerifAngle(ad[2] - tsl[2] + GetTSL());
    c3.y = de[2];

    Azimut(c3.x, c3.y, &mm.x, &mm.y);
    c3.x = cos(mm.x) * cos(mm.y);
    c3.y = sin(mm.x) * cos(mm.y);
    c3.z = sin(mm.y);

    m1.x = VerifAngle(delta_ad[0] - tsl[0] + GetTSL());
    m1.y = delta_de[0];

    m2.x = VerifAngle(delta_ad[1] - tsl[1] + GetTSL());
    m2.y = delta_de[1];

    m3.x = VerifAngle(delta_ad[2] - tsl[2] + GetTSL());
    m3.y = delta_de[2];


    if (Chercher)
    {
        InitParametresInstrument();

        CoeffMoteur = CoeffMoteur0;

        //  for (i=-interv; i< interv; i += pas)
        {
            I0 = I00 ;//+ i ;

            for (j=-interv; j<interv; j += pas)
            {
                ED = ED0 + j;

                for (k=-interv; k<interv; k += pas)
                {
                    CD = CD0 + k;

                    for (l=-interv; l < interv; l+= pas)
                    {
                        BC = BC0 + l;

                        Theta = (atan(CD / BC) - atan(18.5 / ED)) * N180divPi;

                        for (m=-interv; m< interv; m += pas)
                        {
                            AE = AE0 + m;

                            Azimut(m1.x, m1.y, &mm.x, &mm.y);

                            mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

                            mm1.x = cos(mm.x) * cos(mm.y);
                            mm1.y = sin(mm.x) * cos(mm.y);
                            mm1.z = sin(mm.y);

                            Azimut(m2.x, m2.y, &mm.x, &mm.y);

                            mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

                            mm2.x = cos(mm.x) * cos(mm.y);
                            mm2.y = sin(mm.x) * cos(mm.y);
                            mm2.z = sin(mm.y);

                            Azimut(m3.x, m3.y, &mm.x, &mm.y);

                            mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

                            mm3.x = cos(mm.x) * cos(mm.y);
                            mm3.y = sin(mm.x) * cos(mm.y);
                            mm3.z = sin(mm.y);

                            // Calcul de la matrice de correction en fonction de la méthode d'alignement

                            Matrice_ok = ElementsMatriceSimple(c1, c2, c3 ,mm1, mm2, mm3);

                            det = Determinant();



                            if ( fabs(det - 1.0) <= 0.08/*maxdet*/ )
                            {

                                double targetAz, targetAlt;
                                double distmax = 0.0;
                                double dist;

                                for (int n=0; n<nbrcorrections; n++)
                                {
                                    Azimut(VerifAngle(ad[n] - tsl[n] + GetTSL()),  de[n], &mmm.x, &mmm.y);

                                    mmm.y = Motor2Alt(Alt2Motor2(mmm.y * N180divPi)) * Pidiv180;

                                    c4.x = cos(mmm.x) * cos(mmm.y);
                                    c4.y = sin(mmm.x) * cos(mmm.y);
                                    c4.z = sin(mmm.y);

                                    AppliquerMatriceCorrectionSimple(&result, c4);

                                    if (result.x != 0.0)
                                    {
                                        targetAz = atan( result.y / result.x );

                                        if (result.x < 0) targetAz += Pi;
                                    }
                                    else targetAz = Pidiv2;

                                    targetAz  = VerifAngle(targetAz);

                                    targetAlt = asin(result.z);


                                    Azimut(VerifAngle(delta_ad[n] - tsl[n] + GetTSL()),  delta_de[n], &mm.x, &mm.y);

                                    dist = DistanceAngulaireEntre2Points(targetAz, targetAlt, mm.x, mm.y);

                                    if (dist> distmax) distmax=dist;
                                }



                                if (distmax< distmax2)
                                {
                                    os << "Determinant=" << det << "\n";

                                    os << "I0=" << I0 << " CD="<< CD << " ED=" << ED << " BC=" << BC << " AE=" << AE << " Theta=" << Theta << " CoeffMoteur=" << CoeffMoteur << "\n";

                                    os << DHMS(distmax*N180divPi, false) << "\n\n";

                                    AfficherLog(os.str(), true);
                                    os.str("");

                                    distmax2=distmax;
                                    //maxdet=distmax;

                                    memCD = CD;
                                    memI0 = I0;
                                    memED = ED;
                                    memAE = AE;
                                    memBC = BC;
                                    memTheta = Theta;
                                    memCoeffMoteur = CoeffMoteur;
                                }

                                maxdet = fabs(det - 1.0);
                            }
                        }
                    }
                }
                if ((int)Arrondi((interv - j) / (2.0 * interv) * 100.0) % 10 == 0)
                {
                    os << 100.0 - (interv - j) / (2.0 * interv) * 100.0 << " %\n";
                    AfficherLog(os.str(), true);
                    os.str("");
                }

            }

        }


        CD = memCD;
        ED = memED;
        AE = memAE;
        BC = memBC;
        I0 = memI0;
        Theta = memTheta;
        CoeffMoteur = memCoeffMoteur;

        os << "\n\nParamètres  : I0=" << I0 << " CD=" << CD << " ED=" << ED << " BC=" << BC << " AE=" << AE << " Theta=" << Theta << " CoeffMoteur=" << CoeffMoteur << "\n\nFin de l'optimisation\n";


        AfficherLog(os.str(), true);

    }

    Azimut(m1.x, m1.y, &mm.x, &mm.y);

    mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

    mm1.x = cos(mm.x) * cos(mm.y);
    mm1.y = sin(mm.x) * cos(mm.y);
    mm1.z = sin(mm.y);

    Azimut(m2.x, m2.y, &mm.x, &mm.y);

    mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

    mm2.x = cos(mm.x) * cos(mm.y);
    mm2.y = sin(mm.x) * cos(mm.y);
    mm2.z = sin(mm.y);

    Azimut(m3.x, m3.y, &mm.x, &mm.y);

    mm.y = Motor2Alt2(Alt2Motor(mm.y * N180divPi)) * Pidiv180;

    mm3.x = cos(mm.x) * cos(mm.y);
    mm3.y = sin(mm.x) * cos(mm.y);
    mm3.z = sin(mm.y);

    // Calcul de la matrice de correction en fonction de la méthode d'alignement

    Matrice_ok = ElementsMatriceSimple(c1, c2, c3 ,mm1, mm2, mm3);

    AfficherMatrice();
}





/**************************************************************************************
** Calcule la matrice de correction issue de la procédure d'alignement des antennes
**
** Routine principale. On construit ici trois vecteurs c1, c2 et c3 qui contiennent
** les coordonnées (issues du catalogue) de trois objets visés pendant la procédure
** d'alignement. Les vecteurs m1, m2 et m3 contiennent les coordonnées mesurées
** de ces mêmes objets dans le viseur de l'instrument.
**
***************************************************************************************/

void Alignement::CalculerMatriceCorrection(double ar, double dec)
{
    Coord c1, c2, c3, m1, m2, m3;

    double a1, b1, a2, b2, a3, b3, a4, b4, a5, b5;

    int map[MAXALIGNEMENTANTENNE];

    stringstream os;

    int best1 = -1;
    int best2 = -1;
    int best3 = -1;


    // On calcule le temps sidéral local, etc...

    CalculTSL();

    if ( BC!=BC0 || CD!=CD0 || ED!=ED0 || AE!=AE0 || I0!=I00 || CoeffMoteur!=CoeffMoteur0 )
    {
        OptimisationGeometrieAntenne(false);

        return;
    }

    switch ( MethodeAlignement )
    {
    case SIMPLE :
    {
        best1 = 0;
        best2 = 1;
        best3 = 2;

        for (int i=0; i<MAXALIGNEMENTANTENNE; i++) map[i] = i;
    }
    break;


    case AFFINE:
    case TAKI  :
    {
        double *distances;

        distances = new double [nbrcorrections+1];

        os << "Coordonnées visees : ad=" << DHMS(ar * 15.0


                ,true) << "  de=" << DHMS(dec,false) << "\n";

        Azimut(ar * 15.0 * Pidiv180, dec * Pidiv180, &a2, &b2);

        for (int i=0; i<nbrcorrections; i++)
        {
            Azimut(VerifAngle(ad[i]- tsl[i] + GetTSL() ),de[i], &a1, &b1);

            distances[i] = DistanceAngulaireEntre2Points(a1, b1, a2, b2);

            map[i]=i;
        }

        for (int i=0; i<nbrcorrections; i++)
        {
            for (int j=i+1; j<nbrcorrections; j++)
            {
                if ( distances[j] < distances[i] )
                {
                    double bout  = distances[i];
                    distances[i] = distances[j];
                    distances[j] = bout;

                    int bout2 = map[i];
                    map[i]    = map[j];
                    map[j]    = bout2;
                }
            }
        }

        for (int i=0; i<nbrcorrections; i++)
        {
            os << "Point " << map[i] << "  d=" << distances[i] * N180divPi << "°\n";
        }

        for (int i=0; i<nbrcorrections; i++)
        {
            for (int j=i+1; j<nbrcorrections; j++)
            {
                for (int k=j+1; k<nbrcorrections; k++)
                {
                    os << "Triangle : " << map[i] << " " << map[j] << " " << map[k]  << "\n";

                    os << "Objet ";

                    os << ": az=" << DHMS(a2*N180divPi, false) << "  ha=" <<  DHMS(b2*N180divPi, false) << "\n";

                    os << "Etoile " << map[i] ;

                    Azimut(VerifAngle(ad[map[i]] - tsl[map[i]] + GetTSL()), de[map[i]], &a3, &b3);

                    os << ": az=" << DHMS(a3*N180divPi, false) << "  ha=" <<  DHMS(b3*N180divPi, false) << "\n";

                    os << "Etoile " << map[j] ;

                    Azimut(VerifAngle(ad[map[j]] - tsl[map[j]] + GetTSL()), de[map[j]], &a4, &b4);

                    os << ": az=" << DHMS(a4*N180divPi, false) << "  ha=" <<  DHMS(b4*N180divPi, false) << "\n";

                    os << "Etoile " << map[k] ;

                    Azimut(VerifAngle(ad[map[k]] - tsl[map[k]] + GetTSL()), de[map[k]], &a5, &b5);

                    os << ": az=" << DHMS(a5*N180divPi, false) << "  ha=" <<  DHMS(b5*N180divPi, false) << "\n";

                    AfficherLog(&os, true);

                    if (PointSitueDansSurfaceTriangle(a2, b2,
                                                      a3, b3,
                                                      a4, b4,
                                                      a5, b5))
                    {
                        best1=i;
                        best2=j;
                        best3=k;
                        k=1000;
                        j=1000;
                        i=1000;
                    }
                }
            }
        }

        delete [] distances;
    }
    break;
    }



    if ( best1 == -1 && best2 == -1 && best3 == -1 )
    {
        if (MethodeAlignement == AFFINE || MethodeAlignement == TAKI)
        {
            ErreurLog("Erreur ! Pas de triangle disponible pour la correction !\nLe positionnement de l antenne sera approximatif !\n");
        }

        Identity();

        Matrice_ok = false;

        return;
    }
    else
    {
        os << "\nUtilisation du triangle : " << map[best1] << " " << map[best2] << " " << map[best3]  << "\n" ;

        // if (MethodeAlignement == SIMPLE)
        {
            c1.x = VerifAngle(ad[map[best1]] - tsl[map[best1]] + GetTSL());
            c1.y = de[map[best1]];
            c1.z = 1.0;

            c2.x = VerifAngle(ad[map[best2]] - tsl[map[best2]] + GetTSL());
            c2.y = de[map[best2]];
            c2.z = 1.0;

            c3.x = VerifAngle(ad[map[best3]] - tsl[map[best3]] + GetTSL());
            c3.y = de[map[best3]];
            c3.z = 1.0;
        }
        /*    else
            {
                c1.x = VerifAngle(ad[map[best1]] - tsl[map[best1]] + GetTSL());
                c1.y = de[map[best1]];
                c1.z = 1.0;

                c2.x = VerifAngle(ad[map[best2]] - tsl[map[best2]] + GetTSL());
                c2.y = de[map[best2]];
                c2.z = 1.0;

                c3.x = VerifAngle(ad[map[best3]] - tsl[map[best3]] + GetTSL());
                c3.y = de[map[best3]];
                c3.z = 1.0;
            }*/

        m1.x = VerifAngle(delta_ad[map[best1]] - tsl[map[best1]] + GetTSL());
        m1.y = delta_de[map[best1]];
        m1.z = 1.0;

        m2.x = VerifAngle(delta_ad[map[best2]] - tsl[map[best2]] + GetTSL());
        m2.y = delta_de[map[best2]];
        m2.z = 1.0;

        m3.x = VerifAngle(delta_ad[map[best3]] - tsl[map[best3]] + GetTSL());
        m3.y = delta_de[map[best3]];
        m3.z = 1.0;
    }

    // Calcul de la matrice de correction en fonction de la méthode d'alignement

    switch (MethodeAlignement)
    {
    case SIMPLE:
        os << "Methode d alignement SIMPLE\n";
        Matrice_ok = Calculer_Matrice_Simple(c1, c2, c3 ,m1, m2, m3);
        break;
    case AFFINE:
        os << "Methode d alignement AFFINE\n";
        Matrice_ok = Calculer_Matrice_Affine( VerifAngle( ar * 15.0 * Pidiv180 ), dec * Pidiv180, c1, c2, c3 ,m1, m2, m3);
        break;
    case TAKI:
        os << "Methode d alignement TAKI\n";
        Matrice_ok = Calculer_Matrice_Taki  ( VerifAngle( ar * 15.0 * Pidiv180 ), dec * Pidiv180, c1, c2, c3 ,m1, m2, m3);
        break;
    }

    //Afficher les messages de cette fonction

    AfficherLog(os.str(), true);

    AfficherMatrice();


    for (int i=0; i<nbrcorrections; i++) SelectionnePourCalculMatrice[i] = false;

    SelectionnePourCalculMatrice[map[best1]] = true;
    SelectionnePourCalculMatrice[map[best2]] = true;
    SelectionnePourCalculMatrice[map[best3]] = true;
}



/**************************************************************************************
** Le fichier file existe-t-il et n'est-il pas vide ?
**************************************************************************************/

bool Alignement::is_readable( const std::string & file )
{
    std::ifstream fichier( file.c_str() );
    if (fichier.fail()) return false;

    // sauvegarder la position courante
    // long pos = fichier.tellg();
    // se placer en fin de fichier
    fichier.seekg( 0 , std::ios_base::end );
    // récupérer la nouvelle position = la taille du fichier
    long size = fichier.tellg();
    return (size > 0);
}


/**************************************************************************************
** Chargement des paramètres d'alignement des antennes
**************************************************************************************/

bool Alignement::ChargementParametresAlignement(string IP, string fileName, double a, double b)
{
    static double mema = a;

    static double memb = b;

    string value;

    stringstream os, os2;

    char *delta_az = NULL;

    char *delta_ha = NULL;

    char *ad_data = NULL;

    char *de_data = NULL;

    char *delta_ad_data = NULL;

    char *delta_de_data = NULL;

    char *align_data = NULL;

    char *alignment_method_data = NULL;

    char *tsl_data = NULL;

    char *c_BC = NULL;

    char *c_CD = NULL;

    char *c_ED = NULL;

    char *c_AE = NULL;

    char *c_I0 = NULL;

    char *c_CoeffMoteur = NULL;

    char *c_Theta = NULL;

    if (!is_readable(fileName)) return false;

    // Chargement des corrections des antennes

    bool modificationAlignement = false;

    // On sélectionne la bonne antenne (à partir de son adresse ip).

    os << "Alignement antenne ip " << IP;


    nbrcorrections = 0;

    for (int j=0; j < MAXALIGNEMENTANTENNE; j++)
    {
        os2 << "ad " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &ad_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "de " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &de_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "delta_ad " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_ad_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "delta_de " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_de_data,  (char*)fileName.c_str());

        os2.str("");

        os2 << "tsl " << j;

        readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &tsl_data,  (char*)fileName.c_str());

        os2.str("");


        if ( ad_data && de_data && delta_ad_data && delta_de_data
                && tsl_data && atof(tsl_data)!=0.0 && atof(ad_data)!=0.0 && atof(de_data)!=0.0)
        {
            // Si les corrections de l'antenne i ont été modifiées depuis le démarrage du programme
            // -> on les applique...



            if (delta_ad[nbrcorrections] != atof(delta_ad_data) || delta_de[nbrcorrections] != atof(delta_de_data))
            {
                ad[nbrcorrections] = atof(ad_data);

                de[nbrcorrections] = atof(de_data);

                delta_ad[nbrcorrections] = atof(delta_ad_data);

                delta_de[nbrcorrections] = atof(delta_de_data);

                tsl[nbrcorrections] = atof(tsl_data);

                os2 << "Correction antenne ip=" << IP <<  " ad=" << ad[nbrcorrections] << " de=" << de[nbrcorrections] << " deltaAD=" <<
                    delta_ad[nbrcorrections] << " deltaDe=" << delta_de[nbrcorrections] << " tsl=" << tsl[nbrcorrections] << "\n";

                AfficherLog(&os2, true);

                modificationAlignement = true;

                nbrcorrections++;
            }
        }

        SAFEDELETE_TAB(ad_data);

        SAFEDELETE_TAB(de_data);

        SAFEDELETE_TAB(delta_ad_data);

        SAFEDELETE_TAB(delta_de_data);

        SAFEDELETE_TAB(tsl_data);
    }

    os2 << "delta_az";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_az,  (char*)fileName.c_str());

    os2.str("");

    if (delta_az)
    {
        if ( deltaAZ != atol(delta_az) )
        {
            deltaAZ = atol(delta_az);

            modificationAlignement = true;
        }
    }

    os2 << "delta_ha";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &delta_ha,  (char*)fileName.c_str());

    os2.str("");

    if (delta_ha)
    {
        if ( deltaHA != atol(delta_ha) )
        {
            deltaHA= atol(delta_ha);

            modificationAlignement = true;
        }
    }

    os2 << "alignment_method";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &alignment_method_data,  (char*)fileName.c_str());

    os2.str("");

    if (alignment_method_data)
    {
        if ( MethodeAlignement != atoi(alignment_method_data) )
        {
            MethodeAlignement = atoi(alignment_method_data);

            modificationAlignement = true;
        }
    }

    os2 << "align";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &align_data,  (char*)fileName.c_str());

    os2.str("");


    if (align_data)
    {
        // Si la procédure d'alignement se termine, il faut calculer la matrice de correction

        if (AlignementEnCours != atoi(align_data))
        {
            AlignementEnCours = atoi(align_data);

            modificationAlignement = true;
        }
    }

    if (mema!=a || memb!=b)
    {
        mema = a;

        memb = b;

        modificationAlignement = true;
    }

    // si nbrcorrections=0 (suite à l'utilisation de la commande reset dans BAOcontrol par ex) il faut
    // réinitialiser la matrice

    if (nbrcorrections == 0 ) modificationAlignement = true;


    os2 << "BC";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_BC,  (char*)fileName.c_str());

    os2.str("");

    os2 << "CD";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_CD,  (char*)fileName.c_str());

    os2.str("");

    os2 << "ED";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_ED,  (char*)fileName.c_str());

    os2.str("");

    os2 << "AE";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_AE,  (char*)fileName.c_str());

    os2.str("");

    os2 << "I0";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_I0,  (char*)fileName.c_str());

    os2.str("");

    os2 << "CoeffMoteur";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_CoeffMoteur,  (char*)fileName.c_str());

    os2.str("");

    os2 << "Theta";

    readINI((char*)os.str().c_str(), (char*)os2.str().c_str(), &c_Theta,  (char*)fileName.c_str());

    os2.str("");

    if ( c_BC && c_CD && c_ED && c_AE && c_I0 && c_CoeffMoteur && c_Theta &&
            atof(c_BC)!=0.0 && atof(c_CD)!=0.0 && atof(c_ED)!=0.0 && atof(c_AE)!=0.0 && atof(c_I0)!=0.0 && atof(c_CoeffMoteur)!=0.0 && atof(c_Theta)!=0.0 )
    {
        BC = atof(c_BC);

        CD = atof(c_CD);

        ED = atof(c_ED);

        AE = atof(c_AE);

        I0 = atof(c_I0);

        CoeffMoteur = atof(c_CoeffMoteur);

        Theta = atof(c_Theta);

        IDLog("Chargement des parametres de l antenne\n", true);

        modificationAlignement = true;
    }

    // Si un des paramètres d'une antenne change pendant la procédure d'alignement ou que l'on a
    // plus de deux points d'alignement sur une antenne, alors on calcule la matrice de correction

    if (modificationAlignement)
    {
        if  (nbrcorrections >=3 && AlignementEnCours == 0)
        {
            CalculerMatriceCorrection(a, b);
        }
        else
        {
            Identity();

            Matrice_ok = false;
        }
    }

    SAFEDELETE_TAB(delta_az);

    SAFEDELETE_TAB(delta_ha);

    SAFEDELETE_TAB(alignment_method_data);

    SAFEDELETE_TAB(align_data);

    SAFEDELETE_TAB(c_BC);

    SAFEDELETE_TAB(c_CD);

    SAFEDELETE_TAB(c_ED);

    SAFEDELETE_TAB(c_AE);

    SAFEDELETE_TAB(c_I0);

    SAFEDELETE_TAB(c_CoeffMoteur);

    SAFEDELETE_TAB(c_Theta);

    return true;
}



/**************************************************************************************
** Sauvegarde des paramètres d'alignement des antennes
***************************************************************************************/

bool Alignement::EnregistrementParametresAlignement(string IP, string fileName)
{
    string section;
    string key;
    string value;

    stringstream os;


    //AfficherLog("Enregistrement de l'alignement des antennes dans le fichier " + fileName + "\n", true);

    //Enregistrement des corrections des l'antennes


    os << "Alignement antenne ip " << IP;
    section = os.str();
    os.str("");

    os << "align";
    key     = os.str();
    os.str("");
    (AlignementEnCours == true) ? os << "1" : os << "0";
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "alignment_method";
    key     = os.str();
    os.str("");
    os << MethodeAlignement;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "delta_az";
    key     = os.str();
    os.str("");
    os << deltaAZ;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "delta_ha";
    key     = os.str();
    os.str("");
    os << deltaHA;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());


    for (int j = 0; j < MAXALIGNEMENTANTENNE; j++)
    {
        os << "ad " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "de " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_ad " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "delta_de " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

        os << "tsl " << j;
        key     = os.str();
        os.str("");
        os << 0.0;
        value = os.str();
        os.str("");

        writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());
    }

    for (int j = 0; j < nbrcorrections; j++) if (ad[j]!=0.0 && de[j]!=0.0)
        {
            os << "ad " << j;
            key     = os.str();
            os.str("");
            os << ad[j];
            value = os.str();
            os.str("");

            writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

            os << "de " << j;
            key     = os.str();
            os.str("");
            os << de[j];
            value = os.str();
            os.str("");

            writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

            os << "delta_ad " << j;
            key     = os.str();
            os.str("");
            os << delta_ad[j];
            value = os.str();
            os.str("");

            writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

            os << "delta_de " << j;
            key     = os.str();
            os.str("");
            os << delta_de[j];
            value = os.str();
            os.str("");

            writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

            os << "tsl " << j;
            key     = os.str();
            os.str("");
            os << tsl[j];
            value = os.str();
            os.str("");

            writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());
        }

    os << "BC";
    key     = os.str();
    os.str("");
    os << BC;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "CD";
    key     = os.str();
    os.str("");
    os << CD;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "ED";
    key     = os.str();
    os.str("");
    os << ED;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());


    os << "AE";
    key     = os.str();
    os.str("");
    os << AE;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "I0";
    key     = os.str();
    os.str("");
    os << I0;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "CoeffMoteur";
    key     = os.str();
    os.str("");
    os << CoeffMoteur;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());

    os << "Theta";
    key     = os.str();
    os.str("");
    os << Theta;
    value = os.str();
    os.str("");

    writeINI((char*)section.c_str(), (char*)key.c_str(), (char*)value.c_str(), (char*)fileName.c_str());


    return true;
}


bool Alignement::EnregistrementParametresAlignement(int IP, string fileName)
{
    stringstream os;

    os << IP;

    return EnregistrementParametresAlignement(os.str(), fileName);
}
