/*
#############################
##
## BAORadio Indi driver
## Franck RICHARD
## franckrichard033@gmail.com
## Décembre 2011
##
#############################
*/

#include "BAO.h"


auto_ptr<BAO> telescope(0);

const int  POLLMS = 1;					// Period of update, 1 ms.

const char *mydev = "BAO";				// Name of our device.

const char *BASIC_GROUP    = "Main Control";		// Main Group
const char *OPTIONS_GROUP  = "Options";			// Options Group
const char *RAQ_GROUP      = "Raquette";		// Options Group
const char *ALIG_GROUP     = "Alignement";		// Options Group

#define targetRA    EquatorialCoordsWN[0].value
#define targetDEC   EquatorialCoordsWN[1].value



static void ISPoll(void *);

void* LancementThread(BAO * appli);



/**************************************************************************************
** Initialisation de la classe BAO
**
***************************************************************************************/

BAO::BAO()
{
    init_properties();

    // Le bouton connect de la boîte de dialogue d'indi_BAO
    // est dans l'état IDLE au démarrage

    ConnectSP.s           = IPS_IDLE;

    // dernière actualisation

    lastRA                = 0.0;
    lastDEC               = 0.0;

    // Jour julien de la dernière actualisation

    JJAnc                 = 0.0;

    // Variables d'état

    currentSet            =  0;

    // Nombre d'antennes connectées

    SocketsNumber         =  1;

    // Le mode tracking est activé par défaut

    TrackingMode          =  BAO_TRACKING;

    // Méthode d'alignement par défaut

    MethodeAlignement     = SIMPLE;

    // délais en sec entre deux actualisations
    // dans les modes transit et tracking

    // Délai entre 2 actualisations dans le mode transit (en sec)

    ActualisationTMTransit  = UPDATETRANSITDELAY;

    // Délai entre 2 actualisations dans le mode tracking (en sec)

    ActualisationTMTracking = UPDATETRACKINGDELAY;

    // cette variable vaut "true" lorsque le thread de l'aplication a été initialisé

    InitThreadOK   	  = false;

    // UpdatedGoto = true lorsque le dernier goto a bien été réalisé jusqu'au bout

    UpdatedGoto     	  = true;

    // RealisationGoto = true lorsque les étapes nécessaires à la réalisation d'un goto
    // sont en cours d'execution

    RealisationGoto       = false;

    // = true si l'utilisateur demande l'annulation du mouvement en cours

    Abort          	  = false;

    // = true si l'utilisateur demande de mettre les antennes en mode park

    Park           	  = false;

    // = true si un objet est actuellement suivi par les antennes

    Suivi         	  = false;

    // = true si l'utilisateur sort de l'application -> permet de fermer le thread

    Exit           	  = false;

    AutoriserTourComplet  = false;


    // Initialisation des paramètres atmosphériques par défaut

    Pression              = 1013.0;

    Temp                  = 10.0;

    // Paramètres de la raquette

    delta_az 		  = 0;
    delta_ha              = 0;
    VitesseRaquette       = 10;
    targetAlignmentIP     = -1;

    // Ces deux variables définissent un intervalle où l'azimut (exprimée en pas codeur)
    // de l'object pointé va évoluer au cours du suivi...

    azmincodeur = MINAZCODEURS;
    azmaxcodeur = MAXAZCODEURS;


    // initialisation des sockets (Antennes)

    for (int i=0; i<MAXHOSTNAME; i++)
    {
        Sockets[i].Connected         = false;
        Sockets[i].IP                = "";
        Sockets[i].TargetPosition.x  = 0.0;
        Sockets[i].TargetPosition.y  = 0.0;
        Sockets[i].AlignementAntenne = new Alignement();

        if (Sockets[i].AlignementAntenne) Sockets[i].AlignementAntenne->InitAlignement();
    }

    // initialisations supplémentaires

    InitAntennes();

    // Numéro de version

    AfficherLog("Indi server BAORadio...\nDriver Version: %s (%s)\n",VERSION, VERSION_DATE);

    //connect_telescope();
}



/**************************************************************************************
** Destructeur
**
***************************************************************************************/

BAO::~BAO()
{
    // On informe le thread que nous allons sortir

    Exit = true;

    // On lui laisse une seconde pour qu'il se termine

    sleep(1);

    // destruction du thread

    pthread_join(th1, NULL);

    // destruction des objets AlignementAntennes

    for (int i=0; i<MAXHOSTNAME; i++)
    {
        delete Sockets[i].AlignementAntenne;
    }

    // Petit message à l'attention de l'utilisateur
    // permet de savoir si le destructeur a bien été atteint

    AfficherLog("Sortie de indi_BAO\n");
}



/**************************************************************************************
** Affiche le message à l'écran puis sauvegarde le message dans le fichierBAO_indi.log
**
***************************************************************************************/

void BAO::AfficherLog(const char* fmt,...)
{
    static bool avertissement = false;

    va_list ap;

    FILE *pFile = NULL;

    string fichier;

    va_start (ap, fmt);

    vfprintf (stderr, fmt, ap);

    va_end (ap);

    fichier = "/home/" + (string)getenv("USER") + "/BAO_indi.log";

    //Si le fichier log a une taille > 3M -> on efface le fichier

    //if ( Taille(fichier) > TAILLEMAXLOGS ) remove(fichier.c_str());

    if ( (pFile = fopen(fichier.c_str(), "a")) != NULL)
    {
        time_t rawtime;
        struct tm * timeinfo;

        char buffer[80];

        // On sauvegarde la date et l'heure

        time ( &rawtime );
        timeinfo = localtime ( &rawtime );
        strftime (buffer, 80,"%c    ",timeinfo);
        fprintf(pFile, "%s", buffer);

        // On sauvegarde le message dans le fichier log
        va_start (ap, fmt);

        vfprintf(pFile, fmt, ap);

        va_end (ap);

        fclose(pFile);
    }
    else
    {
        if (!avertissement)
        {
            string chaine = "Impossible d'écrire dans le fichier " + fichier;
            chaine +="\nMerci de vérifier les permissions\n";
            chaine += "ou exécutez le programme en mode superutilisateur.\n\n";

            va_start (ap, fmt);

            vfprintf (stderr, chaine.c_str(), ap);

            va_end (ap);

            avertissement = true;
        }
    }
}



/************************************************************************************
** Initialisation des paramètres des antennes
**
************************************************************************************/

void BAO::InitAntennes()
{
    for (int i=0; i<MAXHOSTNAME; i++)
    {
        Sockets[i].status             = 0;
        Sockets[i].sendalertes        = 0;
        Sockets[i].AttenteExecution   = 0;
        Sockets[i].AnomaliesExecution = 0;
        Sockets[i].etape              = 0;

        Sockets[i].ack_status         = false;
        Sockets[i].ack_pos            = false;
        Sockets[i].ack_park           = false;
        Sockets[i].ack_abort          = false;
        Sockets[i].ack_goto           = false;

        Sockets[i].PosValides         = false;
        Sockets[i].GotoOk             = false;
    }
}



/**************************************************************************************
** Initialisation des boutons et des zones d'affichage dans la boîte de dialogue INDI
** se référer à la documentation d'INDI
**
***************************************************************************************/

void BAO::init_properties()
{
    // Connection
    IUFillSwitch(&ConnectS[0], "CONNECT", "Connect", ISS_OFF);
    IUFillSwitch(&ConnectS[1], "DISCONNECT", "Disconnect", ISS_ON);
    IUFillSwitchVector(&ConnectSP, ConnectS, NARRAY(ConnectS), mydev, "CONNECTION", "Connection", BASIC_GROUP, IP_RW, ISR_1OFMANY, 60, IPS_IDLE);

    // Coord Set
    IUFillSwitch(&OnCoordSetS[0], "TRANSIT", "Transit", ISS_ON);
    IUFillSwitch(&OnCoordSetS[1], "TRACKING", "Tracking", ISS_OFF);
    IUFillSwitchVector(&OnCoordSetSP, OnCoordSetS, NARRAY(OnCoordSetS), mydev, "ON_COORD_SET", "On Set", BASIC_GROUP, IP_RW, ISR_1OFMANY, 0, IPS_IDLE);

    // Abort
    IUFillSwitch(&AbortSlewS[0], "ABORT", "Abort", ISS_OFF);
    IUFillSwitchVector(&AbortSlewSP, AbortSlewS, NARRAY(AbortSlewS), mydev, "ABORT_MOTION", "Abort", BASIC_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Park
    IUFillSwitch(&ParkS[0], "PARK", "Park", ISS_OFF);
    IUFillSwitchVector(&ParkSP, ParkS, NARRAY(ParkS), mydev, "", "Park", BASIC_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Object Name
    IUFillText(&ObjectT[0], "OBJECT_NAME", "Name", "--");
    IUFillTextVector(&ObjectTP, ObjectT, NARRAY(ObjectT), mydev, "OBJECT_INFO", "Object", BASIC_GROUP, IP_RW, 0, IPS_IDLE);

    // Equatorial Coords - SET
    IUFillNumber(&EquatorialCoordsWN[0], "RA", "RA  H:M:S", "%10.6m",  0., 24., 0., 0.);
    IUFillNumber(&EquatorialCoordsWN[1], "DEC", "Dec D:M:S", "%10.6m", -90., 90., 0., 0.);
    IUFillNumberVector(&EquatorialCoordsWNP, EquatorialCoordsWN, NARRAY(EquatorialCoordsWN), mydev, "EQUATORIAL_EOD_COORD_REQUEST" , "Equatorial JNow", BASIC_GROUP, IP_WO, 0, IPS_IDLE);

    //Command
    IUFillText(&CommandT[0], "COMMAND", "Command", "");
    IUFillTextVector(&CommandTP, CommandT, NARRAY(CommandT), mydev, "COMMAND_SET", "Command", BASIC_GROUP, IP_WO, 0, IPS_IDLE);


    // Alignement- bouton Alt+
    IUFillSwitch(&AlignementAltp[0], "ALTP", "Alt+", ISS_OFF);
    IUFillSwitchVector(&AlignementAltpP, AlignementAltp, NARRAY(AlignementAltp), mydev, "ALT_P", "Alt+", RAQ_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Alignement- bouton Az
    IUFillSwitch(&AlignementAz[0], "AZM", "Az-", ISS_OFF);
    IUFillSwitch(&AlignementAz[1], "AZP", "Az+", ISS_OFF);
    IUFillSwitchVector(&AlignementAzP, AlignementAz, NARRAY(AlignementAz), mydev, "AZ", "Azimut", RAQ_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);


    // Alignement- bouton Alt-
    IUFillSwitch(&AlignementAltn[0], "ALTN", "Alt-", ISS_OFF);
    IUFillSwitchVector(&AlignementAltnP, AlignementAltn, NARRAY(AlignementAltn), mydev, "ALT_N", "Alt-", RAQ_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);

    // Vitesse Raquette - bouton Az
    IUFillSwitch(&RaquetteN[0], "RAQ1", "1x", ISS_OFF);
    IUFillSwitch(&RaquetteN[1], "RAQ10", "10x", ISS_OFF);
    IUFillSwitchVector(&RaquetteNP, RaquetteN, NARRAY(RaquetteN), mydev, "RAQ", "Vitesse raquette", RAQ_GROUP, IP_RW, ISR_ATMOST1, 0, IPS_IDLE);


    // Alignement Coords - SET
    IUFillText(&AlignementIP[0], "IP", "IP", "");
    IUFillTextVector(&AlignementIPP, AlignementIP, NARRAY(AlignementIP), mydev, "ALIGNEMENT_IP", "Alignment IP", ALIG_GROUP, IP_WO, 0, IPS_IDLE);


    // Alignment validation
    IUFillSwitch(&AlignementOk[0], "OK", "Valider", ISS_OFF);
    IUFillSwitch(&AlignementOk[1], "SAUV", "Sauvegarder", ISS_OFF);
    IUFillSwitchVector(&AlignementOkP, AlignementOk, NARRAY(AlignementOk), mydev, "ALIGNMENT_OK", "Validation", ALIG_GROUP, IP_RW, ISR_1OFMANY, 0, IPS_IDLE);

    // Alignment reset
    IUFillSwitch(&AlignementReset[0], "RESET", "Reset", ISS_OFF);
    IUFillSwitchVector(&AlignementResetP, AlignementReset, NARRAY(AlignementReset), mydev, "ALIGNMENT_RESET", "Reset", ALIG_GROUP, IP_RW, ISR_1OFMANY, 0, IPS_IDLE);

    // Alignment Set
    IUFillSwitch(&AlignmentS[0], "SIMPLE", "Simple", ISS_ON);
    IUFillSwitch(&AlignmentS[1], "AFFINE", "Affine", ISS_OFF);
    IUFillSwitch(&AlignmentS[2], "TAKI", "Taki", ISS_OFF);
    IUFillSwitchVector(&AlignmentSP, AlignmentS, NARRAY(AlignmentS), mydev, "ALIGNMENT_SET", "Alignment method", ALIG_GROUP, IP_RW, ISR_1OFMANY, 0, IPS_IDLE);


    // Geographic coord - SET
    IUFillNumber(&GeographicCoordsWN[0], "LAT", "Lat  D", "%10.6m",  -90., 90., 0., 0.);
    IUFillNumber(&GeographicCoordsWN[1], "LONG", "Long D", "%10.6m", 0., 360., 0., 0.);
    IUFillNumberVector(&GeographicCoordsWNP, GeographicCoordsWN, NARRAY(GeographicCoordsWN), mydev, "GEOGRAPHIC_COORD" , "Geographic coords", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);

    // Pression température - SET
    IUFillNumber(&PressionTempWN[0], "Pression", "Pression mb", "%4.0f",  0., 1500., 0., 0.);
    IUFillNumber(&PressionTempWN[1], "Temperature", "Temperature °c", "%3.0f", -50., +50., 0., 0.);
    IUFillNumberVector(&PressionTempWNP, PressionTempWN, NARRAY(PressionTempWN), mydev, "PRESSION_DATA" , "Pression, Temperature", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);

    // Actualisation - SET
    IUFillNumber(&ActualisationN1[0], "DELAY", "Transit delay (s)", "%3.0f",  0., 3600., 0., 0.);
    IUFillNumberVector(&ActualisationNP1, ActualisationN1, NARRAY(ActualisationN1), mydev, "DELAY1" , "", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);

    IUFillNumber(&ActualisationN2[0], "DELAY", "Tracking delay (s)", "%3.0f",  0., 3600., 0., 0.);
    IUFillNumberVector(&ActualisationNP2, ActualisationN2, NARRAY(ActualisationN2), mydev, "DELAY2" , "", OPTIONS_GROUP, IP_WO, 0, IPS_IDLE);
}


/**************************************************************************************
** Définition de tous les vecteurs de la boîte de dialogue INDI
** Cette procédure doit être obligatoirement présente dans tous pilotes Indi
**
***************************************************************************************/

void BAO::ISGetProperties(const char *dev)
{

    if (dev && strcmp (mydev, dev))
        return;

    // Main Control
    IDDefSwitch(&ConnectSP, NULL);
    IDDefText(&ObjectTP, NULL);
    IDDefNumber(&EquatorialCoordsWNP, NULL);
    IDDefText(&CommandTP, NULL);
    IDDefSwitch(&OnCoordSetSP, NULL);
    IDDefSwitch(&AbortSlewSP, NULL);
    IDDefSwitch(&ParkSP, NULL);

    // Raquette
    IDDefSwitch(&AlignementAltpP, NULL);
    IDDefSwitch(&AlignementAzP, NULL);
    IDDefSwitch(&AlignementAltnP, NULL);

    //alignement
    IDDefSwitch(&RaquetteNP, NULL);
    IDDefText(&AlignementIPP, NULL);
    IDDefSwitch(&AlignementOkP, NULL);
    IDDefSwitch(&AlignementResetP, NULL);
    IDDefSwitch(&AlignmentSP, NULL);

    // Options
    IDDefNumber(&ActualisationNP1, NULL);
    IDDefNumber(&ActualisationNP2, NULL);
    IDDefNumber(&PressionTempWNP, NULL);
    IDDefNumber(&GeographicCoordsWNP, NULL);
}


/**************************************************************************************
** Initialisation des vecteurs de la boîte Indi
** Cette procédure doit être obligatoirement présente dans tous pilotes Indi
**
***************************************************************************************/

void BAO::reset_all_properties()
{
    ConnectSP.s			= IPS_IDLE;
    OnCoordSetSP.s		= IPS_IDLE;
    AlignmentSP.s		= IPS_IDLE;
    AlignementAltpP.s		= IPS_IDLE;
    AlignementAltnP.s		= IPS_IDLE;
    AlignementAzP.s		= IPS_IDLE;
    AlignementIPP.s             = IPS_IDLE;
    AlignementOkP.s	        = IPS_IDLE;
    RaquetteNP.s	        = IPS_IDLE;
    AlignementResetP.s  	= IPS_IDLE;
    AbortSlewSP.s		= IPS_IDLE;
    ParkSP.s			= IPS_IDLE;
    ObjectTP.s			= IPS_IDLE;
    EquatorialCoordsWNP.s	= IPS_IDLE;
    CommandTP.s			= IPS_IDLE;
    PressionTempWNP.s           = IPS_IDLE;
    GeographicCoordsWNP.s	= IPS_IDLE;
    ActualisationNP1.s		= IPS_IDLE;
    ActualisationNP2.s		= IPS_IDLE;

    IUResetSwitch(&OnCoordSetSP);
    IUResetSwitch(&AlignmentSP);
    IUResetSwitch(&AlignementAltpP);
    IUResetSwitch(&AlignementAzP);
    IUResetSwitch(&AlignementAltnP);
    IUResetSwitch(&AlignementOkP);
    IUResetSwitch(&AlignementResetP);
    IUResetSwitch(&RaquetteNP);
    IUResetSwitch(&AbortSlewSP);
    IUResetSwitch(&ParkSP);

    OnCoordSetS[0].s = ISS_ON;
    AlignmentS[0].s = ISS_ON;
    AlignementAltp[0].s = ISS_OFF;
    AlignementAltn[0].s = ISS_OFF;
    AlignementAz[0].s = ISS_OFF;
    AlignementAz[1].s = ISS_OFF;
    ConnectS[0].s = ISS_OFF;
    ConnectS[1].s = ISS_ON;

    IDSetSwitch(&ConnectSP, NULL);
    IDSetSwitch(&OnCoordSetSP, NULL);
    IDSetSwitch(&AlignmentSP, NULL);
    IDSetSwitch(&AlignementAltpP, NULL);
    IDSetSwitch(&AlignementAzP, NULL);
    IDSetSwitch(&AlignementAltnP, NULL);
    IDSetSwitch(&AlignementOkP, NULL);
    IDSetSwitch(&AlignementResetP, NULL);
    IDSetSwitch(&RaquetteNP, NULL);
    IDSetSwitch(&AbortSlewSP, NULL);
    IDSetSwitch(&ParkSP, NULL);
    IDSetText(&ObjectTP, NULL);
    IDSetText(&CommandTP, NULL);
    IDSetText(&AlignementIPP, NULL);
    IDSetNumber(&EquatorialCoordsWNP, NULL);
    IDSetNumber(&PressionTempWNP, NULL);
    IDSetNumber(&GeographicCoordsWNP, NULL);
    IDSetNumber(&ActualisationNP1, NULL);
    IDSetNumber(&ActualisationNP2, NULL);
}






/**************************************************************************************
** En cas de changement de texte dans la boîte de dialogue (par exemple : changement du
** nom de l'objet) alors suivre l'objet...
** Cette fonction n'est pas utilisée
** TODO: faut-il proposer cette fonction depuis la boîte alors qu'il est déjà
** possible de le faire depuis  Kstars et BAOcontrol ?
**
**
***************************************************************************************/

void BAO::ISNewText (const char *dev, const char *name, char *texts[], char *names[], int n)
{
    // Ignore if not ours
    if (strcmp (dev, mydev))
        return;

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! Please connect before issuing any commands.");
        reset_all_properties();
        return;
    }

    // ===================================
    // Object Name
    // ===================================
    if (!strcmp (name, ObjectTP.name))
    {
        if (IUUpdateText(&ObjectTP, texts, names, n) < 0)
            return;

        ObjectTP.s = IPS_OK;
        IDSetText(&ObjectTP, NULL);
        return;
    }


    // ===================================
    // Adresse IP de l'antenne qu'il faut aligner
    // ===================================
    if (!strcmp (name, AlignementIPP.name))
    {
        if (IUUpdateText(&AlignementIPP, texts, names, n) < 0)
            return;

        IText *eqp = IUFindText (&AlignementIPP, names[0]);

        int i;

        if (eqp == &AlignementIP[0])
        {
            string ip = texts[0];

            bool IPvalide = false;

            // est-ce que l'adresse ip correspond à l'adresse d'une antenne connectée ?

            for ( i=1; i<SocketsNumber; i++) if (Sockets[i].Connected)
                {
                    if (Sockets[i].IP == ip) {
                        IPvalide = true;
                        break;
                    }
                }

            if ( IPvalide )
            {
                targetAlignmentIP = i;
                AlignementIPP.s = IPS_OK ;
                IDSetText(&AlignementIPP, "L antenne %s est maintenant prete pour l alignement", Sockets[targetAlignmentIP].IP.c_str() );

                Sockets[targetAlignmentIP].AlignementAntenne->AlignementEnCours = 1;

                Sockets[targetAlignmentIP].AlignementAntenne->Matrice_ok = false;
            }
            else
            {
                targetAlignmentIP = -1;
                AlignementIPP.s = IPS_ALERT ;
                IDSetText(&AlignementIPP, "L antenne indiquee n est pas connectee !");
            }
        }
        return;
    }


    // ===================================
    // Commands
    // ===================================
    if (!strcmp (name, CommandTP.name))
    {
        if (IUUpdateText(&CommandTP, texts, names, n) < 0)
            return;

        IText *eqp = IUFindText (&CommandTP, names[0]);


        if (eqp == &CommandT[0])
        {
            char chaine[1000];

            bool result = false;

            strcpy(chaine, texts[0]);

            if (chaine[0] == 'G')
            {
                //commande goto

                for (int i = 1; i<=strlen(chaine); i++) chaine[i-1]=chaine[i];

                IDLog("%s", chaine);

                for (int i = 1; i<SocketsNumber; i++ )
                {
                    if (Sockets[i].Connected)
                    {
                        result = COMMANDE(i, (char*)"G", chaine);
                    }
                }
            }
            else if (chaine[0] == 'X')
            {
                if (targetAlignmentIP != -1)
                {
                    IDSetText(&CommandTP, "Position:/%05d/%05d\n", Sockets[targetAlignmentIP].Pos.x, Sockets[targetAlignmentIP].Pos.y );

                    return;
                }
            }
            else if (chaine[0] == 'D')
            {
                if (targetAlignmentIP != -1)
                {
                    IDSetText(&CommandTP, "Deltas:/%05d/%05d\n", delta_az, delta_ha );

                    return;
                }
            }
            else if (chaine[0] == 'M')
            {
                IDSetText(&CommandTP, "Calcul des matrices de correction\n" );

                if  (Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections >=3 )
                {
                    Sockets[targetAlignmentIP].AlignementAntenne->AlignementEnCours = 0;
                    Sockets[targetAlignmentIP].AlignementAntenne->CalculerMatriceCorrection(targetRA, targetDEC);
                }
                else
                {
                    Sockets[targetAlignmentIP].AlignementAntenne->Identity();

                    Sockets[targetAlignmentIP].AlignementAntenne->Matrice_ok = false;
                }
            }
            else if (chaine[0] == 'O')
            {
                IDSetText(&CommandTP, "Optimisation de la geometrie de l antenne\n" );

                if (targetAlignmentIP != -1)
                {
                    Sockets[targetAlignmentIP].AlignementAntenne->OptimisationGeometrieAntenne(true);

                    IDSetSwitch(&AlignementOkP, "Les parametres de l alignement ont ete sauvegardes apres optimisation");

                    Sockets[targetAlignmentIP].AlignementAntenne->EnregistrementParametresAlignement(
                        Sockets[targetAlignmentIP].IP,
                        "/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg");
                }
            }
            else if (chaine[0] == 'C')
            {
                if (targetAlignmentIP != -1)
                {
                    stringstream os;

                    double a, b;

                    os << "Corrections:";

                    Azimut( targetRA * 15.0 * Pidiv180,
                            targetDEC * Pidiv180, &a, &b);
                    os << a << ",";
                    os << b << ",";

                    os << "/";

                    for (int i=0; i<Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections; i++)
                    {
                        if (Sockets[targetAlignmentIP].AlignementAntenne->SelectionnePourCalculMatrice[i])
                        {
                            os << i << ",";
                        }
                    }

                    os << "/";

                    for (int i=0; i<Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections; i++)
                    {
                        if (Sockets[targetAlignmentIP].AlignementAntenne->tsl[i] != 0.0)
                        {
                            Azimut( Sockets[targetAlignmentIP].AlignementAntenne->ad[i]
                                    - Sockets[targetAlignmentIP].AlignementAntenne->tsl[i] + GetTSL(),
                                    Sockets[targetAlignmentIP].AlignementAntenne->de[i], &a, &b);
                            os << a << ",";
                            os << b << ",";

                            Azimut( Sockets[targetAlignmentIP].AlignementAntenne->delta_ad[i]
                                    - Sockets[targetAlignmentIP].AlignementAntenne->tsl[i] + GetTSL(),
                                    Sockets[targetAlignmentIP].AlignementAntenne->delta_de[i]
                                    , &a, &b);

                            os << a << ",";
                            os << b << ",";
                        }
                    }

                    os << "/";

                    for (int i=0; i<Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections; i++)
                    {
                        if (Sockets[targetAlignmentIP].AlignementAntenne->tsl[i] != 0.0 && Sockets[targetAlignmentIP].AlignementAntenne->Matrice_ok)
                        {
                            Azimut( Sockets[targetAlignmentIP].AlignementAntenne->ad[i]
                                    - Sockets[targetAlignmentIP].AlignementAntenne->tsl[i] + GetTSL(),
                                    Sockets[targetAlignmentIP].AlignementAntenne->de[i], &a, &b);

                            Coord vect, r;


                            if (Sockets[targetAlignmentIP].AlignementAntenne->MethodeAlignement == SIMPLE)
                            {

                                b = Sockets[targetAlignmentIP].AlignementAntenne->Motor2Alt(Sockets[targetAlignmentIP].AlignementAntenne->Alt2Motor2(b * N180divPi)) * Pidiv180;

                                vect.x = cos(a) * cos(b);
                                vect.y = sin(a) * cos(b);
                                vect.z = sin(b);


                                // Application de la matrice SIMPLE

                                Sockets[targetAlignmentIP].AlignementAntenne->AppliquerMatriceCorrectionSimple(&r, vect);


                                // Là on doit repasser des coordonnées rectangulaires
                                // aux coordonnées sphériques

                                if (r.x != 0.0)
                                {
                                    a = atan( r.y / r.x );

                                    if (r.x < 0) a += Pi;
                                }
                                else a = Pidiv2;

                                b = asin(r.z);
                            }
                            else
                            {
                                double c, d;

                                Sockets[targetAlignmentIP].AlignementAntenne->AzHa2XY(a, b, &c, &d);
                                vect.x = c;
                                vect.y = d;
                                vect.z = 1.0;

                                AfficherLog("az=%5.2f ha=%5.2f x=%5.2f y=%5.2f\n", a, b, c, d);
                                // Application de la matrice SIMPLE

                                Sockets[targetAlignmentIP].AlignementAntenne->AppliquerMatriceCorrectionAffine(&r, vect);


                                AfficherLog("r.x=%5.2f r.y=%5.2f d=%5.2f\n", r.x, r.y,sqrt(r.x*r.x+r.y*r.y));
                                a = atan(r.y/r.x);
                                if (r.x < 0) a += Pi;

                                b= Pidiv2-(sqrt(r.x*r.x+r.y*r.y));
                            }


                            os << a << ",";
                            os << b << ",";
                        }
                    }

                    IDSetText(&CommandTP, "%s", os.str().c_str() );

                    return;
                }
            }
            else
            {
                for (int i = 1; i<SocketsNumber; i++ )
                {
                    if (Sockets[i].Connected)
                    {
                        result = COMMANDE(i, chaine, (char*)"");
                    }
                }
            }

            (result) ? CommandTP.s = IPS_OK : CommandTP.s = IPS_ALERT;

            IDSetText(&CommandTP, NULL);
        }
        return;
    }
}



/**************************************************************************************
** En cas de changement d'une valeur numérique dans la boîte de dialogue Indi
** Exemple : longitude, latitude, ar, dec etc...) -> prendre en compte les modifications
**
** const char *dev  contient le nom du dispositif recevant le message (ici indi_BAO)
** const char *name reçoit  le nom de la rubrique modifiée par l'utilisateur
** (ex : les coordonnées géographiques, les coordonnées horaires de l'objet etc...)
** double names[] contient la liste de tous les champs modifiables dans chaque rubrique
** (ex : longitude et latitude dans la rubrique coordonnées géographiques
** char *values[] contient toutes les valeurs (numériques ou non) mais toujours exprimées
** sous la forme d'une chaîne de caractères
**
***************************************************************************************/

void BAO::ISNewNumber (const char *dev, const char *name, double values[], char *names[], int n)
{

    // Ignore if not ours

    if (strcmp (dev, mydev))
        return;

    // Si pas de connexion -> on sort

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! BAO is offline. Please connect before issuing any commands.");

        reset_all_properties();

        return;
    }


    // ===================================
    // Geographic  Coords
    // ===================================

    if (!strcmp (name, GeographicCoordsWNP.name))
    {
        // l'utilisateur a modifié une des coordonnées géographiques de la boite indi

        int i = 0, nset = 0;

        // En cas d'erreur, on doit pouvoir sortir avec des paramètres par défaut
        // ici la latitude et la longitude valent 0.0

        Latitude  = 0.0;
        Longitude = 0.0;

        // pour ce vecteur, on sait qu'il y a n éléments possibles
        // Exemple pour la rubrique coordonnées géographique de la boîte indi
        // il y a deux éléments : la longitude et la latitude
        // pour la rubrique alignement, il y a en a trois : simple, affine et taki

        for (i = 0; i < n; i++)
        {
            // pour chacun d'entre eux, on regarde si le nom passé en argument de la fonction (*name)
            // correspond bien à l'un des éléments du vecteur GeographicCoordsWNP

            INumber *eqp = IUFindNumber (&GeographicCoordsWNP, names[i]);

            if (eqp == &GeographicCoordsWN[0])
            {
                // ici on a identifié la zone latitude de la boîte de dialogue
                // Value[i] contient la chaîne de caractère correspondante dans la boîte
                // C'est la valeur saisie par l'utilisateur

                Latitude = values[i];

                // On doit vérifier que la latitude est dans un intervalle correct

                nset += Latitude >= -90.0 && Latitude <= 90.0;

                // on convertit en radians

                Latitude *= Pidiv180;
            }
            else if (eqp == &GeographicCoordsWN[1])
            {
                // ici on a identifié une modification dans la rubrique longitude

                Longitude = values[i];

                // dans l'intervalle ?

                nset += Longitude >= 0.0 && Longitude <= 360.0;

                // en radians

                Longitude *= -Pidiv180;
            }
        }

        // Si la longitude et la latitude sont correctes
        // on envoie les coordonnées à la classe Astro
        if (nset == 2)
        {
            // Vérification
            // AfficherLog("Geographic : RA %5.2f - DEC %5.2f\n", Latitude, Longitude);

            // nset vaut 2, nous sommes donc sûrs qu'il n'y a pas de problème dans les valeurs
            // saisies par l'utilisateur
            // voir le code plus haut pour comprendre...

            // On change la couleur de la "diode" de la rubrique coordonnées
            // géographiques de la boîte en vert

            GeographicCoordsWNP.s = IPS_OK;

            // pas de message d'erreur dans la boîte

            IDSetNumber(&GeographicCoordsWNP, NULL);
        }
        else
        {
            // quelque chose cloche
            // peut-être l'une des valeurs saisies par l'utilisateur n'est-elle pas dans
            // le bon intervalle ? ex : lat = 150°

            // on change la couleur de la diode

            GeographicCoordsWNP.s = IPS_ALERT;

            // on affiche un message d'erreur

            IDSetNumber(&GeographicCoordsWNP, "Latitude or Longitude missing or invalid");

            // on fixe arbitrairement les valeurs à 0

            Latitude  = 0.0;
            Longitude = 0.0;
        }

        // c'est bon. On peut donc transmettre les nouvelles valeurs à la classe astro
        // qui utilisera ces informations pur calculer l'azimut des objets et le temps sidéral local en particulier...

        DefinirLongitudeLatitude(Longitude, Latitude);

        return;
    }


    // ===================================
    // Equatorial Coords
    // ===================================
    if (!strcmp (name, EquatorialCoordsWNP.name))
    {
        int i = 0, nset = 0;

        double newRA =0.0, newDEC =0.0;

        // nous avons le même principe de fonctionnement que pour les coordonnées géographiques

        for (i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&EquatorialCoordsWNP, names[i]);

            if (eqp == &EquatorialCoordsWN[0])
            {
                // on a compris que l'utilisateur avait changé l'ascension droite de l'objet

                // on affecte la nouvelle valeur à newRA

                newRA = values[i];

                // Est-ce que cette valeur est dans le bon intervalle ?

                nset += newRA >= 0 && newRA <= 24.0;
            }
            else if (eqp == &EquatorialCoordsWN[1])
            {
                // même chose pour la déclinaison
                newDEC = values[i];

                nset += newDEC >= -90.0 && newDEC <= 90.0;
            }
        }


        // si les coordonnées de l'objet sont correctes

        if (nset == 2)
        {
            char RAStr[32], DecStr[32];
            double targetAZ, targetAlt;

            // On garde une trace des nouvelles coordonnées saisies par l'utilisateur

            targetRA  = newRA;
            targetDEC = newDEC;

            // Réinitialisation des deltas

            delta_az = 0;
            delta_ha = 0;

            if (targetAlignmentIP != -1 && Sockets[targetAlignmentIP].AlignementAntenne->AlignementEnCours != 0)
            {
                Sockets[targetAlignmentIP].AlignementAntenne->delta_ad[Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections] =
                    targetRA * 15.0 * Pidiv180;

                Sockets[targetAlignmentIP].AlignementAntenne->delta_de[Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections] =
                    targetDEC * Pidiv180;
            }

            for (int i=0; i<SocketsNumber; i++) if (Sockets[i].Connected)
                {
                    if ( Sockets[i].AlignementAntenne->AlignementEnCours == 0 )
                    {
                        Sockets[i].AlignementAntenne->CalculerMatriceCorrection(targetRA, targetDEC);
                    }
                }



            // on les affiche dans les logs

            fs_sexa(RAStr, newRA, 2, 3600);
            fs_sexa(DecStr, newDEC, 2, 3600);

            AfficherLog("We received JNow RA %s - DEC %s\n", RAStr, DecStr);

            // on convertit les coordonnées équatoriales de la zone du ciel observée
            // en unités de codeurs des moteurs

            ADDEC2Motor(newRA, newDEC);

            // on déclenche le goto

            if (process_coords() == false)
            {
                EquatorialCoordsWNP.s = IPS_ALERT;

                IDSetNumber(&EquatorialCoordsWNP, NULL);
            }
        }
        else
        {
            EquatorialCoordsWNP.s = IPS_ALERT;

            IDSetNumber(&EquatorialCoordsWNP, "Error ! RA or Dec missing or invalid");
        }

        return;
    }


    // ===================================
    // Pression, Temperature
    // ===================================
    if (!strcmp (name, PressionTempWNP.name))
    {
        int i = 0, nset = 0;

        double newPression =0.0, newTemperature =0.0;

        // nous avons le même principe de fonctionnement que pour les coordonnées géographiques

        for (i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&PressionTempWNP, names[i]);

            if (eqp == &PressionTempWN[0])
            {
                // on a compris que l'utilisateur a changé l'ascension droite de l'objet

                // on affecte la nouvelle valeur à newRA

                newPression = values[i];

                // Est-ce que cette valeur est dans le bon intervalle ?

                nset += newPression >= 0.0 && newPression <= 1500.0;
            }
            else if (eqp == &PressionTempWN[1])
            {
                //même chose pour la déclinaison
                newTemperature = values[i];

                nset += newTemperature >= -50.0 && newTemperature <= 50.0;
            }
        }


        // si les coordonnées de l'objet sont correctes

        if (nset == 2)
        {
            PressionTempWNP.s = IPS_OK;

            IDSetNumber(&PressionTempWNP, NULL);

            Pression = newPression;
            Temp = newTemperature;
        }
        else
        {
            PressionTempWNP.s = IPS_ALERT;

            IDSetNumber(&PressionTempWNP, "Error ! Bad values for pression or temperature");
        }

        return;
    }


    // ===================================
    // Actualisation
    // ===================================
    if (!strcmp (name, ActualisationNP1.name))
    {
        // on a ici exactement le même fonctionnement que précédemment
        // on règle ici les valeurs des délais entre deux actualisations
        // de la position en mode transit et tracking
        // Retourne un message d'erreur si les durées ne sont pas exprimées
        // dans un intervalle > 0 et < 3600

        double newAct1 = 1.0;

        for (int i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&ActualisationNP1, names[i]);

            if (eqp == &ActualisationN1[0])
            {
                newAct1 = values[i];

                if ( newAct1 >= 0.0 && newAct1 <= 3600.0 )
                {
                    ActualisationTMTransit = newAct1;

                    ActualisationNP1.s = IPS_OK;

                    IDSetNumber(&ActualisationNP1, NULL);
                }
                else
                {
                    ActualisationNP1.s = IPS_ALERT;

                    IDSetNumber(&ActualisationNP1, "Error ! Delay invalid");
                }
            }
        }
    }

    if (!strcmp (name, ActualisationNP2.name))
    {
        double newAct2 = 1.0;

        for (int i = 0; i < n; i++)
        {
            INumber *eqp = IUFindNumber (&ActualisationNP2, names[i]);

            if (eqp == &ActualisationN2[0])
            {
                newAct2 = values[i];

                if ( newAct2 >= 0.0 && newAct2 <= 3600.0 )
                {
                    ActualisationTMTracking = newAct2;

                    ActualisationNP2.s = IPS_OK;

                    IDSetNumber(&ActualisationNP2, NULL);
                }
                else
                {
                    ActualisationNP2.s = IPS_ALERT;
                    IDSetNumber(&ActualisationNP2, "Error ! Delay invalid");
                }
            }
        }
    }
}



/**************************************************************************************
** L'utilisateur clique sur l'un des boutons de la boîte Indi
** Même observation que pour la procédure précédente
**
***************************************************************************************/

void BAO::ISNewSwitch (const char *dev, const char *name, ISState *states, char *names[], int n)
{
    // ignore if not ours //
    if (strcmp (mydev, dev))
        return;



    double az, ha, ad, de;

    ad = targetRA * 15.0 * Pidiv180;

    de = targetDEC * Pidiv180;

    CalculTSL();

    Azimut(ad, de, &az, &ha);

    ha = RefractionAtmospherique(ha) ;




    // ===================================
    // Connect Switch
    // ===================================
    if (!strcmp (name, ConnectSP.name))
    {
        if (IUUpdateSwitch(&ConnectSP, states, names, n) < 0)
            return;

        connect_telescope();

        return;
    }

    if (is_connected() == false)
    {
        IDMessage(mydev, "Error ! BAORadio is offline. Please connect before issuing any commands.");
        reset_all_properties();
        return;
    }

    // ===================================
    // Coordinate Set
    // ===================================
    if (!strcmp(name, OnCoordSetSP.name))
    {
        if (IUUpdateSwitch(&OnCoordSetSP, states, names, n) < 0)
            return;

        currentSet = get_switch_index(&OnCoordSetSP);
        OnCoordSetSP.s = IPS_OK;
        IDSetSwitch(&OnCoordSetSP, NULL);
    }

    // ===================================
    // Alignment Set
    // ===================================
    if (!strcmp(name, AlignmentSP.name))
    {
        if (IUUpdateSwitch(&AlignmentSP, states, names, n) < 0)
            return;

        MethodeAlignement = get_switch_index(&AlignmentSP) + 1 ;

        AlignmentSP.s = IPS_OK;
        IUResetSwitch(&AlignmentSP);
        IDSetSwitch(&AlignmentSP, NULL);

        for (int i=0; i<MAXHOSTNAME; i++)
        {
            switch (MethodeAlignement)
            {
            case 1 :
                Sockets[i].AlignementAntenne->MethodeAlignement = SIMPLE;
                break;
            case 2 :
                Sockets[i].AlignementAntenne->MethodeAlignement = AFFINE;
                break;
            case 3 :
                Sockets[i].AlignementAntenne->MethodeAlignement = TAKI;
                break;
            }
        }

        return;
    }

    // ===================================
    // Alignment Speed
    // ===================================
    if (!strcmp(name, RaquetteNP.name))
    {
        if (IUUpdateSwitch(&RaquetteNP, states, names, n) < 0)
            return;

        RaquetteNP.s = IPS_OK;

        switch (get_switch_index(&RaquetteNP))
        {
        case 0 :
            IUResetSwitch(&RaquetteNP);
            IDSetSwitch(&RaquetteNP, "La vitesse de la raquette est fixee a 1x");
            VitesseRaquette = 1;
            break;
        case 1 :
            IUResetSwitch(&RaquetteNP);
            IDSetSwitch(&RaquetteNP, "La vitesse de la raquette est fixee a 10x");
            VitesseRaquette = 10;
            break;
        }

        return;
    }

    // ===================================
    // Alignment Reset
    // ===================================
    if (!strcmp(name, AlignementResetP.name))
    {
        if (IUUpdateSwitch(&AlignementResetP, states, names, n) < 0)
            return;

        if ( targetAlignmentIP < 0 )
        {
            AlignementResetP.s = IPS_ALERT;
            IDSetSwitch(&AlignementResetP, "Veuillez indiquer l adresse IP de l antenne a calibrer");

            return;
        }

        AlignementResetP.s = IPS_OK;
        IDSetSwitch(&AlignementResetP, "Les parametres de l alignement de l antenne ont ete reinitialises");

        Sockets[targetAlignmentIP].AlignementAntenne->InitAlignement();

        sleep(1);

        Sockets[targetAlignmentIP].AlignementAntenne->EnregistrementParametresAlignement(
            Sockets[targetAlignmentIP].IP,
            "/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg");


        IUResetSwitch(&AlignementResetP);
        AlignementResetP.s = IPS_IDLE;
        IDSetSwitch(&AlignementResetP, NULL);

        return;
    }



    // ===================================
    // Alignment buttons
    // ===================================
    if (!strcmp(name, AlignementAltpP.name))
    {
        if (IUUpdateSwitch(&AlignementAltpP, states, names, n) < 0)
            return;

        delta_ha += VitesseRaquette;

        az  = VerifAngle(az + (double)delta_az * PasDeltaAz);

        ha += (double)delta_ha * PasDeltaHa;

        //   ISPoll();

        AlignementAltpP.s = IPS_IDLE;

        IDSetSwitch(&AlignementAltpP, "Delta en azimut = %s  delta en hauteur = %s  az = %s  ha = %s (Vitesse raquette = %d)\n",
                    DHMS(delta_az * PasDeltaAz * N180divPi, false).c_str(),
                    DHMS(delta_ha * PasDeltaHa * N180divPi, false).c_str(),
                    DHMS(az * N180divPi, false).c_str(),
                    DHMS(ha * N180divPi, false).c_str(),
                    VitesseRaquette);
        IUResetSwitch(&AlignementAltpP);

        return;
    }

    if (!strcmp(name, AlignementAltnP.name))
    {
        if (IUUpdateSwitch(&AlignementAltnP, states, names, n) < 0)
            return;

        delta_ha -= VitesseRaquette;

        az  = VerifAngle(az + (double)delta_az * PasDeltaAz);

        ha += (double)delta_ha * PasDeltaHa;

        //   ISPoll();

        AlignementAltnP.s = IPS_IDLE;

        IDSetSwitch(&AlignementAltnP, "Delta en azimut = %s  delta en hauteur = %s  az = %s  ha = %s (Vitesse raquette = %d)\n",
                    DHMS(delta_az * PasDeltaAz * N180divPi, false).c_str(),
                    DHMS(delta_ha * PasDeltaHa * N180divPi, false).c_str(),
                    DHMS(az * N180divPi, false).c_str(),
                    DHMS(ha * N180divPi, false).c_str(),
                    VitesseRaquette);
        IUResetSwitch(&AlignementAltnP);

        return;
    }

    if (!strcmp(name, AlignementAzP.name))
    {
        if (IUUpdateSwitch(&AlignementAzP, states, names, n) < 0)
            return;

        switch (get_switch_index(&AlignementAzP))
        {
        case 0 :
            delta_az -= VitesseRaquette;
            break;
        case 1 :
            delta_az += VitesseRaquette;
            break;
        }

        az  = VerifAngle(az + (double)delta_az * PasDeltaAz);

        ha += (double)delta_ha * PasDeltaHa;

        // ISPoll();

        AlignementAzP.s = IPS_IDLE;

        IDSetSwitch(&AlignementAzP, "Delta en azimut = %s  delta en hauteur = %s  az = %s  ha = %s (Vitesse raquette = %d)\n",
                    DHMS(delta_az * PasDeltaAz * N180divPi, false).c_str(),
                    DHMS(delta_ha * PasDeltaHa * N180divPi, false).c_str(),
                    DHMS(az * N180divPi, false).c_str(),
                    DHMS(ha * N180divPi, false).c_str(),
                    VitesseRaquette);
        IUResetSwitch(&AlignementAzP);

        return;
    }

    if (!strcmp(name, AlignementOkP.name))
    {
        if (IUUpdateSwitch(&AlignementOkP, states, names, n) < 0)
            return;

        if ( targetAlignmentIP < 0 )
        {
            AlignementOkP.s = IPS_ALERT;
            IDSetSwitch(&AlignementOkP, "Veuillez indiquer l adresse IP de l antenne a calibrer");

            return;
        }

        AlignementOkP.s = IPS_OK;
        IDSetSwitch(&AlignementOkP, NULL);

        switch (get_switch_index(&AlignementOkP))
        {
            // Validation de l'alignement en cours...

        case 0 :
        {
            float a, b, c;
            double az, ha, ad, de, ad2, de2;

            int num;

            num = Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections;

            ad = targetRA * 15.0 * Pidiv180;

            Sockets[targetAlignmentIP].AlignementAntenne->ad[num] = ad;

            de = targetDEC * Pidiv180;

            Sockets[targetAlignmentIP].AlignementAntenne->de[num] = de;

            CalculTSL();

            Sockets[targetAlignmentIP].AlignementAntenne->tsl[num] = GetTSL();

            Azimut(ad, de, &az, &ha);

            //ha = RefractionAtmospherique(ha) ;

            az  = VerifAngle(az + (double)delta_az * PasDeltaAz);

            ha += (double)delta_ha * PasDeltaHa;

            AzHa2ADDe(az, ha, &ad2, &de2);


            Sockets[targetAlignmentIP].AlignementAntenne->delta_ad[num] = ad2;

            Sockets[targetAlignmentIP].AlignementAntenne->delta_de[num] = de2;

            stringstream os;

            AfficherLog("Alignement de l antenne %s avec l objet situe en RA %s - DEC %s\n",
                        Sockets[targetAlignmentIP].IP.c_str(),
                        DHMS(targetRA * 15.0, true).c_str(),
                        DHMS(targetDEC, false     ).c_str());

            AfficherLog("(ad1=%s de1=%s) -> (ad2=%s de2=%s)\n",
                        DHMS(ad  * N180divPi, true ).c_str(),
                        DHMS(de  * N180divPi, false).c_str(),
                        DHMS(ad2 * N180divPi, true ).c_str(),
                        DHMS(de2 * N180divPi, false).c_str());

            AfficherLog("delta_az=%s delta_ha=%s\n",
                        DHMS(delta_az * PasDeltaAz, false).c_str(),
                        DHMS(delta_ha * PasDeltaHa, false).c_str());

            Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections++;

            delta_az = 0;
            delta_ha = 0;

            IDSetSwitch(&AlignementOkP, "Les parametres de l alignement ont ete valides");

        }
        break;

        // Sauvegarde de l'alignement en cours...

        case 1 :
        {
            if  (Sockets[targetAlignmentIP].AlignementAntenne->nbrcorrections >=3 )
            {
                // IDLog("Calcul matrice\n", true);
                Sockets[targetAlignmentIP].AlignementAntenne->AlignementEnCours = 0;
                Sockets[targetAlignmentIP].AlignementAntenne->CalculerMatriceCorrection(targetRA, targetDEC);
            }
            else
            {
                Sockets[targetAlignmentIP].AlignementAntenne->Identity();

                Sockets[targetAlignmentIP].AlignementAntenne->Matrice_ok = false;
            }

            IDSetSwitch(&AlignementOkP, "Les parametres de l alignement ont ete sauvegardes");

            Sockets[targetAlignmentIP].AlignementAntenne->EnregistrementParametresAlignement(
                Sockets[targetAlignmentIP].IP,
                "/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg");
        }
        break;
        }


        AlignementOkP.s = IPS_IDLE;
        IUResetSwitch(&AlignementOkP);
        IDSetSwitch(&AlignementOkP, NULL);

        return;
    }


    // ===================================
    // Abort slew
    // ===================================
    if (!strcmp (name, AbortSlewSP.name))
    {
        Abort = true;

        IUResetSwitch(&AbortSlewSP);

        if (EquatorialCoordsWNP.s == IPS_OK)
        {
            AbortSlewSP.s = IPS_OK;
            EquatorialCoordsWNP.s = IPS_IDLE;
            ObjectTP.s = IPS_IDLE;
            CommandTP.s = IPS_IDLE;

            IDSetSwitch(&ConnectSP, "Envoi de la commande Abort\n");
            IDSetNumber(&EquatorialCoordsWNP, NULL);
            IDSetText(&ObjectTP, NULL);
            IDSetText(&CommandTP, NULL);
        }

        return;
    }


    // ===================================
    // Park
    // ===================================
    if (!strcmp (name, ParkSP.name))
    {
        Park=true;

        IUResetSwitch(&ParkSP);

        if (EquatorialCoordsWNP.s == IPS_OK)
        {
            AbortSlewSP.s = IPS_OK;
            EquatorialCoordsWNP.s = IPS_IDLE;
            ObjectTP.s = IPS_IDLE;
            CommandTP.s = IPS_IDLE;

            IDSetSwitch(&ConnectSP, "Envoi de la commande Park\n");
            IDSetNumber(&EquatorialCoordsWNP, NULL);
            IDSetText(&ObjectTP, NULL);
            IDSetText(&CommandTP, NULL);
        }

        return;
    }
}



/**************************************************************************************
** Gestion du thread
** permet de suivre la connexion/déconnexion des antennes toutes les secondes
**
** l'utilisation d'un thread permet de contourner le problème de la fonction accept
** qui est bloquante.
** L'adresse IP de l'antenne qui a effectué la connexion est consignée dans la variable IP
** de la structure Sockets. L'état de l'antenne (connected) est placée à true
**
***************************************************************************************/

void *BAO::pThreadSocket ()
{
    do
    {
        try
        {
            server.accept( Sockets[SocketsNumber].new_sock );

            Sockets[SocketsNumber].IP = server.recupip(Sockets[SocketsNumber].new_sock);

            Sockets[SocketsNumber].AlignementAntenne->MethodeAlignement = MethodeAlignement;

            Sockets[SocketsNumber++].Connected = true;

            InitThreadOK = true;
        }
        catch ( SocketException& e )
        {
            /*AfficherLog("Indi_BAO, pThreadSocket exception : ");
            AfficherLog(e.description().c_str());
            AfficherLog("\n");*/
        }

        sleep(1); // faire une pause pour éviter de consommer trop de temps CPU -> à vérifier
    }
    while (!Exit);

    pthread_exit (0);
}



/**************************************************************************************
** Astuce pour lancer le thread depuis la classe BAO
**
***************************************************************************************/

void* LancementThread(BAO * appli)
{
    appli->pThreadSocket();

    return 0;
}


/**************************************************************************************
** Extraction de la position de l'antenne après l'envoi de la commande P
** Le retour de la commande P est POSITION/valeur_az/valeur_alt/
** Ce retour est envoyé dans la chaîne str
** ExtractPosition retourne une structure Position contenant Valeur_az et Valeur_alt
** Ex: ExtractPosition("POSITION/0100/0001/", result)  retourne result.x=100 et result.y=1
**
***************************************************************************************/

bool BAO::ExtractPosition(string str, Position *result)
{
    string str2;

    int pos = str.find("/");

    if (pos != string::npos)
    {
        str2 = str.substr(pos + 1);

        pos = str2.find("/");

        if (pos != string::npos)
        {
            result->x = atol( str2.substr(0, pos).c_str() );

            result->y = atol( str2.substr(pos + 1).c_str() );

            return true;
        }
    }

    AfficherLog((str +" failed !\n").c_str());

    return false;
}


/************************************************************************************
** Cette procédure, après avoir appliqué la matrice de rotation aux coordonnées de l'objet
** visées par l'utilisateur, effectue les conversions en unités codeurs des paraboles
** (càd en nb de tours des deux axes moteurs depuis la position PARK)
**
************************************************************************************/

void BAO::ADDEC2Motor(double newRA, double newDEC)
{
    double targetAz;
    double targetAlt;
    double newRA2  = newRA * 15.0 * Pidiv180;
    double newDEC2 = newDEC * Pidiv180;
    char AzStr[32];
    char AltStr[32];
    Coord result, vect;

    
    // Pour toutes les antennes connectées

    for (int i=0; i<SocketsNumber; i++)
    {
        // On vérifie qu'elles sont encore connectées

        if (Sockets[i].Connected)
        {
            newRA  = newRA2;
            newDEC = newDEC2;

            // Dans les modes d'alignement AFFINE et TAKI, on applique la matrice de correction
            // sur les coordonnées horaires de l'objet visé

            // il faut vérifier auparavant que la matrice de rotation est disponible
            // pour l'antenne i

            if  (Sockets[i].AlignementAntenne->Matrice_ok)
            {
                // On est bien dans le cas de l'alignement AFFINE ou Taki

                /* if ( Sockets[i].AlignementAntenne->MethodeAlignement ==  AFFINE ||
                         Sockets[i].AlignementAntenne->MethodeAlignement ==  TAKI )
                 {
                     // On crée un vecteur avec pour coordonnées
                     // x = Angle horaire de l'objet visé
                     // y = déclinaison de l'objet
                     // z = 1.0
                     // Voir la documentation pour plus d'explications sur le calcul affine/taki

                     vect.x = VerifAngle( newRA2 + GetTSL() );
                     vect.y = newDEC2;
                     vect.z = 1.0;

                     // Message pour l'utilisateur et les logs

                     AfficherLog("Application de la matrice AFFINE/TAKI\n", true);
                     AfficherLog("Coordonnees initiales: AD = %s  Dec = %s\n", DHMS(newRA2*N180divPi, true ).c_str(),
                                 DHMS(newDEC2*N180divPi, false ).c_str());

                     // On applique la matrice de transformation

                     switch (Sockets[i].AlignementAntenne->MethodeAlignement)
                     {
                     case AFFINE :
                         Sockets[i].AlignementAntenne->AppliquerMatriceCorrectionAffine(&result, vect);
                         break;
                     case TAKI   :
                         Sockets[i].AlignementAntenne->AppliquerMatriceCorrectionTaki( &result, vect);
                         break;
                     }

                     // On récupère les nouvelles coordonnées. z ne nous intéresse pas

                     newRA  = VerifAngle(result.x - GetTSL());
                     newDEC = result.y;

                     AfficherLog("Coordonnees finales: AD = %s  Dec = %s\n", DHMS(newRA*N180divPi, true ).c_str(),
                                 DHMS(newDEC*N180divPi, false ).c_str());
                 }*/
            }
            else
            {
                if (Sockets[i].AlignementAntenne->AlignementEnCours != 0)
                {
                    // Si la matrice n'est pas prête, c'est que la procédure d'alignement de l'antenne i n'est pas encore achevée.
                    // On se contente ici d'appliquer des deltas en ascension droite et en déclinaison. Ces deltas sont envoyés de BAOcontrol
                    // au driver indi_BAO par l'intermédiaire du fichier AlignementAntennes.cfg
                    // Consultez la documentation pour comprendre le principe de fonctionnement de l'alignement et de la communication
                    // entre les deux programmes

                    newRA  = Sockets[i].AlignementAntenne->delta_ad[Sockets[i].AlignementAntenne->nbrcorrections];
                    newDEC = Sockets[i].AlignementAntenne->delta_de[Sockets[i].AlignementAntenne->nbrcorrections];
		    
		    // if newRA et newDEC = 0 -> on sort
    
                    if ( newRA == 0 && newDEC == 0) return;

                    AfficherLog("Alignement en cours.\n", true);

                    AfficherLog("Nouvelles Coordonnées AD =%s  Dec = %s    Ncorrections=%i\n",
                                DHMS(newRA*N180divPi, true).c_str(),
                                DHMS(newDEC*N180divPi, false).c_str(),Sockets[i].AlignementAntenne->nbrcorrections);
                }
            }

            // Calcule la hauteur et l'azimut de la zone du ciel pointée (en fonction de la date et du lieu d'observation)

            Azimut( newRA, newDEC, &targetAz, &targetAlt);

            // Si la méthode d'alignement est la méthode SIMPLE, alors on doit cette fois-ci appliquer la matrice
            // de correction au coordonnées horizontales et non aux coordonnées horaires comme ce que nous
            // avons vu plus haut

            if  (Sockets[i].AlignementAntenne->Matrice_ok /*&& Sockets[i].AlignementAntenne->MethodeAlignement == SIMPLE*/)
            {
                // Après s'être assurer que nous utilisons la méthode SIMPLE et que la matrice de rotation est ok

                AfficherLog("Application de la matrice SIMPLE (det=%10.8f)\n", Sockets[i].AlignementAntenne->Determinant(), true);
                AfficherLog("Coordonnees initiales: Azi = %s  Haut = %s\n", DHMS(targetAz*N180divPi, false ).c_str(),
                            DHMS(targetAlt*N180divPi, false ).c_str());

                // On constitue un vecteur vect
                // Les coordonnées du vecteur correspondent à la position
                // d'un point défini par une latitude targetAlt et une longitude targetAz
                // sur une sphère de rayon 1.
                // On reconnaît les formules de passage des coordonnées
                // sphériques aux coordonnées rectangulaires

                if (Sockets[i].AlignementAntenne->MethodeAlignement == SIMPLE)
                {
                    //TODO :: à vérifier en détails

                    double targetAltC = Sockets[i].AlignementAntenne->Motor2Alt(Sockets[i].AlignementAntenne->Alt2Motor2(targetAlt * N180divPi)) * Pidiv180;

                    vect.x = cos(targetAz) * cos(targetAltC);
                    vect.y = sin(targetAz) * cos(targetAltC);
                    vect.z = sin(targetAltC);

                    Sockets[i].AlignementAntenne->AppliquerMatriceCorrectionSimple(&result, vect);


                    // Là on doit repasser des coordonnées rectangulaires
                    // aux coordonnées sphériques

                    if (result.x != 0.0)
                    {
                        targetAz = atan( result.y / result.x );

                        if (result.x < 0) targetAz += Pi;
                    }
                    else targetAz = Pidiv2;

                    targetAz  = VerifAngle(targetAz);

                    targetAlt = asin(result.z);
                }
                else
                {
                    Sockets[i].AlignementAntenne->AzHa2XY(targetAz, targetAlt, &vect.x, &vect.y);

                    vect.z = 1.0;



                    Sockets[i].AlignementAntenne->AppliquerMatriceCorrectionAffine(&result, vect);

                    // Là on doit repasser des coordonnées rectangulaires
                    // aux coordonnées sphériques



                    targetAz = atan(result.y/result.x);
                    if (result.x < 0) targetAz += Pi;

                    targetAlt= Pidiv2 - sqrt(result.x*result.x+result.y*result.y);


                    AfficherLog("Coordonnees finales: Azi = %s  Haut = %s\n", DHMS(VerifAngle(targetAz)*N180divPi, false ).c_str(),
                                DHMS(targetAlt*N180divPi, false ).c_str());
                }
            }

            // Correction de la réfraction atmosphérique

            targetAlt = RefractionAtmospherique(targetAlt);

            // On ajoute Pi pour que le sud soit à 0° d'azimut

            targetAz = VerifAngle( targetAz + Pi );

            // On convertit les angles précédents en degrés

            targetAlt *= N180divPi;
            targetAz  *= N180divPi;

            // Affichage dans les logs

            fs_sexa(AzStr, targetAz, 2, 3600);
            fs_sexa(AltStr, targetAlt, 2, 3600);

            AfficherLog("Coordonnees horizontales finales: Azi = %s  Haut= %s\n", AzStr, AltStr);


            // Conversion des deux angles en pas codeurs

            if ( targetAlt < HAUTMIN )
            {
                // L'objet est trop bas ( < 30°). On annule le suivi...

                IDSetSwitch(&OnCoordSetSP, "Erreur ! L objet suivi est situe a moins de 30 deg au-dessus de l horizon. Goto annule.");

                Suivi = false;

                RealisationGoto = false;

                InitAntennes();

                return;
            }
            else
            {
                // Si la hauteur est supérieure à 90°, on doit ajouter 180° à l'azimut et corriger
                // la hauteur en appliquant hauteur=180°-hauteur

                if (targetAlt > 90.0)
                {
                    targetAlt = 180.0 - targetAlt;
                    targetAz += 180.0;
                }

                if ( targetAlt < HAUTMIN ) targetAlt = HAUTMIN;

                // On applique la formule de Marc pour convertir la hauteur en nombre de pas codeur alt

                Sockets[i].TargetPosition.y = (int) Arrondi( Sockets[i].AlignementAntenne->Alt2Motor( targetAlt ) );

                //4000 pas pour 360° sur l'axe az

                Sockets[i].TargetPosition.x = (int) Arrondi( targetAz * (double)NBREPASCODEURSAZ / 360.0);

                // On ajoute les deltas en AZ et en HA pour la raquette hors procédure d'alignement

                Sockets[i].TargetPosition.x += delta_az;

                Sockets[i].TargetPosition.y += delta_ha;

                //TODO : solution à trouver pour maintenir efficacement l'antenne dans l'intervalle

                // est-ce que le transit de l'objet suivi va nous emmener à sortir de l'intervalle [MINAZCODEURS..MAXAZCODEURS] ?

                AutoriserTourComplet = false;

                if ( azmincodeur < MINAZCODEURS ) {
                    azmincodeur += NBREPASCODEURSAZ;
                    azmaxcodeur += NBREPASCODEURSAZ;
                    //  AutoriserTourComplet = true;
                    AfficherLog("azmincodeur < MINAZCODEURS\n");
                }

                if ( azmaxcodeur > MAXAZCODEURS ) {
                    azmincodeur -= NBREPASCODEURSAZ;
                    azmaxcodeur -= NBREPASCODEURSAZ;
                    // AutoriserTourComplet = true;
                    AfficherLog("azmaxcodeur > MAXAZCODEURS\n");
                }

                AfficherLog("Intervalle autorise pour le suivi de l objet en azimut [%i..%i]\n", azmincodeur, azmaxcodeur );
                AfficherLog("Position initiale en Az = %i\n", Sockets[i].TargetPosition.x);

                if ( Sockets[i].TargetPosition.x < azmincodeur ) {
                    Sockets[i].TargetPosition.x += NBREPASCODEURSAZ;
                    AutoriserTourComplet = true;
                    AfficherLog("TargetPosition.x < azmincodeur\n");
                }

                if ( Sockets[i].TargetPosition.x > azmaxcodeur ) {
                    Sockets[i].TargetPosition.x -= NBREPASCODEURSAZ;
                    AutoriserTourComplet = true;
                    AfficherLog("TargetPosition.x > azmaxcodeur\n");
                }

                if ( Sockets[i].TargetPosition.x < MINAZCODEURS ) {
                    Sockets[i].TargetPosition.x += NBREPASCODEURSAZ;
                    AutoriserTourComplet = true;
                    AfficherLog("TargetPosition.x < MINAZCODEURS\n");
                }

                if ( Sockets[i].TargetPosition.x > MAXAZCODEURS ) {
                    Sockets[i].TargetPosition.x -= NBREPASCODEURSAZ;
                    AutoriserTourComplet = true;
                    AfficherLog("TargetPosition.x > MAXAZCODEURS\n");
                }

                if (fabs(Sockets[i].Pos.x - Sockets[i].TargetPosition.x) > MAXAZCODEURS / 2)
                {
                    AutoriserTourComplet = true;
                    AfficherLog("| Pos.x - TargetPosition.x | > MAXAZCODEURS / 2\n");
                }

                // Message de debug

                AfficherLog("Position finale :  Az = %i  Alt = %i\n", Sockets[i].TargetPosition.x, Sockets[i].TargetPosition.y);
            }
        }
    }
}



/************************************************************************************
** Retourne le nombre d'antennes actuellement connectées
**
************************************************************************************/

int BAO::AntennesConnectees()
{
    int num = 0;

    for (int i=1; i<SocketsNumber; i++) if (Sockets[i].Connected) num++;

    return num;
}



/**************************************************************************************
** En cas de problème
** Déconnecter l'antenne utilisant le socket num
***************************************************************************************/

void BAO::DeconnecterSocket(int num)
{
    if (Sockets[num].Connected) AfficherLog("Deconnexion de l antenne : %s\n", Sockets[num].IP.c_str());
    Sockets[num].new_sock.shutdown();
    Sockets[num].Connected = false;
    Sockets[num].IP = "";
}




/**************************************************************************************
** Procédure principale
** Elle est appelée toutes les ms
**
***************************************************************************************/

void BAO::ISPoll()
{
    static int memSocketsNumber = -1;      // Combien y avait-il d'antennes connectées lors de l'appel précédent d'ISPoll ?
    static unsigned int compt   =  0;      // Compteur de la fonction ISPoll

    struct tm date;
    time_t t;
    struct timeval tv;
    struct timezone tz;

    //si pas de connexion avec le serveur d'indi -> on sort

    if (!is_connected()) return;


    // toutes les 100 millisec, on actualise le jour julien
    // le temps sidéral local etc...

    if ( compt%100 == 0 )
    {
        //Récupération de la date et de l'heure

        time(&t);
        date=*gmtime(&t);
        gettimeofday(&tv, &tz);

        double Annee = (double)(date.tm_year + 1900);
        double Mois  = (double)(date.tm_mon + 1);
        double Jour  = (double)date.tm_mday;
        double Heu   = (double)date.tm_hour;
        double Min   = (double)date.tm_min;
        double Sec   = (double)date.tm_sec+tv.tv_usec/1.0E6;
        double UTCP  = 0.0;//(double)date.tm_isdst;

        // On transmet la date et l'heure à la classe Astro

        DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);

        // Puis on calcule le temps sidéral local, le JJ etc.

        CalculTSL();


        // On charge les paramètres de correction des antennes toutes les demi-secondes

        /*
        if ( compt%500 == 0  && tv.tv_usec/1.0E6 > 0.5 )
            {
                for (int i=1; i<SocketsNumber; i++)
                {
                    if (Sockets[i].Connected)
                    {
                        // Si l'antenne i est connectée, on charge les paramètres
                        Sockets[i].AlignementAntenne->TransmettreParametresClasseAstro(Annee, Mois, Jour, Heu, Min, Sec, Longitude, Latitude, Pression, Temp);

                        // AfficherLog("chargement taki\n");
                        //  Sockets[i].AlignementAntenne->ChargementParametresAlignement(Sockets[i].IP, "/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg", targetRA * 15 * Pidiv180, targetDEC * Pidiv180);
                    }
                }
            }*/
    }



    // Il faut que le thread soit actif

    if (InitThreadOK)
    {

        // Nouvelle connexion sur le socket ?

        if (SocketsNumber > memSocketsNumber)
        {
            memSocketsNumber = SocketsNumber;

            IDSetSwitch(&ConnectSP, "Connexion de l antenne %s (Antennes connectees : %i)",
                        Sockets[SocketsNumber-1].IP.c_str(), AntennesConnectees());

            Sockets[SocketsNumber-1].AlignementAntenne->TransmettreParametresClasseAstro(GetAnnee(), GetMois(), GetJour(), GetHeure(), GetMin(), GetSec(), GetLongitude(), GetLatitude(), GetPression(), GetTemperature());


            // Chargement des paramètres d'alignement de l'antenne

            Sockets[SocketsNumber-1].AlignementAntenne->ChargementParametresAlignement(
                Sockets[SocketsNumber-1].IP,
                "/home/" + (string)getenv("USER") + "/AlignementAntennes.cfg",
                targetRA * 15 * Pidiv180,
                targetDEC * Pidiv180);

        }


        /////////////////////////////////////////////////
        // Début des échanges avec les micro-contrôleurs

        // Analyse des réponses des micro-contrôleurs

        for (int i=1; i<SocketsNumber; i++)
        {
            if (Sockets[i].Connected)
            {
                try
                {
                    string reponse, buffereponse;

                    // on récupère la réponse du micro-contrôleur

                    Sockets[i].new_sock >> reponse;

                    // Dans le cas où plusieurs trames seraient arrivées entre deux appels de POLLMS
                    // les traiter successivement

                    // d'où l'intérêt de mettre un '\n' à la fin des trames
                    // pour différencier une trame de la précédente

                    int pos = reponse.find("\n");

                    // S'il y a une réponse

                    while ((pos != string::npos) && (reponse.length() > 1))
                    {
                        // on garde en stock la deuxième partie de la trame
                        // pour un traitement ultérieur

                        buffereponse = reponse.substr(pos + 1);

                        // Partie traitée

                        reponse = reponse.substr(0, pos);

                        // on l'affiche dans le log

                        AfficherLog("Reponse recue de %s : %s\n", Sockets[i].IP.c_str(), reponse.c_str());

                        // On vérifie ici les acknowledges

                        if ((reponse.find("ACK") != string::npos) && (reponse.find("NACK") == string::npos))
                        {
                            if (reponse.find("POSITION")  != string::npos)
                            {
                                Sockets[i].ack_pos   = true;
                            }
                            else if (reponse.find("GOTO") != string::npos)
                            {
                                Sockets[i].ack_goto  = true;
                            }
                            else if (reponse.find("PARK") != string::npos)
                            {
                                Sockets[i].ack_park  = true;
                            }
                            else if (reponse.find("ABORT")!= string::npos)
                            {
                                Sockets[i].ack_abort = true;
                            }
                        }
                        else
                        {
                            // Réponse à la requête POSITION

                            if (reponse.find("POSITION") != string::npos)
                            {
                                // Il y a une erreur signalée par le micro-contrôleur

                                if (reponse.find("NACK") != string::npos)
                                {
                                    // Problème concernant la commande P
                                    // On affiche dans la boîte indi un voyant de couleur rouge
                                    // et on affiche un message d'erreur

                                    OnCoordSetSP.s = IPS_ALERT;

                                    IDSetSwitch(&OnCoordSetSP, "ALERTE antenne %s : position de l antenne inconnue !\n",
                                                Sockets[i].IP.c_str());
                                    AfficherLog("ALERTE antenne %s : position de l antenne inconnue !\n",
                                                Sockets[i].IP.c_str());

                                    // La position de l'antenne st donc inconnue

                                    Sockets[i].PosValides = false;

                                    // On déconnecte l'antenne

                                    Sockets[i].Connected  = false;
                                }
                                else if (Sockets[i].ack_pos)
                                {
                                    // La position devrait être valide si on arrive ici...

                                    Sockets[i].PosValides = ExtractPosition(reponse, &Sockets[i].Pos);

                                    if ( Sockets[i].PosValides )
                                    {
                                        // La fonction extractPositon confirme que les positions transmises
                                        // par le micro-contrôleur sont valides

                                        OnCoordSetSP.s = IPS_OK;

                                        // message dans les logs

                                        IDSetSwitch(&OnCoordSetSP, "Antenne %s : POSITION OK  (x=%i, y=%i)\n",
                                                    Sockets[i].IP.c_str(), Sockets[i].Pos.x, Sockets[i].Pos.y);
                                    }
                                    else
                                    {
                                        // Erreur repérée par la fonction ExtractPosition

                                        OnCoordSetSP.s = IPS_ALERT;

                                        // On en informe l'utilisateur

                                        IDSetSwitch(&OnCoordSetSP, "Antenne %s : La position n est pas valide !\n",
                                                    Sockets[i].IP.c_str());
                                        AfficherLog("Antenne %s : La position n est pas valide !\n",
                                                    Sockets[i].IP.c_str());

                                    }
                                }
                            }

                            //réponse à la requête PARK

                            if (reponse.find("PARK") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    // Une erreur  est retournée par le micro-contrôleur concernant la commande Park
                                    // On affiche le voyant Park en rouge dans la boîte indi...

                                    ParkSP.s = IPS_ALERT;

                                    // ... et on affiche un message d'erreur

                                    IDSetSwitch(&ParkSP, "ALERTE antenne %s : erreur PARK !\n", Sockets[i].IP.c_str());
                                    AfficherLog("ALERTE antenne %s : erreur ABORT !\n",  Sockets[i].IP.c_str());
                                }
                                else if (Sockets[i].ack_park && reponse.find("OK")!=string::npos)
                                {
                                    // PARK ok !

                                    ParkSP.s = IPS_OK;

                                    IDSetSwitch(&ParkSP, "Antenne %s : PARK OK\n",  Sockets[i].IP.c_str());
                                }
                            }

                            //réponse à la requête ABORT

                            if (reponse.find("ABORT") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    // Une erreur  est retournée par le micro-contrôleur concernant la commande Abort
                                    // On affiche le voyant Park en rouge dans la boîte indi...

                                    AbortSlewSP.s = IPS_ALERT;

                                    // ... et on affiche un message d'erreur

                                    IDSetSwitch(&AbortSlewSP, "ALERTE antenne %s : erreur ABORT !\n",  Sockets[i].IP.c_str());
                                    AfficherLog("ALERTE antenne %s : erreur ABORT !\n",  Sockets[i].IP.c_str());
                                }
                                else if (Sockets[i].ack_abort && reponse.find("OK")!=string::npos)
                                {
                                    // OK !

                                    AbortSlewSP.s = IPS_OK;

                                    IDSetSwitch(&AbortSlewSP, "Antenne %s : ABORT OK\n",  Sockets[i].IP.c_str());
                                }
                            }

                            //réponse à la requête GOTO

                            if (reponse.find("GOTO") != string::npos)
                            {
                                if (reponse.find("NACK") != string::npos)
                                {
                                    // Une erreur  est retournée par le micro-contrôleur concernant la commande Goto
                                    // On affiche le voyant Park en rouge dans la boîte indi...

                                    OnCoordSetSP.s = IPS_ALERT;

                                    // Message d'erreur

                                    IDSetSwitch(&OnCoordSetSP, "ALERTE antenne %s : Erreur NACK GOTO ! (Retour : %s)\n",  Sockets[i].IP.c_str(), reponse.c_str());
                                    AfficherLog("ALERTE antenne %s : Erreur NACK GOTO ! (Retour : %s)\n",  Sockets[i].IP.c_str(), reponse.c_str());
                                    // On déconnecte l'antenne. Elle ne semble pas en mesure d'exécuter les ordres goto

                                    // 02/04/2012 Le fait de recevoir un nack/eos ne nécessite pas de devoir déconnecter l'antenne
                                    // DeconnecterSocket(i);
                                }
                                else if (Sockets[i].ack_goto)
                                {
                                    if (reponse.find("OK") != string::npos)
                                    {
                                        // On a ici la confirmation que l'antenne 'i' a bien réalisé le goto

                                        // On prend note

                                        Sockets[i].GotoOk = true;

                                        // Voyant en vert dans la boîte Indi

                                        OnCoordSetSP.s = IPS_OK;

                                        // Message pour l'utilisateur

                                        IDSetSwitch(&OnCoordSetSP, "Antenne %s : GOTO OK.\n",  Sockets[i].IP.c_str());

                                        // Fin du Goto pour toutes les antennes ?
                                        // On compte les antennes connectées qui ont réalisé le dernier ordre goto

                                        int num = 0;

                                        for (int j=1; j<SocketsNumber; j++)
                                        {
                                            if (Sockets[j].Connected)
                                            {
                                                if (Sockets[j].GotoOk) num++;
                                            }
                                        }

                                        // Si le nombre d'antennes connectées est > 0
                                        // et que toutes les antennes connectées ont
                                        // réalisé le goto alors...

                                        if ((num == AntennesConnectees()) && (num>0))
                                        {
                                            // C'est bon ! Tout marche bien...
                                            // On actualise l'AD et la Dec dans la boîte de dialogue

                                            lastRA  = targetRA;
                                            lastDEC = targetDEC;

                                            // On a fini le mouvement. Il n'y a donc plus d'étape à faire
                                            // dans le processus de réalisation d'un goto
                                            // RealisationGoto vaut donc false

                                            RealisationGoto = false;

                                            // Réinitialisation des paramètres des antennes en vue d'un prochain goto

                                            InitAntennes();

                                            // On dessine les voyants de la boîte de dialogue en vert

                                            OnCoordSetSP.s = IPS_OK;
                                            EquatorialCoordsWNP.s = IPS_OK;
                                            IDSetNumber (&EquatorialCoordsWNP, NULL);

                                            // Confirmation dans la boîte de dialogue que toutes
                                            // les antennes sont OK

                                            IDSetSwitch(&OnCoordSetSP, "GOTO OK !");

                                            UpdatedGoto = true;
                                        }
                                    }
                                }
                            }
                        }

                        // On passe à la trame suivante si memreponse n'est pas vide

                        reponse = buffereponse;
                        pos     = reponse.find("\n");
                    }
                }
                catch (SocketException& e) //Aïe
                {
                    // Une exception concerne le socket i

                    // On déconnecte l'antenne pour plus de sécurité

                    DeconnecterSocket(i);

                    AfficherLog("Indi_BAO, SocketException IsPoll : ");
                    AfficherLog(e.description().c_str());
                    AfficherLog("\n");
                }
            }
        }


        ///////////////////////////////////////
        // L'utilisateur a demandé l'annulation du mouvement en cours

        if (Abort)
        {
            // On arrête le suivi d'un objet

            Suivi = false;

            // On arrête l'enchaînement des actions nécessaires à la réalisation d'un goto

            RealisationGoto = false;

            // On envoie l'ordre ABORT à toutes les antennes

            for (int i=1; i<SocketsNumber; i++)
            {
                if (Sockets[i].Connected)
                {
                    if (!ABORT(i)) Sockets[i].sendalertes++;
                }
            }

            // Message à destination de l'utilisateur et des logs

            IDSetSwitch(&OnCoordSetSP, "ABORT OK !");

            // Réinititialisation des paramètres des antennes

            InitAntennes();

            //Pour permettre de refaire un abort

            Abort = false;
        }

        ///////////////////////////////////////
        // L'utilisateur a demandé de mettre les antennes au repos

        if (Park)
        {
            // On arrête le suivi d'un objet

            Suivi = false;

            // On arrête l'enchaînement des actions
            // pour réaliser un goto

            RealisationGoto = false;

            // On envoie l'ordre PARK à toutes les antennes

            for (int i=1; i<SocketsNumber; i++)
            {
                if (Sockets[i].Connected)
                {
                    if (!PARK(i)) Sockets[i].sendalertes++;
                }
            }

            IDSetSwitch(&OnCoordSetSP, "PARK OK !");

            // Réinititialisation des paramètres des antennes

            InitAntennes();

            // Pour permettre de refaire un park

            Park = false;
        }


        ///////////////////////////////////////
        // Gestion du suivi

        if ((Suivi) && (UpdatedGoto))
        {
            // Délais entre deux actualisations

            // Actualisation toutes les 15 minutes en mode transit (par défaut)

            double delai = ActualisationTMTransit / 3600.0 / 24.0;

            // et 5 secs en mode tracking (par défaut)

            if (TrackingMode == BAO_TRACKING) delai = ActualisationTMTracking / 3600.0 / 24.0;


            // On actualise la position si le délai est dépassé

            if (GetJJ() - JJAnc > delai)
            {
                // Cette variable vaut true lorsque le goto a été réalisé

                UpdatedGoto = false;

                // Conversion des coordonnées en pas moteur

                ADDEC2Motor(targetRA, targetDEC);

                // On réinitialise les antennes en vue du goto

                InitAntennes();

                // On lance le processus d'actualisation du goto

                RealisationGoto = true;

                // On sauvegarde la date

                JJAnc = GetJJ();
            }

            //Plus d'antenne ! On arrête le suivi

            if (AntennesConnectees() == 0)
            {
                if ( compt % 1000 == 0)
                {
                    IDSetSwitch(&OnCoordSetSP, "Erreur ! Plus d antennes connectees !");

                    if (Suivi) AfficherLog("Arrêt du suivi !");

                    RealisationGoto=false;

                    Suivi=false;

                    InitAntennes();
                }
            }
        }



        // Exécution de la procédure complète d'un goto :
        // 1ère étape : envoi de la commande POSITION à toutes les antennes
        // 2ème étape : Vérification de l'acknowledge de la commande POSITION pour chaque antenne
        // 3ème étape : Les positions retournées par les antennes sont-elles valides ?
        // 4ème étape : Si oui, envoie de la commande goto à toutes les antennes
        // 5ème étape : est-ce que toutes les antennes ont envoyé l'acknowledge de la commande goto ?
        // 6ème étape : les antennes ont toutes répondu GOTO OK !

        if (RealisationGoto)
        {
            for (int i=1; i<SocketsNumber; i++)
            {
                // On ne parle qu'aux antennes connectées

                if (Sockets[i].Connected)
                {
                    // En fonction de l'étage de la réalisation d'un goto par l'antenne i

                    switch (Sockets[i].etape)
                    {

                        // Envoi de la commande POS

                    case 0 :
                    {
                        // On doit initialiser l'acknowledge à false

                        Sockets[i].ack_pos    = false;

                        // et considérer que la position de l'antenne n'est pas valide

                        Sockets[i].PosValides = false;

                        // avant d'envoyer l'ordre POSITION

                        if (!POSITION(i)) Sockets[i].sendalertes++;

                        // On passe à l'étage suivante

                        Sockets[i].etape++;
                    }
                    break;


                    // A-ton bien reçu l'ack POS ?

                    case 1 :
                    {
                        if (Sockets[i].ack_pos)
                        {
                            // tout marche bien. On a bien reçu l'acknowledge de l'antenne i

                            // Il n'y a donc aucune attente supplémentaire pour recevoir la réponse
                            Sockets[i].AttenteExecution = 0;

                            // Pas d'anomalie à consigner
                            Sockets[i].AnomaliesExecution = 0;

                            // On passe à l'étape suivante
                            Sockets[i].etape++;

                            //permet de rester sur le même socket malgré la boucle -> plus rapide
                            i--;
                        }
                        else
                        {
                            // On réitère l'ordre précédent si rien ne se passe

                            // On garde une trace de l'anomalie

                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // on recommence depuis le début.
                                // Peut-être que l'antenne n'a pas reçu l'ordre ?
                                Sockets[i].etape = 0;
                                Sockets[i].AttenteExecution = 0;

                                // Pas de réponse. On consigne une erreur grave
                                Sockets[i].AnomaliesExecution++;
                            }

                            // Les erreurs graves s'accumulent. Pas de réponse après plusieurs minutes
                            // -> Il faut déconnecter l'antenne
                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
                                //Voyant en rouge dans la boîte Indi

                                OnCoordSetSP.s = IPS_ALERT;

                                //Message à l'attention de l'utilisateur

                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : pas d acknowledge recu apres l ordre POSITION. \
Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());
                                AfficherLog("Erreur sur l antenne %s : pas d acknowledge recu apres l ordre POSITION. \
Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());

                                Suivi=false;

                                RealisationGoto = false;

                                InitAntennes();

                                // On déconnecte l'antenne

                                // DeconnecterSocket(i);
                            }
                        }
                    }
                    break;


                    //Les valeurs retournées pas la commande POSITION sont-elles valides ?

                    case 2 :
                    {
                        if (Sockets[i].PosValides)
                        {
                            // Tout vas bien
                            // On ne consigne aucune anomalie

                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;

                            //On passe à l'étape suivante

                            Sockets[i].etape++;
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe

                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // on attend encore la réponse posvalides

                                Sockets[i].etape = 2;
                                Sockets[i].AttenteExecution = 0;

                                // On consigne une erreur grave. L'antenne tarde à répondre

                                Sockets[i].AnomaliesExecution++;
                            }

                            // Aucune réponse de l'antenne depuis plusieurs minutes
                            // On la déconnecte

                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
                                // Voyant en rouge

                                OnCoordSetSP.s = IPS_ALERT;

                                //Message d'erreur

                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : la position retournee n est pas valide. \
Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());
                                AfficherLog("Erreur sur l antenne %s : la position retournee n est pas valide. \
Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());

                                Suivi=false;

                                RealisationGoto = false;

                                InitAntennes();

                                //Déconnexion de l'antenne

                                // DeconnecterSocket(i);
                            }
                        }
                    }
                    break;



                    // On peut remarquer qu'il n'y a pas de case 3 ici...
                    // L'envoie de la commande goto se fait portant pendant l'étape 3 en dehors du switch
                    // Voir le code plus loin...


                    // A-ton reçu l'acknowledge de la commande goto ?

                    case 4 :
                    {
                        if (Sockets[i].ack_goto)
                        {
                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;
                            Sockets[i].etape++; // on passe à l'étape suivante
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // On prolonge l'attente pour recevoir l'acknowledge du goto
                                Sockets[i].etape = 4;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

                            if (Sockets[i].AnomaliesExecution > MAXANOMALIES)
                            {
                                OnCoordSetSP.s = IPS_ALERT;

                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : pas d acknowledge recu apres l ordre GOTO. Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());
                                AfficherLog("Erreur sur l antenne %s : pas d acknowledge recu apres l ordre GOTO. Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());

                                Suivi=false;

                                RealisationGoto = false;

                                InitAntennes();

                                // DeconnecterSocket(i);
                            }
                        }
                    }
                    break;


                    //  Confirmation goto ok ?

                    case 5 :
                    {
                        if (Sockets[i].GotoOk)
                        {
                            // On a bien reçu Goto Ok pour l'antenne i !

                            Sockets[i].AttenteExecution = 0;
                            Sockets[i].AnomaliesExecution = 0;

                            //On passe à l'étape suivante
                            Sockets[i].etape++;
                        }
                        else
                        {
                            // on réitère l'ordre précédent si rien ne se passe
                            Sockets[i].AttenteExecution++;

                            if (Sockets[i].AttenteExecution > MAXATTENTE)
                            {
                                // On prolonge l'attente afin de recevoir un GOTO OK

                                Sockets[i].etape = 5;
                                Sockets[i].AttenteExecution = 0;
                                Sockets[i].AnomaliesExecution++;
                            }

                            // On déconnecte l'antenne s'il n'y a pas de confirmation du goto au bout de 2 minutes
                            if (Sockets[i].AnomaliesExecution > MAXANOMALIESGOTO)
                            {
                                // Alerte sur une antenne
                                OnCoordSetSP.s = IPS_ALERT;

                                // Message d'erreur
                                IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : l antenne n a pas renvoye GOTO/OK. Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());
                                AfficherLog("Erreur sur l antenne %s : l antenne n a pas renvoye GOTO/OK. Fin du suivi. Verifier les connexions.\n", Sockets[i].IP.c_str());

                                Suivi=false;

                                RealisationGoto = false;

                                InitAntennes();


                                //Déconnexion de l'antenne
                                // DeconnecterSocket(i);
                            }
                        }
                    }
                    break;
                    }
                }
            }
        }


        ///////////////////////////////////////
        // Réalisation d'un goto - traitement de l'étape 3
        // On place cette partie du traitement en dehors du switch et de la boucle
        // pour pouvoir envoyer les gotos à toutes les antennes lorsque l'on a la confirmation
        // qu'elles sont toutes prêtes à exécuter cet ordre (être à l'étape 3 pour un socket)
        // -> meilleure synchronisation

        // On compte les antennes rendues à l'étape 3 et qui attendent

        int num = 0;

        for (int i=1; i<SocketsNumber; i++)
        {
            // Uniquement les antennes connectées

            if (Sockets[i].Connected)
            {
                if (Sockets[i].etape == 3) num++; // num antennes sont prêtes à recevoir l'ordre GOTO
            }
        }

        // Toutes les antennes connectées sont prêtes à recevoir l'ordre goto

        if ((num == AntennesConnectees()) && (num>0))
        {
            for (int i=1; i<SocketsNumber; i++ )
            {
                if (Sockets[i].Connected)
                {
                    // On envoie l'ordre

                    Sockets[i].ack_goto = false;
                    Sockets[i].AttenteExecution = 0;
                    Sockets[i].AnomaliesExecution = 0;

                    if (!GOTO(i, Sockets[i].TargetPosition.x - Sockets[i].Pos.x, Sockets[i].TargetPosition.y - Sockets[i].Pos.y )) Sockets[i].sendalertes++;

                    Sockets[i].etape++;
                }
            }
        }

        ///////////////////////////////////////
        // Opération garbage
        // Détection d'anomalies sur le socket i.
        // Ce n'est pas normal ici. Il faut déconnecte l'antenne

        for (int i=1; i<SocketsNumber; i++)
        {
            if (Sockets[i].Connected)
            {
                if (Sockets[i].sendalertes > 0)
                {
                    // Alarme dans la boîte Indi

                    OnCoordSetSP.s = IPS_ALERT;

                    // Erreur dans les logs

                    IDSetSwitch(&OnCoordSetSP, "Erreur sur l antenne %s : Fin du suivi. Vérifier les connexions.\n", Sockets[i].IP.c_str());
                    AfficherLog("Erreur sur l antenne %s : Fin du suivi. Vérifier les connexions.\n", Sockets[i].IP.c_str());

                    Suivi=false;

                    RealisationGoto = false;

                    InitAntennes();

                    // Déconnexion antenne

                    // DeconnecterSocket(i);
                }
            }
        }
    }

    //incrémentation du compteur

    compt++;
}



/**************************************************************************************
** Mode transit ou tracking
**
***************************************************************************************/

bool BAO::process_coords()
{
    int posaz = 0;

    switch (currentSet)
    {
        // Transit

    case BAO_TRANSIT:

        // Éteindre les voyants dans la boîte de dialogue Indi
        EquatorialCoordsWNP.s = IPS_BUSY;
        AbortSlewSP.s = IPS_IDLE;
        ParkSP.s = IPS_IDLE;

        IDSetNumber (&EquatorialCoordsWNP, NULL);
        IDSetSwitch (&AbortSlewSP, NULL);
        IDSetSwitch (&ParkSP, NULL);

        // On cherche l'azimut de l'objet lorsque celui-ci passera en-dessous de 30°
        // Cette recherche est nécessaire pour optimiser le suivi et éviter
        // les rotations inutiles

        for (int i=1; i<SocketsNumber; i++ )
        {
            if ((Sockets[i].Connected) && (Sockets[i].PosValides))
            {
                posaz = Sockets[i].Pos.x;
                break;
            }
        }

        RechercheAzimutFinSuivi(targetRA * 15.0 * Pidiv180, targetDEC * Pidiv180, posaz, &azmincodeur, &azmaxcodeur);

        // On prépare les antennes pour le prochain goto

        InitAntennes();

        // On garde la trace du début du Goto pour enchaîner les actualisations

        JJAnc = GetJJ();

        // Conversion des coordonnées horaires en pas moteur

        ADDEC2Motor(targetRA, targetDEC);

        // Mode transit activé

        TrackingMode = BAO_TRANSIT;

        // On suit un objet

        Suivi = true;

        // Aucun goto n'a été encore réalisé

        UpdatedGoto = false;

        // Mais on se lance dans la réalisation d'un goto

        RealisationGoto = true;

        break;

        // Tracking

    case BAO_TRACKING:

        // Éteindre les voyants dans la boîte de dialogue Indi

        EquatorialCoordsWNP.s = IPS_BUSY;
        AbortSlewSP.s = IPS_IDLE;
        ParkSP.s = IPS_IDLE;

        IDSetNumber (&EquatorialCoordsWNP, NULL);
        IDSetSwitch (&AbortSlewSP, NULL);
        IDSetSwitch (&ParkSP, NULL);

        // On cherche l'azimut de l'objet lorsque celui-ci passera en-dessous de 30°
        // Cette recherche est nécessaire pour optimiser le suivi et éviter
        // les rotations inutiles

        for (int i=1; i<SocketsNumber; i++ )
        {
            if ((Sockets[i].Connected) && (Sockets[i].PosValides))
            {
                posaz = Sockets[i].Pos.x;
                break;
            }
        }

        RechercheAzimutFinSuivi(targetRA * 15.0 * Pidiv180, targetDEC * Pidiv180, posaz, &azmincodeur, &azmaxcodeur);

        InitAntennes();

        JJAnc = GetJJ();

        ADDEC2Motor(targetRA, targetDEC);

        TrackingMode = BAO_TRACKING;

        Suivi = true;

        UpdatedGoto = false;

        RealisationGoto = true;

        break;
    }

    return true;
}



/**************************************************************************************
** Connexion / Déconnexion avec le télescope
**
***************************************************************************************/

void BAO::connect_telescope()
{
    switch (ConnectSP.sp[0].s)
    {
    case ISS_ON:

        // Etats des voyants

        ConnectS[0].s = ISS_ON;
        ConnectS[1].s = ISS_OFF;
        ConnectSP.s = IPS_OK;
        IDSetSwitch (&ConnectSP, "BAORadio is online. Retrieving basic data...");

        // On lance le thread !
        // Exit ne faudra true que lorsque sera venu le moment de sortir !

        Exit = false;

        // Création du thread

        if (pthread_create (&th1, NULL, (void*(*)(void*))LancementThread, this) < 0)
        {
            AfficherLog("pthread_create error for threadSocket\n");
        }

        break;

    case ISS_OFF:

        // Etat des voyants

        ConnectS[0].s = ISS_OFF;
        ConnectS[1].s = ISS_ON;
        ConnectSP.s = IPS_IDLE;
        IDSetSwitch (&ConnectSP, "BAORadio is offline.");
        AfficherLog("Telescope is offline.");

        // On déconnecte tous les sockets

        for (int i=0; i<MAXHOSTNAME; i++)
        {
            DeconnecterSocket(i);
        }

        // init

        InitAntennes();

        SocketsNumber = 1;

        // On sort du thread

        // On dit au thread de sortir de la boucle

        Exit = true;

        // Désactiver la boucle de traitement des messages des ISPOLL

        InitThreadOK = false;

        // On laisse 1 s au thread pour sortir de la boucle

        sleep(1);

        // On détruit le thread

        pthread_join (th1, NULL);

        // on sort du programme

        exit(EXIT_SUCCESS);

        break;
    }
}




/**************************************************************************************
**  Envoie une commande sur le socket numsocket
**
***************************************************************************************/

bool BAO::COMMANDE(int numsocket, char* Commande, char* Params)
{
    char chaine[MAXCARACTERES];

    try
    {
        sprintf(chaine, "%s%s\n", Commande, Params);

        Sockets[numsocket].new_sock << chaine;

        AfficherLog("Commande envoyee a %s: %s", Sockets[numsocket].IP.c_str(), chaine);
    }
    catch (SocketException& e)
    {
        // Consignation d'une anomalie sur le socket

        DeconnecterSocket(numsocket);

        AfficherLog("Indi_BAO, COMMANDE exception : ");
        AfficherLog(e.description().c_str());
        AfficherLog("\n");

        return false;
    }

    return true;
}


/**************************************************************************************
** Commande POSITION
**
***************************************************************************************/

bool BAO::POSITION(int numsocket)
{
    return COMMANDE(numsocket, (char*)"P", (char*)"");
}

/**************************************************************************************
** Commande PARK
**
***************************************************************************************/

bool BAO::PARK(int numsocket)
{
    return COMMANDE(numsocket, (char*)"Z", (char*)"");
}

/**************************************************************************************
** Commande ABORT
**
***************************************************************************************/

bool BAO::ABORT(int numsocket)
{
    return COMMANDE(numsocket, (char*)"A", (char*)"");
}


/**************************************************************************************
** Commande GOTO
**
***************************************************************************************/

bool BAO::GOTO(int numsocket, int deltaAz, int deltaAlt)
{
    char Params[MAXCARACTERES];
    char sensAz;
    char sensAlt;

    sensAlt = 1;
    sensAz  = 1;

    // gestion des signes des deltas

    if ( deltaAz < 0 )
    {
        deltaAz = -deltaAz;
        sensAz  = 0;
    }

    if ( deltaAlt < 0 )
    {
        deltaAlt = -deltaAlt;
        sensAlt  = 0;
    }

    // Vérification du nombre de pas à faire au niveau de l'axe azimut
    // Rappel : un tour complet autour de l'axe az fait 4000 pas codeur (voir #define NBREPASCODEURSAZ dans BAO.h)

    // problème 1 : si deltaAz > à un demi-tours - soit 2000 pas codeur alors
    // on fait le trajet en sens inverse pour aller plus vite

    // problème 2 : Passage au méridien
    // on se situe à quelques secondes de degrés avant le sud par exemple (nb de pas 3995 par exemple)
    // et on franchit le méridien (nb de pas codeur = 5 par exemple)
    // On risque alors de faire un tour complet en sens inverse pour rattraper l'objet -> à éviter

    // ne pas faire des tours complets en Az pour rien...

    // Corrections sup pour rester dans l'intervalle [-1693.. 3867]

    AfficherLog("deltaAz1=%i\n", deltaAz, true);

    while (deltaAz > NBREPASCODEURSAZ) deltaAz -= NBREPASCODEURSAZ;


    //TODO : a verifier en details

    if ( !AutoriserTourComplet )
    {

        // Doit résoudre tous les problèmes concernant l'azimut...

        if (deltaAz > NBREPASCODEURSAZ / 2 )
        {
            deltaAz = NBREPASCODEURSAZ - deltaAz;

            sensAz = 1 - sensAz;
        }
    }

    AfficherLog("deltaAz2=%i\n", deltaAz, true);

    //on envoie les coordonnées au driver

    (sensAz == 1 ) ? sensAz='f' : sensAz='b';

    (sensAlt == 1 ) ? sensAlt='f': sensAlt='b';

    sprintf(Params, "%c%04i%c%04i", sensAz, deltaAz, sensAlt, deltaAlt);

    return COMMANDE(numsocket, (char*)"G", Params);
}



/**************************************************************************************
** Les fonctions qui suivent sont nécessaires pour construire le pilote Indi_BAO
** Elles sont appelées par le noyau d'Indi
***************************************************************************************/

/**************************************************************************************
** Initialisation du pilote BAO
***************************************************************************************/

void ISInit()
{
    //Il ne faut exécuter la fonction qu'une seule fois

    static int isInit = 0;

    if (isInit) return;

    if (telescope.get() == 0) telescope.reset(new BAO());

    isInit = 1;

    //initialisation du timer

    IEAddTimer (POLLMS, ISPoll, NULL);
}


void ISGetProperties (const char *dev)
{
    ISInit();

    telescope->ISGetProperties(dev);
}

void ISNewSwitch (const char *dev, const char *name, ISState *states, char *names[], int n)
{
    ISInit();
    telescope->ISNewSwitch(dev, name, states, names, n);
}

void ISNewText (const char *dev, const char *name, char *texts[], char *names[], int n)
{
    ISInit();
    telescope->ISNewText(dev, name, texts, names, n);
}

void ISNewNumber (const char *dev, const char *name, double values[], char *names[], int n)
{
    ISInit();
    telescope->ISNewNumber(dev, name, values, names, n);
}

void ISNewBLOB (const char *dev, const char *name, int sizes[], int blobsizes[], char *blobs[], char *formats[], char *names[], int n)
{
    INDI_UNUSED(dev);
    INDI_UNUSED(name);
    INDI_UNUSED(sizes);
    INDI_UNUSED(blobsizes);
    INDI_UNUSED(blobs);
    INDI_UNUSED(formats);
    INDI_UNUSED(names);
    INDI_UNUSED(n);
}

void ISSnoopDevice (XMLEle *root)
{
    INDI_UNUSED(root);
}

void ISPoll (void *p)
{
    INDI_UNUSED(p);

    telescope->ISPoll();

    IEAddTimer (POLLMS, ISPoll, NULL);
}


/**************************************************************************************
**
***************************************************************************************/

int BAO::get_switch_index(ISwitchVectorProperty *sp)
{
    for (int i=0; i < sp->nsp ; i++)
        if (sp->sp[i].s == ISS_ON)
            return i;

    return -1;
}



/**************************************************************************************
**
***************************************************************************************/

bool BAO::is_connected()
{
    // return (ConnectSP.sp[0].s == ISS_ON);
    return (ConnectSP.s == IPS_OK);
}

/**************************************************************************************
**
***************************************************************************************/
void BAO::connection_lost()
{
    ConnectSP.s = IPS_IDLE;
    IDSetSwitch(&ConnectSP, "The connection to the telescope is lost.");
    AfficherLog("arret");
    return;
}

/**************************************************************************************
**
***************************************************************************************/
void BAO::connection_resumed()
{
    ConnectS[0].s = ISS_ON;
    ConnectS[1].s = ISS_OFF;
    ConnectSP.s = IPS_OK;

    IDSetSwitch(&ConnectSP, "The connection to the telescope has been resumed.");
}


/**************************************************************************************
** Gère les erreurs de communication avec la boîte Indi
***************************************************************************************/
/*
void BAO::handle_error(INumberVectorProperty *nvp, int err, const char *msg)
{
    nvp->s = IPS_ALERT;

    // If the error is a time out, then the device doesn't support this property
    if (err == -2)
    {
        nvp->s = IPS_ALERT;
        IDSetNumber(nvp, "Device timed out. Current device may be busy or does not support %s. Will retry again.", msg);
    }
    else
        // Changing property failed, user should retry.
        IDSetNumber( nvp , "%s failed.", msg);

    fault = true;
}*/


/**************************************************************************************
**
***************************************************************************************/
/*
void BAO::correct_fault()
{
    fault = false;
    IDMessage(mydev, "Telescope is online.");
}*/




