/*
The stellarium telescope library helps building
telescope server programs, that can communicate with stellarium
by means of the stellarium TCP telescope protocol.
It also contains smaple server classes (dummy, Meade LX200).

Author and Copyright of this file and of the stellarium telescope library:
Johannes Gajdosik, 2006

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "ServerBAO.hpp"
#include "Socket2.hpp" // GetNow

#include <math.h>

ServerDummy::ServerDummy(int port)
            :Server2(port)
{
	last_pos[0] = current_pos[0] = desired_pos[0] = 1.0;
	last_pos[1] = current_pos[1] = desired_pos[1] = 0.0;
	last_pos[2] = current_pos[2] = desired_pos[2] = 0.0;
	next_pos_time = -0x8000000000000000LL;


       float a1, a2, a3;

    // Le choix de la couleur des caractères est adapté pour une fenêtre de terminal
    // ayant un fond blanc. pour un fond noir, mettre couleurs = 2 dans le fichier params

    ChoixCouleurs=1;

    // Affichage de la version du programme

    AfficherLog("\n\n\n\n\n \
 ******************************************\n \
 *             BAORadio Control           *\n \
 * Laboratoire de l'Accélérateur Linéaire *\n \
 *                                        *\n \
 *             v0.52 19/04/2012           *\n \
 *        franckrichard033@gmail.com      *\n \
 ******************************************\n\n\n", true);


    // Initialisation des variables globales et des pointeurs
    // pour la signification des variables, voir baocontrol.h


    delaitransit           = 0;
    delaitracking          = 0;
    lognum                 = 0;
    numAntennes            = 0;
    numobjets              = 0;
    numEtoiles             = 0;
    runnum                 = 1;

    MethodeAlignement      = SIMPLE;

    Pression               = 1013.0;
    Temperature            = 10.0;

    NoExit                 = true;
    Transit                = false;
    run                    = false;
    exitrun                = false;
    ModificationAlignement = false;

    fd                     = NULL;
    client_socket          = NULL;

    // chaînes de caractères et tableaux

    LatitudeChar           = "";
    LongitudeChar          = "";
    Serveur                = "";
    Port                   = "";

    Antennes 	           = new DefAntenne[ MAXANTENNES ];
    objets                 = new DefObjets [ MAXOBJETS ];
    Etoiles                = new DefEtoiles[ MAXETOILES] ;


    // on initialise aussi les structures

    for (int i=0; i<MAXANTENNES; i++)
    {
        Antennes[i].ok = 0;
        Antennes[i].ip = 0;

        Antennes[i].AlignementAntenne = new Alignement();
    }

    for (int i=0; i<MAXOBJETS; i++)
    {
        objets[i].JJ   = 0.0;
        objets[i].exec = false;
    }

    for (int i=0; i<MAXETOILES; i++)
    {
        Etoiles[i].nom          = "";
        Etoiles[i].ad           = 0.0;
        Etoiles[i].de           = 0.0;
        Etoiles[i].az           = 0.0;
        Etoiles[i].ha           = 0.0;
        Etoiles[i].mag          = 0.0;
        Etoiles[i].selectionnee = false;
    }


    // Chargement du fichier contenant les paramètres de l'application BAOcontrol
    // La variable permet de récupérer le nom de l'utilisateur ayant ouvert la session sous linux
    // c'est dans le répertoire /home/USER/ que BAOcontrol cherche le fichier de paramètre baocontrol_params

    if (!ChargementParametres("/home/" + (string)getenv("USER") + "/baocontrol_params"))
    {
        ErreurLog("Le fichier de configuration 'params' contient des erreurs ou n'existe pas.\n");
        ErreurLog("Il devrait se trouver dans le répertoire /home/" + (string)getenv("USER") +"\n");
        ErreurLog("Merci de vérifier...\n\n");
        exit(1);
    }


    // Conversion de la latitude et de la longitude en radians

    Decomposition(LatitudeChar, 0, &a1, &a2, &a3);

    double Latitude = (a1 + a2 / 60.0 + a3 / 3600.0) * Pidiv180;

    Decomposition(LongitudeChar, 1, &a1, &a2, &a3);

    double Longitude = Pi2 - ( a1 + a2 / 60.0 + a3 / 3600.0 ) * Pidiv180;


    // Transfert de la latitude, longitude, pression, température à la classe Astro

    DefinirLongitudeLatitude(Longitude, Latitude);

    DefinirPressionTemp(Pression, Temperature);
    // On lance les calculs (Temps sidéral local etc...)

    CalculTSL();


    //Chargement du catalogue d'étoiles que l'on utilise pour l'alignement des antennes

  /* if ( !ChargementCatalogueEtoiles("/home/" + (string)getenv("USER") + "/bao_catalogue.dat") )
    {
        ErreurLog("Le catalogue d'étoiles 'bao_catalogue' est introuvable.\n");
        ErreurLog("Il devrait se trouver dans le répertoire /home/" + (string)getenv("USER") +"\n");
        ErreurLog("Merci de vérifier...\n\n");
        exit(1);
    }
*/


    // Ouverture du socket avec indi_BAO

    client_socket = new ClientSocket("localhost", atoi(Port.c_str()) );

    // On se connecte au pilote indi_BAO

   Connect(true);
        // On lance la fenêtre graphique

        initialiserFenetre();

        XSelectInput(d,w, ExposureMask );
}

void ServerDummy::step(long long int timeout_micros)
{

  if (last_pos[0]!=current_pos[0] || last_pos[1]!=current_pos[1] || last_pos[2]!=current_pos[2])
  {
		current_pos[0] = desired_pos[0];
			current_pos[1] = desired_pos[1];
			current_pos[2] = desired_pos[2];
			
		const double ra = atan2(current_pos[1],current_pos[0]);
		const double dec = atan2(current_pos[2],
		                         sqrt(current_pos[0]*current_pos[0]+current_pos[1]*current_pos[1]));
		
			//On actualise le jour julien et le temps sidéral local
		
		Goto(DHMS(VerifAngle(ra)*180.0/M_PI, true), DHMS(dec*180.0/M_PI, false), Transit, true);
		
		last_pos[0] = current_pos[0];
			last_pos[1] = current_pos[1];
			last_pos[2] = current_pos[2];
  }

    Update();

    

    // on lit et affiche la réponse des microcontrolleurs

    LireReponse();

    Dessiner();
    
	
	

	
	Server2::step(timeout_micros);
}

void ServerDummy::gotoReceived(unsigned int ra_int, int dec_int)
{
	const double ra = ra_int*(M_PI/(unsigned int)0x80000000);
	const double dec = dec_int*(M_PI/(unsigned int)0x80000000);
	const double cdec = cos(dec);
	desired_pos[0] = cos(ra)*cdec;
	desired_pos[1] = sin(ra)*cdec;
	desired_pos[2] = sin(dec);
}


/**************************************************************************************
** Affiche le message à l'écran (si AfficheEcran==true) et le sauvegarde dans le fichier
** BAOControl.log
***************************************************************************************/

void ServerDummy::Affiche(string Message, bool AfficheEcran)
{
    FILE *pFile = NULL;

    if ( (pFile = fopen ("BAOcontrol.log", "a")) != NULL)
    {
        if (AfficheEcran) cout << Message;

        fprintf (pFile, Message.c_str());

        fclose (pFile);
    }
}

void ServerDummy::Affiche(stringstream *Message, bool AfficheEcran)
{
    Affiche(Message->str(), AfficheEcran);

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Affiche les messages d'erreur en rouge
***************************************************************************************/

void ServerDummy::Erreur(string chaine)
{
    ChoixCouleurs==1 ? cout << red1  : cout << red2;
    Affiche(chaine, true);
    ChoixCouleurs==1 ? cout << grey1 : cout << grey2;
}

void ServerDummy::Erreur(stringstream *Message)
{
    Erreur(Message->str());

    // on efface le buffer

    Message->str("");
}


/**************************************************************************************
** Initialise la fenêtre graphique
***************************************************************************************/

void ServerDummy::initialiserFenetre()
{
    XGCValues gcv;

    // contacte le serveur graphique

    if ((d = XOpenDisplay(getenv("DISPLAY"))) == NULL)
    {
        Erreur("Impossible de contacter le serveur\n\n");
        exit(1);
    }

    // crée une fenêtre

    w = XCreateSimpleWindow(d, RootWindow(d, DefaultScreen(d)),
                            0, 0, larg_fenetre, haut_fenetre,
                            2, BlackPixel(d, DefaultScreen(d)),
                            WhitePixel(d, DefaultScreen(d)));

    // Création du double buffer

    db = XCreatePixmap(d, w, larg_fenetre, haut_fenetre, DefaultDepth(d, DefaultScreen(d)));

    // Chargement de la police de caractères

    if ((fd = XLoadQueryFont(d, "fixed")) == NULL)
    {
        Erreur("Impossible de charger la police fixed\n\n");
        exit(1);
    }

    // Création des couleurs

    gcv.font = fd->fid;
    gcv.foreground = BlackPixel(d, DefaultScreen(d));
    noir = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*255;
    vert = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 255*256*256;
    rouge = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    gcv.foreground = 200*256*256+200*256+200;
    gris = XCreateGC(d, w, GCFont | GCForeground, &gcv);

    // Affichage de fenêtre

    XMapWindow(d, w);
}


/**************************************************************************************
** Dessine les éléments dans la fenêtre graphique
***************************************************************************************/

void ServerDummy::Dessiner()
{
    char *chaine;
    GC color;

    chaine = new char [100];

    // On efface la fenêtre

    XSetForeground(d, noir, WhitePixel(d, DefaultScreen(d)));
    XFillRectangle(d, db, noir, 0, 0, larg_fenetre, haut_fenetre);
    XSetForeground(d, noir, BlackPixel(d, DefaultScreen(d)));

    // On récupère la date et l'heure contenue dans la classe astro (et le jour julien)

    double h  = GetHeure();
    double mi = GetMin();
    double s  = GetSec();
    double a  = GetAnnee();
    double m  = GetMois();
    double j  = GetJour();
    double JJ = GetJJ();

    // on évite les affichages de l'heure du genre 12:56:60 ou 24:00:00

    if (s==60) {
        s = 0;
        mi++;
        if (mi == 60) {
            mi = 0;
            h++;
            if (h == 24) {
                h = 0;
                j++;
            }
        }
    }

    //affichage de la date et de l'heure

    sprintf(chaine, "%02.0f:%02.0f:%02.1f TU    %02.0f/%02.0f/%04.0f    JJ=%10.5f",
            h, mi, s, j, m, a, JJ);

    XDrawString(d, db, noir, 10, 10, chaine, strlen(chaine));

    //En cas d'exécution d'un fichier de mouvements, on affiche la prochaine étape

    if (run)
    {
        sprintf(chaine, "Prochaine etape dans %4.1fs", (objets[runnum].JJ - JJ) * 3600.0 * 24.0);

        if ((objets[runnum].JJ - JJ) * 3600.0 * 24.0 >= 0.0)
            XDrawString(d, db, noir, 350, 10, chaine, strlen(chaine));
    }

    // affichage des cercles d'état
    // vert = l'antenne fonctionne
    // rouge = problème détecté sur l'antenne
    // gris = antenne non connectée

    for (int i=0; i <20; i++)
    {
        for (int j=0; j<2; j++)
        {
            switch (Antennes[20*j+i].ok)
            {
            case 1 :
                color=vert;
                break;
            case 2 :
                color=rouge;
                break;
            default:
                color=gris;
                break;
            }

            XFillArc(d, db ,color, 10+i*30, 20+j*30, 15, 15, 0, 360*64);
            XDrawArc(d, db ,noir,  10+i*30, 20+j*30, 15, 15, 0, 360*64);
        }
    }


    // Affichage des messages d'indi_BAO
    // gère aussi le défilement

    if (lognum>10)
    {
        for (int i=lognum-1; i>lognum-11; i--)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+(i-lognum+10)*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }
    else
    {
        for (int i=0; i<lognum; i++)
        {
            if ((logs[i].find("Err")!=string::npos)
                    || (logs[i].find("ALERTE")!=string::npos)) color=rouge;
            else color=vert;
            XDrawString(d, db, color, 10, 9*12+i*12, logs[i].c_str(), strlen(logs[i].c_str()));
        }
    }

    // On affiche la copie de l'écran une fois que tout est dessiné
    // on évite ainsi les scientillements de la fenêtre

    XCopyArea(d, db, w, noir, 0, 0, larg_fenetre, haut_fenetre, 0, 0);

    // On force le rafraichissement de l'écran

    XFlush(d);

    delete [] chaine;
}


/**************************************************************************************
** Gestion des événements affectant la fenêtre graphique
***************************************************************************************/

void ServerDummy::rouler()
{
    XEvent e;

    if (XCheckWindowEvent(d, w, ExposureMask, &e))
    {
        switch (e.type)
        {
        default:
            //cout << "J'ai reçu un événement " <<  e.type << endl
            break;
        case Expose :
            Dessiner();
            break;
        }
    }
}


/**************************************************************************************
** Récupère les messages envoyés par indi_BAO
***************************************************************************************/

void ServerDummy::LireReponse()
{
    string reponse, memreponse, decomp;

    bool test = false;

    // on récupère les messages provenant d'indi_BAO

    *client_socket >> reponse;

    do
    {
        // s'il contient le mot message (et vient donc d'indi_BAO)

        if (reponse.find("message") != string::npos)
        {
            // On ne garde que la partie intéressante
            // La partie qui suit "message="...

            reponse = reponse.substr(reponse.find("message") + 9);

            // (on en garde une trace)

            memreponse = reponse;

            // ...on extrait le message jusqu'au caractère "

            reponse = reponse.substr(0, reponse.find("\""));

            // On récupère l'adresse ip de l'antenne qui nous parle
            // et on met à jour le tableau des états de l'antenne correspondante
            // Dans les messages d'indi_BAO, l'adresse ip précède les mots "Antennes connectées"

            if (reponse.find(" (Antennes connectees")!=string::npos)
            {
                decomp = reponse.substr(0, reponse.find(" (Antennes connectees"));
                decomp = decomp.substr(decomp.rfind(".")+1);

                // si cette antenne est déjà connectée, on change son état si nécessaire

                if ( atoi(decomp.c_str()) != 0 ) for (int i=0; i<numAntennes; i++)
                    {
                        // on a trouvé l'antenne correspondante

                        if (atoi(decomp.c_str()) == Antennes[i].ip)
                        {
                            Antennes[i].ok = 1;
                            test = true;
                            break;
                        }
                    }

                // ou si c'est une nouvelle antenne, on l'ajoute au tableau Antennes

                if (!test)
                {
                    Antennes[numAntennes].ip = atoi(decomp.c_str());

                    Antennes[numAntennes++].ok = 1;
                }
            }

            // erreur sur une antenne -> on change son état dans la fenêtre
            // on trouve les erreurs dans les messages d'indi_BAO formulée
            // de la façon suivante: "ALERTE antenne" ou bien "Erreur sur l antenne"

            if  ((reponse.find("ALERTE antenne ") != string::npos) ||
                    (reponse.find("Erreur sur l antenne ") != string::npos))
            {
                decomp = reponse.substr(0, reponse.find(" :"));
                decomp = decomp.substr(decomp.rfind(".") + 1);

                for (int i=0; i<numAntennes; i++)
                {
                    // On identifie l'antenne dans le tableau Antennes et on change son état

                    if (atoi(decomp.c_str()) == Antennes[i].ip)
                    {
                        Antennes[i].ok = 2;  // Erreur sur l'antenne i
                    }
                }
            }

            // On sauvegarde le message dans le fichier logs mais on ne l'affiche pas

            stringstream os;

            os << lognum << " : " << reponse;

            AfficherLog(os.str() + "\n", false);

            if ( lognum < MAXLOG - 1 ) logs[ lognum++ ] = os.str();

            reponse = memreponse;
        }
        else reponse = "";


    } while ( reponse != "" );

    // on actualise la fenêtre

    Dessiner();

    // on envoie un message pour actualiser la fenêtre graphique

    rouler();
}


/**************************************************************************************
** Est-ce que la réponse du pilote indi_BAO est conforme à ce que l'on attendait ?
***************************************************************************************/

bool ServerDummy::VerifReponse(string reponseattendue, string *retourreponse)
{
    string reponse = "";
    string memreponse = "";
    int    duree = 0;
    bool   test = false;


    // TODO :La réponse du driver ne vient pas tout de suite et on peut recevoir
    // des messages intermédiaires un peu étranges impliquant la fonction CONNECT, pourquoi ?
    // Là on lit les messages cinq fois de suite pour essayer de voir passer le bon
    // Essayer de voir si on ne peut pas faire mieux...

    if (retourreponse) *retourreponse = "";

    for (int i=0; i<5 ; i++)
    {
        do
        {
            usleep(1000); // attendre 1 ms pour laisser le temps au pilote indi_BAO de répondre

            *client_socket >> reponse;

            duree++;
        }
        while (reponse.length() == 0 && duree < MAX_DELAI_REPONSE ); // on attend un message pendant 10 ms max

        // on ajoute tous les messages reçus dans memmreponse
        // ->intéressant pour le debug

        memreponse += reponse;

        if (reponse.find(reponseattendue) != string::npos)
        {
            test = true;
            break;
        }
    }


    if (!test)
    {
        // réponse inconnue -> une erreur ?

       long  unsigned int pos = memreponse.find("message=");

        if ( pos != string::npos )
        {
            memreponse = memreponse.substr(pos + 9);

            memreponse = memreponse.substr(0, memreponse.find("\""));
        }

        if (memreponse != "")  ErreurLog("Réponse du pilote indi_BAO :" + memreponse + "\n\n");

        return false;
    }

    if (retourreponse)
    {
        long unsigned int pos = memreponse.find(reponseattendue);

        if ( pos != string::npos )
        {
            memreponse = memreponse.substr(pos);

            memreponse = memreponse.substr(0, memreponse.find("\""));
        }

        *retourreponse = memreponse;
    }

    return true;
}


/**************************************************************************************
** Calcul du jour julien et du temps sidéral local
***************************************************************************************/

void ServerDummy::Update()
{
    struct tm date;
    time_t t;
    struct timeval tv;
    struct timezone tz;

    // On récupère la date et l'heure depuis l'horloge du système

    time(&t);
    date=*gmtime(&t);
    gettimeofday(&tv, &tz);

    double Annee = (double)date.tm_year + 1900.0;
    double Mois  = (double)date.tm_mon + 1.0;
    double Jour  = (double)date.tm_mday;
    double Heu   = (double)date.tm_hour;
    double Min   = (double)date.tm_min;
    double Sec   = (double)date.tm_sec + tv.tv_usec / 1.0E6;
    //double UTCP  = 0.0;//(double)date.tm_isdst;

    /*Annee=2011;
    Mois=6;
    Jour=17;
    Heu=15;
    Min=53.0;
    Sec=00.0;*/

    // On actualise la date et l'heure dans la classe Astro

    DefinirDateHeure(Annee, Mois, Jour, Heu, Min, Sec);


    // On lance le calcul

    CalculTSL();
}



/**************************************************************************************
** Envoie la longitude et la latitude du lieu d'observation au pilote indi_BAO
** sous la forme de vecteurs xml
** (voir les particularités des drivers développés sous Indi)
**
***************************************************************************************/

bool ServerDummy::EnvoyerCoordGeographiques()
{
    // la communication avec le pilote indi_BAO
    // se fait par l'envoi de trames au format xml

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"GEOGRAPHIC_COORD\">";
        *client_socket << "<oneNumber name=\"LAT\">";
        *client_socket << LatitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"LONG\">";
        *client_socket << LongitudeChar;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"GEOGRAPHIC_COORD\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerCoordGeographiques : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les coordonnées géographiques ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}


/**************************************************************************************
** Transmet la pression et la température au pilote indi_BAO
**
***************************************************************************************/

bool ServerDummy::EnvoyerPressionTemperature()
{
    stringstream os;

    try
    {
        *client_socket << "<newNumberVector device=\"BAO\" name=\"PRESSION_DATA\">";
        *client_socket << "<oneNumber name=\"Pression\">";
        os << Pression;
        *client_socket << os.str();
        os.str("");
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"Temperature\">";
        os << Temperature;
        *client_socket << os.str();
        os.str("");
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"PRESSION_DATA\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerPressionTemperature : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("La pression et la température ont bien été envoyées au pilote indi_BAO\n\n", true);

    return true;
}

/**************************************************************************************
** Envoie les paramètres delaitransit et delaitracking au pilote indi_BAO
** Ces paramètres définissent la durée (en sec) entre deux actualisations
** dans les modes tracking et transit
**
***************************************************************************************/

bool ServerDummy::EnvoyerDelaisModesTransitEtTracking()
{
    stringstream os;

    try
    {
        os << delaitransit;
        *client_socket << "<newNumberVector device=\"BAO\" name=\"DELAY1\">";
        *client_socket << "<oneNumber name=\"DELAY\">";
        *client_socket << os.str();
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";
        os.str("");

        if (!VerifReponse("name=\"DELAY1\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerDelaisModesTransitEtTracking : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }

        os << delaitracking;
        *client_socket << "<newNumberVector device=\"BAO\" name=\"DELAY2\">";
        *client_socket << "<oneNumber name=\"DELAY\">";
        *client_socket << os.str();
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";
        os.str("");

        if (!VerifReponse("name=\"DELAY2\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerDelaisModesTransitEtTracking : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    return true;
}



/**************************************************************************************
** Envoie la méthode d'alignement au pilote indi_bao
** Ce paramètre peut être SIMPLE, AFFINE ou TAKI
**
***************************************************************************************/

bool ServerDummy::EnvoyerMethodeAlignement()
{
    stringstream os;

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_SET\">";
        *client_socket << "<oneSwitch name=\"SIMPLE\">";
        (MethodeAlignement == SIMPLE) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "<oneSwitch name=\"AFFINE\">";
        (MethodeAlignement == AFFINE) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "<oneSwitch name=\"TAKI\">";
        (MethodeAlignement == TAKI)   ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ALIGNMENT_SET\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerMethodeAlignement : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }
    }

    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    return true;
}


/**************************************************************************************
** Envoie une commande aux antennes
** exemples : P, Z, A, Gf1000f0100
**
***************************************************************************************/

bool ServerDummy::EnvoyerCommande(string commande)
{
    stringstream os;

    try
    {
        *client_socket << "<newTextVector device=\"BAO\" name=\"COMMAND_SET\">";
        *client_socket << "<oneText name=\"COMMAND\">";
        *client_socket << commande;
        *client_socket << "</oneText>";
        *client_socket << "</newTextVector>";

        if (!VerifReponse("name=\"COMMAND_SET\" state=\"Ok\"", NULL))
        {
            ErreurLog("ERREUR fct EnvoyerCommande : pas de réponse du pilote indi_BAO.\n\n");

            return false;
        }

    }


    // un problème ?

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    return true;
}



/**************************************************************************************
** Place les antennes en position de repos
**
***************************************************************************************/

bool ServerDummy::Park()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"\" >";
        *client_socket << "<oneSwitch name=\"PARK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("PARK OK", NULL))
        {
            ErreurLog("La commande PARK a échoué.\n\n");;

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien reçu la commande PARK.\n\n", true);

    return true;
}



/**************************************************************************************
** Annule le mouvement en cours
**
***************************************************************************************/

bool ServerDummy::Abort()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ABORT_MOTION\" >";
        *client_socket << "<oneSwitch name=\"ABORT\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("ABORT OK", NULL))
        {
            ErreurLog("L'annulation a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien reçu la commande ABORT.\n\n", true);

    return true;
}


/**************************************************************************************
** Alignement de l'antenne ip
**
***************************************************************************************/

bool ServerDummy::AlignementIP(string ip)
{
    try
    {
        *client_socket << "<newTextVector device=\"BAO\" name=\"ALIGNEMENT_IP\">";
        *client_socket << "<oneText name=\"IP\">";
        *client_socket << ip;
        *client_socket << "</oneText>";
        *client_socket << "</newTextVector>";


        if (!VerifReponse("est maintenant prete pour l alignement", NULL))
        {
            ErreurLog("La sélection de l'antenne située à l'adresse ip a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Le pilote indi_BAO a bien sélectionné l'antenne situé à l'adresse ip pour l'alignement.\n\n", true);

    return true;
}


/**************************************************************************************
** Augmenter ou diminuer le delta en ascension droite ou en déclinaison
**
***************************************************************************************/

bool ServerDummy::AlignementDelta(string delta, bool azimut)
{
    string Message = "";

    try
    {
        if (azimut)
        {
            if (delta == "-1")
            {   *client_socket << "<newSwitchVector device=\"BAO\" name=\"AZ\" >";
                *client_socket << "<oneSwitch name=\"AZM\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";
            }
            else
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"AZ\" >";
                *client_socket << "<oneSwitch name=\"AZP\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";
            }

            if (!VerifReponse("Delta", &Message))
            {
                ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (delta == "-1")
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALT_N\" >";
                *client_socket << "<oneSwitch name=\"ALTN\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";

                if (!VerifReponse("Delta", &Message))
                {
                    ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                    return false;
                }
            }
            else
            {
                *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALT_P\" >";
                *client_socket << "<oneSwitch name=\"ALTP\">";
                *client_socket << "On";
                *client_socket << "</oneSwitch>";
                *client_socket << "</newSwitchVector>";

                if (!VerifReponse("Delta", &Message))
                {
                    ErreurLog("BAOcontrol:AlignementDelta a échoué.\n\n");

                    return false;
                }
            }
        }

        if (Message != "") AfficherLog(Message + "\n\n", true);
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

//   AfficherLog("Le pilote indi_BAO a bien sélectionné l'antenne situé à l'adresse ip pour l'alignement.\n\n", true);

    return true;
}


/**************************************************************************************
** Validation de l'alignement sur l'étoile visée actuellement
**
***************************************************************************************/

bool ServerDummy::ValidationAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_OK\" >";
        *client_socket << "<oneSwitch name=\"OK\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("alignement ont ete valides", NULL))
        {
            ErreurLog("L'alignement sur l'étoile n'a pu être validé.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("L'alignement sur l'étoile a bien été validé.\n\n", true);

    return true;
}


/**************************************************************************************
** Sauvegarde de l'alignement
**
***************************************************************************************/

bool ServerDummy::SauvegardeAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_OK\" >";
        *client_socket << "<oneSwitch name=\"SAUV\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("Les parametres de l alignement ont ete sauvegardes", NULL))
        {
            ErreurLog("La sauvegarde des paramètres de l'alignement a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les paramètres de l'alignemeent ont bien été sauvegardés.\n\n", true);

    return true;
}



/**************************************************************************************
** Sauvegarde de l'alignement
**
***************************************************************************************/

bool ServerDummy::ResetAlignement()
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ALIGNMENT_RESET\" >";
        *client_socket << "<oneSwitch name=\"RESET\">";
        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("reinitialises", NULL))
        {
            ErreurLog("Impossible de réinitialiser les paramètres d'alignement de l'antenne.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les paramètres d'alignement de l'antenne ont été réinitialisés.\n\n", true);

    return true;
}


/**************************************************************************************
** Sélection vitesse alignement
**
***************************************************************************************/

bool ServerDummy::VitesseAlignement(int vit)
{
    string Message = "";

    try
    {
        if (vit == 1)
        {
            *client_socket << "<newSwitchVector device=\"BAO\" name=\"RAQ\" >";
            *client_socket << "<oneSwitch name=\"RAQ1\">";
            *client_socket << "On";
            *client_socket << "</oneSwitch>";
            *client_socket << "</newSwitchVector>";

            if (!VerifReponse("La vitesse de la raquette est fixee a 1x", &Message))
            {
                ErreurLog("Impossible de modifier la vitesse de la raquette.\n\n");

                return false;
            }
        }
        else
        {
            *client_socket << "<newSwitchVector device=\"BAO\" name=\"RAQ\" >";
            *client_socket << "<oneSwitch name=\"RAQ10\">";
            *client_socket << "On";
            *client_socket << "</oneSwitch>";
            *client_socket << "</newSwitchVector>";

            if (!VerifReponse("La vitesse de la raquette est fixee a 10x", &Message))
            {
                ErreurLog("Impossible de modifier la vitesse de la raquette.\n\n");

                return false;
            }
        }



    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    if (Message != "") AfficherLog(Message +"\n\n", true);

    return true;
}

/**************************************************************************************
** Dirige l'antenne vers les coordonnées ar et dec et suit l'objet  en activant le mode
** transit ou tracking.
**
** si J2000 == true, cela signifie que les coordonnées ar et dec sont données dans le
** le système de coordonnées J2000 (écliptique et équinoxe du 1 janvier 2000 à 0 h TU)
** Des calculs supplémentaires (précession, nutation, aberration) sont alors réalisés
** pour ramener les coordonnées horaires à l'écliptique et l'équinoxe de la date de
** l'observation.
**
***************************************************************************************/

bool ServerDummy::Goto(string ar, string dec, bool Transit, bool J2000)
{
    double arf, decf;
    float  ar1, ar2, ar3;
    float  dec1, dec2, dec3;


    // Conversion de l'AD et de la déclinaison en radians

    Decomposition(ar, 2, &ar1, &ar2, &ar3);

    Decomposition(dec, 0, &dec1, &dec2, &dec3);

    arf  = ( ar1  + ar2 / 60.0  + ar3 / 3600.0 ) * 15.0 * Pidiv180;

    decf = ( fabs(dec1) + dec2 / 60.0 + dec3 / 3600.0 ) * Pidiv180;

    if (dec[0] == '-') decf = -decf;

    // Calculs supplémentaires pour ramener les coordonnées horaires
    // à l'époque de l'observation

    if ( J2000 )
    {
        Precession(&arf, &decf);
        NutationEtoile(&arf, &decf);
        AberrationAnnuelle(&arf, &decf);
    }

    // on réécrit l'ar et la dec dans le format d'indi_BAO

    ar  = DHMS(arf * N180divPi, true);

    dec = DHMS(decf * N180divPi, false);

    // on en informe l'utilisateur

    AfficherLog("Coordonnées apparentes de l'objet :\n", true);

    AfficherLog("AD=" + ar  + "   Dec=" + dec +"\n", true);


    // On transmet les coordonnées au pilote BAO

    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"ON_COORD_SET\">";
        *client_socket << "<oneSwitch name=\"TRANSIT\">";
        (Transit) ? *client_socket << "On" : *client_socket << "Off";
        *client_socket << "</oneSwitch>";

        *client_socket << "<oneSwitch name=\"TRACKING\">";
        (Transit) ? *client_socket << "Off" : *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (!VerifReponse("name=\"ON_COORD_SET\"", NULL))
        {
            ErreurLog("Le changement de mode TRANSIT/TRACKING a échoué.\n\n");

            return false;
        }

        *client_socket << "<newNumberVector device=\"BAO\" name=\"EQUATORIAL_EOD_COORD_REQUEST\">";
        *client_socket << "<oneNumber name=\"RA\">";
        *client_socket << ar;
        *client_socket << "</oneNumber>";
        *client_socket << "<oneNumber name=\"DEC\">";
        *client_socket << dec;
        *client_socket << "</oneNumber>";
        *client_socket << "</newNumberVector>";

        if (!VerifReponse("name=\"EQUATORIAL_EOD_COORD_REQUEST\"", NULL))
        {
            ErreurLog("Le transfert des coordonnées de l'objet a échoué.\n\n");

            return false;
        }
    }
    catch ( SocketException& e)
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    AfficherLog("Les nouvelles coordonnées AD=" + ar + " Dec=" + dec, true);
    AfficherLog(" ont été envoyées au pilote indi_BAO.\n\n", true);

    return true;
}

/**************************************************************************************
** Se connecte ou se déconnecte au pilote indi_BAO
**
***************************************************************************************/

bool ServerDummy::Connect(bool connect)
{
    try
    {
        *client_socket << "<newSwitchVector device=\"BAO\" name=\"CONNECTION\">";

        (connect) ? *client_socket << "<oneSwitch name=\"CONNECT\">" : *client_socket << "<oneSwitch name=\"DISCONNECT\">";

        *client_socket << "On";
        *client_socket << "</oneSwitch>";
        *client_socket << "</newSwitchVector>";

        if (connect)
        {
            if (!VerifReponse("BAORadio is online", NULL))
            {
                ErreurLog("La connexion a échoué.\n\n");

                return false;
            }
        }
        else
        {
            if (!VerifReponse("BAORadio is offline", NULL))
            {
                ErreurLog("La déconnexion a échoué.\n\n");

                return false;
            }
        }
    }

    catch ( SocketException& e )
    {
        ErreurLog("Exception was caught:" + e.description() + "\n");

        return false;
    }

    if (connect)
    {
        AfficherLog("La connexion a été établie avec le pilote indi_BAO.\n\n", true);

        // Au moment de la connexion avec le driver indi_BAO
        // On transmet les paramètres du lieu de l'observation, le mode de suivi
        // ainsi que la méthode d'alignement utilisée

        usleep(500000);

        EnvoyerCoordGeographiques();

        EnvoyerPressionTemperature();

        EnvoyerDelaisModesTransitEtTracking();

        EnvoyerMethodeAlignement();
    }
    else
    {
        AfficherLog("BAOcontrol s'est bien déconnecté du pilote indi_BAO.\n\n", true);
    }

    return true;
}



/**************************************************************************************
** Décomposition et vérification des dates, heures, AD et déclinaisons
**
** type = 0 -> Déclinaison/latitude
** type = 1 -> longitude
** type = 2 -> AD ou heure
** type = 3 -> date
**
** Exemple de Decomposition("15:23:12", 2, a, b, c) retourne a=15, b=23, c=12
** si la chaine a un format incorrect, la fct retourne false
***************************************************************************************/

bool ServerDummy::Decomposition(string chaine, char type, float *a1, float *a2, float *a3)
{
    string car, s;
    float a, b, c;

    // pour les heures et les coordonnées, on attend ":" comme caractère séparateur, sinon
    // on attend d'avoir des "/" pour une date

    (type==3) ? car="/" : car=":";

    // Y a-t-il 2 caractères ':' ou '/' dans la chaine ?
    // C'est indispensable dans tous les cas

    int test=0;
    for (int i=0; i<(int)chaine.length(); i++) if (chaine[i] == car[0]) test++;
    if (test<2) return false;

    // Extraction des trois nombres

    s = chaine.substr(0, chaine.find(car));

    a = atoi(s.c_str());

    s = chaine.substr(chaine.find(car)+1, chaine.rfind(car) - chaine.find(car) - 1);

    b = atoi(s.c_str());

    s = chaine.substr(chaine.rfind(car)+1);

    c = Arrondi(atof(s.c_str())*100.0)/100.0;

    //vérification de la cohérence des infos contenues dans la chaine

    if (type < 3 )
    {
        // pour une déclinaison
        if ((type==0) && (a>90.0  || a<-90.0)) return false;
        // pour une AD
        if ((type==1) && (a>360.0 || a<0.0  )) return false;
        // pour une heure
        if ((type==2) && (a>23.0  || a<0.0  )) return false;
        // pour les minutes
        if (b<0.0 || b>59.0 ) return false;
        //pour les secondes
        if (c<0.0 || c>=60.0) return false;
    }
    else
    {
        //pour les jours
        if (a<0.0 || a>31.0) return false;
        //pour les mois
        if (b<0.0 || b>12.0) return false;
    }

    if (a1!=NULL) *a1 = a;
    if (a2!=NULL) *a2 = b;
    if (a3!=NULL) *a3 = c;

    return true;
}






/**************************************************************************************
** Lecteur d'un fichier ligne après ligne
** retourne -1 à la fin
***************************************************************************************/

int ServerDummy::readline (FILE * pfile, char *tab)
{
    int nbchar = 0;
    char c;

    while ((c = getc (pfile)) != '\n')
    {
        if (c == EOF)
        {
            break;
        }
        tab[nbchar++] = c;
    }
    tab[nbchar] = '\0';

    while (nbchar>0 && tab[nbchar-1]==' ')
    {
        tab[--nbchar] = '\0';
    }

    while (tab[0]==' ') for (int i=1; i<=nbchar; i++) tab[i-1] = tab[i];

    if (c == EOF) nbchar=-1;

    return (nbchar);
}



/**************************************************************************************
** chargement des paramètres contenus dans le fichier params
***************************************************************************************/

bool ServerDummy::ChargementParametres(string fileName)
{
     string section;
    string key;
    char * value;
    stringstream os;


    // Si le fichier n'existe pas -> on sort

    if (!is_readable(fileName)) return false;


    AfficherLog("Lecture du fichier de configuration 'params' :\n\n", true);


    /////////////////////////////////////////
    // Rubrique Coordonnées géographiques

    section = "coordonnees geographiques";

    key = "latitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 0, NULL, NULL, NULL)))
    {
        LatitudeChar = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("latitude = " + LatitudeChar +"\n", true);
    }
    else
    {
        ErreurLog("La latitude est incorrecte !\n");

        return false;
    }


    key = "longitude";

    if ((readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
            && (Decomposition(value, 1, NULL, NULL, NULL)))
    {
        LongitudeChar = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("longitude = " + LongitudeChar +"\n\n", true);
    }
    else
    {
        ErreurLog("La longitude est incorrecte !\n");

        return false;
    }


    /////////////////////////////////////////
    // rubrique connexion

    section = "connexion indi";

    key = "serveur";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Serveur = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("serveur = " + Serveur +"\n", true);
    }
    else
    {
        ErreurLog("Nom du serveur invalide !\n");

        return false;
    }

    /////////////////////////////////////////
    // Port de connexion du serveur Indi

    key = "port";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Port = (string)value;
        SAFEDELETE_TAB(value);
        AfficherLog("port = " + Port +"\n\n", true);
    }
    else
    {
        ErreurLog("Numéro de port incorrect !\n");

        return false;
    }


    /////////////////////////////////////////
    // Rubrique paramètres de l'atmosphère

    section = "atmosphere";

    key = "pression";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Pression = atof(value);
        SAFEDELETE_TAB(value);
        os << "pression = " << Pression << endl;
        AfficherLog(&os, true);
    }
    else
    {
        os << "La pression atmosphérique est incorrecte !" << endl;
        ErreurLog(&os);

        return false;
    }

    key = "temperature";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Temperature = atof(value);
        SAFEDELETE_TAB(value);
        os << "température = " << Temperature << endl << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("La température est incorrecte !\n");

        return false;
    }

    /////////////////////////////////////////
    // Rubrique alignement

    section = "alignement";

    key = "methode_alignement";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        AfficherLog("methode d'alignement = " + (string)value + "\n", true);
        switch (value[0])
        {
        case 's' :
        case 'S' :
            MethodeAlignement = SIMPLE;
            break;

        case 'a' :
        case 'A' :
            MethodeAlignement = AFFINE;
            break;

        case 't' :
        case 'T' :
            MethodeAlignement = TAKI;
            break;
        }

        SAFEDELETE_TAB(value);
    }
    else
    {
        ErreurLog("Le paramètre methode_alignement est incorrect !\n");

        return false;
    }



    /////////////////////////////////////////
    // Rubrique suivi

    section = "suivi";

    key = "mode";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        Transit = (strstr(value, "transit") != NULL);
        AfficherLog("mode suivi = " + (string)value + "\n", true);
        SAFEDELETE_TAB(value);
    }
    else
    {
        ErreurLog("Le paramètre mode suivi est incorrect !\n");

        return false;
    }



    key = "delai_transit";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitransit = atoi(value);
        SAFEDELETE_TAB(value);
        os << "delai transit = " << delaitransit << " sec" << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("Le paramètre delai_transit est incorrect !\n");

        return false;
    }


    key = "delai_tracking";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        delaitracking = atoi(value);
        SAFEDELETE_TAB(value);
        os << "delai tracking = " << delaitracking << " sec" << endl << endl;
        AfficherLog(&os, true);
    }
    else
    {
        ErreurLog("Le paramètre delai_tracking est incorrect !\n");

        return false;
    }


    /////////////////////////////////////////
    // Rubrique divers

    section = "divers";

    key = "couleurs";

    if (readINI(section.c_str(), key.c_str(), &value, fileName.c_str()))
    {
        ChoixCouleurs=atoi(value);
        SAFEDELETE_TAB(value);
    }
    else
    {
        /*os << "Le paramètre couleurs est incorrect !" << endl;
        ErreurLog(os.str());
        return false;*/
    }

    return true;
}



/**************************************************************************************
** Le fichier file existe-t-il et n'est-il pas vide ?
**
**************************************************************************************/

bool ServerDummy::is_readable( const string & file )
{
    ifstream fichier( file.c_str() );
    if (fichier.fail()) return false;

    // sauvegarder la position courante

 //   long pos = fichier.tellg();

    // se placer en fin de fichier

    fichier.seekg( 0 , ios_base::end );

    // récupérer la nouvelle position = la taille du fichier

    long size = fichier.tellg();

    return (size > 0);
}


