/*
The stellarium telescope library helps building
telescope server programs, that can communicate with stellarium
by means of the stellarium TCP telescope protocol.
It also contains smaple server classes (dummy, Meade LX200).

Author and Copyright of this file and of the stellarium telescope library:
Johannes Gajdosik, 2006

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _SERVER_BAO_HPP_
#define _SERVER_BAO_HPP_

#include "Server.hpp"
#include <iostream>
#include <string.h>
#include <sstream>
#include "../../communs/ClientSocket.h"
#include "../../communs/SocketException.h"
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include "../../communs/const.h"
#include "../../communs/astro.h"
#include "../../communs/logs.h"
#include "../../communs/alignement.h"
#include "math.h"
#include <time.h>
#include <unistd.h>
#include <sys/time.h>


using namespace std;


// Paramètres des étoiles utilisées pour l'alignement des antennes

struct DefEtoiles
{
    string nom;                         // Nom de l'objet
    string cons;                        // Nom de la constellation
    double ad;				// Ascension droite J2000 de l'objet (en rad)
    double de;				// déclinaison J2000 de l'objet (en rad)
    double az;                          // Azimut (en rad)
    double ha;                          // hauteur au-dessus de l'horizon (en rad)
    double mag;				// magnitude
    bool   selectionnee;                // l'objet a-t-il été sélectionné pour calibrer l'antenne ?
};



// structure définissant les paramètres des antennes

struct DefAntenne
{
    unsigned char ip;			// Adresse ip d'une antenne. Seul le dernier nombre nous intéresse

    char ok;				// actuellement disponible ? 
                                        // 0 = antenne non connectée
                                        // 1 = antenne connectée fonctionnant normalement
                                        // 2 = antenne connectée mais présentant un défaut

    Alignement *AlignementAntenne;      // Contient les paramètres d'alignement de l'antenne définie par cette structure  
};



// objets ou zone du ciel stockés dans le fichier des mouvements planifiés
// Se reporter à la notice de la commande run pour savoir comment utiliser ces fichiers de mouvement

struct DefObjets
{
    double JJ;				// Date julienne du début de l'observation de l'objet i défini dans le fichier de mouvements
    double Duree;			// Durée de l'observation de l'objet en sec
    string ad;				// Ascension droite de l'objet dans le repère J2000
    string de;				// Déclinaison de l'objet dans le repère J2000
    bool exec;				// L'objet est-il actuellement observé ?
};


//! Telescope server class for a virtual telescope that requires no physical device.
//! Useful for remote connection testing.
class ServerDummy : public Server2, public Astro, public Logs
{
public:
	ServerDummy(int port);
	void step(long long int timeout_micros);

        void Affiche(string Message, bool AfficheEcran);
        void Affiche(stringstream *Message, bool AfficheEcran);
        void Erreur(string chaine);
        void Erreur(stringstream *Message);
        void initialiserFenetre();
        void Dessiner();
        void rouler();
        void LireReponse();
        bool VerifReponse(string reponseattendue, string *Message);
        void Update();
     bool Connect(bool connect);
    bool EnvoyerCoordGeographiques();
    bool EnvoyerPressionTemperature();
    bool EnvoyerDelaisModesTransitEtTracking();
    bool EnvoyerMethodeAlignement();
    bool EnvoyerCommande(string commande);
    bool Park();
    bool Abort();
    bool Goto(string ar, string dec, bool Transit, bool J2000);
    bool AlignementAntenneIP(string ip);
   
    bool AlignementIP(string ip);
    bool AlignementDelta(string delta, bool azimut);
    bool ValidationAlignement();
    bool SauvegardeAlignement();
    bool ResetAlignement();
    bool VitesseAlignement(int vit);
        bool Decomposition(string chaine, char type, float *a1, float *a2, float *a3);
        int  readline (FILE * pfile, char *tab);
        bool ChargementParametres(string fileName);
	
	bool is_readable( const string & file );
	
private:
	void gotoReceived(unsigned int ra_int,int dec_int);
	double current_pos[3];
	double desired_pos[3];
	double last_pos[3];
	long long int next_pos_time;

       // Variables globales

    XFontStruct * fd;                           // Police de caractères utilisée dans la fenêtre graphique

    string logs[MAXLOG];                        // Sauvegarde de toutes les actions et réponses dans un tableau logs

    int delaitransit;				// Délai entre deux actualisations de la position en mode transit
    int delaitracking;				// Délai entre deux actualisations de la position en mode tracking
    int lognum;					// Nbre d'enregistrements dans le tableau logs
    int numAntennes;				// Nbre d'antennes connectées
    int numobjets;				// Nbre d'objets dans le dernier fichier de mouvements chargé
    int runnum;					// Numéro de l'objet actuellement visé dans le fichier de mouvement
    int numEtoiles;				// Nbre d'étoiles utilisées pour la procédure d'alignement des antennes
    int MethodeAlignement;                      // Méthode d'alignement utilisée


    double Pression;				// Pression atmosphérique en mbar
    double Temperature; 			// température de l'atmosphère en °C

    bool NoExit;				// A-t-on tapé exit pour sortir ? Permet de fermer le thread à ce moment-là
    bool Transit;				// Si oui, le mode transit est activé, sinon, c'est le mode tracking
    bool run;					// exécute actuellement un fichier de mouvements
    bool exitrun;				// doit-on sortir du programme à la fin de l'exécution de la commande run ?
    bool J2000;					// est-ce que les coordonnées du fichier de mouvements sont en coordonnées J2000 ?
    bool ModificationAlignement;		// Est-ce que l'une des antennes est en cours d'alignement et est-ce qu'un paramètre 
                                                // d'alignement à changé ?

    struct DefAntenne *Antennes;		// Sauvegarde de la situation actuelle de l'antenne i
    struct DefObjets  *objets;			// Liste des objets chargés depuis un fichier de mouvements
    struct DefEtoiles *Etoiles;			// Liste des étoiles servant à la procédure d'alignement des antennes

    string LatitudeChar;			// Latitude du lieu d'observation. Chargée depuis le fichier de configuration
    string LongitudeChar;			// Longitude du lieu d'observation. Chargée depuis le fichier de configuration
    string Serveur;				// Nom ou IP du serveur faisant tourner indi_BAO. Par defaut contient 'localhost'
    string Port; 				// Port utilisé pour connecter le serveur indi_BAO

    ClientSocket *client_socket;		// Socket permettant la connexion avec le pilote indi_BAO

    Display * d;				// contexte graphique
    Window w;					// fenêtre graphique
    Pixmap db;					// copie de la fenêtre pour permettre une actualisation sans scintillement
   
    GC noir, vert, rouge, gris;			// couleurs utilisées dans la fenêtre graphique
};

#endif
