
// Geant4:
#include <G4ios.hh>
#include <G4RunManager.hh>
#include <G4UImanager.hh>
#include <G4SDManager.hh>

//AIDA :
#include <AIDA/IAnalysisFactory.h>

// std::
#include <iostream>

// Lib :
#include <Lib/Interfaces/IManager.h>
#include <Lib/Interfaces/ISession.h>
#include <Lib/Interfaces/ITypeManager.h>
#include <Lib/Manager.h>
#include <Lib/System.h>
#include <Lib/Out.h>
#include <Lib/Cast.h>

// G4Lab :
#include <G4Lab/UIOnX.h>
#include <G4Lab/TrackingAction.h>
#include <G4Lab/Interfaces/IGeant4Manager.h>
#include <G4Lab/DigitsCollectionType.h>

//MEMPHYS:
#include "MEMPHYS/Analysis.hh"
#include "MEMPHYS/DetectorConstruction.hh"
#include "MEMPHYS/PhysicsList.hh"
#include "MEMPHYS/PhysicsMessenger.hh"
#include "MEMPHYS/PrimaryGeneratorAction.hh"
#include "MEMPHYS/EventAction.hh"
#include "MEMPHYS/RunAction.hh"
#include "MEMPHYS/StackingAction.hh"
#include "MEMPHYS/TrackingAction.hh"
#include "MEMPHYS/SteppingAction.hh"
#include "MEMPHYS/WCDigi.hh"
#include <MEMPHYS/HitsCollectionType.h>

// Create a manager to access the MEMPHYSAnalysis from the callbacks :

//JEC 10/1/06 introduce MEMPHYS namespace
namespace MEMPHYS {

class AppManager 
  :public virtual IManager
  ,public Analysis  {
 public: //IManager
  virtual const std::string& name() const { return fName;}
  virtual void* cast(const std::string& aTo) const {
    if(aTo=="MEMPHYS::AppManager") {
      return Lib_SCast(MEMPHYS::AppManager);
    } else if(aTo=="MEMPHYS::IAppManager") {
      return Lib_SCast(MEMPHYS::IAppManager);
    } else if(aTo=="MEMPHYS::Analysis") {
      return Lib_SCast(MEMPHYS::Analysis);
    } else if(aTo=="IManager") {
      return Lib_SCast(IManager);
    } else {
      return 0;
    }
  }
 public://IAppManager
  virtual bool initialize() {
    //GB :
    // Used in the MEMPHYS_Initialize callback.
    // Do all the below here because of platforms having 
    // not Geant4 shared libs.
    IGeant4Manager* g4Manager = 
      Lib_findManager(fSession,"Geant4Manager",IGeant4Manager);
    if(!g4Manager) {
      Lib::Out out(fSession.printer());
      out << "MEMPHYS_Initialize :"
          << " Geant4Manager not found."
          << Lib::endl;
      return false;
    }

    // To initialize G4Lab Types (PV, Trajectory, HitsCollection, etc.. types).
    // To initialize G4Lab SoNodes.
    g4Manager->initialize();

    // Overload the HitsCollectionType of G4Lab :
    Lib::ITypeManager* typeManager = 
      Lib_findManager(fSession,"TypeManager",Lib::ITypeManager);
    if(typeManager) {
      G4SDManager* sdManager = g4Manager->sdManager();
      if(sdManager) {
        G4HCtable* hcTable  = sdManager->GetHCtable();
        if(hcTable) {
          int number = hcTable->entries();
          for(int index=0;index<number;index++) {
            G4String hcName = hcTable->GetHCname(index);
            typeManager->removeType(hcName);
            typeManager->addType
              (new MEMPHYS::HitsCollectionType(fSession,hcName));
          }
        }
      }

      typeManager->addType
        (new G4Lab::DigitsCollectionType<MEMPHYS::WCDigi>
          (fSession,"WCDigitizedCollection"));
    }
    return true;
  }
 public:
  AppManager(ISession& aSession,AIDA::IAnalysisFactory& aAIDA,bool aBatch)
    :Analysis(aAIDA,aBatch),fSession(aSession),fName("MEMPHYS::AppManager") {}
  virtual ~AppManager(){}
 private:
  ISession& fSession;
  std::string fName;
};
}

//////////////////////////////////////////////////////////////////////////////
int main(
	 int aArgc
	 ,char** aArgv
	 ) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  //Lib::Debug::checkByClass(true);

  G4RunManager* runManager = new G4RunManager;

  // Need to pass the G4RunManager at UIOnX creation
  std::string gui = "$MEMPHYSROOT/scripts/OnX/MEMPHYS_session.onx";
  G4Lab::UIOnX* session = new G4Lab::UIOnX(*runManager,gui,aArgc,aArgv);

  MEMPHYS::AppManager* appManager = 0;

  AIDA::IAnalysisFactory* aida = session->findAIDA();
  if(!aida) {
    std::cout << "MEMPHYS_session : AIDA not found." << std::endl;
  } else {
    appManager = new MEMPHYS::AppManager(*session->session(),*aida,false);
    session->addManager(appManager);
  }

  G4UImanager* UI = G4UImanager::GetUIpointer();
  if (UI) {
    std::string file = 
      Lib::System::getenv("MEMPHYSROOT")+"/scripts/Geant4/jobOptions.mac";
    UI->ApplyCommand("/control/execute "+file);  
  }

  
  //JEC 18/11/05 give the appManager to fill geometry ITuple
  MEMPHYS::DetectorConstruction* MEMPHYSdetector = 
    new MEMPHYS::DetectorConstruction(*appManager); 

  runManager->SetUserInitialization(MEMPHYSdetector);

  runManager->SetUserInitialization(new MEMPHYS::PhysicsList);

  // Set user action classes
  MEMPHYS::PrimaryGeneratorAction* myGeneratorAction = 
    new MEMPHYS::PrimaryGeneratorAction(MEMPHYSdetector);
  runManager->SetUserAction(myGeneratorAction);

  MEMPHYS::RunAction* myRunAction = new MEMPHYS::RunAction();

  runManager->SetUserAction(myRunAction);


  runManager->SetUserAction(new MEMPHYS::EventAction(*appManager,
						   myRunAction, 
						   MEMPHYSdetector,
						   myGeneratorAction));
  runManager->SetUserAction(new MEMPHYS::TrackingAction);

  runManager->SetUserAction(new MEMPHYS::StackingAction);

  runManager->SetUserAction(new MEMPHYS::SteppingAction); //JEC 15/12/05 Add user SteppingAction

  // Initialize G4 kernel
  runManager->Initialize();

  if (UI) {
    std::string file = 
      Lib::System::getenv("MEMPHYSROOT")+"/scripts/Geant4/init.g4m";
    UI->ApplyCommand("/control/execute "+file);  
  }


  // Main UI loop :
  // GUI is really constructed here. 
  // Create callbacks are executed here too.
  session->SessionStart();

  delete session; //It will delete the appManager.

  delete runManager;

  return 0;
}

