
// this :
#include <G4Lab/Manager.h>

// Other includes :

// Inventor :
#include <Inventor/nodes/SoNode.h>

#ifdef WIN32
#undef pascal // Clash between windef.h and Geant4/SystemOfUnits.hh
#endif

// Geant4 :
#include <G4UImanager.hh>
#include <G4UIsession.hh>
#include <G4RunManager.hh>
#include <G4UIcommandTree.hh>
#include <G4SDManager.hh>
#include <G4DigiManager.hh>

// Lib :
#include <Slash/Core/IManager.h>
#include <Slash/Core/ISession.h>
#include <Slash/Data/IProcessor.h>
#include <Lib/Manager.h>
#include <Lib/Cast.h>
#include <Lib/Out.h>
#include <Lib/sout.h>
#include <Lib/smanip.h>

// OnX :
#include <Slash/UI/IScriptManager.h>

// AIDA :
#include <AIDA/IAnalysisFactory.h>

// G4Lab :
#include <G4Lab/PhysicalVolumeAccessor.h>
#include <G4Lab/TrajectoryContainerAccessor.h>
#include <G4Lab/TrajectoryAccessor.h>
#include <G4Lab/PhysicsTableAccessor.h>
#include <G4Lab/HitsCollectionAccessor.h>
#include <G4Lab/Tree.h>
#include <G4Lab/State.h>
#include <G4Lab/SoG4Trajectories.h>
#include <G4Lab/SoG4RunManager.h>

// From G4VBasicShell (tag geant4-05-00-ref-01)
static G4String Complete(G4String);
static G4String FindMatchingPath(G4UIcommandTree*,G4String);

// Class to redirect G4 output to OnX console and do command completion :
// Avoid establishing a relationship with G4interfaces (by inheriting
// G4VBasicShell). This library may be reconstructed with various GUI drivers.

namespace G4Lab {
class UIsession : public G4UIsession {
public: //G4UIsession
  virtual G4int ReceiveG4cout(G4String aString){
    fOut << aString;
    return 0;
  }
  virtual G4int ReceiveG4cerr(G4String aString){
    fOut << aString;
    return 0;
  }
public:
  UIsession(Slash::Core::IWriter& aPrinter):fOut(aPrinter){}
  std::string completeCommand(const std::string& aString) {
    return Complete(aString);
  }
private:
  Lib::Out fOut;
};

}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
G4Lab::Manager::Manager(
 Slash::Core::ISession& aSession                    
,const std::string& aName
,G4RunManager* aRunManager
,bool aDeleteRunManager
)
:fSession(aSession)
,fName(aName)
,fUIsession(0)
,fState(0)
,fScriptManager(0)
,fRunManager(aRunManager)
,fDeleteRunManager(aDeleteRunManager)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4UImanager* UI = G4UImanager::GetUIpointer();
  if(UI) {
    fUIsession = new UIsession(fSession.printer());
    UI->SetSession(fUIsession);
    UI->SetCoutDestination(fUIsession);
  }

  fScriptManager = 
    Lib_findManager(fSession,"ScriptManager",Slash::UI::IScriptManager);
  if(!fScriptManager) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::Manager :"
        << " ScriptManager not found." 
        << Lib::endl;
  } else {
    fScriptManager->addInterpreter("G4","","","","","",executeScript,0); 
  }

  // The G4Lab::State will be deleted by the run manager.
  fState = new G4Lab::State(fScriptManager);
}
//////////////////////////////////////////////////////////////////////////////
G4Lab::Manager::~Manager(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4UImanager* UI = G4UImanager::GetUIpointer();
  if(UI) {
    UI->SetSession(0);
    UI->SetCoutDestination(0);
  }
  delete fUIsession;

  // We should remove G4Lab things in the AccessorManager...

  // The G4Lab::State is deleted by the run manager.

  if(fDeleteRunManager) {
    delete fRunManager;
    fRunManager = 0;
  } else {
    // The fRunManager is deleted by the user.
  }
}
//////////////////////////////////////////////////////////////////////////////
AIDA::IAnalysisFactory* G4Lab::Manager::findAIDA(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  AIDA::IAnalysisFactory* aida = 
    Lib_findManager(fSession,"AnalysisFactory",AIDA::IAnalysisFactory);
  if(!aida) {
    if(!fScriptManager) return 0;
    fScriptManager->executeScript("Session","OnXLab OnXLab_createAIDA");
    aida = Lib_findManager(fSession,"AnalysisFactory",AIDA::IAnalysisFactory);
  }
  return aida;
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::Manager::initialize(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  AIDA::IAnalysisFactory* aida = 
    Lib_findManager(fSession,"AnalysisFactory",AIDA::IAnalysisFactory);

  Slash::Data::IProcessor* accessorManager = 
    Lib_findManager(fSession,"AccessorManager",Slash::Data::IProcessor);
  if(!accessorManager) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::initialize :"
	<< " AccessorManager not found." << Lib::endl;
    return false;
  } else {
    if(!accessorManager->findAccessor("PV")) {
      accessorManager->addAccessor(new PhysicalVolumeAccessor(fSession));
      accessorManager->addAccessor(new TrajectoryContainerAccessor(fSession));
      accessorManager->addAccessor(new TrajectoryAccessor(fSession));
      accessorManager->addAccessor(new PhysicsTableAccessor(fSession,aida));

      // Declare HitsCollections types :
      if(fRunManager) {
        if(!fRunManager->GetUserDetectorConstruction()) {
          Lib::Out out(fSession.printer());
          out << "G4Lab::Manager::initialize :"
              << " G4RunManager did not receive yet the"
              << " user detector construction class."
              << Lib::endl;   
          out << "G4Lab::Manager::initialize :"
              << " We can't then build the hits collection types."
              << Lib::endl;   
        } else {
          fRunManager->Initialize(); //FIXME : should be able to check if done.
          G4SDManager* sdManager = G4SDManager::GetSDMpointer();
          if(sdManager) {
            G4HCtable* hcTable  = sdManager->GetHCtable();
            if(hcTable) {
              int number = hcTable->entries();
              for(int index=0;index<number;index++) {
                G4String hcName = hcTable->GetHCname(index);
                accessorManager->addAccessor
                  (new HitsCollectionAccessor(fSession,hcName));
                if(fSession.verboseLevel()) {
                  Lib::Out out(fSession.printer());
                  out << "G4Lab::Manager::initialize :"
  	            << " declare HitsCollection type " << Lib::sout(hcName)
                      << Lib::endl;
                }
              }
            }
          }
        }
      }

      //Declare the generic HitsCollection type :
      accessorManager->addAccessor(new HitsCollectionAccessor(fSession));

      // Should be initialized once after Inventor and HEPVis :
      SoG4Trajectories::initClass();
      SoG4RunManager::initClass();
    }
  }
  return true;
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::addAccessor(
 Slash::Data::IAccessor* aAccessor
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  Slash::Data::IProcessor* accessorManager = 
    Lib_findManager(fSession,"AccessorManager",Slash::Data::IProcessor);
  if(!accessorManager) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::addAccessor :"
	<< " AccessorManager not found." << Lib::endl;
    return;
  }
  accessorManager->addAccessor(aAccessor);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::executeG4Script(
 const std::string& aString
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fScriptManager) return;
  fScriptManager->executeScript("G4",aString);
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Manager::completeCommand(
 const std::string& aString
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fUIsession) return "";
  return fUIsession->completeCommand(aString);
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::Manager::executeScript(
 const std::string& aString
,void*
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4UImanager* UI = G4UImanager::GetUIpointer();
  if(!UI) return false;
  std::vector<std::string> text;
  Lib::smanip::lines(aString,text);
  int linen = text.size();
  if(linen) {
    for(int count=0;count<linen;count++) {
      std::string line = text[count];
      Lib::smanip::strip(line);
      if( (line[0]=='\0') || (line[0]=='#') ) continue;
      UI->ApplyCommand(line);
    }
  }
  return true;
}
#include <G4TransportationManager.hh>
#include <G4Navigator.hh>
#include <G4ParticleTable.hh>
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Manager::physicalVolumes(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
#define UNLIMITED (-1)
  G4TransportationManager* tsp = 
    G4TransportationManager::GetTransportationManager();
  if(!tsp) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::physicalVolumes : " 
        << "can't find the G4TransportationManager."
        << Lib::endl;
    return "";
  }
  G4Navigator* nav = tsp->GetNavigatorForTracking();
  if(!nav) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::physicalVolumes : " 
        << "can't get the G4Navigator on G4TransportationManager." 
        << Lib::endl;
    return "";
  }
  G4VPhysicalVolume* topVolume = nav->GetWorldVolume(); 
  if(!topVolume) {
    Lib::Out out(fSession.printer());
    out << "G4Lab::Manager::physicalVolumes : " 
        << "no world volume in G4Navigator." << Lib::endl;
    return "";
  }
  std::string sout;
  sout += "<tree>";
  XML_VisitedVolume xmlVisitedVolume(sout);
  GeometryVisitor geometryVisitor;
  geometryVisitor.visit(topVolume,
                        UNLIMITED,
                        G4Transform3D(),
                        &xmlVisitedVolume);
  sout += "</tree>";
  return sout;
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Manager::hitsCollections(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4SDManager* sdManager = G4SDManager::GetSDMpointer();
  if(!sdManager) return "";
  G4HCtable* hcTable  = sdManager->GetHCtable();
  if(!hcTable) return "";
  std::string sout;
  sout += "<tree>";
  int number = hcTable->entries();
  for(int index=0;index<number;index++) {
    G4String hcName = hcTable->GetHCname(index);
    sout += "<treeItem><label>";
    sout += hcName;
    sout += "</label></treeItem>";
  }
  sout += "</tree>";
  return sout;
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Manager::digitsCollections(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4DigiManager* digiManager = G4DigiManager::GetDMpointer();
  if(!digiManager) return "";
  G4DCtable* dcTable  = digiManager->GetDCtable();
  if(!dcTable) return "";
  std::string sout;
  sout += "<tree>";
  int number = dcTable->entries();
  for(int index=0;index<number;index++) {
    G4String dcName = dcTable->GetDCname(index);
    sout += "<treeItem><label>";
    sout += dcName;
    sout += "</label></treeItem>";
  }
  sout += "</tree>";
  return sout;
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::setRunBeginScript(
 const std::string& aInterp
,const std::string& aScript
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fState->setRunBeginScript(aInterp,aScript);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::setRunEndScript(
 const std::string& aInterp
,const std::string& aScript
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fState->setRunEndScript(aInterp,aScript);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::setEventBeginScript(
 const std::string& aInterp
,const std::string& aScript
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fState->setEventBeginScript(aInterp,aScript);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::setEventEndScript(
 const std::string& aInterp
,const std::string& aScript
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fState->setEventEndScript(aInterp,aScript);
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Manager::name(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fName;
}
//////////////////////////////////////////////////////////////////////////////
void* G4Lab::Manager::cast(
 const std::string& aClass
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if_Lib_SCast(G4Lab::Manager)
  else if_Lib_SCast(Slash::Core::IManager)
  else if_Lib_SCast(IGeant4Manager)
  else return 0;
}
//////////////////////////////////////////////////////////////////////////////
G4RunManager* G4Lab::Manager::runManager(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fRunManager;
}
//////////////////////////////////////////////////////////////////////////////
G4SDManager* G4Lab::Manager::sdManager(
) const
//////////////////////////////////////////////////////////////////////////////
// Used by callbacks if having not shared libs for Geant4.
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return G4SDManager::GetSDMpointer();
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::beamOn(
 int aNumber
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fRunManager) return;
  fRunManager->BeamOn(aNumber);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::abortRun(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fRunManager) return;
  fRunManager->AbortRun();
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::Manager::isRunning(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fState->isRunning();
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Manager::particleTableDump(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4ParticleTable* table = G4ParticleTable::GetParticleTable();
  if(!table) return;
  table->DumpTable("ALL");
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// From G4VBasicShell (tag geant4-05-00-ref-01)
//////////////////////////////////////////////////////////////////////////////
static G4String Complete(
 G4String commandName
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4String rawCommandLine = commandName;
  G4String commandLine = rawCommandLine.strip(G4String::both);
  size_t i = commandLine.index(" ");
  if( i != std::string::npos ) return rawCommandLine; // Already entering parameters, 
                                            // assume command path is correct.
  G4String commandString = commandLine; 
  //G4String targetCom = ModifyPath(commandString);
  G4String targetCom = commandString;
  G4UIcommandTree* tree = G4UImanager::GetUIpointer()->GetTree();
  G4String value = FindMatchingPath(tree,targetCom);
  if(value=="") return rawCommandLine;
  return value;
}
//////////////////////////////////////////////////////////////////////////////
static G4String FindMatchingPath(
 G4UIcommandTree* aTree
,G4String aCommandPath
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
// From intercoms/src/G4UIcommandTree::FindPath.
{
  G4String empty = "";
  if(aTree==NULL) return empty;
  G4String pathName = aTree->GetPathName();
  if( aCommandPath.index( pathName ) == std::string::npos ) return empty;
  G4String remainingPath = aCommandPath;
  remainingPath.remove(0,pathName.length());
  size_t i = remainingPath.first('/');
  if( i == std::string::npos ) {
    // Look for number of matching commands :
    std::vector<G4UIcommand*> commands;
    G4int n_commandEntry = aTree->GetCommandEntry();
    for( G4int i_thCommand = 1; i_thCommand <= n_commandEntry; i_thCommand++ ) {
      G4UIcommand* cmd = aTree->GetCommand(i_thCommand);
      G4String ss = cmd->GetCommandName();
      ss.resize(remainingPath.length());
      if( remainingPath == ss ) commands.push_back(cmd);
    }
    n_commandEntry = commands.size();
    if(n_commandEntry==1) {
      return (pathName + commands[0]->GetCommandName());
    } else if (n_commandEntry>=2) {
      G4cout << "Matching commands :" << G4endl; 
      for( G4int i_thCommand = 0; i_thCommand < n_commandEntry; i_thCommand++ ) {
        G4UIcommand* cmd = commands[i_thCommand];
        G4cout << cmd->GetCommandName() << G4endl; 
      }
      return empty;
    }
    // Look for sub tree :
    std::vector<G4UIcommandTree*> trees;
    G4String nextPath = pathName;
    nextPath.append(remainingPath);
    G4int n_treeEntry = aTree->GetTreeEntry();
    for( G4int i_thTree = 1; i_thTree <= n_treeEntry; i_thTree++ ) {
      G4UIcommandTree* tree = aTree->GetTree(i_thTree);
      G4String ss = tree->GetPathName();
      ss.resize(nextPath.length());
      if( nextPath == ss ) trees.push_back(tree);
    }
    n_treeEntry = trees.size();
    if(n_treeEntry==1) {
      return trees[0]->GetPathName();
    } else if (n_treeEntry>=2) {
      G4cout << "Matching directories :" << G4endl; 
      for( G4int i_thTree = 0; i_thTree < n_treeEntry; i_thTree++ ) {
        G4UIcommandTree* tree = trees[i_thTree];
        G4cout << tree->GetPathName() << G4endl; 
      }
      return empty;
    } else {
      return empty; // No match.
    }
  } else {
    // Find path
    G4String nextPath = pathName;
    nextPath.append(remainingPath(0,i+1));
    G4int n_treeEntry = aTree->GetTreeEntry();
    for( G4int i_thTree = 1; i_thTree <= n_treeEntry; i_thTree++ ) {
      G4UIcommandTree* tree = aTree->GetTree(i_thTree);
      if( nextPath == tree->GetPathName() ) { 
        return FindMatchingPath(tree,aCommandPath ); 
      }
    }
  }
  return empty;
}
