// this :
#include <G4Lab/TrajectoryAccessor.h>

// Inventor :
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/engines/SoTimeCounter.h>
#include <Inventor/engines/SoCalculator.h>

// HEPVis :
#include <HEPVis/nodes/SoHighlightMaterial.h>

#ifdef WIN32
#undef pascal // Clash between windef.h and Geant4/SystemOfUnits.hh
#endif

// Geant4 :
#include <G4RunManager.hh>

// Slash :
#include <Slash/Core/ISession.h>
#include <Slash/Data/IIterator.h>

// Lib :
#include <Lib/Debug.h>
#include <Lib/Out.h>
#include <Lib/Value.h>
#include <Lib/smanip.h>

// G4Lab :
#include <G4Lab/Trajectory.h>
#include <G4Lab/SoG4Trajectories.h>

//////////////////////////////////////////////////////////////////////////////
G4Lab::TrajectoryAccessor::TrajectoryAccessor(
 Slash::Core::ISession& aSession
)
:OnX::InventorAccessor(aSession)
,fType("Trajectory")
,fModeling("immediate_all")
,fG4Lab(false)
,fSeparator(0)
,fCoordinate(0)
,fIndex(0)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  Lib::Debug::increment("G4Lab::TrajectoryAccessor");
}
//////////////////////////////////////////////////////////////////////////////
G4Lab::TrajectoryAccessor::~TrajectoryAccessor(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  Lib::Debug::decrement("G4Lab::TrajectoryAccessor");
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::TrajectoryAccessor::name(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fType;
}
namespace G4Lab {
  class TrajectoryIterator : public virtual Slash::Data::IIterator {
  public: //Slash::Data::IIterator
    virtual Slash::Data::IAccessor::Data object() {
      if(fIterator==fVector.end()) return 0;
      return *fIterator;
    }
    virtual void next() { ++fIterator; }
    virtual void* tag() { return 0;}
  public:
    TrajectoryIterator(G4TrajectoryContainer& aVector)
      :fVector(*(aVector.GetVector())) {
      fIterator = fVector.begin();
    }
    virtual ~TrajectoryIterator() {}
  private:
    TrajectoryVector& fVector;
    TrajectoryVector::iterator fIterator;
  };
}
//////////////////////////////////////////////////////////////////////////////
Slash::Data::IIterator* G4Lab::TrajectoryAccessor::iterator(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  // Set properties here since a user G4Trajectory may
  // had been declared very lately.
  clearProperties();
  // G4Trajectory properties :
  addProperty("particle",Lib::Property::STRING);
  addProperty("track",Lib::Property::INTEGER);
  addProperty("parent",Lib::Property::INTEGER);
  addProperty("charge",Lib::Property::DOUBLE);
  if(isG4LabTrajectory()) {
    // G4Lab::Trajectory properties :
    addProperty("energy",Lib::Property::DOUBLE);
    addProperty("totalEnergy",Lib::Property::DOUBLE);
    addProperty("globalTime",Lib::Property::DOUBLE);
  }
  //addProperty("id",Lib::Property::POINTER);

  G4RunManager* runManager = G4RunManager::GetRunManager();
  if(!runManager) return 0;
  const G4Event* event = runManager->GetCurrentEvent();
  if(!event) {
    Lib::Out out(fSession.printer());
    out << "No event." << Lib::endl;
    return 0;
  }
  G4TrajectoryContainer* trajectoryContainer = 
    event->GetTrajectoryContainer();
  if(!trajectoryContainer) {
    Lib::Out out(fSession.printer());
    out << "No trajectory container." << Lib::endl;
    return 0;
  }

  fG4Lab = isG4LabTrajectory();

  return new TrajectoryIterator(*trajectoryContainer);
}
//////////////////////////////////////////////////////////////////////////////
Slash::Core::IValue* G4Lab::TrajectoryAccessor::findValue(
 Slash::Data::IAccessor::Data aData
,const std::string& aName
,void*
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4Trajectory* obj = (G4Trajectory*)aData;
  //if(aName=="id") {
    //return new Lib::Value((void*)obj);
  //} else 
  if(aName=="particle") {
    return new Lib::Value(obj->GetParticleName());
  } else if(aName=="track") {
    return new Lib::Value(obj->GetTrackID());
  } else if(aName=="parent") {
    return new Lib::Value(obj->GetParentID());
  } else if(aName=="charge") {
    return new Lib::Value(obj->GetCharge());
  } else if(!fG4Lab) {
    return new Lib::Value();
  } else { // We have G4Lab::Trajectory objects :
    G4Lab::Trajectory* obj = (G4Lab::Trajectory*)aData;
    if(aName=="energy") {
      return new Lib::Value(obj->GetKineticEnergy());
    } else if(aName=="totalEnergy") {
      return new Lib::Value(obj->GetTotalEnergy());
    } else if(aName=="globalTime") {
      return new Lib::Value(obj->GetGlobalTime());
    } else {
      return new Lib::Value();
    }
  }
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::TrajectoryAccessor::beginVisualize(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  OnX::InventorAccessor::beginVisualize();

  // modeling.trajectories:immediate_all
  // modeling.trajectories:immediate_time
  // modeling.trajectories:pickable
  if(!fSession.parameterValue("modeling.trajectories",fModeling))
    fModeling = "immediate_all";

  //fSession.out().println("debug : modeling \"%s\".",fModeling.c_str());

  if(fModeling=="pickable") {

    fSeparator = new SoSeparator;
    fCoordinate = new SoCoordinate3;
    fSeparator->addChild(fCoordinate);
    fIndex = 0;

  } else if( (fModeling=="immediate_all") || (fModeling=="immediate_time") ) {
    // Scene graph :
    SoSeparator* separator = new SoSeparator;
    separator->setName("sceneGraph");
    
    separator->addChild(fSoGC.getHighlightMaterial());
    separator->addChild(fSoGC.getDrawStyle());
    separator->addChild(fSoGC.getLightModel());

    SoG4Trajectories* soG4Trajectories = new SoG4Trajectories;
    separator->addChild(soG4Trajectories);

    if(fModeling=="immediate_time") {
      soG4Trajectories->model.setValue(SoG4Trajectories::TIMED);
      //soG4Trajectories->timeSteps.setValue(1000); 

      SoCalculator* calculator = new SoCalculator();
      calculator->a.connectFrom(&soG4Trajectories->timeSteps);
      calculator->expression.set1Value(0,"oa=2.0/a");

      SoTimeCounter* timeCounter = new SoTimeCounter;
#ifdef WIN32
#undef max
#endif
      timeCounter->max.connectFrom(&soG4Trajectories->timeSteps);
      timeCounter->frequency.connectFrom(&calculator->oa);
      timeCounter->reset.connectFrom(&soG4Trajectories->timeIndex);

      soG4Trajectories->timeIndex.connectFrom(&timeCounter->output);

    }

    fSoRegion->doIt(SbAddNode(separator,"dynamicScene"));
  }
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::TrajectoryAccessor::visualize(
 Slash::Data::IAccessor::Data aData
,void*
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fSeparator) return; // not in "pickable" mode.

  G4Trajectory* obj = (G4Trajectory*)aData;
  if(!obj) return;

  int pointn = obj->GetPointEntries();
  if(pointn<=0) return;
  
  SbVec3f* points = new SbVec3f[pointn];
  int32_t* coordIndex = new int32_t[pointn+1];

  G4TrajectoryPoint* tp;
  G4ThreeVector pos;
  for (int i = 0; i < pointn ; i++) {
    tp = (G4TrajectoryPoint*)(obj->GetPoint(i)); //?
    pos = tp->GetPosition();
    points[i].setValue((float)pos.x(),(float)pos.y(),(float)pos.z());
    coordIndex[i] = fIndex+i;
  }
  coordIndex[pointn] = SO_END_LINE_INDEX;
  
  //Get a style corresponding to Trajectory type.
 {std::string s = obj->GetParticleName();
  Lib::smanip::strip(s);
  std::string style = "Trajectory("+s+")";
  if(!isStyle(style)) {
    style = "Trajectory_"+s; //Backward compatibility.
    if(!isStyle(style)) {
      style = "Trajectory"; //Default.
    }
  }
  fillSoGC(style);}

  // Build name (for picking) :
  std::string s;
  Lib::smanip::printf(s,128,"Trajectory/0x%lx",(unsigned long)obj);
  SbName name(s.c_str());

  // Scene graph :
  SoSeparator* separator = new SoSeparator;
  separator->setName("sceneGraph");
  fSeparator->addChild(separator);
  
  separator->addChild(fSoGC.getHighlightMaterial());
  separator->addChild(fSoGC.getDrawStyle());
  separator->addChild(fSoGC.getLightModel());

  fCoordinate->point.setValues(fIndex,pointn,points);
  fIndex += pointn;

  SoIndexedLineSet* lineSet = new SoIndexedLineSet;
  lineSet->coordIndex.setValues(0,pointn+1,coordIndex);
  lineSet->setName(name);
  separator->addChild(lineSet);

  delete [] coordIndex;
  delete [] points;
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::TrajectoryAccessor::endVisualize(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(fSeparator) {
    if(fSeparator->getNumChildren()==1) {
      fSeparator->unref();
    } else {
      fSoRegion->doIt(SbAddNode(fSeparator,"dynamicScene"));
    }
    fSeparator = 0;
    fCoordinate = 0;
    fIndex = 0;
  }
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::TrajectoryAccessor::isG4LabTrajectory(
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4RunManager* runManager = G4RunManager::GetRunManager();
  if(!runManager) return false;
  const G4Event* event = runManager->GetCurrentEvent();
  if(!event) return false;
  G4TrajectoryContainer* trajectoryContainer = 
    event->GetTrajectoryContainer();
  if(!trajectoryContainer) return false;
  int number = trajectoryContainer->entries();
  if(number<=0) return false;
  G4Lab::Trajectory* trajectory = 
    dynamic_cast<G4Lab::Trajectory*>((G4Trajectory*)(*trajectoryContainer)[(size_t)0]);
  return (trajectory?true:false);
}
