// this :
#include <G4Lab/DigitsCollectionAccessor.h>

// Inventor :
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoCoordinate3.h>

// HEPVis :
#include <HEPVis/misc/SoStyleCache.h>
#include <HEPVis/nodes/SoPolyhedron.h>
#include <HEPVis/nodes/SoHighlightMaterial.h>

#ifdef WIN32
#undef pascal // Clash between windef.h and Geant4/SystemOfUnits.hh
#endif

// Geant4 :
#include <G4RunManager.hh>
#include <G4Event.hh>
#include <G4DigiManager.hh>
#include <G4DCofThisEvent.hh>
#include <G4LogicalVolume.hh>
#include <G4Transform3D.hh>
#include <G4AttDef.hh>
#include <G4AttValue.hh>
#include <G4Polyhedron.hh>
#include <G4Colour.hh>
#include <G4VDigi.hh>
#include <G4VDigiCollection.hh>

// Lib :
#include <Slash/Core/ISession.h>
#include <Slash/Data/IIterator.h>
#include <Lib/Out.h>
#include <Lib/Value.h>
#include <Lib/smanip.h>
#include <Lib/sout.h>

// G4Lab :
#include <G4Lab/Transform3D.h>
#include <G4Lab/Polyhedron.h>

//////////////////////////////////////////////////////////////////////////////
G4Lab::DigitsCollectionBaseAccessor::DigitsCollectionBaseAccessor(
 Slash::Core::ISession& aSession
,G4RunManager* aRunManager
,const std::string& aDC
)
:OnX::InventorAccessor(aSession)
,fType("DC")
,fRunManager(aRunManager)
,fAttDefs(0)
,fDC(aDC)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(fDC!="") fType = fDC;
}
//////////////////////////////////////////////////////////////////////////////
G4Lab::DigitsCollectionBaseAccessor::~DigitsCollectionBaseAccessor(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::DigitsCollectionBaseAccessor::name(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fType;
}
//////////////////////////////////////////////////////////////////////////////
const std::string& G4Lab::DigitsCollectionBaseAccessor::DCName(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fDC;
}
//////////////////////////////////////////////////////////////////////////////
Slash::Core::IValue* G4Lab::DigitsCollectionBaseAccessor::findValue(
 Slash::Data::IAccessor::Data aData
,const std::string& aName
,void*
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fAttDefs) {
    Lib::Out out(printer());
    out << "G4Lab::DigitsCollectionBaseAccessor::value :" 
        << " problem to get AttDefs fot digi collection " << Lib::sout(fDC) 
        << "." 
        << Lib::endl;
    return new Lib::Value();
  }

  G4VDigi* obj = (G4VDigi*)aData;  
  std::vector<G4AttValue>* vec = obj->CreateAttValues();
  if(!vec) {
    Lib::Out out(printer());
    out << "G4Lab::DigitsCollectionBaseAccessor::value :" 
        << " problem to get AttValues fot digi collection " << Lib::sout(fDC) 
        << "." 
        << Lib::endl;
    return new Lib::Value();
  }
  
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(aName==val.GetName()) {
      const std::string& stype = (*fAttDefs)[val.GetName()].GetValueType();
      if(stype=="G4int") {
        int v;
        if(!Lib::smanip::toint(val.GetValue(),v)) {
          Lib::Out out(printer());
          out << "G4Lab::DigitsCollectionBaseAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a G4int." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        delete vec;
        return new Lib::Value(v);
      } else if((stype=="G4double")||(stype=="G4float")) {
        double v;
        if(!Lib::smanip::todouble(val.GetValue(),v)) {
          Lib::Out out(printer());
          out << "G4Lab::DigitsCollectionBaseAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a double." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        delete vec;
        return new Lib::Value(v);
      } else {
        delete vec;
        return new Lib::Value(val.GetValue());
      }
    }
  }
  Lib::Out out(printer());
  out << "G4Lab::DigitsCollectionBaseAccessor::value :" 
      << " AttValue not found for property " << Lib::sout(aName) << "." 
      << Lib::endl;
  delete vec;
  return new Lib::Value();
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::DigitsCollectionBaseAccessor::beginVisualize(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  OnX::InventorAccessor::beginVisualize();

  std::string style = "DC("+fDC+")";
  if(!isStyle(style)) style = "DC"; //Default.
  fillSoGC(style);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::DigitsCollectionBaseAccessor::visualize(
 Slash::Data::IAccessor::Data aData
,void*
) 
//////////////////////////////////////////////////////////////////////////////
// The digi must have the "LV" and "TSF" AttDef.
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4VDigi* obj = (G4VDigi*)aData;  
  G4LogicalVolume* lv = digiLogicalVolume(*obj);
  if(!lv) return;
  G4Transform3D* tsf = digiTransform3D(*obj);
  if(!tsf) return;
  G4VSolid* solid = lv->GetSolid();
  if(!solid) return;
  //G4Material* material = lv->GetMaterial();

  int old_nstep = HepPolyhedron::GetNumberOfRotationSteps();
  HepPolyhedron::SetNumberOfRotationSteps(fSoGC.getRotationSteps());
  G4Polyhedron* g4Polyhedron = solid->CreatePolyhedron();
  HepPolyhedron::SetNumberOfRotationSteps(old_nstep);
  if(!g4Polyhedron) return;

  G4Lab::Polyhedron* polyhedron = new G4Lab::Polyhedron(*g4Polyhedron);
  delete g4Polyhedron;
  if(!polyhedron) return;

  SoSeparator* separator = new SoSeparator;
  separator->setName("sceneGraph");
      
  G4Colour color;
  if(digiColor(*obj,color)) {
    SbColor sbColor((float)color.GetRed(),
                    (float)color.GetGreen(),
                    (float)color.GetBlue());
    float transp = 1.0F - (float)color.GetAlpha();
    SoStyleCache* styleCache = fSoGC.getStyleCache();
    separator->addChild(
      styleCache->getHighlightMaterial
        (sbColor,fSoGC.getHighlightColor(),transp));
  } else {
    separator->addChild(fSoGC.getHighlightMaterial());
  }

  separator->addChild(fSoGC.getDrawStyle());
  separator->addChild(fSoGC.getLightModel());

  SoTransform* transform = new SoTransform;
  G4Lab::Transform3D* t = new G4Lab::Transform3D(*tsf);
  SbMatrix* matrix = t->getMatrix();
  transform->setMatrix(*matrix);
  delete matrix;
  delete t;
  separator->addChild(transform);
          
  // Build name (for picking) :
  std::string s;
  Lib::smanip::printf(s,128,"%s/0x%lx",fDC.c_str(),(unsigned long)obj);
  SbName name(s.c_str());
  //FIXME : can't cache due to the below setName for picking.
  //FIXME SoPolyhedron* soPolyhedron = fSoGC.getPolyhedron(*polyhedron);
  SoPolyhedron* soPolyhedron = new SoPolyhedron(*polyhedron);
  if(fSoGC.getModeling()==SbModeling_wire_frame) {
    soPolyhedron->solid.setValue(FALSE);
    //FIXME : handle reduceWireFrame.
    soPolyhedron->reducedWireFrame.setValue(TRUE);
  } else {
    soPolyhedron->solid.setValue(TRUE);
  }
  delete polyhedron;
  soPolyhedron->setName(name);
        
  separator->addChild(soPolyhedron);
          
  fSoRegion->doIt(SbAddNode(separator,"dynamicScene"));
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::DigitsCollectionBaseAccessor::setProperties(
 G4VDigi& aDigi
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fAttDefs = (std::map<G4String,G4AttDef>*)aDigi.GetAttDefs();
  if(!fAttDefs) return;
  std::map<G4String,G4AttDef>::const_iterator it;
  for(it=fAttDefs->begin();it!=fAttDefs->end();++it) {
    const std::string& sname = (*it).second.GetName();
    if((sname!="LV") && 
       (sname!="TSF") && 
       (sname!="Color")) {
      const std::string& stype = (*it).second.GetValueType();    
      if(stype=="G4int") {
        addProperty((*it).first,Lib::Property::INTEGER);
      } else if((stype=="G4double")||(stype=="G4float")) {
        addProperty((*it).first,Lib::Property::DOUBLE);
      } else {
        addProperty((*it).first,Lib::Property::STRING);
      }
    }
  }
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
G4VDigiCollection* G4Lab::DigitsCollectionBaseAccessor::getCollection(
 Slash::Core::IWriter& aPrinter
,const std::string& aName
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fRunManager) {
    Lib::Out out(aPrinter);
    out << "No G4RunManager." << Lib::endl;
    return 0;
  }
  const G4Event* event = fRunManager->GetCurrentEvent();
  if(!event) {
    Lib::Out out(aPrinter);
    out << "No event." << Lib::endl;
    return 0;
  }
  G4DigiManager* digiManager = G4DigiManager::GetDMpointer();
  if(!digiManager) {
    Lib::Out out(aPrinter);
    out << "No G4DigiManager." << Lib::endl;
    return 0;
  }

  // Get Digi collection of this event
  G4DCofThisEvent* dce  = event->GetDCofThisEvent();
  if(!dce) {
    Lib::Out out(aPrinter);
    out << "No G4DCofThisEvent." << Lib::endl;
    return 0;
  }
  G4int cid = digiManager->GetDigiCollectionID(aName);
  int nc = dce->GetNumberOfCollections();
  if((cid<0)||(cid>=nc)) {
    Lib::Out out(aPrinter);
    out << "Collection id not found for " << Lib::sout(aName) << Lib::endl;
    return 0;
  }

  G4VDigiCollection* dc = dce->GetDC(cid);
  if(!dce) {
    Lib::Out out(aPrinter);
    out << "No G4VDigiCollection for " << Lib::sout(aName) << Lib::endl;
    return 0;
  }
  return dc;
}
//////////////////////////////////////////////////////////////////////////////
G4LogicalVolume* G4Lab::DigitsCollectionBaseAccessor::digiLogicalVolume(
 const G4VDigi& aDigi
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  std::vector<G4AttValue>* vec = aDigi.CreateAttValues();
  if(!vec) return 0;
  std::string value;
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(val.GetName()=="LV") {
      std::string s = val.GetValue();
      void* p;
      if(!Lib::smanip::topointer(s,p)) return 0;
      return (G4LogicalVolume*)p; //Beurk
    }
  }
  return 0;
}
//////////////////////////////////////////////////////////////////////////////
G4Transform3D* G4Lab::DigitsCollectionBaseAccessor::digiTransform3D(
 const G4VDigi& aDigi
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  std::vector<G4AttValue>* vec = aDigi.CreateAttValues();
  if(!vec) return 0;
  std::string value;
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(val.GetName()=="TSF") {
      std::string s = val.GetValue();
      void* p;
      if(!Lib::smanip::topointer(s,p)) return 0;
      return (G4Transform3D*)p; //Beurk
    }
  }
  return 0;
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::DigitsCollectionBaseAccessor::digiColor(
 const G4VDigi& aDigi
,G4Colour& aColor
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{  
  std::vector<G4AttValue>* vec = aDigi.CreateAttValues();
  if(vec) {
    std::string value;
    unsigned int number = vec->size();
    for(unsigned int index=0;index<number;index++) {
      const G4AttValue& val = (*vec)[index];
      if(val.GetName()=="Color") {
        std::string s = val.GetValue();
        double r,g,b,a;
        if(!Lib::smanip::torgba(s,r,g,b,a)) break;
        aColor = G4Colour(r,g,b,a);
        return true;
      }
    }
  }
  aColor = G4Colour();
  return false;
}
