// this :
#include <G4Lab/HitsCollectionAccessor.h>

// Inventor :
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>

// HEPVis :
#include <HEPVis/misc/SoStyleCache.h>
#include <HEPVis/nodes/SoPolyhedron.h>
#include <HEPVis/nodes/SoHighlightMaterial.h>

#ifdef WIN32
#undef pascal // Clash between windef.h and Geant4/SystemOfUnits.hh
#endif

// Geant4 :
#include <G4RunManager.hh>
#include <G4Event.hh>
#include <G4SDManager.hh>
#include <G4HCofThisEvent.hh>
#include <G4LogicalVolume.hh>
#include <G4Transform3D.hh>
#include <G4AttDef.hh>
#include <G4AttValue.hh>
#include <G4Polyhedron.hh>
#include <G4Colour.hh>

// Lib :
#include <Slash/Core/ISession.h>
#include <Slash/Data/IIterator.h>
#include <Lib/Out.h>
#include <Lib/Value.h>
#include <Lib/smanip.h>
#include <Lib/sout.h>

// G4Lab :
#include <G4Lab/Transform3D.h>
#include <G4Lab/Polyhedron.h>

//////////////////////////////////////////////////////////////////////////////
G4Lab::HitsCollectionAccessor::HitsCollectionAccessor(
 Slash::Core::ISession& aSession
,G4RunManager* aRunManager
,const std::string& aHC
)
:OnX::InventorAccessor(aSession)
,fType("HC")
,fRunManager(aRunManager)
,fAttDefs(0)
,fHC(aHC)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(fHC!="") fType = fHC;
}
//////////////////////////////////////////////////////////////////////////////
G4Lab::HitsCollectionAccessor::~HitsCollectionAccessor(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::HitsCollectionAccessor::name(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fType;
}
//////////////////////////////////////////////////////////////////////////////
const std::string& G4Lab::HitsCollectionAccessor::HCName(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fHC;
}
namespace G4Lab {
  class HitsCollectionIterator : public virtual Slash::Data::IIterator {
  public: //Slash::Data::IIterator
    virtual Slash::Data::IAccessor::Data object() {
      if(fCurrent>=fVector.GetSize()) return 0;
      G4VHit* hit = fVector.GetHit(fCurrent);
      return hit;
    }
    virtual void next() { fCurrent++;}
    virtual void* tag() { return 0;}
  public:
    HitsCollectionIterator(G4VHitsCollection& aVector)
    :fVector(aVector),fCurrent(0) {}
    virtual ~HitsCollectionIterator() {}
  private:
    G4VHitsCollection& fVector;
    unsigned int fCurrent;
  };
}
//////////////////////////////////////////////////////////////////////////////
Slash::Data::IIterator* G4Lab::HitsCollectionAccessor::iterator(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  const std::vector<std::string>& args = iteratorArguments();
  if(args.size()) fHC = args[0];

  // Initialize properties here since we declare the hit AttDefs
  // as properties and then we need at least one hit.
  clearProperties();
  fAttDefs = 0;

  G4VHitsCollection* hc = getCollection(printer(),fHC);
  if(!hc) return 0;

  if(hc->GetSize()) {
    G4VHit* hit = hc->GetHit(0);
    if(hit) {
      fAttDefs = (std::map<G4String,G4AttDef>*)hit->GetAttDefs();
      if(fAttDefs) {
        std::map<G4String,G4AttDef>::const_iterator it;
        for(it=fAttDefs->begin();it!=fAttDefs->end();++it) {
          const std::string& sname = (*it).second.GetName();
          if((sname!="LV") && 
             (sname!="TSF") && 
             (sname!="Color")) {
            const std::string& stype = (*it).second.GetValueType();    
            if(stype=="G4int") {
              addProperty((*it).first,Lib::Property::INTEGER);
            } else if((stype=="G4double")||(stype=="G4float")) {
              addProperty((*it).first,Lib::Property::DOUBLE);
            } else if((stype=="std::vector<G4float>")||
                      (stype=="std::vector<G4double>")) {
              addProperty((*it).first,Lib::Property::VECTOR_DOUBLE);
            } else {
              addProperty((*it).first,Lib::Property::STRING);
            }
          }
        }
      }
    }
  }

  return new HitsCollectionIterator(*hc);
}
//////////////////////////////////////////////////////////////////////////////
Slash::Core::IValue* G4Lab::HitsCollectionAccessor::findValue(
 Slash::Data::IAccessor::Data aData
,const std::string& aName
,void*
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fAttDefs) {
    Lib::Out out(printer());
    out << "G4Lab::HitsCollectionAccessor::value :" 
        << " problem to get AttDefs fot hit collection " << Lib::sout(fHC) 
        << "." 
        << Lib::endl;
    return new Lib::Value();
  }

  G4VHit* obj = (G4VHit*)aData;  
  std::vector<G4AttValue>* vec = obj->CreateAttValues();
  if(!vec) {
    Lib::Out out(printer());
    out << "G4Lab::HitsCollectionAccessor::value :" 
        << " problem to get AttValues fot hit collection " << Lib::sout(fHC) 
        << "." 
        << Lib::endl;
    return new Lib::Value();
  }
  
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(aName==val.GetName()) {
      const std::string& stype = (*fAttDefs)[val.GetName()].GetValueType();
      if(stype=="G4int") {
        int v;
        if(!Lib::smanip::toint(val.GetValue(),v)) {
          Lib::Out out(printer());
          out << "G4Lab::HitsCollectionAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a G4int." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        delete vec;
        return new Lib::Value(v);
      } else if((stype=="G4double")||(stype=="G4float")) {
        double v;
        if(!Lib::smanip::todouble(val.GetValue(),v)) {
          Lib::Out out(printer());
          out << "G4Lab::HitsCollectionAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a double." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        delete vec;
        return new Lib::Value(v);
      } else if(stype=="std::vector<G4float>") {
        void* p;
        if(!Lib::smanip::topointer(val.GetValue(),p)) {
          Lib::Out out(printer());
          out << "G4Lab::HitsCollectionAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a pointer." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        std::vector<float>* vvec = (std::vector<float>*)p;
        unsigned int vn = vvec->size();
	std::vector<double> array(vn);
        for(unsigned int vi=0;vi<vn;vi++) array[vi] = double((*vvec)[vi]);
        delete vec;
        return new Lib::Value(array);
      } else if(stype=="std::vector<G4double>") {
        void* p;
        if(!Lib::smanip::topointer(val.GetValue(),p)) {
          Lib::Out out(printer());
          out << "G4Lab::HitsCollectionAccessor::value :" 
              << " " << Lib::sout(val.GetValue()) << " not a pointer." 
              << Lib::endl;
          delete vec;
          return new Lib::Value();
        }
        std::vector<double>* vvec = (std::vector<double>*)p;
        delete vec;
        return new Lib::Value(*vvec);
      } else {
        delete vec;
        return new Lib::Value(val.GetValue());
      }
    }
  }
  Lib::Out out(printer());
  out << "G4Lab::HitsCollectionAccessor::value :" 
      << " AttValue not found for property " << Lib::sout(aName) << "." 
      << Lib::endl;
  delete vec;
  return new Lib::Value();
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::HitsCollectionAccessor::beginVisualize(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  OnX::InventorAccessor::beginVisualize();

  std::string style = "HC("+fHC+")";
  if(!isStyle(style)) style = "HC"; //Default.
  fillSoGC(style);
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::HitsCollectionAccessor::visualize(
 Slash::Data::IAccessor::Data aData
,void*
) 
//////////////////////////////////////////////////////////////////////////////
// The hit must have the "LV" and "TSF" AttDef.
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  G4VHit* obj = (G4VHit*)aData;  
  G4LogicalVolume* lv = hitLogicalVolume(*obj);
  if(!lv) return;
  G4Transform3D* tsf = hitTransform3D(*obj);
  if(!tsf) return;
  G4VSolid* solid = lv->GetSolid();
  if(!solid) return;
  //G4Material* material = lv->GetMaterial();

  int old_nstep = HepPolyhedron::GetNumberOfRotationSteps();
  HepPolyhedron::SetNumberOfRotationSteps(fSoGC.getRotationSteps());
  G4Polyhedron* g4Polyhedron = solid->CreatePolyhedron();
  HepPolyhedron::SetNumberOfRotationSteps(old_nstep);
  if(!g4Polyhedron) return;

  G4Lab::Polyhedron* polyhedron = new G4Lab::Polyhedron(*g4Polyhedron);
  delete g4Polyhedron;
  if(!polyhedron) return;

  SoSeparator* separator = new SoSeparator;
  separator->setName("sceneGraph");
      
 {G4Colour color;
  if(hitColor(*obj,color)) {
    SbColor sbColor((float)color.GetRed(),
                    (float)color.GetGreen(),
                    (float)color.GetBlue());
    float transp = 1.0F - (float)color.GetAlpha();
    SoStyleCache* styleCache = fSoGC.getStyleCache();
    separator->addChild(
      styleCache->getHighlightMaterial
        (sbColor,fSoGC.getHighlightColor(),transp));
  } else {
    separator->addChild(fSoGC.getHighlightMaterial());
  }
  separator->addChild(fSoGC.getDrawStyle());
  separator->addChild(fSoGC.getLightModel());}

 {SoTransform* transform = new SoTransform;
  G4Lab::Transform3D* t = new G4Lab::Transform3D(*tsf);
  SbMatrix* matrix = t->getMatrix();
  transform->setMatrix(*matrix);
  delete matrix;
  delete t;
  separator->addChild(transform);}
          
  // Build name (for picking) :
 {std::string s;
  Lib::smanip::printf(s,128,"%s/0x%lx",fHC.c_str(),(unsigned long)obj);
  SbName name(s.c_str());
  //FIXME : can't cache due to the below setName for picking.
  //FIXME SoPolyhedron* soPolyhedron = fSoGC.getPolyhedron(*polyhedron);
  SoPolyhedron* soPolyhedron = new SoPolyhedron(*polyhedron);
  if(fSoGC.getModeling()==SbModeling_wire_frame) {
    soPolyhedron->solid.setValue(FALSE);
    //FIXME : handle reduceWireFrame.
    soPolyhedron->reducedWireFrame.setValue(TRUE);
  } else {
    soPolyhedron->solid.setValue(TRUE);
  }
  delete polyhedron;
  soPolyhedron->setName(name);        
  separator->addChild(soPolyhedron);}
          
  fSoRegion->doIt(SbAddNode(separator,"dynamicScene"));
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
G4VHitsCollection* G4Lab::HitsCollectionAccessor::getCollection(
 Slash::Core::IWriter& aPrinter
,const std::string& aName
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fRunManager) {
    Lib::Out out(aPrinter);
    out << "No G4RunManager." << Lib::endl;
    return 0;
  }
  const G4Event* event = fRunManager->GetCurrentEvent();
  if(!event) {
    Lib::Out out(aPrinter);
    out << "No event." << Lib::endl;
    return 0;
  }
  G4SDManager* sdManager = G4SDManager::GetSDMpointer();
  if(!sdManager) {
    Lib::Out out(aPrinter);
    out << "No G4SDManager." << Lib::endl;
    return 0;
  }

  // Get Hit collection of this event
  G4HCofThisEvent* hce  = event->GetHCofThisEvent();
  if(!hce) {
    Lib::Out out(aPrinter);
    out << "No G4HCofThisEvent." << Lib::endl;
    return 0;
  }
  G4int cid = sdManager->GetCollectionID(aName);
  int nc = hce->GetNumberOfCollections();
  if((cid<0)||(cid>=nc)) {
    Lib::Out out(aPrinter);
    out << "Collection id not found for " << Lib::sout(aName) << Lib::endl;
    return 0;
  }

  G4VHitsCollection* hc = hce->GetHC(cid);
  if(!hce) {
    Lib::Out out(aPrinter);
    out << "No G4VHitsCollection for " << Lib::sout(aName) << Lib::endl;
    return 0;
  }
  return hc;
}
//////////////////////////////////////////////////////////////////////////////
G4LogicalVolume* G4Lab::HitsCollectionAccessor::hitLogicalVolume(
 const G4VHit& aHit
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  std::vector<G4AttValue>* vec = aHit.CreateAttValues();
  if(!vec) return 0;
  std::string value;
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(val.GetName()=="LV") {
      std::string s = val.GetValue();
      void* p;
      if(!Lib::smanip::topointer(s,p)) return 0;
      return (G4LogicalVolume*)p; //Beurk
    }
  }
  return 0;
}
//////////////////////////////////////////////////////////////////////////////
G4Transform3D* G4Lab::HitsCollectionAccessor::hitTransform3D(
 const G4VHit& aHit
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  std::vector<G4AttValue>* vec = aHit.CreateAttValues();
  if(!vec) return 0;
  std::string value;
  unsigned int number = vec->size();
  for(unsigned int index=0;index<number;index++) {
    const G4AttValue& val = (*vec)[index];
    if(val.GetName()=="TSF") {
      std::string s = val.GetValue();
      void* p;
      if(!Lib::smanip::topointer(s,p)) return 0;
      return (G4Transform3D*)p; //Beurk
    }
  }
  return 0;
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::HitsCollectionAccessor::hitColor(
 const G4VHit& aHit
,G4Colour& aColor
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{  
  std::vector<G4AttValue>* vec = aHit.CreateAttValues();
  if(vec) {
    std::string value;
    unsigned int number = vec->size();
    for(unsigned int index=0;index<number;index++) {
      const G4AttValue& val = (*vec)[index];
      if(val.GetName()=="Color") {
        std::string s = val.GetValue();
        double r,g,b,a;
        if(!Lib::smanip::torgba(s,r,g,b,a)) break;
        aColor = G4Colour(r,g,b,a);
        return true;
      }
    }
  }
  aColor = G4Colour();
  return false;
}
