
// this :
#include <G4Lab/Trajectory.h>

// Lib :
#include <Lib/Debug.h>

// Geant4 :
#include <G4Track.hh>
#include <G4VProcess.hh>

static G4Allocator<G4Lab::Trajectory> sTrajectoryAllocator; // Beurk.

namespace G4Lab {
class TrajectoryPoint {
public:
  TrajectoryPoint(G4TrajectoryPoint* aPoint,double aGlobalTime)
    :fPoint(aPoint),fGlobalTime(aGlobalTime){}
  virtual ~TrajectoryPoint() { delete fPoint;}
  G4TrajectoryPoint* point() const { return fPoint;}
  double globalTime() const { return fGlobalTime;}
private:
  G4TrajectoryPoint* fPoint;
  double fGlobalTime;
  Lib::Debug fDebug; // To check memory balance.
};
}

//////////////////////////////////////////////////////////////////////////////
G4Lab::Trajectory::Trajectory(
 const G4Track* aTrack
)
:G4Trajectory(aTrack) 
,fProcess(0)
,fSave(false)
,fStoppingVolume(0)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fKineticEnergy = aTrack->GetKineticEnergy();
  fTotalEnergy = aTrack->GetTotalEnergy();
  fGlobalTime = aTrack->GetGlobalTime();
  fProcess = aTrack->GetCreatorProcess();

  fStoppingPoint  = aTrack->GetPosition();
  fStoppingVolume = aTrack->GetVolume();

  /*
  printf("debug : G4Lab::Trajectory : \"%s\", kineticEnergy : %g, totalEnergy : %g, globalTime : %g\n",
         GetParticleName().c_str(),
         fKineticEnergy,fTotalEnergy,fGlobalTime);
  */
  TrajectoryPoint* tp = 
    new TrajectoryPoint(
     new G4TrajectoryPoint(aTrack->GetPosition()),
     aTrack->GetGlobalTime());
  fPoints.push_back(tp);

}
//////////////////////////////////////////////////////////////////////////////
G4Lab::Trajectory::Trajectory(
 Trajectory& aFrom
)
:G4Trajectory(aFrom) 
//,IGeant4Trajectory(aFrom)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fKineticEnergy = aFrom.fKineticEnergy;
  fTotalEnergy = aFrom.fTotalEnergy;
  fGlobalTime = aFrom.fGlobalTime;
  fProcess = aFrom.fProcess;
  fSave = aFrom.fSave;

  size_t number = aFrom.fPoints.size();
  for(size_t index=0;index<number;index++) {
    G4TrajectoryPoint* point = aFrom.fPoints[index]->point();
    TrajectoryPoint* tp = 
      new TrajectoryPoint(new G4TrajectoryPoint(*point),aFrom.fGlobalTime);
    fPoints.push_back(tp);
  }
  
}
//////////////////////////////////////////////////////////////////////////////
G4Lab::Trajectory::~Trajectory(
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  size_t number = fPoints.size();
  for(size_t index=0;index<number;index++) {
    delete fPoints[index];
  }
  fPoints.clear();
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
double G4Lab::Trajectory::kineticEnergy(
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fKineticEnergy;
}
//////////////////////////////////////////////////////////////////////////////
double G4Lab::Trajectory::totalEnergy(
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fTotalEnergy;
}
//////////////////////////////////////////////////////////////////////////////
double G4Lab::Trajectory::globalTime(
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fGlobalTime;
}
//////////////////////////////////////////////////////////////////////////////
unsigned int G4Lab::Trajectory::pointEntries(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fPoints.size();
}
//////////////////////////////////////////////////////////////////////////////
double G4Lab::Trajectory::pointGlobalTime(
 unsigned int aIndex
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fPoints[aIndex]->globalTime();
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Trajectory::creatorProcessName(
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fProcess) return "";
  return fProcess->GetProcessName();
}
//////////////////////////////////////////////////////////////////////////////
std::string G4Lab::Trajectory::creatorProcessType(
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!fProcess) return "NotDefined";
  switch(fProcess->GetProcessType()){
  case fNotDefined: return "NotDefined";
  case fTransportation: return "Transportation";
  case fElectromagnetic: return "Electromagnetic";
  case fOptical: return "Optical";
  case fHadronic: return "Hadronic";
  case fPhotolepton_hadron: return "Photolepton_hadron";
  case fDecay: return "Decay";
  case fGeneral: return "General";
  case fParameterisation: return "Parameterisation";
  case fUserDefined: return "UserDefined";
  };
  return "NotDefined";
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::setSave(
 bool aFlag
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  fSave = aFlag;
}
//////////////////////////////////////////////////////////////////////////////
bool G4Lab::Trajectory::save(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fSave;
}
//////////////////////////////////////////////////////////////////////////////
std::vector<double> G4Lab::Trajectory::stoppingPoint(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  std::vector<double> v(3);
  v[0] = fStoppingPoint.x();
  v[1] = fStoppingPoint.y();
  v[2] = fStoppingPoint.z();
  return v;
}
//////////////////////////////////////////////////////////////////////////////
G4VPhysicalVolume* G4Lab::Trajectory::stoppingPhysicalVolume(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fStoppingVolume;
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void* G4Lab::Trajectory::operator new(
 size_t
)  
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return (void*)sTrajectoryAllocator.MallocSingle();
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::operator delete(
 void* aTrajectory
) 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  sTrajectoryAllocator.FreeSingle((G4Lab::Trajectory*)aTrajectory);
}
//////////////////////////////////////////////////////////////////////////////
int G4Lab::Trajectory::operator == (
 const G4Lab::Trajectory& aRight
) const 
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return (this==&aRight);
} 
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::ShowTrajectory(
 std::ostream&
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::DrawTrajectory(
 G4int
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::AppendStep(
 const G4Step* aStep
)
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  TrajectoryPoint* tp = 
    new TrajectoryPoint(
      new G4TrajectoryPoint(aStep->GetPostStepPoint()->GetPosition()),
      aStep->GetPostStepPoint()->GetGlobalTime());
  fPoints.push_back(tp);
}
//////////////////////////////////////////////////////////////////////////////
int G4Lab::Trajectory::GetPointEntries(
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fPoints.size();
}
//////////////////////////////////////////////////////////////////////////////
G4VTrajectoryPoint* G4Lab::Trajectory::GetPoint(
 G4int aIndex
) const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  return fPoints[aIndex]->point();
}
//////////////////////////////////////////////////////////////////////////////
void G4Lab::Trajectory::MergeTrajectory(
 G4VTrajectory* aTrajectory
)
//////////////////////////////////////////////////////////////////////////////
// Transfer points from aTrajectory to this.
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  if(!aTrajectory) return;
  Trajectory* tj = (Trajectory*)aTrajectory;
  size_t number = tj->fPoints.size();
  // initial point of the second trajectory should not be merged
  for(size_t index=1;index<number;index++) { 
    fPoints.push_back(tj->fPoints[index]);
  }
  delete tj->fPoints[0];
  tj->fPoints.clear();
}
//////////////////////////////////////////////////////////////////////////////
const std::map<G4String,G4AttDef>* G4Lab::Trajectory::GetAttDefs() const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  //return G4Trajectory::GetAttDefs();
  return 0;
}
//////////////////////////////////////////////////////////////////////////////
std::vector<G4AttValue>* G4Lab::Trajectory::CreateAttValues() const
//////////////////////////////////////////////////////////////////////////////
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
{
  //return G4Trajectory::CreateAttValues();
  return 0;
}

