// -*- mode: C++; -*-
/* prog_snova.cc
 * SuperNEMO module G4 based simulation program
 */

#include <string>

#include <snova/snova_const.hh>
#include <snova/geometry2.hh>         // Geometry definition
#include <snova/physics.hh>           // Physical proceses to consider
#include <snova/generator.hh>         // Initial particles generator
#include <snova/run.hh>               // User actions per run basis
#include <snova/event.hh>             // User actions per event basis
#include <snova/tracking.hh>          // User actions per track basis
#include <snova/stacking.hh>          // User actions related with tracks stacking
#include <snova/stepping.hh>          // User actions per step basis
#include <snova/stepping_verbose.hh>  // Information dump per step basis

#include "G4RunManager.hh"  // Program manager
#include "G4UImanager.hh"   // User Interface manager
#include "G4UIterminal.hh"  // User Interface manager through the terminal

#ifdef G4UI_USE_TCSH
#include "G4UItcsh.hh"      // For terminal tcsh use
#endif

#undef G4VIS_USE

#ifdef G4VIS_USE
#include "G4VisExecutive.hh"
#endif

#include <bhep/bhep_svc.h>
#include <bhep/sreader.h>

using namespace snova;

int main( int argc_ , char** argv_ ) 
{
  bool        debug           = false;
  std::string params_filename = snova::constants::DEF_PARAMS_FILENAME;
  bool        automatic       = true;
  bool        visual_mode     = false;
  std::string G4_macro        = "";

  // parse switches:
  int iarg=1;
  while ( iarg < argc_ ) {
    
    std::string arg = argv_[iarg];
    
    if ( arg[0] == '-' ) {

      if ( arg == "-d" || arg == "--debug" ) debug=true;
      
      if ( arg == "-p" || arg == "--params" ) {
	iarg++;
	if (iarg==argc_) {
	  throw std::runtime_error("Missing simulation parameters file!");
	}
	params_filename=argv_[iarg];
      }

    }
    else {
      G4cerr << "arg=" << arg << G4endl;

      if ( arg == "vis" || arg == "visual" ) {
	visual_mode = true;
      }
      else {
	G4_macro=arg;
      }

    }
    
    iarg++;
  }

  if ( visual_mode || !G4_macro.empty() ) automatic=false;

  // create a generic store:
  bhep::gstore store;
  G4cout << G4endl << "***** Reading parameters from '" 
	 << params_filename << "' file ..." 
	 << G4endl << G4endl;

  // and a reader:
  bhep::sreader reader(store);
  reader.file(params_filename.c_str());
  reader.info_level(NORMAL);
  reader.group("GLOBAL");
  reader.read();

  G4String sim_verbosity = store.fetch_sstore("simulation_verbosity");
  G4cout << "***** Simulation verbosity set to: " << sim_verbosity << G4endl;

  G4String G4Tracking_verbosity = store.fetch_sstore("G4Tracking_verbosity");
  G4cout << "***** G4 Tracking verbosity set to: " << G4Tracking_verbosity << G4endl;

  G4String dst_fname = store.fetch_sstore("dst_fname");
  G4cout << "***** DST output file name:" << dst_fname << G4endl;

  G4String gen_source = store.fetch_sstore("gen_source");
  G4double gen_min_E  = store.fetch_dstore("gen_min_E");
  G4double gen_max_E  = store.fetch_dstore("gen_max_E");
  G4int    num_events = store.fetch_istore("num_events");
  G4String part_name  = store.fetch_sstore("part_name");

  if (gen_source == "random") {
    G4cout << "***** Generator data will follow a flat random distribution in: " << G4endl;
    G4cout << "*****           Ekin  (Mev): (" << gen_min_E << " , " 
	   << gen_max_E << ")" << G4endl;
    G4cout << "*****           Theta (deg): (0 , 90)" <<  G4endl;
    G4cout << "*****           Phi   (deg): (0 , 180)" <<  G4endl;
    G4cout << "***** Initial events: " << num_events << " " << part_name << G4endl;
  }
  else {
    G4cout << "***** Generator data source read from:" << gen_source << G4endl;
    G4cout << "***** Initial events: " << num_events << " pairs of " << part_name << G4endl;
  }

  G4String geom_filename = store.fetch_sstore("geom_file");
  G4cout << "***** Reading geometry parameters from: '" 
	 << geom_filename << "'" << G4endl;
  G4cout << "***** Reading physics parameters from:  '" 
	 << params_filename << "'" << G4endl;
  
  G4cout << "**********************************************************"
	 << G4endl << G4endl;
  
  // The ihep::event service is generated. This server must be instantiated
  // wherever you want to access all the info.
  bhep::bhep_svc bsvc;  

  // my Verbose output class:
  G4VSteppingVerbose* verbosity = new stepping_verbose;
  G4VSteppingVerbose::SetInstance(verbosity);

  // Run manager:
  G4RunManager * runManager = new G4RunManager;

  // UserInitialization classes (mandatory)

  // geometry
  geometry* geom = new geometry(geom_filename,params_filename);
  geom->set_info_level(sim_verbosity);
  runManager->SetUserInitialization(geom);

  // physics
  physics* phys = new physics(params_filename);
  phys->set_info_level(sim_verbosity);
  runManager->SetUserInitialization(phys);


#ifdef G4VIS_USE
  // Visualization, if you choose to have it!
  G4VisManager* visManager = new G4VisExecutive;
  visManager->Initialize();
#endif

  // UserAction classes
  // generator
  generator* my_gen = new generator( part_name , 
				     gen_source , 
				     gen_min_E , 
				     gen_max_E , 
				     geom->get_geom_mgr() , 
				     params_filename );
  my_gen->set_info_level(sim_verbosity);
  runManager->SetUserAction(my_gen);

  // run
  run* my_run = new run(dst_fname);
  runManager->SetUserAction(my_run);  

  //event
  snova::event* my_evt = new snova::event(part_name, dst_fname);
  my_evt->set_info_level(sim_verbosity);
  runManager->SetUserAction(my_evt);

  // track
  tracking* my_tracking = new tracking();
  my_tracking->set_info_level(sim_verbosity);
  runManager->SetUserAction(my_tracking);

  // step
  stepping* my_stepping = new stepping(part_name);
  my_stepping->set_info_level(sim_verbosity);
  runManager->SetUserAction(my_stepping);

  // stack
  stacking* my_stacking = new stacking(part_name);
  my_stacking->set_info_level(sim_verbosity);
  runManager->SetUserAction(my_stacking);


  //Initialize G4 kernel
  runManager->Initialize();


  //////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////
  //get the pointer to the User Interface manager
  //in order for the user to issue commands to the program 
  G4UImanager * UI = G4UImanager::GetUIpointer();  


  //////////////////////////////////////////////////////////////
  // Automatic mode
  if( automatic ) {

#ifdef G4VIS_USE
    visManager->SetVerboseLevel("quiet");
#endif

    G4String command = "/control/shell cp currentEvent.rndm currentRun.rndm";
    UI->ApplyCommand(command);
    command = "/random/setSavingFlag true";
    UI->ApplyCommand(command);
    command = "/random/resetEngineFrom currentRun.rndm";
    UI->ApplyCommand(command);

    command = "/tracking/verbose ";
    UI->ApplyCommand(command + G4Tracking_verbosity);

    command = "/run/beamOn ";
    UI->ApplyCommand(command+to_string(num_events));

  }
  else {      
    // Visual mode
    if ( visual_mode ) {
      
      // G4UIterminal is a (dumb) terminal.
      G4UIsession *session = 0;
      
#ifdef G4UI_USE_TCSH
      session = new G4UIterminal(new G4UItcsh);      
#else
      session = new G4UIterminal();
#endif     
      
      UI->ApplyCommand("/control/execute snova_vis.mac");    
      session->SessionStart();
      delete session;
    }
    // Batch mode
    else {
      
#ifdef G4VIS_USE
      visManager->SetVerboseLevel("quiet");
#endif
      
      G4String command = "/control/execute ";
      G4String fileName = G4_macro.c_str();
      UI->ApplyCommand(command + fileName);

    }
    
  }
  
#ifdef G4VIS_USE
  delete visManager;
#endif
  
  delete runManager;
  delete verbosity;
  
  return 0;
}

// end of prog_snova.cc
