// **********************************************************************
//
// Copyright (c) 2000
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#ifndef JTC_HANDLE_H
#define JTC_HANDLE_H

//
// Provide some handy typedefs.
//
template <class T> class JTCHandleT;

class JTCThread;
typedef JTCHandleT<JTCThread> JTCThreadHandle;

class JTCThreadGroup;
typedef JTCHandleT<JTCThreadGroup> JTCThreadGroupHandle;

class JTCRunnable;
typedef JTCHandleT<JTCRunnable> JTCRunnableHandle;

//
// JTCRefCount class. This is a base class to be used in conjunction
// with the JTCHandleT class.
//
class JTCRefCount
{
public:
    int _jtc_refCount_;
    JTCMutex refMutex_;
    
    JTCRefCount()
        : _jtc_refCount_(0)
    {
    }

    virtual ~JTCRefCount()
    {
    }
};

//
// This class is a handle to a reference counted object. This ensures
// that the object is deleted when it is no longer referenced.
//
template <class T>
class JTCHandleT
{
    T* tg_; // Referenced object

public:

    //
    // Constructors.
    //
    JTCHandleT(T* tg = 0);
    JTCHandleT(const JTCHandleT<T>& rhs);

    //
    // Destructor. Deletes object when reference count drops to
    // zero.
    //
    ~JTCHandleT();

    //
    // Assignment operator.
    //
    JTCHandleT<T>& operator=(const JTCHandleT<T>& rhs);

    //
    // Equality and inequality.
    //
    bool operator==(const JTCHandleT<T>& rhs) const
    { return tg_ == rhs.tg_; }
    bool operator!=(const JTCHandleT<T>& rhs) const
    { return tg_ != rhs.tg_; }

    //
    // Logical not operator. Are we referring to a valid Thread?
    //
    bool operator!() const { return tg_ == 0; }
    operator bool() const { return tg_ != 0; }
    
    //
    // Make access to the Thread intuitive.
    //
    T* operator->() const { return tg_; }
    T* get() const { return tg_; }
    T& operator*() { return *tg_; }

    void _jtc_deref();
};

#if defined(HAVE_NO_EXPLICIT_TEMPLATES) && !defined(HAVE_PRAGMA_DEFINE)
#   include "JTC/HandleI.h"
#endif

#endif
