// **********************************************************************
//
// Copyright (c) 2000
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#ifndef JTC_THREAD_GROUP_H
#define JTC_THREAD_GROUP_H

//
// This class represents a grouping of threads.
//
class JTCThreadGroup : public JTCMonitor, public virtual JTCRefCount
{
    //
    // Hide copy constructor and assignment operator.
    //
    JTCThreadGroup(const JTCThreadGroup&);
    void operator=(const JTCThreadGroup&);

    //
    // This class represents a collection of ThreadGroupHandles.
    //
    class JTCThreadGroupSet
    {
	//
        // Hide copy constructor and assignment operator.
	//
        JTCThreadGroupSet(const JTCThreadGroupSet&);
        JTCThreadGroupSet& operator=(const JTCThreadGroupSet&);
        
        JTCThreadGroupHandle* handles_; // The set of handles.
        int len_; // Length of the set.

    public:

        //
        // Constructor/destructor.
        //
        JTCThreadGroupSet() : handles_(0), len_(0) { }
        ~JTCThreadGroupSet() { delete [] handles_; }

        //
        // Allocate n members for the set.
        //
        void allocate(int n);

        //
        // How many members in the set.
        //
        int length() const;

        //
        // Return a reference to the ith member of the set.
        //
        JTCThreadGroupHandle& operator[](int i);
    };
    
    //
    // Copy current set of groups.
    //
    void createGroupSnapshot(JTCThreadGroupSet& set) const;

    //
    // Add and remove ThreadGroup.  Add and remove threads.
    //
    void add(JTCThreadGroup* g);
    void remove(JTCThreadGroup* g);
    void add(JTCThreadHandle t);
    void remove(JTCThreadHandle t);

    //
    // Copy current set of threads, and ThreadGroups.
    //
    int enumerate(JTCThreadHandle* list, int len, int n, bool recurse) const;
    int enumerate(JTCThreadGroupHandle* list, int len, int n,
		  bool recurse) const;

    //
    // Constructor helper.
    //
    void init(JTCThreadGroup* parent, const char* name);

    JTCThreadGroupHandle parent_; // Parent of this ThreadGroup.
    char* name_; // Name of ThreadGroup.
    int maxPriority_; // Maximum priority of this ThreadGroup.
    bool destroyed_; // Has this ThreadGroup been destroyed?

    int nthreads_; // Number of threads in this ThreadGroup.
    JTCThreadHandle* threads_; // Pointer to each thread in this ThreadGroup.
    int threadsLength_; // Capacity of the above array.

    int ngroups_; // Number of child ThreadGroups.
    JTCThreadGroup** groups_; // Pointer to each child ThreadGroup.
    int groupsLength_; // Capacity of the above array.

    bool daemon_; // Is this a daemon ThreadGroup?

    friend class JTCThread;

    JTCThreadGroup(bool main);

public:

    //
    // Constructor and destructor.
    //
    JTCThreadGroup(const char* name);
    JTCThreadGroup(JTCThreadGroup* parent, const char* name);

    virtual ~JTCThreadGroup();

    //
    // Get the name of this ThreadGroup.
    //
    const char* getName() const;

    //
    // Return parent of this ThreadGroup.
    //
    JTCThreadGroupHandle getParent() const;

    //
    // Is this a daemon Threadgroup?
    //
    bool isDaemon() const;

    //
    // Set the daemon status of a ThreadGroup.
    //
    void setDaemon(bool daemon);

    //
    // Called on an uncaught exception.
    //
    virtual void uncaughtException(JTCThreadHandle t, const JTCException& e);

    //
    // Called on an uncaught unknown exception.
    //
    virtual void uncaughtException(JTCThreadHandle t);

    //
    // Get maximum priority of this ThreadGroup.
    //
    int getMaxPriority() const;

#ifdef HAVE_JTC_STOP
    //
    // Stop, suspend and resume all threads in this ThreadGroup, and child
    // ThreadGroups.
    //
    void stop();
    void resume();
    void suspend();
#endif

    //
    // Has this ThreadGroup been destroyed?
    //
    bool isDestroyed() const;

    //
    // Destroy this ThreadGroup.
    //
    void destroy();

    //
    // Set maximum priority of threads in this ThreadGroup, and our
    // child ThreadGroups.
    //
    void setMaxPriority(int pri);

    //
    // Is this ThreadGroup a parent of the argument?
    //
    bool parentOf(JTCThreadGroupHandle g);

    //
    // Number of active threads in this group, and child ThreadGroups.
    // Number of active ThreadGroups in this group, and child ThreadGroups.
    //
    int activeCount() const;
    int activeGroupCount() const;

    //
    // Enumerate the set of threads in this ThreadGroup, and child
    // ThreadGroups, if recurse is true.
    //
    int enumerate(JTCThreadHandle* list, int len, bool recurse = true) const;

    //
    // Enumerate the set of ThreadGroups in this ThreadGroup, and child
    // ThreadGroups, if recurse is true.
    //
    int enumerate(JTCThreadGroupHandle* list, int len,
		  bool recurse = true) const;

    //
    // Display the set of threads and child ThreadGroups
    // in this ThreadGroup to cout.
    //
    void list();

    //
    // Display the set of threads and child ThreadGroups
    // in this ThreadGroup to stream os, with given indent level.
    //
    void list(JTC_STD(ostream)& os, int indent);
};

//
// Insert operator for the iostream library.
//
JTC_STD(ostream)& operator<<(JTC_STD(ostream)& os, const JTCThreadGroup& g);

#endif
