// **********************************************************************
//
// Copyright (c) 2000
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#ifndef JTC_TYPES_H
#define JTC_TYPES_H

//
// Configuration stuff
//
#include <JTC/Config.h>

//
// If we're in a pthreads environment we want to define _REENTRANT
//
#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)
#   ifndef _REENTRANT
#       define _REENTRANT  1
#   endif
#endif

// 
// Under HP/UX we want to define _CMA_NOWRAPPERS_ to stop read,
// write and other friends from being redefined to the cma_              
// versions.
//                
#if defined(__hpux)
#   include <errno.h>
#   ifndef _CMA_NOWRAPPERS_
#       define _CMA_NOWRAPPERS_
#   endif
#endif

//
// DLL support
//
#ifdef JTC_DLL
#   ifdef JTC_BUILD_DLL
#        define JTC_IMPORT /**/
#   else
#        define JTC_IMPORT __declspec(dllimport)
#   endif
#else
#   define JTC_IMPORT /**/
#endif

//
// Include common header files.
//
#ifdef HAVE_IOSTREAM
#   include <iostream>
#else
#   include <iostream.h>
#endif

#ifdef HAVE_STD_IOSTREAM
#   define JTC_STD(x) std::x
#else
#   define JTC_STD(x) x
#endif

//
// Typedef for bool type if bool is not supported.
//
#if SIZEOF_BOOL == 0
#   ifndef HAVE_BOOL_TYPEDEF
#       define HAVE_BOOL_TYPEDEF
typedef int bool;
#       if !defined(true)
const bool true = 1;
#       endif
#       if !defined(false)
const bool false = 0;
#       endif
#   endif
#endif

#if !defined(HAVE_POSIX_THREADS) && !defined(HAVE_DCE_THREADS) && \
    !defined(HAVE_WIN32_THREADS)
#   error Environment must support either POSIX threads or WIN32 threads.
#endif

#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)

#   include <pthread.h>

//
// This class represents a thread id under POSIX threads, and DCE
// threads.
//
class JTCThreadId
{
    pthread_t id_; // The thread id.
    bool nullId_; // Is this a null thread id?

public:
    //
    // Constuctor.  By default a thread id is a null thread id.
    //
    JTCThreadId()
    	: nullId_(true)
    {
    }
    
    JTCThreadId(pthread_t id)
	: id_(id), nullId_(false)
    {
    }

    //
    // Use default copy constructor and assignment operator.
    //

    //
    // Equality and inequality.
    //
    bool operator!=(const JTCThreadId& rhs) const
    {
	return !(*this == rhs);
    }
    
    bool operator==(const JTCThreadId& rhs) const
    {
        //
        // Two thread ids that are not null should use pthread_equal.
        //
	if (!nullId_ && !rhs.nullId_)
	{
	    return pthread_equal(id_, rhs.id_);
	}
        //
        // If they are both null then they are equal.
        //
	else if (nullId_ && rhs.nullId_)
	{
	    return true;
	}
        //
        // Else one is null and the other isnt.  Therefore they are
        // not equal.
        
        //
	//
	// dont remove this else.  This works around a bug with
	// the SUN C++ 4.1 compiler.
	//
	else return false;
    }

    //
    // Return the id of the current thread.
    //
    static JTCThreadId self()
    {
	return JTCThreadId(pthread_self());
    }

    //
    // Conversion operator.
    //
    operator pthread_t() const
    {
	return id_;
    }
    friend JTC_STD(ostream)& operator<<(JTC_STD(ostream)&, const JTCThreadId&);
};

//
// IOstream insertion operator. This method is different for various
// thread packages since the type pthread_t is not specified.
// Therefore it may not be printable.
//
inline JTC_STD(ostream)&
operator<<(JTC_STD(ostream)& os, const JTCThreadId& id)
{
    if (id == JTCThreadId())
    {
        os << "NIL";
    }
    else
    {
#if defined(HAVE_POSIX_THREADS)
        os << id.id_;
#endif
#if defined(HAVE_DCE_THREADS)
        os << "(" << id.id_.field1
           << "," << id.id_.field2
#    if !defined(__alpha)
           << "," << id.id_.field3
#endif
           << ")";
#endif
    }
    return os;
}

#endif

#if defined(HAVE_WIN32_THREADS)

//
// This class represents a thread id under WIN32.
//
class JTCThreadId
{
    HANDLE handle_; // Handle to the thread.

public:

    // This field should be private but VC++ 6 doesn't allow this.
    DWORD threadId_; // The thread id.

    //
    // Constructor.
    //
    JTCThreadId(DWORD threadId = 0, HANDLE handle = 0)
	: handle_(handle), threadId_(threadId)
    {
    }

    //
    // Use default copy constructor and assignment operator.
    //

    //
    // Equality and inequality.
    //
    bool operator==(const JTCThreadId& rhs) const
    {
	return threadId_ == rhs.threadId_;
    }
    
    bool operator!=(const JTCThreadId& rhs) const
    {
	return !(*this == rhs);
    }

    //
    // Conversion operator.
    //
    operator DWORD() const
    {
    	return threadId_;
    }

    //
    // Return the id of the current thread.
    //
    static JTCThreadId self()
    {
	return JTCThreadId(GetCurrentThreadId(), GetCurrentThread());
    }

//    friend ostream& operator<<(ostream&, const JTCThreadId&);
    friend class JTCThread;
};

//
// IOstream insertion operator.
//
inline JTC_STD(ostream)&
operator<<(JTC_STD(ostream)& os, const JTCThreadId& id)
{
    os << id.threadId_;
    return os;
}

#endif

//
// The type JTCThreadKey represents a thread specific storage key.
//
#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)
typedef pthread_key_t JTCThreadKey;
#endif
#if defined(HAVE_WIN32_THREADS)
typedef DWORD JTCThreadKey;
#endif

#endif
