// **********************************************************************
//
// Copyright (c) 2000
// Object Oriented Concepts, Inc.
// Billerica, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#ifndef JTC_THREAD_H
#define JTC_THREAD_H

class JTCMonitor;
class JTCThreadGroup;

#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)
typedef void (*JTCAttrHook)(pthread_attr_t*);
#endif

//
// This class encapsulates the functionality of a thread.
//
class JTCThread : public virtual JTCRefCount
{
    //
    // Hide copy constructor and assignment operator.
    //
    JTCThread(const JTCThread&);
    void operator=(const JTCThread&);

#ifdef HAVE_JTC_STOP
    //
    // Check running status of this thread.  This determines whether
    // the thread needs to be suspended or terminated.
    //
    void _checkRunningStatus() const;

    //
    // Set the monitor associated with this thread.
    //
    void setMonitor(JTCMonitor* monitor);
    JTCMonitor* getMonitor();
#endif

    //
    // Spawn a new thread.
    //
    void spawnThread();

    //
    // Helper for the constructor.
    //
    void init(const JTCThreadGroupHandle& group, JTCRunnableHandle target,
	      const char* name);

    //
    // Private constructor for booting JTC.
    //
    JTCThread(JTCThreadId tid, bool main);

    JTCThreadId thrId_; // Threads ID.
    JTCThreadGroupHandle group_; // Group of this thread.

    //
    // Event variable used for waiting until thread data is set.
    //
    JTCEvent setInitialData_;

    char* name_; // Name of the thread.
    JTCRunnableHandle target_; // Target of the thread.
    enum
    {
	NewThread,
	Runnable,
	NotRunnable,
	Dead
    } state_; // The thread state.

    JTCEvent startRequest_; // Used for starting the thread.

    //
    // State flags.
    // Future enhancement could be to convert these
    // flags to a bitfield to save space.
    //
    enum { not_terminated, terminated, throw_termination };
    int terminated_; // Has the thread been terminated.

    JTCRecursiveMutex resumeMut_; // Mutex, associated with resumeCond.

#ifdef HAVE_JTC_STOP
    bool suspended_; // Has the thread been suspended.

    JTCCond resumeCond_; // Condition variable used for resume.

    JTCMutex monMutex_; // Mutex to protect monitor
    JTCMonitor* monitor_; // Threads current monitor.
#endif

    JTCCond joinCond_; // Join condition variable.
    JTCRecursiveMutex joinMut_; // Join mutex.

    bool adopted_; // Is this thread adopted?
    
    static JTCThreadKey thrKey_; // TSS key.
#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)
    static JTCAttrHook attrHook_;
#endif

    friend class JTCInitialize;
    friend class JTCAdoptCurrentThread;

public:

    JTCThread (const char* name);
    JTCThread (JTCRunnableHandle target = JTCRunnableHandle(),
    	       const char* name = 0);
    JTCThread (JTCThreadGroupHandle& group, JTCRunnableHandle target,
	       const char* name = 0);
    JTCThread (JTCThreadGroupHandle& group, const char* name = 0);

    virtual ~JTCThread();

    //
    // Get the threads group.
    //
    JTCThreadGroupHandle getThreadGroup();

    //
    // Set the name of this thread.
    //
    void setName(const char*);

    //
    // Get the name of this thread.
    //
    const char* getName() const;

#ifdef HAVE_JTC_STOP
    //
    // Suspend the thread.
    //
    void suspend();

    //
    // Resume the thread.
    //
    void resume();

    //
    // Stop the thread.  Subclasses should override this method to
    // provide special functionality for stopping.
    //
    virtual void stop();
#endif
    
    //
    // Start the thread.
    //
    void start();

    //
    // Override this method to provide functionality to your thread.
    //
    virtual void run();

    //
    // Determine if the thread is currently running.
    //
    bool isAlive() const;

    //
    // Wait for this thread to terminate.
    //
    void join();

    //
    // Wait at most millis for this thread to terminate.
    //
    void join(long millis);

    //
    // Wait at most millis plus nanos for this thread to terminate.
    //
    void join(long millis, int nanos);
    
    //
    // Set the priority of this thread.
    //
    void setPriority(int newPri);

    //
    // Get the priority of this thread.
    //
    int getPriority() const;

    //
    // Enumerate all threads in this threads group.
    //
    static int enumerate(JTCThreadHandle* list, int len);

    enum // Under WIN32 MIN_PRIORITY is already defined in winspool.h
    {
	JTC_MIN_PRIORITY  = 0,
	JTC_NORM_PRIORITY = 1,
	JTC_MAX_PRIORITY  = 2
    };

    //
    // Get a pointer to the current thread object.
    //
    static JTCThread* currentThread();

    //
    // Sleep for millis milliseconds, and nano nanoseconds.
    //
    static void sleep(long millis, int nano = 0);

    //
    // Yield this threads current timeslice.
    //
    static void yield();

    //
    // Determine the number of threads currently running in this
    // threads ThreadGroup.
    //
    static int activeCount();

    //
    // Get the id of this thread.
    //
    JTCThreadId getId() const;

#if defined(HAVE_POSIX_THREADS) || defined(HAVE_DCE_THREADS)
    //
    // This function sets a "hook" that is used to initialize custom
    // POSIX thread attributes. This function is called before every
    // thread that is created
    //
    static void setAttrHook(JTCAttrHook);
#endif

    //
    // These two methods are part of the internal thread interface.
    // They shouldnt be directly called by clients of the JTC
    // threading package.
    //
    void _JTC_startThreadHook();
    void _JTC_exit();

#ifdef HAVE_JTC_STOP
    static void _JTC_checkRunningStatus();

    static void _JTC_setMonitor(JTCMonitor* monitor);
#endif
};

//
// Insert operator for the iostream library.
//
JTC_STD(ostream)& operator<<(JTC_STD(ostream)& os, const JTCThread& thr);

//
// Forward declaration
//
class JTCTSSManager;

//
// This class is used to initialize the JTC library.
//
class JTCInitialize
{
    static int init_; // Number of times the library has been initialized.
    
    JTCThreadHandle mainThread_; // The JTC main thread.
    JTCTSSManager* manager_;

    void initialize();
    
public:

    JTCInitialize();
    JTCInitialize(int&, char**);
    ~JTCInitialize();

    //
    // Wait for all running threads to terminate.
    //
    void waitTermination();

    //
    // Determine if JTC has been initialized.
    //
    static bool initialized();
};

//
// This class is used to incorporate a non-JTC thread into the
// JThreads/C++ family. If an application wants to use a non-JTC
// thread with JTC primitives (for instance, if an ORBacus method is
// called from a non-JTC thread) an instance of this class must be
// instantiated before using any JTC primitive.
//
// What this class actually does is setup a fake
// JTCThread::currentThread() object, and create another root
// ThreadGroup.
//
class JTCAdoptCurrentThread
{
    JTCThreadHandle adoptedThread_; // The JTC adopted thread

public:

    JTCAdoptCurrentThread();
    ~JTCAdoptCurrentThread();
};

#endif
