#ifndef HISPROF_SEEN
#define HISPROF_SEEN

#include <stdio.h>
#include "peida.h"
#include "tvector.h"
#include "ppersist.h"
#include "histos.h"

namespace SOPHYA {

//! 1 dimension profile histograms
class HProf : public Histo {
  friend class ObjFileIO<HProf>;
public:

  // CREATOR / DESTRUCTOR
  HProf();
  HProf(float xMin, float xMax, int nBin=100, float yMin=1., float yMax=-1.);
  HProf(const HProf& H);
  virtual ~HProf();

  // UPDATING or SETTING
  //! Calcul la partie histogramme du profile si elle n'est pas a jour.
  virtual inline void UpdateHisto(bool force=false) const
                        {if(!Ok || force) updatehisto();}
  void SetErrOpt(bool spread = true);
  void Zero();
  void Add(float x, float y, float w = 1.);
  void AddBin(int numBin, float y, float w = 1.);
  
  // Acces a l information
  //! Retourne l'histogramme de profil.
  inline Histo GetHisto()
               {UpdateHisto(); return (Histo) *this;}
  //! Retourne le contenu de la moyenne dans le vecteur v
  inline void GetValue(TVector<r_8>& v)
              {UpdateHisto(); Histo::GetValue(v);}
  //! Retourne le contenu au carre de la dispersion/erreur dans le vecteur v
  inline void GetError2(TVector<r_8>& v)
              {UpdateHisto(); Histo::GetError2(v);}
  //! Retourne le contenu au carre de la dispersion/erreur dans le vecteur v
  inline void GetError(TVector<r_8>& v)
              {UpdateHisto(); Histo::GetError(v);}
  //! Retourne le contenu du bin i
  inline float operator()(int i) const
               {UpdateHisto(); return data[i];}
  //! Retourne le carre de la dispersion/erreur du bin i
  inline double Error2(int i) const
                {UpdateHisto(); return (float) err2[i];}
  //! Retourne la dispersion/erreur du bin i
  inline float Error(int i) const
               {UpdateHisto();
                return (err2[i]>0.) ? (float) sqrt(err2[i]) : 0.f;}

  // Operators
  HProf& operator = (const HProf& h);
  HProf& operator += (const HProf& a);

  // Info, statistique et calculs sur les histogrammes
  virtual void HRebin(int nbinew);

  // Fit
  //! Fit du profile par ``gfit''.
  inline int Fit(GeneralFit& gfit)
         {UpdateHisto(); return Histo::Fit(gfit,0);}
  //! Retourne l'Histogramme des residus par ``gfit''.
  inline Histo FitResidus(GeneralFit& gfit)
         {UpdateHisto(); return Histo::FitResidus(gfit);}
  //! Retourne l'Histogramme de la fonction fittee par ``gfit''.
  inline Histo FitFunction(GeneralFit& gfit)
         {UpdateHisto(); return Histo::FitFunction(gfit);}

  // Print
  //! Print, voir detail dans Histo::Print
  inline void Print(int dyn=100,float hmin=1.,float hmax=-1.
                   ,int pflag=0,int il=1,int ih=-1)
         {UpdateHisto(); Histo::Print(dyn,hmin,hmax,pflag,il,ih);}
  //! PrintF, voir detail dans Histo::PrintF
  inline void PrintF(FILE * fp,int dyn=100,float hmin=1.,float hmax=-1.
                    ,int pflag=0,int il=1,int ih=-1)
         {UpdateHisto(); Histo::PrintF(fp,dyn,hmin,hmax,pflag,il,ih);}

protected:
  void Delete();
  void updatehisto() const;

  double*        SumY;  //!< somme
  double*        SumY2; //!< somme des carres
  double*        SumW;  //!< somme des poids
  mutable bool   Ok;    //!< true if update fait
  float          YMin;  //!< limite minimum Y pour somme
  float          YMax;  //!< limite maximum Y pour somme
  uint_2         Opt;   //!< options pour les erreurs
};


/*! \ingroup HiStats \fn operator<<(POuttPersist&, HProf)
  \brief Persistance management */
inline POutPersist& operator << (POutPersist& os, HProf & obj)
{ ObjFileIO<HProf> fio(&obj);  fio.Write(os);  return(os); }
/*! \ingroup HiStats \fn operator>>(PInPersist&, HProf)
  \brief Persistance management */
inline PInPersist& operator >> (PInPersist& is, HProf & obj)
{ ObjFileIO<HProf> fio(&obj);  fio.Read(is);  return(is); }
// Classe pour la gestion de persistance
// ObjFileIO<HProf>

/*! \ingroup HiStats \fn operator+(const HProf&, const HProf&)
  \brief Operateur H = H1 + H2
  \warning Meme commentaire que pour l'operateur +=
*/
inline HProf operator + (const HProf& a, const HProf& b)
{
if (b.NBins()!=a.NBins()) THROW(sizeMismatchErr);
HProf c(a);
return (c += b);
}

} // Fin du namespace

#endif // HISPROF_SEEN
