//     Utility classes for template numerical arrays
//                     R. Ansari, C.Magneville   03/2000

#include "machdefs.h"
#include "utilarr.h"
#include "srandgen.h"

// Classe utilitaires

Sequence::~Sequence()
{
}

//////////////////////////////////////////////////////////
/*!
  \class SOPHYA::RandomSequence
  \ingroup TArray
  Class to generate a random sequence of values
*/

//! Constructor
/*!
  \param typ : generator type
  \param m : mean parameter of the generator (if needed)
  \param s : sigma parameter of the generator (if needed)
 */
RandomSequence::RandomSequence(int typ, double m, double s)
{
  typ_ = (typ == Flat) ? Flat : Gaussian;
  mean_ = m;
  sig_ = s;
}
RandomSequence::~RandomSequence()
{
}

//! Return random sequence values.
/*!
  \return If typ = Flat : return [-1,+1]*sig + mean
  \return If typ = Gaussian : return gaussian distributed
                          with \b mean mean and sigma \b sig
 */
double RandomSequence::Rand()
{
  if (typ_ == Flat) 
    return(drandpm1()*sig_ + mean_);
  else return(GauRnd(mean_, sig_));
}

MuTyV & RandomSequence::Value(uint_8 k) const 
{
  if (typ_ == Flat) retv_ = drandpm1()*sig_ + mean_;
  else retv_ = GauRnd(mean_, sig_);
  return retv_;
}


//////////////////////////////////////////////////////////
/*!
  \class SOPHYA::RegularSequence
  \ingroup TArray
  Class to generate a sequence of values
*/

//! Constructor
/*!
  \param start : start value
  \param step : step value
  \param f : pointer to the sequence function

  See \ref RegularSequenceOperat "operator()"
 */
RegularSequence::RegularSequence(double start, double step, Arr_DoubleFunctionOfX f)
{
  start_ = start;
  step_ = step;
  myf_ = f;
}

RegularSequence::~RegularSequence()
{
}

//! Get the \b k th value
/*!
  \param k : index of the value
  \anchor RegularSequenceOperat

  If the constructor was done with RandomSequence, return a RandomSequence
  and \b k doesn't matter.

  If the constructor has a NULL Arr_DoubleFunctionOfX, return start+k*step

  If the constructor has a not NULL Arr_DoubleFunctionOfX, return f(start+k*step)
  \return the \b k th value
 */

MuTyV & RegularSequence::Value (uint_8 k) const 
{
  double x = start_+(double)k*step_;
  if (myf_)  x = myf_(x);
  retv_ = x;
  return(retv_);
}

EnumeratedSequence::~EnumeratedSequence()
{
}

MuTyV & EnumeratedSequence::Value (uint_8 k) const 
{
  if (k >= vecv_.size())  retv_ = 0;
  else retv_ = vecv_[k];
  return(retv_);
}

EnumeratedSequence & EnumeratedSequence::operator , (MuTyV const & v)
{
  vecv_.push_back(v);
  return(*this);
}

//////////////////////////////////////////////////////////
/*!
  \class SOPHYA::Range
  \ingroup TArray
  Class to define a range of indexes
*/

//! Constructor
/*!
  Define a range of indexes
  \param start : start index
  \param end : start end
  \param size : size
  \param step : step

  \warning If \b end \> \b start, \b size is computed automatically
  \warning If not \b size is fixed and \b end recomputed
 */
Range::Range(uint_4 start, uint_4 end, uint_4 size, uint_4 step)
{
  start_ = start;
  step_ = (step > 0) ? step : 1;
  if (end > start) {  // Taille calcule automatiquement 
    end_ = end;
    if (step_ > ((end_-start_)+1))  size_ = 1;
    else size_ = ((end-start)+1)/step_;
  }
  else {     // Taille fixee
    size_ = size;
    end_ = start_+size_*step_;
  }
}

/*
Range & Range::operator = (uint_4 start)
{
  start_ = start;
  size_ = 1;
  step_ = 1;
  return (*this);
}
*/


//////////////////////////////////////////////////////////
/*!
  \class SOPHYA::IdentityMatrix
  \ingroup TArray
  Class to define an identity matrix
*/

//! Constructor of a (n,n) diagonal matrix with value diag on the diagonal
IdentityMatrix::IdentityMatrix(double diag, uint_4 n)
{
  size_ = n;
  diag_ = diag;
}
