// This may look like C code, but it is really -*- C++ -*-
//     Utility classes for template numerical arrays 
//                     R. Ansari, C.Magneville   03/2000

#ifndef UtilArray_SEEN
#define UtilArray_SEEN

#include "machdefs.h"
#include "mutyv.h"

#include <stdlib.h>
#include <vector>

namespace SOPHYA {   

/* Quelques utilitaires pour les tableaux (Array) */             

/*! \ingroup TArray
  \typedef Arr_DoubleFunctionOfX
  \brief define a function of float which returns a double
*/
typedef double (* Arr_DoubleFunctionOfX) (double x);
/*! \ingroup TArray
  \typedef Arr_FloatFunctionOfX
  \brief define a function of float which returns a double
*/
typedef float  (* Arr_FloatFunctionOfX)  (float x);

//////////////////////////////////////////////////////////
//! Class to generate a random sequence of values
class Sequence {
public:
  virtual ~Sequence();
  virtual MuTyV & Value(uint_8 k) const = 0;
  inline MuTyV & operator () (uint_8 k) const { return(Value(k)) ; }
};

class RandomSequence : public Sequence {
public:
  //! to define the generator type
  enum {
    Gaussian = 0, //!< gaussian generator
    Flat = 1      //!< Flat generator
    };

  explicit RandomSequence(int typ = RandomSequence::Gaussian, double m=0., double s=1.);
  virtual  ~RandomSequence();
  virtual MuTyV & Value(uint_8 k) const ;
  double   Rand();

protected:
  int typ_;            //!< random generation type
  double mean_, sig_;  //!< generation parameters mean and sigma (if needed)
  mutable MuTyV retv_;
};


//////////////////////////////////////////////////////////
//! Class to generate a sequence of values
class RegularSequence : public Sequence {
public:
  explicit RegularSequence (double start=0., double step=1., Arr_DoubleFunctionOfX f=NULL);
  virtual  ~RegularSequence();

  //! return start value of the sequence
  inline double & Start() { return start_; }
  //! return step value of the sequence
  inline double & Step() { return step_; }

  virtual MuTyV & Value(uint_8 k) const ;

protected:
  double start_;              //!< start value of the sequence
  double step_;               //!< step value of the sequence
  Arr_DoubleFunctionOfX myf_; //!< pointer to the sequence function
  mutable MuTyV retv_;
};

class EnumeratedSequence : public Sequence  {
public:
  virtual ~EnumeratedSequence();
  virtual MuTyV & Value(uint_8 k) const ;
  EnumeratedSequence & operator , (MuTyV const & v);
private:
  vector<MuTyV> vecv_;
  mutable MuTyV retv_;
};

inline EnumeratedSequence operator , (MuTyV const & a, MuTyV const & b)
{ EnumeratedSequence seq;   return ((seq,a),b) ; }

//////////////////////////////////////////////////////////
//! Class to define a range of indexes
class Range {
public:
  explicit Range(uint_4 start=0, uint_4 end=0, uint_4 size=1, uint_4 step=1);
  //! Return the start index
  inline uint_4 & Start()  {  return start_; }
  //! Return the last index
  inline uint_4 & End()    {  return end_; }
  //! Return the size
  inline uint_4 & Size()   {  return size_; }
  //! Return the step
  inline uint_4 & Step()   {  return step_; }
protected:
  uint_4 start_; //!< start index
  uint_4 end_;   //!< end index
  uint_4 size_;  //!< size
  uint_4 step_;  //!< step
};

//////////////////////////////////////////////////////////
//! Class to define an identity matrix
class IdentityMatrix {
public:
  explicit IdentityMatrix(double diag=1., uint_4 n=0);
  //! return the size of the identity matrix
  inline uint_4 Size() { return size_; }
  //! return the value of the diagonal elements
  inline double Diag() { return diag_; }
protected:
  uint_4 size_; //!< size of the matrix
  double diag_; //!< value of the diagonal elements
};

} // Fin du namespace

#endif
