// Module PI : Peida Interactive     PIColorMap
// Gestion de table de couleurs        96
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include "picmap.h"

//++
// Class	PIColorMap
// Lib		PI
// include	picmap.h
//
//   	Classe fournissant les services de gestion des tables de couleur.
//	Une hirachie un peu complexe a t mise en place pour 
//	l'implmentation des tables de couleurs pour chaque systme
//	(Mac, XWindow, ...) :
//|	PIColorMapGen  -----  PIColorMapX     (=PIColorMapNative)
//|	                 |--  PIColorMapMac   (=PIColorMapNative)
//	et la gestion de table de couleurs standard partage.
//|	PIColorMapNative (=X,Mac,...) ----- PIColorMap.
//	La seule classe  utiliser est *PIColorMap* qui fournit 
//	l'ensemble des services de gestion de table de couleurs.
//--

#define  MXMAPIDS   15
static   int    MapNCols[MXMAPIDS] = {256, 256, 256, 256, 256, 256, 256, 256, 256, 256,
                                      256, 256, 256, 256, 256 };
static   const char*  MapNoms[MXMAPIDS]  = {"Grey32","InvGrey32","ColRJ32","ColBR32",
                               "ColRV32", "Grey128", "InvGrey128", "ColRJ128",
                               "ColBR128", "Col16",
                               "MIDAS_Pastel", "MIDAS_Heat", "MIDAS_Rainbow3", 
                               "MIDAS_BlueRed", "MIDAS_BlueWhite" };
static   CMapId MapIds[MXMAPIDS] = { CMAP_GREY32, CMAP_GREYINV32, CMAP_COLRJ32,
                               CMAP_COLBR32, CMAP_COLRV32, CMAP_GREY128, 
                               CMAP_GREYINV128, CMAP_COLRJ128, CMAP_COLBR128,  
                               CMAP_COL16,
                               CMAP_MIDAS_Pastel, CMAP_MIDAS_Heat, CMAP_MIDAS_Rainbow3, 
                               CMAP_MIDAS_BlueRed, CMAP_MIDAS_BlueWhite } ;

static   int mColTNums[8192];   // Max 8192 tables differentes pour le moment  
static   int CTNum = 0;        //  Numero des tables ds mColTNum - s'incremente a chaque creation 

static   int NMaxTableAuto = 0;
static   PIColorMap* mMaps[15] = {NULL, NULL, NULL, NULL, NULL, 
                                  NULL, NULL, NULL, NULL, NULL, 
                                  NULL, NULL, NULL, NULL, NULL  };

int  PIColorMap::NumberStandardColorMaps()
{
  return ( MXMAPIDS );
}

CMapId PIColorMap::GetStandardColorMapId(int num)
{
  if ( (num < 0) || (num >= MXMAPIDS) )  num = 0;
  return (MapIds[num]) ;
}

string PIColorMap::GetStandardColorMapName(int num)
{
  if ( (num < 0) || (num >= MXMAPIDS) )  num = 0;
  return (MapNoms[num]) ;
}


//++
// Titre	Constructeurs, mthodes
//--
//++
// PIColorMap(CMapId id)
//	Constructeur d'une table de couleur standard, identifie par "CMapId id"
//|	CMAP_GREY32     :  32 niveaux de gris
//|	CMAP_GREYINV32  :  32 niveaux gris du blanc au noir
//|	CMAP_COLRJ32    :  32 couleurs, du Rouge au Jaune/blanc
//|	CMAP_COLBR32    :  32 couleurs du bleu au rouge (arcenciel)
//|	CMAP_COLRV32    :  32 couleurs du rouge au violet
//|	CMAP_GREY128    :  128 niveaux de gris
//|	CMAP_GREYINV128 :  128 niveaux gris invers
//|	CMAP_COLRJ128   :  128 couleurs, du Rouge au Jaune/blanc
//|	CMAP_COLBR128   :  128 couleurs du bleu au rouge (arcenciel)
//|	CMAP_COL16      :  16 Couleurs arcenciel
//|	CMAP_OTHER      :  Table non standard
//	Toutes les tables de couleurs standard possdent 256 cellules de couleurs,
//	contenant 16 ou 32 ou 128 couleurs distinctes.
//
// PIColorMap(string const& nom, int nCol)
//	Constructeur d'une table de "nCol" couleurs identifi par la chane "nom".
// PIColorMap(PIColorMap&)
//	Constructeur par copie.
// int Type()
//	Retourne le type de la table, correspondant  "CMapId id"
// int NCol()
//	Retourne le nombre de cellules de couleurs dans la table
// string const& Nom()
//	Renvoie le nom de la table.
//--

PIColorMap::PIColorMap(CMapId id)
: PIColorMapNative()
{
   int k, kdx;
   string mnom;

   if (NMaxTableAuto == 0) { //  Il faut allouer les tables de depart
     if ( TotNbColors() > 4096 ) NMaxTableAuto = MXMAPIDS-1;
     else NMaxTableAuto = 3; 
     //     for(k=0; k<NMaxTableAuto; k++)  { 
     for(k=0; k<3; k++)  {   // On alloue les 3 tables de depart
        mnom = MapNoms[k]; 
	cout << " PIColorMap::PIColorMap(CMapId id) Allocating ColorMap " 
	     << mnom << " ... " << endl;
        mMaps[k] = new PIColorMap(MapIds[k], MapNCols[k], mnom);
     }
   }

//  Protection pour valeur de Id  
   for(kdx=0; kdx<MXMAPIDS; kdx++)  if (id == MapIds[kdx])  break;
   if (kdx == MXMAPIDS)  { id = MapIds[0];  kdx = 0; }
   for(k=0; k<NMaxTableAuto; k++) {
     if ((MapIds[k] == id) && (mMaps[k] == NULL))  { //  Table pas encore alloue 
       mnom = MapNoms[k]; 
       cout << " PIColorMap::PIColorMap(CMapId id) Allocating ColorMap " 
	     << mnom << " ... " << endl;
       mMaps[k] = new PIColorMap(MapIds[k], MapNCols[k], mnom);
     }
     if (mMaps[k] && (id == mMaps[k]->Type()) ) {
       CopyFrom(mMaps[k]);  
       mCTId = mMaps[k]->mCTId;  mColTNums[mCTId]++;
       break;
     }
   }
   if (Type() != id)  {  // Table pas encore trouve dans la liste
     k = NMaxTableAuto;
     if (!mMaps[k] || mMaps[k]->Type() != id) {
       if ( mMaps[k] ) {  
         mColTNums[mMaps[k]->mCTId] = 1;  // Je force a liberer les couleurs 
         delete mMaps[k] ;
       }
       mnom = MapNoms[kdx];
       mMaps[k] = new PIColorMap(MapIds[kdx],  MapNCols[kdx], mnom); 
     }
    CopyFrom(mMaps[k]);  
    mCTId = mMaps[k]->mCTId;  mColTNums[mCTId]++;
   }
}


PIColorMap::PIColorMap(string const& nom, int nCol)
: PIColorMapNative(CMAP_OTHER, nCol, nom)
{
  MakeColMap();
  mCTId = CTNum;
  mColTNums[mCTId] = 1;
  CTNum++;
}

PIColorMap::PIColorMap(CMapId id, int nCol, string const& nom)
: PIColorMapNative(id, nCol, nom)
{
  MakeColMap();
  mCTId = CTNum;
  mColTNums[mCTId] = 1;
  CTNum++;
//  printf("PIColorMap/Debug-Creation:  %d %d - %s ", Type(), NCol(), Nom().c_str());
//  printf("  ..CRE.. %d %d \n", CTNum-1, mColTNums[CTNum-1]);
}

PIColorMap::PIColorMap(PIColorMap& cm)
{
  CopyFrom(&cm);
  mCTId = cm.mCTId;
  mColTNums[mCTId]++;
}

PIColorMap::~PIColorMap()
{
//  printf("PIColorMap/Debug-Delete:  %d %d - %s ", Type(), NCol(), Nom().c_str());
//  printf("...DEL.CTNum,mColTNums= %d %d (%d %d) ", 
//        CTNum, mColTNums[CTNum], mCTId,mColTNums[mCTId] );
  mColTNums[mCTId]--; 
  if ( mColTNums[mCTId] == 0 )  { FreeColors();  mColTNums[mCTId] = 0; }
//  printf("\n");
}


//++
// Titre	Gestion des couleurs
//	Une structure *PIColor* permet la manipulation des couleurs  
//	travers 3 champs de type "unsigned short: 0 ... 65535" dfinissant
//	les intensit des couleurs de base: Rouge, Bleue, Vert.
//|	struct PIColor {
//|	  unsigned short red;
//|	  unsigned short green;
//|	  unsigned short blue; }
//
//--
//++
// PIColor GetColor(int n)	
//	Renvoie la couleur de la cellule "n"
// bool AllocColor(PIColor const& col, int index)
//	Alloue la couleur dfinie par "col" dans la cellule numro "index".
//	Renvoie "true" si OK, "false" sinon.
// void  FreeColors()
//	Libre les couleurs alloues par la table
// long  TotNbColors()
//	Renvoie le nombre total de couleurs disponibles 
//	(Ex 256 pour un systme  8 plans image, 65536 pour 16 plans, etc ...)
// int  NbAllocColors()
//	Renvoie le nombre total de couleurs dj utilise par l'application
//--

void PIColorMap::CleanColorMaps()
{
  int k;
  for(k=0; k<MXMAPIDS; k++)  { 
    if (!mMaps[k])  continue;
    mMaps[k]->FreeColors();
    delete mMaps[k];
  }
}

/*  Definition d' une table de couleurs  R G B   */

static float R_RGB1[16] =
{0.0,0.00,0.0,0.0,0.0,0.00,0.0,0.6,0.8,1.0,1.00,1.0,0.75,1.0,0.85,1.0};
static float G_RGB1[16] =
{0.0,0.00,0.0,0.8,1.0,1.00,1.0,0.8,0.8,1.0,0.75,0.6,0.00,0.0,0.0,0.0};
static float B_RGB1[16] =
{0.0,0.75,1.0,1.0,1.0,0.75,0.0,0.0,0.0,0.0,0.00,0.5,0.75,1.0,0.0,0.0};



void PIColorMap::MakeColMap()
{
  PIColor mycol;
  int i,j,k;

  switch (mType) {

    case CMAP_GREY32 :
      for(i=0; i<32; i++) {
        mycol.red = mycol.green = mycol.blue = 3535+(i*62000/32);
        for(k=0; k<8; k++)     AllocColor(mycol, i*8+k);
      }
      break;
    

    case CMAP_GREYINV32 :
      for(i=0; i<32; i++) {
        mycol.red = mycol.green = mycol.blue = 3535+(i*62000/32);
        for(k=0; k<8; k++)     AllocColor(mycol, mNCol-(i*8+k)-1);
      }
      break;

    case CMAP_COL16 :
      for(i=0; i<16; i++) {
        mycol.red = (int)(R_RGB1[i]*65535.);
        mycol.green = (int)(G_RGB1[i]*65535.);
        mycol.blue =(int)(B_RGB1[i]*65535.) ;
        for(k=0; k<16; k++)     AllocColor(mycol, i*16+k);
      }
      break;

    case CMAP_COLRJ32 :
      for( i=0; i<32; i++) {
        if (i < 12)  { 
          mycol.green = mycol.blue = 0;  
          mycol.red = 21535+(i*4000);  }
        else if (i < 24) { 
          mycol.blue = 0;   mycol.red = 65535;
          mycol.green = 21535+(i-12)*4000;  }
        else { 
          mycol.green = mycol.red = 65535;
          mycol.blue = 17535+(i-23)*6000; }
        for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
        }
      break;

  case CMAP_COLBR32 :
    for( i=0; i<32; i++) {
      if (i < 5) { 
        mycol.green = mycol.red = 0;    // Du bleu sombre au bleu clair
        mycol.blue = 5535+i*15000; }
      else if (i < 10) { 
        mycol.blue = 65535;             // Plein bleu - On rajoute du vert -> bleu clair
        mycol.green = 5535+(i-5)*9000;
        mycol.red = 0;  }
      else if (i < 15) { 
        mycol.blue = 60000-(i-10)*15000;   // On passe au vert
        mycol.green = 41535+(i-10)*6000;
        mycol.red = 0;  }
      else if (i < 21) { 
        mycol.blue = 0;
        mycol.green = 65535;               // Plein vert, on rajoute du rouge -> jaune
        mycol.red = 5535+(i-15)*12000;  }
      else if (i < 27) {  
        mycol.blue = 0;                     //  On diminue vert -> Orange
        mycol.green = 60000-(i-21)*12000;
        mycol.red = 65535-(i-21)*2000;  }
      else {
        mycol.blue = 0;                     // Plein rouge
        mycol.green = 0;
        mycol.red = 57535+(i-27)*2000; }

      for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
      }
    break;

  case CMAP_COLRV32 :
    for( i=0; i<32; i++) {
      if (i < 8) {
        mycol.green = 15000+(i*5000); mycol.blue = 0;  
        mycol.red = 15000+(i*5000); }
      else if (i < 21) {
        mycol.green = 50000-(i-7)*5000;   
        if (mycol.green < 0)  mycol.green = 0;
        mycol.blue = 0;
        mycol.red = 53535+(i-8)*1000;  }
      else { 
        mycol.green = 0;  mycol.red = 65535;
        mycol.blue = 15535+(i-21)*5000; }

      for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
      }
    break;

    case CMAP_GREY128 :
      for(i=0; i<128; i++) {
        mycol.red = mycol.green = mycol.blue = 2035+(i*500);
        for(k=0; k<2; k++)     AllocColor(mycol, i*2+k);
      }
      break;
    

    case CMAP_GREYINV128 :
      for(i=0; i<128; i++) {
        mycol.red = mycol.green = mycol.blue = 2035+(i*500);
        for(k=0; k<2; k++)     AllocColor(mycol, mNCol-(i*2+k)-1);
      }
      break;

    case CMAP_COLRJ128 :
      for( i=0; i<128; i++) {
        if (i < 48)  { 
          mycol.green = mycol.blue = 0;  
          mycol.red = 5375+(i*1280);  }
        else if (i < 96) { 
          mycol.blue = 0;   mycol.red = 65535;
          mycol.green = 5375+((i-48)*1280);  }
        else { 
          mycol.green = mycol.red = 65535;
          mycol.blue = 3535+(i-96)*2000; }
        for(k=0; k<2; k++)   AllocColor(mycol, i*2+k); 
        }
      break;

    case CMAP_COLBR128 :
      for( i=0; i<128; i++) {
        if (i < 20) { 
          mycol.green = mycol.red = 0;  // Du bleu sombre au bleu clair
          mycol.blue = 2835+i*3300; }
        else if (i < 36) { 
          mycol.blue = 65535;           // Plein bleu - On rajoute du vert -> bleu clair
          mycol.green = 3000+(i-20)*2000;
          mycol.red = 0;  }
        else if (i < 56) { 
          mycol.blue = 64600-(i-36)*3400;    // On passe au vert 
          mycol.green = 32300+(i-36)*1700;
          mycol.red = 0;  }
        else if (i < 81) {         
          mycol.blue = 0;              // Plein vert, on rajoute du rouge -> jaune
          mycol.green = 65535;
          mycol.red = 3135+(i-56)*2600;  }
        else if (i < 96) { 
          mycol.blue = 0;
          mycol.green = 63535-(i-80)*2000;
          mycol.red = 65535;  }
        else if (i < 112) { 
          mycol.blue = 0;
          mycol.green = 30000-(i-96)*2000;
          mycol.red = 65535-(i-96)*1000;  }
        else {
          mycol.blue = 0;
          mycol.green = 0 ;
          mycol.red = 50535+(i-112)*1000; }

        for(k=0; k<2; k++)   AllocColor(mycol, i*2+k); 
      }
      break;

  case CMAP_MIDAS_Pastel :   // Les valeurs correspondent a  je ne sais quel table MIDAS
  case CMAP_MIDAS_Heat :      // A faire  pour CMAP_MIDAS_Pastel CMAP_MIDAS_Heat 
  case CMAP_MIDAS_Rainbow3 :  //    A faire  pour Heat Rainbow3 BlueRed BlueWhite
  case CMAP_MIDAS_BlueRed :   //    BlueRed -> idl11
  case CMAP_MIDAS_BlueWhite : //    BlueWhite -> blulut.lutlis
   {
    static float rgb_r[256] = {
0, 0, 0.01961, 0.0549, 0.08627, 0.1098, 0.13725, 0.15686, 0.18039, 0.2, 
0.21569, 0.23529, 0.25098, 0.26275, 0.28235, 0.29412, 0.30588, 0.31765, 
0.33333, 0.34118, 0.35294, 0.36078, 0.37255, 0.38431, 0.39216, 0.4, 
0.41176, 0.41961, 0.43137, 0.43529, 0.44314, 0.45098, 0.45882, 0.46667, 
0.47059, 0.48235, 0.48627, 0.49412, 0.50196, 0.50588, 0.5098, 0.51765, 
0.52157, 0.53333, 0.53725, 0.54118, 0.54902, 0.55294, 0.55686, 0.56078, 
0.56471, 0.57255, 0.58039, 0.58431, 0.58824, 0.59216, 0.59608, 0.6, 0.60392,
0.60784, 0.61176, 0.61569, 0.61961, 0.62745, 0.63137, 0.63529, 0.63922, 
0.64314, 0.64706, 0.65098, 0.6549, 0.6549, 0.65882, 0.66275, 0.66667 ,
0.67059, 0.67843, 0.67843, 0.68235, 0.68627, 0.6902, 0.6902, 0.69412, 
0.69804, 0.70196, 0.70196, 0.70588, 0.7098, 0.71373, 0.71373, 0.71765, 
0.72157, 0.72157, 0.72941, 0.73333, 0.73333, 0.73725, 0.74118, 0.74118, 
0.7451, 0.7451, 0.74902, 0.75294, 0.75294, 0.75686, 0.76078, 0.76078, 
0.76471, 0.76471, 0.76863, 0.76863, 0.77255, 0.78039, 0.78039, 0.78431, 
0.78431, 0.78824, 0.78824, 0.79216, 0.79216, 0.79608, 0.79608, 0.8, 0.8, 
0.80392, 0.80392, 0.80784, 0.80784, 0.81176, 0.81176, 0.81569, 0.81569, 
0.81961, 0.81961, 0.82745, 0.82745, 0.83137, 0.83137, 0.83529, 0.83529, 
0.83922, 0.83922, 0.83922, 0.84314, 0.84314, 0.84706, 0.84706, 0.85098, 
0.85098, 0.8549, 0.8549, 0.8549, 0.85882, 0.85882, 0.86275, 0.86275, 
0.86275, 0.86667, 0.86667, 0.87059, 0.87059, 0.87059, 0.87843, 0.87843, 
0.88235, 0.88235, 0.88235, 0.88627, 0.88627, 0.8902, 0.8902, 0.8902, 0.89412, 
0.89412, 0.89412, 0.89804, 0.89804, 0.89804, 0.90196, 0.90196, 0.90588, 
0.90588, 0.90588, 0.9098, 0.9098, 0.9098, 0.91373, 0.91373, 0.91373, 
0.91765, 0.91765, 0.91765, 0.92157, 0.92157, 0.92157, 0.92941, 0.92941, 
0.92941, 0.93333, 0.93333, 0.93333, 0.93725, 0.93725, 0.93725, 0.94118, 
0.94118, 0.94118, 0.94118, 0.9451, 0.9451, 0.9451, 0.94902, 0.94902, 
0.94902, 0.95294, 0.95294, 0.95294, 0.95686, 0.95686, 0.95686, 0.95686, 
0.96078, 0.96078, 0.96078, 0.96471, 0.96471, 0.96471, 0.96471, 0.96863, 
0.96863, 0.96863, 0.97255, 0.97255, 0.97255, 0.97255, 0.98039, 0.98039, 
0.98039, 0.98039, 0.98431, 0.98431, 0.98431, 0.98824, 0.98824, 0.98824, 
0.98824, 0.99216, 0.99216, 0.99216, 0.99216, 0.99608, 0.99608, 0.99608, 
0.99608, 1, 1 };
    static float rgb_g[256] = {
0, 0, 0, 0, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00784, 
0.00784, 0.00784, 0.00784, 0.01176, 0.01176, 0.01176, 0.01176, 0.01176, 
0.01569, 0.01569, 0.01569, 0.01569, 0.01961, 0.01961, 0.01961, 0.01961, 
0.02353, 0.02353, 0.02353, 0.02745, 0.02745, 0.02745, 0.02745, 0.03137, 
0.03137, 0.03137, 0.03529, 0.03529, 0.03529, 0.03529, 0.04314, 0.04314, 
0.04314, 0.04706, 0.04706, 0.04706, 0.05098, 0.05098, 0.05098, 0.0549, 
0.0549, 0.0549, 0.05882, 0.05882, 0.05882, 0.06275, 0.06275, 0.06275, 0.06667, 
0.06667, 0.06667, 0.07059, 0.07059, 0.07451, 0.07451, 0.07451, 0.07843, 
0.07843, 0.08235, 0.08235, 0.08235, 0.08627, 0.08627, 0.0902, 0.0902, 
0.0902, 0.09412, 0.09412, 0.09804, 0.09804, 0.10196, 0.10196, 0.10588, 
0.10588, 0.1098, 0.1098, 0.1098, 0.11765, 0.11765, 0.12157, 0.12157, 
0.12549, 0.12549, 0.12941, 0.12941, 0.13333, 0.13725, 0.13725, 0.14118, 
0.14118, 0.1451, 0.1451, 0.14902, 0.14902, 0.15294, 0.15686, 0.15686, 
0.16078, 0.16078, 0.16471, 0.16863, 0.16863, 0.17255, 0.17255, 0.17647, 
0.18039, 0.18039, 0.18431, 0.18824, 0.18824, 0.19608, 0.2, 0.2, 0.20392, 
0.20784, 0.20784, 0.21176, 0.21569, 0.21961, 0.21961, 0.22353, 0.22745, 
0.23137, 0.23137, 0.23529, 0.23922, 0.24314, 0.24314, 0.24706, 0.25098, 
0.2549, 0.25882, 0.26275, 0.26275, 0.27059, 0.27451, 0.27843, 0.28235, 
0.28627, 0.2902, 0.29412, 0.29804, 0.29804, 0.30196, 0.30588, 0.3098, 
0.31373, 0.31765, 0.32157, 0.32549, 0.33333, 0.33725, 0.34118, 0.3451, 
0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.37255, 0.37647, 0.38039, 
0.38431, 0.38824, 0.39216, 0.4, 0.40392, 0.40784, 0.41176, 0.41961, 0.42353, 
0.42745, 0.43529, 0.43922, 0.44314, 0.45098, 0.4549, 0.45882, 0.46667, 
0.47059, 0.47843, 0.48235, 0.4902, 0.49412, 0.50196, 0.50588, 0.51373, 
0.51765, 0.52549, 0.52941, 0.53725, 0.54118, 0.54902, 0.55686, 0.56078, 
0.56863, 0.57647, 0.58039, 0.58824, 0.59608, 0.6, 0.60784, 0.61569, 
0.62353, 0.63137, 0.63529, 0.64314, 0.65098, 0.65882, 0.66667, 0.67451, 
0.68235, 0.6902, 0.69804, 0.70588, 0.71373, 0.72157, 0.72941, 0.73725, 0.7451, 
0.75294, 0.76078, 0.77255, 0.78039, 0.78824, 0.79608, 0.80392, 0.81569, 
0.82353, 0.83137, 0.84314, 0.85098, 0.86275, 0.87059, 0.87843, 0.8902 ,
0.89804, 0.9098, 0.91765, 0.92941, 0.94118, 0.94902, 0.96078, 0.97255, 
0.98039, 0.99216 };
    static float rgb_b[256] = {
0, 0, 0, 0, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00784, 
0.00784, 0.00784, 0.00784, 0.01176, 0.01176, 0.01176, 0.01176, 0.01176, 
0.01569, 0.01569, 0.01569, 0.01569, 0.01961, 0.01961, 0.01961, 0.01961, 
0.02353, 0.02353, 0.02353, 0.02745, 0.02745, 0.02745, 0.02745, 0.03137, 
0.03137, 0.03137, 0.03529, 0.03529, 0.03529, 0.03529, 0.04314, 0.04314, 
0.04314, 0.04706, 0.04706, 0.04706, 0.05098, 0.05098, 0.05098, 0.0549, 
0.0549, 0.0549, 0.05882, 0.05882, 0.05882, 0.06275, 0.06275, 0.06275, 0.06667, 
0.06667, 0.06667, 0.07059, 0.07059, 0.07451, 0.07451, 0.07451, 0.07843, 
0.07843, 0.08235, 0.08235, 0.08235, 0.08627, 0.08627, 0.0902, 0.0902, 
0.0902, 0.09412, 0.09412, 0.09804, 0.09804, 0.10196, 0.10196, 0.10588, 
0.10588, 0.1098, 0.1098, 0.1098, 0.11765, 0.11765, 0.12157, 0.12157, 
0.12549, 0.12549, 0.12941, 0.12941, 0.13333, 0.13725, 0.13725, 0.14118, 
0.14118, 0.1451, 0.1451, 0.14902, 0.14902, 0.15294, 0.15686, 0.15686, 
0.16078, 0.16078, 0.16471, 0.16863, 0.16863, 0.17255, 0.17255, 0.17647, 
0.18039, 0.18039, 0.18431, 0.18824, 0.18824, 0.19608, 0.2, 0.2, 0.20392, 
0.20784, 0.20784, 0.21176, 0.21569, 0.21961, 0.21961, 0.22353, 0.22745, 
0.23137, 0.23137, 0.23529, 0.23922, 0.24314, 0.24314, 0.24706, 0.25098, 
0.2549, 0.25882, 0.26275, 0.26275, 0.27059, 0.27451, 0.27843, 0.28235, 
0.28627, 0.2902, 0.29412, 0.29804, 0.29804, 0.30196, 0.30588, 0.3098, 
0.31373, 0.31765, 0.32157, 0.32549, 0.33333, 0.33725, 0.34118, 0.3451, 
0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.37255, 0.37647, 0.38039, 
0.38431, 0.38824, 0.39216, 0.4, 0.40392, 0.40784, 0.41176, 0.41961, 0.42353, 
0.42745, 0.43529, 0.43922, 0.44314, 0.45098, 0.4549, 0.45882, 0.46667, 
0.47059, 0.47843, 0.48235, 0.4902, 0.49412, 0.50196, 0.50588, 0.51373, 
0.51765, 0.52549, 0.52941, 0.53725, 0.54118, 0.54902, 0.55686, 0.56078, 
0.56863, 0.57647, 0.58039, 0.58824, 0.59608, 0.6, 0.60784, 0.61569, 0.62353, 
0.63137, 0.63529, 0.64314, 0.65098, 0.65882, 0.66667, 0.67451, 0.68235, 
0.6902, 0.69804, 0.70588, 0.71373, 0.72157, 0.72941, 0.73725, 0.7451, 
0.75294, 0.76078, 0.77255, 0.78039, 0.78824, 0.79608, 0.80392, 0.81569, 
0.82353, 0.83137, 0.84314, 0.85098, 0.86275, 0.87059, 0.87843, 0.8902, 
0.89804, 0.9098, 0.91765, 0.92941, 0.94118, 0.94902, 0.96078, 0.97255, 
0.98039, 0.99216 };
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;

  default :
    for(i=0; i<mNCol; i++) {
      mycol.green = mycol.red = mycol.blue = 0;
      AllocColor(mycol, i);
    }
    break;
  }
}
