#ifndef FITSFILE_H
#define FITSFILE_H

#include "ndatablock.h"
#include "dvlist.h"
#include "FitsIO/fitsio.h" 

#define OPENFILE    0
#define CREATEFILE  1
#define LEN_KEYWORD 9

// classes for saving/loading SOPHYA objects to/from FITS files...
// Guy le Meur (september 2000)


namespace SOPHYA {

  struct BnTblLine;
  class FitsFile;
  class FitsInFile;
  class FitsOutFile;
  enum WriteMode {append, clear, unknown};
  

//
//! Class for managing Interface for SOPHYA objects to FITS Format Files (uses cfitsio lib)

 class FitsIOHandler {


 public:

   virtual ~FitsIOHandler() {}
   void   Read(char flnm[],int hdunum= 0);
   void   Write(char flnm[]) ;
   void   Read(FitsInFile& ifts, int hdunum=0);
   void   Write(FitsOutFile& ofts) ;


 protected:
  
   virtual void    ReadFromFits(FitsInFile& is)=0;           
   virtual void    WriteToFits(FitsOutFile& os) =0;   
        
   friend class FitsInFile;
   friend class FitsOutFile;
  };


//! Class (virtual) for managing FITS format files
 class FitsFile {

 public:

   FitsFile() { InitNull(); };
   virtual ~FitsFile();
   static string GetErrStatus(int status);
   inline  int   statusF() const { return fits_status_;}
 

 protected:

   void         ResetStatus(int& status) ; 
   static  void printerror(int&) ;
   static  void printerror(int&,char* texte) ;
   inline void  InitNull() {fptr_ = NULL; hdutype_= 0; hdunum_ = 1; 
   fits_status_ = 0;}

   fitsfile *fptr_;     /**<  pointer to the FITS file, defined in fitsio.h */
   int hdutype_;        /**<  image or bintable ? */
   int hdunum_;         /**<   index of header to be read/written */
   int fits_status_;    /**< last status returned by fitsio library. updated only by several methods */

 };

//! Class for saving  SOPHYA objects on FITS Format Files (uses cfitsio lib)

 class FitsInFile : public  FitsFile {

 public:
   FitsInFile();
   FitsInFile(char flnm[]);
   ~FitsInFile() { ; };

   static int  NbBlocks(char flnm[]);
   static void GetBlockType(char flnm[], int hdunum, string& typeOfExtension, int& naxis, vector<int>& naxisn, string& dataType, DVList& dvl  );
   void        ReadFInit(int hdunum);
 
  /*! \return a reference on a DVList containing the keywords from FITS file */
  inline const DVList& DVListFromFits() const { return dvl_;}

  DVList  DVListFromPrimaryHeader() const;
  void    moveToFollowingHeader();




       //////////////////////////////////////////////////////////
       ///////   methods for managing extensions ////////////////
       //////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////
//       methods for managing FITS IMAGE extension
///////////////////////////////////////////////////



/*! \return true if the current header  corresponds to a FITS image extension */
inline bool IsFitsImage() const { return (hdutype_ == IMAGE_HDU);}



  /*! \return number of dimensions of an image extension : NAXIS parameter (in FITS notations)   */
inline int nbDimOfImage() const {return naxis_;}

/*! \return a reference on a vector containing sizes of the NAXIS dimensions : NAXIS1, NAXIS2, NAXIS3 etc.  */
 inline const vector<int>& dimOfImageAxes() const { return naxisn_;}


/*! \return total number of data in the current IMAGE extension */
inline int nbOfImageData() const { return nbData_; }



//////////////////////////////////////////////////////////////////////////
//       methods for managing FITS BINARY TABLE or ASCII TABLE extension
////////////////////////////////////////////////////////////////////////




/*! \return true if the current header  corresponds to a FITS ASCII or BINTABLE extension */
inline bool IsFitsTable() const {return (hdutype_ == ASCII_TBL || hdutype_ == BINARY_TBL);}


 static  void GetBinTabParameters(fitsfile* fileptr, int& nbcols, int& nrows,
				  vector<int>& repeat,
				  vector<string>& noms, 
				  vector<char>& types,   
				  vector<int>&  taille_des_chaines);
 char   ColTypeFromFits(int nocol) const;
 string ColNameFromFits(int nocol) const;
 int    ColStringLengthFromFits(int nocol) const;
 void   GetBinTabLine(int NoLine, double* ddata, float* fdata, int* idata, char
** cdata) ;
 void   GetBinTabLine(long NoLine, BnTblLine& ligne) ;
 void   GetBinTabLine(int NoLine, float* fdata) ;
 void   GetBinTabFCol(double* valeurs, int nentries, int NoCol) const;
 void   GetBinTabFCol(float* valeurs, int nentries, int NoCol) const;
 void   GetBinTabFCol(int* valeurs, int nentries,  int NoCol) const;
 void   GetBinTabFCol(char** valeurs,int nentries, int NoCol) const;

/////////////////////////////////////////////////////////////
//       methods for managing any type of FITS extension
////////////////////////////////////////////////////////

  int     NbColsFromFits() const;
  int     NentriesFromFits(int nocol) const;


  void    GetSingleColumn(double* map, int nentries) const;

  void    GetSingleColumn(float*  map, int nentries) const;

  void    GetSingleColumn(int* map, int nentries) const;

  private :

void InitNull();
static void KeywordsIntoDVList(fitsfile* fileptr, DVList& dvl, int hdunum); 
static  void GetImageParameters (fitsfile* fileptr,int& bitpix,int& naxis,vector<int>& naxisn);

 int bitpix_;          /**< fits-Image parameter */
 int naxis_;           /**< fits-Image parameter */
 vector<int> naxisn_;  /**< fits-Image parameters : sizes of dimensions */
 int nbData_;          /*< fits-Image parameter: number of data */
 int nrows_;           /**< Bintable parameter */
 vector<int> repeat_;  /**< Bintable parameter */
 int nbcols_;          /**< Bintable parameter */
 vector<string> noms_; /**< Bintable parameter: column names */
 vector<char> types_;  /**< Bintable parameters: types of columns (D: double, E: float, I: integers,  A: char*) */
 DVList dvl_;          /**< DVList for transferring keywords */
 vector<int>  taille_des_chaines_; /**< Bintable parameters:   length of the char* variables */

 };

//! Class for loading  SOPHYA objects from FITS Format Files (uses cfitsio lib)

 class FitsOutFile : public  FitsFile {

 public:


   FitsOutFile();
   FitsOutFile(char flnm[], WriteMode wrm = unknown );
   ~FitsOutFile() { ;};
   inline void InitNull() {imageOnPrimary_=false;}

       //////////////////////////////////////////////////////////
       ///////   methods for managing extensions ////////////////
       //////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////
//       methods for managing FITS IMAGE extension
///////////////////////////////////////////////////


   inline void firstImageOnPrimaryHeader() {imageOnPrimary_=true;}
   void makeHeaderImageOnFits(char type, int nbdim, int* naxisn, DVList &dvl) ;
   void PutImageToFits( int nbData, double* map) const;
   void PutImageToFits(int nbData, float* map ) const;
   void PutImageToFits(int nbData, int* map) const;



//////////////////////////////////////////////////////////////////////////
//       methods for managing FITS BINARY TABLE or ASCII TABLE extension
////////////////////////////////////////////////////////////////////////



   void makeHeaderBntblOnFits ( string fieldType, vector<string> Noms, int nentries, int tfields, DVList &dvl, string extname,  vector<int> taille_des_chaines) ;
   void PutColToFits(int nocol, int nentries, double* donnees) const;
   void PutColToFits(int nocol, int nentries, float* donnees) const;
   void PutColToFits(int nocol, int nentries, int* donnees) const;
   void PutColToFits(int nocol, int nentries, char** donnees) const;
   void PutBinTabLine(long NoLine,  BnTblLine& ligne) const;


/////////////////////////////////////////////////////////////
//       methods for managing any type of FITS extension
////////////////////////////////////////////////////////


void  DVListIntoPrimaryHeader(DVList& dvl) const;



  private :

  void writeSignatureOnFits() const; 
  void addKeywordsOfDVList(DVList& dvl) const;

  bool imageOnPrimary_;

 };

 struct BnTblLine 
 {
   BnTblLine() {}
   void setFormat(int dc, int fc, int ic, int cc, vector<string> names);
   bool sameFormat(const BnTblLine& btl) const;

   void Print();

   vector<double> ddata_;
   vector<float>  fdata_;
   vector<int>    idata_;
   vector<string>  cdata_;
   vector<string> ColName_;

 };


} // Fin du namespace


#endif
