#include "fftpserver.h"
#include "fftpackc.h"

#include <iostream.h>


/*! 
  \class SOPHYA::FFTPackServer
  \ingroup NTools
  An implementation of FFTServerInterface based on fftpack, for 
  one dimensional arrays.

The class calls the c library ``fftpack'', which is accessible and documented
at http://www.netlib.org/fftpack/.  However, the class functions do not
necessarily correspond with the equivalent fftpack function.  For example,
fftpack "forward" transformations are in fact inverse fourier transformations.
Otherwise, the output is in the fftpack format.


Due to the way that fftpack manages
its work arrays, an object can run faster if the length of the input arrays
does not change.  For example, if you need to do a series of FFT's
of differing length, it may be more efficient to create an fftserver object
for each length.
*/


FFTPackServer::FFTPackServer()
   : FFTServerInterface("FFTPackServer using extended FFTPack (C-version) package")
  , ckR4(true, true) , ckR8(true, true)
{
//the working array and its size for the different
//possible numerical types
  sz_rfft = 0;     
  ws_rfft = NULL;  
  sz_dfft = 0;     
  ws_dfft = NULL;  
  sz_cfft = 0;
  ws_cfft = NULL;
  sz_cdfft = 0;
  ws_cdfft = NULL;
}

FFTPackServer::~FFTPackServer()
{
if (ws_rfft) delete[] ws_rfft;
if (ws_dfft) delete[] ws_dfft;
if (ws_cfft) delete[] ws_cfft;
if (ws_cdfft) delete[] ws_cdfft;
}

FFTServerInterface * FFTPackServer::Clone()
{
  return (new FFTPackServer);
}


void FFTPackServer::FFTForward(TArray< complex<r_8> > const & in, TArray< complex<r_8> > & out)
{
  ckR8.CheckResize(in, out);
  out = in;
  fftf(out.Size(), out.Data());
  if (getNormalize()) out *= (1./(r_8)(in.Size()));
}

void FFTPackServer::FFTBackward(TArray< complex<r_8> > const & in, TArray< complex<r_8> > & out)
{
  ckR8.CheckResize(in, out);
  out = in;
  fftb(out.Size(), out.Data());
}



void FFTPackServer::FFTForward(TArray< complex<r_4> > const & in, TArray< complex<r_4> > & out)
{
  ckR4.CheckResize(in, out);  
  out = in;
  fftf(out.Size(), out.Data());
  if (getNormalize()) out *= (1./(r_4)(in.Size()));
}

void FFTPackServer::FFTBackward(TArray< complex<r_4> > const & in, TArray< complex<r_4> > & out)
{
  ckR4.CheckResize(in, out);  
  out = in;
  fftb(out.Size(), out.Data());
}

void FFTPackServer::FFTForward(TArray< r_4 > const & in, TArray< complex<r_4> > & out)
{
  ckR4.CheckResize(in, out);  
  TArray< r_4 > inout(in, false);
  fftf(inout.Size(), inout.Data());
  ckR4.ReShapetoCompl(inout, out);
  if (getNormalize()) out *= complex<r_4>((1./(r_4)(in.Size())), 0.);
}

void FFTPackServer::FFTBackward(TArray< complex<r_4> > const & in, TArray< r_4 > & out)
{
  ckR4.CheckResize(in, out);    
  ckR4.ReShapetoReal(in, out);
  fftb(out.Size(), out.Data());
}


void FFTPackServer::FFTForward(TArray< r_8 > const & in, TArray< complex<r_8> > & out)
{
  ckR8.CheckResize(in, out);  
  TArray< r_8 > inout(in, false);
  fftf(inout.Size(), inout.Data());
  ckR8.ReShapetoCompl(inout, out);
  if (getNormalize()) out *= complex<r_8>((1./(r_8)(in.Size())), 0.);
}

void FFTPackServer::FFTBackward(TArray< complex<r_8> > const & in, TArray< r_8 > & out)
{
  ckR8.CheckResize(in, out);  
  ckR8.ReShapetoReal(in, out);
  fftb(out.Size(), out.Data());
}

 

void FFTPackServer::checkint_rfft(int_4 l)
{
  if (sz_rfft == l) return;       //checkint functions check and reallocate
                                  //memory for the work arrays when performing
  if (ws_rfft) delete[] ws_rfft;  //a transform
  sz_rfft = l;
  ws_rfft = new r_4[2*l+15];
  rffti_(&l, ws_rfft);
}

void FFTPackServer::checkint_cfft(int_4 l)
{
  if (sz_cfft == l) return;

  if (ws_cfft) delete[] ws_cfft;
  sz_cfft = l;
  ws_cfft = new r_4[4*l+15];
  cffti_(&l, ws_cfft);
}

void FFTPackServer::checkint_dfft(int_4 l)
{
  if (sz_dfft == l) return;

  if (ws_dfft) delete[] ws_dfft;
  sz_dfft = l;
  ws_dfft = new r_8[2*l+15];
  dffti_(&l, ws_dfft);
}

void FFTPackServer::checkint_cdfft(int_4 l)
{
  if (sz_cdfft == l) return;

  if (ws_cdfft) delete[] ws_cdfft;
  sz_cdfft = l;
  ws_cdfft = new r_8[4*l+15];
  cdffti_(&l, ws_cdfft);
}

/* In general forward transformations are resorted since fftpack functions
   return inverse transformations */

void FFTPackServer::fftf(int_4 l, r_4* inout)
{
  checkint_rfft(l);
  rfftf_(&l, inout, ws_rfft);
  //  for (int k= 2;k<=(l+1)/2;k++) inout[2*k-2]=-inout[2*k-2];
}

void FFTPackServer::fftf(int_4 l, r_8* inout)
{
  checkint_dfft(l);
  dfftf_(&l, inout, ws_dfft);
  //  for (int k= 2;k<=(l+1)/2;k++) inout[2*k-2]=-inout[2*k-2];
}

void FFTPackServer::fftf(int_4 l, complex<r_4>* inout)
{
  checkint_cfft(l);
  cfftf_(&l, (r_4 *)(inout), ws_cfft);
}

void FFTPackServer::fftf(int_4 l, complex<r_8>* inout)
{
  checkint_cdfft(l);
  cdfftf_(&l, (r_8*)(inout), ws_cdfft);
}

void FFTPackServer::fftb(int_4 l, r_4* inout)
{
  checkint_rfft(l);
  rfftb_(&l, inout, ws_rfft);
}

void FFTPackServer::fftb(int_4 l, r_8* inout)
{
  checkint_dfft(l);
  dfftb_(&l, inout, ws_dfft);
}

void FFTPackServer::fftb(int_4 l, complex<r_4>* inout)
{
  checkint_cfft(l);
  cfftb_(&l, (r_4 *)(inout), ws_cfft);
}

void FFTPackServer::fftb(int_4 l, complex<r_8>* inout)
{
  checkint_cdfft(l);
  cdfftb_(&l, (r_8 *)(inout), ws_cdfft);
}


