// This may look like C code, but it is really -*- C++ -*-
//  Classes image heritant de TMatrix<T>
//                       R.Ansari, C.Magneville 07/2000    
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#ifndef CIMAGE_SEEN
#define CIMAGE_SEEN

#include "tmatrix.h"
#include "fioarr.h"


// **********************************************************
// Classe Image

namespace SOPHYA {

//! Class for handling images
template <class T>
class Image : public TMatrix<T> {

public:
  Image();
  Image(sa_size_t sizx, sa_size_t sizy, r_8 szpx=1., r_8 szpy=1., r_8 orgx=0., r_8 orgy=0.);
  Image(const Image<T>& a);
  Image(const Image<T>& a, bool share);

  virtual ~Image();

  // Inline element acces methods 
  inline T const& operator()(sa_size_t ix, sa_size_t jy) const;
  inline T&       operator()(sa_size_t ix, sa_size_t jy);

  //! Returns the image size along X (corresponding to the number of columns)  
  inline sa_size_t  XSize() const {return NCols();}
  //! Returns the image size along Y (corresponding to the number of lines)  
  inline sa_size_t  YSize() const {return NRows();}

  //! Returns the X position, for pixel(0,0)
  inline r_8     XOrg() const {return org_x;}
  //! Returns the Y position, for pixel(0,0)
  inline r_8     YOrg() const {return org_y;}

  //! Returns pixel size along X 
  inline r_8     XPixSize() const {return pxsz_x;}
  //! Returns pixel size along Y
  inline r_8     YPixSize() const {return pxsz_y;}

  //! Returns the pixel position along X, for pixels with X index \b i
  inline r_8     XPos(sa_size_t i) const {return org_x + i*pxsz_x;}
  //! Returns the pixel position along Y, for pixels with Y index \b j
  inline r_8     YPos(sa_size_t j) const {return org_y + j*pxsz_y;}

  //! Set the position for pixel(0,0)
  inline void    SetOrg(r_8 orgx=0., r_8 orgy=0.)
                       { org_x = orgx;  org_y = orgy; }  
  //! Set the pixel size
  inline void    SetPixelSize(r_8 szx=1., r_8 szy=1.)
                       { pxsz_x = szx;   pxsz_y = szy; }

protected:
  r_8   org_x,org_y;        //  Coordonnees pixel(0,0)
  r_8   pxsz_x, pxsz_y;     //  Taille des pixels
};

//! () : Return the pixel value (element) for pixel \b ix (column index) and \b jy (line index)
template <class T>
inline T const& Image<T>::operator()(sa_size_t ix, sa_size_t jy) const
{
  return (TMatrix<T>::operator() (jy, ix));
}

//! () : Return the pixel value (element) for pixel \b ix (column index) and \b jy (line index)
template <class T>
inline T & Image<T>::operator()(sa_size_t ix, sa_size_t jy) 
{
  return (TMatrix<T>::operator() (jy, ix));
}

/////////////////////////////////////////////////////////////////////////
//! Class for persistent management of Image
template <class T>
class FIO_Image : public FIO_TArray<T>   {
public:
  FIO_Image();
  FIO_Image(string const & filename); 
  FIO_Image(const Image<T> & obj);
  FIO_Image(Image<T> * obj);
  //  virtual ~FIO_Image();  
  virtual void        SetDataObj(AnyDataObj & o);
  inline operator Image<T>() { return(*(dynamic_cast<Image<T> * >(dobj))); }
protected :
  virtual void ReadSelf(PInPersist&);           
  virtual void WriteSelf(POutPersist&) const;  
};

/*! \ingroup NTools \fn operator<<(POutPersist&,Image<T>&)
  \brief Write Image \b obj into POutPersist stream \b os */
template <class T>
inline POutPersist& operator << (POutPersist& os, Image<T> & obj)
{ FIO_Image<T> fio(&obj);  fio.Write(os);  return(os); }

/*! \ingroup NTools \fn operator>>(PInPersist&,Image<T>&)
  \brief Read Image \b obj from PInPersist stream \b os */
template <class T>
inline PInPersist& operator >> (PInPersist& is, Image<T> & obj)
{ FIO_Image<T> fio(&obj);  fio.Read(is);  return(is); }



typedef Image<uint_2> ImageU2;
typedef Image<int_4>  ImageI4;
typedef Image<r_4>    ImageR4;
typedef Image<r_8>    ImageR8;

class GeneralFit;
//! OBSOLETE - Should not be used - for compatibility with peida in (s)piapp
class RzImage {
public:
  RzImage() { }
  ~RzImage() { }
  inline uint_4  XSize() const {return 0;}
  inline uint_4  YSize() const {return 0;}

  inline r_8     XOrg() const {return 0;}
  inline r_8     YOrg() const {return 0;}

  inline r_8     XPixSize() const {return 1.;}
  inline r_8     YPixSize() const {return 1.;}
  inline r_8     DValue(int i, int j) const { return 0.; }
  // Resultats d'un GeneralFit
  inline RzImage*      FitResidus(GeneralFit& gfit) { return NULL; }
  inline RzImage*      FitFunction(GeneralFit& gfit) { return NULL; }

};

}  // End of namespace SOPHYA

#endif


