#include "machdefs.h"    // Definitions specifiques SOPHYA

#include <math.h>
#include <iostream.h>

#include "nbmath.h"
#include "timing.h"

#include "array.h"
#include "skymap.h"
#include "samba.h"
#include "sambainit.h"
#include "fitsspherehealpix.h"
#include "fitstarray.h"


/*!
  \ingroup PrgUtil
  \file map2cl.cc
  \brief \b map2cl: Computes power spectra (C(l)) on spherical maps.

  \verbatim

  csh> map2cl -h
  SOPHYA Version  1.1 Revision 0 (V_Fev2001) -- Mar  9 2001 15:45:31 cxx

  map2cl : Spherical harmonics analysis - HEALPix map -> Power spectrum C_l
  Usage: map2cl [-float/-r4] [-lmax lval] [-thetacut dtdeg]
         [-fitsin] [-fitsout] InFileName OutFileName
   -float (-r4): single precision C_l and map (default = double)
   -lmax lval: Maximum value for l index (default=255)
   -thetacut dtdeg : Symmetric delta-theta cut (in degree) along equator
                    (default=0 -> no cut)
   -fitsout: Select the FITS format for the output map (default PPF format)
   -fitsin : Select the FITS format for the input vector (default PPF format)
   InFileName : Input file name (HEALPix map)
   OutFileName : Output file name (the C_l vector)

  \endverbatim
*/

/* Nouvelle-Fonction */
void Usage(bool fgerr)
{
  cout << endl;
  if (fgerr) {
    cout << " map2cl : Argument Error ! map2cl -h for usage " << endl; 
    exit(1);
  }
  else {
    cout << " map2cl : Spherical harmonics analysis - HEALPix map -> Power spectrum C_l \n" 
	 << " Usage: map2cl [-float/-r4] [-lmax lval] [-thetacut dtdeg] \n" 
	 << "        [-fitsin] [-fitsout] InFileName OutFileName \n" 
	 << "   -float (-r4): single precision C_l and map (default = double) \n"
	 << "   -lmax lval: Maximum value for l index (default=255)\n" 
	 << "   -thetacut dtdeg : Symmetric delta-theta cut (in degree) along equator \n" 
	 << "                    (default=0 -> no cut)\n"
	 << "   -fitsout: Select the FITS format for the output map (default PPF format) \n"
	 << "   -fitsin : Select the FITS format for the input vector (default PPF format) \n"
	 << "   InFileName : Input file name (HEALPix map) \n" 
	 << "   OutFileName : Output file name (the C_l vector) \n" << endl; 
    exit(0);
  }  
}

/* Nouvelle-Fonction */
template <class T> 
class _Map2Cl {
public :
static void ComputeCl(string & infile, string & outfile, int lmax, double tcut, 
		      bool fgfitsin, bool fgfitsout)
{
  double deg2rad =  M_PI/180.;
  double minute2rad =  M_PI/(180.*60.);

  SphereHEALPix<T> sph;
  if (fgfitsin) {
    cout << "--- Reading Input FITS file " << infile << endl; 
    FitsInFile fii(infile);
    fii >> sph;
  }
  else {
    cout << "--- Reading Input PPF file " << infile << endl; 
    PInPersist ppi(infile);
    ppi >> sph;
  }

  cout << " Input map : NbPixels= " <<  sph.NbPixels() << " NSide= " 
       << sph.SizeIndex() << " Resolution= "
       << sqrt(sph.PixSolAngle(0))/minute2rad << " Arcminutes " << endl;
  
  double ctcut = (tcut < 1.e-19) ? 0. : cos((90.-tcut)*deg2rad);
  cout << "--- Calling  DecomposeToCl() (lmax= " << lmax 
       << " cos_theta_cut= " << ctcut << ") theta_cut=" << tcut << " deg" << endl;
  // Decomposition de la carte en C_l
  SphericalTransformServer<T> sphtr;
  TVector<T> clvec = sphtr.DecomposeToCl(sph, lmax, ctcut);
 
  T min, max;
  double mean, sigma;
  clvec.MinMax(min, max);
  MeanSigma(clvec, mean, sigma);
  cout << "--- Statistics on the computed C_l vector: Size=" << clvec.Size() << endl; 
  cout << " C_l.Min= " << min << " C_l.Max= " << max 
       << " C_l.Mean= " << mean << " C_l.Sigma= " << sigma << endl;
  
  if (fgfitsout) {
    cout << "--- Writing C_l vector to Output FITS file " << outfile << endl; 
    FitsOutFile fio(outfile);
    fio << clvec;
  }
  else {
    cout << "--- Writing C_l vector to Output PPF file " << outfile << endl; 
    POutPersist ppo(outfile);
    ppo << clvec;
  }
}

};

/* Main-Program */
int main(int narg, char *arg[])
{
  if ((narg > 1) && (strcmp(arg[1],"-h") == 0) ) Usage(false);
  
  int lmax = 255;
  double tcut = 0.;
  string infile;
  string outfile;
  bool fgfitsin = false;
  bool fgfitsout = false;
  bool fgr4 = false;
  cout << " map2cl : Decoding command line options ... " << endl;

  int ko=1;
  for (int k=1; k<narg; k++)   {
    if (strcmp(arg[k], "-lmax") == 0)  {
      if (k == narg-1) Usage(true);  // -lmax est suivi d'un argument 
      lmax = atoi(arg[k+1]);  k++;       // k++ pour sauter au suivant
    }
    else if (strcmp(arg[k], "-thetacut") == 0)  {
      if (k == narg-1) Usage(true);  // -thetacut est suivi d'un argument 
      tcut = atof(arg[k+1]);  k++;       // k++ pour sauter au suivant
    }
    else if (strcmp(arg[k], "-fitsin") == 0) {
      fgfitsin = true;  
    }
    else if (strcmp(arg[k], "-fitsout") == 0) {
      fgfitsout = true;  
    }
    else if ((strcmp(arg[k], "-float") == 0) || (strcmp(arg[k], "-r4") == 0) ) {
      fgr4 = true;  
    }
   
    else { ko = k; break; }  // Debut des noms
  }

  if ((narg-ko) < 2)  Usage(true); 
  infile = arg[ko];
  outfile = arg[ko+1];
  
  // Bloc try de gestion des exception 
  try {
    InitTim();
    SophyaInit();
    if (fgr4) {
      cout << " SphereHEALPix<r_4>  --> Power spectrum C_l<r_4> (float)" << endl;
      _Map2Cl<r_4>::ComputeCl(infile, outfile, lmax, tcut, fgfitsin, fgfitsout);
    }
    else {
      cout << " SphereHEALPix<r_8>  --> Power spectrum C_l<r_8> (double)" << endl;
      _Map2Cl<r_8>::ComputeCl(infile, outfile, lmax, tcut, fgfitsin, fgfitsout);
    }
  }
  catch (PThrowable & exc) {   // Exceptions de SOPHYA
    cerr << " map2cl: Catched Exception " << (string)typeid(exc).name()
         << " - Msg= " << exc.Msg() << endl;
  }
  catch (...) {    // Autres Exceptions
    cerr << " map2cl: some other exception was caught ! " << endl;
  }
  
  PrtTim("End of map2cl ");
  return(0);
}


  
