// Module PI : Peida Interactive     PIColorMap
// Gestion de table de couleurs        96
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include "picmap.h"
#include <iostream.h>

//++
// Class	PIColorMap
// Lib		PI
// include	picmap.h
//
//   	Classe fournissant les services de gestion des tables de couleur.
//	Une hirachie un peu complexe a t mise en place pour 
//	l'implmentation des tables de couleurs pour chaque systme
//	(Mac, XWindow, ...) :
//|	PIColorMapGen  -----  PIColorMapX     (=PIColorMapNative)
//|	                 |--  PIColorMapMac   (=PIColorMapNative)
//	et la gestion de table de couleurs standard partage.
//|	PIColorMapNative (=X,Mac,...) ----- PIColorMap.
//	La seule classe  utiliser est *PIColorMap* qui fournit 
//	l'ensemble des services de gestion de table de couleurs.
//--

#define  MXMAPIDS   22
static   int    MapNCols[MXMAPIDS] = {256, 256, 256, 256, 256, 256, 256, 256, 256, 256,
                            256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256 };

// Tables de couleurs :
// 5 Standard PI , 32  couleurs
// 4 Standard PI , 128 couleurs
// 12 MIDAS , 256 couleurs
// 1 Standard PI 16 couleurs

static   const char*  MapNoms[MXMAPIDS]  = {
                      "Grey32","InvGrey32","ColRJ32","ColBR32","ColRV32", 
                      "Grey128","InvGrey128","ColRJ128","ColBR128",
                      "MIDAS_Pastel","MIDAS_Heat","MIDAS_Rainbow3", 
                      "MIDAS_BlueRed","MIDAS_BlueWhite","MIDAS_Stairs8",
                      "MIDAS_Stairs9","MIDAS_StairCase","MIDAS_Color",
                      "MIDAS_ManyCol","MIDAS_Idl14","MIDAS_Idl15",
                      "Rainbow16"};

static   CMapId MapIds[MXMAPIDS] = { 
                      CMAP_GREY32, CMAP_GREYINV32, CMAP_COLRJ32, CMAP_COLBR32, CMAP_COLRV32, 
                      CMAP_GREY128, CMAP_GREYINV128, CMAP_COLRJ128, CMAP_COLBR128,  
                      CMAP_MIDAS_Pastel, CMAP_MIDAS_Heat, CMAP_MIDAS_Rainbow3, 
                      CMAP_MIDAS_BlueRed, CMAP_MIDAS_BlueWhite, CMAP_MIDAS_Stairs8, 
                      CMAP_MIDAS_Stairs9, CMAP_MIDAS_StairCase, CMAP_MIDAS_Color, 
                      CMAP_MIDAS_ManyCol, CMAP_MIDAS_Idl14, CMAP_MIDAS_Idl15, 
                      CMAP_COL16 } ;

static   int mColTNums[8192];   // Max 8192 tables differentes pour le moment  
static   int CTNum = 0;        //  Numero des tables ds mColTNum - s'incremente a chaque creation 

static   int NMaxTableAuto = 0;
static   PIColorMap* mMaps[MXMAPIDS] = {
                                  NULL, NULL, NULL, NULL, NULL, 
                                  NULL, NULL, NULL, NULL, NULL, 
                                  NULL, NULL, NULL, NULL, NULL,
                                  NULL  };

int  PIColorMap::NumberStandardColorMaps()
{
  return ( MXMAPIDS );
}

CMapId PIColorMap::GetStandardColorMapId(int num)
{
  if ( (num < 0) || (num >= MXMAPIDS) )  num = 0;
  return (MapIds[num]) ;
}

string PIColorMap::GetStandardColorMapName(int num)
{
  if ( (num < 0) || (num >= MXMAPIDS) )  num = 0;
  return (MapNoms[num]) ;
}


//++
// Titre	Constructeurs, mthodes
//--
//++
// PIColorMap(CMapId id)
//	Constructeur d'une table de couleur standard, identifie par "CMapId id"
//|	CMAP_GREY32     :  32 niveaux de gris
//|	CMAP_GREYINV32  :  32 niveaux gris du blanc au noir
//|	CMAP_COLRJ32    :  32 couleurs, du Rouge au Jaune/blanc
//|	CMAP_COLBR32    :  32 couleurs du bleu au rouge (arcenciel)
//|	CMAP_COLRV32    :  32 couleurs du rouge au violet
//|	CMAP_GREY128    :  128 niveaux de gris
//|	CMAP_GREYINV128 :  128 niveaux gris invers
//|	CMAP_COLRJ128   :  128 couleurs, du Rouge au Jaune/blanc
//|	CMAP_COLBR128   :  128 couleurs du bleu au rouge (arcenciel)
//|	CMAP_COL16      :  16 Couleurs arcenciel
//|	CMAP_OTHER      :  Table non standard (Midas,Idl, ...)
//	Toutes les tables de couleurs standard possdent 256 cellules de couleurs,
//	contenant 16 ou 32 ou 128 couleurs distinctes.
//
// PIColorMap(string const& nom, int nCol)
//	Constructeur d'une table de "nCol" couleurs identifi par la chane "nom".
// PIColorMap(PIColorMap&)
//	Constructeur par copie.
// int Type()
//	Retourne le type de la table, correspondant  "CMapId id"
// int NCol()
//	Retourne le nombre de cellules de couleurs dans la table
// string const& Nom()
//	Renvoie le nom de la table.
//--

PIColorMap::PIColorMap(CMapId id)
: PIColorMapNative()
{
   int k, kdx;
   string mnom;

   if (NMaxTableAuto == 0) { //  Il faut allouer les tables de depart
     if ( TotNbColors() > 4096 ) NMaxTableAuto = MXMAPIDS-1;
     else NMaxTableAuto = 3; 
     //     for(k=0; k<NMaxTableAuto; k++)  { 
     for(k=0; k<3; k++)  {   // On alloue les 3 tables de depart
        mnom = MapNoms[k]; 
	cout << " PIColorMap::PIColorMap(CMapId id) Allocating ColorMap " 
	     << mnom << " ... " << endl;
        mMaps[k] = new PIColorMap(MapIds[k], MapNCols[k], mnom);
     }
   }

//  Protection pour valeur de Id  
   for(kdx=0; kdx<MXMAPIDS; kdx++)  if (id == MapIds[kdx])  break;
   if (kdx == MXMAPIDS)  { id = MapIds[0];  kdx = 0; }
   for(k=0; k<NMaxTableAuto; k++) {
     if ((MapIds[k] == id) && (mMaps[k] == NULL))  { //  Table pas encore alloue 
       mnom = MapNoms[k]; 
       cout << " PIColorMap::PIColorMap(CMapId id) Allocating ColorMap " 
	     << mnom << " ... " << endl;
       mMaps[k] = new PIColorMap(MapIds[k], MapNCols[k], mnom);
     }
     if (mMaps[k] && (id == mMaps[k]->Type()) ) {
       CopyFrom(mMaps[k]);  
       mCTId = mMaps[k]->mCTId;  mColTNums[mCTId]++;
       break;
     }
   }
   if (Type() != id)  {  // Table pas encore trouve dans la liste
     k = NMaxTableAuto;
     if (!mMaps[k] || mMaps[k]->Type() != id) {
       if ( mMaps[k] ) {  
         mColTNums[mMaps[k]->mCTId] = 1;  // Je force a liberer les couleurs 
         delete mMaps[k] ;
       }
       mnom = MapNoms[kdx];
       mMaps[k] = new PIColorMap(MapIds[kdx],  MapNCols[kdx], mnom); 
     }
    CopyFrom(mMaps[k]);  
    mCTId = mMaps[k]->mCTId;  mColTNums[mCTId]++;
   }
}


PIColorMap::PIColorMap(string const& nom, int nCol)
: PIColorMapNative(CMAP_OTHER, nCol, nom)
{
  MakeColMap();
  mCTId = CTNum;
  mColTNums[mCTId] = 1;
  CTNum++;
}

PIColorMap::PIColorMap(CMapId id, int nCol, string const& nom)
: PIColorMapNative(id, nCol, nom)
{
  MakeColMap();
  mCTId = CTNum;
  mColTNums[mCTId] = 1;
  CTNum++;
//  printf("PIColorMap/Debug-Creation:  %d %d - %s ", Type(), NCol(), Nom().c_str());
//  printf("  ..CRE.. %d %d \n", CTNum-1, mColTNums[CTNum-1]);
}

PIColorMap::PIColorMap(PIColorMap& cm)
{
  CopyFrom(&cm);
  mCTId = cm.mCTId;
  mColTNums[mCTId]++;
}

PIColorMap::~PIColorMap()
{
//  printf("PIColorMap/Debug-Delete:  %d %d - %s ", Type(), NCol(), Nom().c_str());
//  printf("...DEL.CTNum,mColTNums= %d %d (%d %d) ", 
//        CTNum, mColTNums[CTNum], mCTId,mColTNums[mCTId] );
  mColTNums[mCTId]--; 
  if ( mColTNums[mCTId] == 0 )  { FreeColors();  mColTNums[mCTId] = 0; }
//  printf("\n");
}


//++
// Titre	Gestion des couleurs
//	Une structure *PIColor* permet la manipulation des couleurs  
//	travers 3 champs de type "unsigned short: 0 ... 65535" dfinissant
//	les intensit des couleurs de base: Rouge, Bleue, Vert.
//|	struct PIColor {
//|	  unsigned short red;
//|	  unsigned short green;
//|	  unsigned short blue; }
//
//--
//++
// PIColor GetColor(int n)	
//	Renvoie la couleur de la cellule "n"
// bool AllocColor(PIColor const& col, int index)
//	Alloue la couleur dfinie par "col" dans la cellule numro "index".
//	Renvoie "true" si OK, "false" sinon.
// void  FreeColors()
//	Libre les couleurs alloues par la table
// long  TotNbColors()
//	Renvoie le nombre total de couleurs disponibles 
//	(Ex 256 pour un systme  8 plans image, 65536 pour 16 plans, etc ...)
// int  NbAllocColors()
//	Renvoie le nombre total de couleurs dj utilise par l'application
//--

void PIColorMap::CleanColorMaps()
{
  int k;
  for(k=0; k<MXMAPIDS; k++)  { 
    if (!mMaps[k])  continue;
    mMaps[k]->FreeColors();
    delete mMaps[k];
  }
}

/*  Definition d' une table de couleurs  R G B   */

static float R_RGB1[16] =
{0.0,0.00,0.0,0.0,0.0,0.00,0.0,0.6,0.8,1.0,1.00,1.0,0.75,1.0,0.85,1.0};
static float G_RGB1[16] =
{0.0,0.00,0.0,0.8,1.0,1.00,1.0,0.8,0.8,1.0,0.75,0.6,0.00,0.0,0.0,0.0};
static float B_RGB1[16] =
{0.0,0.75,1.0,1.0,1.0,0.75,0.0,0.0,0.0,0.0,0.00,0.5,0.75,1.0,0.0,0.0};



void PIColorMap::MakeColMap()
{
  PIColor mycol;
  int i,j,k;

  switch (mType) {

    case CMAP_GREY32 :
      for(i=0; i<32; i++) {
        mycol.red = mycol.green = mycol.blue = 3535+(i*62000/32);
        for(k=0; k<8; k++)     AllocColor(mycol, i*8+k);
      }
      break;
    

    case CMAP_GREYINV32 :
      for(i=0; i<32; i++) {
        mycol.red = mycol.green = mycol.blue = 3535+(i*62000/32);
        for(k=0; k<8; k++)     AllocColor(mycol, mNCol-(i*8+k)-1);
      }
      break;

    case CMAP_COL16 :
      for(i=0; i<16; i++) {
        mycol.red = (int)(R_RGB1[i]*65535.);
        mycol.green = (int)(G_RGB1[i]*65535.);
        mycol.blue =(int)(B_RGB1[i]*65535.) ;
        for(k=0; k<16; k++)     AllocColor(mycol, i*16+k);
      }
      break;

    case CMAP_COLRJ32 :
      for( i=0; i<32; i++) {
        if (i < 12)  { 
          mycol.green = mycol.blue = 0;  
          mycol.red = 21535+(i*4000);  }
        else if (i < 24) { 
          mycol.blue = 0;   mycol.red = 65535;
          mycol.green = 21535+(i-12)*4000;  }
        else { 
          mycol.green = mycol.red = 65535;
          mycol.blue = 17535+(i-23)*6000; }
        for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
        }
      break;

  case CMAP_COLBR32 :
    for( i=0; i<32; i++) {
      if (i < 5) { 
        mycol.green = mycol.red = 0;    // Du bleu sombre au bleu clair
        mycol.blue = 5535+i*15000; }
      else if (i < 10) { 
        mycol.blue = 65535;             // Plein bleu - On rajoute du vert -> bleu clair
        mycol.green = 5535+(i-5)*9000;
        mycol.red = 0;  }
      else if (i < 15) { 
        mycol.blue = 60000-(i-10)*15000;   // On passe au vert
        mycol.green = 41535+(i-10)*6000;
        mycol.red = 0;  }
      else if (i < 21) { 
        mycol.blue = 0;
        mycol.green = 65535;               // Plein vert, on rajoute du rouge -> jaune
        mycol.red = 5535+(i-15)*12000;  }
      else if (i < 27) {  
        mycol.blue = 0;                     //  On diminue vert -> Orange
        mycol.green = 60000-(i-21)*12000;
        mycol.red = 65535-(i-21)*2000;  }
      else {
        mycol.blue = 0;                     // Plein rouge
        mycol.green = 0;
        mycol.red = 57535+(i-27)*2000; }

      for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
      }
    break;

  case CMAP_COLRV32 :
    for( i=0; i<32; i++) {
      if (i < 9) {
        mycol.green = i*5000; mycol.blue = 0;  
        mycol.red = i*6000; }
      else if (i < 15) {
	mycol.green = 40000;
	mycol.blue = 0;  
	mycol.red = 50000+(i-9)*1000;
      }
      else if (i < 21) {
        mycol.green = 40000-(i-15)*8000;
        mycol.blue = 0;
        mycol.red = 57535+(i-15)*1600;  }
      else { 
        mycol.green = 0;  mycol.red = 65535;
        mycol.blue = 15535+(i-21)*5000; }

      for(k=0; k<8; k++)   AllocColor(mycol, i*8+k); 
      }
    break;

    case CMAP_GREY128 :
      for(i=0; i<128; i++) {
        mycol.red = mycol.green = mycol.blue = 2035+(i*500);
        for(k=0; k<2; k++)     AllocColor(mycol, i*2+k);
      }
      break;
    

    case CMAP_GREYINV128 :
      for(i=0; i<128; i++) {
        mycol.red = mycol.green = mycol.blue = 2035+(i*500);
        for(k=0; k<2; k++)     AllocColor(mycol, mNCol-(i*2+k)-1);
      }
      break;

    case CMAP_COLRJ128 :
      for( i=0; i<128; i++) {
        if (i < 48)  { 
          mycol.green = mycol.blue = 0;  
          mycol.red = 5375+(i*1280);  }
        else if (i < 96) { 
          mycol.blue = 0;   mycol.red = 65535;
          mycol.green = 5375+((i-48)*1280);  }
        else { 
          mycol.green = mycol.red = 65535;
          mycol.blue = 3535+(i-96)*2000; }
        for(k=0; k<2; k++)   AllocColor(mycol, i*2+k); 
        }
      break;

    case CMAP_COLBR128 :
      for( i=0; i<128; i++) {
        if (i < 20) { 
          mycol.green = mycol.red = 0;  // Du bleu sombre au bleu clair
          mycol.blue = 2835+i*3300; }
        else if (i < 36) { 
          mycol.blue = 65535;           // Plein bleu - On rajoute du vert -> bleu clair
          mycol.green = 3000+(i-20)*2000;
          mycol.red = 0;  }
        else if (i < 56) { 
          mycol.blue = 64600-(i-36)*3400;    // On passe au vert 
          mycol.green = 32300+(i-36)*1700;
          mycol.red = 0;  }
        else if (i < 81) {         
          mycol.blue = 0;              // Plein vert, on rajoute du rouge -> jaune
          mycol.green = 65535;
          mycol.red = 3135+(i-56)*2600;  }
        else if (i < 96) { 
          mycol.blue = 0;
          mycol.green = 63535-(i-80)*2000;
          mycol.red = 65535;  }
        else if (i < 112) { 
          mycol.blue = 0;
          mycol.green = 30000-(i-96)*2000;
          mycol.red = 65535-(i-96)*1000;  }
        else {
          mycol.blue = 0;
          mycol.green = 0 ;
          mycol.red = 50535+(i-112)*1000; }

        for(k=0; k<2; k++)   AllocColor(mycol, i*2+k); 
      }
      break;

  case CMAP_MIDAS_Pastel :   // pastel.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.01961, 0.05490, 0.08627, 0.10980, 0.13725, 0.15686, 
0.18039, 0.20000, 0.21569, 0.23529, 0.25098, 0.26275, 0.28235, 0.29412, 
0.30588, 0.31765, 0.33333, 0.34118, 0.35294, 0.36078, 0.37255, 0.38431, 
0.39216, 0.40000, 0.41176, 0.41961, 0.43137, 0.43529, 0.44314, 0.45098, 
0.45882, 0.46667, 0.47059, 0.48235, 0.48627, 0.49412, 0.50196, 0.50588, 
0.50980, 0.51765, 0.52157, 0.53333, 0.53725, 0.54118, 0.54902, 0.55294, 
0.55686, 0.56078, 0.56471, 0.57255, 0.58039, 0.58431, 0.58824, 0.59216, 
0.59608, 0.60000, 0.60392, 0.60784, 0.61176, 0.61569, 0.61961, 0.62745, 
0.63137, 0.63529, 0.63922, 0.64314, 0.64706, 0.65098, 0.65490, 0.65490, 
0.65882, 0.66275, 0.66667, 0.67059, 0.67843, 0.67843, 0.68235, 0.68627, 
0.69020, 0.69020, 0.69412, 0.69804, 0.70196, 0.70196, 0.70588, 0.70980, 
0.71373, 0.71373, 0.71765, 0.72157, 0.72157, 0.72941, 0.73333, 0.73333, 
0.73725, 0.74118, 0.74118, 0.74510, 0.74510, 0.74902, 0.75294, 0.75294, 
0.75686, 0.76078, 0.76078, 0.76471, 0.76471, 0.76863, 0.76863, 0.77255, 
0.78039, 0.78039, 0.78431, 0.78431, 0.78824, 0.78824, 0.79216, 0.79216, 
0.79608, 0.79608, 0.80000, 0.80000, 0.80392, 0.80392, 0.80784, 0.80784, 
0.81176, 0.81176, 0.81569, 0.81569, 0.81961, 0.81961, 0.82745, 0.82745, 
0.83137, 0.83137, 0.83529, 0.83529, 0.83922, 0.83922, 0.83922, 0.84314, 
0.84314, 0.84706, 0.84706, 0.85098, 0.85098, 0.85490, 0.85490, 0.85490, 
0.85882, 0.85882, 0.86275, 0.86275, 0.86275, 0.86667, 0.86667, 0.87059, 
0.87059, 0.87059, 0.87843, 0.87843, 0.88235, 0.88235, 0.88235, 0.88627, 
0.88627, 0.89020, 0.89020, 0.89020, 0.89412, 0.89412, 0.89412, 0.89804, 
0.89804, 0.89804, 0.90196, 0.90196, 0.90588, 0.90588, 0.90588, 0.90980, 
0.90980, 0.90980, 0.91373, 0.91373, 0.91373, 0.91765, 0.91765, 0.91765, 
0.92157, 0.92157, 0.92157, 0.92941, 0.92941, 0.92941, 0.93333, 0.93333, 
0.93333, 0.93725, 0.93725, 0.93725, 0.94118, 0.94118, 0.94118, 0.94118, 
0.94510, 0.94510, 0.94510, 0.94902, 0.94902, 0.94902, 0.95294, 0.95294, 
0.95294, 0.95686, 0.95686, 0.95686, 0.95686, 0.96078, 0.96078, 0.96078, 
0.96471, 0.96471, 0.96471, 0.96471, 0.96863, 0.96863, 0.96863, 0.97255, 
0.97255, 0.97255, 0.97255, 0.98039, 0.98039, 0.98039, 0.98039, 0.98431, 
0.98431, 0.98431, 0.98824, 0.98824, 0.98824, 0.98824, 0.99216, 0.99216, 
0.99216, 0.99216, 0.99608, 0.99608, 0.99608, 0.99608, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00784, 0.00784, 0.00784, 0.00784, 0.01176, 0.01176, 0.01176, 
0.01176, 0.01176, 0.01569, 0.01569, 0.01569, 0.01569, 0.01961, 0.01961, 
0.01961, 0.01961, 0.02353, 0.02353, 0.02353, 0.02745, 0.02745, 0.02745, 
0.02745, 0.03137, 0.03137, 0.03137, 0.03529, 0.03529, 0.03529, 0.03529, 
0.04314, 0.04314, 0.04314, 0.04706, 0.04706, 0.04706, 0.05098, 0.05098, 
0.05098, 0.05490, 0.05490, 0.05490, 0.05882, 0.05882, 0.05882, 0.06275, 
0.06275, 0.06275, 0.06667, 0.06667, 0.06667, 0.07059, 0.07059, 0.07451, 
0.07451, 0.07451, 0.07843, 0.07843, 0.08235, 0.08235, 0.08235, 0.08627, 
0.08627, 0.09020, 0.09020, 0.09020, 0.09412, 0.09412, 0.09804, 0.09804, 
0.10196, 0.10196, 0.10588, 0.10588, 0.10980, 0.10980, 0.10980, 0.11765, 
0.11765, 0.12157, 0.12157, 0.12549, 0.12549, 0.12941, 0.12941, 0.13333, 
0.13725, 0.13725, 0.14118, 0.14118, 0.14510, 0.14510, 0.14902, 0.14902, 
0.15294, 0.15686, 0.15686, 0.16078, 0.16078, 0.16471, 0.16863, 0.16863, 
0.17255, 0.17255, 0.17647, 0.18039, 0.18039, 0.18431, 0.18824, 0.18824, 
0.19608, 0.20000, 0.20000, 0.20392, 0.20784, 0.20784, 0.21176, 0.21569, 
0.21961, 0.21961, 0.22353, 0.22745, 0.23137, 0.23137, 0.23529, 0.23922, 
0.24314, 0.24314, 0.24706, 0.25098, 0.25490, 0.25882, 0.26275, 0.26275, 
0.27059, 0.27451, 0.27843, 0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 
0.29804, 0.30196, 0.30588, 0.30980, 0.31373, 0.31765, 0.32157, 0.32549, 
0.33333, 0.33725, 0.34118, 0.34510, 0.34902, 0.35294, 0.35686, 0.36078, 
0.36471, 0.37255, 0.37647, 0.38039, 0.38431, 0.38824, 0.39216, 0.40000, 
0.40392, 0.40784, 0.41176, 0.41961, 0.42353, 0.42745, 0.43529, 0.43922, 
0.44314, 0.45098, 0.45490, 0.45882, 0.46667, 0.47059, 0.47843, 0.48235, 
0.49020, 0.49412, 0.50196, 0.50588, 0.51373, 0.51765, 0.52549, 0.52941, 
0.53725, 0.54118, 0.54902, 0.55686, 0.56078, 0.56863, 0.57647, 0.58039, 
0.58824, 0.59608, 0.60000, 0.60784, 0.61569, 0.62353, 0.63137, 0.63529, 
0.64314, 0.65098, 0.65882, 0.66667, 0.67451, 0.68235, 0.69020, 0.69804, 
0.70588, 0.71373, 0.72157, 0.72941, 0.73725, 0.74510, 0.75294, 0.76078, 
0.77255, 0.78039, 0.78824, 0.79608, 0.80392, 0.81569, 0.82353, 0.83137, 
0.84314, 0.85098, 0.86275, 0.87059, 0.87843, 0.89020, 0.89804, 0.90980, 
0.91765, 0.92941, 0.94118, 0.94902, 0.96078, 0.97255, 0.98039, 0.99216
};
float rgb_b[256] = {
0.00000, 1.00000, 0.98039, 0.94510, 0.91373, 0.89020, 0.86275, 0.84314, 
0.81961, 0.80000, 0.78431, 0.76471, 0.74902, 0.73725, 0.71765, 0.70588, 
0.69412, 0.68235, 0.66667, 0.65882, 0.64706, 0.63922, 0.62745, 0.61569, 
0.60784, 0.60000, 0.58824, 0.58039, 0.56863, 0.56471, 0.55686, 0.54902, 
0.54118, 0.53333, 0.52941, 0.51765, 0.51373, 0.50588, 0.49804, 0.49412, 
0.49020, 0.48235, 0.47843, 0.46667, 0.46275, 0.45882, 0.45098, 0.44706, 
0.44314, 0.43922, 0.43529, 0.42745, 0.41961, 0.41569, 0.41176, 0.40784, 
0.40392, 0.40000, 0.39608, 0.39216, 0.38824, 0.38431, 0.38039, 0.37255, 
0.36863, 0.36471, 0.36078, 0.35686, 0.35294, 0.34902, 0.34510, 0.34510, 
0.34118, 0.33725, 0.33333, 0.32941, 0.32157, 0.32157, 0.31765, 0.31373, 
0.30980, 0.30980, 0.30588, 0.30196, 0.29804, 0.29804, 0.29412, 0.29020, 
0.28627, 0.28627, 0.28235, 0.27843, 0.27843, 0.27059, 0.26667, 0.26667, 
0.26275, 0.25882, 0.25882, 0.25490, 0.25490, 0.25098, 0.24706, 0.24706, 
0.24314, 0.23922, 0.23922, 0.23529, 0.23529, 0.23137, 0.23137, 0.22745, 
0.21961, 0.21961, 0.21569, 0.21569, 0.21176, 0.21176, 0.20784, 0.20784, 
0.20392, 0.20392, 0.20000, 0.20000, 0.19608, 0.19608, 0.19216, 0.19216, 
0.18824, 0.18824, 0.18431, 0.18431, 0.18039, 0.18039, 0.17255, 0.17255, 
0.16863, 0.16863, 0.16471, 0.16471, 0.16078, 0.16078, 0.16078, 0.15686, 
0.15686, 0.15294, 0.15294, 0.14902, 0.14902, 0.14510, 0.14510, 0.14510, 
0.14118, 0.14118, 0.13725, 0.13725, 0.13725, 0.13333, 0.13333, 0.12941, 
0.12941, 0.12941, 0.12157, 0.12157, 0.11765, 0.11765, 0.11765, 0.11373, 
0.11373, 0.10980, 0.10980, 0.10980, 0.10588, 0.10588, 0.10588, 0.10196, 
0.10196, 0.10196, 0.09804, 0.09804, 0.09412, 0.09412, 0.09412, 0.09020, 
0.09020, 0.09020, 0.08627, 0.08627, 0.08627, 0.08235, 0.08235, 0.08235, 
0.07843, 0.07843, 0.07843, 0.07059, 0.07059, 0.07059, 0.06667, 0.06667, 
0.06667, 0.06275, 0.06275, 0.06275, 0.05882, 0.05882, 0.05882, 0.05882, 
0.05490, 0.05490, 0.05490, 0.05098, 0.05098, 0.05098, 0.04706, 0.04706, 
0.04706, 0.04314, 0.04314, 0.04314, 0.04314, 0.03922, 0.03922, 0.03922, 
0.03529, 0.03529, 0.03529, 0.03529, 0.03137, 0.03137, 0.03137, 0.02745, 
0.02745, 0.02745, 0.02745, 0.01961, 0.01961, 0.01961, 0.01961, 0.01569, 
0.01569, 0.01569, 0.01176, 0.01176, 0.01176, 0.01176, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00392, 0.00392, 0.00392, 0.00392, 0.00000, 0.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Heat :      //  heat.lutlis 
   {
float rgb_r[256] = {
0.00000, 0.01176, 0.02353, 0.03529, 0.04706, 0.05882, 0.07059, 0.08235, 
0.09412, 0.10588, 0.11765, 0.12941, 0.14118, 0.15294, 0.16471, 0.17647, 
0.18824, 0.20000, 0.21176, 0.22353, 0.23529, 0.24706, 0.25882, 0.27059, 
0.28235, 0.29412, 0.30588, 0.31765, 0.32941, 0.34118, 0.35294, 0.36471, 
0.37647, 0.38824, 0.40000, 0.41176, 0.42353, 0.43529, 0.44706, 0.45882, 
0.47059, 0.48235, 0.49412, 0.50588, 0.51765, 0.52941, 0.54118, 0.55294, 
0.56471, 0.57647, 0.58824, 0.60000, 0.61176, 0.62353, 0.63529, 0.64706, 
0.65882, 0.67059, 0.68235, 0.69412, 0.70588, 0.71765, 0.72941, 0.74118, 
0.75294, 0.76471, 0.77647, 0.78824, 0.80000, 0.81176, 0.82353, 0.83529, 
0.84706, 0.85882, 0.87059, 0.88235, 0.89412, 0.90588, 0.91765, 0.92941, 
0.94118, 0.95294, 0.96471, 0.97647, 0.98824, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 
0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 0.05882, 
0.06275, 0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 0.09020, 
0.09412, 0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 0.12157, 
0.12549, 0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 0.15294, 
0.15686, 0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 0.18431, 
0.18824, 0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 0.21569, 
0.21961, 0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 0.24706, 
0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 
0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 
0.31373, 0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.33725, 0.34118, 
0.34510, 0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.36863, 0.37255, 
0.37647, 0.38039, 0.38431, 0.38824, 0.39216, 0.39608, 0.40000, 0.40392, 
0.40784, 0.41176, 0.41569, 0.41961, 0.42353, 0.42745, 0.43137, 0.43529, 
0.43922, 0.44314, 0.44706, 0.45098, 0.45490, 0.45882, 0.46275, 0.46667, 
0.47059, 0.47451, 0.47843, 0.48235, 0.48627, 0.49020, 0.49412, 0.49804, 
0.50196, 0.50588, 0.50980, 0.51373, 0.51765, 0.52157, 0.52549, 0.52941, 
0.53333, 0.53725, 0.54118, 0.54510, 0.54902, 0.55294, 0.55686, 0.56078, 
0.56471, 0.56863, 0.57255, 0.57647, 0.58039, 0.58431, 0.58824, 0.59216, 
0.59608, 0.60000, 0.60392, 0.60784, 0.61176, 0.61569, 0.61961, 0.62353, 
0.62745, 0.63137, 0.63529, 0.63922, 0.64314, 0.64706, 0.65098, 0.65490, 
0.65882, 0.66275, 0.66667, 0.67059, 0.67451, 0.67843, 0.68235, 0.68627, 
0.69020, 0.69412, 0.69804, 0.70196, 0.70588, 0.70980, 0.71373, 0.71765, 
0.72157, 0.72549, 0.72941, 0.73333, 0.73725, 0.74118, 0.74510, 0.74902, 
0.75294, 0.75686, 0.76078, 0.76471, 0.76863, 0.77255, 0.77647, 0.78039, 
0.78431, 0.78824, 0.79216, 0.79608, 0.80000, 0.80392, 0.80784, 0.81176, 
0.81569, 0.81961, 0.82353, 0.82745, 0.83137, 0.83529, 0.83922, 0.84314, 
0.84706, 0.85098, 0.85490, 0.85882, 0.86275, 0.86667, 0.87059, 0.87451, 
0.87843, 0.88235, 0.88627, 0.89020, 0.89412, 0.89804, 0.90196, 0.90588, 
0.90980, 0.91373, 0.91765, 0.92157, 0.92549, 0.92941, 0.93333, 0.93725, 
0.94118, 0.94510, 0.94902, 0.95294, 0.95686, 0.96078, 0.96471, 0.96863, 
0.97255, 0.97647, 0.98039, 0.98431, 0.98824, 0.99216, 0.99608, 1.00000
};
float rgb_b[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.01176, 0.02353, 
0.03529, 0.04706, 0.05882, 0.07059, 0.08235, 0.09412, 0.10588, 0.11765, 
0.12941, 0.14118, 0.15294, 0.16471, 0.17647, 0.18824, 0.20000, 0.21176, 
0.22353, 0.23529, 0.24706, 0.25882, 0.27059, 0.28235, 0.29412, 0.30588, 
0.31765, 0.32941, 0.34118, 0.35294, 0.36471, 0.37647, 0.38824, 0.40000, 
0.41176, 0.42353, 0.43529, 0.44706, 0.45882, 0.47059, 0.48235, 0.49412, 
0.50588, 0.51765, 0.52941, 0.54118, 0.55294, 0.56471, 0.57647, 0.58824, 
0.60000, 0.61176, 0.62353, 0.63529, 0.64706, 0.65882, 0.67059, 0.68235, 
0.69412, 0.70588, 0.71765, 0.72941, 0.74118, 0.75294, 0.76471, 0.77647, 
0.78824, 0.80000, 0.81176, 0.82353, 0.83529, 0.84706, 0.85882, 0.87059, 
0.88235, 0.89412, 0.90588, 0.91765, 0.92941, 0.94118, 0.95294, 0.96471, 
0.97647, 0.98824, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Rainbow3 :  //  rainbow3.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.04706, 0.09412, 0.14118, 0.18824, 0.23529, 0.28235, 0.32941, 
0.37647, 0.42353, 0.47059, 0.51765, 0.56471, 0.61176, 0.65882, 0.70588, 
0.72549, 0.74510, 0.76471, 0.78431, 0.80392, 0.82353, 0.84314, 0.86275, 
0.88235, 0.90196, 0.92157, 0.94118, 0.96078, 0.98039, 1.00000, 0.99608, 
0.99608, 0.99608, 0.99608, 0.99608, 0.99216, 0.99216, 0.99216, 0.99216, 
0.99216, 0.98824, 0.98824, 0.98824, 0.98824, 0.98824, 0.98824, 0.98824, 
0.98824, 0.98824, 0.99216, 0.99216, 0.99216, 0.99216, 0.99216, 0.99608, 
0.99608, 0.99608, 0.99608, 0.99608, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.03137, 0.06275, 
0.09412, 0.12549, 0.15686, 0.18824, 0.21961, 0.25490, 0.28627, 0.31765, 
0.34902, 0.38039, 0.41176, 0.44314, 0.47843, 0.49804, 0.51765, 0.53725, 
0.55686, 0.58039, 0.60000, 0.61961, 0.63922, 0.65882, 0.68235, 0.70196, 
0.72157, 0.74118, 0.76078, 0.78431, 0.79608, 0.81176, 0.82745, 0.83922, 
0.85490, 0.87059, 0.88235, 0.89804, 0.91373, 0.92549, 0.94118, 0.95686, 
0.96863, 0.98431, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.98039, 
0.96078, 0.94510, 0.92549, 0.90588, 0.89020, 0.87059, 0.85098, 0.83529, 
0.81569, 0.79608, 0.78039, 0.76078, 0.74118, 0.72549, 0.70588, 0.69020, 
0.67451, 0.65490, 0.63922, 0.62353, 0.60392, 0.58824, 0.57255, 0.55294, 
0.53725, 0.52157, 0.50196, 0.48627, 0.47059, 0.43922, 0.40784, 0.37647, 
0.34510, 0.31373, 0.28235, 0.25098, 0.21569, 0.18431, 0.15294, 0.12157, 
0.09020, 0.05882, 0.02745, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.03529, 
0.07059, 0.10588, 0.14118, 0.18039, 0.21569, 0.25098, 0.28627, 0.32549, 
0.36078, 0.39608, 0.43137, 0.47059, 0.48627, 0.50588, 0.52157, 0.54118, 
0.56078, 0.57647, 0.59608, 0.61176, 0.63137, 0.65098, 0.66667, 0.68627, 
0.70588, 0.74510, 0.78824, 0.83137, 0.87059, 0.91373, 0.95686, 1.00000
};
float rgb_b[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.03922, 0.07843, 0.11765, 
0.15686, 0.20000, 0.23922, 0.27843, 0.31765, 0.35686, 0.40000, 0.43922, 
0.47843, 0.51765, 0.55686, 0.60000, 0.63922, 0.67843, 0.71765, 0.75686, 
0.80000, 0.83922, 0.87843, 0.91765, 0.95686, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 0.98039, 0.96078, 0.94118, 0.92157, 0.90196, 
0.88235, 0.86275, 0.84314, 0.82353, 0.80392, 0.78431, 0.76471, 0.74510, 
0.72549, 0.70588, 0.65882, 0.61176, 0.56471, 0.51765, 0.47059, 0.42353, 
0.37647, 0.32549, 0.27843, 0.23137, 0.18431, 0.13725, 0.09020, 0.04314, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.04706, 0.09412, 0.14118, 0.18824, 0.23529, 
0.28235, 0.32941, 0.37647, 0.42353, 0.47059, 0.51765, 0.56471, 0.61176, 
0.65882, 0.70588, 0.72549, 0.74902, 0.77255, 0.79608, 0.81569, 0.83922, 
0.86275, 0.88627, 0.90588, 0.92941, 0.95294, 0.97647, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_BlueRed :   //   idl11.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.01569, 0.03137, 0.04706, 0.06275, 0.07843, 0.09412, 0.10980, 
0.12549, 0.14118, 0.15686, 0.17255, 0.18824, 0.20392, 0.21961, 0.23529, 
0.25098, 0.26667, 0.28235, 0.29804, 0.31373, 0.33333, 0.34902, 0.36471, 
0.38039, 0.39608, 0.41176, 0.42745, 0.44314, 0.45882, 0.47451, 0.49020, 
0.50588, 0.52157, 0.53725, 0.55294, 0.56863, 0.58431, 0.60000, 0.61569, 
0.63137, 0.64706, 0.66667, 0.68235, 0.69804, 0.71373, 0.72941, 0.74510, 
0.76078, 0.77647, 0.79216, 0.80784, 0.82353, 0.83922, 0.85490, 0.87059, 
0.88627, 0.90196, 0.91765, 0.93333, 0.94902, 0.96471, 0.98039, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.03137, 0.04706, 0.06275, 
0.08235, 0.09804, 0.11373, 0.12941, 0.14902, 0.16471, 0.18039, 0.19608, 
0.21569, 0.23137, 0.24706, 0.26275, 0.28235, 0.29804, 0.31373, 0.32941, 
0.34902, 0.36471, 0.38039, 0.39608, 0.41569, 0.43137, 0.44706, 0.46275, 
0.48235, 0.49804, 0.51373, 0.52941, 0.54902, 0.56471, 0.58039, 0.59608, 
0.61569, 0.63137, 0.64706, 0.66275, 0.68235, 0.69804, 0.71373, 0.72941, 
0.74902, 0.76471, 0.78039, 0.79608, 0.81569, 0.83137, 0.84706, 0.86275, 
0.88235, 0.89804, 0.91373, 0.92941, 0.94902, 0.96471, 0.98039, 1.00000, 
1.00000, 0.98431, 0.96863, 0.95294, 0.93725, 0.92157, 0.90588, 0.89020, 
0.87451, 0.85882, 0.84314, 0.82745, 0.81176, 0.79608, 0.78039, 0.76471, 
0.74902, 0.73333, 0.71765, 0.70196, 0.68627, 0.66667, 0.65098, 0.63529, 
0.61961, 0.60392, 0.58824, 0.57255, 0.55686, 0.54118, 0.52549, 0.50980, 
0.49412, 0.47843, 0.46275, 0.44706, 0.43137, 0.41569, 0.40000, 0.38431, 
0.36863, 0.35294, 0.33333, 0.31765, 0.30196, 0.28627, 0.27059, 0.25490, 
0.23922, 0.22353, 0.20784, 0.19216, 0.17647, 0.16078, 0.14510, 0.12941, 
0.11373, 0.09804, 0.08235, 0.06667, 0.05098, 0.03529, 0.01961, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000
};
float rgb_b[256] = {
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.03137, 0.04706, 0.06275, 
0.08235, 0.09804, 0.11373, 0.12941, 0.14902, 0.16471, 0.18039, 0.19608, 
0.21569, 0.23137, 0.24706, 0.26275, 0.28235, 0.29804, 0.31373, 0.32941, 
0.34902, 0.36471, 0.38039, 0.39608, 0.41569, 0.43137, 0.44706, 0.46275, 
0.48235, 0.49804, 0.51373, 0.52941, 0.54902, 0.56471, 0.58039, 0.59608, 
0.61569, 0.63137, 0.64706, 0.66275, 0.68235, 0.69804, 0.71373, 0.72941, 
0.74902, 0.76471, 0.78039, 0.79608, 0.81569, 0.83137, 0.84706, 0.86275, 
0.88235, 0.89804, 0.91373, 0.92941, 0.94902, 0.96471, 0.98039, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 0.98431, 0.96863, 0.95294, 0.93725, 0.92157, 0.90588, 0.89020, 
0.87451, 0.85490, 0.83922, 0.82353, 0.80784, 0.79216, 0.77647, 0.76078, 
0.74510, 0.72941, 0.70980, 0.69412, 0.67843, 0.66275, 0.64706, 0.63137, 
0.61569, 0.60000, 0.58431, 0.56471, 0.54902, 0.53333, 0.51765, 0.50196, 
0.48627, 0.47059, 0.45490, 0.43922, 0.41961, 0.40392, 0.38824, 0.37255, 
0.35686, 0.34118, 0.32549, 0.30980, 0.29412, 0.27451, 0.25882, 0.24314, 
0.22745, 0.21176, 0.19608, 0.18039, 0.16471, 0.14902, 0.12941, 0.11373, 
0.09804, 0.08235, 0.06667, 0.05098, 0.03529, 0.01961, 0.00000, 0.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_BlueWhite : //    blulut.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00001, 0.00001, 0.00001, 
0.00002, 0.00002, 0.00002, 0.00003, 0.00004, 0.00005, 0.00006, 0.00007, 
0.00008, 0.00009, 0.00011, 0.00013, 0.00015, 0.00017, 0.00019, 0.00022, 
0.00025, 0.00028, 0.00032, 0.00035, 0.00040, 0.00044, 0.00049, 0.00055, 
0.00061, 0.00067, 0.00074, 0.00081, 0.00089, 0.00097, 0.00106, 0.00115, 
0.00126, 0.00136, 0.00148, 0.00160, 0.00173, 0.00187, 0.00201, 0.00216, 
0.00233, 0.00250, 0.00268, 0.00287, 0.00307, 0.00327, 0.00349, 0.00373, 
0.00397, 0.00422, 0.00449, 0.00477, 0.00506, 0.00536, 0.00568, 0.00601, 
0.00636, 0.00672, 0.00709, 0.00748, 0.00789, 0.00831, 0.00875, 0.00921, 
0.00969, 0.01018, 0.01069, 0.01122, 0.01177, 0.01235, 0.01294, 0.01355, 
0.01418, 0.01484, 0.01552, 0.01622, 0.01694, 0.01769, 0.01847, 0.01926, 
0.02009, 0.02094, 0.02181, 0.02272, 0.02365, 0.02461, 0.02560, 0.02662, 
0.02767, 0.02875, 0.02986, 0.03100, 0.03218, 0.03338, 0.03463, 0.03590, 
0.03721, 0.03856, 0.03994, 0.04136, 0.04282, 0.04432, 0.04585, 0.04743, 
0.04904, 0.05070, 0.05239, 0.05413, 0.05591, 0.05774, 0.05961, 0.06153, 
0.06349, 0.06549, 0.06755, 0.06965, 0.07180, 0.07400, 0.07625, 0.07856, 
0.08091, 0.08331, 0.08577, 0.08829, 0.09086, 0.09348, 0.09616, 0.09890, 
0.10169, 0.10455, 0.10746, 0.11044, 0.11347, 0.11657, 0.11973, 0.12296, 
0.12624, 0.12960, 0.13302, 0.13651, 0.14007, 0.14369, 0.14739, 0.15116, 
0.15500, 0.15891, 0.16289, 0.16695, 0.17109, 0.17530, 0.17959, 0.18395, 
0.18840, 0.19292, 0.19753, 0.20222, 0.20699, 0.21185, 0.21679, 0.22182, 
0.22693, 0.23213, 0.23742, 0.24280, 0.24827, 0.25384, 0.25949, 0.26524, 
0.27109, 0.27703, 0.28307, 0.28920, 0.29544, 0.30178, 0.30821, 0.31476, 
0.32140, 0.32815, 0.33500, 0.34196, 0.34903, 0.35621, 0.36350, 0.37090, 
0.37841, 0.38603, 0.39377, 0.40163, 0.40960, 0.41769, 0.42590, 0.43423, 
0.44268, 0.45126, 0.45996, 0.46878, 0.47773, 0.48681, 0.49601, 0.50535, 
0.51482, 0.52442, 0.53415, 0.54402, 0.55403, 0.56417, 0.57445, 0.58487, 
0.59543, 0.60613, 0.61698, 0.62798, 0.63911, 0.65040, 0.66184, 0.67342, 
0.68516, 0.69705, 0.70909, 0.72129, 0.73365, 0.74616, 0.75883, 0.77167, 
0.78466, 0.79782, 0.81115, 0.82464, 0.83830, 0.85213, 0.86612, 0.88029, 
0.89464, 0.90915, 0.92385, 0.93872, 0.95377, 0.96899, 0.98441, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00001, 0.00001, 0.00001, 
0.00002, 0.00002, 0.00002, 0.00003, 0.00004, 0.00005, 0.00006, 0.00007, 
0.00008, 0.00009, 0.00011, 0.00013, 0.00015, 0.00017, 0.00019, 0.00022, 
0.00025, 0.00028, 0.00032, 0.00035, 0.00040, 0.00044, 0.00049, 0.00055, 
0.00061, 0.00067, 0.00074, 0.00081, 0.00089, 0.00097, 0.00106, 0.00115, 
0.00126, 0.00136, 0.00148, 0.00160, 0.00173, 0.00187, 0.00201, 0.00216, 
0.00233, 0.00250, 0.00268, 0.00287, 0.00307, 0.00327, 0.00349, 0.00373, 
0.00397, 0.00422, 0.00449, 0.00477, 0.00506, 0.00536, 0.00568, 0.00601, 
0.00636, 0.00672, 0.00709, 0.00748, 0.00789, 0.00831, 0.00875, 0.00921, 
0.00969, 0.01018, 0.01069, 0.01122, 0.01177, 0.01235, 0.01294, 0.01355, 
0.01418, 0.01484, 0.01552, 0.01622, 0.01694, 0.01769, 0.01847, 0.01926, 
0.02009, 0.02094, 0.02181, 0.02272, 0.02365, 0.02461, 0.02560, 0.02662, 
0.02767, 0.02875, 0.02986, 0.03100, 0.03218, 0.03338, 0.03463, 0.03590, 
0.03721, 0.03856, 0.03994, 0.04136, 0.04282, 0.04432, 0.04585, 0.04743, 
0.04904, 0.05070, 0.05239, 0.05413, 0.05591, 0.05774, 0.05961, 0.06153, 
0.06349, 0.06549, 0.06755, 0.06965, 0.07180, 0.07400, 0.07625, 0.07856, 
0.08091, 0.08331, 0.08577, 0.08829, 0.09086, 0.09348, 0.09616, 0.09890, 
0.10169, 0.10455, 0.10746, 0.11044, 0.11347, 0.11657, 0.11973, 0.12296, 
0.12624, 0.12960, 0.13302, 0.13651, 0.14007, 0.14369, 0.14739, 0.15116, 
0.15500, 0.15891, 0.16289, 0.16695, 0.17109, 0.17530, 0.17959, 0.18395, 
0.18840, 0.19292, 0.19753, 0.20222, 0.20699, 0.21185, 0.21679, 0.22182, 
0.22693, 0.23213, 0.23742, 0.24280, 0.24827, 0.25384, 0.25949, 0.26524, 
0.27109, 0.27703, 0.28307, 0.28920, 0.29544, 0.30178, 0.30821, 0.31476, 
0.32140, 0.32815, 0.33500, 0.34196, 0.34903, 0.35621, 0.36350, 0.37090, 
0.37841, 0.38603, 0.39377, 0.40163, 0.40960, 0.41769, 0.42590, 0.43423, 
0.44268, 0.45126, 0.45996, 0.46878, 0.47773, 0.48681, 0.49601, 0.50535, 
0.51482, 0.52442, 0.53415, 0.54402, 0.55403, 0.56417, 0.57445, 0.58487, 
0.59543, 0.60613, 0.61698, 0.62798, 0.63911, 0.65040, 0.66184, 0.67342, 
0.68516, 0.69705, 0.70909, 0.72129, 0.73365, 0.74616, 0.75883, 0.77167, 
0.78466, 0.79782, 0.81115, 0.82464, 0.83830, 0.85213, 0.86612, 0.88029, 
0.89464, 0.90915, 0.92385, 0.93872, 0.95377, 0.96899, 0.98441, 1.00000
};
float rgb_b[256] = {
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 
0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 0.05882, 
0.06275, 0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 0.09020, 
0.09412, 0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 0.12157, 
0.12549, 0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 0.15294, 
0.15686, 0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 0.18431, 
0.18824, 0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 0.21569, 
0.21961, 0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 0.24706, 
0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 
0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 
0.31373, 0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.33725, 0.34118, 
0.34510, 0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.36863, 0.37255, 
0.37647, 0.38039, 0.38431, 0.38824, 0.39216, 0.39608, 0.40000, 0.40392, 
0.40784, 0.41176, 0.41569, 0.41961, 0.42353, 0.42745, 0.43137, 0.43529, 
0.43922, 0.44314, 0.44706, 0.45098, 0.45490, 0.45882, 0.46275, 0.46667, 
0.47059, 0.47451, 0.47843, 0.48235, 0.48627, 0.49020, 0.49412, 0.49804, 
0.50196, 0.50588, 0.50980, 0.51373, 0.51765, 0.52157, 0.52549, 0.52941, 
0.53333, 0.53725, 0.54118, 0.54510, 0.54902, 0.55294, 0.55686, 0.56078, 
0.56471, 0.56863, 0.57255, 0.57647, 0.58039, 0.58431, 0.58824, 0.59216, 
0.59608, 0.60000, 0.60392, 0.60784, 0.61176, 0.61569, 0.61961, 0.62353, 
0.62745, 0.63137, 0.63529, 0.63922, 0.64314, 0.64706, 0.65098, 0.65490, 
0.65882, 0.66275, 0.66667, 0.67059, 0.67451, 0.67843, 0.68235, 0.68627, 
0.69020, 0.69412, 0.69804, 0.70196, 0.70588, 0.70980, 0.71373, 0.71765, 
0.72157, 0.72549, 0.72941, 0.73333, 0.73725, 0.74118, 0.74510, 0.74902, 
0.75294, 0.75686, 0.76078, 0.76471, 0.76863, 0.77255, 0.77647, 0.78039, 
0.78431, 0.78824, 0.79216, 0.79608, 0.80000, 0.80392, 0.80784, 0.81176, 
0.81569, 0.81961, 0.82353, 0.82745, 0.83137, 0.83529, 0.83922, 0.84314, 
0.84706, 0.85098, 0.85490, 0.85882, 0.86275, 0.86667, 0.87059, 0.87451, 
0.87843, 0.88235, 0.88627, 0.89020, 0.89412, 0.89804, 0.90196, 0.90588, 
0.90980, 0.91373, 0.91765, 0.92157, 0.92549, 0.92941, 0.93333, 0.93725, 
0.94118, 0.94510, 0.94902, 0.95294, 0.95686, 0.96078, 0.96471, 0.96863, 
0.97255, 0.97647, 0.98039, 0.98431, 0.98824, 0.99216, 0.99608, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Stairs8 :  //  stairs8.lutlis
   {
float rgb_r[256] = {
0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 
0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 
0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 
0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 0.76471, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_b[256] = {
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Stairs9 :  //    stairs9.lutlis
   {
float rgb_r[256] = {
0.00000, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 
0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 
0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 0.60784, 
0.60784, 0.60784, 0.60784, 0.60784, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 
0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 
0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 
0.92549, 0.92549, 0.92549, 0.92549, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000
};
float rgb_g[256] = {
0.00000, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 
0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 
0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 0.65490, 
0.65490, 0.65490, 0.65490, 0.65490, 0.56863, 0.56863, 0.56863, 0.56863, 
0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 
0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 
0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 0.56863, 
0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 
0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 
0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 0.96471, 
0.96471, 0.96471, 0.96471, 0.96471, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 
0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 
0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 0.69412, 
0.69412, 0.69412, 0.69412, 0.69412, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000
};
float rgb_b[256] = {
0.00000, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 0.19608, 
0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 
0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 
0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 0.47451, 
0.47451, 0.47451, 0.47451, 0.47451, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_StairCase :  //    staircase.lutlis
   {
float rgb_r[256] = {
0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 0.03137, 
0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 0.05882, 0.06275, 
0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 0.09020, 0.09412, 
0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 0.12157, 0.12549, 
0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 0.15294, 0.15686, 
0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 0.18431, 0.18824, 
0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 0.21569, 0.21961, 
0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 0.24706, 0.25098, 
0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 0.28235, 
0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 0.31373, 
0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.00392, 0.00784, 0.01176, 
0.01569, 0.01961, 0.02353, 0.02745, 0.03137, 0.03529, 0.03922, 0.04314, 
0.04706, 0.05098, 0.05490, 0.05882, 0.06275, 0.06667, 0.07059, 0.07451, 
0.07843, 0.08235, 0.08627, 0.09020, 0.09412, 0.09804, 0.10196, 0.10588, 
0.10980, 0.11373, 0.11765, 0.12157, 0.12549, 0.12941, 0.13333, 0.13725, 
0.14118, 0.14510, 0.14902, 0.15294, 0.15686, 0.16078, 0.16471, 0.16863, 
0.17255, 0.17647, 0.18039, 0.18431, 0.18824, 0.19216, 0.19608, 0.20000, 
0.20392, 0.20784, 0.21176, 0.21569, 0.21961, 0.22353, 0.22745, 0.23137, 
0.23529, 0.23922, 0.24314, 0.24706, 0.25098, 0.25490, 0.25882, 0.26275, 
0.26667, 0.27059, 0.27451, 0.27843, 0.28235, 0.28627, 0.29020, 0.29412, 
0.29804, 0.30196, 0.30588, 0.30980, 0.31373, 0.31765, 0.32157, 0.32549, 
0.32941, 0.33333, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 
0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 
0.31373, 0.31373, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 
0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 
0.47059, 0.47059, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 
0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 
0.62745, 0.62745, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 
0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 
0.94118, 0.94118, 0.94902, 0.96078, 0.97255, 0.98431, 0.99216, 1.00000
};
float rgb_g[256] = {
0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 0.03137, 
0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 0.05882, 0.06275, 
0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 0.09020, 0.09412, 
0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 0.12157, 0.12549, 
0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 0.15294, 0.15686, 
0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 0.18431, 0.18824, 
0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 0.21569, 0.21961, 
0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 0.24706, 0.25098, 
0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 0.28235, 
0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 0.31373, 
0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.31373, 0.31373, 0.31373, 
0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 
0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.47059, 0.47059, 0.47059, 
0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 
0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.62745, 0.62745, 0.62745, 
0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 
0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.94118, 0.94118, 0.94118, 
0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 
0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.95294, 0.96471, 0.97647, 
0.98824, 1.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 
0.02745, 0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 
0.05882, 0.06275, 0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 
0.09020, 0.09412, 0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 
0.12157, 0.12549, 0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 
0.15294, 0.15686, 0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 
0.18431, 0.18824, 0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 
0.21569, 0.21961, 0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 
0.24706, 0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 
0.27843, 0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 
0.30980, 0.31373, 0.39216, 0.52941, 0.66667, 0.80392, 0.80000, 1.00000
};
float rgb_b[256] = {
0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 
0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 0.31373, 
0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 
0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 0.47059, 
0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 
0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 0.62745, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 
0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 0.94118, 
0.95294, 0.96471, 0.97647, 0.98824, 1.00000, 0.00392, 0.00784, 0.01176, 
0.01569, 0.01961, 0.02353, 0.02745, 0.03137, 0.03529, 0.03922, 0.04314, 
0.04706, 0.05098, 0.05490, 0.05882, 0.06275, 0.06667, 0.07059, 0.07451, 
0.07843, 0.08235, 0.08627, 0.09020, 0.09412, 0.09804, 0.10196, 0.10588, 
0.10980, 0.11373, 0.11765, 0.12157, 0.12549, 0.12941, 0.13333, 0.13725, 
0.14118, 0.14510, 0.14902, 0.15294, 0.15686, 0.16078, 0.16471, 0.16863, 
0.17255, 0.17647, 0.18039, 0.18431, 0.18824, 0.19216, 0.19608, 0.20000, 
0.20392, 0.20784, 0.21176, 0.21569, 0.21961, 0.22353, 0.22745, 0.23137, 
0.23529, 0.23922, 0.24314, 0.24706, 0.25098, 0.25490, 0.25882, 0.26275, 
0.26667, 0.27059, 0.27451, 0.27843, 0.28235, 0.28627, 0.29020, 0.29412, 
0.29804, 0.30196, 0.30588, 0.30980, 0.31373, 0.31765, 0.32157, 0.32549, 
0.32941, 0.33333, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 
0.02745, 0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 
0.05882, 0.06275, 0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 
0.09020, 0.09412, 0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 
0.12157, 0.12549, 0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 
0.15294, 0.15686, 0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 
0.18431, 0.18824, 0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 
0.21569, 0.21961, 0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 
0.24706, 0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 
0.27843, 0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 
0.30980, 0.31373, 0.39216, 0.52941, 0.66667, 0.80392, 0.80000, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Color :  //   color.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 
0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902
};
float rgb_g[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 
0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 0.62353, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000
};
float rgb_b[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 0.18431, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 0.37255, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 0.56078, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 0.93725, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 0.74902, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 0.49804, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 
0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980, 0.30980
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_ManyCol :  //  manycol.lutlis
   {
float rgb_r[256] = {
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 
0.44706, 0.44706, 0.44706, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 
0.72549, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.44706, 0.44706, 0.44706, 
0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 
0.72549, 0.72549, 0.72549, 0.72549, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 0.44706, 
0.44706, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.72549, 0.72549
};
float rgb_g[256] = {
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.69020, 0.69020, 0.69020, 0.69020, 
0.69020, 0.69020, 0.69020, 0.69020, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.69020, 0.69020, 0.69020, 
0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.88235, 
0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.78431, 0.78431, 0.78431, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.69020, 
0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 
0.69020, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 
0.88235, 0.88235, 0.88235, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 0.78431, 
0.78431, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 
0.69020, 0.69020, 0.69020, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.69020, 0.69020, 0.69020, 0.69020, 
0.69020, 0.69020, 0.69020, 0.69020, 0.69020, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.88235, 0.88235, 0.88235, 
0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.88235, 0.00000, 0.00000
};
float rgb_b[256] = {
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 
0.92549, 0.92549, 0.92549, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 
0.72549, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.92549, 0.92549, 0.92549, 
0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.72549, 0.72549, 0.72549, 0.72549, 0.72549, 
0.72549, 0.72549, 0.72549, 0.72549, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 0.34902, 0.34902, 
0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 0.34902, 
0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 0.92549, 
0.92549, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.72549, 0.72549
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Idl14 :  //  idl14.lutlis
   {
float rgb_r[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.02745, 0.05882, 0.09020, 0.12157, 0.15294, 0.18431, 0.21569, 
0.24706, 0.27451, 0.30588, 0.33725, 0.36863, 0.40000, 0.43137, 0.46275, 
0.49412, 0.52549, 0.55686, 0.59216, 0.62353, 0.65882, 0.69020, 0.72157, 
0.75686, 0.78824, 0.82353, 0.85490, 0.88627, 0.92157, 0.95294, 0.98824, 
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 
0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05490, 0.06275, 0.07059, 
0.07843, 0.09020, 0.09804, 0.10588, 0.11373, 0.12549, 0.13333, 0.14118, 
0.14902, 0.16078, 0.17255, 0.18431, 0.19608, 0.20784, 0.21961, 0.23137, 
0.24706, 0.25882, 0.27059, 0.28235, 0.29412, 0.30588, 0.32157, 0.33333, 
0.34902, 0.36471, 0.38039, 0.39608, 0.41176, 0.42353, 0.43922, 0.45490, 
0.47059, 0.48627, 0.50196, 0.51373, 0.52941, 0.54510, 0.56078, 0.57647, 
0.59216, 0.60784, 0.62353, 0.63922, 0.65490, 0.67059, 0.68627, 0.69804, 
0.70980, 0.72549, 0.73725, 0.75294, 0.76471, 0.77647, 0.79216, 0.80392, 
0.81961, 0.83137, 0.84706, 0.85490, 0.86275, 0.87451, 0.88235, 0.89020, 
0.90196, 0.90980, 0.91765, 0.92941, 0.93725, 0.94510, 0.95686, 0.95686, 
0.96078, 0.96471, 0.96863, 0.96863, 0.97255, 0.97647, 0.98039, 0.98039, 
0.98431, 0.98824, 0.99216, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 
0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 
0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 0.99608, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 
1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000
};
float rgb_g[256] = {
0.00000, 0.16471, 0.33333, 0.49804, 0.66667, 0.83137, 1.00000, 0.96471, 
0.92549, 0.88627, 0.84706, 0.80784, 0.77255, 0.73333, 0.69412, 0.65490, 
0.61569, 0.58039, 0.54118, 0.50196, 0.46275, 0.42353, 0.38824, 0.34902, 
0.30980, 0.27059, 0.23137, 0.19608, 0.15686, 0.11765, 0.07843, 0.03922, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00392, 0.00392, 0.01176, 0.01961, 0.03137, 
0.03922, 0.04706, 0.05882, 0.06667, 0.07451, 0.08627, 0.09412, 0.10196, 
0.11373, 0.12157, 0.12941, 0.14118, 0.14118, 0.14902, 0.15686, 0.16471, 
0.17647, 0.18824, 0.20000, 0.21176, 0.22745, 0.23922, 0.25098, 0.26275, 
0.27451, 0.28627, 0.30196, 0.32157, 0.34118, 0.36078, 0.37647, 0.39216, 
0.40784, 0.42353, 0.44314, 0.46667, 0.49020, 0.51373, 0.54118, 0.56471, 
0.59216, 0.61569, 0.64314, 0.67059, 0.69804, 0.72549, 0.75686, 0.78431, 
0.81569, 0.84314, 0.87451, 0.89804, 0.92157, 0.94902, 0.97255, 1.00000
};
float rgb_b[256] = {
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.03137, 0.06275, 0.09412, 0.12549, 0.16078, 0.19216, 0.22353, 
0.25490, 0.29020, 0.32157, 0.35294, 0.38431, 0.41569, 0.45098, 0.48235, 
0.51373, 0.54510, 0.58039, 0.61176, 0.64314, 0.67451, 0.70588, 0.74118, 
0.77255, 0.80392, 0.83529, 0.87059, 0.90196, 0.93333, 0.96471, 1.00000, 
0.00000, 0.01961, 0.03922, 0.05882, 0.08235, 0.10196, 0.12157, 0.14510, 
0.16471, 0.18431, 0.20784, 0.22745, 0.24706, 0.27059, 0.29020, 0.30980, 
0.33333, 0.34902, 0.36863, 0.38431, 0.40392, 0.42353, 0.43922, 0.45882, 
0.47451, 0.49412, 0.51373, 0.52941, 0.54902, 0.56471, 0.58431, 0.60392, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 
0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 0.00392, 
0.00392, 0.00392, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 
0.00784, 0.00784, 0.00784, 0.00784, 0.00784, 0.01176, 0.01569, 0.01961, 
0.02745, 0.03529, 0.04706, 0.05490, 0.06667, 0.07843, 0.09020, 0.10588, 
0.11765, 0.13333, 0.15294, 0.17255, 0.19216, 0.21569, 0.23529, 0.25490, 
0.27843, 0.29804, 0.32157, 0.34902, 0.38039, 0.40784, 0.43922, 0.47059, 
0.50196, 0.53333, 0.56863, 0.60000, 0.63529, 0.67059, 0.70588, 0.74118, 
0.77647, 0.81176, 0.85098, 0.87843, 0.90980, 0.93725, 0.96863, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;
  case CMAP_MIDAS_Idl15 :  //  idl15.lutlis
   {
float rgb_r[256] = {
0.00000, 0.07059, 0.14118, 0.21176, 0.28235, 0.35294, 0.42353, 0.49804, 
0.56863, 0.63922, 0.70980, 0.78039, 0.85098, 0.92157, 0.99608, 0.97647, 
0.95686, 0.93725, 0.91765, 0.89804, 0.87451, 0.85490, 0.83529, 0.81569, 
0.79608, 0.77255, 0.75294, 0.73333, 0.71373, 0.69412, 0.67451, 0.65098, 
0.63137, 0.61176, 0.59216, 0.57255, 0.54902, 0.52941, 0.50980, 0.49020, 
0.47059, 0.45098, 0.42745, 0.40784, 0.38824, 0.36863, 0.34902, 0.32549, 
0.30588, 0.28627, 0.26667, 0.24706, 0.22745, 0.20392, 0.18431, 0.16471, 
0.14510, 0.12549, 0.10196, 0.08235, 0.06275, 0.04314, 0.02353, 0.00000, 
0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 
0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 
0.31373, 0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.33725, 0.34118, 
0.34510, 0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.36863, 0.37255, 
0.37647, 0.38039, 0.38431, 0.38824, 0.39216, 0.39608, 0.40000, 0.40392, 
0.40784, 0.41176, 0.41569, 0.41961, 0.42353, 0.42745, 0.43137, 0.43529, 
0.43922, 0.44314, 0.44706, 0.45098, 0.45490, 0.45882, 0.46275, 0.46667, 
0.47059, 0.47451, 0.47843, 0.48235, 0.48627, 0.49020, 0.49412, 0.49804, 
0.50196, 0.50588, 0.50980, 0.51373, 0.51765, 0.52157, 0.52549, 0.52941, 
0.53333, 0.53725, 0.54118, 0.54510, 0.54902, 0.55294, 0.55686, 0.56078, 
0.56471, 0.56863, 0.57255, 0.57647, 0.58039, 0.58431, 0.58824, 0.59216, 
0.59608, 0.60000, 0.60392, 0.60784, 0.61176, 0.61569, 0.61961, 0.62353, 
0.62745, 0.63137, 0.63529, 0.63922, 0.64314, 0.64706, 0.65098, 0.65490, 
0.65882, 0.66275, 0.66667, 0.67059, 0.67451, 0.67843, 0.68235, 0.68627, 
0.69020, 0.69412, 0.69804, 0.70196, 0.70588, 0.70980, 0.71373, 0.71765, 
0.72157, 0.72549, 0.72941, 0.73333, 0.73725, 0.74118, 0.74510, 0.74902, 
0.75294, 0.75686, 0.76078, 0.76471, 0.76863, 0.77255, 0.77647, 0.78039, 
0.78431, 0.78824, 0.79216, 0.79608, 0.80000, 0.80392, 0.80784, 0.81176, 
0.81569, 0.81961, 0.82353, 0.82745, 0.83137, 0.83529, 0.83922, 0.84314, 
0.84706, 0.85098, 0.85490, 0.85882, 0.86275, 0.86667, 0.87059, 0.87451, 
0.87843, 0.88235, 0.88627, 0.89020, 0.89412, 0.89804, 0.90196, 0.90588, 
0.90980, 0.91373, 0.91765, 0.92157, 0.92549, 0.92941, 0.93333, 0.93725, 
0.94118, 0.94510, 0.94902, 0.95294, 0.95686, 0.96078, 0.96471, 0.96863, 
0.97255, 0.97647, 0.98039, 0.98431, 0.98824, 0.99216, 0.99608, 1.00000
};
float rgb_g[256] = {
0.00000, 0.00392, 0.00784, 0.01176, 0.01569, 0.01961, 0.02353, 0.02745, 
0.03137, 0.03529, 0.03922, 0.04314, 0.04706, 0.05098, 0.05490, 0.05882, 
0.06275, 0.06667, 0.07059, 0.07451, 0.07843, 0.08235, 0.08627, 0.09020, 
0.09412, 0.09804, 0.10196, 0.10588, 0.10980, 0.11373, 0.11765, 0.12157, 
0.12549, 0.12941, 0.13333, 0.13725, 0.14118, 0.14510, 0.14902, 0.15294, 
0.15686, 0.16078, 0.16471, 0.16863, 0.17255, 0.17647, 0.18039, 0.18431, 
0.18824, 0.19216, 0.19608, 0.20000, 0.20392, 0.20784, 0.21176, 0.21569, 
0.21961, 0.22353, 0.22745, 0.23137, 0.23529, 0.23922, 0.24314, 0.24706, 
0.25098, 0.25490, 0.25882, 0.26275, 0.26667, 0.27059, 0.27451, 0.27843, 
0.28235, 0.28627, 0.29020, 0.29412, 0.29804, 0.30196, 0.30588, 0.30980, 
0.31373, 0.31765, 0.32157, 0.32549, 0.32941, 0.33333, 0.33725, 0.34118, 
0.34510, 0.34902, 0.35294, 0.35686, 0.36078, 0.36471, 0.36863, 0.37255, 
0.37647, 0.38039, 0.38431, 0.38824, 0.39216, 0.39608, 0.40000, 0.40392, 
0.40784, 0.41176, 0.41569, 0.41961, 0.42353, 0.42745, 0.43137, 0.43529, 
0.43922, 0.44314, 0.44706, 0.45098, 0.45490, 0.45882, 0.46275, 0.46667, 
0.47059, 0.47451, 0.47843, 0.48235, 0.48627, 0.49020, 0.49412, 0.49804, 
0.50196, 0.50588, 0.50980, 0.51373, 0.51765, 0.52157, 0.52549, 0.52941, 
0.53333, 0.53725, 0.54118, 0.54510, 0.54902, 0.55294, 0.55686, 0.56078, 
0.56471, 0.56863, 0.57255, 0.57647, 0.58039, 0.58431, 0.58824, 0.59216, 
0.59608, 0.60000, 0.60392, 0.60784, 0.61176, 0.61569, 0.61961, 0.62353, 
0.62745, 0.63137, 0.63529, 0.63922, 0.64314, 0.64706, 0.65098, 0.65490, 
0.65882, 0.66275, 0.66667, 0.67059, 0.67451, 0.67843, 0.68235, 0.68627, 
0.69020, 0.69412, 0.69804, 0.70196, 0.70588, 0.70980, 0.71373, 0.71765, 
0.72157, 0.72549, 0.72941, 0.73333, 0.73725, 0.74118, 0.74510, 0.74902, 
0.75294, 0.75686, 0.76078, 0.76471, 0.76863, 0.77255, 0.77647, 0.78039, 
0.78431, 0.78824, 0.79216, 0.79608, 0.80000, 0.80392, 0.80784, 0.81176, 
0.81569, 0.81961, 0.82353, 0.82745, 0.83137, 0.83529, 0.83922, 0.84314, 
0.84706, 0.85098, 0.85490, 0.85882, 0.86275, 0.86667, 0.87059, 0.87451, 
0.87843, 0.88235, 0.88627, 0.89020, 0.89412, 0.89804, 0.90196, 0.90588, 
0.90980, 0.91373, 0.91765, 0.92157, 0.92549, 0.92941, 0.93333, 0.93725, 
0.94118, 0.94510, 0.94902, 0.95294, 0.95686, 0.96078, 0.96471, 0.96863, 
0.97255, 0.97647, 0.98039, 0.98431, 0.98824, 0.99216, 0.99608, 1.00000
};
float rgb_b[256] = {
0.00000, 0.00392, 0.01176, 0.01961, 0.02745, 0.03529, 0.04314, 0.05098, 
0.05882, 0.06667, 0.07451, 0.08235, 0.09020, 0.09804, 0.10588, 0.11373, 
0.12157, 0.12941, 0.13725, 0.14510, 0.15294, 0.16078, 0.16863, 0.17647, 
0.18431, 0.19216, 0.20000, 0.20784, 0.21569, 0.22353, 0.23137, 0.23922, 
0.24706, 0.25490, 0.26275, 0.27059, 0.27843, 0.28627, 0.29412, 0.30196, 
0.30980, 0.31765, 0.32549, 0.33333, 0.34118, 0.34902, 0.35686, 0.36471, 
0.37255, 0.38039, 0.38824, 0.39608, 0.40392, 0.41176, 0.41961, 0.42745, 
0.43529, 0.44314, 0.45098, 0.45882, 0.46667, 0.47451, 0.48235, 0.49020, 
0.49804, 0.50588, 0.51373, 0.52157, 0.52941, 0.53725, 0.54510, 0.55294, 
0.56078, 0.56863, 0.57647, 0.58431, 0.59216, 0.60000, 0.60784, 0.61569, 
0.62353, 0.63137, 0.63922, 0.64706, 0.65490, 0.66275, 0.67059, 0.67843, 
0.68627, 0.69412, 0.70196, 0.70980, 0.71765, 0.72549, 0.73333, 0.74118, 
0.74902, 0.75686, 0.76471, 0.77255, 0.78039, 0.78824, 0.79608, 0.80392, 
0.81176, 0.81961, 0.82745, 0.83529, 0.84314, 0.85098, 0.85882, 0.86667, 
0.87451, 0.88235, 0.89020, 0.89804, 0.90588, 0.91373, 0.92157, 0.92941, 
0.93725, 0.94510, 0.95294, 0.96078, 0.96863, 0.97647, 0.98431, 0.99216, 
1.00000, 0.98431, 0.96863, 0.95294, 0.93333, 0.91765, 0.90196, 0.88627, 
0.86667, 0.85098, 0.83529, 0.81961, 0.80000, 0.78431, 0.76863, 0.75294, 
0.73333, 0.71765, 0.70196, 0.68627, 0.66667, 0.65098, 0.63529, 0.61961, 
0.60000, 0.58431, 0.56863, 0.55294, 0.53333, 0.51765, 0.50196, 0.48627, 
0.46667, 0.45098, 0.43529, 0.41961, 0.40000, 0.38431, 0.36863, 0.35294, 
0.33333, 0.31765, 0.30196, 0.28627, 0.26667, 0.25098, 0.23529, 0.21961, 
0.20000, 0.18431, 0.16863, 0.15294, 0.13333, 0.11765, 0.10196, 0.08627, 
0.06667, 0.05098, 0.03529, 0.01961, 0.00000, 0.01176, 0.02745, 0.04314, 
0.05882, 0.07451, 0.08627, 0.10196, 0.11765, 0.13333, 0.14902, 0.16078, 
0.17647, 0.19216, 0.20784, 0.22353, 0.23529, 0.25098, 0.26667, 0.28235, 
0.29804, 0.30980, 0.32549, 0.34118, 0.35686, 0.37255, 0.38431, 0.40000, 
0.41569, 0.43137, 0.44706, 0.45882, 0.47451, 0.49020, 0.50588, 0.52157, 
0.53725, 0.54902, 0.56471, 0.58039, 0.59608, 0.61176, 0.62353, 0.63922, 
0.65490, 0.67059, 0.68627, 0.69804, 0.71373, 0.72941, 0.74510, 0.76078, 
0.77255, 0.78824, 0.80392, 0.81961, 0.83529, 0.84706, 0.86275, 0.87843, 
0.89412, 0.90980, 0.92157, 0.93725, 0.95294, 0.96863, 0.98431, 1.00000
};
    for(i=0; i<mNCol; i++) {
      mycol.red = (rgb_r[i]*65535.); 
      mycol.green = (rgb_g[i]*65535.); 
      mycol.blue = (rgb_b[i]*65535.); 
      AllocColor(mycol, i);      
    }
   }
    break;

  default :
    for(i=0; i<mNCol; i++) {
      mycol.green = mycol.red = mycol.blue = 0;
      AllocColor(mycol, i);
    }
    break;
  }
}
