#include <unistd.h>
#include <stdexcept>

#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toiseqbuff.h"

#include "sambainit.h"
#include "map2toi.h"
#include "fitsspherehealpix.h" 
#include "timing.h"

void usage(void);
void usage(void) {
 cout<<"tstmap2toi [-h] [-p lp] [-s samplemin,samplemax] [-w data_window_size]"<<endl
     <<"           [-a label_alpha] [-d label_delta] [-b label_bolomuv]"<<endl
     <<"           fitsin fitsphere fitsout"<<endl;
 return;
}

////////////////////////////////////////////////////////////////
int main(int narg, char** arg) {

TOIManager* mgr = TOIManager::getManager();

//-- Decodage arguments
int lp = 0, width = 8192;
char *label_alpha = "alpha", *label_delta = "delta", *label_bolomuv = "boloMuV";
long sdeb,sfin;
char c;
while((c = getopt(narg,arg,"hp:s:w:a:d:b:")) != -1) {
  switch (c) {
  case 's' :
    sscanf(optarg,"%ld,%ld",&sdeb,&sfin);
    cout<<"Requested Samples from "<<sdeb<<" , "<<sfin<<endl;
    if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);
    else {cout<<"Bad sample interval "<<endl; exit(-2);}
    break;
  case 'w' :
    sscanf(optarg,"%d",&width);
    if(width<=0) width=8192;
    cout<<"Data window size "<<width<<endl;
    break;
  case 'p' :
    sscanf(optarg,"%d",&lp);
    if(lp<0) lp=0;
    break;
  case 'a' :
    label_alpha = optarg;
    break;
  case 'd' :
    label_delta = optarg;
    break;
  case 'b' :
    label_bolomuv = optarg;
    break;
  case 'h' :
    usage(); exit(-1);
    break;
  default:
    usage(); exit(-1);
  }
}
if(optind+2>=narg) {usage(); exit(-2);}
char * fitsin          = arg[optind];
string const fitsphere = arg[optind+1];
char * fitsout         = arg[optind+2];

cout<<">>>> tstmap2toi:"<<endl
    <<"Fits Infile(snum,alpha,delta)= "<<fitsin<<endl
    <<"  ...label_alpha "<<label_alpha<<"  ,  label_delta "<<label_delta<<endl
    <<"Fits Sphere Healpix"<<fitsphere<<endl
    <<"Fits Outfile(snum,[alpha,delta],boloMuV)"<<fitsout<<endl
    <<"  ...label_bolomuv "<<label_bolomuv<<endl;

SophyaInit();
InitTim();

//--------------------------------------------------------------------
try {
//--------------------------------------------------------------------

 // FITS reader et writer
 FITSTOIReader rfits(fitsin);
 int ncol = rfits.getNOut();
 cout<<"Number of columns in fits Infile : "<<ncol<<endl;
 if(ncol<2) exit(-3);

 FITSTOIWriter wfits(fitsout);
 wfits.setOutFlags(true);
 cout << "fits reader and writer created"<<endl;

 // Lecture de la sphere Healpix
 SphereHEALPix<r_8> sph;        
 FitsInFile sfits(fitsphere);
 sfits >> sph;
 cout<<"SphereHEALPix: Type de map : "<<sph.TypeOfMap()<<endl
     <<"               Nombre de pixels : "<<sph.NbPixels()<<endl;

 // TOI Processor
 Map2TOI m2toi(sph);
 cout<<"Map2TOI created"<<endl;

 // Definition des tuyaux
 TOISeqBuffered * toialphain = new TOISeqBuffered("toi_alpha_in",width);
 if(lp) toialphain->setDebugLevel(1);
 rfits.addOutput(label_alpha,toialphain);
 m2toi.addInput("AlphaIn",toialphain);

 TOISeqBuffered * toialphaout = new TOISeqBuffered("toi_alpha_out",width);
 m2toi.addOutput("AlphaOut",toialphaout);
 wfits.addInput(label_alpha,toialphaout);

 TOISeqBuffered * toideltain = new TOISeqBuffered("toi_delta_in",width);
 if(lp) toideltain->setDebugLevel(1);
 rfits.addOutput(label_delta,toideltain);
 m2toi.addInput("DeltaIn",toideltain);

 TOISeqBuffered * toideltaout = new TOISeqBuffered("toi_delta_out",width);
 m2toi.addOutput("DeltaOut",toideltaout);
 wfits.addInput(label_delta,toideltaout);

 TOISeqBuffered * toibolout = new TOISeqBuffered("toi_bolo_out",width);
 if(lp) toibolout->setDebugLevel(1);
 m2toi.addOutput("BoloOut",toibolout);
 wfits.addInput(label_bolomuv,toibolout);

 // Run
 cout<<"----- FITSReaderTOI::PrintStatus() : -----"<<endl;
 rfits.PrintStatus(cout);
 cout<<"----- FITSWriterTOI::PrintStatus() : -----"<<endl;
 wfits.PrintStatus(cout);  

 PrtTim("starting threads");
 rfits.start();
 m2toi.start();
 wfits.start();

 if(lp>1)
   for(int jjjj=0;jjjj<5;jjjj++) {
     cout<<*toialphain; 
     cout<<*toibolout; 
     sleep(2);
   }

 mgr->joinAll();
 PrtTim("End threads");

//--------------------------------------------------------------------
} catch (PThrowable & exc) {
 cout<<"\ntstmap2toi: Catched Exception \n"<<(string)typeid(exc).name() 
     <<" - Msg= "<<exc.Msg()<<endl;
} catch (const std::exception & sex) {
 cout<<"\ntstmap2toi: Catched std::exception \n" 
     <<(string)typeid(sex).name()<<endl; 
} catch (...) {
 cout<<"\ntstmap2toi: some other exception was caught ! "<<endl;
}
//--------------------------------------------------------------------

exit(0);
}
