#include <unistd.h>
#include <stdexcept>

#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toiseqbuff.h"

#include "sambainit.h"
#include "toi2map.h"
#include "fitsspherehealpix.h" 
#include "timing.h"

void usage(void);
void usage(void) {
 cout<<"tsttoi2map [-h] [-p lp] [-s samplemin,samplemax] [-w data_window_size]"<<endl
     <<"           [-a label_alpha] [-d label_delta] [-b label_bolomuv]"<<endl
     <<"           [-n nlat] [-G] fitsin fitsphout fitsphwout"<<endl;
 return;
}

////////////////////////////////////////////////////////////////
int main(int narg, char** arg) {

TOIManager* mgr = TOIManager::getManager();

//-- Decodage arguments
int lp = 0, width = 8192;
int nlat = 128;   // npixel = 12 * nlat^2
char *label_alpha = "alpha", *label_delta = "delta", *label_bolomuv = "boloMuV";
bool coorgal=false;
long sdeb,sfin;
int c;
while((c = getopt(narg,arg,"hGp:s:w:a:d:b:n:")) != -1) {
  switch (c) {
  case 's' :
    sscanf(optarg,"%ld,%ld",&sdeb,&sfin);
    cout<<"Requested Samples from "<<sdeb<<" , "<<sfin<<endl;
    if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);
    else {cout<<"Bad sample interval "<<endl; exit(-2);}
    break;
  case 'w' :
    sscanf(optarg,"%d",&width);
    if(width<=0) width=8192;
    cout<<"Data window size "<<width<<endl;
    break;
  case 'p' :
    sscanf(optarg,"%d",&lp);
    if(lp<0) lp=0;
    break;
  case 'a' :
    label_alpha = optarg;
    break;
  case 'd' :
    label_delta = optarg;
    break;
  case 'b' :
    label_bolomuv = optarg;
    break;
  case 'n' :
    sscanf(optarg,"%d",&nlat);
    if(nlat<0) nlat=128;
    break;
  case 'G' :
    coorgal = true;
    break;
  case 'h' :
    usage(); exit(-1);
    break;
  default:
    usage(); exit(-1);
  }
}
if(optind+1>=narg) {usage(); exit(-2);}
char * fitsin          = arg[optind];
string const fitsphout = arg[optind+1];
string fitsphwout = "";
if(optind+2<narg) fitsphwout = arg[optind+2];

cout<<">>>> tsttoi2map:"<<endl
    <<"Fits Infile(snum,alpha,delta,bolomuv)= "<<fitsin<<endl
    <<"  ...label_alpha "<<label_alpha<<"  ,  label_delta "<<label_delta<<endl
    <<"  ...label_bolomuv "<<label_bolomuv<<endl
    <<"Fits Sphere Healpix"<<fitsphout<<endl
    <<"  ...nlat "<<nlat<<endl
    <<"Fits Sphere Healpix Error "<<fitsphwout<<endl;

SophyaInit();
InitTim();

//--------------------------------------------------------------------
try {
//--------------------------------------------------------------------

 // FITS reader
 FITSTOIReader rfits(fitsin);
 int ncol = rfits.getNOut();
 cout<<"Number of columns in fits Infile : "<<ncol<<endl;
 if(ncol<3) exit(-3);

 // Creation de la sphere Healpix
 SphereHEALPix<r_8>* sph = new SphereHEALPix<r_8>(nlat);
 cout<<"SphereHEALPix: Type de map : "<<sph->TypeOfMap()<<endl
     <<"               Nombre de pixels : "<<sph->NbPixels()<<endl;

 // Creation de la sphere de poids Healpix
 SphereHEALPix<r_8>* wsph = NULL;
 if(fitsphwout.size()>0) {
   wsph = new SphereHEALPix<r_8>;
   cout<<"SphereHEALPix Weight: Type de map : "<<wsph->TypeOfMap()<<endl
       <<"               Nombre de pixels : "<<wsph->NbPixels()<<endl;
 }

 // TOI Processor
 TOI2Map toi2m(sph,wsph);
 cout<<"TOI2Map created"<<endl;
 toi2m.SetCoorGal(coorgal,2000.);  // equinoxe de ref. 2000. (pour archtoi)

 // Definition des tuyaux
 TOISeqBuffered * toialphain = new TOISeqBuffered("toi_alpha_in",width);
 if(lp) toialphain->setDebugLevel(1);
 rfits.addOutput(label_alpha,toialphain);
 toi2m.addInput("AlphaIn",toialphain);

 TOISeqBuffered * toideltain = new TOISeqBuffered("toi_delta_in",width);
 if(lp) toideltain->setDebugLevel(1);
 rfits.addOutput(label_delta,toideltain);
 toi2m.addInput("DeltaIn",toideltain);

 TOISeqBuffered * toibolin = new TOISeqBuffered("toi_bolo_in",width);
 if(lp) toibolin->setDebugLevel(1);
 rfits.addOutput(label_bolomuv,toibolin);
 toi2m.addInput("BoloIn",toibolin);

 // Run
 cout<<"----- FITSReaderTOI::PrintStatus() : -----"<<endl;
 rfits.PrintStatus(cout);

 PrtTim("starting threads");
 rfits.start();
 toi2m.start();

 if(lp>1)
   for(int jjjj=0;jjjj<5;jjjj++) {
     cout<<*toialphain; 
     cout<<*toibolin; 
     sleep(2);
   }

 mgr->joinAll();
 PrtTim("End threads");

 // Ecriture de la sphere Healpix sur fits
 {
 FitsOutFile sfits(fitsphout);
 cout<<"tsttoi2map: Creating sphere fits file "<<fitsphout<<endl;
 sfits << *sph;
 }

 // Ecriture de la sphere Healpix sur fits
 if(wsph) {
   FitsOutFile swfits(fitsphwout);
   cout<<"tsttoi2map: Creating sphere weight fits file "<<fitsphwout<<endl;
   swfits << *wsph;
 }

 // Nettoyage
 delete sph;
 if(wsph) delete wsph;

//--------------------------------------------------------------------
} catch (PThrowable & exc) {
 cout<<"\ntsttoi2map: Catched Exception \n"<<(string)typeid(exc).name() 
     <<" - Msg= "<<exc.Msg()<<endl;
} catch (const std::exception & sex) {
 cout<<"\ntsttoi2map: Catched std::exception \n" 
     <<(string)typeid(sex).name()<<endl; 
} catch (...) {
 cout<<"\ntsttoi2map: some other exception was caught ! "<<endl;
}
//--------------------------------------------------------------------

exit(0);
}
