#include "array.h"
#include <math.h>
#include "genwproc.h"
#include "toimanager.h"
#include "pexceptions.h"
#include "ctimer.h"

// -------------------------------------------------------------
//   Class GenWindowTOIProcessor : generic processor with window
// -------------------------------------------------------------

////////////////////////////////////////////////////////////////
GenWindowTOIProcessor::GenWindowTOIProcessor(int nbinput,int nboutput
                                   ,int wsz, int wstep, int wsztot)
{
  if(nbinput<1) throw ParmError("GenWindowTOIProcessor::Creator nbinput<1 !");
  if(nboutput<0) nboutput=0;
  if(wsz<2) throw ParmError("GenWindowTOIProcessor::Creator wsz<2 !");
  if(wsz%2==0) wsz++;
  if(wstep<1) wstep=1;
  if(wsztot<wsz) wsztot = 2*wsz;

  NbInput = nbinput;
  NbOutput = nboutput;
  WSize = wsz;
  WStep = wstep;
  WSizeTot = wsztot;

  SNdeb = SNend = StartSample = CurSnInd = -1;
  TotNsCount = 0;

  TVector<r_8> vr8;
  TVector<int_8> vi8;
  
  int k;
  for(k=0;k<NbInput;k++) {
    WDataIn.push_back(vr8);
    WFlagIn.push_back(vi8);
    WInFlg.push_back(false);
  }
  if(NbOutput) for(k=0;k<NbOutput;k++) {
    WDataOut.push_back(vr8);
    WFlagOut.push_back(vi8);
    WOutFlg.push_back(false);
    WPutOutFlg.push_back(false);
    OutSample.push_back(0);
  }
}

GenWindowTOIProcessor::~GenWindowTOIProcessor()
{
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::PrintStatus(ostream & os)
{
  os<<"\n ------------------------------------------------------ \n" 
    <<" GenWindowTOIProcessor::PrintStatus() - ["
    <<NbInput<<","<<NbOutput<<"] (wtot="<<WSizeTot
    <<" WindowSize="<<GetWSize()
    <<" WStep= "<<GetWStep()<<endl;
  TOIProcessor::PrintStatus(os);
  os<<"ProcessedSampleCount="<<ProcessedSampleCount()<<endl;
  os<<"------------------------------------------------------ "<<endl;
}

////////////////////////////////////////////////////////////////
TVector<r_8> GenWindowTOIProcessor::GetWData(int numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWData : toi out of range !");
  if(!WInFlg[numtoi])
    throw RangeCheckError("GenWindowTOIProcessor::GetWData : toi not connected!");
  return WDataIn[numtoi];
}

TVector<int_8> GenWindowTOIProcessor::GetWFlag(int numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWFlag : toi out of range !");
  if(!WInFlg[numtoi])
    throw RangeCheckError("GenWindowTOIProcessor::GetWFlag : toi not connected!");
  return WFlagIn[numtoi];
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::PutWData(int numtoi,int_8 numsample
                                    ,TVector<r_8>& data,TVector<int_8>& flag)
{
  if(numtoi<0 || numtoi>=NbOutput)
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi out of range !");
  if(!WOutFlg[numtoi])
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi not connected!");
  if(data.Size()!=flag.Size())
    throw ParmError("GenWindowTOIProcessor::PutWFlag : data.Size()!=flag.Size()!");
  WDataOut[numtoi].Share(data);
  WFlagOut[numtoi].Share(flag);
  OutSample[numtoi] = numsample;
  WPutOutFlg[numtoi] = true;
}

void GenWindowTOIProcessor::PutWData(int numtoi,int_8 numsample
                                    ,r_8 data,int_8 flag)
{
  if(numtoi<0 || numtoi>=NbOutput)
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi out of range !");
  if(!WOutFlg[numtoi])
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi not connected!");
  WDataOut[numtoi].Realloc(1,BaseArray::SameVectorType,true);
    (WDataOut[numtoi])(0) = data;
  WFlagOut[numtoi].Realloc(1,BaseArray::SameVectorType,true);
    (WFlagOut[numtoi])(0) = flag;
  OutSample[numtoi] = numsample;
  WPutOutFlg[numtoi] = true;
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::UserInit()
{
  cout<<"GenWindowTOIProcessor::UserInit() Default implementation does nothing"<<endl;
}

void GenWindowTOIProcessor::UserProc()
{
  cout<<"GenWindowTOIProcessor:UserProc() Default implementation does nothing"<<endl;
}

void GenWindowTOIProcessor::UserEnd()
{
  cout<<"GenWindowTOIProcessor::UserEnd() Default implementation does nothing"<<endl;
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::init()
{
  cout << "GenWindowTOIProcessor::init" << endl;
  char buff[64];
  int k;
  for(k=0; k<NbInput; k++) {
    sprintf(buff,"in%d", k);
    declareInput(buff);
  }
  for(k=0; k<NbOutput; k++) {
    sprintf(buff,"out%d",k);
    declareOutput(buff);
  }
  name = "GenWindowTOIProcessor";
  // upExtra = 1; $CHECK a quoi ca sert EA?
}

void GenWindowTOIProcessor::run()
{
  //  TOIManager* mgr = TOIManager::getManager();
  int_8 SNdeb = getMinIn();
  int_8 SNend = getMaxIn();
  if(SNend-SNdeb<WSize)
    throw RangeCheckError("GenWindowTOIProcessor::run : sne-snb<WSize !");

  // Allocation des tailles pour les vecteurs
  int kc, nc=0;
  for(kc=0;kc<NbInput;kc++) {
    if( !(WInFlg[kc]=checkInputTOIIndex(kc)) ) continue;
     WDataIn[kc].ReSize(WSizeTot);
     WFlagIn[kc].ReSize(WSizeTot);
     nc++;
  }
  if(nc==0) {
    cerr<<" GenWindowTOIProcessor::run() - No input TOI connected!"<<endl;
    throw ParmError("GenWindowTOIProcessor::run() No input TOI connected!");
  }
  for(kc=0;kc<NbOutput;kc++)
      WOutFlg[kc] = checkOutputTOIIndex(kc);

  // Lecture des samples et remplissage des vecteurs
  cout<<"GenWindowTOIProcessor::run() SNRange="<<SNdeb<<" - "<<SNend<<endl; 
  try {
    Timer tm("GenWindowTOIProcessor::run()");
    int_8 ki = -1;

    for(int_8 ks=SNdeb;ks<=SNend;ks++) {
      Remplissage(ks);
      TotNsCount++;
    }
    cout << " GenWindowTOIProcessor::run() - End of processing " << endl;

  } catch(PException & exc) {
    cerr<<"GenWindowTOIProcessor: Catched Exception "<<(string)typeid(exc).name()
        <<"\n .... Msg= "<<exc.Msg()<<endl;
  }
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::Remplissage(int_8 ks)
// INPUT:
//   ks : numero du sample CENTRAL
//      samples :   sn-ws/2        sn          sn+ws/2
//      fenetre :   0              ws/2+1      ws-1
//   ki : index des vecteurs a remplir.
//        C'est l'index ou on veut remplir la donnee
//        dans un vecteur (fictif) de taille WSize.
//        La routine gere le decalage dans le grand buffer WSizeTot.
//  Si ki depasse la taille maximale permise on decale, sinon on ne fait rien
// RETURN:
//   numero du PROCHAIN index a remplir (ki+1 si pas decale !)
{
if(ks<SNdeb || ks>SNend)
  throw RangeCheckError("GenWindowTOIProcessor::remplissage : ks<SNdeb || ks>SNend !");

int_8 wsz2=WSize/2;

StartSample = ks - wsz2; // peut etre < snb au debut

// Premier remplissage ???? Gestion de la borne inferieure
if(CurSnInd<0) {
  CurSnInd = 0;
  for(int_8 k=ks-wsz2; k<=ks+wsz2; k++) { // Lecture TOI
    for(int kc=0; kc<NbInput; kc++) {
      if(!WInFlg[kc]) continue;
      if(k>=SNdeb && k<=SNend) {
        getData(kc,k,(WDataIn[kc])(CurSnInd), (WFlagIn[kc])(CurSnInd));
      } else {
        (WDataIn[kc])(CurSnInd) = (r_8) 0.;
        (WFlagIn[kc])(CurSnInd) = (int_8) 0;
      }
    }
    CurSnInd++;
  }
  return;
}

// Faut-il decaler ????
if(CurSnInd == WSizeTot) { // On decale
  for(int kc=0; kc<NbInput; kc++) {
    if(!WInFlg[kc]) continue;
      for(int_8 k=1;k<WSize;k++) { // un en moins car on va remplir apres
        (WDataIn[kc])(k-1) = (WDataIn[kc])(WSizeTot-WSize+k);
        (WFlagIn[kc])(k-1) = (WFlagIn[kc])(WSizeTot-WSize+k);
      }
  }
  CurSnInd = WSize-1;
}

// Remplissage de ks+wsz2 (dernier element de la fenetre pour ks central)
int_8 kse = ks+wsz2;
for(int kc=0; kc<NbInput; kc++) {
  if(!WInFlg[kc]) continue;
  if(kse>=SNdeb && kse<=SNend) {
    getData(kc,kse,(WDataIn[kc])(CurSnInd),(WFlagIn[kc])(CurSnInd));
  } else {
    (WDataIn[kc])(CurSnInd) = (r_8) 0.;
    (WFlagIn[kc])(CurSnInd) = (int_8) 0;
  }
}
CurSnInd++;

return;
}
