#include <unistd.h>
#include <stdexcept>
#include <stdlib.h>
#include <stdio.h>
#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toiseqbuff.h"

#include "sambainit.h"
#include "toi2map.h"
#include "fitsspherehealpix.h" 
#include "timing.h"

void usage(void);
void usage(void) {
 cout<<"tsttoi2map [-h] [-p lp] [-s samplemin,samplemax] [-w data_window_size]"<<endl
     <<"           [-a label_coord1] [-d label_coord2] [-b label_bolomuv]"<<endl
     <<"           [-n nlat] [-i c,h] [-o c,h]"<<endl
     <<"           fitsin_point fitsin_bolo fitsphout [fitsphwout]"<<endl
     <<" -p lp : print level (def=0)"<<endl
     <<" -s samplemin,samplemax : sample range to be treated (def=all)"<<endl
     <<" -w data_window_size : window size for pipe (def=8192)"<<endl
     <<" -a label_coord1 : label fits for alpha/gLong (def=coord1)"<<endl
     <<" -d label_coord2 : label fits for delta/gLat (def=coord2)"<<endl
     <<"          coord1 = alpha or gLong ; coord2 = delta or gLat"<<endl
     <<" -b label_bolomuv : label fits for bolo value (def=boloMuV)"<<endl
     <<" -n nlat : nlat for Healpix sphere (def=128)"<<endl
     <<" -i c,h : coord1 caracteristics (c=G/E h=H/D) (def=G,D)"<<endl
     <<" -o c,h : idem -i for coord2"<<endl
     <<" fitsin_point : fits file for pointing"<<endl
     <<" fitsin_bolo : fits file for bolo values"<<endl
     <<" fitsphout : fits file for output Healpix sphere"<<endl
     <<" fitsphwout : fits file for output Healpix nFilled sphere (def=no)"<<endl;
}

unsigned long typecoord(char typc=' ',char hd=' ');
unsigned long typecoord(char typc,char hd)
// typc : G=galactiques, E=equatoriales, autres=galactiques
// hd : H=heure, D=degre, autres=(heure si typc==E, degre si typc==G)
{
  if(typc!='G' && typc!='E') typc='G';
  if(hd!='H' && hd!='D') {if(typc=='E') hd='H'; else hd='D';}
 unsigned long rc=TypCoordUndef;
  if(typc=='G') rc |= TypCoordGal;
    else        rc |= TypCoordEq;
  if(hd=='D')   rc |= TypCoordDD;
    else        rc |= TypCoordHD;
  return rc;
}

////////////////////////////////////////////////////////////////
int main(int narg, char** arg) {

TOIManager* mgr = TOIManager::getManager();

//-- Decodage arguments
int lp = 0, width = 8192;
int nlat = 128;   // npixel = 12 * nlat^2
char *label_coord1 = "coord1", *label_coord2 = "coord2", *label_bolomuv = "boloMuV";
long sdeb,sfin;
string fitsphwout = "";
unsigned long tcoorin=typecoord(), tcoorout=typecoord();
int c; char t=' ',h=' ';
while((c = getopt(narg,arg,"hp:s:w:a:d:b:n:i:o:")) != -1) {
  switch (c) {
  case 's' :
    sscanf(optarg,"%ld,%ld",&sdeb,&sfin);
    cout<<"Requested Samples from "<<sdeb<<" , "<<sfin<<endl;
    if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);
    else {cout<<"Bad sample interval "<<endl; exit(2);}
    break;
  case 'w' :
    sscanf(optarg,"%d",&width);
    if(width<=0) width=8192;
    cout<<"Data window size "<<width<<endl;
    break;
  case 'p' :
    sscanf(optarg,"%d",&lp);
    if(lp<0) lp=0;
    break;
  case 'a' :
    label_coord1 = optarg;
    break;
  case 'd' :
    label_coord2 = optarg;
    break;
  case 'b' :
    label_bolomuv = optarg;
    break;
  case 'n' :
    sscanf(optarg,"%d",&nlat);
    if(nlat<0) nlat=128;
    break;
  case 'i' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoorin=typecoord(t,h);
    break;
  case 'o' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoorout=typecoord(t,h);
    break;
  case 'h' :
  default:
    usage(); exit(1);
    break;
  }
}
if(optind+2>=narg) {usage(); exit(3);}
 
char * fitsin_point = arg[optind];
char * fitsin_bolo = arg[optind+1];
string const fitsphout = arg[optind+2];
if(optind+3<narg) fitsphwout = arg[optind+3];

{
unsigned long tg,te,hd,dd;
cout<<">>>> tsttoi2map:"<<endl
    <<"Pipe Window Size "<<width<<endl
    <<"Fits Infile Bolo "<<fitsin_bolo<<endl
    <<"  ...label_bolomuv "<<label_bolomuv<<endl;
tg = tcoorin&TypCoordGal; te = tcoorin&TypCoordEq;
hd = tcoorin&TypCoordHD;  dd = tcoorin&TypCoordDD;
cout<<"  ...label_coord1 "<<label_coord1<<endl
    <<"  ...label_coord2 "<<label_coord2<<endl
    <<"  ...... Gal="<<tg<<" Eq="<<te<<"   hour="<<hd<<" deg="<<dd<<endl;
tg = tcoorout&TypCoordGal; te = tcoorout&TypCoordEq;
hd = tcoorout&TypCoordHD;  dd = tcoorout&TypCoordDD; 
cout<<"Fits Healpix Sphere "<<fitsphout<<endl
    <<"  ...nlat "<<nlat<<endl
    <<"  ...... Gal="<<tg<<" Eq="<<te<<"   hour="<<hd<<" deg="<<dd<<endl;
cout<<"Fits Healpix Weight Sphere "<<fitsphwout<<endl;
}

SophyaInit();
InitTim();

//--------------------------------------------------------------------
try {
//--------------------------------------------------------------------

 // FITS reader
 FITSTOIReader rfitsb(fitsin_bolo);
 int ncolb = rfitsb.getNOut();
 cout<<"Number of columns in fits Infile_bolo : "<<ncolb<<endl;
 if(ncolb<1) exit(-4);

 FITSTOIReader rfitsp(fitsin_point);
 int ncolp = rfitsp.getNOut();
 cout<<"Number of columns in fits Infile_point : "<<ncolp<<endl;
 if(ncolp<2) exit(-5);

 // Creation de la sphere Healpix
 SphereHEALPix<r_8>* sph = new SphereHEALPix<r_8>(nlat);
 cout<<"SphereHEALPix: Type de map : "<<sph->TypeOfMap()<<endl
     <<"               Nombre de pixels : "<<sph->NbPixels()<<endl;

 // Creation de la sphere de poids Healpix
 SphereHEALPix<r_8>* wsph = NULL;
 if(fitsphwout.size()>0) {
   wsph = new SphereHEALPix<r_8>;
   cout<<"SphereHEALPix Weight: Type de map : "<<wsph->TypeOfMap()<<endl
       <<"              Nombre de pixels : "<<wsph->NbPixels()<<endl;
 }

 // TOI Processor
 TOI2Map toi2m(sph,wsph);
 cout<<"TOI2Map created"<<endl;
 toi2m.SetEquinox(2000.);
 toi2m.SetCoorIn((TypAstroCoord) tcoorin);
 toi2m.SetCoorOut((TypAstroCoord) tcoorout);

 // Definition des tuyaux
 TOISeqBuffered * toicoord1in = new TOISeqBuffered("toi_coord1_in",width);
 // toicoord1in->setDebugLevel(1);
 rfitsp.addOutput(label_coord1,toicoord1in);
 toi2m.addInput("Coord1In",toicoord1in);

 TOISeqBuffered * toicoord2in = new TOISeqBuffered("toi_coord2_in",width);
 // toicoord2in->setDebugLevel(1);
 rfitsp.addOutput(label_coord2,toicoord2in);
 toi2m.addInput("Coord2In",toicoord2in);
 
 TOISeqBuffered * toibolin = new TOISeqBuffered("toi_bolo_in",width);
 // toibolin->setDebugLevel(1);
 rfitsb.addOutput(label_bolomuv,toibolin);
 toi2m.addInput("BoloIn",toibolin);

 // Run
 cout<<"----- FITSReaderTOI::PrintStatus() : -----"<<endl;
 rfitsp.PrintStatus(cout);
 rfitsb.PrintStatus(cout);

 PrtTim("starting threads");
 rfitsp.start();
 rfitsb.start();
 toi2m.start();

 if(lp>1)
   for(int jjjj=0;jjjj<5;jjjj++) {
     cout<<*toicoord1in; 
     cout<<*toibolin; 
     sleep(2);
   }

 mgr->joinAll();
 PrtTim("End threads");

 // Ecriture de la sphere Healpix sur fits
 {
 FitsOutFile sfits(fitsphout);
 cout<<"tsttoi2map: Creating sphere fits file "<<fitsphout<<endl;
 sfits << *sph;
 }

 // Ecriture de la sphere Healpix sur fits
 if(wsph) {
   FitsOutFile swfits(fitsphwout);
   cout<<"tsttoi2map: Creating sphere weight fits file "<<fitsphwout<<endl;
   swfits << *wsph;
 }

 // Nettoyage
 delete sph;
 if(wsph) delete wsph;

//--------------------------------------------------------------------
} catch (PThrowable & exc) {
 cout<<"\ntsttoi2map: Catched Exception \n"<<(string)typeid(exc).name() 
     <<" - Msg= "<<exc.Msg()<<endl;
} catch (const std::exception & sex) {
 cout<<"\ntsttoi2map: Catched std::exception \n" 
     <<(string)typeid(sex).name()<<endl; 
} catch (...) {
 cout<<"\ntsttoi2map: some other exception was caught ! "<<endl;
}
//--------------------------------------------------------------------

exit(0);
}
