#include <unistd.h>
#include <stdexcept>

#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toiseqbuff.h"

#include "sambainit.h"
#include "map2toi.h"
#include "fitsspherehealpix.h" 
#include "timing.h"

void usage(void);
void usage(void) {
 cout<<"tstmap2toi [-h] [-p lp] [-s samplemin,samplemax] [-w data_window_size]"<<endl
     <<"           [-a label_coord1In] [-d label_coord2In] [-b label_bolomuv]"<<endl
     <<"           [-i c,h] [-m c,h]"<<endl
     <<"           fitsin_point fitsphere fitsout"<<endl
     <<" -p lp : print level (def=0)"<<endl
     <<" -s samplemin,samplemax : sample range to be treated (def=all)"<<endl
     <<" -w data_window_size : window size for pipe (def=8192)"<<endl
     <<" -a label_coord1 : label fits for alpha/gLong (def=coord1)"<<endl
     <<" -d label_coord2 : label fits for delta/gLat (def=coord2)"<<endl
     <<"          coord1 = alpha or gLong ; coord2 = delta or gLat"<<endl
     <<" -b label_bolomuv : label fits for bolo value (def=boloMuV)"<<endl
     <<" -i c,h : coordIn caracteristics (c=G/E h=H/D) (def=G,D)"<<endl
     <<" -m c,h : idem -i for Sphere"<<endl
     <<" fitsin_point : fits file for pointing"<<endl
     <<" fitsphere : fits file for input Healpix sphere"<<endl
     <<" fitsout : fits file for output"<<endl;
 return;
}

unsigned long typecoord(char typc=' ',char hd=' ');
unsigned long typecoord(char typc,char hd)
// typc : G=galactiques, E=equatoriales, autres=galactiques
// hd : H=heure, D=degre, autres=(heure si typc==E, degre si typc==G)
{
  if(typc!='G' && typc!='E') typc='G';
  if(hd!='H' && hd!='D') {if(typc=='E') hd='H'; else hd='D';}
 unsigned long rc=TypCoordUndef;
  if(typc=='G') rc |= TypCoordGal;
    else        rc |= TypCoordEq;
  if(hd=='D')   rc |= TypCoordDD;
    else        rc |= TypCoordHD;
  return rc;
}

////////////////////////////////////////////////////////////////
int main(int narg, char** arg) {

TOIManager* mgr = TOIManager::getManager();

//-- Decodage arguments
int lp = 0, width = 8192;
char *label_coord1 = "coord1", *label_coord2 = "coord2", *label_bolomuv = "boloMuV";
long sdeb,sfin;
unsigned long tcoorin=typecoord(), tcoormap=typecoord();
int c; char t=' ',h=' ';
while((c = getopt(narg,arg,"hp:s:w:a:d:b:i:m:")) != -1) {
  switch (c) {
  case 's' :
    sscanf(optarg,"%ld,%ld",&sdeb,&sfin);
    cout<<"Requested Samples from "<<sdeb<<" , "<<sfin<<endl;
    if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);
    else {cout<<"Bad sample interval "<<endl; exit(1);}
    break;
  case 'w' :
    sscanf(optarg,"%d",&width);
    if(width<=0) width=8192;
    cout<<"Data window size "<<width<<endl;
    break;
  case 'p' :
    sscanf(optarg,"%d",&lp);
    if(lp<0) lp=0;
    break;
  case 'a' :
    label_coord1 = optarg;
    break;
  case 'd' :
    label_coord2 = optarg;
    break;
  case 'b' :
    label_bolomuv = optarg;
    break;
  case 'i' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoorin=typecoord(t,h);
    break;
  case 'm' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoormap=typecoord(t,h);
    break;
  case 'h' :
  default:
    usage(); exit(1);
  }
}
if(optind+2>=narg) {usage(); exit(2);}
char * fitsin_point    = arg[optind];
string const fitsphere = arg[optind+1];
char * fitsout         = arg[optind+2];

{
unsigned long tg,te,hd,dd;
cout<<">>>> tstmap2toi:"<<endl
    <<"Pipe Window Size "<<width<<endl
    <<"Fits OutFile "<<fitsout<<endl
    <<"  ...label_bolomuv "<<label_bolomuv<<endl;
tg = tcoorin&TypCoordGal; te = tcoorin&TypCoordEq;
hd = tcoorin&TypCoordHD;  dd = tcoorin&TypCoordDD;
cout<<"Fits Infile Pointing "<<fitsin_point<<endl
    <<"  ...label_coord1 "<<label_coord1<<endl
    <<"  ...label_coord2 "<<label_coord2<<endl
    <<"  ...... Gal="<<tg<<" Eq="<<te<<"   hour="<<hd<<" deg="<<dd<<endl;
tg = tcoormap&TypCoordGal; te = tcoormap&TypCoordEq;
hd = tcoormap&TypCoordHD;  dd = tcoormap&TypCoordDD; 
cout<<"Fits Healpix Sphere "<<fitsphere<<endl
    <<"  ...... Gal="<<tg<<" Eq="<<te<<"   hour="<<hd<<" deg="<<dd<<endl;
}

SophyaInit();
InitTim();

//--------------------------------------------------------------------
try {
//--------------------------------------------------------------------

 // FITS reader et writer
 FITSTOIReader rfits(fitsin_point);
 int ncol = rfits.getNOut();
 cout<<"Number of columns in fits Infile Pointing : "<<ncol<<endl;
 if(ncol<2) exit(3);

 FITSTOIWriter wfits(fitsout);
 //wfits.setOutFlags(true);
 cout << "fits reader and writer created"<<endl;

 // Lecture de la sphere Healpix
 SphereHEALPix<r_8> sph;        
 FitsInFile sfits(fitsphere);
 sfits >> sph;
 cout<<"SphereHEALPix: Type de map : "<<sph.TypeOfMap()<<endl
     <<"               Nombre de pixels : "<<sph.NbPixels()<<endl;

 // TOI Processor
 Map2TOI m2toi(sph);
 cout<<"Map2TOI created"<<endl;
 m2toi.SetEquinox(2000.);
 m2toi.SetCoorIn((TypAstroCoord) tcoorin);
 m2toi.SetCoorMap((TypAstroCoord) tcoormap);
 m2toi.Print(cout);

 // Definition des tuyaux
 TOISeqBuffered * toicoord1in = new TOISeqBuffered("toi_coord1_in",width);
 // toicoord1in->setDebugLevel(1);
 rfits.addOutput(label_coord1,toicoord1in);
 m2toi.addInput("Coord1In",toicoord1in);

 TOISeqBuffered * toicoord1out = new TOISeqBuffered("toi_coord1_out",width);
 m2toi.addOutput("Coord1Out",toicoord1out);
 wfits.addInput(label_coord1,toicoord1out);

 TOISeqBuffered * toicoord2in = new TOISeqBuffered("toi_coord2_in",width);
 // toicoord2in->setDebugLevel(1);
 rfits.addOutput(label_coord2,toicoord2in);
 m2toi.addInput("Coord2In",toicoord2in);

 TOISeqBuffered * toicoord2out = new TOISeqBuffered("toi_coord2_out",width);
 m2toi.addOutput("Coord2Out",toicoord2out);
 wfits.addInput(label_coord2,toicoord2out);

 TOISeqBuffered * toibolout = new TOISeqBuffered("toi_bolo_out",width);
 // toibolout->setDebugLevel(1);
 m2toi.addOutput("BoloOut",toibolout);
 wfits.addInput(label_bolomuv,toibolout);

 // Run
 cout<<"----- FITSReaderTOI::PrintStatus() : -----"<<endl;
 rfits.PrintStatus(cout);
 cout<<"----- FITSWriterTOI::PrintStatus() : -----"<<endl;
 wfits.PrintStatus(cout);  

 PrtTim("starting threads");
 rfits.start();
 m2toi.start();
 wfits.start();

 if(lp>1)
   for(int jjjj=0;jjjj<5;jjjj++) {
     cout<<*toicoord1in; 
     cout<<*toibolout; 
     sleep(2);
   }

 mgr->joinAll();
 PrtTim("End threads");

//--------------------------------------------------------------------
} catch (PThrowable & exc) {
 cout<<"\ntstmap2toi: Catched Exception \n"<<(string)typeid(exc).name() 
     <<" - Msg= "<<exc.Msg()<<endl;
} catch (const std::exception & sex) {
 cout<<"\ntstmap2toi: Catched std::exception \n" 
     <<(string)typeid(sex).name()<<endl; 
} catch (...) {
 cout<<"\ntstmap2toi: some other exception was caught ! "<<endl;
}
//--------------------------------------------------------------------

exit(0);
}
