#include <unistd.h>
#include <stdexcept>
#include <stdlib.h>
#include <stdio.h>
#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toiseqbuff.h"

#include "sambainit.h"
#include "toi2map.h"
#include "fitsspherehealpix.h" 
#include "timing.h"

void usage(void);
void usage(void) {
 cout<<"tsttoi2map [-h] [-p lp] [-s samplemin,samplemax] [-w data_window_size]"<<endl
     <<"           [-a label_coord1] [-d label_coord2] [-b label_bolomuv]"<<endl
     <<"           [-n nlat] [-i c,h] [-o c,h]"<<endl
     <<"           [-m vmin] [-M vmax] [-f flag]"<<endl
     <<"           fitsin_point fitsin_bolo fitsphout [fitsphwout]"<<endl
     <<" -p lp : print level (def=0)"<<endl
     <<" -s samplemin,samplemax : sample range to be treated (def=all)"<<endl
     <<" -w data_window_size : window size for pipe (def=8192)"<<endl
     <<" -a label_coord1 : label fits for alpha/gLong (def=coord1)"<<endl
     <<" -d label_coord2 : label fits for delta/gLat (def=coord2)"<<endl
     <<"          coord1 = alpha or gLong ; coord2 = delta or gLat"<<endl
     <<" -b label_bolomuv : label fits for bolo value (def=boloMuV)"<<endl
     <<" -n nlat : nlat for Healpix sphere (def=128)"<<endl
     <<" -i c,h : coordIn caracteristics (c=G/E h=H/D/R) (def=G,D)"<<endl
     <<" -o c,h : idem -i for coordOut"<<endl
     <<" -m vmin : samples are good if sample value >= vmin"<<endl
     <<" -M vmax : samples are good if sample value <= vmax"<<endl
     <<" -f flag : samples are bad if match flag"<<endl
     <<" fitsin_point : fits file for pointing"<<endl
     <<" fitsin_bolo : fits file for bolo values"<<endl
     <<" fitsphout : fits file for output Healpix sphere"<<endl
     <<" fitsphwout : fits file for output Healpix nFilled sphere (def=no)"<<endl;
}

unsigned long typecoord(char typc=' ',char hd=' ');
unsigned long typecoord(char typc,char hd)
// typc : G=galactiques, E=equatoriales, autres=galactiques
// hd : H=heure, D=degre, R=radian, autres=(heure si typc==E, degre si typc==G)
{
  if(typc!='G' && typc!='E') typc='G';
  if(hd!='H' && hd!='D' && hd!='R') {if(typc=='E') hd='H'; else hd='D';}
 unsigned long rc=TypCoordUndef;
  if(typc=='G') rc |= TypCoordGal;
    else        rc |= TypCoordEq;
  if(hd=='D')        rc |= TypCoordDD;
    else if(hd=='R') rc |= TypCoordRR;
      else           rc |= TypCoordHD;
  return rc;
}

////////////////////////////////////////////////////////////////
int main(int narg, char** arg) {

TOIManager* mgr = TOIManager::getManager();

//-- Decodage arguments
int lp = 0, width = 8192;
int nlat = 128;   // npixel = 12 * nlat^2
bool tflg=false, tmin=false, tmax=false;
r_8 vmin=-1.e30, vmax=1.e30; int_8 badflg=0;
char *label_coord1 = "coord1", *label_coord2 = "coord2"
   , *label_bolomuv = "boloMuV";
long sdeb,sfin;
string fitsphwout = "";
unsigned long tcoorin=typecoord(), tcoorout=typecoord();
int c; char t=' ',h=' ';
while((c = getopt(narg,arg,"hp:s:w:a:d:b:n:i:o:m:M:f:")) != -1) {
  switch (c) {
  case 's' :
    sscanf(optarg,"%ld,%ld",&sdeb,&sfin);
    cout<<"Requested Samples from "<<sdeb<<" , "<<sfin<<endl;
    if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);
    else {cout<<"Bad sample interval "<<endl; exit(2);}
    break;
  case 'w' :
    sscanf(optarg,"%d",&width);
    if(width<=0) width=8192;
    cout<<"Data window size "<<width<<endl;
    break;
  case 'p' :
    sscanf(optarg,"%d",&lp);
    if(lp<0) lp=0;
    break;
  case 'a' :
    label_coord1 = optarg;
    break;
  case 'd' :
    label_coord2 = optarg;
    break;
  case 'b' :
    label_bolomuv = optarg;
    break;
  case 'n' :
    sscanf(optarg,"%d",&nlat);
    if(nlat<0) nlat=128;
    break;
  case 'i' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoorin=typecoord(t,h);
    break;
  case 'o' :
    sscanf(optarg,"%c,%c",&t,&h);
    tcoorout=typecoord(t,h);
    break;
  case 'm' :
    sscanf(optarg,"%lf",&vmin);
    tmin = true;
    break;
  case 'M' :
    sscanf(optarg,"%lf",&vmax);
    tmax = true;
    break;
  case 'f' :
    sscanf(optarg,"%ld",&badflg);
    tflg = true;
    break;
  case 'h' :
  default:
    usage(); exit(1);
    break;
  }
}
if(optind+2>=narg) {usage(); exit(3);}
 
char * fitsin_point = arg[optind];
char * fitsin_bolo = arg[optind+1];
string const fitsphout = arg[optind+2];
if(optind+3<narg) fitsphwout = arg[optind+3];

cout<<">>>> tsttoi2map:"<<endl
    <<"Pipe Window Size "<<width<<endl
    <<"Fits Infile Bolo "<<fitsin_bolo<<endl
    <<"  ...label_bolomuv "<<label_bolomuv<<endl;
cout<<"Fits Infile Pointing "<<fitsin_point<<endl
    <<"  ...label_coord1 "<<label_coord1<<endl
    <<"  ...label_coord2 "<<label_coord2<<endl;
cout<<"Fits Healpix Sphere "<<fitsphout<<endl
    <<"  ...nlat "<<nlat<<endl;
cout<<"Fits Healpix Weight Sphere "<<fitsphwout<<endl;

SophyaInit();
InitTim();

//--------------------------------------------------------------------
try {
//--------------------------------------------------------------------

 // FITS reader
 FITSTOIReader rfitsb(fitsin_bolo);
 int ncolb = rfitsb.getNOut();
 cout<<"Number of columns in fits Infile_bolo : "<<ncolb<<endl;
 if(ncolb<1) exit(-4);

 FITSTOIReader rfitsp(fitsin_point);
 int ncolp = rfitsp.getNOut();
 cout<<"Number of columns in fits Infile_point : "<<ncolp<<endl;
 if(ncolp<2) exit(-5);

 // Creation de la sphere Healpix
 SphereHEALPix<r_8>* sph = new SphereHEALPix<r_8>(nlat);
 cout<<"SphereHEALPix: Type de map : "<<sph->TypeOfMap()<<endl
     <<"               Nombre de pixels : "<<sph->NbPixels()<<endl
     <<"               Nlat : "<<sph->SizeIndex()<<endl;

 // Creation de la sphere de poids Healpix
 SphereHEALPix<r_8>* wsph = NULL;
 if(fitsphwout.size()>0) {
   wsph = new SphereHEALPix<r_8>;
   cout<<"SphereHEALPix Weight Created"<<endl;
 }

 // TOI Processor
 TOI2Map toi2m(sph,wsph);
 cout<<"TOI2Map created"<<endl;
 toi2m.SetEquinox(2000.);
 toi2m.SetCoorIn((TypAstroCoord) tcoorin);
 toi2m.SetCoorOut((TypAstroCoord) tcoorout);
 toi2m.SetTestFlag(tflg,badflg);
 toi2m.SetTestMin(tmin,vmin);
 toi2m.SetTestMax(tmax,vmax);
 toi2m.Print(cout);

 // Definition des tuyaux
 TOISeqBuffered * toicoord1in = new TOISeqBuffered("toi_coord1_in",width);
 // toicoord1in->setDebugLevel(1);
 rfitsp.addOutput(label_coord1,toicoord1in);
 toi2m.addInput("Coord1In",toicoord1in);

 TOISeqBuffered * toicoord2in = new TOISeqBuffered("toi_coord2_in",width);
 // toicoord2in->setDebugLevel(1);
 rfitsp.addOutput(label_coord2,toicoord2in);
 toi2m.addInput("Coord2In",toicoord2in);
 
 TOISeqBuffered * toibolin = new TOISeqBuffered("toi_bolo_in",width);
 // toibolin->setDebugLevel(1);
 rfitsb.addOutput(label_bolomuv,toibolin);
 toi2m.addInput("BoloIn",toibolin);

 // Run
 cout<<"----- FITSReaderTOI::PrintStatus() : -----"<<endl;
 rfitsp.PrintStatus(cout);
 rfitsb.PrintStatus(cout);

 PrtTim("starting threads");
 rfitsp.start();
 rfitsb.start();
 toi2m.start();

 if(lp>1)
   for(int jjjj=0;jjjj<5;jjjj++) {
     cout<<*toicoord1in; 
     cout<<*toibolin; 
     sleep(2);
   }

 mgr->joinAll();
 PrtTim("End threads");

 // Ecriture de la sphere Healpix sur fits
 {
 FitsOutFile sfits(fitsphout);
 cout<<"tsttoi2map: Creating sphere fits file "<<fitsphout<<endl;
 sfits << *sph;
 }

 // Ecriture de la sphere Healpix sur fits
 if(wsph) {
   FitsOutFile swfits(fitsphwout);
   cout<<"tsttoi2map: Creating sphere weight fits file "<<fitsphwout<<endl;
   swfits << *wsph;
 }

 // Nettoyage
 delete sph;
 if(wsph) delete wsph;

//--------------------------------------------------------------------
} catch (PThrowable & exc) {
 cout<<"\ntsttoi2map: Catched Exception \n"<<(string)typeid(exc).name() 
     <<" - Msg= "<<exc.Msg()<<endl;
} catch (const std::exception & sex) {
 cout<<"\ntsttoi2map: Catched std::exception \n" 
     <<(string)typeid(sex).name()<<endl; 
} catch (...) {
 cout<<"\ntsttoi2map: some other exception was caught ! "<<endl;
}
//--------------------------------------------------------------------

exit(0);
}
