#include "array.h"
#include <math.h>
#include "genwproc.h"
#include "toimanager.h"
#include "pexceptions.h"
#include "ctimer.h"

// -------------------------------------------------------------
//   Class GenWindowTOIProcessor : generic processor with window
// -------------------------------------------------------------

////////////////////////////////////////////////////////////////
GenWindowTOIProcessor::GenWindowTOIProcessor(int_4 nbinput,int_4 nboutput
                                   ,int_4 wsz, int_4 wstep, int_4 wsztot)
{
  if(nbinput<1) throw ParmError("GenWindowTOIProcessor::Creator nbinput<1 !");
  if(nboutput<0) nboutput=0;
  if(wsz<2) throw ParmError("GenWindowTOIProcessor::Creator wsz<2 !");
  if(wsz%2==0) {if(wsztot==wsz) wsztot++; wsz++;}
  if(wstep<1) wstep=1;
  if(wsztot<wsz) wsztot = 2*wsz;
  // cas sans intersection entre 2 samples successifs
  if(wstep>=wsz) wsztot = wsz;

  NbInput = nbinput;
  NbOutput = nboutput;
  WSize = wsz;
  WStep = wstep;
  WSizeTot = wsztot;

  CurWtIndex = -1;
  SNbegin = SNend = StartSample = LastFilledSn = -1;
  TotNsCount = TotDecalCount = 0;

  SetDbgLevel();
  SetWSizeLCR();
  SetDefaultValue();

  TVector<r_8> vr8(1);     // CMV+RZ supprimer taille(1) apres correction de 
  TVector<int_8> vi8(1);   // constructeur copie de TArray (taille nulle)
  
  int_4 k;
  for(k=0;k<NbInput;k++) {
    WDataIn.push_back(vr8);
    WFlagIn.push_back(vi8);
    WInFlg.push_back(false);
  }
  if(NbOutput)
    for(k=0;k<NbOutput;k++) {
      WDataOut.push_back(vr8);
      WFlagOut.push_back(vi8);
      WOutFlg.push_back(false);
      WPutOutFlg.push_back(false);
      WPutOutOwnVector.push_back(false);
      OutSample.push_back(0);
    }
}

GenWindowTOIProcessor::~GenWindowTOIProcessor()
{
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::PrintStatus(ostream & os)
{
os<<"\n ------------------------------------------------------ \n" 
  <<" GenWindowTOIProcessor::PrintStatus() - ["<<NbInput<<","<<NbOutput<<"]"<<endl
  <<" WSizeTot="<<WSizeTot<<" WSize="<<GetWSize()<<" WStep= "<<GetWStep()<<endl;
os<<" WindowLCR: "
  <<" L=("<<W0Left<<","<<GetWSize('l')<<")"
  <<" C=("<<W0Center<<","<<GetWSize('c')<<")"
  <<" R=("<<W0Right<<","<<GetWSize('r')<<")"
  <<endl;

TOIProcessor::PrintStatus(os);
os<<"ProcessedSampleCount="<<ProcessedSampleCount()<<" NbDecal="<<TotDecalCount<<endl;
os<<"------------------------------------------------------ "<<endl;
}

////////////////////////////////////////////////////////////////
TVector<r_8> GenWindowTOIProcessor::GetWData(int_4 numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWData : toi out of range !");
  if(!WInFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::GetWData : toi not connected!");
  return (WDataIn[numtoi])(Range(StartWtIndex(),0,WSize));
}

TVector<uint_8> GenWindowTOIProcessor::GetWFlag(int_4 numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWFlag : toi out of range !");
  if(!WInFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::GetWFlag : toi not connected!");
  return (WFlagIn[numtoi])(Range(StartWtIndex(),0,WSize));
}

r_8 * GenWindowTOIProcessor::GetWDataPointer(int_4 numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWDataPointer : toi out of range !");
  if(!WInFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::GetWDataPointer : toi not connected!");
  return (WDataIn[numtoi].Data()+StartWtIndex());

}
 
uint_8 * GenWindowTOIProcessor::GetWFlagPointer(int_4 numtoi)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWFlagPointer : toi out of range !");
  if(!WInFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::GetWFlagPointer : toi not connected!");
  return (WFlagIn[numtoi].Data()+StartWtIndex());
}

void GenWindowTOIProcessor::GetData(int_4 numtoi, int_8 numsample, r_8 & data, uint_8 & flag)
{
  if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetData : toi out of range !");
  if(!WInFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::GetData : toi not connected!");
  int_8 k = numsample-GetWStartSample();
  if ((k<0) || (k >= GetWSize()))
    throw RangeCheckError("GenWindowTOIProcessor::GetData : numsample out of window!");
  k += StartWtIndex();
  data = (WDataIn[numtoi])(k);
  flag = (WFlagIn[numtoi])(k);
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::PutWData(int numtoi,int_8 numsample
                                    ,TVector<r_8>& data,TVector<uint_8>& flag)
{
  if(numtoi<0 || numtoi>=NbOutput || NbOutput<=0)
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi out of range !");
  if(!WOutFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::PutWFlag : toi not connected!");
  if(data.Size()!=flag.Size())
    throw ParmError("GenWindowTOIProcessor::PutWFlag : data.Size()!=flag.Size()!");
  if(data.Size() == 0)
    throw ParmError("GenWindowTOIProcessor::PutWFlag : data.Size()==0 !");
  WDataOut[numtoi].Share(data);
  WFlagOut[numtoi].Share(flag);
  OutSample[numtoi] = numsample;
  WPutOutFlg[numtoi] = true;
  WPutOutOwnVector[numtoi] = false;
}

void GenWindowTOIProcessor::PutWData(int numtoi,int_8 numsample
                                    , r_8 data, uint_8 flag)
{
  if(numtoi<0 || numtoi>=NbOutput || NbOutput<=0)
    throw RangeCheckError("GenWindowTOIProcessor::PutWFlag : toi out of range !");
  if(!WOutFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::PutWFlag : toi not connected!");
  if (!WPutOutOwnVector[numtoi]) {
    WDataOut[numtoi].Realloc(1,BaseArray::SameVectorType,true);
    WFlagOut[numtoi].Realloc(1,BaseArray::SameVectorType,true);
    WPutOutOwnVector[numtoi] = true;
  }
  (WDataOut[numtoi])(0) = data;
  (WFlagOut[numtoi])(0) = flag;
  OutSample[numtoi] = numsample;
  WPutOutFlg[numtoi] = true;
}

/******* cmv routines
void GenWindowTOIProcessor::PutWData(int_4 numtoi,int_8 numsample
                                    ,TVector<r_8>& data,TVector<uint_8>& flag)
{
  if(numtoi<0 || numtoi>=NbOutput || NbOutput<=0)
    throw RangeCheckError("GenWindowTOIProcessor::PutWData : toi out of range !");
  if(!WOutFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::PutWData : toi not connected!");
  if(data.Size()!=flag.Size())
    throw ParmError("GenWindowTOIProcessor::PutWData : data.Size()!=flag.Size()!");
  if(data.Size() == 0)
    throw ParmError("GenWindowTOIProcessor::PutWData : data.Size()==0 !");

  for(int_4 k=0;k<data.Size(); k++)
     putData(numtoi,numsample+k,data(k),flag(k));
}

void GenWindowTOIProcessor::PutWData(int_4 numtoi,int_8 numsample
                                    ,r_8 data, uint_8 flag)
{
  if(numtoi<0 || numtoi>=NbOutput || NbOutput<=0)
    throw RangeCheckError("GenWindowTOIProcessor::PutWData : toi out of range !");
  if(!WOutFlg[numtoi])
    throw ParmError("GenWindowTOIProcessor::PutWData : toi not connected!");

  putData(numtoi,numsample,data,flag);
}
*********/

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::SetWSizeLCR(int_4 wszl,int_4 wszc,int_4 wszr)
// Fenetre a gauche, au centre et a droite du pixel central
// Selon la logique:
//-----------------------------------------------
//                     | pixel central
//                     |
//                     |
//          | wszl | wszc | wszr |
//                     |
//       |           WSize           |
//-----------------------------------------------
// wszc toujours impair SVP.
// Default: 0,0,0 : WSize/2,1,WSize/2
{
 int_4 wsz2 = WSize/2;
 // Default
 if(wszl<=0 && wszc<=0 && wszr<=0) {
   WSizeLeft = WSizeRight = wsz2; WSizeCenter = 1;
   W0Left = 0; W0Center = wsz2;   W0Right = wsz2+1;
   return;
 }

 // Fenetre centrale
 if(wszc<=0)    wszc = 1;
 if(wszc%2==0)  wszc++;
 if(wszc>WSize) wszc = WSize;
 WSizeCenter = wszc;
 W0Center = wsz2 - WSizeCenter/2;

 // Fenetre de gauche
 if(wszl<=0) wszl = WSize;
 W0Left = W0Center - wszl;
 if(W0Left<0) W0Left = 0;
 WSizeLeft = W0Center - W0Left;
 if(WSizeLeft<=0) WSizeLeft = 1;

 // Fenetre de droite
 if(wszr<=0) wszr = WSize;
 W0Right = W0Center + WSizeCenter;
 if(W0Right>=WSize) W0Right = WSize - 1;
 int_4 dum = W0Right + wszr; if(dum>WSize) dum = WSize;
 WSizeRight = dum - W0Right;
 if(WSizeRight<=0) WSizeRight = 1;
}

TVector<r_8> GenWindowTOIProcessor::GetWData(char cw,int_4 numtoi)
// cr='l' fenetre de gauche, 'c' du centre, 'r' de droite, autre = tout
{
 if(numtoi<0 || numtoi>=NbInput)
    throw RangeCheckError("GenWindowTOIProcessor::GetWData(lcr) : toi out of range !");
 if(!WInFlg[numtoi])
   throw ParmError("GenWindowTOIProcessor::GetWData(lcr) : toi not connected!");
 return (WDataIn[numtoi])(Range(StartWtIndex(cw),0,GetWSize(cw)));
}

TVector<uint_8> GenWindowTOIProcessor::GetWFlag(char cw,int_4 numtoi)
// cr='l' fenetre de gauche, 'c' du centre, 'r' de droite, autre = tout
{
 if(numtoi<0 || numtoi>=NbInput)
   throw RangeCheckError("GenWindowTOIProcessor::GetWFlag(lcr) : toi out of range !");
 if(!WInFlg[numtoi])
   throw ParmError("GenWindowTOIProcessor::GetWFlag(lcr) : toi not connected!");
 return (WFlagIn[numtoi])(Range(StartWtIndex(cw),0,GetWSize(cw)));
}

r_8 * GenWindowTOIProcessor::GetWDataPointer(char cw,int_4 numtoi,int_4& n)
// cr='l' fenetre de gauche, 'c' du centre, 'r' de droite, autre = tout
{
 if(numtoi<0 || numtoi>=NbInput)
   throw RangeCheckError("GenWindowTOIProcessor::GetWDataPointe(lcr)r : toi out of range !");
 if(!WInFlg[numtoi])
   throw ParmError("GenWindowTOIProcessor::GetWDataPointer(lcr) : toi not connected!");
 n = GetWSize(cw);
 return (WDataIn[numtoi].Data()+StartWtIndex(cw));

}
 
uint_8 * GenWindowTOIProcessor::GetWFlagPointer(char cw,int_4 numtoi,int_4& n)
// cr='l' fenetre de gauche, 'c' du centre, 'r' de droite, autre = tout
{
 if(numtoi<0 || numtoi>=NbInput)
   throw RangeCheckError("GenWindowTOIProcessor::GetWFlagPointer(lcr) : toi out of range !");
 if(!WInFlg[numtoi])
   throw ParmError("GenWindowTOIProcessor::GetWFlagPointer(lcr) : toi not connected!");
 n = GetWSize(cw);
 return (WFlagIn[numtoi].Data()+StartWtIndex(cw));
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::UserInit(int_8 kstart)
{
  cout<<"GenWindowTOIProcessor::UserInit() Default implementation does nothing"<<endl;
}

void GenWindowTOIProcessor::UserProc(int_8 ks)
{
  cout<<"GenWindowTOIProcessor:UserProc() Default implementation does nothing"<<endl;
}

void GenWindowTOIProcessor::UserEnd(int_8 kend)
{
  cout<<"GenWindowTOIProcessor::UserEnd() Default implementation does nothing"<<endl;
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::init()
{
  cout << "GenWindowTOIProcessor::init" << endl;
  char buff[64];
  int_4 k;
  for(k=0; k<NbInput; k++) {
    sprintf(buff,"in%d", k);
    declareInput(buff);
  }
  if(NbOutput)
    for(k=0; k<NbOutput; k++) {
      sprintf(buff,"out%d",k);
      declareOutput(buff);
    }
  name = "GenWindowTOIProcessor";
  // upExtra = 1; $CHECK a quoi ca sert EA?
}

void GenWindowTOIProcessor::run()
{
  //  TOIManager* mgr = TOIManager::getManager();
  SNbegin = getMinIn();
  SNend   = getMaxIn();

  if(SNend-SNbegin<WSize)
    throw ParmError("GenWindowTOIProcessor::run : sne-snb<WSize !");

  // Allocation des tailles pour les vecteurs
  int_4 kc, nc=0;
  for(kc=0;kc<NbInput;kc++) {
    if( !(WInFlg[kc]=checkInputTOIIndex(kc)) ) continue;
     WDataIn[kc].ReSize(WSizeTot);
     WFlagIn[kc].ReSize(WSizeTot);
     nc++;
  }
  if(nc==0) {
    cerr<<" GenWindowTOIProcessor::run() - No input TOI connected!"<<endl;
    throw ParmError("GenWindowTOIProcessor::run() No input TOI connected!");
  }
  if(NbOutput)
    for(kc=0;kc<NbOutput;kc++) WOutFlg[kc] = checkOutputTOIIndex(kc);

  // Lecture des samples et remplissage des vecteurs
  cout<<"GenWindowTOIProcessor::run() SNRange="<<SNbegin<<" - "<<SNend<<endl; 
  try {
    Timer tm("GenWindowTOIProcessor::run()");
    int_4 wsz2=WSize/2;
    int_8 ksend = SNbegin;
    for(int_8 ks=SNbegin; ks<=SNend; ks+=WStep) {

      Remplissage(ks);
      if(DbgLevel) test_avec_print(ks);

      if(ks == SNbegin) {UserInit(ks); Ecriture();}

      UserProc(ks);
      Ecriture();

      ksend = ks;
      TotNsCount++;
    }
    UserEnd(ksend); Ecriture();
    cout << " GenWindowTOIProcessor::run() - End of processing " << endl;

  } catch(PException & exc) {
    cerr<<"GenWindowTOIProcessor::run Catched Exception "<<(string)typeid(exc).name()
        <<"\n .... Msg= "<<exc.Msg()<<endl;
  }
}

////////////////////////////////////////////////////////////////
void GenWindowTOIProcessor::Remplissage(int_8 ks)
// INPUT:
//   ks : numero du sample CENTRAL
//      samples :   sn-ws/2        sn          sn+ws/2
//      fenetre :   0              ws/2+1      ws-1
// Cette routine doit etre utilisee dans une boucle:
//      for(int k=???; k<???; k+=WStep) .....
// Dans le grand buffer WSizeTot il doit y avoir WSize samples
//      entre ks1=ks-wsz2 et ks2=ks-wsz2  (ks1<=k<=ks2)
// En ENTREE de cette routine :
// - CurWtIndex = c'est l'index du tableau WSizeTot qui doit etre rempli
//                (le dernier index qui a ete rempli par l'appel precedent
//                 est donc CurWtIndex-1)
// - LastFilledSn = c'est le dernier numero de sample qui a ete rempli
//                  dans le tableau WSizeTot par l'appel precedent
{
int_4 wsz2=WSize/2;

// Numero de sample du premier element du tableau WSize
//   (peut etre < SNbegin au debut)
StartSample = ks - wsz2;

// on doit avoir les samples [ks1,ks2] dans le buffer
// (mais selon le step, certain peuvent etre deja remplis)
int_8 ks1 = ks-wsz2, ks2 = ks+wsz2;

if(DbgLevel>1)
  cout<<"DBG-GenWindowTOIProcessor::Remplissage("<<ks
      <<") ["<<ks1<<","<<ks2<<"]"
      <<" CurWtIndex="<<CurWtIndex<<" LastFilledSn="<<LastFilledSn<<endl;

//--------------------------------------------------------
// Premier remplissage ????
// Gestion de la borne inferieure pour le permier sample
//--------------------------------------------------------
if(CurWtIndex<0) {
  if(DbgLevel>1)
    cout<<"DBG-GenWindowTOIProcessor::Remplissage 1ere fois"<<endl; 
  for(int_4 ntoi=0; ntoi<NbInput; ntoi++) {
    if(!WInFlg[ntoi]) continue;
    for(int_8 k=ks1, j=0; k<=ks2; k++, j++) {
      if(k>=SNbegin && k<=SNend) {
        getData(ntoi,k,(WDataIn[ntoi])(j),(WFlagIn[ntoi])(j));
      } else {
        (WDataIn[ntoi])(j) = R8DefVal; (WFlagIn[ntoi])(j) = I8DefVal;
      }
    }
  }
  CurWtIndex = ks2 - ks1 + 1;
  LastFilledSn = ks2;
  return;
}

//--------------------------------------------------------
// Faut-il decaler ????
//--------------------------------------------------------
if(WStep>=WSize) {
  // On ne decale pas puisque entre 2 appels a la routine
  // il faut TOUT recharger (aucun sample deja charge n'est utile)
  CurWtIndex = 0;
} else {
  // Certains samples deja charges sont utiles
  int_8 ifin = ks2 - LastFilledSn + CurWtIndex-1;
  if(ifin >= WSizeTot) { // On decale
    int_8 ideb = CurWtIndex-1 - LastFilledSn + ks1;
    if(DbgLevel>1)
      cout<<"DBG-GenWindowTOIProcessor::Remplissage ... Decalage ideb="<<ideb
          <<" ifin="<<ifin<< endl; 
    for(int_4 ntoi=0; ntoi<NbInput; ntoi++) {
      if(!WInFlg[ntoi]) continue;
      for(int_4 i=ideb, j=0; i<CurWtIndex; i++, j++) {
        (WDataIn[ntoi])(j) = (WDataIn[ntoi])(i);
        (WFlagIn[ntoi])(j) = (WFlagIn[ntoi])(i);
      }
    }
    CurWtIndex = CurWtIndex-ideb;
    TotDecalCount++;
  }
}

// Remplissage des samples utiles pour ks
if(ks1<=LastFilledSn) ks1 = LastFilledSn+1;
if(DbgLevel>1)
  cout<<"DBG-GenWindowTOIProcessor::Normal fill de ["<<ks1<<","<<ks2<<"]"<<endl;
for(int_4 ntoi=0; ntoi<NbInput; ntoi++) {
  if(!WInFlg[ntoi]) continue;
  for(int_4 k=ks1, j=CurWtIndex; k<=ks2; k++, j++) {
    if(k>=SNbegin && k<=SNend) {
      getData(ntoi,k,(WDataIn[ntoi])(j),(WFlagIn[ntoi])(j));
    } else {
      (WDataIn[ntoi])(j) = R8DefVal; (WFlagIn[ntoi])(j) = I8DefVal;
    }
  }
}
CurWtIndex += ks2 - ks1 + 1;
LastFilledSn = ks2;

return;
}

void GenWindowTOIProcessor::Ecriture()
{
  int_8 maxlenout = 0;
  int kc;
  for(kc=0; kc<NbOutput; kc++)
    if(WOutFlg[kc] && WPutOutFlg[kc] && (WDataOut[kc].Size() > maxlenout) )
      maxlenout = WDataOut[kc].Size();

  for(int_8 k=0; k<maxlenout; k++) {
    for(int kc=0; kc<NbOutput; kc++) {
      if(!WOutFlg[kc]) continue;
      if(!WPutOutFlg[kc]) continue;
      if(k>=WDataOut[kc].Size()) continue;
      putData(kc, k+OutSample[kc], (WDataOut[kc])(k), (WFlagOut[kc])(k));      
    }
  }
  for(kc=0; kc<NbOutput; kc++)  WPutOutFlg[kc] = false;
}

////////////////////////////////////////////////////////////////
void  GenWindowTOIProcessor::test_avec_print(int_8 ks)
{
if(DbgLevel<=0) return;

int i,ii=0;
r_8   * datatot = WDataIn[0].Data();
uint_8 * flagtot = WFlagIn[0].Data();
r_8   * data = GetWDataPointer();
uint_8 * flag = GetWFlagPointer();
int_4 nl,nc,nr;
r_8   * datal = GetWDataPointer('l',nl);
uint_8 * flagl = GetWFlagPointer('l',nl);
r_8   * datac = GetWDataPointer('c',nc);
uint_8 * flagc = GetWFlagPointer('c',nc);
r_8   * datar = GetWDataPointer('r',nr);
uint_8 * flagr = GetWFlagPointer('r',nr);

cout<<"-------- ks = "<<ks<<endl;

if(DbgLevel>2) {
  cout<<"datatot = ";
  for(i=0;i<WSizeTot;i++) {ii=0; cout<<" "<<datatot[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"flagtot = ";
  for(i=0;i<WSizeTot;i++) {ii=0; cout<<" "<<flagtot[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
}

cout<<"data = ";
for(i=0;i<GetWSize();i++) {ii=0; cout<<" "<<data[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
cout<<"flag = ";
for(i=0;i<GetWSize();i++) {ii=0; cout<<" "<<flag[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;

if(DbgLevel>2) {
  cout<<"datal = ";
  for(i=0;i<nl;i++) {ii=0; cout<<" "<<datal[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"flagl = ";
  for(i=0;i<nl;i++) {ii=0; cout<<" "<<flagl[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"datac = ";
  for(i=0;i<nc;i++) {ii=0; cout<<" "<<datac[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"flagc = ";
  for(i=0;i<nc;i++) {ii=0; cout<<" "<<flagc[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"datar = ";
  for(i=0;i<nr;i++) {ii=0; cout<<" "<<datar[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
  cout<<"flagr = ";
  for(i=0;i<nr;i++) {ii=0; cout<<" "<<flagr[i]; if(i%10==9) {cout<<endl; ii=1;}}
        if(ii==0) cout<<endl;
}
}
