#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <iostream.h>
#include <string>
#include <list>
#include <map>
#if defined(__KCC__)
using std::string ;
#include <list.h>
#include <map.h>
#endif

#include "strutil.h"
#include "datatypes.h"
#include "fitsimage.h"

#include "nobjmgr.h"
#include "servnobjm.h"
#include "pistdimgapp.h"

#include "pclassids.h"
#include "poly.h"
#include "dvlist.h"
#include "matrix.h"
#include "cvector.h"
#include "zfidu.h"

#include "fft.h"
#include "fct1dfit.h"
#include "fct2dfit.h"

#include "pimgadapter.h"
#include "pipodrw.h"

#include "pihisto.h"
#include "hisprof.h"
#include "pihisto2d.h"
#include "pintuple.h"
#include "pisurfdr.h"  
#include "pintup3d.h"
#include "pigfd1.h"
#include "pigfd2.h"

//  Si le module StarReco++ a ete compile (Pour les StarList, Transfo, etc ...
#ifndef SANS_StarReco 
#include "pistlist.h"
#include "transfo.h"
#endif

//++
// Class	NamedObjMgr
// Lib		PI
// include	nobjmgr.h
//
//	Cette classe fournit les services ncssaire la gestion des objets
//	(l'ensemble des objets PPersist de PEIDA++) au sein du programme
//	d'analyse interactive *piapp* . Elle constitue en outre l'interface
//	entre les fonctions utilisateur et l'application graphique.
//--
//++
// Links	Voir aussi
// PIStdImgApp
// Services2NObjMgr
// PIACmd
//--


// ..................................................................
// ......  Gestion des objets nommes, variables globales ............
struct nobj_item {
  PPersist* obj;
  int num;
  list<int> wrsid;
  bool operator==(nobj_item const& b) const
    { return (this->obj ==  b.obj); }
};

typedef map<string, nobj_item, less<string> > NObjList;

static NObjList* myObjs;
static int fgOInit = 0;
static int myNObj = 0;
static string* lastobj = NULL;

static PIStdImgApp* myImgApp=NULL;
static Services2NObjMgr* servnobjm=NULL;

static string* TmpDir;  // Repertoire pour les compilations / link dynamique

//++
// Titre	Constructeurs
//--
//++
// NamedObjMgr()
//	Constructeur. Les diffrents instantiation de la classe "NamedObjMgr" 
//	dans une mme application crent des objets qui travaillent sur la mme
//	liste d'objets. Les objets de cette classe ne possedent en effet pas
//	de variables membres. 
//--

/* --Methode-- */
NamedObjMgr::NamedObjMgr()
{
if (fgOInit == 0) { 
  myObjs = new NObjList;  
  lastobj = new string(""); 
  char* varenv;
  TmpDir = new string("");
  if ( (varenv=getenv("PEIDA_TMP")) != NULL )  (*TmpDir) = varenv; 
  else if ( (varenv=getenv("TMPDIR")) != NULL )  (*TmpDir) = varenv; 
  int l = (*TmpDir).length();
  if ( (l>0) && ((*TmpDir)[l-1] != '/') )  (*TmpDir) += '/';
  servnobjm = new Services2NObjMgr(NULL, (*TmpDir));
  }
fgOInit++;
}

  
/* --Methode-- */
NamedObjMgr::~NamedObjMgr()
{
fgOInit--;
if (fgOInit == 0)  delete myObjs;
}

//++
// Titre	Mthodes
//--
//++
// void SetImgApp(PIStdImgApp* app)
//	Spcifie l'objet "PIStdImgApp" associ.
// PIStdImgApp* GetImgApp()
//	Accs  l'objet "PIStdImgApp" associ.
//--

/* --Methode-- */
void NamedObjMgr::SetImgApp(PIStdImgApp* app)
{
myImgApp = app;
servnobjm->SetImgApp(app);
}

/* --Methode-- */
PIStdImgApp* NamedObjMgr::GetImgApp()
{
return(myImgApp);
}

/* --Methode-- */
Services2NObjMgr* NamedObjMgr::GetServiceObj()
{
return(servnobjm);
}

//++
// Titre	Gestion de la liste des objets
//--
//++
// void AddObj(PPersist* obj, string& nom)
//	Ajoute l'objet "obj"  la liste, identifi par "nom".
//	Si un objet de mme nom existe, l'ancien objet est renomm en concatnant 
//	un numro  son nom.
// void DelObj(string const& nom, bool fgd=true)
//	Supprime l'objet "nom" de la liste. L'objet est dtruit si "fgd==true"	("delete obj")
// void DelObjects(string const& patt, bool fgd=true)
//	Supprime l'ensemble des objets dont le nom correspond au patron "patt".
//	Le patron peut contenir les caractres "*" et "?" . Les objets sont dtruits si "fgd==true"
// PPersist*  GetObj(string const& nom) 
//	Retourne l'objet identifi par "nom" dans la liste. Retourne "NULL" si "nom" n'est 
//	pas dans la liste.
// void  RenameObj(string const& nom, string& nomnew)
//	Change le nom d'un objet dans la liste.
// string  LastObjName()
//	Renvoie le nom du dernier objet ajout  la liste
//--


/* --Methode-- */
void NamedObjMgr::AddObj(PPersist* obj, string& nom)
{

if (obj == NULL)  return;
// on supprime les blancs de debut et de fin 
size_t p = nom.find_first_not_of(" ");
if(p>nom.length()) {
  nom = "";
} else {
  nom = nom.substr(p);
  p = nom.find_first_of(" ");
  if(p>nom.length()) p=nom.length();
  nom = nom.substr(0, p);
}

myNObj++;
if (nom.length() < 1) nom = servnobjm->PClassIdToShortClassName(obj->ClassId());
NObjList::iterator it = myObjs->find(nom);
if (it != myObjs->end()) { // l'objet existe deja
  char strg[16];
  sprintf(strg, "%d", myNObj);
  string nomnew = nom + strg;
  RenameObj(nom, nomnew);
  }

(*lastobj) = nom;
nobj_item no;
no.obj = obj;
no.num =  myNObj;
(*myObjs)[(*lastobj)] = no;
if (myImgApp)  { 
  string str = (*lastobj) + "    (T= " + servnobjm->PClassIdToClassName(obj->ClassId()) + ")" ;
  (myImgApp->ObjMgrW())->AddObj(str.c_str(), myNObj+1000);
}
 
cout << "NamedObjMgr::AddObj()  Object " << (*lastobj) << " ( "  
     << servnobjm->PClassIdToClassName(obj->ClassId()) << " ) added (Total= " << myObjs->size() << ")" << endl;  
return;
}

/* --Methode-- */
void NamedObjMgr::DelObj(string const& nom, bool fgd)
{
NObjList::iterator it = myObjs->find(nom);
if (it == myObjs->end()) return; 
list<int>::iterator ii;
if (myImgApp) {
//DBG  cerr << " *DBG* NamedObjMgr::DelObj Sz= " << (*it).second.wrsid.size() << endl; 
  for(ii=(*it).second.wrsid.begin(); ii != (*it).second.wrsid.end(); ii++) 
      myImgApp->DelWRsId((*ii));
  (myImgApp->ObjMgrW())->DelObj((*it).second.num+1000);
}
if (fgd) delete (*it).second.obj;
myObjs->erase(it);
if (fgd) cout << "NamedObjMgr::DelObj()  Object " << nom << " deleted (Total= " << myObjs->size() << ")" << endl;
else  cout << "NamedObjMgr::DelObj()  Object " << nom << " removed (Total= " << myObjs->size() << ")" << endl;
return;  
}

/* --Methode-- */
void NamedObjMgr::DelObjects(string const& patt, bool fgd)
{
NObjList::iterator it;  
list<string> odel;
string cn;
for(it = myObjs->begin(); it != myObjs->end(); it++) {
  cn = (*it).first;
  if (csh_parse(cn.c_str(), patt.c_str()) != 0)  odel.push_back(cn);
  }
list<string>::iterator ii;
for(ii=odel.begin(); ii != odel.end(); ii++)  DelObj(*ii, fgd);
}

/* --Methode-- */
PPersist* NamedObjMgr::GetObj(string const& nom)
{
NObjList::iterator it = myObjs->find(nom);
if (it == myObjs->end()) return(NULL); 
return((*it).second.obj);
}

/* --Methode-- */
void NamedObjMgr::RenameObj(string const& nom, string& nomnew)
{
PPersist* obj = GetObj(nom);
if (obj == NULL) return;
DelObj(nom, false);
AddObj(obj, nomnew);
return;
}

/* --Methode-- */
string NamedObjMgr::LastObjName()
{
return((*lastobj));
}

//++
// Titre	Entres-Sorties (I/O) sur les objets
//--
//++
// void  ReadObj(PInPersist& s, int num=-1)
//	Lit l'objet  partir avec le tag numro "num" dans le flot "PInPersist s"
//	et l'ajoute  la liste. Si "num" est ngatif, tous les objets prsents
//	sur le flot "s" sont crs et ajouts  la liste.
// void  ReadObj(string const & nomppf, string nobj="")
//	Lit le premier objet  partir du fichier PPF "nomppf". L'objet est ajout
//	 la liste avec le nom "nobj". Si "nobj" est une chane vide, un nom est 
//	compos  partir du nom de fichier. 
//--

/* --Methode-- */
void NamedObjMgr::ReadObj(string const & flnm, string nobj)
{
PPersist* obj=NULL;
bool ok = true;

TRY{
  PInPersist pis(flnm);
  obj = PPersistMgr::ReadObject(pis);
  if (obj == NULL) ok = false;
}  CATCH(merr) 
  { printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;

if (!ok)  return;
if (nobj.length()<1)  nobj = servnobjm->FileName2Name(flnm);
AddObj(obj, nobj);
return;
}

/* --Methode-- */
void NamedObjMgr::ReadObj(PInPersist& s, int num)
{
int i, cid, key, ln;
int n0, n1;
bool ok = true;
PPersist* obj=NULL;
string nom;

if ( (s.NbTags() < 1) || (num >= s.NbTags()) ) {
  if (num >= 0) { 
    printf("NamedObjMgr::ReadObj(PInPersist, %d) Error! NbTags = %d \n", num, s.NbTags());
    return;
  }
  TRY {
    obj = PPersistMgr::ReadObject(s);
    if (obj == NULL)  ok = false;
  }  CATCH(merr) { 
    printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
    ok = false; 
  } ENDTRY;
  if (!ok)  return;
  nom = "";
  AddObj(obj, nom);
}
 
if (num < 0) { n0 = 0; n1 = s.NbTags(); } 
else { n0 = num; n1 = num+1; }
for(i=n0; i<n1; i++) {
  key = s.TagKey(i, cid, ln); 
  if (ln <= 0)  nom = "";
  else nom = s.TagName(i);
  s.GotoTag(i);
  TRY {
    obj = PPersistMgr::ReadObject(s);
    if (obj == NULL)  ok = false;
  }  CATCH(merr) { 
    printf("NamedObjMgr::ReadObj()/Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
    ok = false; 
  } ENDTRY;
  if (ok)  AddObj(obj, nom);  
}

return;
}
/* --Methode-- */
void NamedObjMgr::ReadAll(string const & flnm)
{
bool ok = true;
PPersist* obj=NULL;

PInPersist* ppin;
TRY{
  ppin = new PInPersist(flnm);
  if (ppin->NbTags() < 1)  obj = PPersistMgr::ReadObject((*ppin));
  else obj = NULL;
}  CATCH(merr) 
  { printf("NamedObjMgr::ReadAll()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;

if (!ok)  return;
if (obj) {
  string nom = servnobjm->FileName2Name(flnm);
  AddObj(obj, nom);
  }
else ReadObj((*ppin), -1);
delete ppin;
return;
}

/* --Methode-- */
void NamedObjMgr::ReadFits(string const & flnm, string nobj)
{
bool ok = true;
RzImage* obj;

TRY{
//  obj = RzReadFits((char*)flnm.c_str(), ImgOffX, ImgOffY, ImgSizX, ImgSizY, ImgBitSgn);
  obj = RzReadFits((char*)flnm.c_str());
  if (obj == NULL)   ok = false;
}  CATCH(merr) { 
  printf("NamedObjMgr::ReadFITS(_Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr)); 
  ok = false; 
} ENDTRY;
if (ok) { 
  if (nobj.length()<1)  nobj = servnobjm->FileName2Name(flnm);
  AddObj((PPersist*)obj, nobj);
}
return;
}


static int key_for_write = 5000;
/* --Methode-- */
void NamedObjMgr::SaveObj(string const& nom, POutPersist& s)
{
PPersist* obj=NULL;
obj = GetObj(nom);
if (obj == NULL)  return;
printf("NamedObjMgr::SaveObj(%s, )  (Type=%s) \n",  
       nom.c_str(), servnobjm->PClassIdToClassName(obj->ClassId()));
string nm = nom;  // A cause de const 
obj->Write(s,  key_for_write, nm);
key_for_write++;
return; 
}

/* --Methode-- */
void NamedObjMgr::SaveAll(string const& flnm)
{
bool ok = true;

POutPersist* pout;
TRY{
  pout = new POutPersist(flnm);
}  CATCH(merr) 
  { printf("NamedObjMgr::SaveAll()/Error Exception= %ld (%s) \n", 
    (long)merr, PeidaExc(merr));  ok = false; } ENDTRY;
if (!ok)  return;
NObjList::iterator it; 
for(it = myObjs->begin(); it != myObjs->end(); it++) SaveObj((*it).first, (*pout));
delete pout;
return;
}

/* --Methode-- */
void NamedObjMgr::SaveFits(string const& nom, string const & flnm)
{
PPersist* obj=NULL;
obj = GetObj(nom);
if (obj == NULL)  return;

int cid = obj->ClassId();
if ( (cid < ClassId_Image) && ( cid > ClassId_Image+kr_8) ) {
  printf("NamedObjMgr::SaveFits(%s, ) Error - Object is NOT an image (type=%s) \n",
         nom.c_str(), servnobjm->PClassIdToClassName(obj->ClassId()) );
  return;
}

RzImage* pim = (RzImage*)obj;
TRY { 
  switch (pim->PixelType()) {
    case kuint_2:
      {
      FitsImageU2 imgfits(*pim);
      imgfits.Save(flnm); 
      }
    break;
    case kint_2:
      {
      FitsImageI2 imgfits(*pim);
      imgfits.Save(flnm); 
      }
      break;
    case kint_4:
      {
      FitsImageI4 imgfits(*pim);
      imgfits.Save(flnm); 
      }
      break;
    case kr_4:
      {
      FitsImageR4 imgfits(*pim);
      imgfits.Save(flnm); 
      }
      break;
    default :
      printf("NamedObjMgr::SaveObj()/Error Type d'image (%d) NON supporte en FITS \n", 
             (int)pim->PixelType());
      break;
  }
}  CATCH(merr) { 
  printf("NamedObjMgr::SaveObj()/Error Exception= %ld (%s) \n", (long)merr, PeidaExc(merr));
} ENDTRY;

return;
}


/* --Methode-- */
void NamedObjMgr::ListObjs()
{
int k;
PPersist* obj=NULL;
string ctyp;
char strg[256];
int cid;

cout << "NamedObjMgr::ListObjs() NObjs= " << myObjs->size() << "\n" ;
NObjList::iterator it;  k = 0;
for(it = myObjs->begin(); it != myObjs->end(); it++) {
  obj = (*it).second.obj;

  cid = obj->ClassId();
  ctyp = servnobjm->PClassIdToClassName(cid);
  sprintf(strg, "%2d/ %16s : %s", k, servnobjm->PClassIdToClassName(obj->ClassId()), ((*it).first).c_str()); 
  ctyp = strg;
  cout << ctyp << "\n" ;
  k++;
}
cout << endl;
return;
}

/* --Methode-- */
void NamedObjMgr::PrintObj(string const& nom)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::PrintObj() Error , Pas d'objet de nom " << nom << endl; 
  return;
}

int cid = obj->ClassId();
string ctyp = servnobjm->PClassIdToClassName(cid);

cout << "NamedObjMgr::PrintObj(" << nom << ") Type: " << ctyp 
     << "Cid= " << cid << ")" << endl;

int i4dum;
switch (cid) {
  case ClassId_Poly1 : 
    ((Poly*)obj)->Print(cout);
    break;
  case ClassId_Poly2 : 
    ((Poly2*)obj)->Print(cout);
    break;
  case ClassId_Matrix : 
  case ClassId_Vector : 
    cout << *((Matrix*)obj);
    break;

  case ClassId_DVList : 
    ((DVList*)obj)->Print();
    break;

  case ClassId_Histo1D : 
    ((Histo*)obj)->Print(60);
    break;
  case ClassId_Histo2D : 
    ((Histo2D*)obj)->Print();
    break;
  case ClassId_HProf : 
    ((HProf*)obj)->Print(60);
    break;
  case ClassId_NTuple : 
    cout << ((NTuple*)obj)->Info();
    cout << *((NTuple*)obj);
    break;
  case ClassId_GeneralFitData : 
    ((GeneralFitData*)obj)->PrintStatus();
    i4dum = ((GeneralFitData*)obj)->NData();
    if(i4dum>0) {
      ((GeneralFitData*)obj)->PrintData(0);
      ((GeneralFitData*)obj)->PrintData(i4dum-1);
    }
    break;

  case ClassId_Image : 
  case ClassId_Image + kuint_1 : 
  case ClassId_Image + kint_1 : 
  case ClassId_Image + kuint_2 : 
  case ClassId_Image + kint_2 : 
  case ClassId_Image + kuint_4 : 
  case ClassId_Image + kint_4 : 
  case ClassId_Image + kr_4 : 
  case ClassId_Image + kr_8 : 
    ((RzImage*)obj)->Print();
    break;

  case ClassId_ZFidu : 
    ((ZFidu*)obj)->Print();
    break;

#ifndef SANS_StarReco 
  case ClassId_StarList : 
    ((StarList*)obj)->Print(10, 1, 50);
    break;
  case ClassId_Transfo : 
    ((Transfo*)obj)->Print(cout);
    break;
  case ClassId_PSF : 
    break;

  case ClassId_Star + BStar_Type : 
  case ClassId_Star + RzStar_Type : 
  case ClassId_Star + PSFStar_Type : 
  case ClassId_Star + MCStar_Type : 
  case ClassId_Star + CircRFixStar_Type : 
  case ClassId_Star + PSFDHStar_Type : 
  case ClassId_Star + PSFSEStar_Type : 
  case ClassId_Star + MCDHStar_Type : 
    ((BStar*)obj)->Print(1);
    break;
#endif

// - Ajout objet PPF
  default:
    break;
  }
cout << endl;
return;
}

/* --Methode-- */
void NamedObjMgr::DisplayObj(string const& nom, string dopt) 
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplayObj() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return;
 
int cid = obj->ClassId();
string ctyp = servnobjm->PClassIdToClassName(cid);

int wrsid = 0;
bool fgsr = true;
// On veut tracer une ligne pour vecteur et histo1D par defaut
if ( (cid == ClassId_Vector) || (cid == ClassId_Histo1D) )  dopt = "thinline," + dopt;
else if(cid == ClassId_HProf)  dopt = "fcirclemarker5," + dopt;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

switch (cid) {
  case ClassId_Poly1 : 
  case ClassId_Poly2 : 
  case ClassId_DVList : 
    cout << "NamedObjMgr::DisplayObj() Error , Pas de display pour " << ctyp << endl; 
    break;

  case ClassId_Matrix : 
    {
    wrsid = myImgApp->DispImage(new POMatrixAdapter((Matrix*)obj, false), nom, opt);
    break;
    }
  case ClassId_Vector : 
    {
    PIYfXDrawer* dr = new   PIYfXDrawer( new POVectorAdapter((Vector*)obj, false), NULL, true);
    wrsid = myImgApp->DispScDrawer( (PIDrawer*)dr, nom, opt);    
    break;
    }
  case ClassId_Histo1D : 
    {
    PIHisto* pih = new PIHisto(((Histo*)obj), false);
    wrsid = myImgApp->DispScDrawer( (PIDrawer*)pih, nom, opt);
    break;
    }
  case ClassId_Histo2D : 
    {
    PIHisto2D* pih2 = new PIHisto2D(((Histo2D*)obj), false);
    wrsid = myImgApp->DispScDrawer( (PIDrawer*)pih2, nom, opt, nom, true);
    break;
    }
  case ClassId_HProf : 
    {
    PIHisto* pihp = new PIHisto(((Histo*)obj), false);
    wrsid = myImgApp->DispScDrawer( (PIDrawer*)pihp, nom, opt);
    break;
    }
  case ClassId_GeneralFitData : 
    {
    GeneralFitData* gfd = (GeneralFitData*)obj;
    if(gfd->NVar()==1) {
      PIGenFitDat* pigfd = new PIGenFitDat(gfd,false);
      wrsid = myImgApp->DispScDrawer((PIDrawer*)pigfd,nom,opt);
    } else if(gfd->NVar()==2) {
      PIGenFitDat3D* pigfd = new PIGenFitDat3D(gfd,false);
      wrsid = myImgApp->Disp3DDrawer(pigfd,nom,opt);
    } else {
      cout<<"NamedObjMgr::DisplayObj() Error , Utilisez DisplayGFD pour GeneralFitData "
          <<ctyp<<"\n de "<<gfd->NVar()<<" variables (!=1,2)"<<endl; 
    }
    break;
    }

  case ClassId_NTuple : 
    cout << "NamedObjMgr::DisplayObj() Error , Utilisez DisplayNt pour NTuple " << ctyp << endl; 
    break;

  case ClassId_Image : 
  case ClassId_Image + kuint_1 : 
  case ClassId_Image + kint_1 : 
  case ClassId_Image + kr_8 : 
  case ClassId_Image + kuint_4 : 
    wrsid = myImgApp->DispImage(new RzImageAdapter((RzImage*)obj, false), nom, opt);
    break;
  case ClassId_Image + kuint_2 : 
    wrsid = myImgApp->DispImage(new ImageAdapter<uint_2>((ImageU2*)obj, false), nom, opt);
    break;
  case ClassId_Image + kint_2 : 
    wrsid = myImgApp->DispImage(new ImageAdapter<int_2>((ImageI2*)obj, false), nom, opt);
    break;
  case ClassId_Image + kint_4 : 
    wrsid = myImgApp->DispImage(new ImageAdapter<int_4>((ImageI4*)obj, false), nom, opt);
    break;
  case ClassId_Image + kr_4 : 
    wrsid = myImgApp->DispImage(new ImageAdapter<r_4>((ImageR4*)obj, false), nom, opt);
    break;

  case ClassId_ZFidu : 
    {
    int nx = ((ZFidu*)obj)->XMax() -  ((ZFidu*)obj)->XMin();
    int ny = ((ZFidu*)obj)->YMax() -  ((ZFidu*)obj)->YMin();
    if (nx > 1024)  nx = 1024;
    if (ny > 1024)  ny = 1024;
    if (nx < 128)  nx = 128;
    if (ny < 128)  nx = 128;
    ImageU2* msk = ((ZFidu*)obj)->MakeBadZoneMask(nx, ny);
    myImgApp->DispImage(new ImageAdapter<uint_2>((ImageU2*)msk, true), nom, opt);
    break;
    }

#ifndef SANS_StarReco 
  case ClassId_StarList : 
    {
    PIStarList* pist = new PIStarList(((StarList*)obj), false);
    wrsid = myImgApp->DispScDrawer( (PIDrawer*)pist, nom, opt);
    break;
    }

  case ClassId_Transfo : 
  case ClassId_PSF : 
  case ClassId_Star + BStar_Type : 
  case ClassId_Star + RzStar_Type : 
  case ClassId_Star + PSFStar_Type : 
  case ClassId_Star + MCStar_Type : 
  case ClassId_Star + CircRFixStar_Type : 
  case ClassId_Star + PSFDHStar_Type : 
  case ClassId_Star + PSFSEStar_Type : 
  case ClassId_Star + MCDHStar_Type : 
    cout << "NamedObjMgr::DisplayObj() Error , Pas de display pour " << ctyp << endl; 
    break;
#endif

// - Ajout objet PPF
  default:
    break;
  }

if(wrsid != 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }
if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}


/* --Methode-- */
void NamedObjMgr::DisplayNT(string const& nom, string& nmx, string& nmy, string& nmz,
                            string& erx, string& ery, string& erz, string dopt)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplayNT() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return; 

int cid = obj->ClassId();
if (cid != ClassId_NTuple) { 
  string ctyp = servnobjm->PClassIdToClassName(cid);
  cout << "NamedObjMgr::DisplayNT() Error , Objet n'est pas un NTuple " << ctyp << endl; 
  return;
  }

int wrsid = 0;
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

if (nmz.length()>0)  { // Display 3D
  PINTuple3D* pin = new PINTuple3D(((NTuple*)obj), false);
  pin->SelectXYZ(nmx.c_str(), nmy.c_str(), nmz.c_str());
  pin->SelectErrBar(erx.c_str(), ery.c_str(), erz.c_str());
  string titre = nmz + "%" + nmy + "%" + nmz;
  wrsid = myImgApp->Disp3DDrawer(pin, nom, opt, titre);  
}
else {
  PINTuple* pin = new PINTuple(((NTuple*)obj), false);
  pin->SelectXY(nmx.c_str(), nmy.c_str());
  pin->SelectErrBar(erx.c_str(), ery.c_str());
  string titre = nmy + "%" + nmz;
  wrsid = myImgApp->DispScDrawer( (PIDrawer*)pin, nom, opt, titre);
  }

if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }

if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::DisplayGFD(string const& nom, string& numvarx, string& numvary, string& err, string dopt)
//	Pour le display 2D ou 3D d'un ``GeneralFitData''.
//|  nom = nom de l'objet GeneralFitData a representer.
//|  numvarx = numero (nombre entier) de la 1ere variable d'abscisse.
//|  numvary = numero (nombre entier) de la 2sd variable d'abscisse (3D).
//|            Pour le display 2D, numvary="" string vide.
//|  err = qu'elles erreurs faut il representer ?
//|      - 2D : x y xy   (display y=f(x))
//|      - 3D : x y z xy xz yz xzy   (display z=f(x,y))
//|      Ceci n'est suivi que si on a PI_NotDefLineAtt, sinon toutes
//|      les barres d'erreurs sont representees.
//|  opt = options generales pour le display.
{
PPersist* obj=GetObj(nom);
if(obj == NULL)
  {cout << "NamedObjMgr::DisplayGFD() Error , Pas d'objet de nom " << nom << endl; 
   return;}
if(!myImgApp) return; 
int cid = obj->ClassId();
if(cid != ClassId_GeneralFitData)
  {string ctyp = servnobjm->PClassIdToClassName(cid);
   cout<<"NamedObjMgr::DisplayGFD() Error , Objet n'est pas un GeneralFitData "<<ctyp<<endl; 
   return;}

// Decodage des options classiques
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
// Decodage des erreurs a representer
bool errx=false, erry=false, errz=false;
if(err.length()>0) {
  for(int i=0;i<err.length();i++)
    if     (err[i]=='x' || err[i]=='X') errx = true;
    else if(err[i]=='y' || err[i]=='Y') erry = true;
    else if(err[i]=='z' || err[i]=='Z') errz = true;
}
// Decodage des numeros de variables en abscisse
int numvx=-1, numvy=-1;
if(numvarx.length()>0) numvx = atoi(numvarx.c_str());
if(numvary.length()>0) numvy = atoi(numvary.c_str());

int wrsid = 0;
if(numvy>=0) {              // Display 3D
  PIGenFitDat3D* pigfd = new PIGenFitDat3D(((GeneralFitData*)obj),false);
  pigfd->SelectXY(numvx,numvy);
  pigfd->SelectErrBar(errx,erry,errz);
  wrsid = myImgApp->Disp3DDrawer(pigfd,nom,opt);  
} else {                   // Display 2D
  PIGenFitDat* pigfd = new PIGenFitDat(((GeneralFitData*)obj),false);
  pigfd->SelectX(numvx);
  pigfd->SelectErrBar(errx,erry);
  wrsid = myImgApp->DispScDrawer((PIDrawer*)pigfd,nom,opt);
}

if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
}
if (fgsr) myImgApp->RestoreGraphicAtt(); 
return;
}

/* --Methode-- 
void NamedObjMgr::DisplayImage(string const& nom, string dopt)
{
  cout << "NamedObjMgr::DisplayImage() a faire ! " << endl;
}
*/


/* --Methode-- */
void NamedObjMgr::DisplaySurf3D(string const& nom, string dopt)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplaySurf3D() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return; 

int cid = obj->ClassId();


P2DArrayAdapter* arr=NULL;
switch (cid) {
  case ClassId_Matrix : 
    arr = new POMatrixAdapter((Matrix*)obj, false);
    break;
  case ClassId_Histo2D : 
    arr = new POH2DAdapter((Histo2D*)obj, false);
    break;
  case ClassId_Image : 
  case ClassId_Image + kuint_1 : 
  case ClassId_Image + kint_1 : 
  case ClassId_Image + kr_8 : 
  case ClassId_Image + kuint_4 : 
    arr = new RzImageAdapter((RzImage*)obj, false);
    break;
  case ClassId_Image + kuint_2 : 
    arr = new ImageAdapter<uint_2>((ImageU2*)obj, false);
    break;
  case ClassId_Image + kint_2 : 
    arr = new ImageAdapter<int_2>((ImageI2*)obj, false);
    break;
  case ClassId_Image + kint_4 : 
    arr = new ImageAdapter<int_4>((ImageI4*)obj, false);
    break;
  case ClassId_Image + kr_4 : 
    arr = new ImageAdapter<r_4>((ImageR4*)obj, false);
    break;

  default :
    cout << "NamedObjMgr::DisplaySurf3D() Error , Pas de display pour " << servnobjm->PClassIdToClassName(cid) << endl; 
    return;
  }

if (!arr) return;
if ((arr->XSize() > 250) || (arr->YSize() > 250)) {
  cout << "NamedObjMgr::DisplaySurf3D() Error , 2D-Array(" <<  arr->XSize() 
       << "x" << arr->YSize() << ") trop grand (max=250x250)" << endl; 
  delete arr;
  return;
  }

int wrsid = 0;
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
if (cid == ClassId_Histo2D)  arr->DefineXYCoordinates(0., 0., 1., 1.);
PISurfaceDrawer* sdr = new PISurfaceDrawer(arr, true, true, true);
wrsid = myImgApp->Disp3DDrawer(sdr, nom, opt);
if(wrsid >= 0) {
  NObjList::iterator it = myObjs->find(nom);
  if (it == myObjs->end()) return; 
  (*it).second.wrsid.push_back(wrsid);
  }

if (fgsr) myImgApp->RestoreGraphicAtt();  
return; 
}


/* --Methode-- */
void NamedObjMgr::SetGraphicAttributes(string gratt)
{
bool fg = false;
servnobjm->DecodeDispOption(gratt, fg);
}
/* --Methode-- */
void NamedObjMgr::SetGraphicWinZone(int nzx, int nzy, bool fcr)
{
if (!myImgApp)  return; 
if (fcr)  myImgApp->CreateGraphWin(nzx, nzy);
else myImgApp->SetZone(nzx, nzy);
}

/* --Methode-- */
void NamedObjMgr::DisplayPoints2D(string const& nom,  string& expx, string& expy,
                                  string& experrx, string& experry,
                                  string& expcut, string dopt)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplayPoints2D() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return; 

// Creation NTuple 
char* ntn[4] = {"expx","expy","expex","expey",};
NTuple* nt = NULL;
bool haserr = false;

if ( (experrx.length() > 0 ) && (experry.length() > 0 ) ) {  haserr = true;   nt = new NTuple(2, ntn); }
else {  haserr = false;  experrx = experry = "0."; nt = new NTuple(2, ntn); }

servnobjm->Nobj_ComputeExpressions(obj, expx, expy, experrx, experry, expcut, nt, NULL, NULL);

if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::DisplayPoints2D() Warning  Zero points satisfy cut !" << endl; 
  delete nt;
  return;
  }

// nt->Show();
// nt->Print(0,10);
PINTuple* pin = new PINTuple(nt, true);
pin->SelectXY(ntn[0], ntn[1]);
if ( haserr ) pin->SelectErrBar(ntn[2], ntn[3]);

bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
string titre =  nom + ":" + expy + "%" + expx;
myImgApp->DispScDrawer( (PIDrawer*)pin, titre, opt);
if (fgsr) myImgApp->RestoreGraphicAtt();
return;
}

/* --Methode-- */
void NamedObjMgr::DisplayPoints3D(string const& nom,  string& expx, string& expy, string& expz,
                                  string& expcut, string dopt)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::DisplayPoints3D() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

char* ntn[3] = {"expx","expy","expz"};
NTuple* nt = new NTuple(3,ntn);  // Creation NTuple

string expwt = "1.";
servnobjm->Nobj_ComputeExpressions(obj, expx, expy, expz, expwt, expcut, nt, NULL, NULL);

if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::DisplayPoints3D() Warning  Zero points satisfy cut !" << endl;
  delete nt;
  return;
  }
nt->Show();
nt->Print(0,10);
PINTuple3D* pin = new PINTuple3D(nt, true);
pin->SelectXYZ(ntn[0], ntn[1], ntn[2]);
bool fgsr = true;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);

// Pour plot a partir de DispScDrawer
// string nomdisp = "_NT3D_";
// myImgApp->DispScDrawer( (PIDrawer*)pin, nomdisp, opt);
// Pour plot a partir de Disp3DDrawer
string titre =  nom + ":" + expy + "%" + expx;
myImgApp->Disp3DDrawer(pin, titre, opt);

if (fgsr) myImgApp->RestoreGraphicAtt();
return;
}

/* --Methode-- */
void NamedObjMgr::ProjectH1(string const& nom, string& expx, string& expwt, string& expcut, string& nomh1, string dopt)
{
PPersist* obj=GetObj(nom);

if (obj == NULL) {
  cout << "NamedObjMgr::ProjectH1() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

Histo* h1 = NULL;
NTuple* nt = NULL;
PPersist* oh = NULL; 
if (nomh1.length() > 0) oh=GetObj(nomh1);
else nomh1 = "H1Proj";
if ( (oh != NULL) && (oh->ClassId() == ClassId_Histo1D) )  h1 = (Histo*)oh;  // Pas de remise a zero ! h1->Zero();
else {
  char* ntn[2]= {"hxval", "hwt"};
  nt = new NTuple(2,ntn);  // Creation NTuple 
  }
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obj, expx, expwt, expz, expwt, expcut, nt, h1, NULL);

if ((!h1) && (!nt)) return;
if (!h1) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectH1() Warning  Zero points satisfy cut !" << endl; 
    delete nt;
    return;
    }
  float xmin, xmax;
  nt->GetMinMax(0, xmin, xmax);
  h1 = new Histo(xmin, xmax, 100);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    h1->Add(xn[0], xn[1]);    
    }
  delete nt;
  AddObj(h1, nomh1);
  }

DisplayObj(nomh1, dopt);
return;  
}

/* --Methode-- */
void NamedObjMgr::ProjectH2(string const& nom, string& expx, string& expy, string& expwt, string& expcut, 
                            string& nomh2, string dopt)
{
PPersist* obj=GetObj(nom);

if (obj == NULL) {
  cout << "NamedObjMgr::ProjectH2() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

Histo2D* h2 = NULL;
NTuple* nt = NULL;
PPersist* oh = NULL;
if (nomh2.length() > 0)  oh=GetObj(nomh2);
else nomh2 = "H2Proj";
if ( (oh != NULL) && (oh->ClassId() == ClassId_Histo2D) )  h2 = (Histo2D*)oh;  // Pas de remise a zero ! h2->Zero();
else {
  char* ntn[3]= {"hxval", "hyval", "hwt"};
  nt = new NTuple(3,ntn);  // Creation NTuple 
  }
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obj, expx, expy, expwt, expwt, expcut, nt, NULL, h2);

if ((!h2) && (!nt)) return;
if (!h2) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectH2() Warning  Zero points satisfy cut !" << endl; 
    delete nt;
    return;
    }
  float xmin, xmax, ymin, ymax;
  nt->GetMinMax(0, xmin, xmax);
  nt->GetMinMax(0, ymin, ymax);
  h2 = new Histo2D(xmin, xmax, 50, ymin, ymax, 50);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    h2->Add(xn[0], xn[1], xn[2]);    
    }
  delete nt;
  AddObj(h2, nomh2);
  }

DisplayObj(nomh2, dopt);
return;  

}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::ProjectHProf(string const& nom, string& expx, string& expy, string& expwt, string& expcut, 
                              string& nomprof, string dopt)
//	Pour remplir un ``GeneralFitData'' a partir de divers objets:
//| nom = nom de l'objet a projeter dans un HProf.
//| expx = expression X de definition du bin.
//| expy = expression Y a additionner dans le bin.
//| expwt = expression W du poids a additionner.
//| expcut = expression du test de selection.
//| nomprof = nom du HProf engendre (optionnel). Si l'objet n'existe pas
//|           les limites Xmin,Xmax sont calculees automatiquement.
//|           sinon ce sont celles de l'objet preexistant.
//| opt = options generales pour le display.
{
PPersist* obj=GetObj(nom);

if (obj == NULL) {
  cout << "NamedObjMgr::ProjectHProf() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

HProf* hprof = NULL;
NTuple* nt = NULL;
PPersist* oh = NULL;
if (nomprof.length() > 0)  oh=GetObj(nomprof);
else nomprof = "ProfProj";
if( (oh!=NULL) && (oh->ClassId()== ClassId_HProf) )  hprof = (HProf*)oh;
else {
  char* ntn[3]= {"hxval", "hyval", "hwt"};
  nt = new NTuple(3,ntn);  // Creation NTuple 
}
string expz = "0.";
servnobjm->Nobj_ComputeExpressions(obj, expx, expy, expwt, expwt, expcut, nt, NULL, NULL, hprof);

if((!hprof) && (!nt)) return;
if(!hprof) {
  if (nt->NEntry() < 1) {
    cout << "NamedObjMgr::ProjectHProf() Warning  Zero points satisfy cut !" << endl;
    delete nt;
    return;
  }
  float xmin, xmax;
  nt->GetMinMax(0, xmin, xmax);
  hprof = new HProf(xmin, xmax, 100);
  int k;
  float* xn;
  for(k=0; k<nt->NEntry(); k++)    {
    xn = nt->GetVec(k);
    hprof->Add(xn[0], xn[1], xn[2]);
    }
  delete nt;
  AddObj(hprof, nomprof);
  }
hprof->UpdateHisto();

DisplayObj(nomprof, dopt);
return;
}

/* --Methode-- */
void NamedObjMgr::FillVect(string const& nom, string& expx, string& expcut, string& nomvec, string dopt)
{
PPersist* obj=GetObj(nom);

if (obj == NULL) {
  cout << "NamedObjMgr::FillVect() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

NTuple* nt = NULL;
if (nomvec.length() < 1) nomvec = "VecFill";

char* ntn[2]= {"vecval", "vecwt"};
nt = new NTuple(1,ntn);  // Creation NTuple
 
string expwt = "1.";
string expz = "0.";   
servnobjm->Nobj_ComputeExpressions(obj, expx, expz, expz, expwt, expcut, nt, NULL, NULL);

if (!nt) return;
if (nt->NEntry() < 1) {
  cout << "NamedObjMgr::FillVect() Warning  Zero points satisfy cut !" << endl; 
  delete nt;
  return;
  }
  
Vector* vec = new Vector(nt->NEntry());
int k;
float* xn;
for(k=0; k<nt->NEntry(); k++)    {
  xn = nt->GetVec(k);
  (*vec)(k) = xn[0];
  }
delete nt;
AddObj(vec, nomvec); 
DisplayObj(nomvec, dopt);
return;  
}

/* --Methode-- */
void NamedObjMgr::FillNT(string const& nom, string& expx, string& expy, string& expz, string& expt, 
                         string& expcut, string& nomnt)
{
PPersist* obj=GetObj(nom);

if (obj == NULL) {
  cout << "NamedObjMgr::FillNT() Error , Pas d'objet de nom " << nom << endl; 
  return;
  }
if (!myImgApp)  return; 

bool fgnnt = false;
NTuple* nt = NULL;
PPersist* oh = NULL;
if (nomnt.length() > 0)  oh=GetObj(nomnt);
else nomnt = "NTFill";
if ( (oh != NULL) && (oh->ClassId() == ClassId_NTuple) )  { 
  nt = (NTuple*)oh;
  if (nt->NVar() > 10) {
    cout << "NamedObjMgr::FillNT() Warning , Max 10 var ds NTuple -> new NTuple" << endl;
    nt = NULL;
    }
  }  
if (nt == NULL) {
  char* ntn[4]= {"x", "y","z","t"};
  nt = new NTuple(4,ntn);  // Creation NTuple 
  fgnnt = true;
  }

servnobjm->Nobj_ComputeExpressions(obj, expx, expy, expz, expt, expcut, nt, NULL, NULL);

if (fgnnt) AddObj(nt, nomnt);
return;  

}

/* --Methode-- cmv 13/10/98 */
void NamedObjMgr::FillGFD(string const& nom, string& expx, string& expy, string& expz,
                          string& experr, string& expcut, string& nomgfd)
//	Pour remplir un ``GeneralFitData'' a partir de divers objets:
//| nom = nom de l'objet a transcrire selon 1D: Z=f(X) ou 2D: Z=f(X,Y) .
//|       Vector,Matrix,Histo,HProf,Histo2D,Image<T>,StarList,NTuple,GeneralFitData
//| expx = expression X du GeneralFitData (1er abscisse)
//| expy = expression Y du GeneralFitData (2sd abscisse si non "", Z=f(X,Y))
//| expz = expression Z du GeneralFitData (valeur de l'ordonnee)
//| experr = expression de l'erreur sur l'ordonnee Z
//| expcut = expression du test de selection
//| nomgfd = nom du GeneralFitData engendre (optionnel)
{
PPersist* obj=GetObj(nom);
if(obj == NULL)
  {cout<<"NamedObjMgr::FillGFD() Error , Pas d'objet de nom "<<nom<<endl;
   return;}
if(!myImgApp)  return;

// 2D ou 3D?
int nvar = 2;
if(expy.length()<=0) {nvar = 1; expy = "0.";}

// Que faut-il copier dans GeneralFitData
int ndata=0;
int cid = obj->ClassId();
switch (cid) {
  case ClassId_Vector :
    ndata=((Vector*)obj)->NElts();
    break;
  case ClassId_Matrix :
    ndata=((Matrix*)obj)->NCol()*((Matrix*)obj)->NRows();
    break;
  case ClassId_Histo1D :
  case ClassId_HProf : 
    ndata=((Histo*)obj)->NBins();
    break;
  case ClassId_Histo2D :
    ndata=((Histo2D*)obj)->NBinX()*((Histo2D*)obj)->NBinY();
    break;
  case ClassId_Image : 
  case ClassId_Image + kuint_1 :
  case ClassId_Image + kint_1 :
  case ClassId_Image + kr_8 :
  case ClassId_Image + kuint_4 :
  case ClassId_Image + kuint_2 :
  case ClassId_Image + kint_2 :
  case ClassId_Image + kint_4 :
  case ClassId_Image + kr_4 :
    ndata=((RzImage*)obj)->XSize()*((RzImage*)obj)->YSize();
    break;
#ifndef SANS_StarReco 
  case ClassId_StarList : 
    ndata=((StarList*)obj)->NbStars();
    break;
#endif
  case ClassId_NTuple :
    ndata=((NTuple*)obj)->NEntry();
    break;
  case ClassId_GeneralFitData :
    ndata=((GeneralFitData*)obj)->NData();
    break;
  default :
    cout<<"FillGFD Erreur: N/A to  "<<servnobjm->PClassIdToClassName(cid)<<endl; 
    return;
}

// Nom du GeneralFitData cree.
if (nomgfd.length() <= 0) nomgfd = "GFDFill";
// Consistence ndata.
if(ndata<=0)
  {cout<<"NamedObjMgr::FillGFD() Warning   ndata="<<ndata<<endl; 
   return;}

// Creation NTuple Buffer
char* ntn[4]= {"x","y","f","e"};
NTuple*nt = new NTuple(4,ntn);

// Remplissage NTuple buffer
servnobjm->Nobj_ComputeExpressions(obj, expx, expy, expz, experr, expcut, nt, NULL, NULL);
if(nt->NEntry() < 1)
  {cout<<"NamedObjMgr::FillGFD() Warning  Zero points satisfy cut !"<<endl; 
   delete nt; return;}

//Remplissage de la structure GeneraFitData
GeneralFitData* gfd = new GeneralFitData(nvar,ndata,0);
int k;
float* xn;
for(k=0; k<nt->NEntry(); k++) {
  xn = nt->GetVec(k);
  gfd->AddData(xn,xn[2],xn[3]);
}

// Menage et table d'objets
delete nt;
AddObj(gfd, nomgfd);
return;
}

/* --Methode-- */
void NamedObjMgr::PlotFunc(string& expfunc, float xmin, float xmax, int np, string dopt)
{
FILE *fip;
string fname = (*TmpDir) + "func1_pia_dl.c";
string fnamer = (*TmpDir) + "func1_pia_dl";
string  cmd;
int rc;

if (!myImgApp)  return; 

cmd = "rm " + fname;
rc = system(cmd.c_str());
printf("PlotFunc_Do> %s  (Rc=%d)\n", cmd.c_str(), rc); 

if ((fip = fopen(fname.c_str(), "w")) == NULL)   { 
  string sn = fname; 
  cout << "NamedObjMgr/PlotFunc_Erreur: Pb. Ouverture " << sn << endl; 
  return; 
  }

// constitution du fichier a compiler
fputs("#include <math.h> \n", fip);
fputs("double func1_pia_dl_func(double x) \n{\n", fip);
fprintf(fip,"return(%s); \n}\n", expfunc.c_str());
fclose(fip);

DlFunctionOfX f = (DlFunctionOfX) servnobjm->LinkFunctionFromFile(fnamer, "func1_pia_dl_func");
if (!f) return; 
PlotFunc(f, xmin, xmax, np, dopt);
servnobjm->CloseDLL();
return;
}

/* --Methode-- */
void NamedObjMgr::PlotFunc2D(string& expfunc, float xmin, float xmax, float ymin, float ymax, 
                             int npx, int npy, string dopt)
{
FILE *fip;
string fname = (*TmpDir) + "func2_pia_dl.c";
string fnamer = (*TmpDir) + "func2_pia_dl";
string cmd;
int rc;

if (!myImgApp)  return; 

cmd = "rm " + fname;
rc = system(cmd.c_str());
printf("PlotFunc2D_Do> %s  (Rc=%d)\n", cmd.c_str(), rc); 

if ((fip = fopen(fname.c_str(), "w")) == NULL)   { 
  string sn = fname; 
  cout << "NamedObjMgr/PlotFunc2D_Erreur: Pb. Ouverture " << sn << endl; 
  return; 
  }

// constitution du fichier a compiler
fputs("#include <math.h> \n", fip);
fputs("double func2_pia_dl_func(double x, double y) \n{\n", fip);
fprintf(fip,"return(%s); \n}\n", expfunc.c_str());
fclose(fip);

DlFunctionOfXY f = (DlFunctionOfXY) servnobjm->LinkFunctionFromFile(fnamer, "func2_pia_dl_func");
if (!f)  return;
PlotFunc2D(f, xmin, xmax, ymin, ymax, npx, npy, dopt);
servnobjm->CloseDLL();
return;
}

/* --Methode-- */
void NamedObjMgr::PlotFunc(DlFunctionOfX f, float xmin, float xmax, int np, string dopt)
{
if (!myImgApp)  return; 

int k;
if (np < 1) np = 1;
if (xmax <= xmin) xmax = xmin+1.;
Vector* vpy = new Vector(np);

double xx;
double dx = (xmax-xmin)/np;
TRY {
  for(k=0; k<np; k++) { xx = xmin+dx*k; (*vpy)(k) = f(xx); }
} CATCH(merr) {
  fflush(stdout); 
  cout << endl; 
  cerr << endl;
  string es = PeidaExc(merr);
  cerr << "NamedObjMgr::PlotFunc()  Exception :" << merr << es;
  delete vpy;
  vpy = NULL;
  } ENDTRY;


if (vpy) {
  string nom = "Func";
  AddObj(vpy, nom);
  P1DArrayAdapter* vya = new POVectorAdapter(vpy, false);
  vya->DefineXCoordinate(xmin, (xmax-xmin)/np);
  PIYfXDrawer* dr = new   PIYfXDrawer(vya, NULL, true) ; 
  bool fgsr = true;
  dopt = "thinline," + dopt;
  int opt = servnobjm->DecodeDispOption(dopt, fgsr);
  int wrsid = myImgApp->DispScDrawer(dr, nom, opt);
  if(wrsid >= 0) {
    NObjList::iterator it = myObjs->find(nom);
    if (it == myObjs->end()) return; 
    (*it).second.wrsid.push_back(wrsid);
    }
  if (fgsr) myImgApp->RestoreGraphicAtt(); 
  }

return;
}

/* --Methode-- */
void NamedObjMgr::PlotFunc2D(DlFunctionOfXY f, float xmin, float xmax, float ymin, float ymax, 
                             int npx, int npy, string dopt)
{
if (!myImgApp)  return; 

if (npx < 1) npx = 1;
if (npy < 1) npy = 1;
if (xmax <= xmin) xmax = xmin+1.;
if (ymax <= ymin) ymax = ymin+1.;

Matrix* mtx = new Matrix(npy, npx);

int i,j;
double xx, yy;
double dx = (xmax-xmin)/npx;
double dy = (ymax-ymin)/npy;
// printf(" -- DBG -- %d %d , %g %g , %g %g \n", npx, npy, xmin, xmax, ymin, ymax);
TRY {
  for(j=0; j<npy; j++) {
    yy = ymin+dy*j; 
    for(i=0; i<npx; i++) {
      xx = xmin+dx*i;
      (*mtx)(j, i) = f(xx, yy);
    }
  }
} CATCH(merr) {
  fflush(stdout); 
  cout << endl; 
  cerr << endl;
  string es = PeidaExc(merr);
  cerr << "NamedObjMgr::PlotFunc2D()  Exception :" << merr << es;
  delete mtx;  mtx = NULL;
  } ENDTRY;

if (mtx) {
  string nom = "Func2";
  AddObj(mtx, nom);
  DisplaySurf3D(nom, dopt);
  }

return;
}

///////////////////// Fit 1D et 2D //////////////////////////
/* --Function static propre aux routines de fit 1D et 2D-- cmv 13/10/98 */
struct DFOptions {
  int okres, okfun;
  int polcx,polcy; double xc,yc;
  double err_e, err_E;
  double stc2;
  int lp,lpg;
  int i1,i2,j1,j2;
};
typedef struct DFOptions DFOPTIONS;
static void DecodeFitsOptions(string par,string step,string min,string max,string opt
                      ,Vector& Par,Vector& Step,Vector& Min,Vector& Max,DFOPTIONS& O);
void DecodeFitsOptions(string par,string step,string min,string max,string opt
                      ,Vector& Par,Vector& Step,Vector& Min,Vector& Max,DFOPTIONS& O)
//| Pour decoder les "string" et remplir les vecteurs du fit (cf commentaires dans Fit1D)
{
// set des vecteurs et decodage des string correspondantes
int NParMax = 100;
Par.Realloc(NParMax); Step.Realloc(NParMax);
Min.Realloc(NParMax); Max.Realloc(NParMax);
{
  Vector* v; string* s;
  {for(int i=0;i<NParMax;i++) {Par(i)=0.; Step(i)=1.; Min(i)=1.; Max(i)=-1.;}}
  for(int j=0;j<4;j++) {
    if(j==0)      {v=&Par; s=&par;}
    else if(j==1) {v=&Step; s=&step;}
    else if(j==2) {v=&Min; s=&min;}
    else if(j==3) {v=&Max; s=&max;}
    if(s->length()>0) *s += ",";
    for(int i=0;i<NParMax;i++) {
      if(s->length()<=0) break;
      sscanf(s->c_str(),"%lf",&(*v)(i));
      size_t p = s->find_first_of(',') + 1;
      if(p>=s->length()) *s = ""; else *s = s->substr(p);
    }
  }
}

// Decodage de options de opt
O.okres = O.okfun = 0;
O.polcx = O.polcy = 0;
O.xc = O.yc = 0.;
O.stc2 = 1.e-3;
O.err_e = O.err_E = -1.;
O.lp = 1; O.lpg = 0;
O.i1 = O.j1 = O.i2 = O.j2 = -1;

if(opt.length()<=0) return;
opt = "," + opt + ",";

if(strstr(opt.c_str(),",r,")) O.okres = 1;  // residus
if(strstr(opt.c_str(),",f,")) O.okfun = 1;  // fonction fittee
if(strstr(opt.c_str(),",x")) { // Demande de centrage (fit X=x-xc)
  O.polcx = 2; // Le centrage est calcule automatiquement
  size_t p = opt.find(",x");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) {
    sscanf(dum.c_str(),",x%lf",&O.xc);
    O.polcx = 1; // Le centrage est fixe par la valeur lue
  }
}
if(strstr(opt.c_str(),",y")) { // Demande de centrage (fit Y=y-yc)
  O.polcy = 2; // Le centrage est calcule automatiquement
  size_t p = opt.find(",y");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) {
    sscanf(dum.c_str(),",y%lf",&O.yc);
    O.polcy = 1; // Le centrage est fixe par la valeur lue
  }
}
if(strstr(opt.c_str(),",E")) { // Erreurs imposees a "sqrt(val)" ou "aa.b*sqrt(val)"
  size_t p = opt.find(",E");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",E%lf",&O.err_E);
  if(O.err_E<=0.) O.err_E = 1.;
  O.err_e=-1.;
}
if(strstr(opt.c_str(),",e")) { // Erreurs imposees a "1" ou "aa.b"
  size_t p = opt.find(",e");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",e%lf",&O.err_e);
  if(O.err_e<=0.) O.err_e = 1.;
  O.err_E=-1.;
}
if(strstr(opt.c_str(),",X")) { // Valeur du StopChi2
  size_t p = opt.find(",X");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",X%lf",&O.stc2);
  if(O.stc2<=0.) O.stc2 = 1.e-3;
}
if(strstr(opt.c_str(),",l")) { // niveau de print
  size_t p = opt.find(",l");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  float ab;
  if(dum.length()>2) sscanf(dum.c_str(),",l%f",&ab);
  if(ab<0) ab = 0.;
  O.lp = (int) ab; O.lpg = (int) 10.*(ab-O.lp);
}
if(strstr(opt.c_str(),",I")) { // intervalle de fit selon X
  size_t p = opt.find(",I");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",I%d/%d",&O.i1,&O.i2);
}
if(strstr(opt.c_str(),",J")) { // intervalle de fit selon Y
  size_t p = opt.find(",J");
  size_t q = opt.find_first_of(',',p+1);
  string dum = opt.substr(p,q-p);
  if(dum.length()>2) sscanf(dum.c_str(),",J%d/%d",&O.j1,&O.j2);
}
return;
}

/* --Methode-- cmv 13/10/98 */
void  NamedObjMgr:: Fit12D(string const& nom, string& func,
                          string par,string step,string min,string max,
                          string opt)
//| --------------- Fit d'objets a 1 et 2 dimensions ---------------
//| nom  : nom de l'objet qui peut etre:
//|        fit-1D:  Vector,Histo1D,HProf ou GeneraFitData(1D)
//|        fit-2D:  Matrix,Histo2D,Imagexx ou GeneraFitData(2D)
//| func : pnn = fit polynome degre nn avec classe Poly (lineaire) 1D ou 2D
//|      : Pnn = fit polynome degre nn avec GeneralFit (non-lineaire) 1D ou 2D
//|      : gnn = fit gaussienne (hauteur) + polynome de degre nn 1D
//|      : g   = fit gaussienne (hauteur) 1D
//|      : enn = fit exponentielle + polynome de degre nn 1D
//|      : e   = fit exponentielle 1D
//|      : Gnn = fit gaussienne (volume) + polynome de degre nn 1D
//|      : G   = fit gaussienne (volume) 1D
//|      :     = fit gaussienne+fond (volume) 2D
//|      : Gi  = fit gaussienne+fond integree (volume) 2D
//|      : d   = fit DL de gaussienne+fond (volume) 2D
//|      : di  = fit DL de gaussienne+fond integree (volume) 2D
//|      : D   = fit DL de gaussienne+fond avec coeff variable p6 (volume) 2D
//|      : Di  = fit DL de gaussienne+fond integree avec coeff variable p6 (volume) 2D
//|      : M   = fit Moffat+fond (expos=p6) (volume) 2D
//|      : Mi  = fit Moffat+fond integree (expos=p6) (volume) 2D
//| par  : p1,...,pn valeur d'initialisation des parametres (def=0)
//| step : s1,...,sn valeur des steps de depart (def=1)
//| min  : m1,...,mn valeur des minima (def=1)
//| max  : M1,...,Mn valeur des maxima (def=-1) (max<=min : pas de limite)
//| opt  : options "Eaa.b,eaa.b,f,r,caa.b,Xaa.b"
//|       f = generation d'un Objet identique contenant la fonction fittee
//|       r = generation d'un Objet identique contenant les residus
//|       Xaa.b = aa.b valeur du DXi2 d'arret (def=1.e-3)
//|       la.b = niveau "a.b" de print: a=niveau de print Fit1/2D
//|                                     b=niveau de debug GeneralFit
//|       Ii1/i2 numeros des bins X de l'histos utilises pour le fit [i1,i2]
//|2D     Jj1/j2 numeros des bins Y de l'histos utilises pour le fit [j1,j2]
//|       - L'erreur est celle associee a l'objet si existe, 1 sinon sauf si
//|       E = erreur prise comme la racine de la valeur a fitter
//|       Eaa.b = erreur prise aa.b*sqrt(val)
//|       e = erreur prise egale a 1 pour toutes les valeurs
//|       eaa.b = erreur prise egale a aa.b
//|       xaa.b = demande de centrage: on fit x-aa.b au lieu de x)
//|       x = demande de centrage: on fit x-xc au lieu de x
//|           avec xc=abscisse du milieu de l'histogramme
//|           Actif pour: exp+poly 1D, poly 1D
//|                       gauss+poly 1D (mais xc est le centre de la gaussienne)
//|2D     yaa.b et y = idem "xaa.b et x" mais pour y
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout<<"NamedObjMgr::Fit12D() Error , Pas d'objet de nom "<<nom<<endl;
  return;
}
if (!myImgApp)  return;
if(func.length()<=0)
  {cout<<"NamedObjMgr::Fit12D() Donnez un nom de fonction a fitter."<<endl;
   return;}
int cid = obj->ClassId();
string ctyp = servnobjm->PClassIdToClassName(cid);

int ndim = 0, nbinx=0, nbiny=0, ndata = 0;
Vector* v = NULL; Histo* h = NULL;
Matrix* m = NULL; Histo2D* h2 = NULL; RzImage* im = NULL;
GeneralFitData* g = NULL;
switch (cid) {
  // 1D
  case ClassId_Vector :
    ndim = 1;
    v = (Vector*) obj; nbinx = v->NElts(); nbiny = 1;
    break;
  case ClassId_HProf :
  case ClassId_Histo1D :
    ndim = 1;
    h = (Histo*)  obj; nbinx = h->NBins(); nbiny = 1;
    break;
  // 2D
  case ClassId_Matrix :
    ndim = 2;
    m = (Matrix*) obj; nbinx = m->NCol(); nbiny = m->NRows();
    break;
  case ClassId_Histo2D :
    ndim = 2;
    h2 = (Histo2D*) obj; nbinx = h2->NBinX(); nbiny = h2->NBinY();
    break;
  // 1D ou 2D selon
  case ClassId_GeneralFitData :
    g = (GeneralFitData*) obj; nbinx = g->NData(); nbiny = 1;
    if(     g->NVar()==1) ndim = 1;
    else if(g->NVar()==2) ndim = 2;
    else {
      cout<<"GeneralFitData ne peut avoir que 1 ou 2 variables d'abscisse: "
          <<((GeneralFitData*) obj)->NVar()<<endl; return;}
    break;
  case ClassId_Image : 
  case ClassId_Image + kuint_1 : 
  case ClassId_Image + kuint_2 : 
  case ClassId_Image + kint_2 : 
  case ClassId_Image + kint_4 : 
  case ClassId_Image + kr_4 :
  case ClassId_Image + kr_8 : 
    ndim = 2;
    im = (RzImage*) obj; nbinx = im->XSize(); nbiny = im->YSize();
    break;
  default:
    cout<<"NamedObjMgr::Fit12D() Error , Objet n'est pas un "
        <<"Histo1D/HProf/Vector/Histo2D/Image/Matrix/GeneralFitData "<<ctyp<<endl;
    return;
    break;
}
ndata = nbinx*nbiny;
if(ndata<=0)
  {cout<<"L'objet a "<<nbinx<<","<<nbiny<<" bins ("<<ndata<<")"<<endl; return;}

// Decodage des options et des parametres, mise en forme
Vector Par(1); Vector Step(1); Vector Min(1); Vector Max(1); DFOPTIONS O;
DecodeFitsOptions(par,step,min,max,opt,Par,Step,Min,Max,O);
O.i1 = (O.i1<0||O.i1>=nbinx)? 0: O.i1;
O.i2 = (O.i2<0||O.i2>=nbinx||O.i2<O.i1)? nbinx-1: O.i2;
if(ndim>=2) {
  O.j1 = (O.j1<0||O.j1>=nbiny)? 0: O.j1;
  O.j2 = (O.j2<0||O.j2>=nbiny||O.j2<O.j1)? nbiny-1: O.j2;
} else O.j2 = O.j1 = 0;
if(O.polcx==2) {
  if(v||m)    O.xc = (O.i2-O.i1+1)/2.;
  else if(h)  O.xc = (h->XMin()+h->XMax())/2.;
  else if(h2) O.xc = (h2->XMin()+h2->XMax())/2.;
  else if(g)  {double mini,maxi; g->GetMinMax(2,mini,maxi); O.xc=(mini+maxi)/2.;}
  else if(im) {O.xc = im->XOrg() * im->XPxSize()*(O.i2-O.i1+1)/2.;}
}
if(O.polcy==2 && ndim>=2) {
  if(m)  O.yc = (O.j2-O.j1+1)/2.;
  if(h2) O.yc = (h2->YMin()+h2->YMax())/2.;
  if(g)  {double mini,maxi; g->GetMinMax(12,mini,maxi); O.yc=(mini+maxi)/2.;}
  if(im) {O.yc = im->YOrg() * im->YPxSize()*(O.j2-O.j1+1)/2.;}
}
if(O.lp>0)
  cout<<"Fit["<<nbinx<<","<<nbiny<<"] ("<<ndata<<") dim="<<ndim<<":"
      <<" Int=["<<O.i1<<","<<O.i2<<"],["<<O.j1<<","<<O.j2<<"]"<<endl
      <<" Cent="<<O.polcx<<","<<O.polcy<<","<<O.xc<<"+x"<<","<<O.yc<<"+y"
      <<" TypE="<<O.err_e<<","<<O.err_E
      <<" StpX2="<<O.stc2
      <<" lp,lpg="<<O.lp<<","<<O.lpg<<endl;

///////////////////////////////////
// Remplissage de GeneralFitData //
///////////////////////////////////
GeneralFitData mydata(ndim,ndata,0);
{for(int i=O.i1;i<=O.i2;i++) for(int j=O.j1;j<=O.j2;j++) {
  double x,y,f,e;

  if(v)
    {x= (double) i; f=(*v)(i); e=1.;}
  else if(h)
    {x=h->BinCenter(i); f=(*h)(i); e=(h->HasErrors())?h->Error(i):1.;}
  else if(m)
    {x=(double) i; y=(double) j; f=(*m)(j,i); e=1.;}
  else if(h2)
    {float xf,yf; h2->BinCenter(i,j,xf,yf); x=(double)xf; y=(double)yf;
     f=(*h2)(i,j); e=(h2->HasErrors())?h2->Error(i,j):1.;}
  else if(im)
    {x=im->XOrg()+(i+0.5)*im->XPxSize(); y=im->YOrg()+(j+0.5)*im->YPxSize();
     f=im->DValue(i,j); e=1.;}
  else if(g&&ndim==1) {x= g->X(i); f=g->Val(i); e=g->EVal(i);}
  else if(g&&ndim==2) {x= g->X(i); y= g->Y(i); f=g->Val(i); e=g->EVal(i);}
  else x=y=f=e=0.;

  // Gestion des erreurs a utiliser
  if(O.err_e>0.) e=O.err_e;
  else if(O.err_E>0.) {e=(y<-1.||y>1.)?O.err_E*sqrt(fabs(y)):O.err_E;}

  // Remplissage de generalfit
  if(func[0]=='p') {x -= O.xc; if(ndim>=2) y -= O.yc;}
  if(ndim==1)      mydata.AddData1(x,f,e);
  else if(ndim==2) mydata.AddData2(x,y,f,e);
}}
if(mydata.NData()<=0)
  {cout<<"Pas de donnees dans GeneralFitData: "<<mydata.NData()<<endl;
   return;}
if(O.lpg>1) {
  mydata.PrintStatus();
  mydata.PrintData(0);
  mydata.PrintData(mydata.NData()-1);
}

////////////////////////////////////////////
// Identification de la fonction a fitter //
////////////////////////////////////////////
GeneralFunction* myfunc = NULL;
if(func[0]=='p' && ndim==1) {
  // Fit de polynome sans passer par les GeneralFit
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit (lineaire) 1D polynome de degre "<<degre<<endl;
  Poly p1(0);
  double c2rl = mydata.PolFit(0,p1,degre);
  cout<<"C2r_lineaire = "<<c2rl<<endl;
  if(O.lp>0) cout<<p1<<endl;
  return;

} else if(func[0]=='P' && ndim==1) {
  // Fit de polynome
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit polynome 1D de degre "<<degre<<endl;
  Polyn1D* myf = new Polyn1D(degre,O.xc);
  myfunc = myf;

} else if(func[0]=='e' && ndim==1) {
  // Fit d'exponentielle
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit d'exponentielle+polynome 1D de degre "<<degre<<endl;
  Exp1DPol* myf;
  if(degre>=0) myf = new Exp1DPol((unsigned int)degre,O.xc);
       else    myf = new Exp1DPol(O.xc);
  myfunc = myf;

} else if(func[0]=='g' && ndim==1) {
  // Fit de gaussienne en hauteur
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit de Gaussienne_en_hauteur+polynome 1D de degre "<<degre<<endl;
  Gauss1DPol* myf;
  if(degre>=0) myf = new Gauss1DPol((unsigned int)degre,((O.polcx)?true:false));
       else    myf = new Gauss1DPol(((O.polcx)?true:false));
  myfunc = myf;

} else if(func[0]=='G' && ndim==1) {
  // Fit de gaussienne en volume
  int degre =-1;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit de Gaussienne_en_volume+polynome 1D de degre "<<degre<<endl;
  GaussN1DPol* myf;
  if(degre>=0) myf = new GaussN1DPol((unsigned int)degre,((O.polcx)?true:false));
       else    myf = new GaussN1DPol(((O.polcx)?true:false));
  myfunc = myf;

} else if(func[0]=='p' && ndim==2) {
  // Fit de polynome 2D sans passer par les GeneralFit
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit (lineaire) polynome 2D de degre "<<degre<<endl;
  Poly2 p2(0);
  double c2rl = mydata.PolFit(0,1,p2,degre);
  cout<<"C2r_lineaire = "<<c2rl<<endl;
  if(O.lp>0) cout<<p2<<endl;
  return;

} else if(func[0]=='P' && ndim==2) {
  // Fit de polynome 2D
  int degre = 0;
  if(func.length()>1) sscanf(func.c_str()+1,"%d",&degre);
  cout<<"Fit polynome 2D de degre "<<degre<<endl;
  Polyn2D* myf = new Polyn2D(degre,O.xc,O.yc);
  myfunc = myf;

} else if(func[0]=='G' && ndim==2) {
  // Fit de gaussienne+fond en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de Gaussienne+Fond 2D integ="<<integ<<endl;
  if(integ) {GauRhInt2D* myf = new GauRhInt2D; myfunc = myf;}
    else    {GauRho2D* myf = new GauRho2D; myfunc = myf;}

} else if(func[0]=='d' && ndim==2) {
  // Fit de DL gaussienne+fond en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de DL de Gaussienne+Fond 2D integ="<<integ<<endl;
  if(integ) {GdlRhInt2D* myf = new GdlRhInt2D; myfunc = myf;}
    else    {GdlRho2D* myf = new GdlRho2D; myfunc = myf;}

} else if(func[0]=='D' && ndim==2) {
  // Fit de DL gaussienne+fond avec coeff variable p6 en volume
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de DL de Gaussienne+Fond avec coeff variable (p6) 2D integ="<<integ<<endl;
  if(integ) {Gdl1RhInt2D* myf = new Gdl1RhInt2D; myfunc = myf;}
    else    {Gdl1Rho2D* myf = new Gdl1Rho2D; myfunc = myf;}

} else if(func[0]=='M' && ndim==2) {
  // Fit de Moffat+fond (volume)
  int integ = 0;
  if(func.length()>1) if(func[1]=='i') integ=1;
  cout<<"Fit de Moffat+Fond (expos=p6) 2D integ="<<integ<<endl;
  if(integ) {MofRhInt2D* myf = new MofRhInt2D; myfunc = myf;}
    else    {MofRho2D* myf = new MofRho2D; myfunc = myf;}

} else {
  cout<<"Fonction "<<func<<" inconnue pour la dim "<<ndim<<endl;
  return;
}

/////////////////////////
// Fit avec generalfit //
/////////////////////////
if(myfunc->NPar()>Par.NElts())
  {cout<<"Trop de parametres: "<<myfunc->NPar()<<">"<<Par.NElts()<<endl;
  if(myfunc) delete myfunc; return;}
GeneralFit myfit(myfunc);
myfit.SetDebug(O.lpg);
myfit.SetData(&mydata);
myfit.SetStopChi2(O.stc2);
{for(int i=0;i<myfunc->NPar();i++) {
  char str[10];
  sprintf(str,"P%d",i);
  myfit.SetParam(i,str,Par(i),Step(i),Min(i),Max(i));
}}
if(O.lp>1) myfit.PrintFit();
double c2r = (double) myfit.Fit();
if(O.lp>0) myfit.PrintFit();
if(c2r>0.) {
  c2r = myfit.GetChi2Red();
  cout<<"C2r_Reduit = "<<c2r<<"  nstep="<<myfit.GetNStep()<<endl;
  Vector ParFit(myfunc->NPar());
  for(int i=0;i<myfunc->NPar();i++) ParFit(i)=myfit.GetParm(i);
} else {
  cout<<"echec Fit, rc = "<<c2r<<"  nstep="<<myfit.GetNStep()<<endl;
}

// Mise a disposition des resultats
if(c2r>=0. && myfunc && (O.okres>0||O.okfun>0)) {
  string nomres = nom + "res";
  string nomfun = nom + "fun";
  if(v) {
    if(O.okres) {Vector* ob = v->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Vector* ob = v->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(h) {
    if(O.okres) {Histo* ob = h->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Histo* ob = h->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(m) {
    if(O.okres) {Matrix* ob = m->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Matrix* ob = m->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(h2) {
    if(O.okres) {Histo2D* ob = h2->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {Histo2D* ob = h2->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(im) {
    if(O.okres) {RzImage* ob = im->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {RzImage* ob = im->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  } else if(g) {
    if(O.okres) {GeneralFitData* ob = g->FitResidus(myfit);  if(ob) AddObj(ob,nomres);}
    if(O.okfun) {GeneralFitData* ob = g->FitFunction(myfit); if(ob) AddObj(ob,nomfun);}
  }
}

// Nettoyage
if(myfunc) delete myfunc;
return;
}

/* --Methode-- */
void NamedObjMgr::FFT(string const& nom, string dopt)
{
PPersist* obj=GetObj(nom);
if (obj == NULL) {
  cout << "NamedObjMgr::FFT() Error , Pas d'objet de nom " << nom << endl; 
  return;
}
if (!myImgApp)  return; 

int cid = obj->ClassId();
if (cid != ClassId_Vector) { 
  string ctyp = servnobjm->PClassIdToClassName(cid);
  cout << "NamedObjMgr::FFT() Error , Objet n'est pas un vecteur " << ctyp << endl; 
  return;
  }

FFT1 fft( *((Vector*)obj) );

if (fft.NData() < 1)  return;

char* ntn[3] = {"refc", "imfc", "pfc"};
NTuple* nt = new NTuple(3,ntn);  // Creation NTuple 
Vector* vf = new Vector(fft.NData()/2);

float xnt[3]; 
double pf;
for(int k=0; k<fft.NData()/2; k++) {
  xnt[0] = fft.ReF(k);  xnt[1] = fft.ImF(k); 
  pf = (xnt[0]*xnt[0]+xnt[1]*xnt[1]);
  if (pf >= 0.)  pf = sqrt(pf);  
  xnt[2] = pf;   (*vf)(k) = pf;
  nt->Fill(xnt);  
  }

P1DArrayAdapter* vya = new POVectorAdapter(vf, true);
// vya->DefineXCoordinate(xmin, xmax);   a determiner
PIYfXDrawer* dr = new   PIYfXDrawer(vya, NULL, true) ; 
bool fgsr = true;
dopt = "thinline," + dopt;
int opt = servnobjm->DecodeDispOption(dopt, fgsr);
myImgApp->DispScDrawer(dr, nom, opt);
if (fgsr) myImgApp->RestoreGraphicAtt(); 

return;
}
