/* Writer de table Fits (binaire ou ASCII) */
#include "machdefs.h"
#include <stdlib.h>
#include <stdio.h>
#include "pexceptions.h"
#include "fabtwriter.h"

/*!
  \class SOPHYA::FitsABTWriter
  \ingroup FitsIOServer
  Class for writing a FITS ASCII or BINARY table
  \verbatim\
  Exemple:
  FitsABTWriter fbtw("myfits.fits",BINARY_TBL,3);
  fbtw.SetExtName("MY_OWN_EXTENSION");
  fbtw.AddCol("vars",TSHORT,"","km");      // col=0
  fbtw.AddCol("vars2",TSHORT,"","km");     // col=1
  fbtw.AddCol("varl",TLONG,"","Degre");    // col=2
  fbtw.AddCol("varf",TFLOAT,"","");        // col=3
  fbtw.AddCol("vard",TDOUBLE,"","arcmin"); // col=4
  fbtw.SetDebug(3);
  for(long i=0;i<1000;i++) {
    double x = i;
    fbtw.Write(0,i,1000.*x); // if overflow, managed by cfitsio
                             // Write(int,long,double) is called
    fbtw.Write(1,i,(short)(1000.*x));
                             // if overflow, managed by language
                             // Write(int,long,short) is called
    fbtw.Write(2,i,x);
    fbtw.Write(3,i,x);
    fbtw.Write(4,i,x);
  }
  cout<<"Number of Overflows when writing: "
      <<fbtw.GetNOverFlow()<<endl;
  \endverbatim
*/

//////////////////////////////////////////////////////////////
/*!
  Constructor.
  \verbatim
  fname : FITS file name to be written
  hdutype : type of extension to be created (BINARY_TBL or ASCII_TBL)
  lp : debug level
  \endverbatim
*/
FitsABTWriter::FitsABTWriter(string fname,int hdutype,int lp)
{
  createfits(fname.c_str(),hdutype,lp);
}

/*! See FitsABTWriter() */
FitsABTWriter::FitsABTWriter(const char* cfname,int hdutype,int lp)
{
  createfits(cfname,hdutype,lp);
}

/*! See FitsABTWriter() */
void FitsABTWriter::createfits(const char *cfname,int hdutype,int lp)
{
 FirstTime = true;
 FitsPtr = NULL;
 HduType = hdutype;
 SetDebug(lp);
 FitsFN = cfname;
 NOverFlow = 0;

 if(DbgLevel)
   cout<<"FitsABTWriter::createfits FitsFN="<<FitsFN
       <<" HduType="<<HduType<<endl;

 if(FitsFN.size() <= 0 )
   throw ParmError("FitsABTWriter::createfits: Fits file name error\n");

 if(HduType!=BINARY_TBL && HduType!=ASCII_TBL)
   throw TypeMismatchExc("FitsABTWriter::createfits: Only BINARY or ASCII table permitted\n");

 // create new FITS file
 int sta=0;
 if(fits_create_file(&FitsPtr,FitsFN.c_str(),&sta)) {
   printerror(sta);
   throw NullPtrError("FitsABTWriter::createfits: Error creating Fits file\n");
 }

 // create d'un Primary HDU */
 //long naxes[1] = {0};
 //if(fits_create_img(FitsPtr,BYTE_IMG,0,naxes,&sta)) {
 //  printerror(sta);
 //  throw NullPtrError("FitsABTWriter::createfits: Error creating Primary extension\n");
 //}

}

/*! Destructor */
FitsABTWriter::~FitsABTWriter()
{
 int sta = 0;
 if(fits_close_file(FitsPtr,&sta)) printerror(sta);
 FitsPtr = NULL;
}

//////////////////////////////////////////////////////////////
/*!
  Add a new column to the FITS table
  \verbatim
  label : column label
  datatype : TSHORT TLONG TFLOAT or TDOUBLE
  tform : fits tform definition
          (can be automatically set if BINARY_TBL and tform="")
  tunit : fits tunit definition (optional)
  \endverbatim
*/
int FitsABTWriter::addcol(const char* label,int datatype
                         ,const char* tform,const char* tunit)
{
  if(!FirstTime)
    throw AllocationError("FitsABTWriter::addcol: table already created\n");

 if(  datatype!=TSHORT && datatype!=TLONG
   && datatype!=TFLOAT && datatype!=TDOUBLE )
   throw ParmError("FitsABTWriter::addcol: datatype not allowed\n");
 
 Label.push_back(label);
 DataType.push_back(datatype);
 // Gestion auto du tform par defaut pour les tables binaires
 if(HduType==BINARY_TBL && strlen(tform)<=0) {
   char str[16];
   if(datatype==TSHORT)       strcpy(str,"1I");
   else if(datatype==TLONG)   strcpy(str,"1J");
   else if(datatype==TFLOAT)  strcpy(str,"1E");
   else if(datatype==TDOUBLE) strcpy(str,"1D");
   TForm.push_back(str);
 } else TForm.push_back(tform);
 TUnit.push_back(tunit);

 int n = (int) Label.size() -1;

 if(DbgLevel)
   cout<<"FitsABTWriter::addcol["<<n<<"] Label="<<Label[n]
       <<" datatype="<<datatype
       <<" TForm="<<TForm[n]
       <<" TUnit="<<TUnit[n]<<endl;

 return n;
}

/*! Create the table. Done at the first write request. */
void FitsABTWriter::createtbl(void)
{
  if(!FirstTime) return;

 int tfields = Label.size();
 if(tfields<=0)
   throw ParmError("FitsABTWriter::createtbl: Zero column asked !\n");

 long nrows = 0;
 char *extname = NULL;
 char **ttype   = (char **) malloc(tfields*sizeof(char *));
 char **tform   = (char **) malloc(tfields*sizeof(char *));
 char **tunit   = (char **) malloc(tfields*sizeof(char *));

 if(ExtName.size()>0) {
   extname = (char *) malloc((strlen(ExtName.c_str())+1)*sizeof(char));
   strcpy(extname,ExtName.c_str());
 }
 for(int i=0;i<tfields;i++) {
   ttype[i] = (char *) malloc((strlen(Label[i].c_str())+1)*sizeof(char));
     strcpy(ttype[i],Label[i].c_str());
   tform[i] = (char *) malloc((strlen(TForm[i].c_str())+1)*sizeof(char));
     strcpy(tform[i],TForm[i].c_str());
   tunit[i] = (char *) malloc((strlen(TUnit[i].c_str())+1)*sizeof(char));
     strcpy(tunit[i],TUnit[i].c_str());
 }

 // append a new empty binary/ascii table onto the FITS file
 int sta=0;
 if(fits_create_tbl(FitsPtr,HduType,nrows,tfields,ttype,tform,tunit,extname,&sta)) {
   printerror(sta);
   throw NullPtrError("FitsABTWriter::createtbl: Error creating Table extension\n");
 }

 // menage
 if(extname) delete [] extname;
 for(int i=0;i<tfields;i++) {
   if(ttype[i]) delete [] ttype[i];
   if(tform[i]) delete [] tform[i];
   if(tunit[i]) delete [] tunit[i];
 }
 if(ttype) delete [] ttype;
 if(tform) delete [] tform;
 if(tunit) delete [] tunit;

 FirstTime = false;
}

//////////////////////////////////////////////////////////////
/*!
  Write a short data to FITS file.
  \verbatim
  col : column number [0,ncol[
  row : row number    [0,nrow[
  val : value to be written
  WARNING: that routine write a SHORT value into column "col"
           which could have been defined with an other type.
           Cast is performed by the cfitsio package.
  WARNING: suppose that the column has be defined to be TSHORT
           and suppose that you wanted to write a double value
   - If you write dummy.Write(col,row,(short(val))
       you call the Write(int,long,short) routine and
       the cast is performed by the C++ language.
   - If you write dummy.Write(col,row,val) where val is double
       you call theWrite(int,long,double) routine and
       the cast is performed by the cfistio package.
  \endverbatim
*/
void FitsABTWriter::Write(int col,long row,short val)
{
  if(FirstTime) createtbl();
  short x[1] = {val};
  int sta=0;
  if(fits_write_col(FitsPtr,TSHORT,col+1,row+1,1,1,x,&sta))
    printerrorwrite("short",col,row,sta);
}

/*! Write long data to FITS file (see below) */
void FitsABTWriter::Write(int col,long row,long val)
{
  if(FirstTime) createtbl();
  long x[1] = {val};
  int sta=0;
  if(fits_write_col(FitsPtr,TLONG,col+1,row+1,1,1,x,&sta))
    printerrorwrite("long",col,row,sta);
}

/*! Write float data to FITS file (see below) */
void FitsABTWriter::Write(int col,long row,float val)
{
  if(FirstTime) createtbl();
  float x[1] = {val};
  int sta=0;
  if(fits_write_col(FitsPtr,TFLOAT,col+1,row+1,1,1,x,&sta))
    printerrorwrite("float",col,row,sta);
}

/*! Write double data to FITS file (see below) */
void FitsABTWriter::Write(int col,long row,double val)
{
  if(FirstTime) createtbl();
  double x[1] = {val};
  int sta=0;
  if(fits_write_col(FitsPtr,TDOUBLE,col+1,row+1,1,1,x,&sta))
    printerrorwrite("double",col,row,sta);
}

void FitsABTWriter::printerrorwrite(char* type,int col,long row,int sta)
{
 if(sta==NUM_OVERFLOW) {NOverFlow++; return;}
 printerror(sta);
 char str[256];
 sprintf(str,"FitsABTWriter::Write_%s: Error Writing Fits c=%d r=%ld sta=%d"
        ,type,col,row,sta);
 throw NotAvailableOperation(str);
}

/////////////////////////////////////////////////
void FitsABTWriter::printerror(int sta) const
{
 int stat = sta;
 fits_report_error(stdout,stat);
 fflush(stdout);
 return;
}
