/**************************************************************
 >>>>>>>> usage: tstdemo [snum1,snum2]
 Test et demonstration du Pipe.
 "demo1.fits": fits table avec sampleNum,boloMuV_10,fg_boloMuV_10,boloMuV_20
               samplenum=[100001,105000]
 "demo2.fits": fits table avec sampleNum,boloMuV_30,fg_boloMuV_30
               samplenum=[100101,105100]
 Operation: sortir un fichier "demo.fits" ou on ecrit:
   Toutes les donnees des 2 fichiers d'entree ET :
   boloSum = boloMuV_10 + boloMuV_20 + boloMuV_30
   fg_boloSum = fg_boloMuV_10 | fg_boloMuV_20 | fg_boloMuV_30
   boloMul = boloMuV_10 * boloMuV_20 * boloMuV_30
   fg_boloMul = fg_boloMuV_10 | fg_boloMuV_20 | fg_boloMuV_30

 Structure:

           demo1.fits                        demo2.fits
               |                                  |
           ----------                         ----------
           | Reader |                         | Reader |
           | rfits1 |                         | rfits2 |
           ----------                         ----------
   "boloMuV_10" "boloMuV_20"                 "boloMuV_30"
             |            |                            | 
            t|           t|                           t| 
            u|           u|                           u| 
            y|           y|                           y| 
            a|           a|                           a| 
            u|           u|                           u| 
             |            |                            | 
            b|           b|                           b| 
            o|           o|                           o| 
            l|           l|                           l| 
            o|           o|                           o| 
             |            |                            | 
            1|           2|                           3| 
            0|           0|                           0| 
             |            |                            | 
             |            |                            | 
            /|           /|                           /| 
           / |          / |                          / | 
          /  |         /  |                         /  | 
         /   |        /   |                        /   | 
        /    |       /    |                       /    |
  _____/     |      /     |                      /     |
  |_________ | ____/      |                     /     / 
  |||_______ | __________ | ___________________/     /  
  |||        |            |                         /
  |||        |            |                        /
  |||        |            |                       /
  |||        |            /                      /
  |||        |           /                      /
  |||        |   _______/                      /
  |||        /  /          ___________________/
  |||       /  /  ________/
  |||      /  /  /_________________________________
  |||     /  /  / | DemoPipe Processor:           |
  |||    /  /  /  | ===================           |
  |||   /  /  |---| entree_bolo_3                 |
  |||   | |       |               sortie_bolo_sum |_______
  |||   | |-------| entree_bolo_2                 |      |
  |||   |         |               sortie_bolo_mul | ___  |
  |||   |---------| entree_bolo_1                 |   |  |
  |||             |_______________________________|   |  |
  |||                                                 |  |
  |||                                                 |  |
  |||                     ---------------             |  |
  |||                     |             |             |  |
  |||--------"boloMuv_30" |   Writter   | "boloMul" --   |
  ||---------"boloMuv_20" |   rfitsw    | "boloSum" ------
  |----------"boloMuV_10" |             |
                          ---------------
                                 |
                             demo.fits

**************************************************************/

#include <stdexcept>
#include "toi.h"
#include "toiprocessor.h"
#include "fitstoirdr.h"
#include "fitstoiwtr.h"
#include "toimanager.h"
#include "toisegment.h"
#include "sophyainit.h"

void crefits(void);  // Juste pour le test, rien a voir avec TOI

////////////////////////////////////////////////////////////////
// Le INCLUDE de la classe du processeur (peut etre mis a part)
class DemoPipe : public TOIProcessor {
public:
  DemoPipe(void);
  virtual ~DemoPipe();

  virtual void	init(void);
  virtual void	run(void);

  void PrintStatus(ostream & os);

  inline int_8 ProcessedSampleCount() const {return totnscount;}
protected:
  int_8 nread,nwrite,totnscount;
};

////////////////////////////////////////////////////////////////
// Le code de la classe du processeur (peut etre mis a part)
DemoPipe::DemoPipe(void)
: nread(0), nwrite(0),totnscount(0)
{
}

DemoPipe::~DemoPipe()
{
}

void DemoPipe::PrintStatus(ostream & os)
{
  os<<"DemoPipe::Print -- nread  = "<<nread<<endl
    <<"                -- nwrite = "<<nwrite<<endl;
}

void DemoPipe::init() {
 // Declaration des tuyaux a connecter. L'ordre de declaration compte!
 cout << "DemoPipe::init" << endl;
 declareInput("entree_bolo_1");     // input  index 0
 declareInput("entree_bolo_2");     // input  index 1
 declareInput("entree_bolo_3");     // input  index 2
 declareOutput("sortie_bolo_sum");  // output index 0
 declareOutput("sortie_bolo_mul");  // output index 1
}

void DemoPipe::run()
{
 // Verification des connections en entree
 if(!checkInputTOIIndex(0) || !checkInputTOIIndex(1) || !checkInputTOIIndex(2)) {
   cout<<"DemoPipe::run() - Input TOI (entree_bolo_1/2/3) not connected! "<<endl;
   throw ParmError("DemoPipe::run() Output TOI (entree_bolo_1/2/3) not connected!");
 }

 // Verification des connections en sortie
 if(!checkOutputTOIIndex(0) || !checkOutputTOIIndex(1)) {
   cout<<"DemoPipe::run() - Output TOI (boloSum/Mul) not connected! "<<endl;
   throw ParmError("DemoPipe::run() Output TOI (boloSum/Mul) not connected!");
 }

 // On recupere les sample numbers
 int snb = getMinIn();
 int sne = getMaxIn();
 cout<<"DemoPipe::run: sn="<<snb<<" sne="<<sne<<endl;
 if(snb>sne) {
   cout<<"DemoPipe::run() - Bad sample interval"<<snb<<" , "<<sne<<endl;
   throw ParmError("DemoPipe::run() - Bad sample interval ");
 }

 //---------------------------------------------------------
 uint_8 flb1,flb2,flb3,flbs,flbm;
 double b1,b2,b3,bs,bm;

 for(int k=snb;k<=sne;k++) {
   totnscount++;

   getData(0,k,b1,flb1);
   getData(1,k,b2,flb2);
   getData(2,k,b3,flb3);
   nread++;

   bs   = b1   +  b2  +  b3;
   flbs = flb1 | flb2 | flb3;
   bm   = b1   *  b2  *  b3;
   flbm = flb1 | flb2 | flb3;
   /*
   cout<<"b1="<<b1<<" flb1 "<<flb1
       <<" / b2="<<b2<<" flb2 "<<flb2
       <<" / b3="<<b3<<" flb3 "<<flb3<<endl;
   cout<<"...bs="<<bs<<" flbs "<<flbs<<"   bm="<<bm<<" flbm "<<flbm<<endl;
   */

   putData(0,k,bs,flbs);
   putData(1,k,bm,flbm);
   nwrite++;
 }
 cout<<"DemoPipe::run: end"<<endl;
}


////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
// Le main program
int main(int narg, char** arg)
{
 // Initialisation de Sophya
 SophyaInit();

 // Creation des fichiers fits utilises par la demo (Rien a voir avec le Pipe!)
 crefits();

 // Ouverture du gestionnaire de TOI.
 TOIManager* mgr = TOIManager::getManager();

 // Selection eventuelle des sample num a traiter
 long sdeb=1,sfin=-1;
 if(narg>1) sscanf(arg[1],"%ld,%ld",&sdeb,&sfin);
 if(sfin>=sdeb) mgr->setRequestedSample(sdeb,sfin);

 //--------------------------------------------------------------------
 try { // On met tous ca dans un bloc "try" pour recuperer les exceptions
 //--------------------------------------------------------------------

 ////////////////////////////////////////////////////////
 //////// Creation des lecteurs de fichiers fits ////////
 ////////////////////////////////////////////////////////

 // FITS reader du premier fichier
 FITSTOIReader rfits1("demo1.fits");
 int ncol1 = rfits1.getNOut();
 cout<<"Lecteur_1: Number of columns in fits  : "<<ncol1<<endl;
 if(ncol1<1) exit(-1);

 // FITS reader du deuxieme fichier
 FITSTOIReader rfits2("demo2.fits");
 int ncol2 = rfits2.getNOut();
 cout<<"Lecteur_2: Number of columns in fits  : "<<ncol2<<endl;
 if(ncol1<2) exit(-1);

 /////////////////////////////////////////////////////////
 //////// Creation de l'ecriveur de fichiers fits ////////
 /////////////////////////////////////////////////////////

 FITSTOIWriter wfits("!demo.fits");
 cout<<"Ecriveur: created"<<endl;

 ////////////////////////////////////////////////////
 //////// Creation du (/des) TOI processeurs ////////
 ////////////////////////////////////////////////////

 DemoPipe demo;

 //////////////////////////////////////////////////////////////////
 //////// Creation des tuyaux et des connections associees ////////
 //////////////////////////////////////////////////////////////////

 int taille = 8192;
 bool writeflag;

 // tuyau bolo10 pour entree processeur
 TOISegmented * bolo10 = new TOISegmented("tuyau_bolo_10",taille);
 // connection a la colonne correspondante du lecteur fits
 rfits1.addOutput("boloMuV_10",bolo10);
 // connection a l'entree correspondante du processeur
 demo.addInput("entree_bolo_1",bolo10);
 // connection directe a l'ecriveur
 writeflag = true;
 wfits.addInput("boloMuV_10",bolo10,writeflag);

 // tuyau bolo20 pour entree processeur (idem cf au dessus)
 TOISegmented * bolo20 = new TOISegmented("tuyau_bolo_20",taille);
 rfits1.addOutput("boloMuV_20",bolo20);
 demo.addInput("entree_bolo_2",bolo20);
 writeflag = false;
 wfits.addInput("boloMuV_20",bolo20,writeflag);

 // tuyau bolo30 pour entree processeur (idem cf au dessus)
 TOISegmented * bolo30 = new TOISegmented("tuyau_bolo_30",taille);
 rfits2.addOutput("boloMuV_30",bolo30);
 demo.addInput("entree_bolo_3",bolo30);
 writeflag = true;
 wfits.addInput("boloMuV_30",bolo30,writeflag);

 // tuyau bolosum pour sortie processeur
 TOISegmented * bolosum = new TOISegmented("tuyau_bolo_sum",taille);
 demo.addOutput("sortie_bolo_sum",bolosum);
 writeflag = true;
 wfits.addInput("boloSum",bolosum,writeflag);

 // tuyau bolomul pour sortie processeur
 TOISegmented * bolomul = new TOISegmented("tuyau_bolo_mul",taille);
 demo.addOutput("sortie_bolo_mul",bolomul);
 writeflag = true;
 wfits.addInput("boloMul",bolomul,writeflag);

 // Print de status avant lancement des taches:
 cout<<"----- FITSReaderTOI::PrintStatus() 1 : -----"<<endl;
 rfits1.PrintStatus(cout);
 cout<<"----- FITSReaderTOI::PrintStatus() 2 : -----"<<endl;
 rfits2.PrintStatus(cout);
 cout<<"----- FITSWriterTOI::PrintStatus() : -----"<<endl;
 wfits.PrintStatus(cout);  
 cout<<"----- TOISegmented::PrintStatus() : -----"<<endl;
 bolo10->PrintStatus(cout);  
 bolo20->PrintStatus(cout);  
 bolo30->PrintStatus(cout);  
 cout<<"----- DemoPipe::PrintStatus() : -----"<<endl;
 demo.PrintStatus(cout);  

 //////////////////////////////////////
 //////// Lancement des taches ////////
 //////////////////////////////////////

 rfits1.start();
 rfits2.start();
 wfits.start();
 demo.start();

 //  Affichage de l'avancement des TOIProcessors (optionnel)
 // ProcSampleCounter<FITSTOIReader> statr(rfits1);
 // statr.InfoMessage() = "tstdemopipe/Info rfits1";
 // statr.PrintStats();

 // ProcSampleCounter<FITSTOIWriter> statw(wfits);
 // statw.InfoMessage() = "tstdemopipe/Info wfits";
 // statw.PrintStats();

 // ProcSampleCounter<DemoPipe> statp(demo);
 // statp.InfoMessage() = "tstdemopipe/Info DemoPipe";
 // statp.PrintStats();

 // Gestion de la re-connection des threads
 mgr->joinAll();
 cout<<"End threads"<<endl;

 //--------------------------------------------------------------------
 // Que fait on avec les exceptions qui ont ete lancees ?
 } catch (PThrowable & exc) {    // Sophya exceptions
  cout<<"\ntstdemopipe: Catched Exception \n"<<(string)typeid(exc).name() 
      <<" - Msg= "<<exc.Msg()<<endl;
 } catch (const std::exception & sex) {    // Standard exceptions
  cout<<"\ntstdemopipe: Catched std::exception \n" 
      <<(string)typeid(sex).name()<<endl; 
 } catch (...) {    // Other exceptions
  cout<<"\ntstdemopipe: some other exception was caught ! "<<endl;
 }
 //--------------------------------------------------------------------

 return(0);
}


///////////////////////////////////////////////////////////
//////// Just for the test, NOT CONNECTED WITH THE PIPE !
///////////////////////////////////////////////////////////
#include "srandgen.h"
#include "fabtwriter.h"
void crefits(void)
{
 const int nro = 500000;
 const int sn1 = 100001;
 const int sn2 = 100001+100;
 FitsABTWriter fw1("!demo1.fits",BINARY_TBL);
    fw1.SetExtName("MON_EXTENSION1");
    fw1.AddCol("sampleNum",NULL,"integer",TDOUBLE);
    fw1.AddCol("boloMuV_10",NULL,"double",TDOUBLE);
    fw1.AddCol("fg_boloMuV_10",NULL,"UInt_8Flag",TLONG);
    fw1.AddCol("boloMuV_20",NULL,"double",TDOUBLE);
    //fw1.AddCol("fg_boloMuV_20",NULL,"UInt_8Flag",TLONG);
    for(int i=0;i<nro;i++) {
      fw1.Write(0,i,sn1+i);
      fw1.Write(1,i,10.+NorRand());
      fw1.Write(2,i,((frand01()<0.5) ? 1: 0)); // 1 pt/10 flaggue
      fw1.Write(3,i,20.+NorRand());
      //fw1.Write(4,i,((frand01()<0.5) ? 2: 0)); // 1 pt/10 flaggue
    }

 FitsABTWriter fw2("!demo2.fits",BINARY_TBL);
    fw2.SetExtName("MON_EXTENSION2");
    fw2.AddCol("sampleNum",NULL,"integer",TDOUBLE);
    fw2.AddCol("boloMuV_30",NULL,"double",TDOUBLE);
    fw2.AddCol("fg_boloMuV_30",NULL,"UInt_8Flag",TLONG);
    for(int i=0;i<nro;i++) {
      fw2.Write(0,i,sn2+i);
      fw2.Write(1,i,30.+NorRand());
      fw2.Write(2,i,((frand01()<0.5) ? 4: 0));
    }
}
