#include <iostream.h>
#include "datacirclefits.h"

/*
   Class used to recover the circles and measures involved in a mission. The parameters needful to create a circle and the number of samples on this circle are read from a FITS data array. In addition, a function returns the measurement at a given angular position psi on the circle.
*/

DataCircleFits::DataCircleFits() : 
  _fptr(NULL)
, _NMeasurements(0)
, _ICircle(0)
, _mesures(NULL)
,_stored_data(false)
{;}

DataCircleFits::DataCircleFits(FitsInFile& is,int hdunum,bool sdata) :
  _fptr(NULL)
, _NMeasurements(0)
, _ICircle(hdunum)
, _mesures(NULL)
,_stored_data(sdata)
{

  // pointer to the FITS file
  _fptr= &is;

  // move to the HDU containing a circle
  _fptr->ReadHeader(hdunum);

  // reference on a DVList containing the keywords
  DVList dvl= _fptr->DVListFromFits();

  // angles of the circle
  double theta= dvl.GetD("CIRTHETA");
  double phi  = dvl.GetD("CIRPHI");
  double aperture= dvl.GetD("CIRAPER");

  UnitVector temp(theta,phi);
  SetCircle(temp,aperture);

 _NMeasurements= dvl.GetI("NSAMPLES");

 if(_stored_data) {
   _mesures= new double[_NMeasurements];
   int i;
   for(i = 0; i < _NMeasurements; i++) {
     _fptr->GetBinTabLine(i,&_mesures[i],NULL,NULL,NULL);
   }
 }
}

DataCircleFits::~DataCircleFits() {

  if(_mesures != NULL) delete [] _mesures;
}

int DataCircleFits::NMeasurements() const {

  return _NMeasurements;
}

double DataCircleFits::getData(double psi) const {

  double dtab;
  int ibin= (int)floor(psi*_NMeasurements/2./M_PI);

  if(_stored_data) {
    dtab= _mesures[ibin];
  } else {
    _fptr->ReadHeader(_ICircle);
    _fptr->GetBinTabLine(ibin,&dtab,NULL,NULL,NULL);
  }
  //cout << "DataCircleFits:: bin= " << ibin << ", " << dtab << endl;
  return dtab;
}

void DataCircleFits::print(ostream& out) const {

  out << " Circle:: ApertureAngle= " << ApertureAngle() << ", Theta= " 
      << Theta() << ", Phi= " << Phi() << ", NSamples= " 
      << NMeasurements() << endl;
}
