// Pour boucher les trous d'une sphere HEALPIX en la clusterisant
// dans une sphere plus petite puis en la re-extrapolant.
//                    cmv 13/6/01
// extrapsph -r 2 sph143k05.fits sph143k05_e.fits
//                sphout.fits sphoutw.fits sphred.fits sphredw.fits
#include "machdefs.h"
#include <unistd.h>
#include <stdexcept>
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include "skymapinit.h"
#include "skymap.h"
#include "fitsspherehealpix.h" 

/*!
  \ingroup PrgMap
  \file extrapsph.cc
  \brief \b extrapsph: Fill holes in a sphere of datas by reducing
  it size and re-extrapolating.
  \verbatim
csh> extrapsph -h
extrapsph [-r reduc_factor] sphin.fits sphinw.fits
          sphout.fits [sphout_w.fits sphred.fits sphred_w.fits]
 -r reduc : reduction factor for clustering (must be 2^n, def=2)
 sphin.fits    : input sphere
 sphin_w.fits  : input sphere filling weight
 sphout.fits   : output sphere
 sphout_w.fits : output sphere filling weight
 sphred.fits   : output reducted sphere
 sphred_w.fits : output reducted sphere filling weight
  \endverbatim
*/

void usage();
void usage()
{
cout<<"extrapsph [-r reduc_factor] sphin.fits sphinw.fits"<<endl
    <<"          sphout.fits [sphout_w.fits"
    <<" sphred.fits sphred_w.fits]"<<endl
    <<" -r reduc : reduction factor for clustering (must be 2^n, def=2)"<<endl
    <<" sphin.fits    : input sphere"<<endl
    <<" sphin_w.fits  : input sphere filling weight"<<endl
    <<" sphout.fits   : output sphere"<<endl
    <<" sphout_w.fits : output sphere filling weight"<<endl
    <<" sphred.fits   : output reducted sphere"<<endl
    <<" sphred_w.fits : output reducted sphere filling weight"<<endl;
}

int main(int narg, char* arg[])
{
int red=2;
string dum;
int c;
while((c = getopt(narg,arg,"hr:")) != -1) {
  switch (c) {
  case 'r' :
    sscanf(optarg,"%d",&red);
    break;
  case 'h' :
  default:
    usage(); exit(1);
  }
}

if(optind+2>=narg) {usage(); exit(1);}
char * fsphin   = arg[optind];
char * fsphinw  = arg[optind+1];
char * fsphout  = arg[optind+2];
char * fsphoutw = NULL;
char * fsphred = NULL;
char * fsphredw = NULL;
if(optind+3<narg) fsphoutw = arg[optind+3];
if(optind+4<narg) fsphred = arg[optind+4];
if(optind+5<narg) fsphredw = arg[optind+5];

cout<<"Input Sphere : "<<fsphin<<endl
    <<"Input Weight Sphere : "<<fsphinw<<endl
    <<"Output Sphere : "<<fsphout<<endl
    <<"Output Weight Sphere : "<<fsphoutw<<endl
    <<"Output Reducted Sphere : "<<fsphred<<endl
    <<"Output Reducted Weight Sphere : "<<fsphredw<<endl
    <<"Reduction : "<<red<<endl;

// Lecture des spheres
SphereHEALPix<r_8> sphin;
{FitsInFile sfits(fsphin); sfits >> sphin;}
int nlat = sphin.SizeIndex();
int nlatr = nlat/red;
cout<<"Opening Input Sphere :"<<endl
    <<"          Type of map : "<<sphin.TypeOfMap()<<endl
    <<"     Number of pixels : "<<sphin.NbPixels()<<endl
    <<"                 Nlat : "<<sphin.SizeIndex()<<endl;

SphereHEALPix<r_8> sphinw;
{FitsInFile sfits(fsphinw); sfits >> sphinw;}
cout<<"Opening Input Weight Sphere :"<<endl
    <<"          Type of map : "<<sphinw.TypeOfMap()<<endl
    <<"     Number of pixels : "<<sphinw.NbPixels()<<endl
    <<"                 Nlat : "<<sphinw.SizeIndex()<<endl;
if(sphinw.SizeIndex()!=nlat) {
  cout<<"Incompatible sphin sphinw"<<endl;
  exit(2);
}

// Creation des spheres reduites
cout<<"Creating Reducted Spheres : nlatr = "<<nlatr<<endl;
SphereHEALPix<r_8> sphred(nlatr);
  sphred.SetPixels(0.);
SphereHEALPix<r_8> sphredw(nlatr);
  sphredw.SetPixels(0.);
cout<<"     Number of pixels : "<<sphred.NbPixels()<<endl
    <<"                 Nlat : "<<sphred.SizeIndex()<<endl;

// Filling reducted spheres
cout<<"...Filling Reducted Spheres"<<endl;
uint_4 n=0;
int_4 i, ir;
for(i=0;i<sphin.NbPixels();i++) {
  if(sphinw(i)<=0.) continue;
  double theta,phi;
  sphin.PixThetaPhi(i,theta,phi);
  ir = sphred.PixIndexSph(theta,phi);
  sphred(ir)  += sphin(i)*sphinw(i);
  sphredw(ir) += sphinw(i);
  n++;
}
cout<<"      Input Sphere: Number of filled pixels "<<n<<endl;
cout<<"...Computing Reducted Spheres"<<endl;
n=0;
for(ir=0;ir<sphred.NbPixels();ir++)
  if(sphredw(ir) > 0.) {sphred(ir) /= sphredw(ir); n++;}
cout<<"      Reducted Sphere: Number of pixels filled "<<n<<endl;

// Filling hole for Output Spheres
cout<<"...Filling hole for Output Spheres"<<endl;
n=0;
for(i=0;i<sphin.NbPixels();i++) {
  if(sphinw(i)>0.) continue;
  double theta,phi;
  sphin.PixThetaPhi(i,theta,phi);
  int_4 ir = sphred.PixIndexSph(theta,phi);
  if(sphredw(ir)<=0.) continue;
  sphin(i)  = sphred(ir);
  // On passe en negatif les pixels qui ont ete extrapoles
  sphinw(i) = -sphredw(ir);
  n++;
}
cout<<"      Output Sphere: Number of pixels filled "<<n<<endl;

// Ecriture des spheres
{
dum = "rm -f "; dum += fsphout; system(dum.c_str());
FitsOutFile sfits(fsphout, FitsFile::clear);
cout<<"Writing Output Sphere Fits file"<<endl;
sfits<<sphin;
}
if(fsphoutw) {
dum = "rm -f "; dum += fsphoutw; system(dum.c_str());
FitsOutFile sfits(fsphoutw, FitsFile::clear);
cout<<"Writing Output Sphere Weight Fits file"<<endl;
sfits<<sphinw;
}
if(fsphred) {
dum = "rm -f "; dum += fsphred; system(dum.c_str());
FitsOutFile sfits(fsphred, FitsFile::clear);
cout<<"Writing Reducted Sphere Fits file"<<endl;
sfits<<sphred;
}
if(fsphredw) {
dum = "rm -f "; dum += fsphredw; system(dum.c_str());
FitsOutFile sfits(fsphredw, FitsFile::clear);
cout<<"Writing Reducted Sphere Weight Fits file"<<endl;
sfits<<sphredw;
}

exit(0);
}
