// This may look like C code, but it is really -*- C++ -*-
// Module PI : Peida Interactive     PIGraphic 
// Primitives de trace graphiques    R. Ansari  97
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#ifndef PIGRAPHICGEN_H_SEEN
#define PIGRAPHICGEN_H_SEEN

#include "pisysdep.h"

#include PIWDG_H
#include PICMAP_H
#include PIFONT_H

#include "picolist.h"  // enum PIColors

// Line drawing attributes
enum PILineJoin { PI_JoinMiter=0, PI_JoinRound=1, PI_JoinBevel=2};
enum PILineCap  { PI_CapButt=0, PI_CapRound=1, PI_CapProjecting=2};
enum PILineDash { PI_LineSolid, PI_LineDashed, PI_LineDotted, 
		  PI_LineDashDotted};
// Predefined line types
enum PILineTypes { PI_NotDefLineAtt,
		   PI_NormalLine, PI_ThinLine, PI_ThickLine,
		   PI_DashedLine, PI_ThinDashedLine, PI_ThickDashedLine,
		   PI_DottedLine, PI_ThinDottedLine, PI_ThickDottedLine,
                   PI_DashDottedLine, PI_ThinDashDottedLine, PI_ThickDashDottedLine};

// Marker types
enum PIMarker  { PI_NotDefMarker = -1,
                 PI_DotMarker = 0, PI_PlusMarker=1, PI_CrossMarker=2,  
                 PI_CircleMarker=3, PI_FCircleMarker=4,
                 PI_BoxMarker=5, PI_FBoxMarker=6,
                 PI_TriangleMarker=7, PI_FTriangleMarker=8,
                 PI_StarMarker=9, PI_FStarMarker=10 };

// ArrowMarker types (oriented markers)
enum PIArrowMarker { PI_NotDefArrowMarker, PI_BasicArrowMarker, 
		     PI_TriangleArrowMarker, PI_FTriangleArrowMarker,
		     PI_ArrowShapedArrowMarker, PI_FArrowShapedArrowMarker };

// Flag de positionnement 
enum PIGrPosHorizontal // Flags de positionnement horizontal
// PI_HorizontalPosition regroupe l'ensemble des bits utilisables
     { PI_HorizontalLeft = 0x1, PI_HorizontalCenter = 0x2, 		 
       PI_HorizontalRight = 0x3, PI_HorizontalPosition = 0xF };

enum PIGrPosVertical  // Flags de positionnement vertical
// PI_VerticalPosition regroupe l'ensemble des bits utilisables
     { PI_VerticalBottom = 0x10, PI_VerticalBaseLine = 0x20, 
       PI_VerticalCenter = 0x30, PI_VerticalTop = 0x40, 
       PI_VerticalPosition = 0xF0 };

// Les differents modes de trace graphique disponible
enum PIGOMode { PI_GOCopy = 0, PI_GOXOR = 1, PI_GOInvert = 2 }; 


//  Classe pour gerer indifferement des coordonnees double ou int 
class PIGrCoord {
public:
  inline PIGrCoord() { iv = 0; dv = 0.;}
  inline PIGrCoord(int a) { iv = a; dv = (double)a;}
  inline PIGrCoord(float a) { iv = (int)a; dv = (double)a;}
  inline PIGrCoord(double a) { iv = (int)a; dv = a;}
  inline PIGrCoord(PIGrCoord const & gc) { iv = gc.iv;  dv = gc.dv; }
  inline operator int() { return(iv); }
  inline operator short() { return((short)iv); }
  inline operator unsigned int() { return((unsigned int)iv); }
  inline operator unsigned short() { return((unsigned short)iv); }
  inline operator float() { return((float)dv); }
  inline operator double() { return(dv); }
  inline PIGrCoord& operator= (int a) { iv = a; dv = (double)a; return(*this); }
  inline PIGrCoord& operator= (float a) { iv = (int)a; dv = (double)a; return(*this); }
  inline PIGrCoord& operator= (double a) { iv = (int)a; dv = a; return(*this); }
  inline PIGrCoord& operator= (PIGrCoord const & gc) { iv = gc.iv; dv = gc.dv;  return(*this); }
  int iv;
  double dv;
};

// Classe pour gerer les attributs de lignes
class PILineAtt {
public:
  explicit inline PILineAtt(int width=1, PILineDash dash=PI_LineSolid, 
			    PILineJoin join=PI_JoinMiter, PILineCap cap=PI_CapButt)
  {_lwidth = (width>0)?width*8:8; _ldash=dash; _ljoincap=join+cap*256; }
  explicit inline PILineAtt(double width, PILineDash dash=PI_LineSolid, 
			    PILineJoin join=PI_JoinMiter, PILineCap cap=PI_CapButt)
  { _lwidth = (width>0)?(unsigned short)(width*8.):8; _ldash=dash; _ljoincap=join+cap*256; } 
  inline PILineAtt(PILineAtt const& b)
  { _lwidth=b._lwidth; _ldash=b._ldash; _ljoincap=b._ljoincap; }
 
  PILineAtt(PILineTypes ltyp);

  inline ~PILineAtt() {}

  inline PILineAtt& operator = (PILineAtt const& b)
  { _lwidth=b._lwidth; _ldash=b._ldash; _ljoincap=b._ljoincap; return(*this); }  
  inline PILineAtt& operator = (PILineTypes ltyp)
  { *this=PILineAtt(ltyp);  return(*this); }
   
  inline bool operator == (PILineAtt const& b) const
  { return ((_lwidth==b._lwidth)&&(_ldash==b._ldash)&&(_ljoincap==b._ljoincap)); }
  inline bool operator == (PILineTypes ltyp) const
  { return ((*this) == PILineAtt(ltyp)); }
  inline bool operator != (PILineAtt const& b) const
  { return ((_lwidth!=b._lwidth)||(_ldash!=b._ldash)||(_ljoincap!=b._ljoincap)); }
  inline bool operator != (PILineTypes ltyp) const
  { return ((*this) != PILineAtt(ltyp)); }

  inline int GetLineWidth() const { return ((_lwidth>0)?_lwidth/8:1); }
  inline int GetLineWidthx8() const { return (_lwidth); }
  inline double GetLineWidthD() const { return ((_lwidth>0)?(double)_lwidth/8.:1.); }
  inline PILineDash GetLineDash() const { return _ldash; }
  inline PILineJoin GetLineJoin() const { return (PILineJoin)(_ljoincap%256); }
  inline PILineCap  GetLineCap() const { return (PILineCap)(_ljoincap/256); }

  inline void SetLineWidth(int lw) { _lwidth = lw*8; }
  inline void SetLineWidth(double lw) { _lwidth = (unsigned short)(lw*8.); }
  inline void SetLineDash(PILineDash ld)  { _ldash = ld; }
  inline void SetLineJoin(PILineJoin lj)  { _ljoincap = lj+(_ljoincap&0xFF00); }
  inline void SetLineCap(PILineCap lc)  { _ljoincap = lc*256+(_ljoincap&0x00FF); }

protected:
  unsigned short _lwidth;  // en 1/8 de pixels (ou points)
  unsigned short _ljoincap;
  PILineDash _ldash;  
};

// Les differents types de classes PIGraphic
enum PIGraphicsType { PI_UnknownGraphics = 0, 
                      PI_ScrWindowGraphics = 2, PI_ScrBufferGraphics = 3, 
                      PI_PSFileGraphics = 8, 
                      PI_UCGraphics = 32, PI_3DGraphics = 36 };

// Classe generique de trace graphique de base :

class PIGraphicGen
{
public:
                     PIGraphicGen();
                     PIGraphicGen(PIWdg* wdg);
                     PIGraphicGen(PIScreenBuffer* grb);
  virtual           ~PIGraphicGen();

  virtual int        kind();

//  Espace des coordonnees
  virtual void       GetGrSpace(PIGrCoord& xmin, PIGrCoord& xmax, PIGrCoord& ymin, PIGrCoord& ymax); 

//  Trace graphiques
  virtual void       Erase(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)     = 0;

  virtual void       DrawString(PIGrCoord x, PIGrCoord y, const char* s, int pos = 0)        = 0;
  virtual void       DrawOpaqueString(PIGrCoord x, PIGrCoord y, const char* s, int pos = 0)  = 0;
  virtual void       DrawCompString(PIGrCoord x, PIGrCoord y, const char* s, 
                                    const char* s_up, const char* s_dn, int pos = 0);

  virtual void       DrawLine(PIGrCoord x1, PIGrCoord y1, PIGrCoord x2, PIGrCoord y2)  = 0;
  virtual void       DrawBox(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)   = 0;
  virtual void       DrawFBox(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)  = 0;
  virtual void       DrawCircle(PIGrCoord x0, PIGrCoord y0, PIGrCoord r)               = 0;
  virtual void       DrawFCircle(PIGrCoord x0, PIGrCoord y0, PIGrCoord r)              = 0;
  virtual void       DrawOval(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)  = 0;
  virtual void       DrawFOval(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy) = 0;
  virtual void       DrawPolygon(PIGrCoord *x, PIGrCoord *y, int n, bool cinc=true)    = 0;  
  virtual void       DrawFPolygon(PIGrCoord *x, PIGrCoord *y, int n, bool cinc=true)   = 0;  
  virtual void       DrawArc(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy,
                             double degdeb, double degfin)                             = 0;
  virtual void       DrawFArc(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy, 
                              double degdeb, double degfin)                            = 0;

  virtual void       DrawMarker(PIGrCoord x0, PIGrCoord y0)                            = 0;
  virtual void       DrawMarkers(PIGrCoord *x, PIGrCoord *y, int n)                    = 0;
  virtual void       DrawArrowMarker(PIGrCoord x1, PIGrCoord y1, PIGrCoord x2, PIGrCoord y2);

  virtual void       DrawPixmap(PIGrCoord x, PIGrCoord y, unsigned char *pix, 
                                int sx, int sy, PIColorMap* cmap)              = 0;

// Modifications attributs graphiques
  virtual void       SelForeground(PIColors col=PI_Black)                      = 0;
  virtual void       SelBackground(PIColors col=PI_White)                      = 0;
  virtual void       SelForeground(PIColorMap& cmap, int cid)                  = 0;
  virtual void       SelBackground(PIColorMap& cmap, int cid)                  = 0;
  virtual void       SelGOMode(PIGOMode mod=PI_GOCopy)                         = 0;
  virtual void       SelLine(PILineAtt const& att)                             = 0;
  inline  void       SelLine(PILineTypes lt=PI_NormalLine) 
                             { SelLine(PILineAtt(lt)); }
  virtual void       SelMarker(int msz=3, PIMarker mrk=PI_DotMarker)           = 0;
  virtual void       SelArrowMarker(int arrsz=5, 
                                    PIArrowMarker arrmrk=PI_BasicArrowMarker);
// Modifications de fonte
  virtual void	     SelFont(PIFont & fnt)				       = 0; 
  virtual void       SelFont(PIFontSize sz=PI_NormalSizeFont, 
                             PIFontAtt att=PI_RomanFont)		       = 0;
  virtual void       SelFontSzPt(int npt=12, PIFontAtt att=PI_RomanFont)       = 0; 

// Selection de zone de trace (clip)
  virtual void       SetClipRectangle(PIGrCoord x0, PIGrCoord y0, PIGrCoord dx, PIGrCoord dy)  = 0;
  virtual void       ClearClipRectangle()                                      = 0;

// Acces aux attributs graphiques 
  virtual PIColors   GetForeground()    = 0;
  virtual PIColors   GetBackground()    = 0;
  virtual PIGOMode   GetGOMode()        = 0;
  virtual PILineAtt  GetLineAtt()       = 0;
  virtual PIMarker   GetMarker()        = 0;
  virtual int        GetMarkerSize()    = 0;
  virtual PIArrowMarker    GetArrowMarker();
  virtual int        GetArrowMarkerSize();

// Acces a la fonte et ses attributs
  inline  PIFont     GetFont() { return (myFont) ; }
  virtual PIGrCoord  GetFontHeight(PIGrCoord& asc, PIGrCoord& desc);
  virtual PIGrCoord  CalcStringWidth(char const* s);
  virtual bool       CalcStringPositionShift(char const* s, int pos,  
					     int& dx, int& dy);

// Sauvegarde des attributs graphiques 
  virtual void       SaveGraphicAtt();
  virtual void       RestoreGraphicAtt();

protected:
// Pour Save/Restore des attributs graphiques 
  PIWdg* myWdg;
  PIScreenBuffer* myGrb;

  PIFont myFont;
  PIArrowMarker myArrowMrk;
  int myArrowMrkSz;
  PIColors  sFCol, sBCol;
  PIGOMode sGOm;
  PIFont sFont;
  PILineAtt sLAtt;
  int sFSize;
  PIMarker sMrk;
  int sMrkSz;
  PIArrowMarker sArrowMrk;
  int sArrowMrkSz;

};

typedef PIGraphicGen PIGraphic;

// hierarchie apparente
//   PIGraphic               ( == Gen, mais pas visible de l'utilisateur).
//     PIGraphicWin   ( == X ou Mac)
//     PIGraphicPS
//     PIGraphicUC

#endif      // PIGRAPHGEN_H_SEEN
