// This may look like C code, but it is really -*- C++ -*-

// ArchTOIPipe           (C)     CEA/DAPNIA/SPP IN2P3/LAL
//                               Eric Aubourg
//                               Christophe Magneville
//                               Reza Ansari
// $Id: simtoipr.h,v 1.12 2002-05-14 13:06:58 ansari Exp $


#ifndef SIMTOIPR_H
#define SIMTOIPR_H

#include "toiprocessor.h"
#include "tvector.h"

// ---------  Un deglitcheur simple
// Dans chaque fenetre de largeur de wsz
// if (val > Mean(Window)+ns*Sigma(Window)) val = Mean(Window)
// Si Pas plus de maxnpt points remplissants cette condition
//
// Structure generale :
// 
//                --------------------
//   toi in  ---> |                  | ---> out    (toi = in_deglitche)
//                | SimpleDeglitcher | ---> mean   (toi - optionnel)  
//                |                  | ---> sigma  (toi - optionnel)
//                |                  | ---> incopie ( toi = in , optionnel)
//                |                  |
//                --------------------
//  Si pas de toi out connecte, seul mean et sigma sont calcule
//  les flags sont mis a jour pour le toi out en sortie

class SimpleDeglitcher : public TOIProcessor {
public:
		SimpleDeglitcher(int wsz=64, double ns=3, 
				 int maxnpt=5, int minnpt=2);
  virtual	~SimpleDeglitcher();

  inline void   SetRange(double min, double max)
    { range_min = min; range_max = max; }
  inline void   GetRange(double& min, double& max) const 
    { min = range_min; max = range_max; }

  inline void   SetWSize(int wsz) 
    { wsize = (wsz < 5) ? 5 : wsz; } 


  void          SetDetectionParam(double ns, double ns2, int maxnpt, 
				  int minnpt, int wszrec=0);
    
  inline void   RepBadSamples(bool gl_samples, bool out_range_samples, bool use_wrec=true)
    {  rec_gl_samples = gl_samples;  rec_out_range_samples = out_range_samples;
    rec_use_wrec =  use_wrec; }

  virtual void	init();  
  virtual void	run();

  inline int	WSize() const { return wsize; }
  inline int	WRecSize() const { return wrecsize; }
  inline double NbSigmas() const { return nsig; }
  inline double NbSigmas2() const { return nsig2; }
  inline int	MaxPoints() const { return maxpoints; }
  inline int	MinPoints() const { return minpoints; }
  
  inline int_8	ProcessedSampleCount() const { return totnscount; }
  inline int_8	GlitchCount() const { return glcount; }
  inline int_8  GlitchSampleCount() const { return glnscount; }
  inline int_8  OutOfRangeSampleCount() const { return out_range_nscount; }
  
  virtual void  PrintStatus(::ostream & os) ; // const plus tard
  
protected:
  int_8 totnscount;   // Nombre total d'echantillon processe
  int_8 glnscount;    // Nombre total de glitch
  int_8 glcount;      // Nombre de glitch detecte
  int_8 out_range_nscount;  // Nombre de sample Out Of Range
  bool deglitchdone;  // Deglitch effectue

  int wsize;        // Taille de fenetre de travail
  int wrecsize;     // Taille de fenetre de calcul pour reconstruite les mauvaises valeur 
                    // pour valeur de glitch 
  double nsig;      // Seuil en nb de sigmas 
  double nsig2;     // Seuil en nb de sigmas, pour les points suivants le 1er
  int maxpoints;    // Nb maxi de points > ns sigmas
  int minpoints;    // Nb mini de points > ns sigmas pour avoir un glitch
  double range_min, range_max;  // Range acceptable pour in

  bool rec_gl_samples;        // if true, replace glitch sample values
  bool rec_out_range_samples; // if true, replace out of range sample values
  bool rec_use_wrec;          // if true, use Mean[Window(wrecsize)] to replace bad samples
                              // else use sliding mean value for 

};


//  Un filtre simple, dans le domaine temporel
//  remplace val -> Somme(val(i)*coeff(i)) ds Fenetre
//
// Structure generale :
// 
//                ------------------
//   toi in  ---> |                | ---> out    (toi = in_filtre)
//                |  SimpleFilter  |  
//                |                | ---> incopie ( toi = in , optionnel)
//                |                |
//                ------------------

class SimpleFilter : public TOIProcessor {
public:
  enum FilterKind {
    UserFilter=0,	// User defined filter function
    MeanFilter=1,       // Replace sample by the window mean value (lowpass)
    SumFilter=2,        // Replace sample by the window sum (lowpass)
    GaussFilter=3,      // Apply a gaussian to the window samples
    DiffFilter=4,	// value -= MeanValue
  };

  static string FilterKind2String(FilterKind fk);
		SimpleFilter(int wsz=128, 
			     FilterKind fk=SimpleFilter::MeanFilter,
			     double a=1., double s=1.);
  		SimpleFilter(Vector const & vc);
		~SimpleFilter();

  inline FilterKind Type() { return fkind; }

  inline int	WSize() const { return wsize; }
  inline int_8	ProcessedSampleCount() const { return totnscount; }
  Vector        FilterCoefficients() const;

  virtual void  PrintStatus(::ostream & os) ; // const plus tard

  virtual void	init();  
  virtual void	run();

protected:
  FilterKind fkind;
  int_8 totnscount;   // Nombre total d'echantillon processe
  int wsize;        // Taille de fenetre de travail
  double* coef;     // Coefficients du filtre 
  
};

//  Classe SimpleAdder
//  Calcule la sortie = Somme_Entree [ coeff[num] * entree[num] ]

class SimpleAdder : public TOIProcessor {
public:
		SimpleAdder(int nbinput); 
		~SimpleAdder();

  void          SetGain(int num, double g);
  double        Gain(int num);

  inline int	NbInput() const { return nb_input; }
  inline int_8	ProcessedSampleCount() const { return totnscount; }

  virtual void  PrintStatus(::ostream & os) ; // const plus tard

  virtual void	init();  
  virtual void	run();

protected:
  int nb_input;
  Vector gains;
  int_8 totnscount;   // Nombre total d'echantillon processe
};


//  Un filtre simple, dans le domaine de Fourier
//  InverseFFT ( FFT(Vecteur(in)) * FilterCoefficient )

class SimpleFourierFilter : public TOIProcessor {
public:
                SimpleFourierFilter(Vector const & vc);  
		~SimpleFourierFilter();

  inline int	WSize() const { return wsize; }
  inline int_8	ProcessedSampleCount() const { return totnscount; }
  inline Vector FilterCoefficients() const
                { Vector rcv; rcv = ffcoef; return(rcv); } 

  virtual void  PrintStatus(::ostream & os) ; // const plus tard

  virtual void	init();  
  virtual void	run();

  inline  void  KeepSpectra(string outname, int nb) 
                { outppfname = outname; nb_keep = nb; }
  inline  void  ComputeMeanSpectra(bool fg)
                {  c_meanspectra = fg; }
protected:
  int_8 totnscount;   // Nombre total d'echantillon processe
  int_8 totnbblock;   // Nombre total de blocs pour FFT
  int wsize;         // Taille de fenetre de travail
  Vector ffcoef;     // Coefficients du filtre
  bool c_meanspectra;
  int nb_keep;
  string outppfname;
};


//  Classe SimpleFanOut
//  Recopie chaque entree sur M lignes de sortie

class SimpleFanOut : public TOIProcessor {
public:
		SimpleFanOut(int nbinput, int mfanout); 
		~SimpleFanOut();

  inline int	NbInput() const { return nb_input; }
  inline int	MFanOut() const { return m_fanout; }
  inline int_8	ProcessedSampleCount() const { return totnscount; }

  virtual void  PrintStatus(::ostream & os) ; // const plus tard

  virtual void	init();  
  virtual void	run();

protected:
  int nb_input;
  int m_fanout;
  int_8 totnscount;   // Nombre total d'echantillon processe
};


#endif
